/*
 * FISG - Fast IRC Statistic Generator
 * Programmed and designed by Matti 'ccr' Hamalainen
 * (C) Copyright 2003 Tecnic Software productions (TNSP)
 *
 * Please read file 'COPYING' for information on license and distribution.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <assert.h>
#include "fisg.h"
#include "th_util.h"
#include "th_args.h"
#include "th_config.h"
#include "in_formats.h"
#include "out_formats.h"

/*
 * Misc globals
 */
int	setOutputFormat = 0;
int	setCurrInputFormat = 0;

int	nverbosity = 2;
int	nsourceFileNames = 0, nconfigFileNames = 0;
int	sourceFileFormats[SET_MAX_INFILES];
char	*sourceFileNames[SET_MAX_INFILES],
	*destFileName = NULL,
	*userFileName = NULL;
char	*configFileNames[SET_MAX_INFILES];

char	*progName = NULL;


t_arg argList[] = {
	{ "?",	"help",		"Show this help", 0 },
	{ "o",	"output",	"Specify output file (default stdout)", 1 },
	{ "u",	"user-file",	"Users file (default: no user-file)", 1 },
	{ "f",	"format",	"Specify input format (logfile format)", 1 },
	{ "of", "output-format","Specify output format", 1 },
	{ "F",	"show-formats",	"Show list of predefined log- and output-formats", 0 },
	{ "c",	"config",	"Specify configuration file (default: none)", 1 },
	{ "q",	"quiet",	"Be less noisy. Good if you run from scripts", 0 }
};

const int argListN = (sizeof(argList) / sizeof(t_arg));


void showHelp()
{
 fprintf(stderr, "\n" RA_NAME " " RA_VERSION " " RA_COPYRIGHT "\n");
 fprintf(stderr, "This software is licensed under GNU General Public License version 2\n");
 fprintf(stderr, "Usage: %s [options] [source#1] [source#2...]\n", progName);

 th_showHelp(argList, argListN);
}


void handleOpt(const int optN, char *optArg, char *currArg)
{
 int i;
 BOOL isFound;
 
 switch (optN) {
 case 0: showHelp(); exit(0); break;
 
 case 1:
 	/* Specify output filename */
 	if (optArg)
 		destFileName = optArg;
 		else {
 		fprintf(stderr, RA_NAME ": No output filename specified!\n");
 		exit(2);
 		}
	break;

 case 2:
 	/* Specify user-file filename */
 	if (optArg)
 		userFileName = optArg;
 		else {
 		fprintf(stderr, RA_NAME ": No user-file filename specified!\n");
 		exit(2);
 		}
	break;

 case 3:
 	/* Specify input format */
 	if (optArg)
 		{
 		/* Go through the list */
 		isFound = FALSE;
 		i = 0;
 		while ((i < nInputFormats) && (!isFound))
 			{
 			if (strcmp(optArg, inputFormats[i].ifName) == 0)
 				isFound = TRUE;
 				else
 				i++;
 			}

 		
 		/* Check */
 		if (!isFound)
 			{
 			fprintf(stderr, RA_NAME ": Invalid input (log-file) format '%s'\n", optArg);
 			exit(2);
 			}
 		
 		setCurrInputFormat = i;
 		} else {
 		fprintf(stderr, RA_NAME ": No input (log-file) format specified!\n");
 		exit(2);
 		}
	break;

 case 4:
 	/* Specify output format */
 	if (optArg)
 		{
 		/* Go through the list */
 		isFound = FALSE;
 		i = 0;
 		while ((i < nOutputFormats) && (!isFound))
 			{
 			if (strcmp(optArg, outputFormats[i].ofName) == 0)
 				isFound = TRUE;
 				else
 				i++;
 			}

 		/* Check */
 		if (!isFound)
 			{
 			fprintf(stderr, RA_NAME ": Invalid output format '%s'\n", optArg);
 			exit(2);
 			}

 		setOutputFormat = i;
 		} else {
 		fprintf(stderr, RA_NAME ": No output format specified!\n");
 		exit(2);
 		}
 	break;

 case 5:
 	/* Show list of input and output formats */
 	fprintf(stderr, RA_NAME ": Available pre-defined INPUT (log) formats:\n");
 	for (i = 0; i < nInputFormats; i++)
 		{
 		fprintf(stderr, "  %-8s	- %s %s\n",
 			inputFormats[i].ifName,
 			inputFormats[i].ifDescription,
 			(i == 0) ? "(default)" : "");
 		}
 		
 	fprintf(stderr, "\n" RA_NAME ": Available OUTPUT formats:\n");
 	for (i = 0; i < nOutputFormats; i++)
 		{
 		fprintf(stderr, "  %-8s	- %s %s\n",
 			outputFormats[i].ofName,
 			outputFormats[i].ofDescription,
 			(i == setOutputFormat) ? "(default)" : "");
 		}
 	
 	fprintf(stderr, "\n");
 	exit(0);
 	break;

 case 6:
 	/* Specify configuration filename */
 	if (optArg)
 		{
		if (nconfigFileNames < SET_MAX_INFILES)
			{
			configFileNames[nconfigFileNames] = optArg;
			nconfigFileNames++;
			}
 		} else {
 		fprintf(stderr, RA_NAME ": No configuration filename specified!\n");
 		exit(2);
 		}
	break;

 case 7:
 	/* Quiet -- lessen verbosity */
 	nverbosity--;
 	break;
 	
 default:
 	/* Error */
 	fprintf(stderr, RA_NAME ": Unknown argument '%s'.\n", currArg);
 	break;
 }
} 	


void handleFile(char *currArg)
{
 /* Was not option argument */
 if (nsourceFileNames < SET_MAX_INFILES)
	{
	sourceFileNames[nsourceFileNames] = currArg;
	sourceFileFormats[nsourceFileNames] = setCurrInputFormat;
	nsourceFileNames++;
	}
}


/*
 * Parsers
 */
int parse_int(char *inLine, int iLen, int *linePos)
{
 int iResult;
 
 iResult = 0;
 while (isdigit(inLine[*linePos]) && (iLen--))
 	{
 	iResult *= 10;
 	iResult += (inLine[(*linePos)++] - '0');
 	}

 return iResult;
}


t_user_entry *parse_newuser(t_stats *pStats, char *newNick, BOOL autoAlias)
{
 int i, j;
 char tmpStr[SET_MAX_NICKLEN + 2 + 1];
 t_user_entry *tmpUser;
 t_nick_entry *tmpNick;
 
 /* Check if nick matches existing user record */
 tmpUser = user_search(pStats->nickList, newNick);
 if (tmpUser == NULL)
	{
	/* No, we need to create a new one */
	tmpUser = user_new(newNick);

	if (autoAlias)
		{
		/* Using auto-aliasing, strip special chars */
		i = j = 0;
		tmpStr[j++] = '*';
		
		while (newNick[i])
			{
			if (isalnum(newNick[i]))
				tmpStr[j++] = newNick[i];
			i++;
			}
		
		tmpStr[j++] = '*';			
		tmpStr[j++] = 0;
					
		/* Add "*strippednick*" */
		tmpNick = nick_new(tmpStr);
		} else
		/* No auto-aliasing, normal nick */
		tmpNick = nick_new(newNick);

	tmpNick->user = tmpUser;
	nick_insert(pStats->nickList, tmpNick);
	user_insert(&pStats->userList, tmpUser);
	}

 return tmpUser;
}



int parse_public(char *inLine, char *fmt, t_stats *pStats, BOOL autoAlias)
{
 int linePos, i;
 t_user_entry *tmpUser;
 char tmpStr[SET_MAX_NICKLEN + 1];
 int iHours, iSeconds, iMinutes;

 /* Initialize */
 linePos = 0;
 iHours = iMinutes = iSeconds = -1;
 tmpUser = NULL;


 /* Parse the line via format-string */
 while (*fmt)
 {
 if (*fmt == '%')
	{
	switch (*(++fmt)) {
	case '?':
		/* Match anything */
		fmt++;
		linePos++;
		break;

	case '*':
		/* Match anything until next char */
		fmt++;
		
		/* Search for it */
		while (inLine[linePos] && (inLine[linePos] != *fmt))
			linePos++;

		break;
		
	case '@':
		/* Match irssi style optional '@|+| ' */
		if ((inLine[linePos] == '@') || (inLine[linePos] == '+') || isspace(inLine[linePos]))
			linePos++;
		fmt++;
		break;

	/* Timestamp */
	case 'H':
		/* Hours */
		iHours = parse_int(inLine, 2, &linePos);
		fmt++;
		break;

	case 'M':
		/* Minutes */
		iMinutes = parse_int(inLine, 2, &linePos);
		fmt++;
		break;
		
	case 'S':
		/* Seconds */
		iSeconds = parse_int(inLine, 2, &linePos);
		fmt++;
		break;

	/* Nick */
	case 'n':
		/* Get ending character */
		fmt++;
		
		/* Find the start of the nick */
		th_findnext(inLine, &linePos);

		/* Get the nick to temp buffer */
		i = 0;
		while ((inLine[linePos]) && (inLine[linePos] != *fmt) && (i < SET_MAX_NICKLEN))
			tmpStr[i++] = inLine[linePos++];

		tmpStr[i++] = 0;
		
		/* Find user or add new */
		if (i > 0)
			tmpUser = parse_newuser(pStats, tmpStr, autoAlias);

		/* And neext .. */
		break;
	
	/* Error */
	default:
		fprintf(stderr, "parse_msg_public: Syntax error in format-string '%s'\n", fmt);
		return -1;
	}
	} else {
	/* Check matches */
	if (*fmt != inLine[linePos])
		return 1;

	fmt++;
	linePos++;
	}

 } /* while(*fmt) */


 /* Check if we got user for this ... */
 if (tmpUser)
 {
 t_int	nWords, nQuestions, nYelling,
 	nHappyFaces, nSadFaces;

 int tmpPos;
 BOOL isWord;


 /* Detect URLs */
 if (strstr(&inLine[linePos], "http://") != NULL)
	{
	/* Add the URL in list */
	tmpUser->nURLs++;
	}

 /* Statisticize the actual public message-line */
 tmpPos = linePos;
 isWord = FALSE;

 nHappyFaces = nSadFaces = nQuestions = nYelling = nWords = 0;

 while (inLine[linePos])
 	{
 	if (isWord && isspace(inLine[linePos]))
 		{
 		nWords++;
 		isWord = FALSE;
 		} else
 	if ((!isWord) && !isspace(inLine[linePos]))
		isWord = TRUE;

	if (isupper(inLine[linePos]))
		tmpUser->nCaps++;

	switch (inLine[linePos]) {
	case ':':
	case ';':
		switch (inLine[linePos + 1]) {
		case ')': /* :) */
		case 'D': /* :D */
		case 'P': /* :P */
		case '>': /* :> */
		case ']': /* :] */
			nHappyFaces++;
			break;
	
		case '(': /* :( */
		case '[': /* :[ */
		case '/': /* :/ */
		case 'I': /* :I */
			nSadFaces++;
			break;
		}
		break;

	case '!':
		nYelling++;
		break;
	
	case '?':
		nQuestions++;
		break;
	}


 	tmpUser->nChars++;

 	linePos++;
 	}

 /* Add to user's stats */
 if (nYelling) tmpUser->nYelling++;
 if (nQuestions) tmpUser->nQuestions++;
 if (nHappyFaces) tmpUser->nHappyFaces++;
 if (nSadFaces) tmpUser->nSadFaces++;
 
 tmpUser->nWords += nWords;
 tmpUser->nPublics++;

 if ((iHours >= 0) && (iHours < SET_HOURS_DAY))
 	{
	tmpUser->nWordsPerHour[iHours] += nWords;
	tmpUser->nPublicsPerHour[iHours]++;

	if (tmpUser->nWords >= (tmpUser->nWordsPerHour[iHours] / (tmpUser->nPublicsPerHour[iHours]+1)))
		{
		if ((!tmpUser->sComment) || (random() < (RAND_MAX / 3)))
			th_strcpy(&tmpUser->sComment, &inLine[tmpPos]);
		}
 	}
 }
 
 /* Done, ok. */ 
 return 0;
}



/*
 * A generic logfile parser
 */
int parse_log(char *logFilename, FILE *inFile, t_stats *pStats, t_logformat *logFmt, t_config *pCfg)
{
 BOOL autoAlias;
 char inLine[SET_MAX_BUF + 1];
 int lineNum, linePos;

 /* Get configuration options */
 autoAlias = th_config_get_bool(pCfg, CFG_GEN_AUTO_ALIAS_NICKS, FALSE);

 /* Initial stats */
 pStats->nLogFiles++;

 /* Read and parse the data */
 lineNum = 0;
 while (fgets(inLine, sizeof(inLine), inFile) != NULL)
 {
 pStats->nChars += strlen(inLine);
 pStats->nLines++;
 lineNum++;
 linePos = 0;

 /* Check if the line is OK and what type it is */ 
 if (inLine[0])
	{
	if (parse_public(inLine, logFmt->fmtPublic, pStats, autoAlias) > 0)
		{
		
		}
 	}
 
 } /* if (fgets()) */

 return 0;
}


/*
 * Userfile parser
 */
int parse_users(char *usrFilename, t_stats *pStats, t_config *pCfg)
{
 char inLine[SET_MAX_BUF + 1], tmpStr[SET_MAX_BUF + 1];
 FILE *inFile;
 int i, lineNum, linePos;
 t_user_entry *tmpUser;
 t_nick_entry *tmpNick;

 assert(pCfg);
 assert(pStats);
 
 /* Get configuration options */

 /* Try to open the file */
 if ((inFile = fopen(usrFilename, "ra")) == NULL)
	return -1;
 
 /* Read and parse the data */
 lineNum = 0;
 while (fgets(inLine, sizeof(inLine), inFile) != NULL)
 {
 lineNum++;
 linePos = 0;

 /* Check if the line is OK and what type it is */ 
 if ((strlen(inLine) > 1) && (inLine[0] != '#'))
	{
	/* Get the handle */
	i = 0;
	while (inLine[linePos] && (inLine[linePos] != ':') && (i < SET_MAX_BUF))
		tmpStr[i++] = inLine[linePos++];
	
	tmpStr[i++] = 0;

	/* Check if next field exists */
	if (inLine[linePos] != ':')
		{
		fprintf(stderr, RA_NAME ": Error in userfile '%s', line %i - missing fields.\n",
			usrFilename, lineNum);
		} else {
		/* Allocate a new user and nick records */
		tmpUser = user_new(tmpStr);
		user_insert(&pStats->userList, tmpUser);
		
		/* Get alias nicks */
		linePos++;
		while (inLine[linePos] && (inLine[linePos] != ':'))
			{
			/* Get one nick */
			i = 0;
			while (inLine[linePos] && (inLine[linePos] != ':') &&
				(!isspace(inLine[linePos])) && (i < SET_MAX_BUF))
				tmpStr[i++] = inLine[linePos++];
	
			tmpStr[i++] = 0;

			if (isspace(inLine[linePos]))
				linePos++;

			/* Add to user */
			tmpNick = nick_new(tmpStr);
			tmpNick->user = tmpUser;
			
			nick_insert(pStats->nickList, tmpNick);
			}
		
		/* Get image path */
		if (inLine[linePos])
			{
			linePos++;
			if (inLine[linePos] != ':')
				{
				/* Get image path */
				i = 0;
				while (inLine[linePos] &&
					(inLine[linePos] != ':') &&
					(i < SET_MAX_BUF))
					tmpStr[i++] = inLine[linePos++];
	
				tmpStr[i++] = 0;
				
				/* Set to user record */
				th_strcpy(&tmpUser->picPath, tmpStr);
				}
			}
		}
 	}
 
 } /* if (fgets()) */

 return 0;
}


/*
 * Allocates and initializes a new t_stats structure
 */
t_stats *stats_new(void)
{
 t_stats *pResult;
 
 /* Allocate memory for new node */
 pResult = (t_stats *) calloc(1, sizeof(t_stats));
 if (pResult == NULL) return NULL;
 
 /* Initialize fields */

 /* Return result */
 return pResult;
}


/*
 * Compare function for qsort() that compares 2 nodes for nTotalScore
 */
int stats_index_cmp(const void *pNode1, const void *pNode2)
{
 t_user_entry *pUser1, *pUser2;

 pUser1 = * (t_user_entry **) pNode1;
 pUser2 = * (t_user_entry **) pNode2;

 if (pUser1->nTotalScore > pUser2->nTotalScore)
 	return -1;
 	else
 if (pUser1->nTotalScore < pUser2->nTotalScore)
 	return 1;
 	else
 	return 0;
}


/*
 * Create a userIndex for given stats from userList
 */
int stats_index(t_stats *pStats)
{
 t_user_entry *pCurr;
 long int i;
 assert(pStats);
 
 /* Create sorted index */
 pCurr = pStats->userList;
 pStats->nUsers = 0;
 while (pCurr)
 	{
 	pStats->nUsers++;
 	pCurr = pCurr->pNext;
 	}

 /* Check number of nodes */
 if (pStats->nUsers > 0)
	{
	/* Allocate memory for index-table */
	pStats->userIndex = (t_user_entry **) malloc(sizeof(t_user_entry *) * pStats->nUsers);
	if (pStats->userIndex == NULL) return -6;

	/* Get node-pointers to table */
	i = 0;
	pCurr = pStats->userList;
	while (pCurr)
		{
		pStats->userIndex[i++] = pCurr;
		pCurr = pCurr->pNext;
		}
	}

 return 0;
}


/*
 * Compute stats from userlist
 */
int stats_compute(t_stats *pStats, t_config *pCfg)
{
 t_user_entry *pCurr;
 int i;
 t_float iTotalActivity, iMaxActivity;
 BOOL usePisgScoring;
 
 /* Get configuration options */
 usePisgScoring = th_config_get_bool(pCfg, CFG_GEN_USE_PISG_SCORING, FALSE);

 /* Initialize */
 pStats->mostStupid = pStats->mostLoud = pStats->mostActions =
 pStats->mostModes = pStats->mostKicks = pStats->mostKicked =
 pStats->mostCaps = pStats->mostHappy = pStats->mostSad =
 pStats->mostURLs = pStats->userList;
 
 /* Go through the userlist */
 pCurr = pStats->userList;
 while (pCurr)
 	{
	/* More loud? */
	if (pCurr->nYelling > pStats->mostLoud->nYelling)
		pStats->mostLoud = pCurr;
		
	/* More stupid */
	if (pCurr->nQuestions > pStats->mostStupid->nQuestions)
		pStats->mostStupid = pCurr;

	/* More happy? */
	pCurr->nHappiness = ((t_float) pCurr->nHappyFaces * (t_float) pCurr->nSadFaces) / (t_float) (pCurr->nPublics + 1);

	if (pCurr->nHappiness > pStats->mostHappy->nHappiness)
		pStats->mostHappy = pCurr;

	/* More sad? */
	if (pCurr->nHappiness < pStats->mostSad->nHappiness)
		pStats->mostSad = pCurr;
		
	/* More caps per chars? */
	if ((pCurr->nChars / (pCurr->nCaps + 1)) > (pStats->mostCaps->nChars / (pStats->mostCaps->nCaps + 1)))
		pStats->mostCaps = pCurr;
	
	/* More URLs pasted? */
	if (pCurr->nURLs > pStats->mostURLs->nURLs)
		pStats->mostURLs = pCurr;

	/* Activity */
	iTotalActivity = 0;
	for (i = 0; i < SET_HOURS_DAY; i++)
		{
		iTotalActivity += ((t_float) pCurr->nWordsPerHour[i] * (t_float) pCurr->nPublicsPerHour[i]);
 		}

	/* Compute activity-% for each hour */
	for (i = 0; i < SET_HOURS_DAY; i++)
		{
		if (iTotalActivity > 0)
			{
			pCurr->activityPercentPerHour[i] =
			(((t_float) pCurr->nWordsPerHour[i] * (t_float) pCurr->nPublicsPerHour[i]) / iTotalActivity) * 100.0f;
			} else
			pCurr->activityPercentPerHour[i] = 0.0f;
		
		pStats->activityPercentPerHour[i] += pCurr->activityPercentPerHour[i];
		}

	
	/* Compute W/P and C/W */
	if (pCurr->nPublics > 0)
		pCurr->nWordsPerPublic = ((t_float) pCurr->nWords / (t_float) pCurr->nPublics);
		else
		pCurr->nWordsPerPublic = 0;

	if (pCurr->nWords > 0)
		pCurr->nCharsPerWord = ((t_float) pCurr->nChars / (t_float) pCurr->nWords);
		else
		pCurr->nCharsPerWord = 0;
	
	/*
	 * Compute total score
	 */
	pCurr->nTotalScore = (pCurr->nWordsPerPublic * pCurr->nCharsPerWord * pCurr->nPublics);

	/* Next node */
 	pCurr = pCurr->pNext;
 	}

 /*
  * Compute total activity %
  */
 pStats->activityPeak = -1;
 iMaxActivity = -1;
 iTotalActivity = 0;
 for (i = 0; i < SET_HOURS_DAY; i++)
	{
	iTotalActivity += pStats->activityPercentPerHour[i];
	if (pStats->activityPercentPerHour[i] > iMaxActivity)
		{
		iMaxActivity = pStats->activityPercentPerHour[i];
		pStats->activityPeak = i;
		}
	}

 if (iTotalActivity > 0) { 
 for (i = 0; i < SET_HOURS_DAY; i++)
	{
	pStats->activityPercentPerHour[i] = (pStats->activityPercentPerHour[i] / iTotalActivity) * 100.0f;
	}
 }
	

 return 0;
}


/*
 * NOTICE: Since this utility is designed to be "one pass/shot" program,
 * we don't free any memory used here. If you take this code, remember it
 * and add possible *free()'s where appropriate.
 */
int main(int argc, char *argv[])
{
 FILE *myFile;
 int iResult, i, j;
 t_config *myConfig = NULL;
 t_stats *myStats = NULL;
 time_t myTime1, myTime2;
 char tmpStr[1024] = "";

 /*
  * Initialize basics
  */
 progName = argv[0];

 time(&myTime1);
 srandom(myTime1);
 
 /*
  * Allocate stats
  */
 myStats = stats_new();
 if (myStats == NULL)
 	{
 	fprintf(stderr, RA_NAME ": Could not allocate memory for statistics!\n");
 	}
 
 /*
  * Parse arguments
  */
 th_processArgs(argc, argv,
 	argList, argListN,
 	handleOpt, handleFile);

 if (nsourceFileNames <= 0)
 	{
 	fprintf(stderr, RA_NAME ": No input files specified!\n");
 	exit(3);
 	}

 /*
  * Read and parse general configuration file
  */
 if (nconfigFileNames <= 0)
 	{
 	fprintf(stderr, RA_NAME ": No configuration file(s) specified.\n");
 	}

 for (i = 0; i < nconfigFileNames; i++)
	{
	if (nverbosity >= 1)
	fprintf(stderr, RA_NAME ": Configuration '%s'\n", configFileNames[i]);

	if (th_config_read(configFileNames[i], &myConfig) < 0)
	 	{
	 	fprintf(stderr, RA_NAME ": Error reading configuration file.\n");
	 	exit(4);
	 	}
	}

 /*
  * Parse the users file
  */
 if (userFileName)
	parse_users(userFileName, myStats, myConfig);
	else
	parse_users(th_config_get_str(myConfig, CFG_GEN_USER_FILE, NULL), myStats, myConfig);
 
 /*
  * Read the source-file(s)
  */
 if (nverbosity >= 1)
 fprintf(stderr, RA_NAME ": Parsing %d sources. Please wait...\n", nsourceFileNames);

 for (i = 0; i < nsourceFileNames; i++)
	{
	/* Try to open the logfile */
	if ((myFile = fopen(sourceFileNames[i], "ra")) == NULL)
		{
		fprintf(stderr, RA_NAME ": Error opening input file '%s' (%s)\n", sourceFileNames[i], strerror(errno));
		return -1;
		}

	/* Parse with selected parser */
	iResult = parse_log(sourceFileNames[i],
		myFile, myStats,
		&inputFormats[sourceFileFormats[i]],
		myConfig);

	if (nverbosity >= 2)
		{
		/* Show progress meter */
		for (j = 0; j < strlen(tmpStr); j++)
			fputc('\b', stderr);

		snprintf(tmpStr, sizeof(tmpStr),
			RA_NAME ": Processed %i%%",  (((i+1) * 100) / nsourceFileNames));

		fputs(tmpStr, stderr);
		}

	/* Close file, report errors */
	fclose(myFile);
	if (iResult < 0)
		{
		fprintf(stderr, RA_NAME ": Error #%i reading file (%s)\n", iResult, strerror(errno));
		return 2;
		}
	}

 if (nverbosity >= 2)
	fprintf(stderr, "\n"); 

 /*
  * Calculate statistics
  */
 if (nverbosity >= 1)
 fprintf(stderr, RA_NAME ": Computing statistics...\n");
 if (stats_compute(myStats, myConfig) < 0)
 	{
 	fprintf(stderr, RA_NAME ": Error while computing statistics!\n");
 	return -10;
 	}

 /*
  * Index the list for sorting
  */
 if (stats_index(myStats) < 0)
 	{
 	fprintf(stderr, RA_NAME ": Error while indexing userlist!\n");
 	return -9;
 	}


 /* Sort the indexes by score */
 fprintf(stderr, RA_NAME ": %li nicks/handles, sorting...\n", myStats->nUsers);

 qsort(myStats->userIndex, myStats->nUsers, sizeof(t_user_entry *), stats_index_cmp);


 /* Calculate time */
 time(&myTime2);
 myStats->nTimeElapsed = (myTime2 - myTime1);
 
 /* Open output-file */
 fprintf(stderr, RA_NAME ": Using %s\n", outputFormats[setOutputFormat].ofDescription);

 if (destFileName == NULL) myFile = stdout; else
 if ((myFile = fopen(destFileName, "wa")) == NULL)
 	{
 	fprintf(stderr, RA_NAME ": Error opening output file '%s' (%s)\n", destFileName, strerror(errno));
 	return -1;
 	}

 
 iResult = outputFormats[setOutputFormat].ofFunction(myFile, myStats, myConfig);

 fclose(myFile);

 /*
  * OK! Show final stats
  */
 if (nverbosity >= 1)
 {
 fprintf(stderr,
 	RA_NAME ": %ld lines in %ld logfile(s), total of %1.2f MB\n",
 	myStats->nLines,
 	myStats->nLogFiles,
 	((t_float) myStats->nChars) / (1024.0f*1024.0f)
 	);
 }
  
 if (iResult == 0)
 	{
 	if (nverbosity >= 1)
	fprintf(stderr, RA_NAME ": Done.");
	} else
	fprintf(stderr, RA_NAME ": Error in output! Return code #%i. ", iResult);

 if (nverbosity >= 1)
 { 
 fprintf(stderr, " Time elapsed %ld hours, %ld minutes and %ld seconds.\n",
 	(myStats->nTimeElapsed / (60*60)),
	(myStats->nTimeElapsed % (60*60)) / 60, 
	(myStats->nTimeElapsed % (60*60)) % 60
	);
 }
 
 return 0;
}

