#include "fluctuate.h"

#ifndef MODELLIKE_INCLUDE
#include "fluc_modellike.h"
#endif

#ifdef DMALLOC_FUNC_CHECK
#include "/usr/local/include/dmalloc.h"
#endif

/***************************************************************************
 *  ALPHA                                                                  *
 *  version 1.40. (c) Copyright 1986, 1991, 1992 by the University of      *
 *  Washington and Joseph Felsenstein.  Written by Joseph Felsenstein,     *
 *  Mary K. Kuhner and Jon A. Yamato, with some additional grunt work by   *
 *  Sean T. Lamont.  Permission is granted to copy and use this program    *
 *   provided no fee is charged for it and provided that this copyright    *
 *  notice is not removed.                                                 *
 *                                                                         *
 ***************************************************************************/

/* This program implements a Hastings-Metropolis Monte Carlo
  Maximum Likelihood sampling method for phylogenetic trees
  without recombination. */

/* Time, 'tyme', in this tree is measured from the tips to the root.
   I.e. the tips are at tyme '0', and the root node has the largest
   value for 'tyme'. */

/* Only long chains included in joint estimates */

/* Modified to accept multiple loci. */

extern double **growi, **theti, **lntheti, xinterval;
extern treerec ***sum;

FILE *infile, *outfile, *treefile, *bestree, *simlog,
     *parmfile, *thetafile, *intree;
long numseq, numnodes, sites, rootnum, categs, apps, inseed, col,
     slidenum, numloci, numtrees, locus, totchains, holding;
boolean  **sametree;
tree *curtree;
double locus_ttratio, freqa, freqc, freqg, freqt, 
	      lambda, probsum, theta0, growth0, branch0;
long		*category,*weight;
double		*rate, *ne_ratio, *mu_ratio, *theta_ratio;
double		*probcat;
contribarr	*contribution;
node		**freenodes;
node		**slidenodes;
rlrec		*alogf;
char            gch;
dnadata         *dna;
option_struct   *op;
longer          seed;

/* the following are used in site aliasing (makesiteptr) */
long *siteptr;

/* the following are for reading in parameters (readparmfile) */
char *booltokens[NUMBOOL] = {"interleaved","printdata","progress",
                          "print-trees","freqs-from-data","categories",
                          "watterson", "usertree", "autocorrelation",
                          "interactive"},
     *numbertokens[NUMNUMBER] = {"ttratio","short-chains",
                          "short-steps","short-inc","long-chains",
                          "long-inc","long-steps","growth-rate"};

/* "Nearly global" variables for maketree: */

double	 	sumweightrat, oldlikelihood, watttheta;
double	 	*weightrat;
valrec   	*tbl;
long		slid, slacc, indecks, chaintype;


void openfile(FILE **fp, char *filename, char *mode, char *application,
   char *perm)
{
  FILE *of;
  char file[100];

  strcpy(file,filename);
  while (1){
    of = fopen(file,mode);
    if (of)
      break;
    else {
      switch (*mode){
      case 'r':
        fprintf(stdout,"%s:  can't read %s\n",application,file);
        file[0] = '\0';
        while (file[0] =='\0'){
          fprintf(stdout,"Please enter a new filename>");
          fgets(file,100,stdin);
          }
        break;
      case 'w':
        fprintf(stdout,"%s: can't write %s\n",application,file);
        file[0] = '\0';
        while (file[0] =='\0'){
          fprintf(stdout,"Please enter a new filename>");
          fgets(file,100,stdin);
          }
        break;
      default:
        fprintf(ERRFILE,"%s: unknown file mode for %s\n",
          application, file);
        exit(-1);
        break;
      }
    }
  }
  *fp=of;
  if (perm != NULL)
    strcpy(perm,file);
} /* openfile */

double randum()
/* Mary's version--faster but needs 32 bits.  Loops have been unrolled
   for speed. */
{
  long newseed0, newseed1, newseed2;

  newseed0 = 1549*seed[0];
    newseed1 = newseed0/2048;
    newseed0 &= 2047;
    newseed2 = newseed1/2048;
    newseed1 &= 2047;
  newseed1 += 1549*seed[1] 
              + 812*seed[0];
    newseed2 += newseed1/2048;
    newseed1 &= 2047;
  newseed2 += 1549*seed[2]
              + 812*seed[1];

  seed[0] = newseed0;
  seed[1] = newseed1;
  seed[2] = newseed2 & 1023;
  return (((seed[0]/2048.0 + seed[1])/2048.0 + seed[2])/1024.0);
}  /* randum */

boolean readseedfile(long *inseedptr)
/* read in the seed from the seed file */
{
    FILE *seedfile;
    boolean success;

    seedfile = fopen("seedfile","r");
    if (seedfile) {
      success = fscanf(seedfile, "%ld%*[^\n]", inseedptr);
      fclose(seedfile);
      return(success);
    }
    return(false); /* didn't find seedfile */
} /* readseedfile */

void printtymelist()
/* prints the entire tymelist: a debug function */
{
  long i;
  tlist *t;
  long limit;

  t = curtree->tymelist;
  fprintf(ERRFILE,"TYMELIST BEGINS\n");
  while (t != NULL) {
    fprintf(ERRFILE,"%3ld age%8.6f branches %3ld--",
           t->eventnode->number, t->age, t->numbranch);
    limit = t->numbranch;
    for (i = 0; i < limit; i++)
      fprintf(ERRFILE," %3ld", t->branchlist[i]->number);
    fprintf(ERRFILE,"\n");
    t = t->succ;
  }
  fprintf(ERRFILE,"TYMELIST ENDS\n");
}  /* printtymelist */

/* The next two functions are utility function to rescale time
   when growth is non-zero */

double to_real (double tyme)
/* convert magic time to real time */
{
double result;

   /* first convert to Joe-Time */
   tyme *= -1.0;
   /* then convert to real time */
   result = -log(1.0-growth0*tyme)/growth0;
   /* now convert back to Mary-Time */
   result *= -1.0;

   /* if we're, theoretically, at infinity,
      don't calculate f(infinity), just return it! */
   if(tyme >= curtree->root->tyme) return(curtree->root->tyme);

   return(result);
} /* to_real */

double to_magic (double tyme)
/* convert real time to magic tyme */
{
double result;

   /* if we're, theoretically, at infinity,
      don't calculate f(infinity), just return it! */
   if(tyme >= curtree->root->tyme) return(curtree->root->tyme);

   /* first convert to Joe-Time */
   tyme *= -1.0;
   /* then convert to magic time */
   result = (1.0-exp(-growth0*tyme))/growth0;
   /* now convert back to Mary-Time */
   result *= -1.0;

   return(result);
} /* to_magic */

/* end time-rescalers */

/*****************************************************************
 * lengthof() returns the length of the branch "rootward" of the *
 * passed node                                                   */
double lengthof(node *p)
{
  return fabs(p->tyme - p->back->tyme);
}  /* lengthof */

/*********************************************************************
 * howlong() returns the amount of tyme passing between 2 entries in *
 * the tymelist.  Magical tyme is returned if growth != 0.           */
double howlong(tlist *t)
{
   double tlength, tyme1, tyme2;

   /* don't try this if growth rate is 0 */
   if (!growth0 || !op->growthused) return(t->age - t->eventnode->tyme);

   /* don't bother if the answer will be zero anyway */
   if (t->age==0.0) return(0.0);
   /* if we're on the theoretically infinitely long branch,
      don't calculate f(infinity), just return it! */
   if (t->age >= curtree->root->tyme) return(rootlength);

   /* otherwise transform to 'magical: fixed theta' time */

   tyme1 = to_magic(t->age);
   tyme2 = to_magic(t->eventnode->tyme);
   tlength = tyme1 - tyme2;

   return(tlength);
} /* howlong */

node *findtop(node *p)
/* findtop returns the first 'top' it finds, for a given node the _same_
   'top' will always be found */
{
  while (!p->top)
    p = p->next;
  return(p);
}  /* findtop */

void VarMalloc(node *p, boolean allokate)  
/* callocs or frees the 'x' [basewise Lnlikelihood] field of a node */
{
   long i;

   if (allokate) {
      if (p->x == NULL) {
         p->x = (phenotype)calloc(sites,sizeof(ratelike));
         p->x[0] = (ratelike)calloc(sites*categs,sizeof(sitelike));
         for(i = 1; i < sites; i++) 
            p->x[i] = p->x[0] + i * categs;
      }
   }
   else {
      if (p->x != NULL) {
         free(p->x[0]);
         free(p->x);
      }
      p->x = NULL;
   }
} /* VarMalloc */

/* "newnode" & "freenode" are paired memory managers for tree nodes.
   They maintain a linked list of unused nodes which should be faster
   to use than "calloc" & "free" (at least for recombination) */
void newnode(node **p)
{
  long i;

  i = 0;
  while (i < numnodes + 3) {   
    if (freenodes[i] != NULL) {
      *p = freenodes[i];
      freenodes[i] = NULL;
      return;
    }
    i++;
  }
  fprintf(ERRFILE,"newnode failed!\n");
  exit(-1);
}  /* newnode */

void freenode(node *p)
{
  long i;

  i = 0;
  while (i < numnodes + 3) {
     if (freenodes[i] == NULL) {
        freenodes[i] = p;
        return;
     }
     i++;
  }
  fprintf(ERRFILE,"freenode failed!\n");
  exit(-1);
}  /* freenode */
/* END of treenode allocation functions */

void newtymenode(tlist **t)
{
  *t = (tlist *)calloc(1,sizeof(tlist));
  (*t)->branchlist = (node **)calloc(numseq,sizeof(node *));
}  /* newtymenode */

void freetymenode(tlist *t)
{
  free(t->branchlist);
  free(t);
}  /* freetymenode*/

void freetymelist(tlist *t)
{
   if (t->succ != NULL) freetymelist(t->succ);
   freetymenode(t);
} /* freetymelist */

void hookup(node *p, node *q)
{
  p->back = q;
  q->back = p;
}  /* hookup */

void atr(node *p) 
/* "atr" prints out a text representation of a tree.  Pass 
   curtree->root->back for normal results */
{
  if (p->back == curtree->root) {
     fprintf(ERRFILE,"next node is root\n");
     fprintf(ERRFILE,"Node %4ld length %12.6f tyme %15.10f",
             p->back->number, lengthof(p), p->back->tyme);
     fprintf(ERRFILE," --> %4ld\n",p->number);
  }
  fprintf(ERRFILE,"Node %4ld length %12.6f tyme %23.18f -->",
         p->number, lengthof(p), p->tyme);
  if (p->top && p->back->top) fprintf(ERRFILE,"TWO TOPS HERE!!!!");
  if (!p->tip) {
     if (!p->next->top) fprintf(ERRFILE,"%4ld",p->next->back->number);
     if (!p->next->next->top) fprintf(ERRFILE,"%4ld",
         p->next->next->back->number);
     fprintf(ERRFILE,"\n");
     if (!p->next->top) atr(p->next->back);
     if (!p->next->next->top)
          atr(p->next->next->back);
  }
  else fprintf(ERRFILE,"\n");
} /* atr */

/* The next set of functions [zerocollis/onecollis/twocollis] compute
   the chance that there are 0/1/2 coalescences [respectively]
   in an interval of length "length", with "numl" active lineages, and
   "numother" inactive lineages */
double zerocollis(long numl, long numother, double length)
{
  double result;

  result = exp(-(numl * (numl - 1) + numl * numother * 2) * (length / theta0));

  if (result == 0.0 && length != 0.0) result = EPSILON;

  return(result);
}  /* zerocollis */


double onecollis(long numl, long numother, double length)
{
  double expon1, expon2, result;

  expon1 = -((numl - 1) * numother * 2 + (numl - 1) * (numl - 2)) *
	   (length / theta0);
  expon2 = -(numl * numother * 2 + numl * (numl - 1)) * (length / theta0);

  result = (numl * (numl - 1.0) / (numother * 2 + (numl - 1) * 2) *
	  (exp(expon1) - exp(expon2)));

  if (result == 0.0 && length != 0.0) result = EPSILON;

  return(result);
}  /* onecollis */


double twocollis(long numother, double length)
/* For this case "numl" is assumed to be equal to 3 */
{
  double expon1, expon2, expon3, result;

  expon1 = numother * -2 * (length / theta0);
  expon2 = -(numother * 4 + 2.0) * (length / theta0);
  expon3 = -(numother * 6 + 6.0) * (length / theta0);

  result = (6.0 / (numother + 1) *
	  (1.0 / (numother * 4 + 6) * (exp(expon1) - exp(expon3)) -
	   1.0 / (numother * 2 + 4) * (exp(expon2) - exp(expon3))));

  if (result == 0.0 && length != 0.0) result = EPSILON;

  return(result);
}  /* twocollis */
/* End of coalescence functions */

tlist *gettymenode(long target)
/* Return a pointer to the tymelist entry whose 'eventnode' has
   the number of 'target'. */
{
  tlist *t;

  if (target == curtree->root->number) {
    return(NULL);
  }
  t = curtree->tymelist;
  if (target <= numseq) return(t); /* it's a tip! */
  while (t != NULL) {
    if (t->eventnode->number == target) return(t);
    t = t->succ;
  }
  fprintf(ERRFILE,"In gettymenode, failed to find node%12ld\n", target);
  fprintf(ERRFILE,"CATASTROPHIC ERROR\n");
  exit(-1);
}  /* gettymenode */

tlist *gettyme(node *p, node *daughter1, node *daughter2, 
   node *ans)
/* Return a pointer to the tymelist entry which encompasses the time
   into which you wish to place node "p".
   tipwards/upper bound: "daughter1" and  "daughter2"
   rootward/lower bound: "ans" */
{
  boolean found;
  tlist *t, *b1, *b2, *before, *after;

  /* first establish a tipward bound on the search */
  before = curtree->tymelist;
  found = false;
  b1 = gettymenode(daughter1->number);
  b2 = gettymenode(daughter2->number);
  while (true) {
     if ((before == b1) || (before == b2)) {
        if ((found) || (b1 == b2)) break;
        found = true;
     }
     before = before->succ;
  }
  /* now establish a rootward bound on the search */
  after = gettymenode(ans->number);
  /* begin the search at the tipward bound */
  t = before;
  found = false;
  while (t != after && !found) {
    if (t->age >= p->tyme)
      found = true;
    else
      t = t->succ;
  }
  if (!found)
  /* prime^.tyme is tied with after, so goes directly in front of it */
    t = t->prev;
  return(t);
}  /* gettyme */

void inserttymelist(node *prime)
/* inserts 2 entries into the tymelist: 
   "prime" and "primans" [prime->back]. */
{
  tlist *t, *temp;
  /* d[2] is primans' daughter, d[0] and [1] parent's daughters. */
  node *parent, *q, *primans, *d[3];
  long i, j;
  d[0] = d[1] = d[2] = parent = primans = NULL; /* just to be careful */

  newtymenode(&t);
  /* find daughters and parents */
  /* this complicated mess is needed because prime must be correctly
  bounded, not by primans (which is not yet in the tymelist), but by
  the parent of primans */
  q = prime;
  j = 0;
  for (i = 1; i <= 3; i++) {
    if (q->top) {
      primans = q->back;
      if (primans->next->top) {
	parent = primans->next->back;
	d[2] = primans->next->next->back;
      } else {
	parent = primans->next->next->back;
	d[2] = primans->next->back;
      }
    } else {
      d[j] = q->back;
      j++;
    }
    q = q->next;
  }
  /* insert prime */
  t->eventnode = prime;
  temp = gettyme(prime, d[0], d[1], parent);
  t->succ = temp->succ;
  t->prev = temp;
  if (temp->succ != NULL)
    temp->succ->prev = t;
  temp->succ = t;
  if (t->succ != NULL)
    t->age = t->succ->eventnode->tyme;
  else
    t->age = t->prev->age;
  t->prev->age = t->eventnode->tyme;
  /* insert primans */
  newtymenode(&t);
  t->eventnode = primans;
  temp = gettyme(primans, prime, d[2], parent);
  t->succ = temp->succ;
  t->prev = temp;
  if (temp->succ != NULL)
    temp->succ->prev = t;
  temp->succ = t;
  if (t->succ != NULL)
    t->age = t->succ->eventnode->tyme;
  else
    t->age = t->prev->age;
  t->prev->age = t->eventnode->tyme;
}  /* inserttymelist */

void subtymelist(node *ndonor, node *nrecip)
/* takes out 2 entries from the tymelist:
   "ndonor" and "nrecip" [which must be tipward/above ndonor] */
{
  long i, j, limit;
  tlist *d, *r, *t;
  node *badbranch, *p;
  boolean found;

  badbranch = NULL; /* just to be careful */

  i = 0;
  
  r = gettymenode(nrecip->number);
  d = gettymenode(ndonor->number);
  p = nrecip;
  for (j = 1; j <= 3; j++) {
    p = p->next;
    if (p->back->number == ndonor->number)
      badbranch = p;
  }

  t = r;

  while (t != d) {
    limit = t->numbranch;
    for (i = 1; i <= limit; i++) {
      if (t->branchlist[i - 1] == badbranch) {
	j = i;
	t->numbranch--;
      }
    }
    for (i = j; i <= t->numbranch; i++)
      t->branchlist[i - 1] = t->branchlist[i];
    t = t->succ;
  }
  p = ndonor;
  p = findtop(p);
  badbranch = p;
  found = true;
  while (t != NULL && found) {
    found = false;
    for (i = 1; i <= t->numbranch; i++) {
      if (t->branchlist[i - 1] == badbranch) {
	j = i;
	t->numbranch--;
	found = true;
      }
    }
    if (found) {
      for (i = j; i <= t->numbranch; i++)
	t->branchlist[i - 1] = t->branchlist[i];
    }
    t = t->succ;
  }
  r->prev->succ = r->succ;
  r->succ->prev = r->prev;
  r->prev->age = r->age;
  freetymenode(r);
  d->prev->succ = d->succ;
  if (d->succ != NULL)
    d->succ->prev = d->prev;
  d->prev->age = d->age;
  freetymenode(d);
}  /* subtymelist */

void ltov(node *p)
/* ltov recalculates the proper "v" value of a branch, from
   the tymes at either end of the branch */
{
  p->v = 1.0 - exp(-(lengthof(p) / dna->fracchange));
  p->back->v = p->v;
}  /* ltov */

void getnums()
{
  /* input number of sequences, number of sites */
  fprintf(outfile, "\n");
  fscanf(infile, "%ld%ld", &numseq, &sites);
    fprintf(outfile, "%4ld Sequences, %4ld Sites\n", numseq, sites);
  numnodes = numseq * 2 - 1;   /*number of nodes in tree, excluding root*/
  rootnum = numnodes + 3;
  setupdata(&dna, sites, numseq);
  freenodes = (node **)calloc(2,sizeof(node *));
  /* number of internal nodes in tree is numseq-1 */
  slidenodes = (node **)calloc(numseq-1,sizeof(node *));
}  /* getnums */

/* boolcheck(), booleancheck(), numbercheck(), and readparmfile() 
   are used in reading the parameter file "parmfile" */
long boolcheck(char ch)
{
   ch = toupper(ch);
   if (ch == 'F') return 0;
   if (ch == 'T') return 1;
   return -1;
} /* boolcheck */

boolean booleancheck(char *var, char *value)
{
   long i, j, check;

   check = boolcheck(value[0]);
   if(check == -1) return false;

   for(i = 0; i < NUMBOOL; i++) {
      if(!strcmp(var,booltokens[i])) {
         if(i == 0) op->interleaved = (boolean)(check);
         if(i == 1) op->printdata = (boolean)(check);
         if(i == 2) op->progress = (boolean)(check);
         if(i == 3) op->treeprint = (boolean)(check);
         if(i == 4) {
            op->freqsfrom = (boolean)(check);
            if(!op->freqsfrom) {
               strtok(value,":");
               freqa = (double)atof(strtok(NULL,";"));
               freqc = (double)atof(strtok(NULL,";"));
               freqg = (double)atof(strtok(NULL,";"));
               freqt = (double)atof(strtok(NULL,";"));
            }
         }
         if(i == 5) {
            op->ctgry = (boolean)(check);
            if(op->ctgry) {
               strtok(value,":");
               categs = (long)atof(strtok(NULL,";"));
               rate = (double *)realloc(rate,categs*sizeof(double));
               probcat = (double *)realloc(probcat,categs*sizeof(double));
               for(j = 0; j < categs; j++) {
                  rate[j] = (double)atof(strtok(NULL,";"));
                  probcat[j] = (double)atof(strtok(NULL,";"));
               }
            }
         }
         if(i == 6) {
            op->watt = (boolean)(check);
            if (!op->watt) {
               strtok(value,":");
               theta0 = (double)atof(strtok(NULL,";"));
            }
         }
         if(i == 7) op->usertree = (boolean)(check);
         if(i == 8) {
            op->autocorr = (boolean)(check);
            if (op->autocorr) {
               strtok(value,":");
               lambda = 1.0 / (double)atof(strtok(NULL,";"));
            }
         }
         if(i == 9) op->interact = (boolean)(check);
         return true;
      }
   }
   return false;
} /* booleancheck */

boolean numbercheck(char *var, char *value)
{
   long i;

   for(i = 0; i < NUMNUMBER; i++) {
      if(!strcmp(var,numbertokens[i])) {
         if(i == 0) locus_ttratio = atof(value);
         if(i == 1) op->numchains[0] = atol(value);
         if(i == 2) op->steps[0] = atol(value);
         if(i == 3) op->increm[0] = atol(value);
         if(i == 4) op->numchains[1] = atol(value);
         if(i == 5) op->increm[1] = atol(value);
         if(i == 6) op->steps[1] = atol(value);
         if(i == 7) {growth0 = atof(value); op->growthused = true;}
         return true;
      }
   }
   return false;
} /* numbercheck */

void readparmfile()
{
   char fileline[LINESIZE],parmvar[LINESIZE],varvalue[LINESIZE];

   parmfile = fopen("parmfile","r");

   if(parmfile) {
      while(fgets(fileline, LINESIZE, parmfile) != NULL) {
         if(fileline[0] == '#') continue;
         if(!strncmp(fileline,"end",3)) break;
         strcpy(parmvar,strtok(fileline,"="));
         strcpy(varvalue,strtok(NULL,"\n"));
         /* now to process... */
         if(!booleancheck(parmvar,varvalue))
            if(!numbercheck(parmvar,varvalue)) {
               fprintf(ERRFILE,
                  "Inappropiate entry in parmfile: %s\n", fileline);
               exit(-1);
            }
      }
   } else
      if (!menu) {
         fprintf(simlog,"Parameter file (parmfile) missing\n");
         exit(-1);
      }
} /* readparmfile */
/* END parameter file read */

void getoptions()
/* interactively set options using a very basic menu */
{
  boolean done, done1, done2;
  char ch;
  long i, j;
  char input[LINESIZE];

  rate    = (double *)calloc(1,sizeof(double));
  probcat = (double *)calloc(1,sizeof(double));

  /* first some multiple rate-categories code stuff */
  op->ctgry = false;
  rate[0] = 1.0;
  probcat[0] = 1.0;
  categs = 1;
  lambda = 1.0;
  op->autocorr = false;  /* false if categs == 1 */
  /* end categories code stuff */

  fscanf(infile,"%ld",&numloci);
  ne_ratio = (double *)calloc(numloci,sizeof(double));
  for(i = 0; i < numloci; i++) ne_ratio[i] = 1.0;
  op->same_ne = true;
  mu_ratio = (double *)calloc(numloci,sizeof(double));
  for(i = 0; i < numloci; i++) mu_ratio[i] = 1.0;
  op->same_mu = true;
  theta_ratio = (double *)calloc(numloci,sizeof(double));
  
  /* default initializations */
  holding = 0;
  op->interleaved = false;
  op->printdata = false;
  op->progress = true;
  op->treeprint = false;
  locus_ttratio = 2.0;
  op->freqsfrom = true;
  op->watt = false;
  op->usertree = false;
  op->interact = false;
  theta0 = 1.0;
  op->growthused = false;
  growth0 = 0.0;
  op->numchains[0] = 10;
  op->increm[0] = 20;
  op->steps[0] = 200;
  op->numchains[1] = 2;
  op->increm[1] = 20;
  op->steps[1] = 20000;
  /* end defaults */

  readparmfile();
  fprintf(outfile, "\nFluctuating population size HMMC ML method");
  fprintf(outfile, " method, version 1.4\n\n");
  if (!readseedfile(&inseed)) {
    if (!menu) {
      fprintf(ERRFILE,"Failed to find seedfile, aborting.\n");
      exit(-1);
    } else {
      printf("Random number seed (must be odd)?\n");
      scanf("%ld%*[^\n]", &inseed);
      getchar();
    }
  }
/* now insure that the randum number seed is of the form 4n+1 */
 /*  inseed = inseed * 4 + 1; */
  for (i = 0; i <= 2; i++)
    seed[i] = 0;
  i = 0;
  for (i = 0; i <= 2; i++) {
    seed[i] = inseed & 2047;
    inseed /= 2048;
    if (inseed == 0) break;
    i++;
  }
  if (!menu) {
    for (i=0; i<numloci; i++) 
      theta_ratio[i] = ne_ratio[i] * mu_ratio[i];
    return;
  }
  putchar('\n');
  do {
    printf("\n%s", op->ansi ? "\033[2J\033[H" : "\n");
    printf("Hastings-Metropolis Markov Chain Monte Carlo");
    printf(" method, version 1.4\n\n");
    printf("INPUT/OUTPUT FORMATS\n");
    printf("  I          Input sequences interleaved?  %s\n",
           op->interleaved ? "Yes" : "No, sequential");
    printf("  E        Echo the data at start of run?  %s\n",
           op->printdata ? "Yes" : "No");
    printf("  P Print indications of progress of run?  %s\n",
           op->progress ? "Yes" : "No");
    printf("  G                Print out genealogies?  %s\n",
           op->treeprint ? "Yes" : "No");
    printf("  Q   Allow interactive design of output?  %s\n",
           op->interact ? "Yes" : "No");
    printf("MODEL PARAMETERS\n");
    printf("  T        Transition/transversion ratio:");
    printf("  %8.4f\n",locus_ttratio);
    printf("  F       Use empirical base frequencies?  %s\n",
	   op->freqsfrom ? "Yes" : "No");
    printf("  C   One category of substitution rates?");
    if (!op->ctgry || categs == 1)
      printf("  Yes\n");
    else {
      printf("  %ld categories\n", categs);
      printf("  R   Rates at adjacent sites correlated?");
      if (!op->autocorr)
	printf("  No, they are independent\n");
      else
	printf("  Yes, mean block length =%6.1f\n", 1.0 / lambda);
    }
    printf("  W      Use Watterson estimate of theta?");
    if (op->watt)
      printf("  Yes\n");
    else
      printf("  No, initial theta = %6.4f\n", theta0);
    if (op->growthused) {
       printf("  H        Population can change in size?  Yes\n");
       printf("  V  Rate of change parameter for growth:  %e\n",
              growth0);
    }
    else
       printf("  H        Population can change in size?  No\n");
    printf("  A           Allow parameters to change?");
    if (holding == 0) printf("  Yes\n");
    else if (holding == 1) printf("  No, theta fixed\n");
         else if (holding == 2) printf("  No, growth-rate fixed\n");
              else printf("  Unknown option!!!\n");
    printf("  U      Use user tree in file \"intree\" ?  %s\n",
           op->usertree ? "Yes" : "No, construct a random tree");
    if (numloci > 1) {
      printf("MULTIPLE LOCI\n");
      printf("  Z     Population size equal among loci?");
      if (op->same_ne)
         printf("  Yes\n");
      else
         printf("  No\n");
      printf("  M       Mutation rate equal among loci?");
      if (op->same_mu)
         printf("  Yes\n");
      else 
         printf("  No\n");
    }
    printf("SEARCH STRATEGY\n");
    printf("  S        Number of short chains to run?  %6ld\n", op->numchains[0]);
    if (op->numchains[0] > 0) {
       printf("  1             Short sampling increment?  %6ld\n",
	   op->increm[0]);
       printf("  2   Number of steps along short chains?  %6ld\n",
           op->steps[0]);
    }
    printf("  L         Number of long chains to run?  %6ld\n", op->numchains[1]);
    if (op->numchains[1] > 0) {
       printf("  3              Long sampling increment?  %6ld\n",
	   op->increm[1]);
       printf("  4    Number of steps along long chains?  %6ld\n",
           op->steps[1]);
    }
    printf("\n");
    printf("Are these settings correct? (type Y or the letter for one to change)\n");
    fgets(input,LINESIZE,stdin);
    ch = input[0];
    ch = toupper(ch);
    done = (ch == 'Y');
    if (!done) {
      ch = toupper(ch);
      if (strchr("AZFGHTIE1234CWVUSLPRQM",ch) != NULL){
	switch (ch) {

        case 'A':
           holding += 1;
           if (holding > 2) holding = 0;
           break;

        case 'Z':
           op->same_ne = !op->same_ne;
           if (!op->same_ne) {
              printf("Enter relative population size for each locus");
              printf(" in input order?\n");   
              i = 0;
              do {
	         scanf("%lf", &ne_ratio[i]);
                 if (ne_ratio[i] <= 0.0) {
                    printf("   Ratios must be positive, please reenter\n");
                 } else i++;
              } while (i < numloci);
           } else for(i = 0; i < numloci; i++) ne_ratio[i] = 1.0;
           break;

        case 'M':
           op->same_mu = !op->same_mu;
           if (!op->same_mu) {
              printf("Enter relative mutation rate for each locus");
              printf(" in input order?\n");   
              i = 0;
              do {
	         scanf("%lf", &mu_ratio[i]);
                 if (mu_ratio[i] <= 0.0) {
                    printf("   Ratios must be positive, please reenter\n");
                 } else i++;
              } while (i < numloci);
           } else for(i = 0; i < numloci; i++) mu_ratio[i] = 1.0;
           break;

	case 'S':
	  do {
	    printf("How many Short Chains?\n");
            fgets(input,LINESIZE,stdin);
            op->numchains[0] = atoi(input);
	    if (op->numchains[0] < 0)
	      printf("Must be non-negative\n");
	  } while (op->numchains[0] < 0);
	  break;

        case 'H':
          op->growthused = !op->growthused;
          break;

        case 'V':
          if (!op->growthused) break;
          printf("What parameter value for growth?\n");
          fgets(input,LINESIZE,stdin);
          growth0 = atof(input);
          break;

	case 'L':
	  do {
	    printf("How many Long Chains?\n");
            fgets(input,LINESIZE,stdin);
            op->numchains[1] = atoi(input);
	    if (op->numchains[1] < 0)
	      printf("Must be non-negative\n");
	  } while (op->numchains[1] < 0);
	  break;

	case 'C':
	  op->ctgry = !op->ctgry;
	  if (!op->ctgry)
	    op->autocorr = false;
	  if (op->ctgry) {
	    do {
	      printf("Number of categories ?");
              fgets(input,LINESIZE,stdin);
              categs = atoi(input);
	    } while (categs < 1);
	    free(rate);
	    free(probcat);
	    printf("Rate for each category? (use a space to");
	    printf(" separate)\n");
	    rate    = (double *)calloc(categs,sizeof(double));
	    probcat = (double *)calloc(categs,sizeof(double));
	    for (j = 0; j < categs; j++)
	      scanf("%lf*[^\n]", &rate[j]);

	    getchar();
	    do {
	      printf("Probability for each category?");
	      printf(" (use a space to separate)\n");
	      for (j = 0; j < categs; j++)
		scanf("%lf", &probcat[j]);
	      scanf("%*[^\n]");
	      getchar();
	      done2 = true;
	      probsum = 0.0;
	      for (j = 0; j < categs; j++)
		probsum += probcat[j];
	      if (fabs(1.0 - probsum) > 0.001) {
		done2 = false;
		printf("Probabilities must add up to");
		printf(" 1.0, plus or minus 0.001.\n");
	      }
	    } while (!done2);
	  }
	  break;

	case 'R':
	  op->autocorr = !op->autocorr;
	  if (op->autocorr) {
	    do {
	      printf("Mean block length of sites having the same ");
	      printf("rate (greater than 1)?\n");
	      scanf("%lf%*[^\n]", &lambda);
	      getchar();
	    } while (lambda <= 1.0);
	    lambda = 1.0 / lambda;
	  }
	  break;

	case 'F':
	  op->freqsfrom = !op->freqsfrom;
	  if (!op->freqsfrom) {
	    printf("Base frequencies for A, C, G, T/U (use blanks to separate)?\n");
	    scanf("%lf%lf%lf%lf", &freqa, &freqc, &freqg, &freqt);
            scanf("%*[^\n]");
	  }
	  break;

	case 'T':
	  do {
	    printf("Transition/transversion ratio?\n");
            fgets(input,LINESIZE,stdin);
            locus_ttratio = atof(input);
	  } while (locus_ttratio < 0.0);
	  break;

	case 'I':
	  op->interleaved = !op->interleaved;
	  break;

	case 'W':
	  op->watt = !op->watt;
	  if (!op->watt) {
	    do {
	      printf("Initial theta estimate?\n");
              fgets(input,LINESIZE,stdin);
              theta0 = atof(input);
	    } while (theta0 <= 0.0);
	  }
	  break;

        case 'U':
          op->usertree = !op->usertree;
          break;

	case 'E':
	  op->printdata = !op->printdata;
	  break;

	case 'P':
	  op->progress = !op->progress;
	  break;

	case 'G':
	  op->treeprint = !op->treeprint;
	  break;

        case 'Q':
          op->interact = !op->interact;
          break;

	case '1':
	  done1 = false;
	  while (!done1) {
	    printf("How often to sample trees?\n");
            fgets(input,LINESIZE,stdin);
            op->increm[0] = atoi(input);
	    if (op->increm[0] > 0)
	      done1 = true;
	    else
	      printf("Must be positive\n");
	  }
	  break;

	case '2':
	  done1 = false;
	  while (!done1) {
	    printf("How many short steps?\n");
            fgets(input,LINESIZE,stdin);
            op->steps[0] = atoi(input);
	    if (op->steps[0] > 0)
	      done1 = true;
	    else
	      printf("Must be a positive integer\n");
	  }
	  break;

	case '3':
	  done1 = false;
	  while (!done1) {
	    printf("How often to sample trees?\n");
            fgets(input,LINESIZE,stdin);
            op->increm[1] = atoi(input);
	    if (op->increm[1] > 0)
	      done1 = true;
	    else
	      printf("Must be positive\n");
	  }
	  break;

	case '4':
	  done1 = false;
	  while (!done1) {
	    printf("How many long steps?\n");
            fgets(input,LINESIZE,stdin);
            op->steps[1] = atoi(input);
	    if (op->steps[1] > 0)
	      done1 = true;
	    else
	      printf("Must be a positive integer\n");
	  }
	  break;

        default:
          fprintf(stderr,"Impossible option %c detected!\n",ch);
          break;     

        }
      } else
	printf("Not a possible option!\n");
    }
  } while (!done);
  for (i=0; i<numloci; i++) {
    theta_ratio[i] = ne_ratio[i] * mu_ratio[i];
  }
}  /* getoptions */

void firstinit()
/* initialization for things that are recorded over multiple loci */
{
  long i;

  totchains = op->numchains[0] + op->numchains[1];

  numtrees = MAX(op->steps[0]/op->increm[0],op->steps[1]/op->increm[1]);

  sametree = (boolean **)calloc(numloci,sizeof(boolean *));
  sametree[0] = (boolean *)calloc(numloci*numtrees,sizeof(boolean));
  for(i = 1; i < numloci; i++)
     sametree[i] = sametree[0] + i*numtrees;

  model_alloc();

}  /* firstinit */

void locusinit()
/* initialization of things that are specific to one locus */ 
{
long i, j;

  getnums();

  if ((op->increm[0] < 0) || (op->increm[1] < 0)) {
     fprintf(ERRFILE,"Error in input sampling increment");
     fprintf(ERRFILE," increment set to 10\n");
     if (op->increm[0] < 0)
        op->increm[0] = 10;
     if (op->increm[1] < 0)
        op->increm[1] = 10;
  }

  for (i = 0; i < 1+op->numchains[1]; i++)
     for (j = 0; j < numtrees; j++) {
       sum[locus][i][j].kk    =
         (long *)calloc(numseq,sizeof(long));
       sum[locus][i][j].kend  =
         (double *)calloc(numseq,sizeof(double));
  }

} /* locusinit */

void inputoptions()
{
  char ch;
  long i, extranum;

  category = (long *)calloc(sites,sizeof(long));
  weight   = (long *)calloc(sites,sizeof(long));

  for (i = 0; i < sites; i++)
    category[i] = 1,
    weight[i] = 1;
  extranum = 0;
  while (!(eoln(infile))) {
    ch = getc(infile);
    if (ch == '\n')
      ch = ' ';
    ch = isupper(ch) ? ch : toupper(ch);
    if (ch == 'C')
      extranum++;
    else if (ch != ' ') {
      printf("BAD OPTION CHARACTER: %c\n", ch);
      exit(-1);
    }
  }
  fscanf(infile, "%*[^\n]");
  getc(infile);
  for (i = 1; i <= extranum; i++) {
    ch = getc(infile);
    if (ch == '\n')
      ch = ' ';
    ch = isupper(ch) ? ch : toupper(ch);
    if (ch != 'W'){
      printf("ERROR: INCORRECT AUXILIARY OPTIONS LINE WHICH STARTS WITH %c\n",
	     ch);
      exit(-1);
    }
  }
  if (categs <= 1)
    return;
  fprintf(outfile, "\nSite category   Rate of change  Probability\n");
  for (i = 1; i <= categs; i++)
    fprintf(outfile, "%12ld%13.3f%13.3f\n", i, rate[i - 1], probcat[i - 1]);
  putc('\n', outfile);
}  /* inputoptions */

void setuptree()
{
  long i, j;
  node *p, *q;

  curtree = (tree *)calloc(1,sizeof(tree));
  curtree->nodep = (node **)calloc(numnodes + 3,sizeof(node *));
  alogf = (rlrec *)calloc(1,sizeof(rlrec));
  alogf->val = (double *)calloc(sites,sizeof(double));
  newtymenode(&curtree->tymelist);
  for (i = 0; i < 2; i++)
    freenodes[i] = NULL;

  /* make the tips first */
  for (i = 0; i < numseq; i++) {
    curtree->nodep[i] = (node *)calloc(1,sizeof(node));
    curtree->nodep[i]->tip = true;
    curtree->nodep[i]->number = i + 1;
    VarMalloc(curtree->nodep[i],true);
  }

  /* now make the interior nodes and the freenodes, but not the root */
  for (i = numseq; i < numnodes+2; i++) {
    q = NULL;
    for (j = 0; j < 3; j++) {
      p = (node *)calloc(1,sizeof(node));
      if (p == NULL) {
        fprintf(ERRFILE,"tree setup fails, allocate more space\n");
        exit(-1);
      }
      p->number = i + 1;
      p->tip = false;
     /* initialize the following pointers to NULL
        space will be allocated as appropiate in procedure
        orient() */
      p->x = NULL;
     /* end NULL assignments */
      p->next = q;
      q = p;
    }
    p->next->next->next = p; /* close up the chain into a loop */
    curtree->nodep[i] = p;
    if (i >= numnodes) {
      freenodes[i - numnodes] = p;
      /* do memory allocation for initial freenodes now, orient
         only covers nodes in initial tree */
      p->top = true;
      VarMalloc(p,true);
    }
  }

  /* now make the root */
  curtree->nodep[rootnum - 1] = (node *)calloc(1,sizeof(node));
  curtree->nodep[rootnum - 1]->tip = true;
  curtree->nodep[rootnum - 1]->number = rootnum;
  VarMalloc(curtree->nodep[rootnum - 1],true);
  strncpy(curtree->nodep[rootnum-1]->nayme,"ROOT",4);
  /* guarantee that the root node contributes nothing to the likelihood
     of a tree (since its supposed to be at the end of a theoretically
     infinite root branch) */
  for (i = 0; i < sites; i++) {
    for (j = 0; j < categs; j++) {
      curtree->nodep[rootnum - 1]->x[i][j][baseA] = 1.0;
      curtree->nodep[rootnum - 1]->x[i][j][baseC] = 1.0;
      curtree->nodep[rootnum - 1]->x[i][j][baseG] = 1.0;
      curtree->nodep[rootnum - 1]->x[i][j][baseT] = 1.0;
    }
  }
  curtree->likelihood = NEGMAX;
}  /* setuptree */

void freetree()
/* we do not free the following arrays:
      sum, theti, lntheti, fixed, numout, ne_ratio, mu_ratio, theta_ratio */
{
   long i;
   node *p;

   free(alogf->val);
   free(alogf);
   free(category);
   free(weight);
   freetymelist(curtree->tymelist);
   /* free the tips */
   for(i = 0; i < numseq; i++) {
      VarMalloc(curtree->nodep[i],false);
      free(curtree->nodep[i]);
   }
   /* free internal nodes including slidenodes */
   for(i = numseq; i < numnodes + 2; i++) {
      p = curtree->nodep[i];
      VarMalloc(p,false);
      VarMalloc(p->next,false);
      VarMalloc(p->next->next,false);
      free(p->next->next);
      free(p->next);
      free(p);
   }
   free(slidenodes);
   free(freenodes);
   /* free the root node */
   VarMalloc(curtree->nodep[rootnum-1],false);
   free(curtree->nodep[rootnum-1]);
   /* free the tree */
   free(curtree->nodep);
   /* free the aliases */
   free(siteptr);
   /* free the working arrays */
   free(weightrat);
   free(tbl);
   free(contribution[0]);
   free(contribution);
} /* freetree */

boolean sitecompare(long site1, long site2)
{
   long i;

   for(i = 0; i < numseq; i++)
      if(dna->seqs[i][site1] != dna->seqs[i][site2])
         return false;

   return true;
} /* sitecompare */

void makesiteptr()
/* create the siteptr array: -1 means do a likelihood calculation
   for this site; a number >= 0 means use the site with that number
   for x (likelihood) values */
{
   long whichsite, i;
   boolean found;

   siteptr = (long *)calloc(sites,sizeof(long));

   siteptr[0] = -1;

   for(whichsite = 1; whichsite < sites; whichsite++) {
      found = false;
      for(i = whichsite - 1; i >= 0; i--) {
         if(sitecompare(i,whichsite)) {
            siteptr[whichsite] = i;
            found = true;
            break;
         }
      }
      if (found) continue;
      siteptr[whichsite] = -1;
   }
} /* makesiteptr */

void getinput()
{
  /* reads the input data */
  inputoptions();
  dna->freqa = freqa;
  dna->freqc = freqc;
  dna->freqg = freqg;
  dna->freqt = freqt;
  setuptree();
  getdata(curtree, dna, op, infile, outfile);
  makesiteptr();
  makevalues(dna, categs, curtree);
  if (op->freqsfrom) {
    empiricalfreqs(curtree, dna, weight);
  }
  getbasefreqs(dna, op, locus_ttratio, outfile);
}  /* getinput */

double watterson()
{
  /* estimate theta using method of Watterson */
  long i, j, kn;
  boolean varies;
  double watter;

  kn = 0;
  for (i = 0; i < sites; i++) {
    varies = false;
    for (j = 1; j < numseq; j++) {
      if (dna->seqs[j][i] != dna->seqs[0][i])
	varies = true;
    }
    if (varies)
      kn++;
  }
  watter = 0.0;
  if (kn > 0) {
    for (i = 1; i < numseq; i++)
      watter += 1.0 / i;
    watter = kn / (sites * watter);
    return watter;
  }
  fprintf(outfile, "Warning:  There are no variable sites");
  fprintf(outfile, " in this data set.\n\n");
  if (menu) printf("Warning:  There are no variable sites in this data set.\n");
  else {
     fprintf(simlog, "Warning:  There are no variable sites");
     fprintf(simlog, " in this data set.\n\n");
  }
  exit(-1);
}  /* watterson */

void orient(node *p)
{
  tlist *t, *u;

  t = curtree->tymelist;

  if (p->tip) {
    p->top = true;
    p->tyme = 0.0;
    t->eventnode = p;
    t->branchlist[p->number - 1] = p;
    return;
  }

  p->top = true;
  curtree->nodep[p->number-1] = p;  /* insure that curtree->nodep points
                                    to nodes with info */

 /* since p is a top nodelet, it needs to actually store
    likelihood information, x is a NULL pointer
    in all other non-tip nodelets */
  VarMalloc(p,true);

  p->next->top = false;
  p->next->next->top = false;

  orient(p->next->back);
  orient(p->next->next->back);
  p->tyme = p->next->length + p->next->back->tyme;
  p->next->tyme = p->tyme;
  p->next->next->tyme = p->tyme;
  if (p->number == curtree->root->back->number) {
    p->back->top = false;
    p->back->tyme = rootlength;
  }
  newtymenode(&u);
  u->eventnode = p;
  while (t != NULL) {
    if (u->eventnode->tyme < t->eventnode->tyme) {
      u->prev = t->prev;
      t->prev = u;
      u->succ = t;
      u->prev->succ = u;
      break;
    }
    if (t->succ != NULL)
      t = t->succ;
    else {
      t->succ = u;
      u->prev = t;
      u->succ = NULL;
      break;
    }
  }
}  /* orient */

void finishsetup(node *p)
{
  if (p->tip) {
    ltov(p);
    return;
  }
  ltov(p);
  finishsetup(p->next->back);
  finishsetup(p->next->next->back);
  return;
} /* finishsetup */

void initbranchlist()
{
  tlist *t;
  node *p, *q;
  long i, j, k, n;

  t = curtree->tymelist;
  n = numseq;
  t->numbranch = n;
  t->age = t->succ->eventnode->tyme;
  t = t->succ;
  for (i = 0; i < (numnodes - numseq); i++) {
    /* for each interior node, do...assumes at least 3 tips */
    n--;
    t->numbranch = n;
    if (n == 1)
      t->age = t->eventnode->tyme + rootlength;
    else
      t->age = t->succ->eventnode->tyme;
    p = t->eventnode->next->back;
    q = t->eventnode->next->next->back;
    k = 0;
    for (j = 0; j < t->prev->numbranch ; j++) {
      /* for the number of branches above the coalescent node, do...*/
      if (t->prev->branchlist[j] != p && t->prev->branchlist[j] != q) {
	t->branchlist[k] = t->prev->branchlist[j];
	k++;
      }
    }
    t->branchlist[t->numbranch - 1] = t->eventnode;
    t = t->succ;
  }
  /* initialize the slidelist, assume that curtree->nodep[numseq] through
     curtree->nodep[numnodes-1] point to all the interior nodes of the
     initial tree (one node will be curtree->root->back and so ineligible
     to be slid) */
  i = 0;
  slidenum = 0;
  for (j = numseq; j < numnodes; j++) {
    if (!(curtree->nodep[j]->back->number == rootnum)) {
      slidenodes[i] = curtree->nodep[j];
      i++;
      slidenum++;
    }
  }
}  /* initbranchlist */

void inittable()
{
  long i;
  tbl = (valrec *)calloc(categs,sizeof(valrec));
  /* Define a lookup table. Precompute values and store them in a table */
  for (i = 0; i < categs; i++) {
    tbl[i].rat_xi = rate[i] * dna->xi;
    tbl[i].rat_xv = rate[i] * dna->xv;
  }
}  /* inittable */

void initweightrat()
{
  long i;
  weightrat = (double *)calloc(sites,sizeof(double));
  sumweightrat = 0.0;
  for (i = 0; i < sites; i++) {
    weightrat[i] = weight[i] * rate[category[i] - 1];
    sumweightrat += weightrat[i];
  }
}  /* initweightrat */

void treeout(node *p, long s, FILE **usefile)
{
  /* write out file with representation of final tree */
  long i, n, w;
  char c;
  double x;

  if (p->tip) {
    n = 0;
    for (i = 1; i <= NMLNGTH; i++) {
      if (p->nayme[i - 1] != ' ')
	n = i;
    }
    for (i = 0; i < n; i++) {
      c = p->nayme[i];
      if (c == ' ')
	c = '_';
      putc(c, *usefile);
    }
    col += n;
  } else {
    putc('(', *usefile);
    col++;
    treeout(p->next->back, s, usefile);
    putc(',', *usefile);
    col++;
    if (col > 45) {
      putc('\n', *usefile);
      col = 0;
    }
    treeout(p->next->next->back, s, usefile);
    putc(')', *usefile);
    col++;
  }
  if (p->v >= 1.0)
    x = -1.0;
  else
    x = lengthof(p);
  if (x > 0.0)
    w = (long)(0.4343 * log(x));
  else if (x == 0.0)
    w = 0;
  else
    w = (long)(0.4343 * log(-x)) + 1;
  if (w < 0)
    w = 0;
  if (p == curtree->root->back)
    putc(';', *usefile);
  else {
    fprintf(*usefile, ":%*.10f", (int)(w + 7), x);
    col += w + 8;
  }
}  /* treeout */

void evaluate(tree *tr, boolean first)
{
  double temp, sum2, sumc, sumterm, lterm, termcheck;
  contribarr like, nulike, term, clai;
  long i, j, k;
  node *p;
  sitelike x1;

  like   = (double *)calloc(categs,sizeof(double));
  nulike = (double *)calloc(categs,sizeof(double));
  term   = (double *)calloc(categs,sizeof(double));
  clai   = (double *)calloc(categs,sizeof(double));

  temp = 0.0;
  p = tr->root->back;

  for (i = 0; i < sites; i++) {
    termcheck = 0.0;
    for (j = 0; j < categs; j++) {
       memcpy((void *)x1, (void *)p->x[i][j], sizeof(sitelike));
       term[j] = dna->freqa * x1[baseA] + dna->freqc * x1[baseC] +
  	      dna->freqg * x1[baseG] + dna->freqt * x1[baseT];
       termcheck += term[j];
    }
    if (!termcheck) {
       fprintf(ERRFILE,"Encountered tree incompatible with data\n");
       if(first) {
          fprintf(ERRFILE,"starting tree needs to be legal\n");
          exit(-1);
       }
       curtree->likelihood = NEGMAX;
       return;
    }
    sumterm = 0.0;
    for (j = 0; j < categs; j++)
      sumterm += probcat[j] * term[j];
    lterm = log(sumterm);
    for (j = 0; j < categs; j++)
      clai[j] = term[j] / sumterm;
    memcpy((void *)contribution[i], (void *)clai, categs*sizeof(double));
    if (!op->autocorr)
      alogf->val[i] = lterm;
    temp += weight[i] * lterm;
  }
  for (j = 0; j < categs; j++)
    like[j] = 1.0;
  for (i = 0; i < sites; i++) {
    sumc = 0.0;
    for (k = 1; k <= categs; k++)
      sumc += probcat[k - 1] * like[k - 1];
    sumc *= lambda;
    memcpy((void *)clai, (void *)contribution[i], categs*sizeof(double));
    for (j = 0; j < categs; j++)
      nulike[j] = ((1.0 - lambda) * like[j] + sumc) * clai[j];
    memcpy((void *)like, (void *)nulike, categs*sizeof(double));
  }
  sum2 = 0.0;
  for (i = 0; i < categs; i++)
    sum2 += probcat[i] * like[i];
  temp += log(sum2);
  curtree->likelihood = temp;
  free(like);
  free(nulike);
  free(term);
  free(clai);
}  /* evaluate */

void nuview(node *p)
{
  long i, j;
  double w1, w2, lw1, lw2, yy1, yy2, ww1zz1, vv1zz1, ww2zz2, vv2zz2,
	 vv1zz1_sumr1, vv2zz2_sumr2, vv1zz1_sumy1, vv2zz2_sumy2, sum1, sum2,
	 sumr1, sumr2, sumy1, sumy2;
  node *q, *r;
  sitelike xx1, xx2, xx3;

  q = p->next->back;
  r = p->next->next->back;

  w1 = 1.0 - q->v;
  w2 = 1.0 - r->v;
  if (w1 > 0.0) {
    lw1 = log(w1);
    for (i = 0; i < categs; i++) {
      tbl[i].ww1 = exp(tbl[i].rat_xi * lw1);
      tbl[i].zz1 = exp(tbl[i].rat_xv * lw1);
      tbl[i].ww1zz1 = tbl[i].ww1 * tbl[i].zz1;
      tbl[i].vv1zz1 = (1.0 - tbl[i].ww1) * tbl[i].zz1;
    }
  }
  if (w2 > 0.0) {
    lw2 = log(w2);
    for (i = 0; i < categs; i++) {
      tbl[i].ww2 = exp(tbl[i].rat_xi * lw2);
      tbl[i].zz2 = exp(tbl[i].rat_xv * lw2);
      tbl[i].ww2zz2 = tbl[i].ww2 * tbl[i].zz2;
      tbl[i].vv2zz2 = (1.0 - tbl[i].ww2) * tbl[i].zz2;
    }
  }
  for (i = 0; i < sites; i++) {
    for (j = 0; j < categs; j++) {
       if(siteptr[i] == -1) { /* if we need to calculate this site */
          if (w1 <= 0.0) {
             ww1zz1 = 0.0;
             vv1zz1 = 0.0;
             yy1 = 1.0;
          } else {
             ww1zz1 = tbl[j].ww1zz1;
             vv1zz1 = tbl[j].vv1zz1;
             yy1 = 1.0 - tbl[j].zz1;
          }
          if (w2 <= 0.0) {
             ww2zz2 = 0.0;
             vv2zz2 = 0.0;
             yy2 = 1.0;
          } else {
             ww2zz2 = tbl[j].ww2zz2;
             vv2zz2 = tbl[j].vv2zz2;
             yy2 = 1.0 - tbl[j].zz2;
          }
          memcpy((void *)xx1, (void *)q->x[i][j], sizeof(sitelike));
          memcpy((void *)xx2, (void *)r->x[i][j], sizeof(sitelike));
          sum1 = yy1 * (dna->freqa * xx1[baseA] + dna->freqc * xx1[baseC] +
    	    dna->freqg * xx1[baseG] + dna->freqt * xx1[baseT]);
          sum2 = yy2 * (dna->freqa * xx2[baseA] + dna->freqc * xx2[baseC] +
    	    dna->freqg * xx2[baseG] + dna->freqt * xx2[baseT]);
          sumr1 = dna->freqar * xx1[baseA] + dna->freqgr * xx1[baseG];
          sumr2 = dna->freqar * xx2[baseA] + dna->freqgr * xx2[baseG];
          sumy1 = dna->freqcy * xx1[baseC] + dna->freqty * xx1[baseT];
          sumy2 = dna->freqcy * xx2[baseC] + dna->freqty * xx2[baseT];
          vv1zz1_sumr1 = vv1zz1 * sumr1;
          vv2zz2_sumr2 = vv2zz2 * sumr2;
          vv1zz1_sumy1 = vv1zz1 * sumy1;
          vv2zz2_sumy2 = vv2zz2 * sumy2;
          xx3[baseA] = (sum1 + ww1zz1 * xx1[baseA] + vv1zz1_sumr1) *
            (sum2 + ww2zz2 * xx2[baseA] + vv2zz2_sumr2);
          xx3[baseC] = (sum1 + ww1zz1 * xx1[baseC] + vv1zz1_sumy1) *
            (sum2 + ww2zz2 * xx2[baseC] + vv2zz2_sumy2);
          xx3[baseG] = (sum1 + ww1zz1 * xx1[baseG] + vv1zz1_sumr1) *
            (sum2 + ww2zz2 * xx2[baseG] + vv2zz2_sumr2);
          xx3[baseT] = (sum1 + ww1zz1 * xx1[baseT] + vv1zz1_sumy1) *
            (sum2 + ww2zz2 * xx2[baseT] + vv2zz2_sumy2);
          memcpy((void *)p->x[i][j], (void *)xx3, sizeof(sitelike));
       }
       else {
      /* this site is just like site #(siteptr[i]), use its values */
          memcpy((void *)p->x[i][j], (void *)p->x[siteptr[i]][j], sizeof(sitelike));
       }
    }
  }
}  /* nuview */

void update(node *p)
{
  if (!p->tip)
    nuview(p);
}  /* update */

void smooth(node *p)
{
  if (!p->tip) {
    if (!p->next->top)
      smooth(p->next->back);
    if (!p->next->next->top)
      smooth(p->next->next->back);
  }
  update(p);
}  /* smooth */

void localsmooth(node *p)
{
  if (p->number != curtree->root->number) {
     p = findtop(p);
     nuview(p);
  }
  if (p->number != curtree->root->number)
     localsmooth(p->back);
} /* localsmooth */

boolean testratio()
{
  /* decide to accept or not */
  double test, x;

  if(curtree->likelihood == NEGMAX)
     return false;
  test = curtree->likelihood - oldlikelihood;
  if (test >= 1.0)
    return true;
  else {
    x = log(randum());
    if (test >= x)
      return true;
    else
      return false;
  }
}  /* testratio */

void seekch(char c) /* use only in reading file intree! */
{
  if (gch == c)
    return;
  do {
    if (eoln(intree)) {
      fscanf(intree, "%*[^\n]");
      getc(intree);
    }
    gch = getc(intree);
    if (gch == '\n')
      gch = ' ';
  } while (gch != c);
}  /* seekch */

void getch(char *c) /* use only in reading file intree! */
{
  /* get next nonblank character */
  do {
    if (eoln(intree)) {
      fscanf(intree, "%*[^\n]");
      getc(intree);
    }
    *c = getc(intree);
    if (*c == '\n')
      *c = ' ';
  } while (*c == ' ');
}  /* getch */

void processlength(node *p)
{
  long digit;
  double valyew, divisor;
  boolean pointread;
 
  pointread = false;
  valyew = 0.0;
  divisor = 1.0;
  getch(&gch);
  digit = gch - '0';
  while (((unsigned long)digit <= 9) || gch == '.'){
    if (gch == '.')
      pointread = true;
    else {
      valyew = valyew * 10.0 + digit;
      if (pointread)
	divisor *= 10.0;
    }
    getch(&gch);
    digit = gch - '0';
  }
  p->length = valyew / divisor;
  p->back->length = p->length;
}  /* processlength */

void addelement(node *p, long *nextnode)
{
  node *q;
  long i, n;
  boolean found;
  char str[NMLNGTH];

  getch(&gch);
  if (gch == '(') {
    (*nextnode)++;
    q = curtree->nodep[(*nextnode) - 1];
    hookup(p, q);
    addelement(q->next,nextnode);
    seekch(',');
    addelement(q->next->next, nextnode);
    seekch(')');
    getch(&gch);
  } else {
    for (i = 0; i < NMLNGTH; i++)
      str[i] = ' ';
    n = 1;
    do {
      if (gch == '_')
	gch = ' ';
      str[n - 1] = gch;
      if (eoln(intree)) {
	fscanf(intree, "%*[^\n]");
	getc(intree);
      }
      gch = getc(intree);
      if (gch == '\n')
	gch = ' ';
      n++;
    } while (gch != ':' && gch != ',' && gch != ')' && n <= NMLNGTH);
    n = 1;
    do {
      found = true;
      for (i = 0; i < NMLNGTH; i++)
	found = (found && str[i] == curtree->nodep[n - 1]->nayme[i]);
      if (!found)
	n++;
    } while (!(n > numseq || found));
    if (n > numseq) {
      printf("Cannot find sequence: ");
      for (i = 0; i < NMLNGTH; i++)
	putchar(str[i]);
      putchar('\n');
    }
    hookup(curtree->nodep[n - 1], p);
  }
  if (gch == ':')
    processlength(p);
}  /* addelement */

void treeread()
{
  long nextnode;
  node *p;

  curtree->root = curtree->nodep[rootnum - 1];
  getch(&gch);
  if (gch == '(') {
    nextnode = numseq + 1;
    p = curtree->nodep[nextnode - 1];
    addelement(p, &nextnode);
    seekch(',');
    addelement(p->next, &nextnode);
    hookup(p->next->next, curtree->nodep[rootnum - 1]);
    p->next->next->length = rootlength;
    curtree->nodep[rootnum - 1]->length = p->next->next->length;
    ltov(curtree->nodep[rootnum - 1]);
  }
  fscanf(intree, "%*[^\n]");
  getc(intree);
}  /* treeread */

void treevaluate()
{

  smooth(curtree->root->back);
  smooth(curtree->root);
  evaluate(curtree,true);
}  /* treevaluate */

void localevaluate(node *p, node *pansdaught)
/* routine assumes that p points to the only 'top' nodelet
   in node 'p' */
{

  /* first update all daughters and p itself */
  if (!p->next->back->tip)
     nuview(p->next->back);
  if (!p->next->next->back->tip)
     nuview(p->next->next->back);
  nuview(p);
  if (!pansdaught->tip)
      nuview(pansdaught);
  /* now update the rest of the tree */
  localsmooth(p->back);
  evaluate(curtree,false);
} /* localevaluate */

void copynode(node *source, node *target)
/* copies source node to target node */
{
  long i, j;
  
  for (i = 1; i <= 3; i++) {
    /* NEVER! target->next := source->next; */
    target->back = source->back;
    target->tip = source->tip;
    /* but NOT target->number := source->number; */
    if (source->x != NULL) {
       VarMalloc(target,true);
       for (j = 0; j < sites; j++) {
          memcpy((void *)target->x[j], (void *)source->x[j], categs*sizeof(sitelike));
       }
    }
    else
       VarMalloc(target,false);
    memcpy((void *)target->nayme, (void *)source->nayme, sizeof(naym));
    target->top = source->top;
    target->v = source->v;
    target->tyme = source->tyme;
    target->length = source->length;
    source = source->next;
    target = target->next;
  }
}  /* copynode */

/* joinnode and constructree are used for constructing a rather bad
   starting tree if the user doesn't provide one */
void joinnode(double length, node *p, node *q)
{
   hookup(p,q);
   p->length = length;
   q->length = length;
   ltov(p);
} /* joinnode */

void constructtree(long numtips, double branchlength)
{
   long i, j, nextnode;
   double height;
   node *p, *q;
 
   curtree->root = curtree->nodep[rootnum - 1];
   nextnode = numseq;
   p = curtree->root;
   q = curtree->nodep[nextnode];
 
   p->back = q;
   q->back = p;
   p->length = rootlength;
   q->length = rootlength;
   ltov(p);
 
   height = (numtips - 1) * branchlength;
   p->tyme = rootlength + height;
   for (i = 0; i < numtips - 1; i++) {
      p = curtree->nodep[i];
      q = curtree->nodep[nextnode]->next;
      joinnode(height,p,q);
      q = q->next;
      if (i != numtips-2) {
         nextnode++;
         p = curtree->nodep[nextnode];
         joinnode(branchlength,p,q);
         height -= branchlength;
      }
      else {
         p = curtree->nodep[numtips - 1];
         joinnode(height,p,q);
      }
      for (j = 0; j < 3; j++)
         q->tyme = height;
   }
} /* constructtree */
/* End bad starting tree construction */

void updateslide(node *p, boolean wanted)
/* pass me FALSE only if sure that the node is invalid */
{
  boolean valid, found;
  node *q;
  long j, k;

  k = 0; /* just to be careful */

  valid = true;
  q = p;
  if (!wanted)
    valid = false;
  else {
    if (q->tip)
      valid = false;
    else {
      q =findtop(q);
      if (q->back->tip)
	valid = false;
    }
  }
  found = false;
  j = 1;
  while (!found && j <= slidenum) {
    if (slidenodes[j - 1]->number == p->number) {
      found = true;
      k = j;
    }
    j++;
  }
  if (valid && !found) {
    slidenum++;
    slidenodes[slidenum - 1] = p;
  }
  if (valid || !found)
    return;
  while (k < slidenum) {
    slidenodes[k - 1] = slidenodes[k];
    k++;
  }
  slidenum--;
}  /* updateslide */

void rebuildbranch()
{
  tlist *t;
  node *p;
  long i, k;
  boolean done;

  t = curtree->tymelist->succ;
  done = false;
  do {
    if (t->succ == NULL) done = true;
    p = t->eventnode;
    k = 1;
    p = findtop(p);
    for (i = 0; i < t->prev->numbranch; i++) {
      if (t->prev->branchlist[i] != p->next->back &&
	  t->prev->branchlist[i] != p->next->next->back) {
	t->branchlist[k - 1] = t->prev->branchlist[i];
	k++;
      }
    }
    t->numbranch = t->prev->numbranch - 1;
    t->branchlist[t->numbranch - 1] = p;
    t = t->succ;
  } while (!done);
}  /* rebuildbranch */

/****************************************************************
 * setlength() returns true if it succeeds in setting a length; *
 * false otherwise                                              */
boolean setlength(long numl, long numother, double tstart,
   double tlength, node *p)
{
  double x, e1, r;

  r = randum();
  e1 = (numl - 1.0) * 2 + numother * 2;
  x = -(theta0 / e1) *
      log(1 - r * (1 - exp(-(e1 * tlength / theta0))));
  if ((unsigned)x > tlength) {
     fprintf(ERRFILE,"disaster in setlength\n");
     return(false);
  }
  if (!growth0 || !op->growthused) p->tyme = tstart + x;
  else {
     tstart = to_magic(tstart);
     x = tstart + x;
     /* if the time is now impossible to big... */
     if (x*growth0 < -1.0) return(false);
     else p->tyme = to_real(x);
  }

  return(true);
}  /* setlength */

#define TRY_SOLVE 20 /* number of times to iteratively solve for
                        length of new branch */
/*********************************************************************
 * setlength2() returns true if it succeeds in setting both lengths; *
 * false otherwise                                                   */
boolean setlength2(long numother, double tstart, double tlength,
   node *p, node *q)
{
  long i;
  double x, xmin, xmax, r, xnew, e1, e2, norm;

  r = randum();
  e1 = exp(numother * -2 * tlength / theta0);
  e2 = exp(-((numother * 2 + 2) * tlength / theta0));
  norm = -3 * e1 / ((numother + 1) * twocollis(numother, tlength));
  xmin = 0.0;
  xmax = tlength;
  for (i = 0; i < TRY_SOLVE; i++) {
    x = (xmax + xmin) / 2.0;
    xnew = norm *
        (1.0 / (numother * 2 + 3) *
            (exp(-((numother * 4 + 6) * x / theta0)) - 1) -
         e2 / (numother + 2) * (exp(-((numother * 2 + 4) * x / theta0)) - 1));
    if (xnew > r)
      xmax = x;
    else
      xmin = x;
  }
  if ((unsigned)x > tlength) {
     fprintf(ERRFILE,"disaster in setlength2\n");
     return(false);
  }
  if (!growth0 || !op->growthused) p->tyme = tstart + x;
  else {
     tstart = to_magic(tstart) + x;
     /* if the time is now impossible to big... */
     if (growth0*tstart < -1.0) return(false);
     else
       p->tyme = to_real(tstart);
  }


  return(setlength(2L, numother, p->tyme, tlength - x, q));
}  /* setlength2 */

void updatebranch(node *oldans, node *oldp, node *prime)
{
  subtymelist(oldans, oldp);
  inserttymelist(prime);
  rebuildbranch();
}  /* updatebranch */

long counttymelist(tlist *first, tlist *last)
{
   long count;
   tlist *t;

  count = 0;
   for (t = first; t != last; t = t->succ)
      count++;
   return(count+1);
} /* counttymelist */

boolean accept_tree(node *prime, node *primans, node *ans,
   node *pansdaught, node *newbr[], tlist *tplus)
{
node *p;
long i, leftout;

/* set up the phylogeny */
if (prime->tyme < tplus->eventnode->tyme) {
   /* case 1 */
   hookup(newbr[0], prime->next);
   hookup(newbr[1], prime->next->next);
   hookup(newbr[2], pansdaught);
} else {
   /* case 2 */
   leftout = (long)(randum() * 3.0) + 1;
   p = prime->next;
   for (i = 1; i <= 3; i++) {
      if (i != leftout) {
      hookup(newbr[i - 1], p);
      p = p->next;
      } else hookup(newbr[i - 1], pansdaught);
  }
}
if (primans->next == pansdaught)
   hookup(primans->next->next, ans);
else hookup(primans->next, ans);

prime->next->tyme = prime->tyme;
prime->next->next->tyme = prime->tyme;
primans->next->tyme = primans->tyme;
primans->next->next->tyme = primans->tyme;

for (i = 0; i <= 2; i++) ltov(newbr[i]);
ltov(prime);
ltov(ans);
/* get acceptance ratio */
if (newbr[1]->tyme == ans->tyme) return(true);

localevaluate(prime,pansdaught->back);
return (testratio());

} /* accept_tree */

boolean slide()
{
/* Local variables for slide: */

  node *prime, *oldp, *oldans, *primans, *pansdaught, *ans, *p,
     *oldbr[3], *newbr[3];
  long i, j, k, cline, numother, numintervals;
  double chance, tlength, normalizer, **coll2, **coll3, c[3];
  tlist *t, *tstart, *tend, *tplus;
  boolean accept_slide, done, skipped, succeeded;

  newbr[0] = newbr[1] = newbr[2] = NULL; /* just to be careful */

  do { /* There exists a chance that randum returns a 1 */
     i = (long)(randum() * slidenum) + 1;
  } while (i == slidenum + 1);
  oldp = slidenodes[i - 1];
  oldp = findtop(oldp);
  oldans = oldp->back;
  /* copy old nodes to new */
  newnode(&prime);
  copynode(oldp, prime);
  newnode(&primans);
  copynode(oldans, primans);
  hookup(prime, primans);
  /* name and connect nodes */
  oldbr[0] = prime->next->back;
  oldbr[1] = prime->next->next->back;
  if (!primans->next->top) {
    pansdaught = primans->next;
    oldbr[2] = primans->next->back;
    ans = primans->next->next->back;
  } else {
    pansdaught = primans->next->next;
    oldbr[2] = primans->next->next->back;
    ans = primans->next->back;
  }
  /* tymelist sort the three branches' tips in arbitrary order */
  j = 1;
  for (i = 0; i <= 2; i++) {
    if (oldbr[i]->tip) {
      newbr[j - 1] = oldbr[i];
      j++;
    }
  }
  /* remainder of tree */
  if (j <= 3) {
    t = curtree->tymelist->succ;
    done = false;
    while (!done) {
      for (i = 0; i <= 2; i++) {
	if (oldbr[i]->number == t->eventnode->number) {
	  newbr[j - 1] = oldbr[i];
	  j++;
	  if (j > 3) done = true;
	}
      }
      t = t->succ;
      if (t == NULL) {
	printf("ERROR IN TYMESORT\n");
        exit(-1);
      }
    }
  }
  tplus = gettymenode(newbr[2]->number);
  skipped = false; /* needed for frees in zero length case */
  succeeded = true; /* records success of setlength operations */
  /* zero length branches are a special case */
  if (newbr[1]->tyme == ans->tyme) {
    prime->tyme = newbr[1]->tyme;
    primans->tyme = newbr[1]->tyme;
    coll2 = NULL; /* just to be careful */
    coll3 = NULL;
    skipped = true;
  } else {
    /* initialize probability arrays for state (ie. 1,2 or 3 branches
       present) */
    tstart = gettymenode(newbr[1]->number);
    tend = gettymenode(ans->number);
    numintervals = counttymelist(tstart,tend);
    coll2 = (double **)calloc(2,sizeof(double *));
    coll2[0] = (double *)calloc(numintervals,sizeof(double));
    coll2[1] = (double *)calloc(numintervals,sizeof(double));
    coll3 = (double **)calloc(3,sizeof(double *));
    coll3[0] = (double *)calloc(numintervals,sizeof(double));
    coll3[1] = (double *)calloc(numintervals,sizeof(double));
    coll3[2] = (double *)calloc(numintervals,sizeof(double));
    t = tstart;
    i = 0;
    cline = 1;
    numother = tstart->numbranch - 2;
    /* initialize 2-array */
    coll2[0][i] = 0.0;
    coll2[1][i] = 1.0;
    /* fill up 2-array */
    while (t != tplus) {
      i++;
      tlength = howlong(t);
      if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	tlength = 20 * theta0 / (numother * 6 + 6);
      coll2[0][i] = coll2[0][i - 1] * zerocollis(1L, numother, tlength)
                  + coll2[1][i - 1] * onecollis(2L, numother, tlength);
      coll2[1][i] = coll2[1][i - 1] * zerocollis(2L, numother, tlength);
      normalizer = coll2[0][i] + coll2[1][i];
      coll2[0][i] /= normalizer;
      coll2[1][i] /= normalizer;
      if (normalizer == 0.0) {
         fprintf(ERRFILE,"Encountered machine precision limits!\n");
         exit(-1);
      }
      t = t->succ;
      if (t->eventnode->number != oldp->number &&
	  t->eventnode->number != oldans->number)
	numother--;
    }
    if (newbr[2]->tyme >= ans->tyme) {
      /* case 2 zero length */
      primans->tyme = tplus->eventnode->tyme;
    } else {
      /* initialize 3-array */
      j = 0;
      numother--;
      coll3[0][j] = 0.0;
      coll3[1][j] = coll2[0][i];
      coll3[2][j] = coll2[1][i];
      /* fill up 3-array */
      done = false;
      while (!done) {
	j++;
	tlength = howlong(t);
	if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	  tlength = 20 * theta0 / (numother * 6 + 6);
	coll3[0][j] = coll3[0][j - 1] * zerocollis(1L, numother, tlength) +
		      coll3[1][j - 1] * onecollis(2L, numother, tlength) +
		      coll3[2][j - 1] * twocollis(numother, tlength);
	coll3[1][j] = coll3[1][j - 1] * zerocollis(2L, numother, tlength) +
		      coll3[2][j - 1] * onecollis(3L, numother, tlength);
	coll3[2][j] = coll3[2][j - 1] * zerocollis(3L, numother, tlength);
        normalizer = coll3[0][j] + coll3[1][j] + coll3[2][j];
        coll3[0][j] /= normalizer;
        coll3[1][j] /= normalizer;
        coll3[2][j] /= normalizer;
        if (normalizer == 0.0) {
           fprintf(ERRFILE,"Encountered machine precision limits!\n");
           exit(-1);
        }
	if (t->succ == tend) {
	  done = true;
	  break;
	}
	t = t->succ;
	if (t->eventnode->number != oldp->number &&
	    t->eventnode->number != oldans->number)
	  numother--;
      }
      /* now find out when prime and primans collide */
      k = j;
      while (cline != 3 && k != 0 && t != NULL) {
	tlength = howlong(t);
	if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	  tlength = 20 * theta0 / (numother * 6 + 6);
	chance = randum();
	if (cline == 1) {
          c[0] = coll3[0][k-1] * zerocollis(1L, numother, tlength);
          c[1] = coll3[1][k-1] * onecollis(2L, numother, tlength);
          c[2] = coll3[2][k-1] * twocollis(numother, tlength);
          normalizer = c[0] + c[1] + c[2];
          c[0] /= normalizer;
          c[1] /= normalizer;
          if (chance > c[0]) {
            chance -= c[0];
            if (chance > c[1]) { /* two collisions */
              cline += 2;
	      if (!setlength2(numother, t->eventnode->tyme, tlength, prime,
			 primans)) succeeded = false;
            } else { /* one collision */
              cline++;
	      if (!setlength(2L, numother, t->eventnode->tyme, 
                         tlength, primans)) succeeded = false;
            }
          }
	} else {  /* cline must equal 2 */
          c[0] = coll3[1][k-1] * zerocollis(2L, numother, tlength);
          c[1] = coll3[2][k-1] * onecollis(3L, numother, tlength);
          normalizer = c[0] + c[1];
          c[0] /= normalizer;
	  if (chance > c[0]) {
	    cline++;
	    if (!setlength(3L, numother, t->eventnode->tyme, tlength, prime))
               succeeded = false;
	  }
	}
	if (t == NULL) {
          fprintf(ERRFILE,"ERROR in slide, ran off tymelist!\n");
          exit(-1);
        }
        if (t->eventnode->number != oldp->number &&
          t->eventnode->number != oldans->number)
	  numother++;
	k--;
	t = t->prev;
      }
      cline--;   /* A lineage dies! */
      numother++;
      if (cline == 0) {
	printf("ERROR in slide, no lineages left!");
	printf("  cline = 0, too few uncollisions\n");
	printf(" in loop %12ld\n", indecks);
        exit(-1);
      }
    }
    k = i;
    t = tplus->prev;
    while (cline == 1 && k != 0 && t != NULL) {
      tlength = howlong(t);
      if ((numother * 6 + 6) * tlength / theta0 > 20.0)
	tlength = 20 * theta0 / (numother * 6 + 6);
      chance = randum();
      c[0] = coll2[0][k-1] * zerocollis(1L, numother, tlength);
      c[1] = coll2[1][k-1] * onecollis(2L, numother, tlength);
      normalizer = c[0] + c[1];
      c[0] /= normalizer;
      if (chance > c[0]) {
	cline++;
	if (!setlength(2L, numother, t->eventnode->tyme, tlength,prime))
           succeeded = false;
      } else {
	k--;
	t = t->prev;
      }
      if (t->eventnode->number != oldp->number &&
	  t->eventnode->number != oldans->number)
	numother++;
    }
  }
  if (succeeded) /* all the setlength operations worked */
     accept_slide = accept_tree(prime, primans, ans, pansdaught, newbr,
                 tplus);
  else /* a setlength failed, garbage present! */
     accept_slide = false;

  slid++;
  if (!skipped) {
     free(coll2[0]);
     free(coll2[1]);
     free(coll2);
     free(coll3[0]);
     free(coll3[1]);
     free(coll3[2]);
     free(coll3);
  }
  if (accept_slide) {
    slacc++;
    updatebranch(oldans,oldp,prime);
    updateslide(oldp, false);
    freenode(oldp);
    updateslide(oldans, false);
    freenode(oldans);
    updateslide(primans, true);
    updateslide(prime, true);
    return (accept_slide);
  }
  hookup(oldbr[0], oldp->next);
  hookup(oldbr[1], oldp->next->next);
  if (!oldans->next->top) {
    hookup(oldbr[2], oldans->next);
    hookup(oldans->next->next, ans);
  } else {
    hookup(oldbr[2], oldans->next->next);
    hookup(oldans->next, ans);
  }
  p = oldp;
  for (i = 1; i <= 2; i++) {
    p = p->next;
    p->back->v = p->v;
  }
  p = oldans;
  for (i = 1; i <= 2; i++) {
    p = p->next;
    p->back->v = p->v;
  }
  localevaluate(oldp,oldbr[2]);
  freenode(prime);
  freenode(primans);
       
  curtree->likelihood = oldlikelihood;

  return (accept_slide);
}  /* slide */

void maketree()
{
  long 		incrprog, i, metout, progout;
  double 	bestlike;
  boolean 	chainend, runend, changetree;
  char          *chainlit[2];

  chainlit[0] = "Short";
  chainlit[1] = "Long";

  contribution = (contribarr *)calloc(sites,sizeof(contribarr));
  contribution[0] = (double *)calloc(sites*categs,sizeof(double));
  for (i=1;i<sites;i++)
     contribution[i] = contribution[0] + i*categs;

  inittable();
  initweightrat();
  getc(infile);
  fprintf(outfile, "Watterson estimate of theta is %12.8f\n", watttheta);
  if (op->usertree)
     treeread();
  else {
     branch0 = watttheta/numseq; 
     constructtree(numseq, branch0);
  }
  orient(curtree->root->back);
  finishsetup(curtree->root->back);
  initbranchlist();
  treevaluate();
  bestlike = NEGMAX;
  growi[locus][0] = growth0;
  theti[locus][0] = theta0;
  lntheti[locus][0] = log(theta0);
  runend = false;
  /* We're going to start sampling thetas with tree 10, and resample after
     10 more trees have been outputted. */
  /**********************************/
  /* Begin Hastings-Metropolis loop */
  /**********************************/
  for (apps = 0; apps < totchains; apps++) {
    if (apps >= op->numchains[0]) chaintype = 1;
    else chaintype = 0;
    if (op->progress) {
      printf("%s chain %ld ",chainlit[chaintype],
         ((chaintype == 0) ? (apps + 1) : (apps + 1 - op->numchains[0])));
      fflush(stdout);
    }
    metout = op->increm[chaintype] - 1;
    incrprog = (long)(op->steps[chaintype] / 10.0);
    progout = incrprog - 1;
    op->numout[chaintype] = 0;
    xinterval = 0.0; /* initialize largest interval */
    slacc = 0;
    slid = 0;
    chainend = false;
    changetree = true; /* just to be careful */

    for (indecks=0; indecks < op->steps[chaintype]; indecks++) {
      oldlikelihood = curtree->likelihood;
      col = 0; /* column number, used in treeout */
      if (slide()) changetree = true;
      if (indecks == op->steps[chaintype] - 1) { /* end of chain? */
        chainend = true;
        if (apps == totchains - 1) /* end of run? */
          runend = true;
      }
      if (curtree->likelihood > bestlike) {
        if (onebestree) {
           FClose(bestree);
           bestree = fopen("bestree","w+");
	   fprintf(bestree, "Chain #%2ld (%s) Step:%8ld\n",apps+1, 
             chainlit[chaintype], indecks+1);
	   treeout(curtree->root->back, 1L, &bestree);
	   fprintf(bestree, " [%12.10f]\n", curtree->likelihood);
	   bestlike = curtree->likelihood;
        }
        else {
	   fprintf(bestree, "Chain #%2ld (%s) Step:%8ld\n",apps+1, 
             chainlit[chaintype], indecks+1);
	   treeout(curtree->root->back, 1L, &bestree);
	   fprintf(bestree, " [%12.10f]\n", curtree->likelihood);
	   bestlike = curtree->likelihood;
        }
      }
      if (indecks == metout) {
        if (op->numout[chaintype] == 0) sametree[locus][0] = false;
        else sametree[locus][op->numout[chaintype]] = !changetree;
        changetree = false;
	op->numout[chaintype]++;
	scoretree(apps);
	metout += op->increm[chaintype];
        if (op->treeprint && apps == totchains-1) {
           fprintf(treefile,"\nlocus = %ld, chain = %ld, tree = %ld,",
                   locus,apps,indecks);
           fprintf(treefile,"likelihood = %e\n",curtree->likelihood);
           treeout(curtree->root->back, 1L, &treefile);
        }
      }
      if (op->progress && indecks == progout) {
	printf(".");
        fflush(stdout);
	progout += incrprog;
      }
    }
    if (op->progress) printf("\nAccepted %ld/%ld rearrangements\n",slacc,slid);
    if (!op->growthused) {
       theta0 = coal_singlechain(apps, chainend, runend);  
    } else {
       fluc_estimate(apps,runend);
       theta0 = theti[locus][apps+1];
       growth0 = growi[locus][apps+1];
    }
  }
  if(slacc == 0) {
     fprintf(outfile,"WARNING--no proposed trees ever accepted\n");
     fprintf(ERRFILE,"WARNING--no proposed trees ever accepted\n");
  }
}  /* maketree */

void finalfree()
/* free everything at end of program; a debugging convenience to check 
   for memory leaks */
{
  free(rate);
  free(probcat);
  free(ne_ratio);
  free(mu_ratio);
  free(theta_ratio);
  free(sametree[0]);
  free(sametree);
  free(curtree);
  freedata(dna);
} /* finalfree */

int main(int argc, char *argv[])
{  /* Fluctuate */
  char infilename[100],outfilename[100],trfilename[100],logfilename[100],
     thfilename[100],intrfilename[100],bestrfilename[100];

  double clearseed;
  long i;

#ifdef MAC
  /*macsetup("Fluctuate","Fluctuate");*/
  argv[0] = "Fluctuate";
#endif

  /* Open various filenames. */

  openfile(&infile,INFILE,"r",argv[0],infilename);
  if (!menu) 
      {
      openfile(&simlog,"simlog","w",argv[0],logfilename);
      }
  if (thetaout) openfile(&thetafile,"thetafile","w",argv[0],thfilename);
  openfile(&outfile,OUTFILE,"w",argv[0],outfilename);

  op = (option_struct *)calloc(1,sizeof(option_struct));
  op->ibmpc = IBMCRT;
  op->ansi = ANSICRT;
  getoptions();
  /* start up randum numbers */
  for (i = 1; i <= 1000; i++)
    clearseed = randum();
  if (op->usertree)
    openfile(&intree,INTREE,"r",argv[0],intrfilename);
  if (op->treeprint)   
    openfile(&treefile,TREEFILE,"w",argv[0],trfilename);      
  openfile(&bestree,"bestree","w",argv[0],bestrfilename);      
  firstinit();
  for (locus = 0; locus < numloci; locus++) {
     if (op->progress) printf("Locus %ld\n",locus+1);
     fprintf(outfile,"\n---------------------------------------\n");
     fprintf(outfile, "Locus %ld\n",locus+1);
     fprintf(outfile,"---------------------------------------\n");
     fprintf(outfile,"Assumed relative Ne = %f\n",ne_ratio[locus]);
     fprintf(outfile,"Assumed relative mu = %f\n",mu_ratio[locus]);
     if (holding)
        if (holding == 1)
           fprintf(outfile,"Theta was held constant.\n");
        else
           fprintf(outfile,"Growth-rate was held constant.\n");
     locusinit();
     getinput();
     watttheta = watterson();
     if (op->watt)
       theta0 = watttheta;
     maketree();
     freetree();
  }
  if (!op->growthused) coal_curveplot(); 
  if (numloci > 1) {
     fluc_locus_estimate();
  }
  modelfree();
  finalfree();
  FClose(infile);
  FClose(outfile);
  FClose(treefile);
  FClose(bestree);
  FClose(simlog);
  FClose(parmfile);
  FClose(thetafile);
#ifdef MAC
  fixmacfile(outfilename);
  fixmacfile(trfilename);
  fixmacfile(bestrfilename);
  fixmacfile(intrfilename);
  fixmacfile(thfilename);
  fixmacfile(logfilename);
#endif
  printf("PROGRAM DONE\n");
  exit(0);
}  /* Fluctuate */

int eof(FILE *f)
{
    register int ch;

    if (feof(f))
        return 1;
    if (f == stdin)
        return 0;
    ch = getc(f);
    if (ch == EOF)
        return 1;
    ungetc(ch, f);
    return 0;
} /* eof */

int eoln(FILE *f)
{
  register int ch;
  
  ch = getc(f);
  if (ch == EOF)
    return 1;
  ungetc(ch, f);
  return (ch == '\n');
} /* eoln */

void memerror()
{
printf("Error allocating memory\n");
exit(-1);
} /* memerror */
