/*****************************************************************************
 *  FTools - Freenet Client Protocol Tools
 *
 *  Copyright (C) 2002 Juergen Buchmueller <juergen@pullmoll.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *	$Id: ftfile.c,v 1.67 2005/07/19 13:52:41 pullmoll Exp $
 *****************************************************************************/
#include "ftfile.h"
#include "ftmime.h"
#include "ftfcp.h"
#include "ftlog.h"

#define	MAXBUFF	32768

typedef struct chk_chunk_s {
	char *name;
	char *chk;
	int retry;
	int	htl;
	FILE *fp;
}	chk_chunk_t;

/*****************************************************************************
 *	put_file
 *	Inserts a single file under a given Freenet URI. Creates redirects for
 *	non-CHK inserts of files > 32K (if limit32k is set) and creates split file
 *	chunks and metadata for files being greater than chunksize (256K).
 *
 *	pf			pointer to a fcp_t freenet client protocol structure
 *	uri			uniform resource identifier where to insert the file
 *	htl			hops to live for the insert
 *	wiggle		htl 'wiggle' range
 *	filename	local filename
 *	mimetype	MIME type for the file (text/plain etc.)
 *****************************************************************************/
int put_file(fcp_t **pf, const char *uri, int htl, int wiggle,
	const char *filename, const char *mimetype)
{
	fcp_t *f = *pf;
	size_t metasize = 0, datasize = 0;
	size_t i, chunk = 0, chk_chunk_cnt = 0;
	chk_chunk_t **chk_chunk = NULL, *b = NULL;
	char *data = NULL, *meta = NULL, *cmdline = NULL, *chk = NULL, *dst, *base;
	FILE *fp = NULL;
	fd_set rdfds, exfds;
	int fd, nfds, minfd, maxfd;
	int retry = 0, rc = -1;

	if (wiggle > 0) {
		htl += (rand() % wiggle) - wiggle / 2;
		if (htl < 1) {
			htl = 1;
		}
	}

	if (0 == strcmp(filename, "-")) {

		fp = stdin;
		datasize = chunksize;
		base = "stdin";

	} else {
		fp = fopen(filename, OPEN_RB);
		if (NULL == fp) {
			LOG(L_ERROR,("cannot open \"%s\" (%s)\n",
				filename, strerror(errno)));
			goto bailout;
		}
		fseek(fp, 0, SEEK_END);
		datasize = ftell(fp);
		fseek(fp, 0, SEEK_SET);
		base = strrchr(filename, '/');
		if (NULL == base) {
			base = strrchr(filename, '\\');
		}
		if (NULL == base) {
			base = (char *)filename;
		} else {
			base++;
		}
	}

	if (datasize > chunksize) {
		cmdline = calloc(MAXBUFF, sizeof(char));
		if (NULL == cmdline) {
			LOG(L_ERROR,("calloc(%d,%d) failed (%s)\n",
				MAXBUFF, sizeof(char), strerror(errno)));
			goto bailout;
		}

		chk_chunk_cnt = (datasize + chunksize - 1) / chunksize;
		chk_chunk = calloc(chk_chunk_cnt, sizeof(chk_chunk_t *));
		if (NULL == chk_chunk) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				sizeof(char *), chk_chunk_cnt, strerror(errno)));
			goto bailout;
		}

		data = calloc(chunksize, sizeof(char));
		if (NULL == data) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				chunksize, sizeof(char), strerror(errno)));
			goto bailout;
		}

		LOG(L_NORMAL,("Inserting %s (%d, %s) as split file\n\twith %d chunks of max. %d bytes each\n",
			base, datasize, mimetype, chk_chunk_cnt, chunksize));
		for (;;) {
			size_t done, offs;

			FD_ZERO(&rdfds);
			FD_ZERO(&exfds);
			nfds = 0;
			minfd = 32767;
			maxfd = 0;
			LOG(L_DEBUGX,("pending:"));
			for (i = 0; i < chk_chunk_cnt; i++) {
				b = chk_chunk[i];
				if (NULL != b && NULL != b->fp && NULL == b->chk) {
					fd = fileno(b->fp);
					FD_SET(fd, &rdfds);
					FD_SET(fd, &exfds);
					if (fd < minfd)
						minfd = fd;
					if (fd > maxfd)
						maxfd = fd;
					nfds += 1;
					LOG(L_DEBUGX,(" #%d (fd:%d)", i+1, fd));
				}
			}
			LOG(L_DEBUGX,("\n"));
		
			while (nfds < sf_threads) {

				for (b = NULL, chunk = 0; chunk < chk_chunk_cnt; chunk++) {
					b = chk_chunk[chunk];
					if (NULL == b || (NULL == b->fp && NULL == b->chk))
						break;
				}

				/* no more chunks waiting */
				if (chunk == chk_chunk_cnt)
					break;

				if (NULL == b) {
					chk_chunk[chunk] = calloc(1, sizeof(chk_chunk_t));
					if (NULL == chk_chunk[chunk]) {
						LOG(L_ERROR,("calloc(%d,%d) failed (%s)\n",
							1, sizeof(chk_chunk_t), strerror(errno)));
						goto bailout;
					}
					b = chk_chunk[chunk];
					b->htl = htl;
					if (wiggle > 0) {
						b->htl += (rand() % wiggle) - wiggle / 2;
						if (b->htl < 1) {
							b->htl = 1;
						}
					}
				}

				offs = chunksize * chunk;
				fseek(fp, offs, SEEK_SET);
				done = fread(data, 1, chunksize, fp);
				if (done <= 0) {
					LOG(L_ERROR,("fread(...,%d,...) failed (%s)\n",
						chunksize, strerror(errno)));
					goto bailout;
				}

				/* insert stdin as a CHK@ key */
				snprintf(cmdline, MAXBUFF,
					"\"%s\" %s -n%s -p%d -l%d -t%d -s%u -v%d %s%s%sCHK@ -",
					program, "put", fcphost, fcpport, b->htl,
					sf_threads, done, verbose,
					delete ? "--delete " : "",
					meta_only ? "--meta-only " : "",
					meta_none ? "--meta-none " : "");

				/* try bidirectional popen() */
				b->fp = popen(cmdline, "r+");
				if (NULL == b->fp) {
					FILE *fo;
					/* this is a unidirectional popen() :-( */
					snprintf(cmdline, MAXBUFF, "%s/ft-%d.%d",
						temppath, getpid(), chunk);
					fo = fopen(cmdline, OPEN_WB);
					if (NULL == fo) {
						LOG(L_ERROR,("fopen(\"%s\",\"%s\") failed (%s)\n",
							cmdline, OPEN_WB, strerror(errno)));
						goto bailout;
					}
					if (done != fwrite(data, 1, done, fo)) {
						LOG(L_ERROR,("fwrite(...,%d,...) failed (%s)\n",
							done, strerror(errno)));
						goto bailout;
					}
					fclose(fo);
					snprintf(cmdline, MAXBUFF,
						"\"%s\" %s -n%s -p%d -l%d -t%d -s%u -v%d %s%s%sCHK@ %s/ft-%d.%d",
						program, "put", fcphost, fcpport, b->htl, sf_threads,
						done, verbose,
						delete ? "--delete " : "",
						meta_only ? "--meta-only " : "",
						meta_none ? "--meta-none " : "",
						temppath, getpid(), chunk);
					LOG(L_DEBUG,("cmdline: '%s'\n", cmdline));
					/* try unidirectional popen() */
					b->fp = popen(cmdline, OPEN_RB);
					if (NULL == b->fp) {
						LOG(L_ERROR,("popen(\"%s\",\"%s\") failed (%s)\n",
							cmdline, "r", strerror(errno)));
						goto bailout;
					}

				} else {

					if (done != fwrite(data, 1, done, b->fp)) {
						LOG(L_ERROR,("fwrite(...,%d,...) failed (%s)\n",
							done, strerror(errno)));
						goto bailout;
					}
					fflush(b->fp);
				}

				chunk++;
				b->name = calloc(256, sizeof(char));
				snprintf(b->name, 256, "chunk #%d/%d (%s)",
					chunk, chk_chunk_cnt, base);
				fd = fileno(b->fp);
				FD_SET(fd, &rdfds);
				FD_SET(fd, &exfds);
				if (fd < minfd)
					minfd = fd;
				if (fd > maxfd)
					maxfd = fd;
				nfds += 1;
				LOG(L_NORMAL,("Inserting chunk #%d/%d (%d @%d) of %s with HTL:%d\n",
					chunk, chk_chunk_cnt, done, offs, base, b->htl));
				LOG(L_DEBUG,("%s\n",
					cmdline));
			}

			/* break out of the loop if all split file threads are done */
			if (0 == nfds) {
				break;
			}

			if (nfds > 0) {
				struct timeval tv = {1,0};

				fd = select(maxfd+1, &rdfds, NULL, &exfds, &tv);
				if (-1 == fd) {
					LOG(L_ERROR,("select(%d,...) failed (%s)\n",
						maxfd+1, strerror(errno)));
					goto bailout;
				}
				for (fd = minfd; fd < maxfd+1; fd++) {
					for (i = 0; i < chk_chunk_cnt; i++) {
						if (NULL != chk_chunk[i] &&
							NULL != chk_chunk[i]->fp &&
							fd == fileno(chk_chunk[i]->fp))
							break;
					}
					b = chk_chunk[i];
					if (FD_ISSET(fd, &rdfds)) {
						char *reply = calloc(MAXBUFF, sizeof(char));
						char *eol, *result, *chk;
						if (NULL == reply) {
							LOG(L_ERROR,("calloc(%d,%d) failed (%s)\n",
								MAXBUFF, sizeof(char), strerror(errno)));
							goto bailout;
						}
						fgets(reply, MAXBUFF, b->fp);
						LOG(L_DEBUG,("reply: \"%s\"\n", reply));
						eol = strtok(reply, "\r\n");
						if (NULL != eol) {
							result = strtok(eol, ":\r\n");
							chk = strtok(NULL, " \r\n");
						} else {
							result = NULL;
							chk = NULL;
						}
						if (NULL == result ||
							NULL == chk ||
							(0 != strcasecmp(result, SUCCESS) &&
							0 != strcasecmp(result, KEYCOLLISION))) {
							if (b->htl > 1) {
								b->htl -= 1;
							}
							b->retry += 1;
							snprintf(reply, MAXBUFF, "RETRY #%d (HTL:%d)",
								b->retry, b->htl);
							result = reply;
							LOG(L_NORMAL,("%s: split file chunk #%d/%d of %s\n",
								result, i+1, chk_chunk_cnt, filename));
						} else {
							while (chk && *chk == ' ')
								chk++;
							LOG(L_NORMAL,("%s: split file chunk #%d/%d of %s\n",
								result, i+1, chk_chunk_cnt, filename));
							LOG(L_MINOR,("%s\n",
								chk));
							b->chk = strdup(chk);
						}
						free(reply);
						pclose(b->fp);
						b->fp = NULL;
						snprintf(cmdline, MAXBUFF, "%s/ft-%d.%d",
							temppath, getpid(), i);
						unlink(cmdline);
					}
					if (FD_ISSET(fd, &exfds)) {
						if (NULL != b->fp) {
							LOG(L_MINOR,("closing chunk #%d %s\n",
								i+1, b->name));
							pclose(b->fp);
							b->fp = NULL;
							snprintf(cmdline, MAXBUFF, "%s/ft-%d.%d",
								temppath, getpid(), i);
							unlink(cmdline);
						}
						if (b->htl > 1) {
							b->htl -= 1;
						}
						b->retry += 1;
					}
				}
			}
		}

		for (;;) {

			if (NULL != f->data) {
				free(f->data);
				f->data = NULL;
			}
			if (NULL != f->meta) {
				free(f->meta);
				f->meta = NULL;
			}
			fcp_free(pf);
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				LOG(L_ERROR,("failed to connect to FCP host %s:%d\n",
					fcphost, fcpport));
			}
			f = *pf;

			/* compute size of required metadata buffer */
			metasize = 512;
			for (i = 0; i < chk_chunk_cnt; i++) {
				metasize += 
					strlen("SplitFile.Block.xxxx=") +
					strlen(chk_chunk[i]->chk) + 1 + 1;
			}
			meta = calloc(metasize + 1, sizeof(char));
			if (NULL == meta) {
				LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
					metasize + 1, sizeof(char), strerror(errno)));
				goto bailout;
			}
	
			dst = meta;
			dst += sprintf(dst, "Version\n");
			dst += sprintf(dst, "Revision=%d\n", revision);
			dst += sprintf(dst, "EndPart\n");
			dst += sprintf(dst, "Document\n");
			dst += sprintf(dst, "SplitFile.Size=%x\n",
				datasize);
			dst += sprintf(dst, "SplitFile.Blocksize=%x\n",
				chunksize);
			dst += sprintf(dst, "SplitFile.BlockCount=%x\n",
				(datasize + chunksize - 1) / chunksize);
			for (i = 0; i < chk_chunk_cnt; i++) {
				dst += sprintf(dst, "SplitFile.Block.%x=%s\n",
					i+1, chk_chunk[i]->chk);
			}
			dst += sprintf(dst, "Info.Format=%s\n", mimetype);
			dst += sprintf(dst, "End\n");

			f->data = NULL;
			f->datasize = 0;
			f->meta = meta;
			f->metasize = (size_t)(dst - meta);

			if (0 == (rc = fcp_put(f, uri, htl))) {
				/* we're done... */
				break;
			}

			if (f->seen_route_not_found) {
				LOG(L_ERROR,(ROUTENOTFOUND ": %s\n",
					uri));
			} else if (f->seen_data_not_found) {
				LOG(L_ERROR,(DATANOTFOUND ": %s\n",
					uri));
			} else if (f->seen_uri_error) {
				LOG(L_ERROR,(URIERROR ": %s\n",
					uri));
				goto bailout;
			} else if (f->seen_size_error) {
				LOG(L_ERROR,(SIZEERROR ": %s\n",
					uri));
				goto bailout;
			} else {
				LOG(L_ERROR,(GENERALERROR ": %s\n",
					uri));
			}
			retry++;
			LOG(L_NORMAL,("RETRY #%d (HTL:%d): %s\n",
				retry, htl, filename));
		}

		meta = NULL;

	} else if (0 != limit32k && datasize > 32000 &&
		0 != strcasecmp(uri, "CHK@")) {

		for (;;) {
			if (0 == meta_none) {
				f->meta = calloc(MAXBUFF, sizeof(char));
				if (NULL == f->meta) {
					LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
						MAXBUFF, sizeof(char), strerror(errno)));
					goto bailout;
				}

				f->metasize = snprintf(f->meta, MAXBUFF,
					"Version\n" \
					"Revision=%d\n" \
					"EndPart\n" \
					"Document\n" \
					"Info.Format=%s\n" \
					"Info.Description=file\n" \
					"End\n",
					revision,
					mimetype);
			}
			f->datasize = datasize;
			f->data = calloc(f->datasize + 1, sizeof(char));
			if (NULL == f->data) {
				LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
					f->datasize + 1, sizeof(char), strerror(errno)));
				goto bailout;
			}
			fseek(fp, 0, SEEK_SET);
			f->datasize = fread(f->data, 1, f->datasize, fp);

			/* insert as CHK@ key */
			if (0 == (rc = fcp_put(f, "CHK@", htl))) {
				break;
			}
			if (f->seen_route_not_found) {
				LOG(L_ERROR,(ROUTENOTFOUND ": %s\n",
					uri));
			} else if (f->seen_data_not_found) {
				LOG(L_ERROR,(DATANOTFOUND ": %s\n",
					uri));
			} else if (f->seen_uri_error) {
				LOG(L_ERROR,(URIERROR ": %s\n",
					uri));
				goto bailout;
			} else if (f->seen_size_error) {
				LOG(L_ERROR,(SIZEERROR ": %s\n",
					uri));
				goto bailout;
			} else {
				LOG(L_ERROR,(GENERALERROR ": %s\n",
					uri));
			}
			retry++;
			LOG(L_NORMAL,("RETRY #%d: %s\n",
				retry, filename));

			fcp_free(pf);
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				LOG(L_ERROR,("failed to connect to FCP host %s:%d\n",
					fcphost, fcpport));
			}
			f = *pf;
		}

		if (NULL == (chk = strdup(f->uri))) {
			LOG(L_ERROR,("barf: strdup(%s) failed (%s)\n",
				f->uri, strerror(errno)));
			goto bailout;
		}

		/* and create a redirect now */
		fcp_free(pf);
		if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
			LOG(L_ERROR,("failed to connect to FCP host %s:%d\n",
				fcphost, fcpport));
		}
		f = *pf;

		for (;;) {
			f->meta = calloc(MAXBUFF, sizeof(char));
			if (NULL == f->meta) {
				LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
					MAXBUFF, sizeof(char), strerror(errno)));
				goto bailout;
			}

			if (0 == meta_none) {
				f->metasize = snprintf(f->meta, MAXBUFF,
					"Version\n" \
					"Revision=%d\n" \
					"EndPart\n" \
					"Document\n" \
					"Redirect.Target=%s\n" \
					"Info.Format=%s\n" \
					"Info.Description=file\n" \
					"End\n",
					revision, chk, mimetype);
			} else {
				/* leave off the Info.Format and Info.Description lines */
				f->metasize = snprintf(f->meta, MAXBUFF,
					"Version\n" \
					"Revision=%d\n" \
					"EndPart\n" \
					"Document\n" \
					"Redirect.Target=%s\n" \
					"End\n",
					revision, chk);
			}

			/* insert as KSK@ or SSK@ key */
			if (0 == (rc = fcp_put(f, uri, htl))) {
				break;
			}
			if (f->seen_route_not_found) {
				LOG(L_ERROR,(ROUTENOTFOUND ": %s\n",
					uri));
			} else if (f->seen_data_not_found) {
				LOG(L_ERROR,(DATANOTFOUND ": %s\n",
					uri));
			} else if (f->seen_uri_error) {
				LOG(L_ERROR,(URIERROR ": %s\n",
					uri));
				goto bailout;
			} else if (f->seen_size_error) {
				LOG(L_ERROR,(SIZEERROR ": %s\n",
					uri));
				goto bailout;
			} else {
				LOG(L_ERROR,(GENERALERROR ": %s\n",
					uri));
			}
			retry++;
			LOG(L_NORMAL,("RETRY #%d: %s\n",
				retry, filename));

			fcp_free(pf);
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				LOG(L_ERROR,("failed to connect to FCP host %s:%d\n",
					fcphost, fcpport));
			}
			f = *pf;
		}

	} else {

		for (;;) {
			if (NULL != *pf) {
				fcp_free(pf);
			}
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				LOG(L_ERROR,("failed to connect to FCP host %s:%d\n",
					fcphost, fcpport));
			}
			f = *pf;
			if (NULL == f) {
				LOG(L_ERROR,("barf: fcp_new() failed (%s)\n",
					strerror(errno)));
				goto bailout;
			}
			if (0 == meta_none) {
				f->meta = calloc(MAXBUFF, sizeof(char));
				if (NULL == f->meta) {
					LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
						MAXBUFF, sizeof(char), strerror(errno)));
					goto bailout;
				}

				f->metasize = snprintf(f->meta, MAXBUFF,
					"Version\n" \
					"Revision=%d\n" \
					"EndPart\n" \
					"Document\n" \
					"Info.Format=%s\n" \
					"Info.Description=file\n" \
					"End\n",
					revision, mimetype);
			}
			f->datasize = datasize;
			f->data = malloc(f->datasize);
			if (NULL == f->data) {
				LOG(L_ERROR,("barf: malloc(%d) failed (%s)\n",
					f->datasize, strerror(errno)));
				goto bailout;
			}
			fseek(fp, 0, SEEK_SET);
			datasize = fread(f->data, 1, f->datasize, fp);

			if (0 == (rc = fcp_put(f, uri, htl))) {
				/* we're done... */
				break;
			}

			retry++;
			LOG(L_NORMAL,("RETRY #%d: %s\n",
				retry, filename));
		}
	}

bailout:
	if (NULL != chk_chunk) {
		char *tempname = calloc(MAXBUFF, sizeof(char));
		for (i = 0; i < chk_chunk_cnt; i++) {
			b = chk_chunk[i];
			if (NULL != b) {
				snprintf(tempname, MAXBUFF, "%s/ft-%d.%d",
					temppath, getpid(), i);
				unlink(tempname);
				if (NULL != b->name) {
					free(b->name);
					b->name = NULL;
				}
				if (NULL != b->chk) {
					free(b->chk);
					b->chk = NULL;
				}
				if (NULL != b->fp) {
					pclose(b->fp);
					b->fp = NULL;
				}
				free(chk_chunk[i]);
				chk_chunk[i] = NULL;
			}
		}
		free(chk_chunk);
		chk_chunk = NULL;
	}
	if (NULL != fp && stdin != fp) {
		fclose(fp);
		fp = NULL;
	}
	if (NULL != meta) {
		free(meta);
		meta = NULL;
	}
	if (NULL != data) {
		free(data);
		data = NULL;
	}
	if (NULL != cmdline) {
		free(cmdline);
		cmdline = NULL;
	}
	if (NULL != chk) {
		free(chk);
		chk = NULL;
	}
	return rc;
}

/*****************************************************************************
 *	chk_file
 *	Computes the content hash key for a file with the given MIME type.
 *
 *	pf			pointer to a fcp_t freenet client protocol structure
 *	filename	local filename
 *	mimetype	MIME type for the file (text/plain etc.)
 *****************************************************************************/
int chk_file(fcp_t **pf, const char *filename, const char *mimetype)
{
	fcp_t *f = *pf;
	size_t datasize, metasize;
	FILE *fp = NULL;
	int rc = -1;

	if (0 == strcmp(filename, "-")) {
		fp = stdin;
		datasize = chunksize;
	} else {
		fp = fopen(filename, OPEN_RB);
		if (NULL == fp) {
			LOG(L_ERROR,("fopen(\"%s\",\"%s\") call failed (%s)\n",
				filename, OPEN_RB, strerror(errno)));
			goto bailout;
		}
		fseek(fp, 0, SEEK_END);
		datasize = ftell(fp);
		fseek(fp, 0, SEEK_SET);
	}

	metasize = MAXBUFF;
	f->meta = calloc(metasize, sizeof(char));
	if (NULL == f->meta) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			metasize, sizeof(char), strerror(errno)));
		goto bailout;
	}

	f->metasize = snprintf(f->meta, MAXBUFF,
		"Version\n" \
		"Revision=%d\n" \
		"EndPart\n" \
		"Document\n" \
		"Info.Format=%s\n" \
		"Info.Description=file\n" \
		"End\n",
		revision,
		mimetype);

	f->datasize = datasize;
	f->data = malloc(f->datasize);
	if (NULL == f->data) {
		LOG(L_ERROR,("barf: malloc(%d) failed (%s)\n",
			f->datasize, strerror(errno)));
		goto bailout;
	}

	datasize = fread(f->data, 1, f->datasize, fp);

	rc = fcp_chk(f);

bailout:
	if (NULL != fp) {
		fclose(fp);
		fp = NULL;
	}

	return rc;
}

/*****************************************************************************
 *	fec_file
 *	Computes the FEC blocks for a file.
 *
 *	pf			pointer to a fcp_t freenet client protocol structure
 *	filename	local filename
 *****************************************************************************/
int fec_file(fcp_t **pf, const char *filename)
{
	FILE *fp = NULL;
	u_int64_t datasize;
	int rc = -1;

	if (0 == strcmp(filename, "-")) {
		fp = stdin;
		datasize = chunksize;
	} else {
		fp = fopen(filename, OPEN_RB);
		if (NULL == fp) {
			LOG(L_ERROR,("fopen(\"%s\",\"%s\") call failed (%s)\n",
				filename, OPEN_RB, strerror(errno)));
			goto bailout;
		}
		fseek(fp, 0, SEEK_END);
		datasize = ftell(fp);
		fseek(fp, 0, SEEK_SET);
	}

	(void)revision;

	rc = fcp_segment_file(pf, fcphost, fcpport,
		"OnionFEC_a_1_2", fp, datasize);

bailout:
	if (NULL != fp) {
		fclose(fp);
		fp = NULL;
	}

	return rc;
}

#define	FILEINFO_ALLOC(newblock) \
	if (newblock && NULL != fileinfo && NULL != fileinfo[fileinfo_cnt] && \
		NULL != fileinfo[fileinfo_cnt]->chk) { \
		fi = fileinfo[fileinfo_cnt]; \
		if (NULL == fi->mimetype) { \
			fi->mimetype = strdup("application/octet-stream"); \
		} \
		fileinfo_cnt++; \
	} \
	if (NULL == fileinfo || NULL == (fi = fileinfo[fileinfo_cnt])) { \
		if (fileinfo_cnt + 1 >= fileinfo_max) { \
			if (0 == fileinfo_max) { \
				fileinfo_max = 2; \
				fileinfo = calloc(fileinfo_max, sizeof(fileinfo_t *)); \
			} else  { \
				fileinfo_max *= 2; \
				fileinfo = realloc(fileinfo, \
					fileinfo_max * sizeof(fileinfo_t *)); \
				memset(&fileinfo[fileinfo_cnt], 0, \
					(fileinfo_max - fileinfo_cnt) * sizeof(fileinfo_t *)); \
			} \
		} \
		fi = fileinfo[fileinfo_cnt] = calloc(1, sizeof(fileinfo_t)); \
		LOG(L_MINOR,("allocted split file data block #%d\n", \
			fileinfo_cnt)); \
	}

#define	FILECHECK_ALLOC(newblock) \
	if (newblock && NULL != filecheck && NULL != filecheck[filecheck_cnt] && \
		NULL != filecheck[filecheck_cnt]->chk) { \
		filecheck_cnt++; \
	} \
	if (NULL == filecheck || NULL == (fc = filecheck[filecheck_cnt])) { \
		if (filecheck_cnt + 1 >= filecheck_max) { \
			if (0 == filecheck_max) { \
				filecheck_max = 2; \
				filecheck = calloc(filecheck_max, sizeof(fileinfo_t *)); \
			} else { \
				filecheck_max *= 2; \
				filecheck = realloc(filecheck, \
					filecheck_max * sizeof(fileinfo_t *)); \
				memset(&filecheck[filecheck_cnt], 0, \
					(filecheck_max - filecheck_cnt) * sizeof(fileinfo_t *)); \
			} \
		} \
		fc = filecheck[filecheck_cnt] = calloc(1, sizeof(fileinfo_t)); \
		LOG(L_MINOR,("allocted split file check block #%d\n", \
			filecheck_cnt)); \
	}

/*****************************************************************************
 *	get_file
 *	Retrieves a file from Freenet from the given URI uri with a request
 *	depth of htl (hops to live). If the metadata of the given key points to
 *	a redirect, date based redirect or split file, this function retrieves
 *	all the bits and pieces of the redirected keys or file fragments.
 *
 *	pf			pointer to a fcp_t freenet client protocol structure
 *	uri			uniform resource identifier of the key to retrieve
 *	htl			hops to live for the request
 *	filename	local filename
 *****************************************************************************/
int get_file(fcp_t **pf, const char *uri, int slimit, int htl, int wiggle,
	const char *filename, const char *infoname)
{
	fcp_t *f = *pf;
	char *dst, *src, *name, *slash;
	char *var = NULL, *val = NULL, *ssk = NULL, *algo = NULL;
	fileinfo_t **fileinfo = NULL, *fi = NULL;
	fileinfo_t **filecheck = NULL, *fc = NULL;
	size_t i, start = 0, size, split_offs = 0, split_size = 0, chunk_size = 0;
	size_t first_block = 0;
	size_t fileinfo_cnt = 0, fileinfo_max = 0;
	size_t filecheck_cnt = 0, filecheck_max = 0;
	char *temppath = NULL, *tempfile = NULL, *cmdline = NULL;
	fd_set rdfds, exfds;
	int fd, nfds, minfd, maxfd;
	FILE *fp = NULL;
	struct stat st;
	struct timeval tv;
	struct tm *tm;
	time_t date;
	int mode = MODE_NONE, rc = -1;

	if (wiggle > 0) {
		htl += (rand() % wiggle) - wiggle / 2;
		if (htl < 1) {
			htl = 1;
		}
	}

	if (NULL == f->meta) {
		goto bailout;
	}

	var = calloc(MAXBUFF, sizeof(char));
	if (NULL == var) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			MAXBUFF, sizeof(char), strerror(errno)));
		goto bailout;
	}

	val = calloc(MAXBUFF, sizeof(char));
	if (NULL == val) {
		LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
			MAXBUFF, sizeof(char), strerror(errno)));
		goto bailout;
	}

	src = f->meta;
	while (*src) {
		dst = var;
		while (*src != '\0' && *src != '\r' && *src != '\n' && *src != '=') {
			*dst++ = *src++;
		}
		*dst = '\0';
		dst = val;
		if (*src == '=') {
			src++;
			while (*src != '\0' && *src != '\r' && *src != '\n') {
				*dst++ = *src++;
			}
		}
		*dst = '\0';
		while (*src == '\r' || *src == '\n') {
			src++;
		}
		if (0 == strcasecmp(var, "Version")) {
			LOG(L_MINOR,("meta: Version\n"));
		} else if (0 == strcasecmp(var, "Revision")) {
			LOG(L_MINOR,("meta: Revision=%s\n", val));
		} else if (0 == strcasecmp(var, "Document")) {
			LOG(L_MINOR,("meta: Document\n"));
			chunk_size = 256 * 1024;	/* Default is 256KB chunks */
		} else if (0 == strcasecmp(var, "Name")) {
			LOG(L_MINOR,("meta: Name=%s\n", val));
			FILEINFO_ALLOC(0);
			fi->name = strdup(val);
		} else if (0 == strcasecmp(var, "Info.Format")) {
			LOG(L_MINOR,("meta: Info.Format=%s\n", val));
			FILEINFO_ALLOC(0);
			fi->mimetype = strdup(val);
		} else if (0 == strcasecmp(var, "Info.Description")) {
			LOG(L_MINOR,("meta: Info.Description=%s\n", val));
		} else if (0 == strcasecmp(var, "Info.Checksum")) {
			LOG(L_MINOR,("meta: Info.Checksum=%s\n", val));
		} else if (0 == strcasecmp(var, "Redirect.Target")) {
			if (MODE_NONE == mode) {
				mode = MODE_REDIR;
			} else {
				mode = MODE_MAPSPACE;
			}
			LOG(L_MINOR,("meta: Redirect.Target=%s\n", val));
			FILEINFO_ALLOC(1);
			fi->chk = strdup(val);
		} else if (0 == strcasecmp(var, "DateRedirect.Target")) {
			LOG(L_MINOR,("meta: DateRedirect.Target=%s\n", val));
			mode = MODE_DATEBASEDREDIR;
			FILEINFO_ALLOC(1);
			fi->chk = strdup(val);
		} else if (0 == strcasecmp(var, "DateRedirect.Offset")) {
			LOG(L_MINOR,("meta: DateRedirect.Offset=%s\n", val));
			date_offs = strtoul(val, NULL, 16);
		} else if (0 == strcasecmp(var, "DateRedirect.Increment")) {
			LOG(L_MINOR,("meta: DateRedirect.Increment=%s\n", val));
			date_incr = strtoul(val, NULL, 16);
		} else if (0 == strcasecmp(var, "SplitFile.Size")) {
			LOG(L_MINOR,("meta: SplitFile.Size=%s\n", val));
			split_size = strtoul(val, NULL, 16);
		} else if (0 == strcasecmp(var, "SplitFile.AlgoName")) {
			LOG(L_MINOR,("meta: SplitFile.AlgoName=%s\n", val));
			if (NULL != algo) {
				free(algo);
				algo = NULL;
			}
			algo = strdup(val);
		} else if (0 == strcasecmp(var, "SplitFile.BlockCount")) {
			mode = MODE_SPLITFILE;
			LOG(L_MINOR,("meta: SplitFile.BlockCount=%s\n", val));
			FILEINFO_ALLOC(0);
		} else if (0 == strcasecmp(var, "SplitFile.CheckBlockCount")) {
			mode = MODE_SPLITFILE;
			LOG(L_MINOR,("meta: SplitFile.CheckBlockCount=%s\n", val));
			FILECHECK_ALLOC(0);
		} else if (0 == strcasecmp(var, "SplitFile.Blocksize")) {
			LOG(L_MINOR,("meta: SplitFile.Blocksize=%s\n", val));
			FILECHECK_ALLOC(0);
			chunk_size = strtoul(val, NULL, 16);
		} else if (0 == strncmp(var, "SplitFile.Block.", 16)) {
			LOG(L_MINOR,("meta: %s=%s\n", var, val));
			i = strtoul(var + 16, NULL, 16);
			FILEINFO_ALLOC(1);
			fi->chk = strdup(val);
			if (NULL == fi->mimetype) {
				fi->mimetype = strdup("application/octet-stream");
			}
		} else if (0 == strncmp(var, "SplitFile.CheckBlock.", 21)) {
			LOG(L_MINOR,("meta: %s=%s\n", var, val));
			i = strtoul(var + 21, NULL, 16);
			FILECHECK_ALLOC(1);
			fc->chk = strdup(val);
			if (NULL == fc->mimetype) {
				fc->mimetype = strdup("application/octet-stream");
			}
		} else if (0 == strcasecmp(var, "EndPart")) {
			LOG(L_MINOR,("meta: EndPart\n"));
			if (NULL != fileinfo && NULL != fileinfo[fileinfo_cnt]) {
				fi = fileinfo[fileinfo_cnt];
				if (NULL == fi->mimetype) {
					fi->mimetype = strdup("application/octet-stream");
				}
				fileinfo_cnt++;
			}
			if (NULL != filecheck && NULL != filecheck[filecheck_cnt]) {
				filecheck_cnt++;
			}
		} else if (0 == strcasecmp(var, "End")) {
			LOG(L_MINOR,("meta: End\n"));
			if (NULL != fileinfo && NULL != fileinfo[fileinfo_cnt]) {
				fi = fileinfo[fileinfo_cnt];
				if (NULL == fi->mimetype) {
					fi->mimetype = strdup("application/octet-stream");
				}
				fileinfo_cnt++;
			}
			if (NULL != filecheck && NULL != filecheck[filecheck_cnt]) {
				filecheck_cnt++;
			}
		} else {
			LOG(L_ERROR,("Unhandled metadata tag: %s\n", var));
		}
	}

	switch (mode) {
	case MODE_REDIR:
		fcp_free(pf);
		if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
			printf("ERROR: connecting to %s:%d\n", fcphost, fcpport);
			goto bailout;
		}
		f = *pf;
		LOG(L_NORMAL,("Following redirect to:\n%s\n", fileinfo[0]->chk));
		rc = fcp_get(f, fileinfo[0]->chk, slimit, htl);
		break;

	case MODE_MAPSPACE:
		/* find the double slash in the name */
		name = strstr(uri, "//");
		if (NULL == name) {
			fprintf(stderr, "Use fcpgetsite to fetch an entire mapspace\n");
			goto bailout;
		} 
		/* find the matching name in the map space */
		for (i = 0; i < fileinfo_cnt; i++) {
			fi = fileinfo[i];
			/* check for default file */
			if (NULL == fi->name && 0 == strcmp(name, "//"))
				break;
			/* check for a matching name */
			if (NULL != fi->name &&
				0 == strcasecmp(name + 2, fi->name))
				break;
		}
		/* found a matching entry in the map space? */
		if (i < fileinfo_cnt) {
			char *chk;
			if (NULL == (chk = strdup(fi->chk))) {
				rc = -1;
				LOG(L_ERROR,("barf: strdup(%s) failed (%s)\n",
					fi->chk, strerror(errno)));
				goto bailout;
			}
			fcp_free(pf);
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				printf("ERROR: connecting to %s:%d\n", fcphost, fcpport);
				goto bailout;
			}
			f = *pf;
			rc = fcp_get(f, fi->chk, slimit, htl);
		} else {
			printf("ERROR: document not found in map space (%s)\n", name + 2);
			rc = -1;
		}
		break;

	case MODE_SPLITFILE:
		if (0 == fileinfo_max) {
			LOG(L_NORMAL,("Split file w/o chunks? I cannot load this type.\n"));
			break;
		}
		cmdline = calloc(MAXBUFF, 1);
		if (NULL == cmdline) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				MAXBUFF, 1, strerror(errno)));
			goto bailout;
		}
		tempfile = calloc(256, 1);
		if (NULL == tempfile) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				256, 1, strerror(errno)));
			goto bailout;
		}
		temppath = calloc(256, 1);
		if (NULL == temppath) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				256, 1, strerror(errno)));
			goto bailout;
		}
		if (0 == strcmp(filename, "-")) {
			strcpy(temppath, "stdout");
		} else {
			strcpy(temppath, filename);
		}
		dst = strrchr(temppath, '/');
		if (NULL == dst) {
			strcpy(tempfile, temppath);
			getcwd(temppath, 256);
		} else {
			*dst++ = '\0';
			strcpy(tempfile, dst);
		}

		if (NULL == algo) {
			LOG(L_NORMAL,("Loading old format split file with %d chunks\n",
				fileinfo_cnt));
			/* non-FEC split files (old format) start with a block # of 1 */
			first_block = 1;
			/* this is a old format split file */
			for (i = 0; i < fileinfo_cnt; i++) {
				fi = fileinfo[i];
				if (i + 1 == fileinfo_cnt) {
					fi->size = split_size - (fileinfo_cnt - 1) * chunk_size;
				} else {
					fi->size = chunk_size;
				}
			}
		} else {
			LOG(L_NORMAL,("Loading FEC format split file with %d chunks\n",
				fileinfo_cnt));
			/* FEC split files (new format) start with a block # of 0 */
			first_block = 0;
			/* this is a split file with FEC encoding algorithm */
			fcp_free(pf);
			if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
				printf("ERROR: connecting to %s:%d\n", fcphost, fcpport);
				goto bailout;
			}
			rc = fcp_segment_file(pf, fcphost, fcpport, algo, NULL, split_size);
			LOG(L_NORMAL,("%d bytes FEC segment file info:\n",
				split_size));
			LOG(L_NORMAL,("\ttotal segments:\t\t%d\n",
				f->sh[0].segments));
			LOG(L_NORMAL,("\tblock count/size:\t%d/%d\n",
				f->sh[0].block_count, f->sh[0].block_size));
			LOG(L_NORMAL,("\tcheck count/size:\t%d/%d\n",
				f->sh[0].check_block_count, f->sh[0].check_block_size));
			LOG(L_NORMAL,("\trequired blocks:\t%d\n",
				f->sh[0].blocks_required));
			/* set block sizes according to segment file info */
			for (i = 0; i < fileinfo_cnt; i++) {
				fi = fileinfo[i];
				fi->size = f->sh[0].block_size;
			}
		}

		start = fileinfo_cnt - 1;
		/* reset the hops to live and retry count */
		for (i = 0; i < fileinfo_cnt; i++) {
			fi = fileinfo[i];
			fi->htl = htl;
			fi->retry = 0;
			if (NULL == fi->name) {
				if (0 == strcmp(filename, "/dev/null")) {
					fi->name = strdup(filename);
				} else {
					snprintf(var, MAXBUFF, "%s/.in-%04x-%s",
						temppath, i+1, tempfile);
					fi->name = strdup(var);
				}
			}
		}
		for (;;) {
			FD_ZERO(&rdfds);
			FD_ZERO(&exfds);
			nfds = 0;
			minfd = 32767;
			maxfd = 0;
			for (i = 0; i < fileinfo_cnt; i++) {
				fi = fileinfo[i];
				if (NULL != fi->fp) {
					fd = fileno(fi->fp);
					FD_SET(fd, &rdfds);
					FD_SET(fd, &exfds);
					if (fd < minfd)
						minfd = fd;
					if (fd > maxfd)
						maxfd = fd;
					nfds += 1;
				}
			}
			for (i = (start + 1) % fileinfo_cnt;
				nfds < sf_threads && i != start;
				i = (i + 1) % fileinfo_cnt) {
				fi = fileinfo[i];
				if (NULL == fi->chk) {
					LOG(L_ERROR,("split file chunk #%d CHK@ is NULL\n",
						i));
					continue;
				}
				if (NULL != fi->fp)
					continue;
				if (0 == stat(fi->name, &st) && st.st_size == fi->size)
					continue;
				snprintf(cmdline, MAXBUFF,
					"\"%s\" %s -n%s -p%d -l%d -t%d -v%d %s%s \"%s\"",
					program, "get", fcphost, fcpport, fi->htl, sf_threads,
					verbose, delete ? "--delete " : "", fi->chk, fi->name);
				LOG(L_DEBUG,("cmdline: '%s'\n", cmdline));
				fi->fp = popen(cmdline, "r");
				if (NULL == fi->fp) {
					LOG(L_ERROR,("popen(\"%s\",\"%s\") failed (%s)\n",
						cmdline, "r", strerror(errno)));
					goto bailout;
				}
				LOG(L_NORMAL,("Requesting chunk #%d/%d %s (%s) with HTL:%d\n",
					i+1, fileinfo_cnt, infoname, fi->mimetype, fi->htl));
				fd = fileno(fi->fp);
				FD_SET(fd, &rdfds);
				FD_SET(fd, &exfds);
				if (fd < minfd)
					minfd = fd;
				if (fd > maxfd)
					maxfd = fd;
				nfds += 1;
				start = i;
			}

			if (0 == nfds) {
				break;
			}

			if (nfds > 0) {
				struct timeval tv = {1,0};

				if (-1 == select(maxfd+1, &rdfds, NULL, &exfds, &tv)) {
					LOG(L_ERROR,("select(%d,...) failed (%s)\n",
						maxfd+1, strerror(errno)));
					goto bailout;
				}
				for (fd = minfd; fd < maxfd+1; fd++) {
					for (i = 0; i < fileinfo_cnt; i++) {
						fi = fileinfo[i];
						if (NULL != fi &&
							NULL != fi->fp &&
							fd == fileno(fi->fp))
						break;
					}
					if (FD_ISSET(fd, &rdfds)) {
						char *reply = calloc(MAXBUFF, 1);
						char *eol, *result, *chk;
						if (NULL == reply) {
							LOG(L_ERROR,("calloc(%d,%d) failed (%s)\n",
								MAXBUFF, 1, strerror(errno)));
							goto bailout;
						}
						fgets(reply, MAXBUFF, fi->fp);
						LOG(L_DEBUG,("reply: \"%s\"\n", reply));
						eol = strtok(reply, "\r\n");
						if (NULL != eol) {
							result = strtok(eol, ":\r\n");
							chk = strtok(NULL, " \r\n");
						} else {
							result = NULL;
							chk = NULL;
						}
						if (NULL == result ||
							NULL == chk ||
							(0 != strcasecmp(result, SUCCESS) &&
							0 != strcasecmp(result, KEYCOLLISION))) {
							if (fi->htl < MAX_HTL) {
								fi->htl += 1;
							}
							fi->retry += 1;
							snprintf(reply, MAXBUFF, "RETRY #%d (HTL:%d)",
								fi->retry, fi->htl);
							result = reply;
							LOG(L_NORMAL,("%s: chunk #%d/%d of %s (%d, %s)\n",
								result, i+1, fileinfo_cnt, infoname, fi->size, fi->mimetype));
						} else {
							while (chk && *chk == ' ')
								chk++;
							LOG(L_NORMAL,("%s: chunk #%d/%d of %s (%d, %s)\n",
								result, i+1, fileinfo_cnt, infoname, fi->size, fi->mimetype));
							LOG(L_MINOR,("%s\n",
								chk));
						}
						free(reply);
						pclose(fi->fp);
						fi->fp = NULL;
					}
					if (FD_ISSET(fd, &exfds)) {
						if (NULL != fi->fp) {
							LOG(L_MINOR,("closing #%d %s\n",
								i+1, fi->name));
							pclose(fi->fp);
							fi->fp = NULL;
						}
					}
				}
			}
		}
		if (0 == strcmp(filename, "-")) {
			fp = stdout;
		} else {
			fp = fopen(filename, OPEN_WB);
		}
		if (NULL == fp) {
			LOG(L_ERROR,("fopen(\"%s\", \"%s\") call failed (%s)\n",
				filename, OPEN_WB, strerror(errno)));
			break;
		}
		for (i = 0, split_offs = 0; i < fileinfo_cnt; i++) {
			fi = fileinfo[i];
			if (NULL == fi->name) {
				LOG(L_ERROR,("FATAL: fileinfo[%d].name is NULL (%s)\n",
					i, fi->chk));
				continue;
			}
			fi->fp = fopen(fi->name, OPEN_RB);
			if (NULL == fi->fp) {
				LOG(L_ERROR,("failed to fopen(\"%s\",\"%s\") (%s)\n",
					fi->name, OPEN_RB, strerror(errno)));
				continue;
			} 
			while (split_offs < split_size && !feof(fi->fp)) {
				size_t done = fread(var, 1, MAXBUFF, fi->fp);
				/* clip output size to specified split_size */
				if (split_offs + done > split_size)
					done = split_size - split_offs;
				if (done != fwrite(var, 1, done, fp)) {
					LOG(L_ERROR,("failed to write \"%s\" (%s)\n",
						filename, strerror(errno)));
					fclose(fi->fp);
					fi->fp = NULL;
					goto bailout;
				}
				split_offs += done;
			}
			fclose(fi->fp);
			fi->fp = NULL;
			unlink(fi->name);
		}
		if (fp != stdout) {
			fclose(fp);
		}
		fp = NULL;
		rc = 0;
		break;

	case MODE_DATEBASEDREDIR:
		if (0 == date_incr) {
			date_incr = 86400;
		}
		gettimeofday(&tv, NULL);
		tm = localtime((const time_t *)&tv.tv_sec);
		date = timegm(tm);
		date = date - (date % date_incr);
		size = MAXPATHLEN;
		if (NULL == (ssk = calloc(size, 1))) {
			LOG(L_ERROR,("barf: calloc(%d,%d) failed (%s)\n",
				size, 1, strerror(errno)));
			goto bailout;
		}
		strcpy(ssk, fileinfo[0]->chk);
		slash = strstr(uri, "//");
		if (NULL == slash) {
			LOG(L_ERROR,("FATAL: no double slash in request URI (%s)\n",
				uri));
			goto bailout;
		}
		strcat(ssk, slash);
		slash = strchr(ssk, '/');
		if (NULL == slash) {
			LOG(L_ERROR,("FATAL: no slash in date based redirect (%s)\n",
				ssk));
			goto bailout;
		}
		/* make room for 8 hex digits and a dash */
		memmove(slash + 8 + 1, slash, strlen(slash) + 2);
		snprintf(slash + 1, 8 + 1, "%08x", (unsigned)date);
		slash[8 + 1] = '-';

		printf("**** %s\n", ssk);

		fcp_free(pf);
		if (0 != (rc = fcp_new(pf, fcphost, fcpport))) {
			printf("ERROR: connecting to %s:%d\n", fcphost, fcpport);
			goto bailout;
		}
		f = *pf;
		LOG(L_NORMAL,("Following date based redirect to:\n%s\n", ssk));
		rc = fcp_get(f, ssk, slimit, htl);
		/* successful but again metadata only? */
		if (0 == rc && NULL == f->data) {
			rc = get_file(pf, ssk, slimit, htl, wiggle, filename, infoname);
		}
		break;
	default:
		LOG(L_NORMAL,("invalid metadata?\n"));
		if (0 != strcmp(filename, "-")) {
			fp = fopen(filename, OPEN_WB);
			fclose(fp);
			fp = NULL;
		}
		rc = 0;
	}

bailout:
	if (NULL != var) {
		free(var);
		var = NULL;
	}
	if (NULL != val) {
		free(val);
		val = NULL;
	}
	if (NULL != ssk) {
		free(ssk);
		ssk = NULL;
	}
	if (NULL != cmdline) {
		free(cmdline);
		cmdline = NULL;
	}
	if (NULL != temppath) {
		free(temppath);
		temppath = NULL;
	}
	if (NULL != tempfile) {
		free(tempfile);
		tempfile = NULL;
	}
	if (NULL != fp) {
		fclose(fp);
		fp = NULL;
	}
	if (NULL != fileinfo) {
		for (i = 0; i < fileinfo_cnt; i++) {
			fi = fileinfo[i];
			if (NULL != fi) {
				if (NULL != fi->name) {
					free(fi->name);
					fi->name = NULL;
				}
				if (NULL != fi->mimetype) {
					free(fi->mimetype);
					fi->mimetype = NULL;
				}
				if (NULL != fi->chk) {
					free(fi->chk);
					fi->chk = NULL;
				}
				if (NULL != fi->fp) {
					pclose(fi->fp);
					fi->fp = NULL;
				}
				free(fileinfo[i]);
				fileinfo[i] = NULL;
			}
		}
		free(fileinfo);
		fileinfo = NULL;
	}
	if (NULL != filecheck) {
		for (i = 0; i < filecheck_cnt; i++) {
			fc = filecheck[i];
			if (NULL != fc) {
				if (NULL != fc->name) {
					free(fc->name);
					fc->name = NULL;
				}
				if (NULL != fc->mimetype) {
					free(fc->mimetype);
					fc->mimetype = NULL;
				}
				if (NULL != fc->chk) {
					free(fc->chk);
					fc->chk = NULL;
				}
				if (NULL != fc->fp) {
					pclose(fc->fp);
					fc->fp = NULL;
				}
				free(filecheck[i]);
				filecheck[i] = NULL;
			}
		}
		free(filecheck);
		filecheck = NULL;
	}
	return rc;
}
