/*****************************************************************************
 *  FTools - Freenet Client Protocol Tools
 *
 *  Copyright (C) 2002 Juergen Buchmueller <juergen@pullmoll.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *	$Id: ftmain.c,v 1.77 2005/07/19 13:52:42 pullmoll Exp $
 *****************************************************************************/
#include "ft.h"
#include "ftsite.h"
#include "ftfile.h"
#include "ftmime.h"
#include "ftfcp.h"
#include "ftlog.h"

enum {
	CMD_FT,
	CMD_HELLO,
	CMD_KEY,
	CMD_CHK,
	CMD_INV,
	CMD_FEC,
	CMD_GET,
	CMD_PUT,
	CMD_GETSITE,
	CMD_PUTSITE,
	CMD_BROADCAST,
	CMD_RECEIVE
}	e_cmd;

char *homepath = NULL;
char *confpath = NULL;
char *temppath = NULL;
char *progname = NULL;
char *progpath = NULL;
char *program = NULL;

#ifndef	VER_MAJ
#define	VER_MAJ	0
#endif
#ifndef	VER_MIN
#define	VER_MIN	2
#endif
#ifndef	VER_BUILD
#define	VER_BUILD	25
#endif

int version = (VER_MAJ << 24) | (VER_MIN << 16) | VER_BUILD;
int verbose = L_NORMAL;
int retries = 5;
char fcphost[256] = "127.0.0.1";
int fcpport = 8482;
int htl = 5;
int wiggle = 0;
int revision = 1;
int limit32k = 0;
size_t chunksize = 1024*1024*1024;	/* 1GB chunks */
// size_t chunksize = 262144;	/* 256KB chunks */
char mimetype[256] = "auto";
char mimeignore[4096] =  "";
int raw = 0;
int sf_threads = 3;
int in_threads = 5;
char default_file[256] = "index.html";
int date_offs = 0;
int date_incr = 86400;
int date_future = 1;
int meta_only = 0;
int meta_none = 0;
int slimit = 0;
int delete = 0;

int cmd_lookup(const char *cmdstr)
{
	int cmd = CMD_FT;
	if (NULL == cmdstr) {
		return cmd;
	}
	if (0 == strcasecmp(cmdstr, "hello")) {
		cmd = CMD_HELLO;
	} else if (0 == strcasecmp(cmdstr, "chk")) {
		cmd = CMD_CHK;
	} else if (0 == strcasecmp(cmdstr, "inv")) {
		cmd = CMD_INV;
	} else if (0 == strcasecmp(cmdstr, "fec")) {
		cmd = CMD_FEC;
	} else if (0 == strcasecmp(cmdstr, "key")) {
		cmd = CMD_KEY;
	} else if (0 == strcasecmp(cmdstr, "get")) {
		cmd = CMD_GET;
	} else if (0 == strcasecmp(cmdstr, "put")) {
		cmd = CMD_PUT;
	} else if (0 == strcasecmp(cmdstr, "getsite")) {
		cmd = CMD_GETSITE;
	} else if (0 == strcasecmp(cmdstr, "putsite")) {
		cmd = CMD_PUTSITE;
	} else if (0 == strcasecmp(cmdstr, "broadcast")) {
		cmd = CMD_BROADCAST;
	} else if (0 == strcasecmp(cmdstr, "receive")) {
		cmd = CMD_RECEIVE;
	}
	return cmd;
}

void usage(int cmd)
{
	switch (cmd) {
	case CMD_HELLO:
		printf("usage: %s [options]\n", progname);
		break;
	case CMD_KEY:
		printf("usage: %s [options]\n", progname);
		break;
	case CMD_CHK:
		printf("usage: %s [options] file\n", progname);
		break;
	case CMD_INV:
		printf("usage: %s [options] private SSK\n", progname);
		break;
	case CMD_FEC:
		printf("usage: %s [options] file\n", progname);
		break;
	case CMD_GET:
		printf("usage: %s [options] URI [file]\n", progname);
		break;
	case CMD_PUT:
		printf("usage: %s [options] URI file\n", progname);
		break;
	case CMD_GETSITE:
		printf("usage: %s [options] SSK@<pubkey>/<site> dstdir\n",
			progname);
		printf("<key>\t\tpublic key of a freesite\n");
		printf("<site>\t\tbase name of this site\n");
		break;
	case CMD_PUTSITE:
		printf("usage: %s [options] SSK@<seckey>/<site> srcdir\n",
			progname);
		printf("<key>\t\tsecret (private) key obtained by e.g. 'fcpkey'\n");
		printf("<site>\t\tbase name of this site\n");
		break;
	case CMD_BROADCAST:
		printf("usage: %s [options] channel sender payload\n",
			progname);
		printf("channel\t\tTo which channel to send the broadcast\n");
		printf("sender\t\tWho is the sender of this broadcast\n");
		printf("payload\t\tPayload of the broadcast (max 936 characters)\n");
		break;
	case CMD_RECEIVE:
		printf("usage: %s [options] id\n",
			progname);
		printf("id\t\tReturn the broadcast with ID 'id', 0 returns min/max\n");
		break;
	default:
		printf("usage: %s [options] <command> [URI] [file|dir]\n",
			progname);
		printf("<command> can be one of:\n");
		printf("hello\t\t\t\tcontact node and print NodeHello info\n");
		printf("key\t\t\t\tgenerate a private/public key pair\n");
		printf("chk file\t\t\tcalculate CHK@ for file\n");
		printf("put URI file\t\t\tstore file under the given URI\n");
		printf("get URI [file]\t\t\tretrieve URI [and store to file]\n");
		printf("get <key>/file\t\t\tretrieve CHK@ key and store to file\n");
		printf("getsite SSK@<key>/<site> dstdir\t" \
			"retrieve site tree to destination directory\n");
		printf("putsite SSK@<key>/<site> srcdir\t" \
			"insert source directory tree as site\n");
	}
	printf("\nparameter description:\n");
	printf("URI\t\t\tCHK@, [KSK@]somename or SSK@<key> " \
			"(key = private key\n" \
			"\t\t\t\tfor inserts; key = public key for requests)\n");
	printf("file\t\t\t[/some/path/]filename[.ext]\n");
	printf("\n[options] can be zero or more of:\n");
	printf("-V|--version \t\tVersion\n");
	printf("-v|--verbose <n>\t0 to 5 for none,normal,minor,... (def. %d)\n",
		verbose);
	printf("-n|--address <addr>\tset FCP server address (def. %s)\n",
		fcphost);
	printf("-p|--port <port>\tset FCP port (def. %d)\n",
		fcpport);
	printf("-l|--htl <n>\t\tset Hops To Live (def. %d)\n",
		htl);
	if (cmd == CMD_FT || cmd == CMD_PUT || cmd == CMD_PUTSITE) {
		printf("-r|--revision <n>\t" \
			"set Revision= meta data (def. %d) DONT CHANGE!\n",
			revision);
		printf("-s|--size <bytes>\t" \
			"set split file size (def. 0x%x)\n",
			chunksize);
		printf("-t|--threads <n>\t" \
			"split file threads (def. %d)\n",
			sf_threads);
		printf("-F|--freenet\t\t" \
			"Freenet compatibility (max. 32K SSK@/KSK@)\n");
		printf("--meta-none\t\t" \
			"do not insert any metadata (def. %s)\n",
			meta_none ? "on" : "off");
	}
	if (cmd == CMD_FT || cmd == CMD_PUTSITE) {
		printf("-i|--insert-threads <n>\t" \
			"number of insert threads (def. %d)\n",
			in_threads);
		printf("-D|--default <n>\t" \
			"default document name (%s)\n",
			default_file);
		printf("-d|--no-dbr\t\t" \
			"insert no date based redirect (DBR)\n");
		printf("-f|--future <n>\t\t" \
			"insert n days (time units) in the future (def. %d)\n",
			date_future);
		printf("--meta-only\t\t" \
			"do not insert data but metadata only (def. %s)\n",
			meta_only ? "on" : "off");
		printf("--delete\t\t" \
			"delete keys from the local store (def. %s)\n",
			delete ? "on" : "off");
	}
	if (cmd == CMD_FT || cmd == CMD_GET) {
		printf("-m|--max-size <n>\t" \
			"maximum file size to retrieve (def.: any)\n");
		printf("-R|--raw\t\t" \
			"raw mode: don't follow redirect(s), return meta data\n");
	}
}

#if	defined(__CYGWIN__) || defined(__INTERIX)
/* Replacement timegm() function for libc's missing it */
time_t timegm(struct tm *tm)
{
	time_t time_temp;
	struct tm *tm_temp;
	

	if (NULL == tm) {
		errno = EINVAL;
		return -1;
	}
	time_temp = mktime(tm);
	tm_temp = gmtime(&time_temp);
	return (time_t)(time_temp + (time_temp - mktime(tm_temp)));
}
#endif

int results(fcp_t *f, int rc, const char *uri)
{
	size_t size;
	char *info = NULL;
	const char *u = f && f->uri ? f->uri : uri;

	if (NULL != f) {
		if (NULL != f->reason) {
			size = 32 + strlen(f->reason) + 32 + strlen(u);
			info = calloc(size + 1, sizeof(char));
			snprintf(info, size, "%s\nReason: %s\n", u, f->reason);
		} else {
			size = 32 + strlen(u);
			info = calloc(size + 1, sizeof(char));
			snprintf(info, size, "%s\n", u);
		}

		if (0 == rc) {
			if (f->seen_key_collision) {
				printf(KEYCOLLISION ": %s", info);
			} else {
				printf(SUCCESS ": %s", info);
			}
		} else {
			if (f->seen_data_not_found) {
				printf(DATANOTFOUND ": %s", info);
			} else if (f->seen_route_not_found) {
				printf(ROUTENOTFOUND ": %s", info);
			} else if (f->seen_uri_error) {
				printf(URIERROR ": %s", info);
			} else if (f->seen_size_error) {
				printf(SIZEERROR ": %s", info);
			} else {
				printf(GENERALERROR ": %s", info);
			}
		}
	} else {
		printf(UNKNOWNERROR ": %s\n", u);
	}
	free(info);
	return -1;
}

int version_info(void)
{
	printf("Freenet Tools version %d.%d.%d\n",
		version >> 24, (version >> 16) % 256, version % 65536);
	return 0;
}

int path_find(char **progpath, char *progname, char *paths)
{
	char *prog = calloc(256, sizeof(char));
	char *dupe = strdup(paths);
	char *path, *slash;
	struct stat st;

	path = strtok(dupe, ":");
	while (NULL != path) {
		if (0 == strncmp(path, "~/", 2))
			snprintf(prog, 256, "%s/%s/%s",
				homepath, path+2, progname);
		else
			snprintf(prog, 256, "%s/%s",
				path, progname);
		if (0 == stat(prog, &st) &&
			(S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))) {
			slash = strrchr(prog, '/');
			if (NULL != slash)
				*slash = '\0';
			*progpath = prog;
			free(dupe);
			return 0;
		}
		path = strtok(NULL, ":");
	}
	free(prog);
	free(dupe);
	return -1;
}

int conf_setup(void)
{
	struct stat st;
	size_t size;

	if (NULL != getenv("HOME")) {
		homepath = strdup(getenv("HOME"));
		if (NULL == homepath) {
			perror("strdup()");
			return -1;
		}
	} else {
		homepath = calloc(256, sizeof(char));
		if (NULL == homepath) {
			perror("calloc()");
			return -1;
		}
		if (0 != getcwd(homepath, 256)) {
			perror("getcwd()");
			return -1;
		}
	}
	size = strlen(homepath) + 1 + strlen(".ft") + 1;
	confpath = calloc(size, sizeof(char));
	if (NULL == confpath) {
		perror("calloc()");
		return -1;
	}
	snprintf(confpath, size, "%s/.ft", homepath);
	if (0 != stat(confpath, &st)) {
		if (0 != mkdir(confpath, 0700)) {
			perror("mkdir(~/.ft)");
			return -1;
		}
	} else if (!S_ISDIR(st.st_mode)) {
		fprintf(stderr, "~/.ft is not a directory?\n");
		return -1;
	}
	if (0 == access("/tmp", 0700)) {
		temppath = strdup("/tmp");
		if (NULL == temppath) {
			perror("strdup()");
			return -1;
		}
	} else if (0 == access("/var/tmp", 0700)) {
		temppath = strdup("/var/tmp");
		if (NULL == temppath) {
			perror("strdup()");
			return -1;
		}
	} else {
		size = strlen(confpath) + 1 + strlen("tmp") + 1;
		temppath = calloc(size, sizeof(char));
		if (NULL == temppath) {
			perror("calloc()");
			return -1;
		}
		snprintf(temppath, size, "%s/tmp", confpath);
		if (0 != mkdir(temppath, 0700) && errno != EEXIST) {
			perror("mkdir(~/.ft/tmp)");
			return -1;
		}
	}
	return 0;
}

int main(int argc, char **argv)
{
	fcp_t *f;
	int cmd = CMD_FT;
	char *cmdstr = NULL;
	char *uri = NULL, *channel = NULL;
	char *filename = NULL, *sender = NULL;
	char *infoname = NULL, *payload = NULL;
	char *prg, *ext;
	int i, options, len;

	if (0 != conf_setup())
		return -1;

	progpath = strdup(argv[0]);
	prg = strrchr(progpath, '/');
	if (NULL != prg) {
		*prg++ = '\0';
		progname = prg;
	} else {
		progname = strdup(argv[0]);
		free(progpath);
		if (0 != path_find(&progpath, progname, getenv("PATH"))) {
			fprintf(stderr, "Cannot find path to executable %s\n", progname);
			exit(1);
		}
	}
	ext = strrchr(progname, '.');
	if (NULL == ext) {
		ext = "";
	}

	/* Full pathname for the main executable /somewhere/ft(.exe) */
	len = strlen(progpath) + 1 + 2 + strlen(ext) + 1;
	program = calloc(len, sizeof(char));
	if (NULL == program) {
		perror("calloc()");
		return -1;
	}
	snprintf(program, len, "%s/ft%s",
		progpath, ext);

	if (0 == strcasecmp(progname, "fcphello") ||
		0 == strcasecmp(progname, "fcphello.exe")) {
		cmdstr = "hello";
	} else if (0 == strcasecmp(progname, "fcpchk") ||
		0 == strcasecmp(progname, "fcpchk.exe")) {
		cmdstr = "chk";
	} else if (0 == strcasecmp(progname, "fcpinv") ||
		0 == strcasecmp(progname, "fcpinv.exe")) {
		cmdstr = "inv";
	} else if (0 == strcasecmp(progname, "fcpfec") ||
		0 == strcasecmp(progname, "fcpfec.exe")) {
		cmdstr = "fec";
	} else if (0 == strcasecmp(progname, "fcpkey") ||
		0 == strcasecmp(progname, "fcpkey.exe")) {
		cmdstr = "key";
	} else if (0 == strcasecmp(progname, "fcpget") ||
		0 == strcasecmp(progname, "fcpget.exe")) {
		cmdstr = "get";
	} else if (0 == strcasecmp(progname, "fcpput") ||
		0 == strcasecmp(progname, "fcpput.exe")) {
		cmdstr = "put";
	} else if (0 == strcasecmp(progname, "fcpgetsite") ||
		0 == strcasecmp(progname, "fcpgetsite.exe")) {
		cmdstr = "getsite";
	} else if (0 == strcasecmp(progname, "fcpputsite") ||
		0 == strcasecmp(progname, "fcpputsite.exe")) {
		cmdstr = "putsite";
	} else if (0 == strcasecmp(progname, "fcpbroadcast") ||
		0 == strcasecmp(progname, "fcpbroadcast.exe")) {
		cmdstr = "broadcast";
	} else if (0 == strcasecmp(progname, "fcpreceive") ||
		0 == strcasecmp(progname, "fcpreceive.exe")) {
		cmdstr = "receive";
	}

	/* translate command string */
	cmd = cmd_lookup(cmdstr);

	for (i = 1, options = 1; i < argc; i++) {
		if (argv[i][0] == '-' && argv[i][1] == '-' && argv[i][2] == '\0') {
			/* double dash terminates options */
			options = 0;
		} else if (0 != options && argv[i][0] == '-' && argv[i][1] != '\0') {
#undef	OPT
#define	OPT	"-n"
			if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					strcpy(fcphost, &argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					strcpy(fcphost, argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--address"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					strcpy(fcphost, &argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					strcpy(fcphost, argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-p"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					fcpport = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					fcpport = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--port"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					fcpport = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					fcpport = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-v"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					verbose = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					verbose = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--verbose"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					verbose = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					verbose = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-i"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					in_threads = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					in_threads = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--insert-threads"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					in_threads = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					in_threads = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-t"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					sf_threads = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					sf_threads = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--threads"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					sf_threads = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					sf_threads = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-s"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					chunksize = strtoul(&argv[i][sizeof(OPT) - 1], NULL, 0);
				} else if (i + 1 < argc) {
					i++;
					chunksize = strtoul(argv[i], NULL, 0);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--size"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					chunksize = strtoul(&argv[i][sizeof(OPT) - 1], NULL, 0);
				} else if (i + 1 < argc) {
					i++;
					chunksize = strtoul(argv[i], NULL, 0);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-m"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					slimit = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					slimit = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--max-size"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					slimit = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					slimit = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-F"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				limit32k = 1;
#undef	OPT
#define	OPT	"--freenet"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				limit32k = 1;
#undef	OPT
#define	OPT	"-l"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					htl = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					htl = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--htl"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					htl = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					htl = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-w"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					wiggle = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					wiggle = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--wiggle"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					wiggle = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					wiggle = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-r"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					revision = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					revision = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--revision"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					revision = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					revision = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-d"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				date_incr = 0;
#undef	OPT
#define	OPT	"--no-dbr"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				date_incr = 0;
#undef	OPT
#define	OPT	"--date-offs"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					date_offs = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					date_offs = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--date-incr"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					date_incr = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					date_incr = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-f"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					date_future = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					date_future = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--future"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					date_future = atoi(&argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					date_future = atoi(argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-D"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					strncpy(default_file, &argv[i][sizeof(OPT) - 1],
						sizeof(default_file));
				} else if (i + 1 < argc) {
					i++;
					strncpy(default_file, argv[i], sizeof(default_file));
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--default"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					strncpy(default_file, &argv[i][sizeof(OPT) - 1],
						sizeof(default_file));
				} else if (i + 1 < argc) {
					i++;
					strncpy(default_file, argv[i], sizeof(default_file));
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-I"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					strcpy(mimeignore, &argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					strcpy(mimeignore, argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"--ignore"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				if (strlen(argv[i]) > sizeof(OPT) - 1) {
					strcpy(mimeignore, &argv[i][sizeof(OPT) - 1]);
				} else if (i + 1 < argc) {
					i++;
					strcpy(mimeignore, argv[i]);
				} else {
					fprintf(stderr, "missing argument for %s\n", OPT);
					usage(cmd);
					return 1;
				}
#undef	OPT
#define	OPT	"-R"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				raw = 1;
#undef	OPT
#define	OPT	"--raw"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				raw = 1;
#undef	OPT
#define	OPT	"-V"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				version_info();
				exit(0);
#undef	OPT
#define	OPT	"--version"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				version_info();
				exit(0);
#undef	OPT
#define	OPT	"--meta-none"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				meta_none = 1;
#undef	OPT
#define	OPT	"--meta-only"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				meta_only = 1;
#undef	OPT
#define	OPT	"--delete"
			} else if (0 == strncmp(argv[i], OPT, sizeof(OPT) - 1)) {
				delete = 1;
#undef	OPT
			} else {
				fprintf(stderr, "unknown switch '%s'\n",
					argv[i]);
				usage(cmd);
				return 1;
			}
		} else if (NULL == cmdstr) {
			cmdstr = argv[i];
		} else if (NULL == uri) {
			uri = channel = argv[i];
		} else if (NULL == filename) {
			filename = sender = argv[i];
		} else if (NULL == infoname) {
			infoname = payload = argv[i];
		}
	}

	/* No command string specified: display usage info */
	if (NULL == cmdstr) {
		usage(cmd);
		return 1;
	}

	/* translate command string */
	cmd = cmd_lookup(cmdstr);

	srand(time(NULL));

	/* No URI and not hello or key commands: display usage info */
	if (NULL == uri && cmd != CMD_HELLO && cmd != CMD_KEY) {
		usage(cmd);
		return 1;
	}
	if (NULL == filename) {
		filename = "/dev/null";
	}
	/* No informational name given: use the uri */
	if (NULL == infoname) {
		infoname = uri;
	}

	if (0 != fcp_new(&f, fcphost, fcpport)) {
		printf(GENERALERROR ": connecting to %s:%d\n", fcphost, fcpport);
		return 1;
	}

	switch (cmd) {
	case CMD_HELLO:
		{
			int rc = fcp_hello(f);

			if (0 == rc) {
				printf("Node Protocol : %s\n", f->protocol);
				printf("Node Version  : %s\n", f->node);
			}
		}
		break;

	case CMD_GET:
		{
			char *src;
			int rc;

			/* Skip first part if the user gave a http://localhost:9999/... link */
			if (0 == strncmp(uri, "http://", 7)) {
				/* find next slash after hostname:port */
				src = strchr(uri + 7, '/');
				if (NULL != src)
					strcpy(uri, src + 1);
			}
			rc = fcp_get(f, uri, slimit, htl);

			/* check if the user specified the CHK@xxx,yyy/filename.ext format */
			if (0 == strcmp(filename, "/dev/null") &&
				0 == strncmp(uri, "CHK@", 4) &&
				NULL != (src = strchr(uri, '/'))) {
				/* terminate URI here and set filename to the remainder */
				*src++ = '\0';
				filename = src;
			}

			/* successful and not raw mode? */
			if (0 == rc && 0 == raw) {

				/* if we have no data, the metadata should tell us what to do */
				if (0 == raw && NULL == f->data) {
					rc = get_file(&f, uri, slimit, htl, wiggle, filename, infoname);
					if (0 != rc) {
						return results(f, rc, uri);
					}
				}

				/* now we should have data (or it was already output) */
				if (NULL != f->data) {
					FILE *fp;

					printf(SUCCESS ": %s\n", f->uri ? f->uri : uri);
					if (0 == strcmp(filename, "-")) {
						fp = stdout;
					} else {
						fp = fopen(filename, "wb");
					}
					if (NULL == fp) {
						fprintf(stderr, "cannot create '%s' (%s)\n",
							filename, strerror(errno));
						return 1;
					}
					if (f->datasize != fwrite(f->data, 1, f->datasize, fp)) {
						fprintf(stderr, "cannot write %d bytes to '%s' (%s)\n",
							f->datasize, filename, strerror(errno));
						return 1;
					}
					if (fp != stdout) {
						fclose(fp);
					}
					fp = NULL;

				} else {

					rc = results(f, rc, uri);

				}

			} else if (0 == rc && 0 != raw) {

				/* we should have metadata */
				if (NULL != f->meta) {
					FILE *fp;

					printf(SUCCESS ": %s\n", f->uri ? f->uri : uri);
					if (0 == strcmp(filename, "-")) {
						fp = stdout;
					} else {
						fp = fopen(filename, "wb");
					}
					if (NULL == fp) {
						fprintf(stderr, "cannot create '%s' (%s)\n",
							filename, strerror(errno));
						return 1;
					}
					if (f->metasize != fwrite(f->meta, 1, f->metasize, fp)) {
						fprintf(stderr, "cannot write %d bytes to '%s' (%s)\n",
							f->metasize, filename, strerror(errno));
						return 1;
					}
					if (fp != stdout) {
						fclose(fp);
					}
					fp = NULL;

				} else {
					printf(GENERALERROR ": %s\n", f->uri ? f->uri : uri);
				}

			} else {
				rc = results(f, rc, uri);
			}
		}
		break;


	case CMD_PUT:
		{
			int rc = put_file(&f, uri, htl, wiggle, filename,
				(strcasecmp(mimetype, "auto")) ?
					mimetype : mimetype_guess(filename));

			rc = results(f, rc, uri);
		}
		break;

	case CMD_CHK:
		{
			int rc;

			/* we don't need no URI for generate CHK */
			if (0 == strcmp(filename, "/dev/null") && NULL != uri) {
				filename = uri;
				uri = "CHK@";
			}

			rc = chk_file(&f, filename,
				(strcasecmp(mimetype, "auto")) ?
					mimetype : mimetype_guess(filename));

			rc = results(f, rc, uri);
		}
		break;

	case CMD_INV:
		{
			int rc;
			char pub[128];

			rc = fcp_inv(f, uri, pub);

			rc = results(f, rc, pub);
		}
		break;

	case CMD_FEC:
		{
			int rc;

			/* we don't need no URI for generate FEC */
			if (0 == strcmp(filename, "/dev/null") && NULL != uri) {
				filename = uri;
				uri = "CHK@";
			}

			rc = fec_file(&f, filename);
		}
		break;

	case CMD_KEY:
		{
			int rc = fcp_key(f);
			if (0 == rc) {
				printf("Public Key  : %s\n", f->public_key);
				printf("Private Key : %s\n", f->private_key);
			}
		}
		break;

	case CMD_GETSITE:
		{
			char *src;
			int rc;

			/* Skip first part if the user gave a http://localhost:9999/... link */
			if (0 == strncmp(uri, "http://", 7)) {
				/* find next slash after hostname:port */
				src = strchr(uri + 7, '/');
				if (NULL != src)
					strcpy(uri, src + 1);
			}
			/* Strip off a trailing double slash */
			src = uri + strlen(uri) - 2;
			if (src > uri && 0 == strcmp(src, "//")) {
				*src = '\0';
			}

			rc = fcp_get(f, uri, slimit, htl);
			rc = get_site(&f, uri, htl, wiggle, filename);
			rc = results(f, rc, uri);
		}
		break;

	case CMD_PUTSITE:
		{
			int rc;

			rc = put_site(&f, uri, htl, wiggle, filename);
			rc = results(f, rc, uri);
		}
		break;

	case CMD_BROADCAST:
		{
			int rc;

			if (NULL == channel) {
				printf(GENERALERROR ": %s\n", "missing channel");
				break;
			}
			if (NULL == sender) {
				printf(GENERALERROR ": %s\n", "missing sender");
				break;
			}
			if (NULL == payload) {
				printf(GENERALERROR ": %s\n", "missing payload");
				break;
			}
			rc = fcp_broadcast(f, htl, channel, sender, payload);
			rc = results(f, rc, "broadcast");
		}
		break;

	case CMD_RECEIVE:
		{
			int rc, id = 0;
			if (NULL != channel)
				id = strtoul(channel, NULL, 0);
			rc = fcp_receive(f, id);
			if (0 == rc) {
				printf(SUCCESS ": %d\n", id);
				if (0 == id) {
					printf("MinID: %d\n", f->min_id);
					printf("MaxID: %d\n", f->max_id);
				} else {
					printf("SHA1: %s\n", f->sha1);
					printf("Time: %x\n", (unsigned)f->time);
					printf("Channel: %s\n", f->channel);
					printf("Sender: %s\n", f->sender);
					printf("Payload: %s\n", f->payload);
				}
			} else {
				rc = results(f, rc, "channel");
			}
		}
	}

	fcp_free(&f);
	return 0;
}
