/*****************************************************************************
 *  FTools - Freenet Client Protocol Tools
 *
 *  Copyright (C) 2002 Juergen Buchmueller <juergen@pullmoll.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *	$Id: ftsock.c,v 1.14 2003/11/04 23:15:36 pullmoll Exp $
 *****************************************************************************/
#include "ftsock.h"
#include "ftlog.h"

#define	RW_SIZE	4096

int fork2(void)
{
	pid_t pid;
	int status;

	if (0 == (pid = fork())) {
		switch (fork()) {
		case 0: return 0;
		case -1: _exit(errno);
		default: _exit(0);
		}
	}

	if (pid < 0 || waitpid(pid, &status, 0) < 0) {
		return -1;
	}

	if (WIFEXITED(status)) {
		if (0 == WEXITSTATUS(status)) {
			return 1;
		} else {
			errno = WEXITSTATUS(status);
		}
	} else {
		errno = EINTR;
	}
	return -1;
}

int sk_outgoing(conn_t *conn, const char *hostname, int port)
{
	struct in_addr addr;
	struct hostent *host;
	int sk;

	addr.s_addr = inet_addr(hostname);
	if (addr.s_addr == (u_int32_t)-1) {
		host = gethostbyname(hostname);
		if (NULL == host) {
			return -1;
		}
		addr.s_addr = ((struct in_addr *) *host->h_addr_list)->s_addr;
	}

	memset((char *)&conn->addr, 0, sizeof(conn->addr));
	conn->sk = -1;
	conn->addr.sin_family = AF_INET;
	conn->addr.sin_port = htons(port);
	conn->addr.sin_addr.s_addr = addr.s_addr;

	sk = socket(AF_INET, SOCK_STREAM, 0);
	if (sk < 0) {
		return -1;
	}

	if (connect(sk, (struct sockaddr *)&conn->addr, sizeof(conn->addr)) < 0) {
		close(sk);
		return -1;
	}
#if	0
	if (fcntl(sk, F_SETFL, O_NONBLOCK) < 0) {
		close(sk);
		return -1;
	}
#endif
	conn->sk = sk;
	return 0;
}

int sk_incoming(const char *hostname, int port, int (*child)(conn_t *conn))
{
	struct in_addr addr;
	struct hostent *host;
	struct sockaddr_in in;
	int sk, arg = 1;

	addr.s_addr = inet_addr(hostname);
	if (addr.s_addr == (u_int32_t)-1) {
		host = gethostbyname(hostname);
		if (NULL == host) {
			return -1;
		}
		addr.s_addr = ((struct in_addr *) *host->h_addr_list)->s_addr;
	}

	memset((char *)&in, 0, sizeof(in));
	in.sin_family = AF_INET;
	in.sin_port = htons(port);
	in.sin_addr.s_addr = addr.s_addr;

	sk = socket(AF_INET, SOCK_STREAM, 0);
	if (sk < 0) {
		return -1;
	}

	setsockopt(sk, SOL_SOCKET, SO_REUSEADDR, (char *)&arg, sizeof(arg));

	if (bind(sk, (struct sockaddr *)&in, sizeof(in)) < 0) {
		close(sk);
		return -1;
	}

	if (0 != listen(sk, 5)) {
		close(sk);
		return -1;
	}

	for (;;) {
		int sk2, rc;

		sk2 = -1;
		while (sk2 < 0) {
			conn_t connection, *conn = &connection;
			socklen_t socklen = sizeof(conn->addr);

			sk2 = accept(sk, (struct sockaddr *)&conn->addr, &socklen);
			if (-1 == sk2) {
				continue;
			}
			conn->sk = sk2;

			switch (fork2()) {
			case -1:
				return -1;
			case 0:
				close(sk);
				rc = (*child)(conn);
				sk_shut(conn);
				_exit(rc);
			default:
				sk_shut(conn);
			}
		}
	}
	close(sk);
	return 0;
}

int sk_write(conn_t *conn, const void *buffer, size_t size)
{
	const char *buff = (const char *)buffer;
	size_t offs;

	for (offs = 0; offs < size; /* */) {
		ssize_t count;
		size_t limit = (size - offs) > RW_SIZE ? RW_SIZE : size - offs;
		struct timeval tv;
		fd_set wrfds, exfds;
		int sk;

		sk = conn->sk;
		if (-1 == conn->sk) {
			return -1;
		}

		FD_ZERO(&wrfds);
		FD_SET(sk, &wrfds);
		FD_ZERO(&exfds);
		FD_SET(sk, &exfds);
		tv.tv_sec = 60;
		tv.tv_usec = 0;
		if (-1 == select(sk+1, NULL, &wrfds, &exfds, &tv)) {
			sk_shut(conn);
			return -1;
		} else if (FD_ISSET(sk, &exfds)) {
			sk_shut(conn);
			return -1;
		} else if (FD_ISSET(sk, &wrfds)) {
			count = write(sk, &buff[offs], limit);
			if (-1 == count && errno == EAGAIN) {
				usleep(25000);
				continue;
			} else if (count < 1) {
				sk_shut(conn);
				return -1;
			}
			offs += count;
		}
	}

	return 0;
}

int sk_read(conn_t *conn, void *buffer, size_t size)
{
	char *buff = (char *)buffer;
	size_t offs;

	for (offs = 0; offs < size; /* */) {
		ssize_t count;
		size_t limit = (size - offs) > RW_SIZE ? RW_SIZE : size - offs;
		struct timeval tv;
		fd_set rdfds, exfds;
		int sk;

		sk = conn->sk;
		if (-1 == conn->sk) {
			return -1;
		}

		FD_ZERO(&rdfds);
		FD_SET(sk, &rdfds);
		FD_ZERO(&exfds);
		FD_SET(sk, &exfds);
		tv.tv_sec = 60;
		tv.tv_usec = 0;
		if (-1 == select(sk+1, &rdfds, NULL, &exfds, &tv)) {
			sk_shut(conn);
			errno = EIO;
			return -1;
		} else if (FD_ISSET(sk, &exfds)) {
			sk_shut(conn);
			errno = EIO;
			return -1;
		} else if (FD_ISSET(sk, &rdfds)) {
			count = read(sk, &buff[offs], limit);
			if (-1 == count && errno == EAGAIN) {
				usleep(25000);
				continue;
			} else if (0 == offs && 0 == count) {
				LOG(L_DEBUG,("EOF (%s)\n", strerror(errno)));
				sk_shut(conn);
				errno = EBADF;
				return -1;
			} else if (count < 1) {
				LOG(L_ERROR,("read() failed (%s)\n", strerror(errno)));
				sk_shut(conn);
				errno = EIO;
				return -1;
			}
			offs += count;
		}
	}

	return 0;
}

int sk_gets(conn_t *conn, char *buff, size_t size)
{
	size_t offs;

	for (offs = 0; offs < size; /* */) {
		ssize_t count;
		struct timeval tv;
		fd_set rdfds, exfds;
		char ch;
		int sk;

		buff[offs] = '\0';

		sk = conn->sk;
		if (-1 == conn->sk) {
			LOG(L_ERROR,("sk is closed\n"));
			return -1;
		}

		FD_ZERO(&rdfds);
		FD_SET(sk, &rdfds);
		FD_ZERO(&exfds);
		FD_SET(sk, &exfds);
		tv.tv_sec = 60;
		tv.tv_usec = 0;
		if (-1 == select(sk+1, &rdfds, NULL, &exfds, &tv)) {
			LOG(L_ERROR,("select() failed (%s)\n", strerror(errno)));
			sk_shut(conn);
			return -1;
		} else if (FD_ISSET(sk, &exfds)) {
			LOG(L_ERROR,("sk is in exfds (%s)\n", strerror(errno)));
			sk_shut(conn);
			return -1;
		} else if (FD_ISSET(sk, &rdfds)) {
			count = read(sk, &ch, 1);
			if (-1 == count && errno == EAGAIN) {
				usleep(25000);
				continue;
			} else if (0 == offs && 0 == count) {
				LOG(L_DEBUG,("EOF (%s)\n", strerror(errno)));
				sk_shut(conn);
				return -1;
			} else if (count < 1) {
				LOG(L_ERROR,("read() failed (%s)\n", strerror(errno)));
				sk_shut(conn);
				return -1;
			}
			buff[offs++] = ch;
			if ('\n' == ch) {
				break;
			}
		}
	}

	buff[offs] = '\0';
	return 0;
}

int sk_printf(conn_t *conn, const char *fmt, ...)
{
	char *buff = calloc(sizeof(char), 32768);
	va_list ap;
	size_t len;

	if (NULL == buff) {
		return -1;
	}

	va_start(ap, fmt);
	len = vsnprintf(buff, 32768, fmt, ap);
	va_end(ap);

	sk_write(conn, buff, len);

	free(buff);
	return 0;
}

int sk_shut(conn_t *conn)
{
	if (NULL == conn) {
		errno = EINVAL;
		return -1;
	}

	if (-1 == conn->sk) {
		errno = EBADF;
		return -1;
	}

	if (0 != shutdown(conn->sk, SHUT_RDWR)) {
		conn->sk = -1;
		errno = EIO;
		return -1;
	}

	if (0 != close(conn->sk)) {
		conn->sk = -1;
		errno = EIO;
		return -1;
	}

	conn->sk = -1;
	return 0;
}
