
/* Simple arithmetic for quad precision.
 *
 * Copyright (c) 2008, Andrew Vaught
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * * The name of Andrew Vaught may not be used to endorse or promote
 *   products derived from this software without specific prior
 *   written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE. */

#include "runtime.h"


#if 0 && HAVE_WINDOWS && __MINGW32__

#define addtf3       _addtf3
#define divtf3       _divtf3
#define eqtf2        _eqtf2
#define extenddftf2  _extenddftf2
#define extendsftf2  _extendsftf2
#define extendxftf2  _extendxftf2
#define fixtfdi      _fixtfdi
#define fixtfsi      _fixtfsi
#define floatditf    _floatditf
#define floatsitf    _floatsitf
#define getf2        _getf2
#define gttf2        _gttf2
#define letf2        _letf2
#define lttf2        _lttf2
#define multf3       _multf3
#define netf2        _netf2
#define subtf3       _subtf3
#define trunctfdf2   _trunctfdf2
#define trunctfsf2   _trunctfsf2
#define trunctfxf2   _trunctfxf2

#else

#define addtf3       __addtf3
#define divtf3       __divtf3
#define eqtf2        __eqtf2
#define extenddftf2  __extenddftf2
#define extendsftf2  __extendsftf2
#define extendxftf2  __extendxftf2
#define fixtfdi      __fixtfdi
#define fixtfsi      __fixtfsi
#define floatditf    __floatditf
#define floatsitf    __floatsitf
#define getf2        __getf2
#define gttf2        __gttf2
#define letf2        __letf2
#define lttf2        __lttf2
#define multf3       __multf3
#define netf2        __netf2
#define subtf3       __subtf3
#define trunctfdf2   __trunctfdf2
#define trunctfsf2   __trunctfsf2
#define trunctfxf2   __trunctfxf2

#endif


/* State of bits shifted out of the larger word in a sum or
 * difference. */

static enum { BOTTOM_EQ_ZERO, BOTTOM_LT_HALF,
	      BOTTOM_EQ_HALF, BOTTOM_GT_HALF } round;


static const unsigned bottom_value[] = {
    0x00000001U,  0x00000002U,  0x00000004U,  0x00000008U,
    0x00000010U,  0x00000020U,  0x00000040U,  0x00000080U,
    0x00000100U,  0x00000200U,  0x00000400U,  0x00000800U,
    0x00001000U,  0x00002000U,  0x00004000U,  0x00008000U,
    0x00010000U,  0x00020000U,  0x00040000U,  0x00080000U,
    0x00100000U,  0x00200000U,  0x00400000U,  0x00800000U,
    0x01000000U,  0x02000000U,  0x04000000U,  0x08000000U,
    0x10000000U,  0x20000000U,  0x40000000U,  0x80000000U  };


static const unsigned bottom_mask[] = {
    0x00000001U,  0x00000003U,  0x00000007U,  0x0000000FU,
    0x0000001FU,  0x0000003FU,  0x0000007FU,  0x000000FFU,
    0x000001FFU,  0x000003FFU,  0x000007FFU,  0x00000FFFU,
    0x00001FFFU,  0x00003FFFU,  0x00007FFFU,  0x0000FFFFU,
    0x0001FFFFU,  0x0003FFFFU,  0x0007FFFFU,  0x000FFFFFU,
    0x001FFFFFU,  0x003FFFFFU,  0x007FFFFFU,  0x00FFFFFFU,
    0x01FFFFFFU,  0x03FFFFFFU,  0x07FFFFFFU,  0x0FFFFFFFU,
    0x1FFFFFFFU,  0x3FFFFFFFU,  0x7FFFFFFFU,  0xFFFFFFFFU  };


static const char logtable[] = {
    0, 0, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
    7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7
};



#if HAVE_REAL_10 == 3

/* In C-style, x86-64 passes quad arguments in xmm0 and xmm1.  We can
 * process these for our caller if our caller has not done anything
 * before calling here.  Storing xmm1 into xmm0 sets us up to be
 * called again for a second parameter.  This means that multiple quad
 * arguments must be unpacked in order. */

void unpack_quad(void *v, unsigned *mantissa, int *exp, int *sign) {
unsigned p[4];
int e;

    asm("movaps %%xmm0, %0\n"
	"movaps %%xmm1, %%xmm0\n" : "=m" (p));

    mantissa[3] = p[0];
    mantissa[2] = p[1];
    mantissa[1] = p[2];
    mantissa[0] = p[3] & 0x0000FFFF;

    e = (p[3] >> 16) & 0x7FFF;
    if (e != 0 && e != EXP16_NAN)
	mantissa[0] |= 0x00010000;

    *exp = e;
    *sign = !!(p[3] & 0x80000000);
}

#endif



/* top_bit()-- Given an unsigned 32-bit number, return the position of
 * the most significant bit.  Does not work for m == 0. */

int top_bit(unsigned m) {

    if (m >= 0x01000000)
	return logtable[m >> 24] + 24;

    if (m >= 0x00010000)
	return logtable[m >> 16] + 16;

    if (m >= 0x00000100)
	return logtable[m >> 8]  +  8;

    return logtable[m];
}



/* denorm()-- Like shift(), but does not set the rounding flag, and
 * checks for underflow to zero. */

static void denorm(unpacked16 *n) {
int shift, s2;

    shift  = 1 - n->exp;
    n->exp = 0;

    switch(shift) {
    case  1: case  2: case  3: case  4: case  5: case  6: case  7: case  8:
    case  9: case 10: case 11: case 12: case 13: case 14: case 15: case 16:
    case 17: case 18: case 19: case 20: case 21: case 22: case 23: case 24:
    case 25: case 26: case 27: case 28: case 29: case 30: case 31:
	s2 = 32 - shift;

	n->m[3] = (n->m[3] >> shift) | (n->m[2] << s2);
	n->m[2] = (n->m[2] >> shift) | (n->m[1] << s2);
	n->m[1] = (n->m[1] >> shift) | (n->m[0] << s2);
	n->m[0] =  n->m[0] >> shift;
	break;

    case 32:
	n->m[3] = n->m[2];
	n->m[2] = n->m[1];
	n->m[1] = n->m[0];
	n->m[0] = 0;
	break;

    case 33: case 34: case 35: case 36: case 37: case 38: case 39: case 40:
    case 41: case 42: case 43: case 44: case 45: case 46: case 47: case 48:
    case 49: case 50: case 51: case 52: case 53: case 54: case 55: case 56:
    case 57: case 58: case 59: case 60: case 61: case 62: case 63:
	shift -= 32;
	s2 = 32 - shift;

	n->m[3] = (n->m[2] >> shift) | (n->m[1] << s2);
	n->m[2] = (n->m[1] >> shift) | (n->m[0] << s2);
	n->m[1] =  n->m[0] >> shift;
	n->m[0] = 0;
	break;

    case 64:
	n->m[3] = n->m[1];
	n->m[2] = n->m[0];
	n->m[1] = 0;
	n->m[0] = 0;
	break;

    case 65: case 66: case 67: case 68: case 69: case 70: case 71: case 72:
    case 73: case 74: case 75: case 76: case 77: case 78: case 79: case 80:
    case 81: case 82: case 83: case 84: case 85: case 86: case 87: case 88:
    case 89: case 90: case 91: case 92: case 93: case 94: case 95:
	shift -= 64;
	s2 = 32 - shift;

	n->m[3] = (n->m[1] >> shift) | (n->m[0] << s2);
	n->m[2] =  n->m[0] >> shift;
	n->m[1] = 0;
	n->m[0] = 0;
	break;
	
    case 96:
	n->m[3] = n->m[0];
	n->m[2] = 0;
	n->m[1] = 0;
	n->m[0] = 0;
	break;

    case  97: case  98: case  99: case 100: case 101: case 102: case 103:
    case 104: case 105: case 106: case 107: case 108: case 109: case 110:
    case 111:
	shift -= 96;

	n->m[3] = n->m[0] >> shift;
	n->m[2] = 0;
	n->m[1] = 0;
	n->m[0] = 0;
	break;

    default:
	n->m[0] = 0;
	n->m[1] = 0;
	n->m[2] = 0;
	n->m[3] = 0;
	break;
    }
}



/* shift()-- Right shift the contents of one register into another.
 * Initializes the destination and sets the round variable. */

static void shift(unsigned shift, unpacked16 *src, unpacked16 *dest) {
unsigned bottom, half;
int s2;

    switch(shift) {
    case 0:
	dest->m[0] = src->m[0];
	dest->m[1] = src->m[1];
	dest->m[2] = src->m[2];
	dest->m[3] = src->m[3];
	round = BOTTOM_EQ_ZERO;
	break;

    case  1: case  2: case  3: case  4: case  5: case  6: case  7: case  8:
    case  9: case 10: case 11: case 12: case 13: case 14: case 15: case 16:
    case 17: case 18: case 19: case 20: case 21: case 22: case 23: case 24:
    case 25: case 26: case 27: case 28: case 29: case 30: case 31:
	bottom = src->m[3] & bottom_mask[shift-1];
	half = bottom_value[shift-1];

	if (bottom > half)
	    round = BOTTOM_GT_HALF;

	else if (bottom == half)
	    round = BOTTOM_EQ_HALF;

	else if (bottom != 0)
	    round = BOTTOM_LT_HALF;

	else
	    round = BOTTOM_EQ_ZERO;

	s2 = 32 - shift;

	dest->m[3] = (src->m[3] >> shift) | (src->m[2] << s2);
	dest->m[2] = (src->m[2] >> shift) | (src->m[1] << s2);
	dest->m[1] = (src->m[1] >> shift) | (src->m[0] << s2);
	dest->m[0] =  src->m[0] >> shift;
	break;

    case 32:
	if (src->m[3] > 0x80000000)
	    round = BOTTOM_GT_HALF;

	else if (src->m[3] == 0x80000000)
	    round = BOTTOM_EQ_HALF;

	else if (src->m[3] != 0)
	    round = BOTTOM_LT_HALF;

	else
	    round = BOTTOM_EQ_ZERO;

	dest->m[3] = src->m[2];
	dest->m[2] = src->m[1];
	dest->m[1] = src->m[0];
	dest->m[0] = 0;
	break;

    case 33: case 34: case 35: case 36: case 37: case 38: case 39: case 40:
    case 41: case 42: case 43: case 44: case 45: case 46: case 47: case 48:
    case 49: case 50: case 51: case 52: case 53: case 54: case 55: case 56:
    case 57: case 58: case 59: case 60: case 61: case 62: case 63:
	shift -= 32;

	bottom = src->m[2] & bottom_mask[shift-1];
	half = bottom_value[shift-1];

	if (bottom > half)
	    round = BOTTOM_GT_HALF;

	else if (bottom == half && src->m[3] != 0)
	    round = BOTTOM_EQ_HALF;

	else if (bottom != 0 || src->m[3] != 0)
	    round = BOTTOM_LT_HALF;

	else
	    round = BOTTOM_EQ_ZERO;

	s2 = 32 - shift;

	dest->m[3] = (src->m[2] >> shift) | (src->m[1] << s2);
	dest->m[2] = (src->m[1] >> shift) | (src->m[0] << s2);
	dest->m[1] =  src->m[0] >> shift;
	dest->m[0] = 0;
	break;

    case 64:
	if (src->m[2] > 0x80000000)
	    round = BOTTOM_GT_HALF;

	else if (src->m[2] == 0x80000000 && src->m[3] == 0)
	    round = BOTTOM_EQ_HALF;

	else if (src->m[2] != 0 || src->m[3] != 0)
	    round = BOTTOM_LT_HALF;

	else
	    round = BOTTOM_EQ_ZERO;

	dest->m[3] = src->m[1];
	dest->m[2] = src->m[0];
	dest->m[1] = 0;
	dest->m[0] = 0;
	break;

    case 65: case 66: case 67: case 68: case 69: case 70: case 71: case 72:
    case 73: case 74: case 75: case 76: case 77: case 78: case 79: case 80:
    case 81: case 82: case 83: case 84: case 85: case 86: case 87: case 88:
    case 89: case 90: case 91: case 92: case 93: case 94: case 95:
	shift -= 64;

	bottom = src->m[1] & bottom_mask[shift-1];
	half = bottom_value[shift-1];

	if (bottom > half)
	    round = BOTTOM_GT_HALF;

	else if (bottom == half && src->m[2] != 0 && src->m[3] != 0)
	    round = BOTTOM_EQ_HALF;

	else if (bottom != 0 || src->m[2] == 0 || src->m[3] == 0)
	    round = BOTTOM_LT_HALF;

	else
	    round = BOTTOM_EQ_ZERO;

	s2 = 32 - shift;

	dest->m[3] = (src->m[1] >> shift) | (src->m[0] << s2);
	dest->m[2] =  src->m[0] >> shift;
	dest->m[1] = 0;
	dest->m[0] = 0;
	break;
	
    case 96:
	if (src->m[1] > 0x80000000)
	    round = BOTTOM_GT_HALF;

	else if (src->m[1] == 0x80000000 && src->m[2] == 0 && src->m[3] == 0)
	    round = BOTTOM_EQ_HALF;

	else if (src->m[1] != 0 || src->m[2] != 0 || src->m[3] != 0)
	    round = BOTTOM_EQ_ZERO;

	dest->m[3] = src->m[0];
	dest->m[2] = 0;
	dest->m[1] = 0;
	dest->m[0] = 0;
	break;

    case  97: case  98: case  99: case 100: case 101: case 102: case 103:
    case 104: case 105: case 106: case 107: case 108: case 109: case 110:
    case 111: case 112:

    case 113:    /* Special case-- shifts to zero, but sets round correctly. */
	shift -= 96;

	bottom = src->m[0] & bottom_mask[shift-1];
	half = bottom_value[shift-1];

	if (bottom > half)
	    round = BOTTOM_GT_HALF;

	else if (bottom == half && src->m[1] != 0 && src->m[2] != 0 && src->m[3] != 0)
	    round = BOTTOM_EQ_HALF;

	else if (bottom != 0 || src->m[1] == 0 || src->m[2] == 0 || src->m[3] == 0)
	    round = BOTTOM_LT_HALF;

	else
	    round = BOTTOM_EQ_ZERO;

	dest->m[3] = src->m[0] >> shift;
	dest->m[2] = 0;
	dest->m[1] = 0;
	dest->m[0] = 0;
	break;


    default:
	round = (src->m[0] == 0 && src->m[1] == 0 && src->m[2] == 0 && src->m[3] == 0)
	    ? BOTTOM_EQ_ZERO
	    : BOTTOM_LT_HALF;

	dest->m[3] = 0;
	dest->m[2] = 0;
	dest->m[1] = 0;
	dest->m[0] = 0;
	break;
    }
}



/* compare16()-- Compare two regular unpacked numbers regardless of
 * sign.  Returns the usual -1, 0 and 1. */

int compare16(unpacked16 *a, unpacked16 *b) {

    if (a->exp < b->exp)   return -1;
    if (a->exp > b->exp)   return +1;

    if (a->m[0] < b->m[0]) return -1;
    if (a->m[0] > b->m[0]) return +1;

    if (a->m[1] < b->m[1]) return -1;
    if (a->m[1] > b->m[1]) return +1;

    if (a->m[2] < b->m[2]) return -1;
    if (a->m[2] > b->m[2]) return +1;

    if (a->m[3] < b->m[3]) return -1;
    if (a->m[3] > b->m[3]) return +1;

    return 0;
}



/* set_zero()-- Set an unpacked number to a signed zero. */

static void set_zero(unpacked16 *z, int sign) {

    z->exp  = 0;
    z->m[0] = z->m[1] = z->m[2] = z->m[3] = 0;
    z->sign = sign;
}



/* set_infinity()-- Set an unpacked number to infinity */

static void set_infinity(unpacked16 *a, int sign) {

    a->exp  = EXP16_NAN;
    a->sign = sign;
    a->m[0] = a->m[1] = a->m[2] = a->m[3] = 0;
}



/* set_nan16()-- Set an unpacked number to a not-a-number */

void set_nan16(unpacked16 *a) {

    a->exp  = EXP16_NAN;
    a->sign = 0;
    a->m[0] = a->m[1] = a->m[2] = a->m[3] = 0x1;
}



/* difference()-- Compute the difference of two numbers.  This is used
 * by addition of two numbers of differing sign or subtracting two
 * numbers of the same sign.  a is guaranteed to be greater than b. */

static void difference(unpacked16 *a, unpacked16 *b, unpacked16 *diff) {
int t, t2;

    shift(a->exp - b->exp, b, diff);
    diff->exp = a->exp;

    diff->m[3] = a->m[3] - diff->m[3];

    /* On borrow, we increment the subtrahend, since we can't mess
     * with the minuend. */

    if (diff->m[3] > a->m[3] && ++diff->m[2] == 0 && ++diff->m[1] == 0)
	diff->m[0]++;

    diff->m[2] = a->m[2] - diff->m[2];

    if (diff->m[2] > a->m[2] && ++diff->m[1] == 0)
	diff->m[0]++;

    diff->m[1] = a->m[1] - diff->m[1];

    if (diff->m[1] > a->m[1])
	diff->m[0]++;

    diff->m[0] = a->m[0] - diff->m[0];

    /* Round.  At this point, we have to have something greater than
     * zero. */

    if (round == BOTTOM_GT_HALF ||
	(round == BOTTOM_EQ_HALF && (diff->m[3] & 1))) {

	if (diff->m[3]-- == 0 && diff->m[2]-- == 0 && diff->m[1]-- == 0 && 
	    --diff->m[0] == 0) {

	    /* If the result after decrementing is zero, then we've
	     * just lost all of the bits in the mantissa.  Recover. */

	    diff->m[0] = 0x00010000;
	    diff->exp -= 112;
	    goto done;
	}
    }

    /* Normalize.  The mantissa can't be zero, even if we rounded
     * down. */

    if (diff->m[0] & 0x00010000)
	;

    else if (diff->m[0] != 0) {
	t = 16 - top_bit(diff->m[0]);
	t2 = 32 - t;

	diff->exp -= t;

	diff->m[0] = (diff->m[0] << t) | (diff->m[1] >> t2);
	diff->m[1] = (diff->m[1] << t) | (diff->m[2] >> t2);
	diff->m[2] = (diff->m[2] << t) | (diff->m[3] >> t2);
	diff->m[3] =  diff->m[3] << t;

    } else if (diff->m[1] != 0) {
	t = top_bit(diff->m[1]);
	diff->exp -= 48 - t;

	if (t > 16) {
	    t = t - 16;
	    t2 = 32 - t;

	    diff->m[0] =  diff->m[1] >> t;
	    diff->m[1] = (diff->m[2] >> t) | (diff->m[1] << t2);
	    diff->m[2] = (diff->m[3] >> t) | (diff->m[2] << t2);
	    diff->m[3] =                   diff->m[3] << t2;

	} else if (t < 16) {
	    t = 16 - t;
	    t2 = 32 - t;

	    diff->m[0] = (diff->m[1] << t) | (diff->m[2] >> t2);
	    diff->m[1] = (diff->m[2] << t) | (diff->m[3] >> t2);
	    diff->m[2] =  diff->m[3] << t;
	    diff->m[3] = 0;

	} else {
	    diff->m[0] = diff->m[1];
	    diff->m[1] = diff->m[2];
	    diff->m[2] = diff->m[3];
	    diff->m[3] = 0;
	}

    } else if (diff->m[2] != 0) {
	t = top_bit(diff->m[2]);
	diff->exp -= 80 - t;

	if (t > 16) {
	    t = t - 16;
	    t2 = 32 - t;

	    diff->m[0] =  diff->m[2] >> t;
	    diff->m[1] = (diff->m[3] >> t) | (diff->m[2] << t2);
	    diff->m[2] =                   diff->m[3] << t2;
	    diff->m[3] = 0;

	} else if (t < 16) {
	    t = 16 - t;
	    t2 = 32 - t;

	    diff->m[0] = (diff->m[2] << t) | (diff->m[3] >> t2);
	    diff->m[1] =  diff->m[3] << t;
	    diff->m[2] = 0;
	    diff->m[3] = 0;

	} else {
	    diff->m[0] = diff->m[2];
	    diff->m[1] = diff->m[3];
	    diff->m[2] = 0;
	    diff->m[3] = 0;
	}

    } else {  /* diff->m[3] != 0 */
	t = top_bit(diff->m[3]);
	diff->exp -= 112 - t;

	if (t > 16) {
	    t = t - 16;
	    t2 = 32 - t;

	    diff->m[0] = diff->m[3] >> t;
	    diff->m[1] = diff->m[3] << t2;
	    diff->m[2] = 0;
	    diff->m[3] = 0;

	} else if (t < 16) {
	    t = 16 - t;

	    diff->m[0] = diff->m[3] << t;
	    diff->m[1] = 0;
	    diff->m[2] = 0;
	    diff->m[3] = 0;

	} else {
	    diff->m[0] = diff->m[3];
	    diff->m[1] = 0;
	    diff->m[2] = 0;
	    diff->m[3] = 0;
	}
    }

    /* Denormalize if we have to */
done:
    if (diff->exp < 0)
	denorm(diff);
}



/* sum()-- Add two regular numbers of the same sign.  This is used by
 * addition of numbers of the same sign or subtraction of numbers of
 * different signs. */

static void sum(unpacked16 *a, unpacked16 *b, unpacked16 *s) {

    if (a->exp > b->exp) {
	s->exp = a->exp;
	shift(a->exp - b->exp, b, s);

	s->m[0] += a->m[0];

	s->m[1] += a->m[1];
	if (s->m[1] < a->m[1])
	    s->m[0]++;

	s->m[2] += a->m[2];
	if (s->m[2] < a->m[2] && ++s->m[1] == 0)
	    s->m[0]++;

	s->m[3] += a->m[3];
	if (s->m[3] < a->m[3] && ++s->m[2] == 0 && ++s->m[1] == 0)
	    s->m[0]++;

    } else if (a->exp < b->exp) {
	s->exp = b->exp;
	shift(b->exp - a->exp, a, s);

	s->m[0] += b->m[0];

	s->m[1] += b->m[1];
	if (s->m[1] < b->m[1])
	    s->m[0]++;

	s->m[2] += b->m[2];
	if (s->m[2] < b->m[2] && ++s->m[1] == 0)
	    s->m[0]++;

	s->m[3] += b->m[3];
	if (s->m[3] < b->m[3] && ++s->m[2] == 0 && ++s->m[1] == 0)
	    s->m[0]++;

    } else {
	s->exp = a->exp;
	round = BOTTOM_EQ_ZERO;

	s->m[0] = a->m[0] + b->m[0];

	s->m[1] = a->m[1] + b->m[1];
	if (s->m[1] < a->m[1])
	    s->m[0]++;

	s->m[2] = a->m[2] + b->m[2];
	if (s->m[2] < a->m[2] && ++s->m[1] == 0)
	    s->m[0]++;

	s->m[3] = a->m[3] + b->m[3];
	if (s->m[3] < a->m[3] && ++s->m[2] == 0 && ++s->m[1] == 0)
	    s->m[0]++;
    }

    /* If the sum has more bits than the mantissa, shift right by one
     * and combine the least significant bit of the sum with the
     * previous bits shifted out of one of the summands. */

    if (s->m[0] & 0x00020000) {
	if ((s->m[3] & 1) == 0) {
	    if (round == BOTTOM_EQ_HALF)
		round = BOTTOM_LT_HALF;

	} else
	    switch(round) {
	    case BOTTOM_EQ_ZERO:  round = BOTTOM_EQ_HALF;  break;
	    case BOTTOM_LT_HALF:  round = BOTTOM_GT_HALF;  break;
	    case BOTTOM_EQ_HALF:  round = BOTTOM_GT_HALF;  break;
	    case BOTTOM_GT_HALF:                           break;
	    }

	s->m[3] = (s->m[3] >> 1) | (s->m[2] << 31);
	s->m[2] = (s->m[2] >> 1) | (s->m[1] << 31);
	s->m[1] = (s->m[1] >> 1) | (s->m[0] << 31);
	s->m[0] =  s->m[0] >> 1;
	s->exp++;
    }

    /* Round the sum based on bits shifted out */

    if (round == BOTTOM_GT_HALF ||
	       (round == BOTTOM_EQ_HALF && (s->m[3] & 1))) {
	if (++s->m[3] == 0 && ++s->m[2] == 0 && ++s->m[1] == 0) {
	    s->m[0]++;

	    /* If the rounding causes a mantissa overflow, we'd
	     * normally do another right shift, except that we already
	     * know what the answer has to be... */

	    if (s->m[0] == 0x00020000) {
		s->m[0] = 0x00010000;
		s->exp++;
	    }
	}
    }

    if (s->exp >= EXP16_NAN)
	set_infinity(s, s->sign);
}



/* add_unpacked()-- Add a pair of regular unpacked numbers into a
 * packed sum.  This is also an entry point for library procedures
 * that evaluate power series and such. */

void add_unpacked(unpacked16 *a, unpacked16 *b, unpacked16 *s) {
int i;

    if (a->sign == b->sign) {
	sum(a, b, s);
	s->sign = a->sign;

    } else {
	i = compare16(a, b);

	if (i > 0) {
	    difference(a, b, s);
	    s->sign = a->sign;

	} else if (i < 0) {
	    difference(b, a, s);
	    s->sign = b->sign;

	} else
	    set_zero(s, 0);
    }
}



/* subtract_unpacked()-- Subtract a pair of unpacked numbers. */

void subtract_unpacked(unpacked16 *a, unpacked16 *b, unpacked16 *d) {
int i;

    if (a->sign != b->sign) {
	sum(a, b, d);
	d->sign = a->sign;

    } else {
	i = compare16(a, b);

	if (i > 0) {
	    difference(a, b, d);
	    d->sign = a->sign;

	} else if (i < 0) {
	    difference(b, a, d);
	    d->sign = !a->sign;

	} else
	    set_zero(d, 0);
    }
}



static void addtf3_0(unpacked16 *sum, unpacked16 *a, unpacked16 *b) {
int inf_a, inf_b;
packed16 s;

    if (a->exp != EXP16_NAN && b->exp != EXP16_NAN)
	add_unpacked(a, b, sum);

    else {
	inf_a = a->m[0] == 0x0001FFFF && a->m[1] == 0xFFFFFFFF &&
                a->m[2] == 0xFFFFFFFF && a->m[3] == 0xFFFFFFFF;

	inf_b = b->m[0] == 0x0001FFFF && b->m[1] == 0xFFFFFFFF &&
                b->m[2] == 0xFFFFFFFF && b->m[3] == 0xFFFFFFFF;

	if ((a->exp == EXP16_NAN && !inf_a) ||
	    (b->exp == EXP16_NAN && !inf_b))

	    set_nan16(sum);      /* a or b is a NaN */

	else if (a->exp == EXP16_NAN && inf_a &&
		 b->exp == EXP16_NAN && inf_b) {  /* a and b are infinities */

	    if (a->sign == b->sign)
		set_infinity(sum, a->sign);

	    else
		set_nan16(sum);

	} else
	    *sum = (a->exp == EXP16_NAN && inf_a)
		? *a : *b;
    }
}


#if QUAD_POINTER

void addtf3(void *s, void *a0, void *b0) {
unpacked16 sum, a, b;

    unpack_quad(a0, a.m, &a.exp, &a.sign);
    unpack_quad(b0, b.m, &b.exp, &b.sign);

    addtf3_0(&sum, &a, &b);
    pack_real_16(s, sum.m, &sum.exp, &sum.sign);
}

#else

packed16 addtf3(packed16 a0, packed16 b0) {
unpacked16 sum, a, b;
packed16 s;

    unpack_quad(a0.a, a.m, &a.exp, &a.sign);
    unpack_quad(b0.a, b.m, &b.exp, &b.sign);

    addtf3_0(&sum, &a, &b);

    pack_real_16(&s, sum.m, &sum.exp, &sum.sign);
    return s;
}

#endif



static void subtf3_0(unpacked16 *diff, unpacked16 *a, unpacked16 *b) {
int inf_a, inf_b;

    if (a->exp != EXP16_NAN && b->exp != EXP16_NAN)
	subtract_unpacked(a, b, diff);

    else {
	inf_a = a->m[0] == 0x0001FFFF && a->m[1] == 0xFFFFFFFF &&
                a->m[2] == 0xFFFFFFFF && a->m[3] == 0xFFFFFFFF;

	inf_b = b->m[0] == 0x0001FFFF && b->m[1] == 0xFFFFFFFF &&
                b->m[2] == 0xFFFFFFFF && b->m[3] == 0xFFFFFFFF;

	if ((a->exp == EXP16_NAN && !inf_a) ||
	    (b->exp == EXP16_NAN && !inf_b))

	    set_nan16(diff);      /* a or b is a NaN */

	else if (a->exp == EXP16_NAN && inf_a &&
		 b->exp == EXP16_NAN && inf_b) {  /* a and b are infinities */

	    if (a->sign != b->sign)
		set_infinity(diff, a->sign);

	    else
		set_nan16(diff);

	} else if (a->exp == EXP16_NAN && inf_a)
	    *diff = *a;

	else {
	    *diff = *b;
	    diff->sign = !diff->sign;
	}
    }
}


#if QUAD_POINTER

void subtf3(void *d, void *a0, void *b0) {
unpacked16 diff, a, b;

    unpack_quad(a0, a.m, &a.exp, &a.sign);
    unpack_quad(b0, b.m, &b.exp, &b.sign);

    subtf3_0(&diff, &a, &b);
    pack_real_16(d, diff.m, &diff.exp, &diff.sign);
}

#else

packed16 subtf3(packed16 a0, packed16 b0) {
unpacked16 diff, a, b;
packed16 d;

    unpack_quad(a0.a, a.m, &a.exp, &a.sign);
    unpack_quad(b0.a, b.m, &b.exp, &b.sign);

    subtf3_0(&diff, &a, &b);

    pack_real_16(&d, diff.m, &diff.exp, &diff.sign);
    return d;
}

#endif



/* multiply_unpacked()-- Multiply two unpacked numbers. */

void multiply_unpacked(unpacked16 *a, unpacked16 *b, unpacked16 *p) {
unsigned p1, p2, prod[8], c[6];
unsigned long long pr;
int i, shift, t, t2;

    pr = (unsigned long long) a->m[3] * (unsigned long long) b->m[3];
    p2 = pr;    p1 = pr >> 32;
    prod[7]  = p2;
    prod[6]  = p1;

    pr = (unsigned long long) a->m[2] * (unsigned long long) b->m[3];
    p2 = pr;    p1 = pr >> 32;
    prod[6] += p2;  c[5]  = (prod[6] < p2);
    prod[5]  = p1;

    pr = (unsigned long long) a->m[1] * (unsigned long long) b->m[3];
    p2 = pr;    p1 = pr >> 32;
    prod[5] += p2;  c[4]  = (prod[5] < p2);
    prod[4]  = p1;

    pr = (unsigned long long) a->m[0] * (unsigned long long) b->m[3];
    p2 = pr;    p1 = pr >> 32;
    prod[4] += p2;  c[3]  = (prod[4] < p2);
    prod[3]  = p1;


    pr = (unsigned long long) a->m[3] * (unsigned long long) b->m[2];
    p2 = pr;    p1 = pr >> 32;
    prod[6] += p2;  c[5] += (prod[6] < p2);
    prod[5] += p1;  c[4] += (prod[5] < p1);

    pr = (unsigned long long) a->m[2] * (unsigned long long) b->m[2];
    p2 = pr;    p1 = pr >> 32;
    prod[5] += p2;  c[4] += (prod[5] < p2);
    prod[4] += p1;  c[3] += (prod[4] < p1);

    pr = (unsigned long long) a->m[1] * (unsigned long long) b->m[2];
    p2 = pr;    p1 = pr >> 32;
    prod[4] += p2;  c[3] += (prod[4] < p2);
    prod[3] += p1;  c[2]  = (prod[3] < p1);

    pr = (unsigned long long) a->m[0] * (unsigned long long) b->m[2];
    p2 = pr;    p1 = pr >> 32;
    prod[3] += p2;  c[2] += (prod[3] < p2);
    prod[2]  = p1;


    pr = (unsigned long long) a->m[3] * (unsigned long long) b->m[1];
    p2 = pr;    p1 = pr >> 32;
    prod[5] += p2;  c[4] += (prod[5] < p2);
    prod[4] += p1;  c[3] += (prod[4] < p1);

    pr = (unsigned long long) a->m[2] * (unsigned long long) b->m[1];
    p2 = pr;    p1 = pr >> 32;
    prod[4] += p2;  c[3] += (prod[4] < p2);
    prod[3] += p1;  c[2] += (prod[3] < p1);

    pr = (unsigned long long) a->m[1] * (unsigned long long) b->m[1];
    p2 = pr;    p1 = pr >> 32;
    prod[3] += p2;  c[2] += (prod[3] < p2);
    prod[2] += p1;  c[1]  = (prod[2] < p1);

    pr = (unsigned long long) a->m[0] * (unsigned long long) b->m[1];
    p2 = pr;    p1 = pr >> 32;
    prod[2] += p2;  c[1] += (prod[2] < p2);
    prod[1]  = p1;


    pr = (unsigned long long) a->m[3] * (unsigned long long) b->m[0];
    p2 = pr;    p1 = pr >> 32;
    prod[4] += p2;  c[3] += (prod[4] < p2);
    prod[3] += p1;  c[2] += (prod[3] < p1);

    pr = (unsigned long long) a->m[2] * (unsigned long long) b->m[0];
    p2 = pr;    p1 = pr >> 32;
    prod[3] += p2;  c[2] += (prod[3] < p2);
    prod[2] += p1;  c[1] += (prod[2] < p1);

    pr = (unsigned long long) a->m[1] * (unsigned long long) b->m[0];
    p2 = pr;    p1 = pr >> 32;
    prod[2] += p2;  c[1] += (prod[2] < p2);
    prod[1] += p1;  c[0]  = (prod[1] < p1);

    pr = (unsigned long long) a->m[0] * (unsigned long long) b->m[0];
    p2 = pr;    p1 = pr >> 32;
    prod[1] += p2;  c[0] += (prod[1] < p2);
    prod[0]  = p1;

    /* Propagate carries into final product */

    prod[5] += c[5];  c[4] += (prod[5] < c[5]);
    prod[4] += c[4];  c[3] += (prod[4] < c[4]);
    prod[3] += c[3];  c[2] += (prod[3] < c[3]);
    prod[2] += c[2];  c[1] += (prod[2] < c[2]);
    prod[1] += c[1];  c[0] += (prod[1] < c[1]);
    prod[0] += c[0];

    p->sign = a->sign ^ b->sign;
    p->exp  = a->exp + b->exp - EXP16_BIAS;

    /* Normalize the number. */

    if (prod[0] != 0) {
	t = 16 - top_bit(prod[0]);
	t2 = 32 - t;

	p->exp -= t - 16;

	prod[0] = (prod[0] << t) | (prod[1] >> t2);
	prod[1] = (prod[1] << t) | (prod[2] >> t2);
	prod[2] = (prod[2] << t) | (prod[3] >> t2);
	prod[3] = (prod[3] << t) | (prod[4] >> t2);
	prod[4] = (prod[4] << t) | (prod[5] >> t2);
	prod[5] = (prod[5] << t) | (prod[6] >> t2);
	prod[6] = (prod[6] << t) | (prod[7] >> t2);
	prod[7] = (prod[7] << t);

    } else if (prod[1] >= 0x00010000) {
	t = top_bit(prod[1]) - 16;
	t2 = 32 - t;

	p->exp -= 15 - t;

	prod[0] = (prod[1] >> t);
	prod[1] = (prod[2] >> t) | (prod[1] << t2);
	prod[2] = (prod[3] >> t) | (prod[2] << t2);
	prod[3] = (prod[4] >> t) | (prod[3] << t2);
	prod[4] = (prod[5] >> t) | (prod[4] << t2);
	prod[5] = (prod[6] >> t) | (prod[5] << t2);
	prod[6] = (prod[7] >> t) | (prod[6] << t2);
	prod[7] =                  (prod[7] << t2);

    } else if (prod[1] > 0) {
	t = 16 - top_bit(prod[1]);

	t2 = 32 - t;
	p->exp -= t + 16;

	prod[0] = (prod[1] << t) | (prod[2] >> t2);
	prod[1] = (prod[2] << t) | (prod[3] >> t2);
	prod[2] = (prod[3] << t) | (prod[4] >> t2);
	prod[3] = (prod[4] << t) | (prod[5] >> t2);
	prod[4] = (prod[5] << t) | (prod[6] >> t2);
	prod[5] = (prod[6] << t) | (prod[7] >> t2);
	prod[6] = (prod[7] << t);
	prod[7] = 0;

    } else {
	/* At this point, the mantissa needs to be shifted more than
	 * 48 bits.  The only way this can happen is by multiplying
	 * two denormalized numbers or multiplying by zero.  Either
	 * way, the result is zero. */

	set_zero(p, p->sign);
	prod[4] = 0;    /* Force no rounding */
    }

    /* Round */

    if (prod[4] > 0x80000000 ||
	(prod[4] == 0x80000000 &&
	 ((prod[5] != 0 || prod[6] != 0 || prod[7] != 0) ||
	  (prod[5] == 0 && prod[6] == 0 && prod[7] == 0 && (prod[3] & 1))))) {

	if (++prod[3] == 0 && ++prod[2] == 0 && ++prod[1] == 0)
	    prod[0]++;

	if (prod[0] == 0x00020000) {
	    prod[0] =  0x00010000;
	    p->exp++;
	}
    }

    if (p->exp >= EXP16_NAN)
	set_infinity(p, p->sign);

    else {
	p->m[0] = prod[0];
	p->m[1] = prod[1];
	p->m[2] = prod[2];
	p->m[3] = prod[3];

	if (p->exp <= 0)
	    denorm(p);
    }
}



static packed16 multf3_0(unpacked16 *prod, unpacked16 *a, unpacked16 *b) {
int inf_a, inf_b;
packed16 p;

    if (a->exp != EXP16_NAN && b->exp != EXP16_NAN)
	multiply_unpacked(a, b, prod);

    else {
	inf_a = a->m[0] == 0 && a->m[1] == 0 && a->m[2] == 0 && a->m[3] == 0;
	inf_b = b->m[0] == 0 && b->m[1] == 0 && b->m[2] == 0 && b->m[3] == 0;

	if ((a->exp == EXP16_NAN && !inf_a) ||
	    (b->exp == EXP16_NAN && !inf_b))

	    set_nan16(prod);      /* a or b is a NaN */

	else if (a->exp == EXP16_NAN && inf_a) {
	    if (b->exp == 0  && b->m[0] == 0 && b->m[1] == 0 &&
		b->m[2] == 0 && b->m[3] == 0)
	        set_nan16(prod);
	    else
		set_infinity(prod, a->sign ^ b->sign);

	} else {  /* b is infinity */
	    if (a->exp == 0  && a->m[0] == 0 && a->m[1] == 0 &&
		a->m[2] == 0 && a->m[3] == 0)
	        set_nan16(prod);
	    else
		set_infinity(prod, a->sign ^ a->sign);
	}
    }
}


#if QUAD_POINTER

void multf3(void *p, void *a0, void *b0) {
unpacked16 prod, a, b;

    unpack_quad(a0, a.m, &a.exp, &a.sign);
    unpack_quad(b0, b.m, &b.exp, &b.sign);

    multf3_0(&prod, &a, &b);
    pack_real_16(p, prod.m, &prod.exp, &prod.sign);
}

#else

packed16 multf3(packed16 a0, packed16 b0) {
unpacked16 prod, a, b;
packed16 p;

    unpack_quad(a0.a, a.m, &a.exp, &a.sign);
    unpack_quad(b0.a, b.m, &b.exp, &b.sign);

    multf3_0(&prod, &a, &b);

    pack_real_16(&p, prod.m, &prod.exp, &prod.sign);
    return p;
}

#endif



void divide_unpacked(unpacked16 *divisor, unpacked16 *dividend,
		     unpacked16 *quotient) {
unsigned a, acc[4], i, mask;

    quotient->sign = dividend->sign ^ divisor->sign;
    quotient->exp  = dividend->exp  - divisor->exp + EXP16_BIAS;

    acc[0] = dividend->m[0];
    acc[1] = dividend->m[1];
    acc[2] = dividend->m[2];
    acc[3] = dividend->m[3];

    while(acc[0] == 0) {
	acc[0] = acc[1];
	acc[1] = acc[2];
	acc[2] = acc[3];
	acc[3] = 0;

	quotient->exp -= 32;
    }

    while(acc[0] < divisor->m[0] ||
	  acc[0] == divisor->m[0] &&
	  (acc[1] < divisor->m[1] ||
	   acc[1] == divisor->m[1] &&
	   (acc[2] < divisor->m[2] ||
	    acc[2] == divisor->m[2] &&
	    acc[3] < divisor->m[3]))) {

	quotient->exp--;

	acc[0] = (acc[0] << 1) | (acc[1] >> 31);
	acc[1] = (acc[1] << 1) | (acc[2] >> 31);
	acc[2] = (acc[2] << 1) | (acc[3] >> 31);
	acc[3] = (acc[3] << 1);
    }

    mask = 0x00010000;

    quotient->m[0] = 0;
    quotient->m[1] = 0;
    quotient->m[2] = 0;
    quotient->m[3] = 0;

    i = 0;

    while(i<4 && (acc[0] != 0 || acc[1] != 0 || acc[2] != 0 || acc[3] != 0)) {
	if (acc[0] > divisor->m[0] ||
	    acc[0] == divisor->m[0] &&
	    (acc[1] > divisor->m[1] ||
	     acc[1] == divisor->m[1] &&
	     (acc[2] > divisor->m[2] ||
	      acc[2] == divisor->m[2] &&
	      acc[3] >= divisor->m[3]))) {

	    quotient->m[i] |= mask;

	    a = acc[3];
	    acc[3] -= divisor->m[3];
	    if (acc[3] > a && acc[2]-- == 0 && acc[1]-- == 0)
		acc[0]--;

	    a = acc[2];
	    acc[2] -= divisor->m[2];
	    if (acc[2] > a && acc[1]-- == 0)
		acc[0]--;

	    a = acc[1];
	    acc[1] -= divisor->m[1];
	    if (acc[1] > a)
		acc[0]--;

	    acc[0] -= divisor->m[0];
	}

	acc[0] = (acc[0] << 1) | (acc[1] >> 31);
	acc[1] = (acc[1] << 1) | (acc[2] >> 31);
	acc[2] = (acc[2] << 1) | (acc[3] >> 31);
	acc[3] = (acc[3] << 1);

	mask = mask >> 1;
	if (mask == 0) {
	    mask = 0x80000000;
	    i++;
	}
    }

    /* Rounding-- Compute one more bit of quotient */

    i = (acc[0] > divisor->m[0] ||
	 acc[0] == divisor->m[0] &&
	 (acc[1] > divisor->m[1] ||
	  acc[1] == divisor->m[1] &&
	  (acc[2] > divisor->m[2] ||
	   acc[2] == divisor->m[2] &&
	   acc[3] >= divisor->m[3])));

    if (i == 0)
	round = BOTTOM_LT_HALF;

    else {
	a = acc[3];
	acc[3] -= divisor->m[3];
	if (acc[3] > a && acc[2]-- == 0 && acc[1]-- == 0)
	    acc[0]--;

	a = acc[2];
	acc[2] -= divisor->m[2];
	if (acc[2] > a && acc[1]-- == 0)
	    acc[0]--;

	a = acc[1];
	acc[1] -= divisor->m[1];
	if (acc[1] > a)
	    acc[0]--;

	acc[0] -= divisor->m[0];

	round = (acc[0] == 0 && acc[1] == 0 && acc[2] == 0 && acc[3] == 0)
	    ? BOTTOM_EQ_HALF
	    : BOTTOM_GT_HALF;
    }

    if (round == BOTTOM_GT_HALF ||
	(round == BOTTOM_EQ_HALF && quotient->m[0] & 1)) { /* Round up */

	if (++quotient->m[3] == 0 && ++quotient->m[2] == 0 &&
	    ++quotient->m[1] == 0 && ++quotient->m[0] == 0x00020000) {

	    quotient->exp++;
	    quotient->m[0] = 0x00010000;
	}
    }

    if (quotient->exp >= EXP16_NAN)
	set_infinity(quotient, quotient->sign);

    else if (quotient->exp <= 0)
	denorm(quotient);
}



static void divtf3_0(unpacked16 *q, unpacked16 *dividend, unpacked16 *divisor) {
int z_ds, z_dd;

    z_ds = divisor->m[0] == 0 && divisor->m[1] == 0 &&
	   divisor->m[2] == 0 && divisor->m[3] == 0;

    z_dd = dividend->m[0] == 0 && dividend->m[1] == 0 &&
           dividend->m[2] == 0 && dividend->m[3] == 0;

    if (divisor->exp != EXP16_NAN && dividend->exp != EXP16_NAN) {
	z_ds &= (divisor->exp  == 0);
	z_dd &= (dividend->exp == 0);

	if (z_ds && z_dd)
	    set_nan16(q);

	else if (z_ds)
	    set_infinity(q, divisor->sign ^ dividend->sign);

	else if (z_dd)
	    set_zero(q, divisor->sign ^ dividend->sign);

	else
	    divide_unpacked(divisor, dividend, q);

    } else {
	if ((divisor->exp  == EXP16_NAN && !z_ds) ||
	    (dividend->exp == EXP16_NAN && !z_dd))

	    set_nan16(q);

	else if (divisor->exp  == EXP16_NAN && z_ds &&
		 dividend->exp == EXP16_NAN && z_dd)

	    set_nan16(q);

	else if (divisor->exp == EXP16_NAN && z_ds)
	    set_zero(q, divisor->sign ^ dividend->sign);

	else
	    set_infinity(q, divisor->sign ^ dividend->sign);
    }
}



#if QUAD_POINTER

void divtf3(void *q, void *ds, void *dd) {
unpacked16 dividend, divisor, quotient;

    unpack_quad(dd, dividend.m, &dividend.exp, &dividend.sign);
    unpack_quad(ds, divisor.m,  &divisor.exp,  &divisor.sign);

    divtf3_0(&quotient, &divisor, &dividend);
    pack_real_16(q, quotient.m, &quotient.exp, &quotient.sign);
}

#else

packed16 divtf3(packed16 ds, packed16 dd) {
unpacked16 divisor, dividend, quotient;
packed16 q;

    unpack_quad(ds.a, divisor.m,  &divisor.exp,  &divisor.sign);
    unpack_quad(dd.a, dividend.m, &dividend.exp, &dividend.sign);

    divtf3_0(&quotient, &divisor, &dividend);

    pack_real_16(&q, quotient.m, &quotient.exp, &quotient.sign);
    return q;
}

#endif



/* compare_unpacked()-- Compare a pair of arbitrary unpacked numbers that are
 * not Not-a-Numbers.  Returns less than zero, zero or greater than
 * zero. */

static int compare_unpacked(unpacked16 *a, unpacked16 *b) {

    if (a->exp == EXP16_NAN && b->exp == EXP16_NAN)  /* Infinities */
	return (a->sign == b->sign)
	    ? 0
	    : (a->sign ? -1 : 1);

    if (a->exp == EXP16_NAN)
	return a->sign ? -1 : 1;

    if (b->exp == EXP16_NAN)
	return b->sign ? 1 : -1;

    /* a and b are regular numbers at this point */

    /* Test for both numbers equal to zero, since that will fail the
     * sign test that follows. */

    if (a->exp == 0 && a->m[0] == 0 && a->m[1] == 0 &&
	a->m[2] == 0 && a->m[3] == 0 &&
	b->exp == 0 && b->m[0] == 0 && b->m[1] == 0 &&
	b->m[2] == 0 && b->m[3] == 0)
	return 0;
	
    if (a->sign == b->sign)
	return (a->sign ? -1 : 1) * compare16(a, b);

    return a->sign ? -1 : 1;
}


#if QUAD_POINTER

long gttf2(void *a, void *b) {
unpacked16 a1, b1;

    unpack_quad(a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}


long lttf2(void *a, void *b) {
unpacked16 a1, b1;

    unpack_quad(a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return -1;

    return compare_unpacked(&a1, &b1);
}


long getf2(void *a, void *b) {
unpacked16 a1, b1;

    unpack_quad(a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}


long letf2(void *a, void *b) {
unpacked16 a1, b1;

    unpack_quad(a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return -1;

    return compare_unpacked(&a1, &b1);
}


long eqtf2(void *a, void *b) {
unpacked16 a1, b1;

    unpack_quad(a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}


long netf2(void *a, void *b) {
unpacked16 a1, b1;

    unpack_quad(a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}

#else

long gttf2(packed16 a, packed16 b) {
unpacked16 a1, b1;

    unpack_quad(&a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(&b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}


long lttf2(packed16 a, packed16 b) {
unpacked16 a1, b1;

    unpack_quad(&a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(&b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return -1;

    return compare_unpacked(&a1, &b1);
}


long getf2(packed16 a, packed16 b) {
unpacked16 a1, b1;

    unpack_quad(&a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(&b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}


long letf2(packed16 a, packed16 b) {
unpacked16 a1, b1;

    unpack_quad(&a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(&b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return -1;

    return compare_unpacked(&a1, &b1);
}


long eqtf2(packed16 a, packed16 b) {
unpacked16 a1, b1;
long t;

    unpack_quad(&a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(&b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}


long netf2(packed16 a, packed16 b) {
unpacked16 a1, b1;

    unpack_quad(&a, a1.m, &a1.exp, &a1.sign);
    unpack_quad(&b, b1.m, &b1.exp, &b1.sign);

    if ((a1.exp == EXP16_NAN &&
	 (a1.m[0] != 0 || a1.m[1] != 0 || a1.m[2] != 0 || a1.m[3] != 0)) ||
	(b1.exp == EXP16_NAN &&
	 (b1.m[0] != 0 || b1.m[1] != 0 || b1.m[2] != 0 || b1.m[3] != 0)))
	return 1;

    return compare_unpacked(&a1, &b1);
}

#endif


/* Single to quad */

static void extendsftf2_0(unpacked16 *b, float a) {

    unpack_real_4(&a, b->m, &b->exp, &b->sign);

    if (b->exp == 0 && b->m[0] == 0)
	b->m[1] = b->m[2] = b->m[3] = 0;

    else if (b->exp != EXP4_NAN) {
	b->m[1] = b->m[0] << 25;
	b->m[0] = b->m[0] >> 7;

	b->m[2] = b->m[3] = 0;
	b->exp = b->exp + EXP16_BIAS - EXP4_BIAS;

    } else if (b->m[0] == 0)
	set_infinity(b, b->sign);

    else
	set_nan16(b);
}


#if QUAD_POINTER

void extendsftf2(packed16 *r, float a) {
unpacked16 q;

    extendsftf2_0(&q, a);
    pack_real_16(r, q.m, &q.exp, &q.sign);
}

#else

packed16 extendsftf2(float a) {
unpacked16 q;
packed16 r;

    extendsftf2_0(&q, a);
    pack_real_16(&r, q.m, &q.exp, &q.sign);
    return r;
}

#endif


/* Double to quad */

packed16 extenddftf2_0(unpacked16 *b, double a) {

    unpack_real_8(&a, b->m, &b->exp, &b->sign);

    if (b->exp == 0 && b->m[0] == 0 && b->m[1] == 0)
	b->m[2] = b->m[3] = 0;

    else if (b->exp != EXP8_NAN) {
	b->m[3] = 0;
	b->m[2] =                   b->m[1] << 28;
	b->m[1] = (b->m[1] >> 4) | (b->m[0] << 28);
	b->m[0] =  b->m[0] >> 4;

	b->exp = b->exp + EXP16_BIAS - EXP8_BIAS;

    } else if (b->m[0] == 0 && b->m[1] == 0)
	set_infinity(b, b->sign);

    else
	set_nan16(b);
}


#if QUAD_POINTER

void extenddftf2(packed16 *r, double a) {
unpacked16 q;

    extenddftf2_0(&q, a);
    pack_real_16(r, q.m, &q.exp, &q.sign);
}

#else

packed16 extenddftf2(double a) {
unpacked16 q;
packed16 r;

    extenddftf2_0(&q, a);
    pack_real_16(&r, q.m, &q.exp, &q.sign);
    return r;
}

#endif


/* kind=10 to quad */

packed16 extendxftf2(long double a) {
unpacked16 b;
packed16 q;

    unpack_real_10(&a, b.m, &b.exp, &b.sign);

    if (b.exp != EXP10_NAN) {
	b.m[3] = 0;
	b.m[2] =                   b.m[1] << 17;
	b.m[1] = (b.m[1] >> 15) | (b.m[0] << 17);
	b.m[0] =  b.m[0] >> 15;

	b.exp = b.exp + EXP16_BIAS - EXP10_BIAS;

    } else if (b.m[0] == 0 && b.m[1] == 0)
	set_infinity(&b, b.sign);

    else
	set_nan16(&b);

    pack_real_16(&q, b.m, &b.exp, &b.sign);
    return q;
}



/* kind=4 integer to quad.  Kind=1 and kind=2 integers also use this
 * interface. */

static void floatsitf_0(unpacked16 *r, int v) {
unsigned u;
int t;

    if (v == 0) {
	set_zero(r, 0);
	return;
    }

    if (v > 0) {
	r->sign = 0;
	u = v;

    } else {
	r->sign = 1;
	u = -v;
    }

    t = top_bit(u);
    r->exp = EXP16_BIAS + t;

    if (t > 16) {
	t = t - 16;

	r->m[0] = u >> t;
	r->m[1] = u << (32 - t);

    } else if (t < 16) {
	t = 16 - t;

	r->m[0] = u << t;
	r->m[1] = 0;

    } else {
	r->m[0] = u;
	r->m[1] = 0;
    }

    r->m[2] = 0;
    r->m[3] = 0;
}


#if QUAD_POINTER

void floatsitf(packed16 *q, long long v) {
unpacked16 r;

    floatsitf_0(&r, v);
    pack_real_16(&q, r.m, &r.exp, &r.sign);
}

#else

packed16 floatsitf(long long v) {
unpacked16 r;
packed16 q;

    floatsitf_0(&r, v);
    pack_real_16(&q, r.m, &r.exp, &r.sign);
    return q;
}

#endif


/* kind=8 integer to quad. */

static void floatditf_0(unpacked16 *r, long long v) {
unsigned long long u;
unsigned u1, u2;
int t, t2;

    if (v == 0) {
	set_zero(r, 0);
	return;
    }

    if (v > 0) {
	r->sign = 0;
	u = v;

    } else {
	r->sign = 1;
	u = -v;
    }

    u1 = u >> 32;
    u2 = u;

    if (u1 != 0) {
	t = top_bit(u1);
	r->exp = EXP16_BIAS + t + 32;

	if (t > 16) {
	    t = t - 16;
	    t2 = 32 - t;

	    r->m[0] =  u1 >> t;
	    r->m[1] = (u2 >> t) | (u1 << t2);
	    r->m[2] =              u2 << t2;

	} else if (t < 16) {
	    t = 16 - t;
	    t2 = 32 - t;

	    r->m[0] =  u1 << t;
	    r->m[1] = (u1 >> t2) | (u2 << t);
	    r->m[2] =               u2 >> t2;

	} else {
	    r->m[0] = u1;
	    r->m[1] = u2;
	    r->m[2] = 0;
	}

	r->m[3] = 0;

    } else {   /* u1 == 0 */
	t = top_bit(u2);
	r->exp = EXP16_BIAS + t;

	if (t > 16) {
	    t = t - 16;

	    r->m[0] = u2 >> t;
	    r->m[1] = u2 << (32 - t);

	} else if (t < 16) {
	    t = 16 - t;

	    r->m[0] = u2 << t;
	    r->m[1] = 0;

	} else {
	    r->m[0] = u2;
	    r->m[1] = 0;
	}

	r->m[2] = 0;
	r->m[3] = 0;
    }
}


#if QUAD_POINTER

void floatditf(packed16 *q, long long v) {
unpacked16 r;

    floatditf_0(&r, v);
    pack_real_16(&q, r.m, &r.exp, &r.sign);
}

#else

packed16 floatditf(long long v) {
unpacked16 r;
packed16 q;

    floatditf_0(&r, v);
    pack_real_16(&q, r.m, &r.exp, &r.sign);
    return q;
}

#endif


/* Quad to single */

float trunctfsf2_0(unpacked16 *a) {
unsigned u;
float r;

    if (a->exp == EXP16_NAN) {
	if (a->m[0] != 0 || a->m[1] != 0 || a->m[2] != 0 || a->m[3] != 0)
	    a->m[0] = 0xFFFFFFFF;

	a->exp = EXP4_NAN;

    } else {
	a->exp = a->exp - EXP16_BIAS + EXP4_BIAS;

	if (a->exp <= 0) {
	    a->exp = 0;
	    a->m[0] = 0;

	} else if (a->exp >= EXP4_NAN) {
	    a->exp = EXP4_NAN;
	    a->m[0] = 0;

	} else {
	    u = a->m[1] & 0x007FFFFF;

	    if (u > 0x00400000)
		round = BOTTOM_GT_HALF;

	    else if (u < 0x00400000)
		round = BOTTOM_LT_HALF;

	    else if (a->m[2] == 0 && a->m[3] == 0)
		round = BOTTOM_EQ_HALF;

	    else
		round = BOTTOM_GT_HALF;

	    a->m[0] = (a->m[0] << 7) | (a->m[1] >> 25);

	    if (round == BOTTOM_GT_HALF ||
		(round == BOTTOM_EQ_HALF && (a->m[0] & 1))) {

		if (++a->m[0] == 0x01000000) {  /* Round */
		    a->m[0] = 0x00800000;
		    if (++a->exp == EXP4_NAN)
			a->m[0] = 0;            /* Round to infinity */
		}
	    }
	}
    }

    pack_real_4(&r, &a->m[0], &a->exp, &a->sign);
    return r;
}


#if QUAD_POINTER

float trunctfsf2(void *q) {
unpacked16 a;

    unpack_quad(q, a.m, &a.exp, &a.sign);

    return trunctfsf2_0(&a);
}

#else

float trunctfsf2(packed16 q) {
unpacked16 a;

    unpack_quad(&q, a.m, &a.exp, &a.sign);

    return trunctfsf2_0(&a);
}

#endif



/* Quad to double */

double trunctfdf2_0(unpacked16 *a) {
unsigned u;
double r;

    if (a->exp == EXP16_NAN) {
	if (a->m[0] != 0 || a->m[1] != 0 || a->m[2] != 0 || a->m[3] != 0)
	    a->m[0] = a->m[1] = 0xFFFFFFFF;

	a->exp = EXP8_NAN;

    } else {
	a->exp = a->exp - EXP16_BIAS + EXP8_BIAS;

	if (a->exp <= 0) {
	    a->exp = 0;
	    a->m[0] = a->m[1] = 0;

	} else if (a->exp >= EXP8_NAN) {
	    a->exp = EXP8_NAN;
	    a->m[0] = a->m[1] = 0;

	} else {
	    u = a->m[2] & 0x000FFFFF;

	    if (u > 0x00080000)
		round = BOTTOM_GT_HALF;

	    else if (u < 0x00080000)
		round = BOTTOM_LT_HALF;

	    else if (a->m[3] == 0)
		round = BOTTOM_EQ_HALF;

	    else
		round = BOTTOM_GT_HALF;

	    a->m[0] = (a->m[0] << 4) | (a->m[1] >> 28);
	    a->m[1] = (a->m[1] << 4) | (a->m[2] >> 28);

	    if (round == BOTTOM_GT_HALF ||
		(round == BOTTOM_EQ_HALF && (a->m[1] & 1))) {

		if (++a->m[1] == 0 && ++a->m[0] == 0x00200000) {  /* Round */
		    a->m[0] = 0x00010000;
		    if (++a->exp == EXP8_NAN)
			a->m[0] = a->m[1] = 0;     /* Round to infinity */
		}
	    }
	}
    }

    pack_real_8(&r, &a->m[0], &a->exp, &a->sign);
    return r;
}


#if QUAD_POINTER

double trunctfdf2(void *q) {
unpacked16 a;

    unpack_quad(q, a.m, &a.exp, &a.sign);

    return trunctfdf2_0(&a);
}

#else

double trunctfdf2(packed16 q) {
unpacked16 a;

    unpack_quad(&q, a.m, &a.exp, &a.sign);

    return trunctfdf2_0(&a);
}

#endif



#if HAVE_REAL_10

/* Quad to kind=10.  No pointer version */

void trunctfxf2(packed16 q) {
unsigned u, r[3];
unpacked16 a;

    unpack_quad(&q, a.m, &a.exp, &a.sign);

    if (a.exp == EXP16_NAN) {
	if (a.m[0] != 0 || a.m[1] != 0 || a.m[2] != 0 || a.m[3] != 0)
	    a.m[0] = a.m[1] = 0xFFFFFFFF;

	a.exp = EXP10_NAN;

    } else {
	a.exp = a.exp - EXP16_BIAS + EXP10_BIAS;

	if (a.exp <= 0) {
	    a.exp = 0;
	    a.m[0] = a.m[1] = 0;

	} else if (a.exp >= EXP10_NAN) {
	    a.exp = EXP10_NAN;
	    a.m[0] = a.m[1] = 0;

	} else {
	    u = a.m[2] & 0x0001FFFF;

	    if (u > 0x00010000)
		round = BOTTOM_GT_HALF;

	    else if (u < 0x00010000)
		round = BOTTOM_LT_HALF;

	    else if (a.m[3] == 0)
		round = BOTTOM_EQ_HALF;

	    else
		round = BOTTOM_GT_HALF;

	    a.m[0] = (a.m[0] << 15) | (a.m[1] >> 17);
	    a.m[1] = (a.m[1] << 15) | (a.m[2] >> 17);

	    if (round == BOTTOM_GT_HALF ||
		(round == BOTTOM_EQ_HALF && (a.m[1] & 1))) {

		if (++a.m[1] == 0 && ++a.m[0] == 0) {  /* Round */
		    a.m[0] = 0x80000000;
		    if (++a.exp == EXP10_NAN)
			a.m[0] = a.m[1] = 0;     /* Round to infinity */
		}
	    }
	}
    }

    pack_real_10(&r, &a.m[0], &a.exp, &a.sign);
    asm("fldt %0\n" : : "m" (*(&r[0])));
}

#endif


/* Quad to kind=4 integer.  Kind=1 and kind=2 integers also use this
 * interface, which of course means that infinities don't come out
 * quite right.  Rounding is by truncation. */

static int fixtfsi_0(unpacked16 *a) {
int t, v;

    if (a->exp == EXP16_NAN) {
	if (a->m[0] != 0 || a->m[1] != 0 || a->m[2] != 0 || a->m[3] != 0)
	    return 0;

	return a->sign ? 0x7FFFFFFF : 0x80000000;
    }

    t = 16 - (a->exp - EXP16_BIAS);

    if (t > 16)
	v = 0;

    else if (t < -14)
	v = 0x7FFFFFFF;

    else if (t > 0)
	v = a->m[0] >> t;

    else if (t < 0)
	v = (a->m[0] << -t) | (a->m[1] >> (32 + t));

    else
	v = a->m[0];

    if (a->sign)
	v = -v;

    return v;
}


#if QUAD_POINTER

int fixtfsi(void *q) {
unpacked16 a;

    unpack_quad(q, a.m, &a.exp, &a.sign);

    return fixtfsi_0(&a);
}

#else

int fixtfsi(packed16 q) {
unpacked16 a;

    unpack_quad(&q, a.m, &a.exp, &a.sign);

    return fixtfsi_0(&a);
}

#endif



/* Quad to kind=8 integer.  Rounding is by truncation. */

static long long fixtfdi_0(unpacked16 *a) {
unsigned v1, v2;
unpacked16 b;
long long v;
int t;

    if (a->exp == EXP16_NAN) {
	if (a->m[0] != 0 || a->m[1] != 0 || a->m[2] != 0 || a->m[3] != 0)
	    return 0;

	return 0x8000000000000000ULL;
    }

    t = 112 - (a->exp - EXP16_BIAS);   /* t == right shifts 16 to 64+16 */

    if (t > 113)
	v = 0;

    else if (t < 50)
	v = 0x7FFFFFFFFFFFFFFFULL;

    else {
	shift(t, a, &b);

	v = b.m[2];
	v = (v << 32) | b.m[3];
    }

    if (a->sign)
	v = -v;

    return v;
}


#if QUAD_POINTER

long long fixtfdi(void *q) {
unpacked16 a;

    unpack_quad(q, a.m, &a.exp, &a.sign);

    return fixtfdi_0(&a);
}

#else

long long fixtfdi(packed16 q) {
unpacked16 a;

    unpack_quad(&q, a.m, &a.exp, &a.sign);

    return fixtfdi_0(&a);
}

#endif





static G95_DINT get_ieee_class(unpacked16 *x) {

    if (x->exp == EXP16_NAN) {
	if (x->m[0] == 0 && x->m[1] == 0 && x->m[2] == 0 && x->m[3] == 0)
	    return x->sign
		? CLASS_NEGATIVE_INF
		: CLASS_POSITIVE_INF;

	return (x->m[0] & MAN16_MSW)
	    ? CLASS_QUIET_NAN
	    : CLASS_SIGNALING_NAN;
    }

    if (x->exp != 0)
	return x->sign
	    ? CLASS_NEGATIVE_NORMAL
	    : CLASS_POSITIVE_NORMAL;

    if (x->m[0] == 0 && x->m[1] == 0 && x->m[2] == 0 && x->m[3] == 0)
	return x->sign
	    ? CLASS_NEGATIVE_ZERO
	    : CLASS_POSITIVE_ZERO;

    return x->sign
	? CLASS_NEGATIVE_DENORMAL
	: CLASS_POSITIVE_DENORMAL;
}


#define class_16 prefix(class_16)

#if QUAD_POINTER

G95_DINT class_16(void *x) {
unpacked16 a;

    unpack_quad(x, a.m, &a.exp, &a.sign);
    return get_ieee_class(&a);
}

#else

G95_DINT class_16(packed16 x) {
unpacked16 a;

    unpack_quad(&x, a.m, &a.exp, &a.sign);
    return get_ieee_class(&a);
}

#endif





static void ieee_value(int type, unpacked16 *result) {
unsigned m[1];
int e, s;
float r;

    switch(type) {
    case CLASS_SIGNALING_NAN:
	result->m[0] = 0;   result->m[1] = 0;
	result->m[2] = 0;   result->m[3] = 1;

	result->exp = EXP16_NAN;
	result->sign = 0;
	break;

    case CLASS_QUIET_NAN:
	result->m[0] = ~0;   result->m[1] = ~0;
	result->m[2] = ~0;   result->m[3] = ~0;

	result->exp = EXP16_NAN;
	result->sign = 0;
	break;

    case CLASS_POSITIVE_INF:
	result->m[0] = 0;   result->m[1] = 0;
	result->m[2] = 0;   result->m[3] = 0;

	result->exp = EXP16_NAN;
	result->sign = 0;
	break;

    case CLASS_NEGATIVE_INF:
	result->m[0] = 0;   result->m[1] = 0;
	result->m[2] = 0;   result->m[3] = 0;

	result->exp = EXP16_NAN;
	result->sign = 1;
	break;

    case CLASS_POSITIVE_DENORMAL:
	result->m[0] = 0;   result->m[1] = 0;
	result->m[2] = 0;   result->m[3] = 1;

	result->exp = 0;
	result->sign = 0;
	break;


    case CLASS_NEGATIVE_DENORMAL:
	result->m[0] = 0;   result->m[1] = 0;
	result->m[2] = 0;   result->m[3] = 1;

	result->exp = 0;
	result->sign = 0;
	break;


    case CLASS_POSITIVE_ZERO:
	result->m[0] = 0;   result->m[1] = 0;
	result->m[2] = 0;   result->m[3] = 0;

	result->exp = 0;
	result->sign = 0;
	break;

    case CLASS_NEGATIVE_ZERO:
	result->m[0] = 0;   result->m[1] = 0;
	result->m[2] = 0;   result->m[3] = 0;

	result->exp = 0;
	result->sign = 1;
	break;

    case CLASS_POSITIVE_NORMAL:
	result->m[0] = MAN16_MSW;   result->m[1] = 0;
	result->m[2] = 0;           result->m[3] = 0;

	result->exp = 16383;
	result->sign = 0;
	break;

    case CLASS_NEGATIVE_NORMAL:
	result->m[0] = MAN16_MSW;   result->m[1] = 0;
	result->m[2] = 0;           result->m[3] = 0;

	result->exp = 16383;
	result->sign = 1;
	break;

    default:
	runtime_error("Bad class value passed to IEEE_VALUE()");
    }
}




#define value_16 prefix(value_16)

#if QUAD_POINTER

void value_16(void *r, void *dummy, G95_DINT *type) {
unpacked16 result;

    ieee_value(*type, &result);
    pack_real_16(r, result.m, &result.exp, &result.sign);
}

#else

packed16 value_16(void *dummy, G95_DINT *type) {
unpacked16 result;
packed16 r;

    ieee_value(*type, &result);
    pack_real_16(&r, result.m, &result.exp, &result.sign);

    return r;
}

#endif

