
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)

*/


/* Runtime library include */

#include "config.h"

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#define stringize(x) expand_macro(x)
#define expand_macro(x) # x

#define DEFAULT_TEMPDIR "/var/tmp"

#define G95_MAX_DIMENSIONS 7
#define G95_MAX_SYMBOL_LEN 64

/* Constants normally found in stdint.h, not present on some
 * platforms.  The values of the constants are, however, pretty much
 * carved in stone.  Other kinds are configured dynamically depending
 * on the platform. */

#define G95_INT4_MAX   2147483647
#define G95_UINT4_MAX  4294967295U

#define G95_INT2_MAX   32767
#define G95_UINT2_MAX  65535U

#define G95_INT1_MAX    127
#define G95_UINT1_MAX   255U


#ifdef G95_INT16
#define G95_MINT G95_INT16
#else
#define G95_MINT G95_INT8
#endif


#if HAVE_REAL_10 == 3
#define EAX "%rax"
#define EBX "%rbx"
#define REAL10_SIZE 16

#else
#define EAX "%eax"
#define EBX "%ebx"
#define REAL10_SIZE 12
#endif


#if (HAVE_REAL_10 == 2) || defined(__APPLE__)
#define FLD_HALF "flds __g95_half\n"
#else
#define FLD_HALF "flds _g95_half\n"
#endif


#define REAL_SIZE(k) (((k) == 10) ? REAL10_SIZE : (k))


/* For a library, a standard prefix is a requirement in order to
 * partition the namespace.  It's ugly to look at and a pain to type,
 * so we hide it behind a macro. */

#define prefix(x) _g95_ ## x

/* external_prefix() is for mangling external names in the same manner
 * as user procedures.  This is not totally correct. */

#define external_prefix(x) x ## _


typedef enum {
    SIGDISP_ABORT, SIGDISP_IGNORE, SIGDISP_DUMP, SIGDISP_DUMP_QUIT
} signal_disp;


typedef struct { char x[10]; } r10;


#if HAVE_WINDOWS
#include <windows.h>

#define OS_HANDLE HANDLE
#define off_t long long

#define win_read prefix(win_read)
int win_read(OS_HANDLE, char *, int);

#define win_write prefix(win_write)
int win_write(OS_HANDLE, char *, int);

#define win_lseek prefix(win_lseek)
int win_lseek(OS_HANDLE, off_t, int);

#define win_close prefix(win_close)
int win_close(OS_HANDLE);

#define win_ftruncate prefix(win_ftruncate)
int win_ftruncate(OS_HANDLE, off_t position);

#define READ win_read
#define WRITE win_write
#define LSEEK_ABS(fd, pos) win_lseek(fd, pos, FILE_BEGIN)
#define LSEEK_REL(fd, pos) win_lseek(fd, pos, FILE_CURRENT)
#define LSEEK_END(fd, pos) win_lseek(fd, pos, FILE_END)
#define CLOSE win_close
#define FTRUNCATE win_ftruncate

#define PATH_LENGTH MAX_PATH

#else

#include <limits.h>
#include <sys/param.h>

#define OS_HANDLE int

#define READ  read
#define WRITE write
#define LSEEK_ABS(fd, pos) lseek(fd, pos, SEEK_SET)
#define LSEEK_REL(fd, pos) lseek(fd, pos, SEEK_CUR)
#define LSEEK_END(fd, pos) lseek(fd, pos, SEEK_END)
#define CLOSE close
#define FTRUNCATE ftruncate

#define PATH_LENGTH PATH_MAX

#endif



#if HAVE_WINDOWS && !defined(__CYGWIN__)
#define G95_CR_DEFAULT 1
#else
#define G95_CR_DEFAULT 0
#endif



/* Runtime options structure */

typedef struct {
    int stdin_unit, stdout_unit, stderr_unit, optional_plus;
    int allocate_init_flag;
    G95_DINT allocate_init_value;
    int locus, seed_rng, minus_zero, ignore_endfile, abort;

    int separator_len;
    char *separator;

    int mem_segments, mem_max_alloc, mem_mxfast, mem_trim_threshold;
    int mem_top_pad, use_stderr, all_unbuffered, default_recl;

    int system_clock_ticks, list_exp, stop_code;

    int fpu_round, fpu_precision, fpu_invalid, fpu_denormal, fpu_zerodiv,
	fpu_overflow, fpu_underflow, fpu_inexact, fpu_no_denormals;

    int view_exceptions, expand_unprintable, cr, input_cr, decimal_comma;
    int quiet;

    signal_disp sighup, sigint, sigquit;
} options_t;


#define options prefix(options)
extern options_t options;


/* Structure for statement options */

typedef struct { 
    char *name;
    int value;
} st_option;

/* Basic types used in data transfers.  */

typedef enum { BT_NULL, BT_INTEGER, BT_LOGICAL, BT_CHARACTER, BT_REAL,
	       BT_COMPLEX, BT_DERIVED } bt;


typedef enum { SUCCESS=1, FAILURE } try;



#define BUFFER_SIZE 16384

typedef struct {
    OS_HANDLE fd;
    off_t physical_offset;   /* Current physical file offset */
    off_t buffer_offset;     /* File offset of the start of the buffer */
    off_t dirty_offset;      /* Start of modified bytes in buffer */
    off_t logical_offset;    /* Current logical file offset */
    off_t logical_size;
    off_t physical_size;

    int active;             /* Length of valid bytes in the buffer */
    int len;                /* Physical length of the current buffer */

    int ndirty;             /* Dirty bytes starting at dirty_offset */
    int unbuffered, truncate;

    char *buffer, small_buffer[BUFFER_SIZE];

} stream;


/* Options for the OPEN statement */

typedef enum { ACCESS_SEQUENTIAL, ACCESS_DIRECT, ACCESS_STREAM, ACCESS_APPEND,
	       ACCESS_UNSPECIFIED } unit_access;

typedef enum { ACTION_READ, ACTION_WRITE, ACTION_READWRITE,
	       ACTION_UNSPECIFIED } unit_action;

typedef enum { BLANK_NULL, BLANK_ZERO, BLANK_UNSPECIFIED } unit_blank;

typedef enum { DELIM_NONE, DELIM_APOSTROPHE, DELIM_QUOTE,
	       DELIM_UNSPECIFIED } unit_delim;

typedef enum { FORM_FORMATTED, FORM_UNFORMATTED, FORM_UNSPECIFIED } unit_form;

typedef enum { POSITION_ASIS, POSITION_REWIND, POSITION_APPEND,
	       POSITION_UNSPECIFIED } unit_position;

typedef enum { STATUS_UNKNOWN, STATUS_OLD, STATUS_NEW, STATUS_SCRATCH,
	       STATUS_REPLACE, STATUS_UNSPECIFIED } unit_status;

typedef enum { PAD_YES, PAD_NO, PAD_UNSPECIFIED } unit_pad;

typedef enum { ADVANCE_YES, ADVANCE_NO, ADVANCE_UNSPECIFIED } unit_advance;

typedef enum { DECIMAL_POINT, DECIMAL_COMMA,
	       DECIMAL_UNSPECIFIED } unit_decimal;


typedef struct {
    char *offset;
    void *base;
    G95_DINT rank, corank, element_size;

    struct {
	G95_AINT mult, lbound, ubound;
    } info[G95_MAX_DIMENSIONS];
} g95_array_descriptor;

#define g95_descriptor_size(rank) \
 (sizeof(g95_array_descriptor) + 3*(rank-G95_MAX_DIMENSIONS)*sizeof(G95_AINT))


typedef struct frame {
    char *f_filename;
    struct frame *next;
    G95_DINT f_line;
} frame;


typedef enum {
    ENDIAN_NATIVE=0, ENDIAN_BIG, ENDIAN_LITTLE, ENDIAN_SWAP, ENDIAN_UNSPECIFIED
} endian_t;


typedef struct {
    unit_access access;
    unit_action action;
    unit_blank blank;
    unit_delim delim;
    unit_form form;
    int is_notpadded;
    unit_position position;
    unit_status status;
    unit_pad pad;
    unit_decimal decimal;
    endian_t endian;
} unit_flags;


/* The default value of record length is defined here.  This value can
 * be overridden by the OPEN statement or by an environment variable. */

#define DEFAULT_RECL 1000000000
#define INITIAL_RECL 2000

typedef struct iounit_t {
    G95_MINT unit_number;

    stream *s;

    struct iounit_t *left, *right;  /* Treap links */
    int priority;

    int read_bad, repos, previous_noadvance;
    enum {
	NO_ENDFILE, AT_ENDFILE, AFTER_ENDFILE
    } endfile;

    unit_flags flags;
    off_t recl, last_record;

    char *record;
    int record_size, record_psize, offset, max_offset, read_eof, reverse;

    /* recl         -- Record length of the file
     * last_record  -- Last record number read or written
     * record       -- Pointer to current buffer
     * record_psize -- Physical size of record buffer allocated on heap
     * record_size  -- Actual size of record being read
     * offset       -- Current pointer within a record
     * max_offset   -- Maximum offset written within a record
     * read_eof     -- EOF has been seen */

    int file_len;
    char file[1];    /* Filename is allocated at the end of the structure */
} iounit_t;


enum {
    LIBRARY_OK=0,
    LIBRARY_ERROR,
    LIBRARY_END,
    LIBRARY_EOR
};

/* Statement parameters.  These are all the things that can appear in
 * an I/O statement.  Some are inputs and some are outputs, but none
 * are both.  All of these values are initially zeroed and are zeroed
 * at the end of a library statement.  The relevant values need to be
 * set before entry to an I/O statement.  This structure needs to be
 * duplicated by the back end. */

typedef struct st_parameter {
    void *unit; G95_DINT unit_kind;
    G95_DINT err, end, eor, list_format;   /* These are flags, not values */

/* Return values from library statements.  These are returned only if
 * the labels are specified in the statement itself and the condition
 * occurs.  In most cases, none of the labels are specified and the
 * return value does not have to be checked.  Must be consistent with
 * the front end. */

    G95_DINT library_rc, sloppy_char, endian;
    void *exist;		G95_DINT exist_kind;
    void *opened;		G95_DINT opened_kind;
    void *named;		G95_DINT named_kind;

    void *iostat;		G95_DINT iostat_kind;
    void *number;		G95_DINT number_kind;
    void *nextrec;		G95_DINT nextrec_kind;
    void *size;			G95_DINT size_kind;
    void *recl_in;		G95_DINT recl_in_kind;
    void *rec;			G95_DINT rec_kind;
    G95_DINT			*recl_out;

    char *file;			G95_DINT file_len;
    char *status;		G95_DINT status_len;
    char *access;		G95_DINT access_len;
    char *form;			G95_DINT form_len;
    char *blank;		G95_DINT blank_len;
    char *position;		G95_DINT position_len;
    char *action;		G95_DINT action_len;
    char *delim;		G95_DINT delim_len;
    char *pad;			G95_DINT pad_len;
    char *format;		G95_DINT format_len;
    char *advance;		G95_DINT advance_len;
    char *name;			G95_DINT name_len;
    char *decimal;		G95_DINT decimal_len;
    char *internal_unit;	G95_DINT internal_unit_len;
    g95_array_descriptor	*internal_array;
    char *namelist;
    char *sequential;		G95_DINT sequential_len;
    char *direct;		G95_DINT direct_len;
    char *formatted;		G95_DINT formatted_len;
    char *unformatted;		G95_DINT unformatted_len;
    char *read;			G95_DINT read_len;
    char *write;		G95_DINT write_len;
    char *readwrite;		G95_DINT readwrite_len;
    char *iomsg;		G95_DINT iomsg_len;
    char *pos;			G95_DINT pos_kind;
    char *convert;              G95_DINT convert_len;

    struct st_parameter *prev;
    void (*transfer)(bt, void *, int);
    iounit_t *unit_save;
    struct fnode *fnode_base, *fnode_save;
    char *file_save;
    int reversion_ok, scale_factor, seen_dollar;
    int iolength_size, reversion_flag;    /* Format reversion has occurred */
    int first_item;

    unit_advance advance_status;
    unit_blank blank_status;
    enum { SIGN_S, SIGN_SS, SIGN_SP } sign_status;
    enum { READING, WRITING } mode;

    unit_decimal current_decimal;

    G95_DINT line_save;
    G95_DINT item_count;      /* Item number in a formatted data transfer */

} st_parameter;


#define ioparm prefix(ioparm)

extern st_parameter *ioparm;


/* Runtime errors.  The EOR and EOF errors are required to be negative. */

typedef enum {
    ERROR_FIRST=-3,          /* Marker for the first error */
    ERROR_EOR=-2,
    ERROR_END=-1,
    ERROR_OK=0,              /* Indicates success, must be zero */
    ERROR_OS,                /* Operating system error, errno is returned. */
    ERROR_OPTION_CONFLICT=200,
    ERROR_BAD_OPTION,
    ERROR_MISSING_OPTION,
    ERROR_ALREADY_OPEN,
    ERROR_BADUNIT,
    ERROR_FORMAT,
    ERROR_BAD_ACTION,
    ERROR_ENDFILE,
    ERROR_READ_VALUE,
    ERROR_READ_OVERFLOW,
    ERROR_NOMEM,
    ERROR_ALREADY_ALLOC,
    ERROR_BADFREE,
    ERROR_CORRUPT_DIRECT,
    ERROR_CORRUPT_SEQUENTIAL,
    ERROR_READ_RECL,
    ERROR_WRITE_RECL,
    ERROR_STAT_STOPPED_IMAGE,   /* Front end assumes = 217 */
    ERROR_LAST                  /* Not a real error, the last error # + 1 */
} error_codes;




/* Global variables.  Putting these in a structure makes it easier to
 * maintain, particularly with the constraint of a prefix.  */

typedef struct {
    iounit_t *unit_root;
    char *brk_save;
    int alarm_value, checkpoint_msg;
} global_t;

extern char **f__xargv;
extern int f__xargc;


#define globals prefix(globals)
extern global_t globals;

#define current_unit prefix(current_unit)
extern iounit_t *current_unit;

/* The filename and line number don't go inside the globals structure.
 * They are set by the rest of the program and must be linked to. */

#define line prefix(line)
extern unsigned G95_DINT line;

#define filename prefix(filename)
extern char *filename;

#define error_filename prefix(error_filename)
extern char *error_filename;

#ifndef NULL
#define NULL (void *) 0
#endif


/* Format tokens.  Only about half of these can be stored in the
 * format nodes. */

typedef enum {
    FMT_NONE=0, FMT_UNKNOWN, FMT_SIGNED_INT, FMT_ZERO, FMT_POSINT, FMT_PERIOD,
    FMT_COMMA, FMT_COLON, FMT_SLASH, FMT_DOLLAR, FMT_T, FMT_TR, FMT_TL,
    FMT_LPAREN, FMT_RPAREN, FMT_X, FMT_S, FMT_SS, FMT_SP, FMT_STRING,
    FMT_BADSTRING, FMT_P, FMT_I, FMT_B, FMT_BN, FMT_BZ, FMT_O, FMT_Z, FMT_F,
    FMT_E, FMT_EN, FMT_ES, FMT_G, FMT_L, FMT_A, FMT_D, FMT_H, FMT_DC, FMT_DP,
    FMT_BAD, FMT_END
} format_token;


/* Format nodes.  A format string is converted into a tree of these
 * structures, which is traversed as part of a data transfer
 * statement.  */

typedef struct fnode {
    format_token format;
    int repeat;
    struct fnode *next;
    char *source;

    union {
	struct {
	    int w, d, e;
	} real;
    
	struct {
	    int length;
	    char *p;
	} string;

	struct {
	    int w, m;
	} integer;

	int w;
	int k;
	int r;
	int n;

	struct fnode *child;
    } u;

    /* Members for traversing the tree during data transfer */

    int count;
    struct fnode *current;

} fnode;


/* Complex number structures */

typedef struct {
    float r, c;
} z4;

typedef struct {
    double r, c;
} z8;


/* Flavors of floating point numbers */

typedef enum {
    FF_REGULAR, FF_PLUS_INFINITY, FF_MINUS_INFINITY, FF_NAN
} float_flavor;


double trunc(double);
float truncf(float);

/* Derived type descriptor */

typedef struct derived_info {
    char *name;
    G95_DINT type, kind, offset, rank, *shape;
    struct derived_info *info;
} derived_info;


/* Namelist descriptor */

typedef struct namelist_info {
    char *name;
    G95_DINT type, kind;

    void *pointer;
    derived_info *dt_info;

    struct namelist_info *next;
} namelist_info;


enum {
    FP_ROUND_NEAREST, FP_ROUND_UP, FP_ROUND_DOWN, FP_ROUND_ZERO
};


typedef struct {
    G95_DINT offset, count, size, coarray;
    void *pointer;
} alloc_struct;



#define my_endian prefix(my_endian)
#define default_endian prefix(default_endian)
extern endian_t my_endian, default_endian;


#define DECIMAL_CHAR() (ioparm->current_decimal == DECIMAL_POINT ? '.' : ',')
#define COMMA_SEP()    (ioparm->current_decimal == DECIMAL_POINT ? ',' : ';')


/* stream.c */

#define fd_to_stream prefix(fd_to_stream)
stream *fd_to_stream(OS_HANDLE fd, int);

#define compare_files prefix(compare_files)
int compare_files(stream *, stream *);

#define unpack_filename prefix(unpack_filename)
int unpack_filename(char *, char *, int);

#define flush_stream prefix(flush_stream)
try flush_stream(stream *);

#define salloc_rline prefix(salloc_rline)
char *salloc_rline(stream *, unsigned *);

#define salloc_w prefix(salloc_w)
char *salloc_w(stream *, unsigned, int);

#define sfree prefix(sfree)
try sfree(stream *);

#define sseek prefix(sseek)
try sseek(stream *, off_t);

#define sclose prefix(sclose)
try sclose(stream *);

#define file_length prefix(file_length)
off_t file_length(stream *);

#define file_position prefix(file_position)
off_t file_position(stream *);

#define is_seekable prefix(is_seekable)
int is_seekable(stream *);

#define default_action prefix(default_action)
unit_action default_action(void);

#define salloc_r prefix(salloc_r)
char *salloc_r(stream *, unsigned *);


/* error.c */

#define st_printf prefix(st_printf)
int st_printf(char *, ...);

#define st_sprintf prefix(st_sprintf)
void st_sprintf(char *, char *, ...);

#define translate_error prefix(translate_error)
char *translate_error(int);

#define generate_error prefix(generate_error)
void generate_error(int, char *);

#define bad_dim prefix(bad_dim)
void bad_dim(void);

#define os_error prefix(os_error)
void os_error(char *);

#define show_locus prefix(show_locus)
void show_locus(void);

#define runtime_error prefix(runtime_error)
void runtime_error(char *);

#define internal_error prefix(internal_error)
void internal_error(char *);

#define write_error prefix(write_error)
void write_error(char *);


#define frame_base prefix(base)
extern frame *frame_base;


/* environ.c */

#define check_unbuffered prefix(check_buffered)
int check_unbuffered(G95_MINT, int);

#define init_variables prefix(init_variables)
void init_variables(void);

#define show_variables prefix(show_variables)
void show_variables(void);


/* io.c */

#define sys_exit prefix(sys_exit)
void sys_exit(int, int);

#define get_oserror prefix(get_oserror)
char *get_oserror(void);

#define get_oserrno prefix(get_oserrno)
int get_oserrno(void);

#define open_file prefix(open_file)
int open_file(unit_action, unit_status, OS_HANDLE *, char *);

#define open_tempfile prefix(open_tempfile)
int open_tempfile(char *, OS_HANDLE *);

#define standard_desc prefix(standard_desc)
int standard_desc(OS_HANDLE);

#define fd_length prefix(fd_length)
off_t fd_length(OS_HANDLE);

#define filename_length prefix(filename_length)
off_t filename_length(void);

#define delete_file prefix(delete_file)
int delete_file(char *);

#define compare_file_filename prefix(compare_file_filename)
int compare_file_filename(iounit_t *, char *, int);

#define init_error_stream prefix(init_error_stream)
stream *init_error_stream(void);

#define terminal_device prefix(terminal_device)
int terminal_device(iounit_t *);

#define file_exists prefix(file_exists)
int file_exists(void);

#define inquire_sequential prefix(inquire_sequential)
char *inquire_sequential(char *, int);

#define inquire_direct prefix(inquire_direct)
char *inquire_direct(char *, int);

#define inquire_sequential_fd prefix(inquire_sequential_fd)
char *inquire_sequential_fd(iounit_t *);

#define inquire_direct_fd prefix(inquire_direct_fd)
char *inquire_direct_fd(iounit_t *);

#define inquire_formatted prefix(inquire_formatted)
char *inquire_formatted(char *, int);

#define inquire_unformatted prefix(inquire_unformatted)
char *inquire_unformatted(char *, int);

#define inquire_formatted_fd prefix(inquire_formatted_fd)
char *inquire_formatted_fd(iounit_t *);

#define inquire_unformatted_fd prefix(inquire_unformatted_fd)
char *inquire_unformatted_fd(iounit_t *);

#define inquire_read prefix(inquire_read)
char *inquire_read(char *, int);

#define inquire_read_fd prefix(inquire_read_fd)
char *inquire_read_fd(iounit_t *);

#define truncate_file prefix(truncate_file)
try truncate_file(stream *);

#define open_external prefix(open_external)
stream *open_external(unit_action, unit_status, char *);

#define input_stream prefix(input_stream)
stream *input_stream(void);

#define output_stream prefix(output_stream)
stream *output_stream(void);

#define error_stream prefix(error_stream)
stream *error_stream(void);

#define find_file prefix(find_file)
iounit_t *find_file(void);

#define inquire_write prefix(inquire_write)
char *inquire_write(char *, int);

#define inquire_write_fd prefix(inquire_write_fd)
char *inquire_write_fd(iounit_t *);

#define inquire_readwrite prefix(inquire_readwrite)
char *inquire_readwrite(char *, int);

#define inquire_readwrite_fd prefix(inquire_readwrite_fd)
char *inquire_readwrite_fd(iounit_t *);


/* unit.c */

#define insert_unit prefix(insert_unix)
void insert_unit(iounit_t *);

#define init_units prefix(init_units)
void init_units(void);

#define close_unit prefix(close_unit)
int close_unit(iounit_t *);

#define close_units prefix(close_units)
void close_units(void);

#define is_internal_unit prefix(is_internal_unit)
int is_internal_unit(void);

#define find_unit prefix(find_unit)
iounit_t *find_unit(void *, G95_DINT);

#define open_unit prefix(open_unit)
iounit_t *open_unit(unit_flags *);

#define get_unit prefix(get_unit)
iounit_t *get_unit(void);

#define flush_units prefix(flush_units)
void flush_units(void);


/* main.c */

#define init_flag prefix(init_flag)
extern int init_flag;

#define junk_stat prefix(junk_stat)
extern G95_DINT junk_stat;

#define library_start prefix(library_start)
void library_start(void);

#define library_end prefix(library_end)
void library_end(void);

#define xorshift128 prefix(xorshift128)
int xorshift128(void);

#define get_ioparm prefix(get_ioparm)
void get_ioparm(void);

void __MAIN(void);

void g95_runtime_start(int, char **);
void g95_runtime_stop(void);


/* string.c */

#define fstrlen prefix(fstrlen)
int fstrlen(char *, int);

#define expanded_string_length prefix(expanded_string_length)
G95_DINT expanded_string_length(char *, G95_DINT, int);

#define copy_string_expand prefix(copy_string_expand)
void copy_string_expand(char *, G95_DINT, char *, G95_DINT, int);

#define copy_string prefix(copy_string)
void copy_string(char *, G95_DINT, char *, G95_DINT);

#define compare_string prefix(compare_string)
G95_DINT compare_string(char *, G95_DINT, char *, G95_DINT);

#define find_option prefix(find_option)
int find_option(char *, int, st_option *, char *);

#define string_copy_in prefix(string_copy_in)
char *string_copy_in(char *, int);

#define string_copy_out prefix(string_copy_out)
void string_copy_out(char *, char *, int);

#define fix_string prefix(fix_string)
void fix_string(char *, int);


/* format.c */

#define parse_format prefix(parse_format)
void parse_format(void);

#define next_format prefix(next_format)
fnode *next_format(void);

#define unget_format prefix(unget_format)
void unget_format(fnode *);

#define format_error prefix(format_error)
void format_error(fnode *, char *);

#define free_fnodes prefix(free_fnodes)
void free_fnodes(void);


/* memory.c */

#define memory_done prefix(memory_done)
void memory_done(void);

#define array_from_section prefix(array_from_section)
g95_array_descriptor *array_from_section(char *);

#define get_mem prefix(get_mem)
void *get_mem(size_t);

#define free_mem prefix(free_mem)
void free_mem(void *);

#define temp_alloc prefix(temp_alloc)
void *temp_alloc(G95_DINT);

#define temp_free prefix(temp_free)
void temp_free(void **);

#define init_memory prefix(init_memory)
void init_memory(void);

#define allocate_array prefix(allocate_array)
void allocate_array(g95_array_descriptor *, G95_DINT, void *, G95_DINT);

#define deallocate_array prefix(deallocate_array)
void deallocate_array(g95_array_descriptor *, alloc_struct *, G95_DINT);

#define deep_dealloc prefix(deep_dealloc)
void deep_dealloc(char *, alloc_struct *);

#define deep_copy prefix(deep_copy)
void deep_copy(char *, char *, G95_DINT, alloc_struct *);


/* open.c */

#define test_endfile prefix(test_endfile)
void test_endfile(iounit_t *);

/* transfer.c */

#define SCRATCH_SIZE 300

#define scratch prefix(scratch)
extern char scratch[];

#define type_name prefix(type_name)
char *type_name(bt);

#define next_list_char prefix(next_list_char)
int next_list_char(void);

#define read_block prefix(read_block)
char *read_block(unsigned *);

#define write_block prefix(write_block)
char *write_block(int);

#define next_record prefix(next_record)
void next_record(void);

#define get_length prefix(get_length)
G95_INT4 get_length(unsigned char *);

/* Data transfer entry points.  The type of the data entity is
 * implicit in the subroutine call.  This prevents us from having to
 * share a common enum with the compiler. */

#define transfer_integer    prefix(transfer_integer)
void transfer_integer(void *, G95_DINT);

#define transfer_real       prefix(transfer_real)
void transfer_real(void *, G95_DINT);

#define transfer_logical    prefix(transfer_logical)
void transfer_logical(void *, G95_DINT);

#define transfer_character  prefix(transfer_character)
void transfer_character(char *, G95_DINT);

#define transfer_complex    prefix(transfer_complex)
void transfer_complex(void *, G95_DINT);

#define transfer_derived    prefix(transfer_derived)
void transfer_derived(char *, derived_info *);

#define transfer_integer_array    prefix(transfer_integer_array)
void transfer_integer_array(g95_array_descriptor *, G95_DINT);

#define transfer_real_array       prefix(transfer_real_array)
void transfer_real_array(g95_array_descriptor *, G95_DINT);

#define transfer_logical_array    prefix(transfer_logical_array)
void transfer_logical_array(g95_array_descriptor *, G95_DINT);

#define transfer_character_array  prefix(transfer_character_array)
void transfer_character_array(g95_array_descriptor *, G95_DINT);

#define transfer_complex_array    prefix(transfer_complex_array)
void transfer_complex_array(g95_array_descriptor *, G95_DINT);

#define transfer_derived_array    prefix(transfer_derived_array)
void transfer_derived_array(g95_array_descriptor *, derived_info *);

#define terminate_record prefix(terminate_record)
void terminate_record(iounit_t *);

/* resume.c */

#define dump_signal prefix(dump_signal)
void dump_signal(int);

/* lread.c */

#define list_formatted_read prefix(list_formatted_read)
void list_formatted_read(bt, void *, int);

#define finish_list_read prefix(finish_list_read)
void finish_list_read(void);

#define namelist_read prefix(namelist_read)
void namelist_read(void);

/* write.c */

#define star_fill(p, n) memset(p, '*', n)
typedef enum { SIGN_NONE, SIGN_MINUS, SIGN_PLUS } sign_t;

#define calculate_sign prefix(calculate_sign)
sign_t calculate_sign(int);

#define write_a prefix(write_a)
void write_a(fnode *, char *, int);

#define write_l prefix(write_l)
void write_l(fnode *, char *, int);

#define write_x prefix(write_x)
void write_x(fnode *);

#define list_formatted_write prefix(list_formatted_write)
void list_formatted_write(bt, void *, int);

#define namelist_write prefix(namelist_write)
void namelist_write(void);

#define extract_logical prefix(extract_logical)
int extract_logical(void *, int);

/* read.c */

#define set_integer prefix(set_integer)
void set_integer(int, void *, int);

#define set_real prefix(set_real)
void set_real(int, void *, int);

#define extract_integer prefix(extract_integer)
int extract_integer(void *, int);

#define extract_mint prefix(extract_mint)
G95_MINT extract_mint(void *, int);

#define set_mint prefix(set_mint)
void set_mint(G95_MINT, void *, int);

#define extract_dint prefix(extract_dint)
G95_DINT extract_dint(void *, int);

#define read_a prefix(read_a)
void read_a(fnode *, char *, int);

#define read_f prefix(read_f)
void read_f(fnode *, char *, int);

#define read_l prefix(read_l)
void read_l(fnode *, char *, int);

#define eat_leading_spaces prefix(eat_leading_spaces)
char *eat_leading_spaces(unsigned *, char *);

/* iread.c */

#define read_radix prefix(read_radix)
void read_radix(fnode *, void *, int, int);

#define read_decimal prefix(read_decimal)
void read_decimal(fnode *, void *, int);

#define read_list_integer prefix(read_list_integer)
int read_list_integer(char *, int, int, void *);

/* iwrite.c */

#define write_b prefix(write_b)
void write_b(fnode *, void *, int);

#define write_o prefix(write_o)
void write_o(fnode *, void *, int);

#define write_z prefix(write_z)
void write_z(fnode *, void *, int);

#define write_i prefix(write_i)
void write_i(fnode *, void *, int);

#define write_integer prefix(write_integer)
void write_integer(void *, int);

#define int_to_a prefix(int_to_a)
char *int_to_a(G95_DINT);

#define long_to_a prefix(long_to_a)
char *long_to_a(G95_AINT);

#define mint_to_a prefix(mint_to_a)
char *mint_to_a(G95_MINT);

#define hex_int prefix(hex_int)
char *hex_int(unsigned G95_DINT);

/* fpu_trap.c */

#define init_fpu_trap prefix(init_fpu_trap)
void init_fpu_trap(void);

/* array.c */

#define section_info prefix(section_info)
extern G95_AINT section_info[];

#define temp_array prefix(temp_array)
g95_array_descriptor *temp_array(int, int, ...);

#define init_multipliers prefix(init_multipliers)
void init_multipliers(g95_array_descriptor *);

#define bump_element prefix(bump_element)
int bump_element(g95_array_descriptor *, G95_AINT []);

#define bump_element_dim prefix(bump_element_dim)
int bump_element_dim(g95_array_descriptor *, G95_AINT [], int);

/* fwrite.c */

#define write_real prefix(write_real)
void write_real(char *, int);

#define write_d prefix(write_d)
void write_d(fnode *, char *, int);

#define write_e prefix(write_e)
void write_e(fnode *, char *, int);

#define write_en prefix(write_en)
void write_en(fnode *, char *, int);

#define write_es prefix(write_es)
void write_es(fnode *, char *, int);

#define write_f prefix(write_f)
void write_f(fnode *, char *, int);

#define write_gi prefix(write_gi)
void write_gi(fnode *, char *, int);

#define write_gr prefix(write_gr)
void write_gr(fnode *, char *, int);

#define st_open prefix(st_open)
#define st_close prefix(st_close)
#define st_inquire prefix(st_inquire)
#define st_rewind prefix(st_rewind)
#define st_read prefix(st_read)
#define st_read_done prefix(st_read_done)
#define st_write prefix(st_write)
#define st_write_done prefix(st_write_done)
#define st_backspace prefix(st_backspace)
#define st_endfile prefix(st_endfile)

/* fpu_set.c */

#define init_fpu prefix(init_fpu)
void init_fpu(void);

#define show_exceptions prefix(show_exceptions)
void show_exceptions(void);

/* random.c */

#define init_random_seed prefix(init_random_seed)
void init_random_seed(int);

/* flavor.c */

#define get_float_flavor prefix(get_float_flavor)
float_flavor get_float_flavor(void *, int, char *);

/* namelist.c */

#define first_namelist prefix(first_namelist)
namelist_info *first_namelist(void);

#define find_namelist prefix(find_namelist)
namelist_info *find_namelist(char *);

#define namelist_done prefix(namelist_done)
void namelist_done(void);

/* ff.c */

#define expand_real prefix(expand_real)
int expand_real(char *, int, int *);

#define get_sign prefix(get_sign)
int get_sign(void *p, int);

#define huge_8 prefix(huge_8)
double huge_8(void);

#define huge_4 prefix(huge_4)
float huge_4(void);

#define huge_10 prefix(huge_10)
void huge_10(void);

#define round_4 prefix(round_4)
void round_4(float *);

#define round_8 prefix(round_8)
void round_8(double *);

#define trunc_4 prefix(trunc_4)
void trunc_4(float *);

#define trunc_8 prefix(trunc_8)
void trunc_8(double *);

#define build_infinity prefix(build_infinity)
void build_infinity(int, void *, int);

#define build_nan prefix(build_nan)
void build_nan(int, int, void *, int);


#define pack_real_4 prefix(pack_real_4)
void pack_real_4(void *, unsigned *, int *, int *);

#define pack_real_8 prefix(pack_real_8)
void pack_real_8(void *, unsigned *, int *, int *);

#define pack_real_10 prefix(pack_real_10)
void pack_real_10(void *, unsigned *, int *, int *);

#define pack_real_16 prefix(pack_real_16)
void pack_real_16(void *, unsigned *, int *, int *);


#define unpack_real_4 prefix(unpack_real_4)
void unpack_real_4(void *, unsigned *, int *, int *);

#define unpack_real_8 prefix(unpack_real_8)
void unpack_real_8(void *, unsigned *, int *, int *);

#define unpack_real_10 prefix(unpack_real_10)
void unpack_real_10(void *, unsigned *, int *, int *);

#define unpack_real_16 prefix(unpack_real_16)
void unpack_real_16(void *, unsigned *, int *, int *);

#define unpack_real prefix(unpack_real)
int unpack_real(void *, int, unsigned *, int *, int *, int *, int *, int *,
		int *);

/* ftoa.c */

#define format_free prefix(format_free)
int format_free(char *, void *, int);

#define format_fixed prefix(format_fixed)
int format_fixed(char, char *, void *, int, int, int *);

#define format_f prefix(format_f)
int format_f(char *, void *, int, int, int);

#define format_en prefix(format_en)
int format_en(char *, void *, int, int);

#define get_f_fmt prefix(get_f_fmt)
int get_f_fmt(int, void *, int);

/* atof.c */

#define convert_real prefix(convert_real)
void convert_real(void *, char *, int);

/* mutex.c */

#define aquire_mutex prefix(aquire_mutex)
void aquire_mutex(void);

#define release_mutex prefix(release_mutex)
void release_mutex(void);

/* coarray.c */

#define init_coarray prefix(init_coarray)
void init_coarray(void);

#define deallocate_coarray prefix(deallocate_coarray)
void deallocate_coarray(g95_array_descriptor *, G95_DINT);


/* mutex.c */

#define cas prefix(cas)
int cas(volatile int *, int, int);

#define aquire_lock prefix(aquire_lock)
void aquire_lock(volatile int *);


/* basic.c */

/* IEEE-754 constants */

#define EXP4_BIAS  127
#define EXP4_NAN   255
#define MAN4_LEN   23
#define MAN4_MSW   0x00800000

#define EXP8_BIAS  1023
#define EXP8_NAN   2047
#define MAN8_LEN   52
#define MAN8_MSW   0x00100000

#define EXP10_BIAS  16383
#define EXP10_NAN   32767
#define MAN10_LEN   64
#define MAN10_MSW   0x00000000

#define EXP16_BIAS  16383
#define EXP16_NAN   32767
#define MAN16_LEN   112
#define MAN16_MSW   0x00010000


/* Values for the IEEE_CLASS parameter.  Must match the version in g95.h */

typedef enum {
    CLASS_OTHER_VALUE=0,
    CLASS_SIGNALING_NAN=1,  CLASS_QUIET_NAN=2,

    CLASS_NEGATIVE_INF=3,   CLASS_NEGATIVE_DENORMAL=4,
    CLASS_NEGATIVE_ZERO=5,  CLASS_NEGATIVE_NORMAL=6,

    CLASS_POSITIVE_INF=7,   CLASS_POSITIVE_DENORMAL=8,
    CLASS_POSITIVE_ZERO=9,  CLASS_POSITIVE_NORMAL=10,
} ieee_class;


typedef struct {
    int exp;
    int sign;
    unsigned m[4];
} unpacked16;


typedef struct {
    unsigned a[4];
} packed16;



const extern packed16 prefix(huge_v16);
const extern packed16 prefix(hugem_v16);



#if defined(FPU_ALPHA)
#define QUAD_POINTER  1
#endif


#if HAVE_REAL_10 == 3
#define unpack_quad prefix(unpack_quad)
#else
#define unpack_quad unpack_real_16
#endif



#define top_bit prefix(top_bit)
int top_bit(unsigned);

#define add_unpacked prefix(add_unpacked)
void add_unpacked(unpacked16 *, unpacked16 *, unpacked16 *);

#define subtract_unpacked prefix(subtract_unpacked)
void subtract_unpacked(unpacked16 *, unpacked16 *, unpacked16 *);

#define multiply_unpacked prefix(multiply_unpacked)
void multiply_unpacked(unpacked16 *, unpacked16 *, unpacked16 *);

#define divide_unpacked prefix(divide_unpacked)
void divide_unpacked(unpacked16 *, unpacked16 *, unpacked16 *);

#define unpacked_sqrt16 prefix(unpacked_sqrt16)
void unpacked_sqrt16(unpacked16 *);

#define set_nan16 prefix(set_nan16)
void set_nan16(unpacked16 *);

#define compare16 prefix(compare16)
int compare16(unpacked16 *, unpacked16 *);

