
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


/* Array support subroutines */

#include <string.h>
#include <stdarg.h>

#include "runtime.h"


/* The section_info[] array holds information on array sections or
 * elements being passed to various subroutines.  The exact format
 * depends on the subroutine. */

G95_AINT section_info[4*G95_MAX_DIMENSIONS];

#define empty_array prefix(empty_array)
g95_array_descriptor empty_array = { NULL, NULL, 1, 0, 0, {{ 0, 1, 0 }} };



/* array_oob1()-- Handle an array out of bounds by terminating the
 * program. */

#define array_oob1 prefix(array_oob1)

void array_oob1(G95_AINT value, G95_DINT dim, G95_AINT lbound) {
char message[100];

    st_sprintf(message, "Array element out of bounds: %L in (%L:*), dim=%d",
	       value, lbound, dim);

    runtime_error(message);
}



/* array_oob2()-- Handle an array out of bounds by terminating the
 * program. */

#define array_oob2 prefix(array_oob2)

void array_oob2(G95_AINT value, G95_DINT dim, G95_AINT lbound,
		G95_AINT ubound) {
char message[100];

    st_sprintf(message, "Array element out of bounds: %L in (%L:%L), dim=%d",
	       value, lbound, ubound, dim);

    runtime_error(message);
}



/* init_assumed_shape()-- This subroutine initializes an assumed shape
 * array from a parameter descriptor.  The multipliers don't change,
 * but the bounds do and the offset must be recalculated. */

#define init_assumed_shape prefix(init_assumed_shape)

void init_assumed_shape(g95_array_descriptor *in, g95_array_descriptor *out,
			void *init) {
G95_AINT extent, index[G95_MAX_DIMENSIONS];
int i, rank, zero_size;
char *p;

    if (in == NULL) {  /* Optional argument that isn't there */
	out->base   = NULL;
	out->offset = NULL;

	for(i=0; i<out->rank; i++)
	    out->info[i].mult = 0;

    } else {
	if (in->base == NULL) {   /* Non-allocated array */
	    out->base   = NULL;
	    out->offset = NULL;
	    return;
	}

	if (in->rank != out->rank)
	    runtime_error("Actual assumed-shape array argument does not "
			  "conform");

	rank = in->rank;
	zero_size = 0;
	out->base = in->base;
	out->offset = in->offset;

	for(i=0; i<rank; i++) {
	    extent = in->info[i].ubound - in->info[i].lbound + 1;
	    if (extent <= 0)
		zero_size = 1;

	    out->info[i].ubound = out->info[i].lbound + extent - 1;
	    out->info[i].mult   = in->info[i].mult;

	    out->offset += out->info[i].mult *
		(in->info[i].lbound - out->info[i].lbound);
	}

	if (init != NULL && !zero_size) {
	    for(i=0; i<rank; i++)
		index[i] = out->info[i].lbound;

	    do {
		p = out->offset;
		for(i=0; i<rank; i++)
		    p += index[i] * out->info[i].mult;

		memcpy(p, init, out->element_size);
	    } while(!bump_element(out, index));
	}
    }
}



/* init_multiplier()-- Given an array descriptor with bounds correctly
 * set, initialize the multipliers and offset to correctly describe
 * the array. */

void init_multipliers(g95_array_descriptor *desc) {
G95_AINT extent;
int i, rank;
char *p;

    desc->info[0].mult = desc->element_size;
    rank = desc->rank;

    for(i=1; i<rank; i++) {
	extent = desc->info[i-1].ubound - desc->info[i-1].lbound + 1;
	if (extent < 0)
	    extent = 0;

	desc->info[i].mult = extent * desc->info[i-1].mult;
    }

    p = desc->base;
    for(i=0; i<rank; i++)
	p -= desc->info[i].lbound * desc->info[i].mult;

    desc->offset = p;
}



/* is_contiguous()-- See if an array is contiguous.  Contiguity is
 * checked by computing the multipliers for a contiguous array and
 * seeing if they are what we have for the input array.  Returns
 * nonzero if contiguous, zero otherwise. */

static int is_contiguous(g95_array_descriptor *desc) {
g95_array_descriptor t;
int i;

    t.rank         = desc->rank;
    t.element_size = desc->element_size; 

    for(i=0; i<t.rank; i++) {
	t.info[i].lbound = desc->info[i].lbound;
	t.info[i].ubound = desc->info[i].ubound;
    }

    init_multipliers(&t);

    for(i=0; i<t.rank; i++)
	if (t.info[i].mult != desc->info[i].mult)
	    return 0;

    return 1;
}


/* contiguous_array()-- Given an array descriptor, return a pointer to
 * the first element of a contiguous array, copying it if necessary.
 * The array_save variable is a void pointer to the temporary array if
 * it was necessary and information on how to copy it back.  A NULL is
 * stored here if it was not necessary to copy-in the array.  The
 * contiguous_array_done must be called with this pointer. */

#define contiguous_array prefix(contiguous_array)

void *contiguous_array(g95_array_descriptor *in, void **array_save,
		       G95_DINT *len) {
G95_AINT mult, size, extent;
char *p, *q, *r;
int i, rank;

    if (in == NULL || in->base == NULL) {
	*array_save = NULL;
	return NULL;
    }

    rank = in->rank;

    /* Calculate the array size.  This is only necessary for
     * a format string that is a full array. */

    if (len != NULL) {
	size = in->element_size;

	for(i=0; i<rank; i++) {
	    extent = in->info[i].ubound - in->info[i].lbound + 1;
	    if (extent < 0)
		extent = 0;

	    size *= extent;
	}

	*len = size;
    }

    if (is_contiguous(in)) {
	*array_save = NULL;
	p = in->offset;

	if (section_info[0]) {
	    for(i=0; i<rank; i++)
		p += in->info[i].mult * section_info[i+1];

	} else
	    for(i=0; i<rank; i++)
		p += in->info[i].mult * in->info[i].lbound;

	return p;
    }

    /* We have to copy the array.  Figure out how much memory is needed */

    if (section_info[0]) 
	memmove(section_info, section_info+1, rank*sizeof(section_info[0]));

    else
	for(i=0; i<rank; i++)
	    section_info[i] = in->info[i].lbound;

    mult = 1;
    size = 0;

    for(i=0; i<rank; i++) {
	extent = in->info[i].ubound - section_info[i] + 1;
	if (extent < 0)
	    extent = 0;

	size += extent * mult;

	extent = in->info[i].ubound - in->info[i].lbound + 1;
	if (extent < 0)
	    extent = 0;

	mult *= extent;
    }

    /* The format of the save area is as follows:
     *    Pointer to the descriptor being saved.
     *    N array integers that give the first index saved, N=array rank.
     *    Possible word of padding
     *    Data area for the array elements 
     */

    p = temp_alloc(sizeof(g95_array_descriptor *) + rank*sizeof(G95_AINT)
		   + size*in->element_size + 8);

    *array_save = p;
    *((g95_array_descriptor **) p) = in;

    memmove(p + sizeof(g95_array_descriptor *), section_info,
	    rank*sizeof(G95_AINT));

    p += sizeof(g95_array_descriptor *) + rank*sizeof(G95_AINT) + 8;
    p = (char *) (((long) p) & -8L);

    q = p;

    for(; size>0; size--) {
	r = in->offset;
	for(i=0; i<rank; i++)
	    r += in->info[i].mult * section_info[i];

	memmove(q, r, in->element_size);
	q += in->element_size;

	i = 0;
	for(;;) {
	    if (++section_info[i] <= in->info[i].ubound)
		break;

	    section_info[i] = in->info[i].lbound;
	    i++;

	    if (i >= in->rank)
		break;
	}
    }

    return p;
}



/* contiguous_array_done()-- Clean things up following a procedure
 * call that passed an array.  This might mean copying the temporary
 * back to the original.  If the copy_out_mask is zero, then copy-out
 * is not performed even if the array was noncontiguous. */

#define contiguous_array_done prefix(contiguous_array_done)

void contiguous_array_done(char *array_save, int copy_out_mask) {
g95_array_descriptor *desc;
G95_AINT *count;
int i, rank;
char *p, *q;

    if (array_save == NULL)
	return;

    if (!copy_out_mask)
	goto done;

    q = array_save;

    desc = *(g95_array_descriptor **) q;
    rank = desc->rank;
    q += sizeof(g95_array_descriptor *);

    count = (G95_AINT *) q;

    q += sizeof(G95_AINT) * desc->rank + 8;
    q = (char *) (((long) q) & -8L);

    for(i=0; i<rank; i++)
	if (desc->info[i].lbound > desc->info[i].ubound)
	    goto done;

    for(;;) {
	p = desc->offset;
	for(i=0; i<rank; i++)
	    p += desc->info[i].mult * count[i];

	memmove(p, q, desc->element_size);
	q += desc->element_size;

	i = 0;
	for(;;) {
	    if (++count[i] <= desc->info[i].ubound)
		break;

	    count[i] = desc->info[i].lbound;
	    i++;

	    if (i >= rank)
		goto done;
	}
    }
  
done:
    temp_free((void **) &array_save);
}



/* temp_array()-- Get memory for a temporary array descriptor and
 * perhaps some storage for the array itself.  Memory for the array
 * storage is obtained in the same block and the base pointer is
 * initialized to point to the data area past the end of the
 * descriptor.  When the descriptor is freed, the array memory goes
 * away as well. */

g95_array_descriptor *temp_array(int rank, int element_size, ...) {
va_list ap;
int i, n;

    va_start(ap, element_size);
    section_info[0] = rank;
    section_info[1] = element_size;
    n = 2;

    for(i=0; i<rank; i++) {
	section_info[n++] = 1;
	section_info[n++] = va_arg(ap, G95_AINT);
    }

    va_end(ap);

    return array_from_section(NULL);
}



/* section_array()-- Given an array, calculate a new descriptor that
 * is the section of the array.  The section_info[] array holds a set
 * of records on each dimension.  The first word of a record is a flag
 * indicating whether the dimension has a fixed value (contracting the
 * dimensionality of the result) or a subscript range.  For fixed
 * values, the next word is the array index.  For ranges, the next
 * three words give the start, end and stride of the range.  From
 * this, the new descriptor is calculated.
 *
 * If a dimension is not a range, the element only contributes to
 * calculating the offset of the new array.
 *
 * For a section, the multiplier of the new descriptor is the product
 * of the old descriptor and the stride.  The bounds are one and the
 * extent of the section, which can be truncated by the range of the
 * original array. */

#define section_array prefix(section_array)

void section_array(g95_array_descriptor *src, g95_array_descriptor *dest,
		   G95_DINT assumed_size) {
G95_AINT start, end, stride, extent, e;
int i, r, dest_rank, n;
char *p;

    p = src->offset;
    n = 0;
    dest_rank = 0;

    for(r=0; r<src->rank; r++) {
	i = section_info[n++];
	start = section_info[n++];

	if (i) {
	    if ((!assumed_size || r != src->rank-1) &&
		(start < src->info[r].lbound || start > src->info[r].ubound))
		array_oob2(start, r+1, src->info[r].lbound,
			   src->info[r].ubound);

	    p = p + src->info[r].mult*start;
	    continue;
	}

	/* Deal with a range */

	end = section_info[n++];
	stride = section_info[n++];

	if (stride == 0)
	    runtime_error("Zero stride in array section");

	/* The original code here clamped the section by the actual extent
	 * of the array.  The problem with this was the last dimension of
	 * an assumed-size array. */

	extent = (end - start + stride) / stride;
	if (extent <= 0)
	    extent = 0;

	else {
	    e = start + stride*(extent-1);
	    if ((!assumed_size || r != src->rank-1) &&
		(start < src->info[r].lbound || start > src->info[r].ubound ||
		 e < src->info[r].lbound || e > src->info[r].ubound))
		runtime_error("Array section out of bounds");
	}

	p = p + src->info[r].mult*start;

	dest->info[dest_rank].mult = src->info[r].mult * stride;
	dest->info[dest_rank].lbound = 1;
	dest->info[dest_rank].ubound = extent;

	p = p - dest->info[dest_rank].mult  /* times one */ ;
	dest_rank++;
    }

    dest->rank = dest_rank;
    dest->offset = p;
    dest->base = src->base;
    dest->element_size = src->element_size;
}



/* compare_section()-- Compare two descriptors to see if they are
 * associated with the same block of memory.  Returns nonzero if so. */

#define compare_section prefix(compare_section)

int compare_section(g95_array_descriptor *a, g95_array_descriptor *b) {
G95_AINT e1, e2;
int i, rank;
char *p, *q;

    if (a->base == NULL || b->base == NULL || a->rank != b->rank)
	return 0;

    rank = a->rank;

    p = a->offset;
    q = b->offset;

    for(i=0; i<rank; i++) {
	p += a->info[i].mult * a->info[i].lbound;
	q += b->info[i].mult * b->info[i].lbound;

	if (a->info[i].lbound > a->info[i].ubound ||
	    b->info[i].lbound > b->info[i].ubound)
	    return 0;
    }

    if (p != q)
	return 0;

    for(i=0; i<rank; i++) {
	e1 = a->info[i].mult;
	if (e1 < 0)
	    e1 = -e1;

	e2 = b->info[i].mult;
	if (e2 < 0)
	    e2 = -e2;

	if (e1 != e2)
	    return 0;

	e1 = a->info[i].ubound - a->info[i].lbound + 1;
	if (e1 < 0)
	    e1 = 0;

	e2 = b->info[i].ubound - b->info[i].lbound + 1;
	if (e2 < 0)
	    e2 = 0;

	if (e1 != e2)
	    return 0;
    }

    return 1;
}



/* bump_element()-- Given an array descriptor and a pointer to an
 * array of integer indexes within the array, bump the indexes to the
 * next element in array element order.  Returns nonzero if the end of
 * the array has been hit, zero otherwise. */

int bump_element(g95_array_descriptor *array, G95_AINT vector[]) {
int i, rank;

    rank = array->rank;
    i = 0;

    for(;;) {
	if (vector[i] != array->info[i].ubound) {
	    vector[i]++;
	    break;
	}

	vector[i] = array->info[i].lbound;
	if (++i >= rank)
	    return 1;
    }

    return 0;
}



/* bump_element_dim()-- Exactly like bump_element(), except, ignores
 * the dim dimension (zero based). */

int bump_element_dim(g95_array_descriptor *array, G95_AINT vector[], int dim) {
int i, rank;

    rank = array->rank;
    i = 0;

    for(;;) {
	if (i != dim) {
	    if (vector[i] != array->info[i].ubound) {
		vector[i]++;
		break;
	    }

	    vector[i] = array->info[i].lbound;
	}

	if (++i >= rank)
	    return 1;
    }

    return 0;
}



#define c_f_pointer prefix(c_f_pointer)

/* c_f_pointer()-- Implement the array version of the c_f_pointer
 * intrinsic.  The idea is we're building an array descriptor fptr. */

void c_f_pointer(char *cptr, g95_array_descriptor *fptr, G95_DINT element_size,
		 g95_array_descriptor *shape) {
int rank, i;

    rank = shape->info[0].ubound - shape->info[0].lbound + 1;

    fptr->rank = rank;
    fptr->element_size = element_size;

    fptr->base = cptr;

    for(i=0; i<rank; i++) {
	fptr->info[i].lbound = 1;
	fptr->info[i].ubound =
	    *((G95_DINT *) (shape->offset +
			    shape->info[0].mult * (shape->info[0].lbound+i)));
    }

    init_multipliers(fptr);
}


