
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/

#include <string.h>
#include <stdlib.h>

#include "safe-ctype.h"
#include "runtime.h"


/* Environment scanner.  Examine the environment for controlling minor
 * aspects of the program's execution.  Our philosophy here that the
 * environment should not prevent the program from running, so an
 * environment variable with a messed-up value will be interpreted in
 * the default way.
 *
 * Most of the environment is checked early in the startup sequence,
 * but other variables are checked during execution of the user's
 * program. */

options_t options;


/* Apple, of course, has to be different from everyone else. */

#if HAVE_CRT_EXTERNS_H
#include <crt_externs.h>
#define environ (*_NSGetEnviron())
#else
extern char ** environ;
#endif


typedef struct variable {
    char *name;
    int value, predef, *var;
    void (*init)(struct variable *);
    void (*show)(struct variable *);
    char *desc;
    int bad;
} variable;


/* Structure for associating names and values.  */

typedef struct {
    char *name;
    G95_DINT value;
} choice;


endian_t my_endian, default_endian;

static char unknown[] = "(Unknown)";

static choice rounding[] = {
    { "NEAREST", FP_ROUND_NEAREST },
    { "UP",      FP_ROUND_UP },
    { "DOWN",    FP_ROUND_DOWN },
    { "ZERO",    FP_ROUND_ZERO },
    { NULL } }, 

#ifdef FPU_387
    precision[] = { { "24", 24 }, { "53", 53 }, { "64", 64 }, { NULL } },
#endif

    signal_choices[] = {
	{ "IGNORE",  SIGDISP_IGNORE },
	{ "ABORT",   SIGDISP_ABORT },
#if HAVE_RESUME
	{ "DUMP",       SIGDISP_DUMP },
	{ "DUMP_QUIT",  SIGDISP_DUMP_QUIT },
#endif
	{ NULL,      0 } },

    endians[] = { { "NATIVE", ENDIAN_NATIVE },
		  { "BIG",    ENDIAN_BIG    },
		  { "LITTLE", ENDIAN_LITTLE } };


static void init_integer(variable *);
static void show_integer(variable *);

static void init_boolean(variable *);
static void show_boolean(variable *);

static void init_string(variable *);
static void show_string(variable *);

static void init_sep(variable *);
static void show_sep(variable *);

static void init_mem(variable *);
static void show_mem(variable *);

static void init_signal(variable *);
static void show_signal(variable *);

static void init_round(variable *);
static void show_round(variable *);

#ifdef FPU_387
static void init_precision(variable *);
static void show_precision(variable *);
#endif

static void init_endian(variable *);
static void show_endian(variable *);


static variable variable_table[] = {
    { "G95_STDIN_UNIT", 5, 0, &options.stdin_unit, init_integer, show_integer,
      "Unit number that will be preconnected to standard input\n"
      "(No preconnection if negative)" },

    { "G95_STDOUT_UNIT", 6, 0, &options.stdout_unit, init_integer,
      show_integer,
      "Unit number that will be preconnected to standard output\n"
      "(No preconnection if negative)" },

    { "G95_STDERR_UNIT", 0, 0, &options.stderr_unit, init_integer,
      show_integer,
      "Unit number that will be preconnected to standard error\n"
      "(No preconnection if negative)" },

    { "G95_USE_STDERR", 1, 0, &options.use_stderr, init_boolean,
      show_boolean,
      "Sends library output to standard error instead of standard output." },

    { "G95_ENDIAN", ENDIAN_NATIVE, 0, (int *) &default_endian, init_endian,
      show_endian,
      "Endian format to use for I/O of unformatted data.  Values are BIG, "
      "LITTLE or\nNATIVE.  Default is NATIVE" },

    { "G95_CR", G95_CR_DEFAULT, 0, &options.cr, init_boolean, show_boolean,
      "Output carriage returns for formatted sequential records.  Default "
      "TRUE on\nnon-Cygwin/Windows, FALSE elsewhere." },

    { "G95_INPUT_CR", 1, 0, &options.input_cr, init_boolean, show_boolean,
      "Treat a carriage return-linefeed as a record marker instead of just a\n"
      "linefeed.  Default TRUE." },

    { "G95_IGNORE_ENDFILE", 0, 0, &options.ignore_endfile, init_boolean,
      show_boolean,
      "Ignore attempts to read past the ENDFILE record in sequential access "
      "mode.\nDefault FALSE." },

    { "G95_TMPDIR", 0, 0, NULL, init_string, show_string,
      "Directory for scratch files.  Overrides the TMP environment variable\n"
      "If TMP is not set " DEFAULT_TEMPDIR " is used." },

    { "G95_UNBUFFERED_ALL", 0, 0, &options.all_unbuffered, init_boolean,
      show_boolean,
      "If TRUE, all output is unbuffered.  This will slow down large writes "
      "but can\nbe useful for forcing data to be displayed immediately."},

    { "G95_SHOW_LOCUS", 1, 0, &options.locus, init_boolean, show_boolean,
      "If TRUE, print filename and line number where runtime errors happen."},

    { "G95_STOP_CODE", 1, 0, &options.stop_code, init_boolean, show_boolean,
      "If TRUE, stop codes are propagated to system exit codes.  Default "
      "TRUE."},

/* G95_UNIT_xx (where xx is a unit number) gives the names of files
 * preconnected to those units. */

/* G95_UNBUFFERED_xx (where xx is a unit number) gives a boolean that is used
 * to turn off buffering for that unit. */

    { "G95_OPTIONAL_PLUS", 0, 0, &options.optional_plus, init_boolean,
      show_boolean,
      "Print optional plus signs in numbers where permitted.  Default FALSE."},

    { "G95_DEFAULT_RECL", DEFAULT_RECL, 0, &options.default_recl,
      init_integer, show_integer,
      "Default maximum record length for sequential files.  Most useful for\n"
      "adjusting line length of preconnected units.  Default "
      stringize(DEFAULT_RECL) },

    { "G95_LIST_SEPARATOR", 0, 0, NULL, init_sep, show_sep, 
      "Separator to use when writing list output.  May contain any number of "
      "spaces\nand at most one comma.  Default is a single space."
    },

    { "G95_LIST_EXP", 6, 0, &options.list_exp, init_integer, show_integer,
      "Last power of ten which does not use exponential format for list "
      "output.\nDefault 6." },

    { "G95_COMMA", 0, 0, &options.decimal_comma, init_boolean, show_boolean,
      "Use a comma character as the default decimal point for I/O.  Default "
      "FALSE." },

    { "G95_EXPAND_UNPRINTABLE", 0, 0, &options.expand_unprintable,
      init_boolean, show_boolean,
      "For formatted output, print otherwise unprintable characters with "
      "\\-sequences\nDefault FALSE" },

    { "G95_QUIET", 0, 0, &options.quiet, init_boolean, show_boolean,
      "Suppress bell characters (\\a) in formatted output.  Default FALSE." },

    { "G95_SYSTEM_CLOCK", 10000, 0, &options.system_clock_ticks, init_integer,
      show_integer,
      "Number of ticks per second reported by the SYSTEM_CLOCK() intrinsic.\n"
      "Zero disables the clock." },

    { "G95_SEED_RNG", 0, 0, &options.seed_rng, init_boolean, show_boolean,
      "If TRUE, seeds the random number generator with a new seed when the "
      "program\nis run.  Default FALSE." },

    { "G95_MINUS_ZERO", 0, 0, &options.minus_zero, init_boolean, show_boolean,
      "If TRUE, prints zero values without a minus sign in formatted "
      "(non-list) output,\neven if the internal value is negative or minus "
      "zero.  This is the traditional\nbut nonstandard way of printing zeros."
      "  Default FALSE."},

    { "G95_ABORT", 0, 0, &options.abort, init_boolean, show_boolean,
      "If TRUE, dumps core on abnormal program end.  Useful for finding the "
      "locus\nof the problem.  Default FALSE." },

    /* Memory related controls */

    { "G95_MEM_INIT", 0, 0, NULL, init_mem, show_mem,
      "How to initialize ALLOCATEd memory.  Default value is NONE for no "
      "initialization\n(faster), NAN for a Not-A-Number with the mantissa "
      "0x00f95, or a custom\nhexadecimal value" },

    { "G95_MEM_SEGMENTS", 25, 0, &options.mem_segments, init_integer,
      show_integer,
      "Maximum number of still-allocated memory segments to display when "
      "program ends.\n0 means show none, less than 0 means show all.  "
      "Default 25" },

    { "G95_MEM_MAXALLOC", 0, 0, &options.mem_max_alloc, init_boolean,
      show_boolean,
      "If TRUE, shows the maximum number of bytes allocated in user memory "
      "during\nthe program run." },

/* Hooks for optimizing malloc() */

#if HAVE_DL_MALLOC
    { "G95_MEM_MXFAST", 64, 0, &options.mem_mxfast, init_integer, show_integer,
      "Maximum request size for handing requests in from fastbins.  Fastbins "
      "are\nquicker but fragment more easily.  Default 64 bytes" },

    { "G95_MEM_TRIM_THRESHOLD", 256*1024, 0, &options.mem_trim_threshold,
      init_integer, show_integer,
      "Amount of top-most memory to keep around until it is returned to the "
      "system.\n-1 prevents returning memory to the system.  Useful in "
      "long-lived programs." },

    { "G95_MEM_TOP_PAD", 0, 0, &options.mem_top_pad, init_integer,
      show_integer,
      "Extra space to allocate when getting memory from the OS.  Can speed up "
      "future\nrequests." },
#endif

  /* Signal handling (Unix). */

#if !HAVE_WINDOWS
    { "G95_SIGHUP", SIGDISP_ABORT, 0, (int *) &options.sighup, init_signal,
      show_signal,
      "Whether the program will IGNORE, ABORT, DUMP or DUMP-QUIT on SIGHUP." },

    { "G95_SIGINT", SIGDISP_ABORT, 0, (int *) &options.sigint, init_signal,
      show_signal,
      "Whether the program will IGNORE, ABORT, DUMP or DUMP-QUIT on SIGINT." },

    { "G95_SIGQUIT", SIGDISP_DUMP, 0, (int *) &options.sigquit, init_signal,
      show_signal,
      "Whether the program will IGNORE, ABORT, DUMP or DUMP-QUIT on SIGQUIT."},
#endif

#if HAVE_RESUME
    { "G95_CHECKPOINT", 0, 0, &globals.alarm_value, init_integer, show_integer,
      "Interval in seconds to dump a checkpoint file, 0 = No dumps.  "
      "Default 0"},

    { "G95_CHECKPOINT_MSG", 1, 0, &globals.checkpoint_msg, init_boolean,
      show_boolean,
      "If TRUE, print a message to stderr when process is checkpointed.  "
      "Default TRUE." },
#endif

    /* Floating point control */

    { "G95_FPU_ROUND", 0, 0, &options.fpu_round, init_round, show_round,
      "Set floating point rounding.  Values can be NEAREST, UP, DOWN, ZERO." },

#ifdef FPU_387
    { "G95_FPU_PRECISION", -1, 0, &options.fpu_precision, init_precision,
      show_precision,
      "Precision of intermediate results.  Value can be 24, 53 and 64.  "
      "Default 64"},

    { "G95_FPU_DENORMAL", 0, 0, &options.fpu_denormal, init_boolean,
      show_boolean,
      "Raise a floating point exception when denormal numbers are "
      "encountered."},

    { "G95_FPU_NO_DENORMALS", 0, 0, &options.fpu_no_denormals, init_boolean,
      show_boolean,
      "Round denormalized numbers to zero (MMX only)" },
#endif

    { "G95_FPU_INVALID", 0, 0, &options.fpu_invalid, init_boolean,
      show_boolean,
      "Raise a floating point exception on an invalid operation." },

    { "G95_FPU_ZERODIV", 0, 0, &options.fpu_zerodiv, init_boolean,
      show_boolean,
      "Raise a floating point exception when dividing by zero." },

    { "G95_FPU_OVERFLOW", 0, 0, &options.fpu_overflow, init_boolean,
      show_boolean,
      "Raise a floating point exception on overflow." },

    { "G95_FPU_UNDERFLOW", 0, 0, &options.fpu_underflow, init_boolean,
      show_boolean,
      "Raise a floating point exception on underflow." },

    { "G95_FPU_INEXACT", 0, 0, &options.fpu_inexact, init_boolean,
      show_boolean,
      "Raise a floating point exception on precision loss." },

    { "G95_FPU_EXCEPTIONS", 0, 0, &options.view_exceptions, init_boolean,
      show_boolean,
      "Whether masked floating point exceptions should be shown after the "
      "program ends." },

    { NULL }
};



/* print_spaces()-- Print a particular number of spaces */

static void print_spaces(int n) {
char buffer[80];
int i;

    if (n <= 0)
	return;

    for(i=0; i<n; i++) 
	buffer[i] = ' ';

    buffer[i] = '\0';

    st_printf(buffer);
}



/* var_source()-- Return a string that describes where the value of a
 * variable comes from */

static char *var_source(variable *v) {

    if (getenv(v->name) == NULL)
	return "Default";

    if (v->bad)
	return "Bad    ";

    return "Set    ";
}



/* init_integer()-- Initialize an integer environment variable */

static void init_integer(variable *v) {
char *p, *q;

    p = getenv(v->name);
    if (p == NULL)
	goto set_default;

    q = p;
    if (*q == '+' || *q == '-')
	q++;

    for(; *q; q++)
	if (!isdigit(*q)) {
	    v->bad = 1;
	    goto set_default;
	}

    *v->var = atoi(p);
    return;

set_default:
    if (!v->predef)
	*v->var = v->value;
}



/* show_integer()-- Show an integer environment variable */

static void show_integer(variable *v) {

    st_printf("%s  %d\n", var_source(v), *v->var);
}



/* init_boolean()-- Initialize a boolean environment variable.  We
 * only look at the first letter of the variable. */

static void init_boolean(variable *v) {
char *p;

    p = getenv(v->name);

    if (p != NULL) {
	if (*p == '1' || *p == 'Y' || *p == 'y' || *p == 't' || *p == 'T') {
	    *v->var = 1;
	    return;
	}

	if (*p == '0' || *p == 'N' || *p == 'n' || *p == 'f' || *p == 'F') {
	    *v->var = 0;
	    return;
	}

	v->bad = 1;
    }

    if (!v->predef)
	*v->var = v->value;
}



/* show_boolean()-- Show a boolean environment variable */

static void show_boolean(variable *v) {

    st_printf("%s  %s\n", var_source(v), *v->var ? "Yes" : "No");
}



/* init_mem()-- Initialize environment variables that have to do with
 * how memory from an ALLOCATE statement is filled.  A single flag
 * enables filling and a second variable gives the value that is used
 * to initialize the memory. */

static void init_mem(variable *v) {
int offset, n;
char *p;

    p = getenv(v->name); 

    options.allocate_init_flag = 0;   /* The default */

    if (p == NULL || strcasecmp(p, "NONE") == 0)
	return;

    /* IEEE-754 Signalling Not-a-Number that will work for single and
     * double precision.  Look for the 'f95' mantissa in debug dumps. */

    if (strcasecmp(p, "NaN") == 0) {
	options.allocate_init_flag = 1;
	options.allocate_init_value = 0xff800f95;
	return;
    }

    /* Interpret the string as a hexadecimal constant */

    if (p[0] == '0' && (p[1] == 'x' || p[1] == 'X'))
	p += 2;

    n = 0;
    while(*p) {
	if (!isxdigit(*p)) {
	    v->bad = 1;
	    return;
	}

	offset = '0';
	if (islower(*p))
	    offset = 'a' - 10;

	else if (isupper(*p))
	    offset = 'A' - 10;

	n = (n << 4) | (*p++ - offset);
    }

    options.allocate_init_flag = 1;
    options.allocate_init_value = n;
}



static void show_mem(variable *v) {
char *p;

    p = getenv(v->name);

    st_printf("%s  ", var_source(v));

    if (options.allocate_init_flag)
	st_printf("0x%x", options.allocate_init_value);

    st_printf("\n");
}



static void init_sep(variable *v) {
int seen_comma;
char *p;

    p = getenv(v->name);
    if (p == NULL)
	goto set_default;

    v->bad = 1;
    options.separator = p;
    options.separator_len = strlen(p);

    /* Make sure the separator is valid */

    if (options.separator_len == 0)
	goto set_default;

    seen_comma = 0;

    while(*p) {
	if (*p == ',') {
	    if (seen_comma) 
		goto set_default;

	    seen_comma = 1;
	    p++;
	    continue;
	}

	if (*p++ != ' ')
	    goto set_default;
    }

    v->bad = 0;
    return;

set_default:
    options.separator = " ";
    options.separator_len = 1;
}



static void show_sep(variable *v) {

    st_printf("%s  \"%s\"\n", var_source(v), options.separator);
}



static void init_string(variable *v) { }


static void show_string(variable *v) {
char *p;

    p = getenv(v->name);
    if (p == NULL)
	p = "";

    st_printf("%s  \"%s\"\n", var_source(v), p);
}


/* check_unbuffered()-- Given a unit number n, determine if an override
 * for the stream exists.  Returns zero if the stream should be
 * unbuffered, one for buffered, or two for unset. */

int check_unbuffered(G95_MINT n, int def) {
char name[40];
variable v;
int rv;

    if (options.all_unbuffered)
	return 1;

    strcpy(name, "G95_UNBUFFERED_");
    strcat(name, mint_to_a(n));

    v.name = name;
    v.value = def;
    v.var = &rv;
    v.predef = 0;

    init_boolean(&v);

    return rv;
}


/* show_choice()-- Show a choice */

static void show_choice(variable *v, choice *c, char *unknown) {

    st_printf("%s  ", var_source(v));

    for(;c->name; c++)
	if (c->value == *v->var)
	    break;

    st_printf("%s\n", c->name ? c->name : unknown);
}


/* init_choice()-- Initialize a choice */

static void init_choice(variable *v, choice *c) {
char *p;

    p = getenv(v->name);
    if (p == NULL)
	goto set_default;

    for(;c->name; c++)
	if (strcasecmp(c->name, p) == 0)
	    break;

    if (c->name == NULL) {
	v->bad = 1;
	goto set_default;
    }

    *v->var = c->value;
    return;

set_default:
    *v->var = v->value;
}


/* init_variables()-- Initialize most runtime variables from
 * environment variables. */

void init_variables(void) {
variable *v;

    for(v=variable_table; v->name; v++) 
	v->init(v);
}



static void init_round(variable *v) { init_choice(v, rounding); }
static void show_round(variable *v) { show_choice(v, rounding, unknown); }

#ifdef FPU_387
static void init_precision(variable *v) { init_choice(v, precision); }
static void show_precision(variable *v) { show_choice(v, precision, "64"); }
#endif

static void init_signal(variable *v) { init_choice(v, signal_choices); }
static void show_signal(variable *v) { show_choice(v, signal_choices,unknown);}



/* init_endian()-- Initialize endianness */

static void init_endian(variable *v) {
union {
    int int_0;
    char char_0[sizeof(int)];
} u;

    init_choice(v, endians);

    u.int_0 = 0x12 << (8*(sizeof(int) - 1));

    my_endian = (u.char_0[0] == 0x12) ? ENDIAN_BIG : ENDIAN_LITTLE;
}



/* show_endian()-- Show endian setting */

static void show_endian(variable *v) {

    show_choice(v, endians, unknown);
}



/* pattern_scan()-- Given an environment string, check that the name
 * has the same name as the pattern followed by an integer.  On a
 * match, a pointer to the value is returned and the integer pointed
 * to by n is updated.  Returns NULL on no match. */

static char *pattern_scan(char *env, char *pattern, G95_DINT *n) {
char *p;
int len;

    len = strlen(pattern);
    if (strncasecmp(env, pattern, len) != 0)
	return NULL;

    p = env + len;

    if (!isdigit(*p))
	return NULL;

    while(isdigit(*p))
	p++;

    if (*p != '=')
	return NULL;

    *p = '\0';
    *n = atoi(env + len);
    *p++ = '=';

    return p;
}



/* show_variable()-- Show all runtime setting in gory detail */

void show_variables(void) {
char *p, **e;
variable *v;
int n, i;

    st_printf("G95 fortran runtime library version " stringize(VERSION)
	      "\n\n");

    st_printf("Environment variables:\n");
    st_printf("----------------------\n");

    for(v=variable_table; v->name; v++) {
	i = st_printf("%s", v->name);
	print_spaces(25 - i);

	if (v->show == show_integer)
	    st_printf("Integer ");
	else if (v->show == show_boolean)
	    st_printf("Boolean ");
	else
	    st_printf("String  ");

	v->show(v);
	st_printf("%s\n\n", v->desc);
    }

    st_printf("\nDefault unit names (G95_UNIT_x):\n");

    for(e=environ; *e; e++) {
	p = pattern_scan(*e, "G95_UNIT_", &n);
	if (p != NULL)
	    st_printf("G95_UNIT_%d         %s\n", n, p);
    }

    st_printf("\nUnit buffering overrides (G95_UNBUFFERED_x):\n");
    for(e=environ; *e; e++) {
	p = pattern_scan(*e, "G95_UNBUFFERED_", &n);
	if (p != NULL)
	    st_printf("G95_UNBUFFERED_%d = %s\n", n, p);
    }

    st_printf("\nUnit endian overrides (G95_UNIT_ENDIAN_x):\n");

    for(e=environ; *e; e++) {
	p = pattern_scan(*e, "G95_UNIT_ENDIAN_", &n);
	if (p != NULL)
	    st_printf("G95_UNIT_ENDIAN_%d = %s\n", n, p);
    }

    /* System error codes */

    st_printf("\nRuntime error codes:");
    st_printf("\n--------------------\n");

    for(n=ERROR_FIRST+1; n<ERROR_LAST; n++)
	if (n == ERROR_OS) {
	    st_printf("      Operating system errno codes (1 - 199)\n");
	    n = 199;

	} else if (n > 99) 
	    st_printf("%d   %s\n", n, translate_error(n));

	else if (n < 0 || n > 9)
	    st_printf(" %d   %s\n", n, translate_error(n));

	else
	    st_printf("  %d   %s\n", n, translate_error(n));

    st_printf("\nCommand line arguments:\n");
    st_printf("  --g95                Print this list\n");

    sys_exit(0, 0);
}
