
/* Copyright (C) 2003-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


/* Write formatted floating point numbers */

/* Real numbers are actually converted in another file using the
 * algorithm described by Robert G. Burger and R. Kent Dybvig in
 * "Proceedings of the ACM SIGPLAN '96 Conference on Programming
 * Language Design and Implementation", pages 108-116, May 1996.  This
 * module formats the output. */


#include "runtime.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

/* Max number of digits in an x87 real*10 in F format */

#define SMALL_SIZE 5000


/* write_real()-- Write a real to list directed output. */

void write_real(char *source, int length) {
char buffer[100], *p, *q, *r;
int e, m, len;

    if (get_float_flavor(source, length, buffer) == FF_REGULAR) {
	q = buffer;
	if (get_sign(source, length))
	    *q++ = '-';

	e = format_free(q, source, length);
	m = strlen(q);

	if (-options.list_exp <= e && e < 0) {  /* Insert leading zeros */
	    memmove(q+1-e, q, m+1);
	    memset(q, '0', 1-e);
	    q[1] = DECIMAL_CHAR();

	} else if (0 <= e && e <= options.list_exp) {
	    if (e < m) {
		memmove(q+e+2, q+e+1, m-e);
		q[e+1] = DECIMAL_CHAR();

	    } else {
		q += m;
		e = e - m;
		while(e >= 0) {
		    *q++ = '0';
		    e--;
		}

		*q++ = DECIMAL_CHAR();
		*q++ = '\0';
	    }

	} else {  /* Exponential format */
	    memmove(q+2, q+1, m);
	    q[1] = DECIMAL_CHAR();
	    q[m+1] = 'E';
      
	    r = int_to_a(e);

	    if (e < 0)
		strcpy(q+m+2, r);
	    else {
		q[m+2] = '+';
		strcpy(q+m+3, r);
	    }
	} 
    }

    len = strlen(buffer);
    p = write_block(len);

    if (p != NULL)
	memmove(p, buffer, len);
}



/* minus_zero()-- Returns nonzero if the number being displayed is a
 * minus zero. */

static int minus_zero(char *p) {

    while(*p == ' ')
	p++;

    if (*p++ != '-')
	return 0;

    for(;;) {
	if (*p == DECIMAL_CHAR()) {
	    p++;
	    continue;
	}

	if ('1' <= *p && *p <= '9')
	    return 0;

	if (*p != '0')
	    break;

	p++;
    }

    return 1;
}



/* format_exponent()-- Format an exponent properly.  Returns nonzero
 * if the exponent overflows its field. */

static int format_exponent(char *p, char exp_char, int exp, int e) {
char *q;
int m;

    switch(e) {
    case 1:  if (exp > 9 || exp < -9) return 1; break;
    case 2:  if (exp > 99 || exp < -99) return 1; break;
    case 3:  if (exp > 999 || exp < -999) return 1; break;
    case 4:  if (exp > 9999 || exp < -9999) return 1; break;
    case 5:  if (exp > 99999 || exp < -99999) return 1; break;
    }

    p = strchr(p, '\0');

    if (e == -1 && (exp > 999 || exp < -999))     /* Squished exponent */
	e = 4;

    else if (e == -1 && (exp > 99 || exp < -99))
	e = 3;

    else {
	*p++ = exp_char;
	if (e == -1)
	    e = 2;
    }

    if (exp >= 0)
	*p++ = '+';

    else {
	*p++ = '-';
	exp = -exp;
    }

    q = int_to_a(exp);
    m = strlen(q);

    while(m < e) {
	*p++ = '0';
	e--;
    }

    strcpy(p, q);
    return 0;
}



/* write_de()-- Write the E and D formats */

static void write_de(fnode *f, char *source, int len, char exp_char) {
char small_buf[SMALL_SIZE], *buffer, *p, *q;
int e, m, n, spaces;
float_flavor ff;

    ff = get_float_flavor(source, len, small_buf);
    if (ff != FF_REGULAR) {
	buffer = small_buf;
	goto done;
    }

    buffer = (f->u.real.w < SMALL_SIZE - 5)
	? small_buf
	: get_mem(f->u.real.w + 10);

    p = buffer;
    switch(calculate_sign(get_sign(source, len))) {
    case SIGN_NONE:
	break;

    case SIGN_PLUS:
	buffer[0] = '+';
	p++;
	break;

    case SIGN_MINUS:
	buffer[0] = '-';
	p++;
	break;
    }

    if (ioparm->scale_factor <= -f->u.real.d ||
	ioparm->scale_factor > f->u.real.d+1)
	goto star;

    format_fixed('E', p, source, len, f->u.real.d, &e);
    m = strlen(p);

    if (m == f->u.real.d) {
	memmove(p+2, p, m+1);
	p[0] = '0';
	p[1] = DECIMAL_CHAR();

    } else {
	n = m - f->u.real.d + ioparm->scale_factor - 1;
	if (n > m)
	    goto star;

	memmove(p+n+1, p+n, f->u.real.d+1);
	p[n] = DECIMAL_CHAR();
    }

/* Now do the exponent */

    e = e + 1 - ioparm->scale_factor;

    if (format_exponent(p, exp_char, e, f->u.real.e))
	goto star;

done:  
    m = strlen(buffer);

    if (options.minus_zero && ff == FF_REGULAR && minus_zero(buffer))
	memmove(buffer, buffer+1, 1+m--);

    if (f->u.real.w == 0) {
	q = write_block(m);
	if (q != NULL)
	    memmove(q, buffer, m);

    } else {
	q = write_block(f->u.real.w);

	if (q != NULL) {
	    if (m-1 > f->u.real.w)
		star_fill(q, f->u.real.w);

	    else {
		if (m-1 == f->u.real.w) {   /* Squeeze out the optional '0'. */
		    p = strchr(buffer, '0');
		    memmove(p, p+1, m+p-buffer);
		    m--;
		}

		spaces = f->u.real.w - m;
		memset(q, ' ', spaces);
		memmove(q+spaces, buffer, m);
	    }
	}
    }

    goto all_done;

star:
    q = write_block(f->u.real.w);
    if (q != NULL)
	star_fill(q, f->u.real.w);

all_done:
    if (buffer != small_buf)
	free_mem(buffer);
}



/* write_d()-- Write the D format */

void write_d(fnode *f, char *p, int len) {

    write_de(f, p, len, 'D');
}



/* write_e()-- Write the E format */

void write_e(fnode *f, char *p, int len) {

    write_de(f, p, len, 'E');
}


/* write_f()-- Write the F format */

void write_f(fnode *f, char *source, int len) {
char small_buf[SMALL_SIZE], *buffer, *p, *q;
int m, n, spaces;
float_flavor ff;

    ff = get_float_flavor(source, len, small_buf);

    if (ff != FF_REGULAR) {
	if (ff == FF_PLUS_INFINITY && f->u.real.w == 3)
	    memmove(small_buf, small_buf+1, 4);

	buffer = p = small_buf;
	goto print;
    }

    buffer = (f->u.real.w < SMALL_SIZE - 5)
	? small_buf
	: get_mem(f->u.real.w + 10);

    switch(calculate_sign(get_sign(source, len))) {
    case SIGN_NONE:
	p = buffer;
	break;

    case SIGN_PLUS:
	buffer[0] = '+';
	p = buffer + 1;
	break;

    case SIGN_MINUS:
	buffer[0] = '-';
	p = buffer + 1;
	break;

    default:
	p = NULL;
	break;
    }

    n = (f->u.real.w == 0)
	? SMALL_SIZE
	: f->u.real.w;

    if (format_f(p, source, len, f->u.real.d, n)) {
	q = write_block(f->u.real.w);
	if (q != NULL)
	    star_fill(q, f->u.real.w);

	goto done;
    }

    m = strlen(p);

    n = f->u.real.d;
    memmove(p+m-n+1, p+m-n, n+1);
    p[m-n] = DECIMAL_CHAR();

print:
    m = strlen(buffer);

    if (options.minus_zero && ff == FF_REGULAR && minus_zero(buffer)) {
	memmove(buffer, buffer+1, 1+m--);
	p = buffer;
    }

    if (f->u.real.w == 0) {
	if (p[0] == '0' && f->u.real.d != 0) {
	    memmove(p, p+1, m);
	    m--;
	}

	q = write_block(m);
	if (q != NULL)
	    memmove(q, buffer, m);

    } else {
	q = write_block(f->u.real.w);
	if (q != NULL) {
	    if (m == f->u.real.w+2 && buffer[0] == '-' && buffer[1] == '0') {

		p = buffer + 3;
		while(*p) {
		    if ((*p) != '0') {
			star_fill(q, f->u.real.w);
			goto done;
		    }

		    p++;
		}

		memmove(buffer, buffer+2, m);  /* Squish minus sign and zero */
		m -= 2;

	    } else if (m == f->u.real.w+1 && p[0] == '0') {
		memmove(p, p+1, m);    /* Squish a leading zero */
		m--;

	    } else if (m > f->u.real.w) {
		star_fill(q, f->u.real.w);
		goto done;
	    }

	    spaces = f->u.real.w - m;
	    memset(q, ' ', spaces);
	    memmove(q+spaces, buffer, m);
	}
    }

done:
    if (buffer != small_buf)
	free_mem(buffer);
}



/* write_es()-- Write the ES format */

void write_es(fnode *f, char *source, int len) {
char small_buf[SMALL_SIZE], *buffer, *p, *q;
int m, i, spaces;

    if (get_float_flavor(source, len, small_buf) != FF_REGULAR) {
	buffer = small_buf;
	goto done;
    }

    buffer = (f->u.real.w < SMALL_SIZE - 5)
	? small_buf
	: get_mem(f->u.real.w + 10);

    switch(calculate_sign(get_sign(source, len))) {
    case SIGN_NONE:
	p = buffer;
	break;

    case SIGN_PLUS:
	buffer[0] = '+';
	p = buffer + 1;
	break;

    case SIGN_MINUS:
	buffer[0] = '-';
	p = buffer + 1;
	break;

    default:
	p = NULL;
    }

    format_fixed('S', p, source, len, f->u.real.d+1, &i);

    m = strlen(p);
    memmove(p+2, p+1, m);
    p[1] = DECIMAL_CHAR();

    if (format_exponent(p, 'E', i, f->u.real.e)) {
	q = write_block(f->u.real.w);
	if (q != NULL)
	    star_fill(q, f->u.real.w);
	goto all_done;
    }

done:  
    m = strlen(buffer);
    if (options.minus_zero && minus_zero(buffer))
	memmove(buffer, buffer+1, 1+m--);

    if (f->u.real.w == 0) {
	q = write_block(m);
	if (q != NULL)
	    memmove(q, buffer, m);

    } else {
	q = write_block(f->u.real.w);

	if (q != NULL) {
	    if (m > f->u.real.w)
		star_fill(q, f->u.real.w);
	    else {
		spaces = f->u.real.w - m;
		memset(q, ' ', spaces);
		memmove(q+spaces, buffer, m);
	    }
	}
    }

all_done:
    if (buffer != small_buf)
	free_mem(buffer);
}


/* write_en()-- Write the EN format */

void write_en(fnode *f, char *source, int len) {
char small_buf[SMALL_SIZE], *buffer, *p, *q;
int e, m, n, spaces;

    if (get_float_flavor(source, len, small_buf) != FF_REGULAR) {
	buffer = small_buf;
	goto done;
    }

    buffer = (f->u.real.w < SMALL_SIZE - 5)
	? small_buf
	: get_mem(f->u.real.w + 10);

    switch(calculate_sign(get_sign(source, len))) {
    case SIGN_NONE:
	p = buffer;
	break;

    case SIGN_PLUS:
	buffer[0] = '+';
	p = buffer + 1;
	break;

    case SIGN_MINUS:
	buffer[0] = '-';
	p = buffer + 1;
	break;

    default:
	p = NULL;
    }

    /* The exponent of the number determines how many digits we really need */

    e = format_en(p, source, len, f->u.real.d);
    m = strlen(p);

    n = m - f->u.real.d;
    e = e - n + 1;

    memmove(p+n+1, p+n, f->u.real.d+1);
    p[n] = DECIMAL_CHAR();

    if (format_exponent(p, 'E', e, f->u.real.e)) {
	q = write_block(f->u.real.w);
	if (q != NULL)
	    star_fill(q, f->u.real.w);

	goto all_done;
    }

done:  
    m = strlen(buffer);

    if (options.minus_zero && minus_zero(buffer))
	memmove(buffer, buffer+1, 1+m--);

    if (f->u.real.w == 0) {
	q = write_block(m);
	if (q != NULL)
	    memmove(q, buffer, m);

    } else {
	q = write_block(f->u.real.w);

	if (q != NULL) {
	    if (m > f->u.real.w)
		star_fill(q, f->u.real.w);
	    else {
		spaces = f->u.real.w - m;
		memset(q, ' ', spaces);
		memmove(q+spaces, buffer, m);
	    }
	}
    }

all_done:
    if (buffer != small_buf)
	free_mem(buffer);
}


/* write_gi()-- Write an integer in G format */

void write_gi(fnode *f, char *source, int length) {
fnode t;

    t.u.integer.w = f->u.integer.w;
    t.u.integer.m = -1;

    write_i(&t, source, length);
}



/* write_gr()-- Write a real in G format.  Basically, the precision
 * field gives the number of digits we want to use to print the number
 * with, with the decimal point stuck at a place that depends on the
 * magnitude of the number.  If this won't work, we fall back to E
 * formatting. */

void write_gr(fnode *f, char *source, int length) {
int f_d, n, scale;
fnode new;
char *p;

    f_d = get_f_fmt(f->u.real.d, source, length);
    if (f_d < 0) {
	write_e(f, source, length);
	return;
    }

    /* Build a new format node based on what we have. */

    n = (f->u.real.e == -1)
	? 4
	: f->u.real.e + 2;

    new.u.real.w = f->u.real.w - n;
    new.u.real.d = f_d;
    new.u.real.e = f->u.real.e;

    if (new.u.real.w < 1) {
	format_error(f, "G descriptor not wide enough");
	return;
    }

    /* "Ignore" the scale factor */

    scale = ioparm->scale_factor;
    ioparm->scale_factor = 0;

    write_f(&new, source, length);
    ioparm->scale_factor = scale;

    p = write_block(n);
    if (p != NULL)
	memset(p, ' ', n);
}

