
/* Copyright (C) 2002-2008 Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


/* transfer.c -- Top level handling of data transfer statements. */

#include <string.h>
#include "runtime.h"


/* Calling conventions:  Data transfer statements are unlike other
 * library calls in that they extend over several calls. 
 *
 * The first call is always a call to st_read() or st_write().  These
 * subroutines return no status unless a namelist read or write is
 * being done, in which case there is the usual status.  No further
 * calls are necessary in this case.
 *
 * For other sorts of data transfer, there are zero or more data
 * transfer statement that depend on the format of the data transfer
 * statement.
 *
 *    transfer_integer
 *    transfer_logical
 *    transfer_character
 *    transfer_real
 *    transfer_complex
 *    transfer_derived
 *
 *  These subroutines do not return status.
 *
 *  The last call is a call to st_[read|write]_done().  While
 *  something can easily go wrong with the initial st_read() or
 *  st_write(), an error inhibits any data from actually being
 *  transferred.
 */

iounit_t *current_unit;
static G95_DINT *iolength_result, size_count;
static G95_AINT internal_array[G95_MAX_DIMENSIONS];

char scratch[SCRATCH_SIZE];
static char *line_buffer = NULL;

static st_option advance_opt[] = {
    { "yes",  ADVANCE_YES },
    { "no",   ADVANCE_NO  },
    { NULL } };

static st_option decimal_opt[] = {
    { "point",  DECIMAL_POINT },
    { "comma",  DECIMAL_COMMA },
    { NULL } };

static derived_info *current_info;

typedef enum {
    FORMATTED_SEQUENTIAL, UNFORMATTED_SEQUENTIAL,
    FORMATTED_DIRECT, UNFORMATTED_DIRECT,
    FORMATTED_STREAM, UNFORMATTED_STREAM
} file_mode;



/* reverse_endian()-- Reverse a string of bytes. */

static void reverse_endian(char *p, int size) {
int i, j;
char c;

    i = 0;
    j = size-1;

    while(i<j) {
	c = p[i];
	p[i] = p[j];
	p[j] = c;

	i++;
	j--;
    }
}



/* reverse_flag()-- Decide if binary data should be reversed or not.
 * Returns nonzero if so.  */

static int reverse_flag(void) {

    return (ioparm->endian == 0)
	? current_unit->reverse
	: ioparm->endian != my_endian;
}


/* get_length()-- Read a record length from memory. */

G95_INT4 get_length(unsigned char *p) {
G95_INT4 m;

    memcpy((char *) &m, p, 4);

    if (reverse_flag())
	reverse_endian((char *) &m, 4);

    return m;
}



/* set_length()-- Store a length into memory. */

static void set_length(char *p, int i) {

    memcpy(p, (char *) &i, 4);

    if (reverse_flag())
	reverse_endian(p, 4);
}



/* recursive_io()-- Return nonzero if the current unit is involved in
 * another nested I/O statement. */

static int recursive_io(void) {
st_parameter *p;

    if (is_internal_unit())
	return 0;

    for(p=ioparm; p; p=p->prev)
	if (p->unit_save == current_unit)
	    return 1;

    return 0;
}



/* next_direct()-- Read the next direct record */

static void next_direct(void) {
unsigned len;
char *p;

    if (sseek(current_unit->s,
	      (current_unit->last_record - 1)*current_unit->recl) == FAILURE) {
	generate_error(ERROR_OS, NULL);
	return;
    }

    len = current_unit->recl; 
    p = salloc_r(current_unit->s, &len);

    if (p == NULL || len != current_unit->recl)
	generate_error(ERROR_CORRUPT_DIRECT, NULL);

    else {
	current_unit->record = p;
	current_unit->record_psize = current_unit->record_size = len;
    }
}


/* next_internal()-- Next record for an internal unit */

static void next_internal(void) {
int rank, i, u;
char *p;

    if (ioparm->internal_unit != NULL) {
	if (current_unit->record != NULL)
	    current_unit->read_eof = 1;
	else {
	    current_unit->record = ioparm->internal_unit;
	    current_unit->record_size = current_unit->recl =
		ioparm->internal_unit_len;
	}

    } else {  /* Vector internal file */
	rank = ioparm->internal_array->rank;

	if (current_unit->record == NULL) {
	    for(i=0; i<rank; i++) {
		internal_array[i] = ioparm->internal_array->info[i].lbound;
		u = ioparm->internal_array->info[i].ubound;

		if (internal_array[i] > u) {
		    if (ioparm->mode == READING)
			current_unit->read_eof = 1;
		    else
			generate_error(ERROR_END, NULL);

		    return;
		}
	    }

	} else if (bump_element(ioparm->internal_array, internal_array)) {
	    if (ioparm->mode == READING)
		current_unit->read_eof = 1;
	    else
		generate_error(ERROR_END, NULL);

	    return;
	}

	p = ioparm->internal_array->offset;
	for(i=0; i<rank; i++)
	    p += internal_array[i] * ioparm->internal_array->info[i].mult;

	current_unit->record = p;
	current_unit->record_size = current_unit->recl =
	    ioparm->internal_array->element_size;
    }
}



/* size_record_buffer()-- Given a position within the write buffer,
 * make sure the buffer is large enough to hold it, resizing if
 * necessary.  Returns nonzero if there was not enough room and
 * generates the EOR error. */

static int size_record_buffer(int size) {
int new_size, err;
char *p;

    if (size == 0)
	size = 1;      /* Make sure buffer is non-null */

    if (size > current_unit->recl) {
	err = ioparm->mode == READING ? ERROR_READ_RECL : ERROR_WRITE_RECL;
	generate_error(err, NULL);
	current_unit->previous_noadvance = 0;
	return 1;
    }

    if (is_internal_unit() ||
	size <= current_unit->record_psize)
	return 0;

    new_size = (current_unit->record_psize == 0)
	? INITIAL_RECL 
	: ((current_unit->record_psize > current_unit->recl/2)
	   ? current_unit->recl
	   : 2*current_unit->record_psize);

    if (new_size < size)
	new_size = size;

    p = get_mem(new_size);
    memcpy(p, current_unit->record, current_unit->record_psize);

    if (current_unit->record != NULL)
	free_mem(current_unit->record);

    current_unit->record = p;
    current_unit->record_psize = new_size;

    return 0;
}



/* next_formatted_sequential()-- Read the next formatted sequential
 * record */

static void next_formatted_sequential(void) {
unsigned len;
int reading;
char *p;

    current_unit->record_size = 0;
    current_unit->read_eof = 0;
    reading = 1;

    while(reading) {
	len = INITIAL_RECL;
	p = salloc_rline(current_unit->s, &len);

	if (p == NULL) {
	    generate_error(ERROR_OS, NULL);
	    return;
	}

	if (len == 0) {
	    current_unit->read_eof = 1;
	    break;
	}

	if (p[len-1] == '\n') {
	    len--;    /* Discard trailing newline */

	    if (len > 0 && p[len-1] == '\r' && options.input_cr)
		len--;

	    reading = 0;
	}

	if (size_record_buffer(current_unit->record_size + len))
	    break;

	memcpy(current_unit->record + current_unit->record_size, p, len);
	current_unit->record_size += len;
    }
}


/* next_unformatted_sequential()-- Read the next sequential
 * unformatted record from a unit.  Returns nonzero if something goes
 * wrong. */

static void next_unformatted_sequential(void) {
unsigned len, m, n;
char *p;

    len = sizeof(int);
    p = salloc_r(current_unit->s, &len);
    current_unit->read_eof = 0;

    if (len == 0) {
	current_unit->read_eof = 1;
	return;
    }

    if (p == NULL)
	return;

    if (len != sizeof(int)) {
	generate_error(ERROR_CORRUPT_SEQUENTIAL, NULL);
	return;
    }

    n = get_length((unsigned char *) p);
    if (size_record_buffer(n))
	return;

    len = m = n + sizeof(int);
    /* The length of this record plus the trailing length */

    p = salloc_r(current_unit->s, &len);
    if (p == NULL || len != m) {
	generate_error(ERROR_CORRUPT_SEQUENTIAL, NULL);
	return;
    }

    memcpy(current_unit->record, p, n);

    current_unit->record_size = n;
}



/* read_next_record()-- Read the next record. */

static void read_next_record(void) {

    current_unit->record_size = 0;
    current_unit->offset = 0;

    if (current_unit->read_eof && !options.ignore_endfile) {
	generate_error(ERROR_END, NULL);
	current_unit->endfile = AT_ENDFILE;
	return;
    }

    switch(current_unit->flags.access) {
    case ACCESS_APPEND:         /* Can't happen */
    case ACCESS_SEQUENTIAL:
	if (current_unit->flags.form == FORM_UNFORMATTED)
	    next_unformatted_sequential();

	else if (is_internal_unit())
	    next_internal();

	else
	    next_formatted_sequential();

	break;

    case ACCESS_DIRECT:
	next_direct();
	break;

    case ACCESS_STREAM:
    case ACCESS_UNSPECIFIED:
	if (ioparm->pos != NULL) {
	    if (recursive_io())
		generate_error(ERROR_OPTION_CONFLICT,
			       "POS not allowed in child I/O statement");

	    else if (sseek(current_unit->s,
			   extract_mint(ioparm->pos,
					ioparm->pos_kind)-1) == FAILURE)
		generate_error(ERROR_OS, NULL);
	}

	return;
    }

    if (ioparm->advance_status == ADVANCE_YES)
	current_unit->last_record++;
}



/* next_list_char()-- Read the next character for a list formatted
 * read.  Returns -1 on end of file, \n on end of record.  This is
 * still fundamentally record-at-a-time reading. */

int next_list_char(void) {
unsigned len;
char *p;

    if (current_unit == NULL || current_unit->endfile != NO_ENDFILE)
	return -1;

    if (current_unit->flags.access == ACCESS_STREAM) {
	len = 1;
	p = salloc_r(current_unit->s, &len);
	if (p == NULL || len == 0)
	    return -1;

	return *p;
    }

    if (current_unit->offset > current_unit->record_size)
	read_next_record();

    if (current_unit->offset == current_unit->record_size) {
	current_unit->offset++;
	return current_unit->read_eof ? -1 : '\n';
    }

    return current_unit->record[current_unit->offset++];
}



/* read_block()-- Returns a pointer to bytes within the current record
 * buffer.  If the read is short, then it is because the current
 * record does not have enough data to satisfy the read request and
 * the file was opened with PAD=YES.  The caller must assume trailing
 * spaces for short reads.  If too many bytes are requested without
 * padding, we generate an error and return NULL. */

char *read_block(unsigned *length) {
int m, n;
char *p;

    if (current_unit->flags.access == ACCESS_STREAM) {
	if (current_unit->flags.form == FORM_UNFORMATTED)
	    p = salloc_r(current_unit->s, length);

	else {
	    p = salloc_rline(current_unit->s, length);

	    if (p != NULL && p[*length - 1] == '\n')
		(*length)--;
	}

	if (p == NULL)
	    generate_error(ERROR_OS, NULL);

	return p;
    }

    p = current_unit->record + current_unit->offset;
    m = current_unit->record_size - current_unit->offset;

    if (m >= *length) {
	n = *length;
	current_unit->offset += n;

    } else {
	n = m;

	if (current_unit->read_eof && !options.ignore_endfile &&
	    current_unit->record_size == 0) {
	    generate_error(ERROR_END, NULL);
	    current_unit->endfile = AFTER_ENDFILE;
	    return NULL;
	}

	if (ioparm->advance_status == ADVANCE_NO) {
	    generate_error(ERROR_EOR, NULL);
	    current_unit->previous_noadvance = 0;
	}

	if (current_unit->flags.form == FORM_FORMATTED &&
	    current_unit->flags.pad == PAD_YES) {
	    *length = m;
	    current_unit->offset = current_unit->record_size;
	} else {
	    generate_error(ERROR_READ_RECL, NULL);
	    p = NULL;
	}
    }

    if (ioparm->size != NULL)
	size_count += n;

    return p;
}



/* write_unformatted_sequential()-- Write an unformatted sequential
 * record.  Extra space has already been reserved at the front and
 * rear of the record for the lengths.  Add these and write the whole
 * thing.  The buffer is left allocated for the next record. */

static void write_unformatted_sequential(void) {
static int serial = 0;
char *q;
int len;

    len = current_unit->max_offset;
    serial++;

    q = salloc_w(current_unit->s, len + 2*sizeof(int), 0);

    if (q == NULL)
	generate_error(ERROR_OS, NULL);
    else {
	set_length(q, len);
	memcpy(q+sizeof(int), current_unit->record, len);
	set_length(q+sizeof(int)+len, len);
	sfree(current_unit->s);
    }
}



/* terminate_record()-- Write a sequential record terminator */

void terminate_record(iounit_t *u) {
char *q;

    q = salloc_w(u->s, 1 + (options.cr ? 1 : 0), 0);

    if (q == NULL)
	generate_error(ERROR_OS, NULL);

    else {
	if (options.cr)
	    *q++ = '\r';

	*q = '\n';
	sfree(u->s);
    }
}



/* write_formatted_sequential()-- Write a formatted sequential record.
 * Space has already been reserved for the trailing newline, so we
 * just add it and write the whole thing. */

static void write_formatted_sequential(int advancing) {
char *p, *q;
int len;

    if (is_internal_unit()) {
	len = current_unit->record_size - current_unit->max_offset;
	if (len > 0)
	    memset(current_unit->record + current_unit->offset, ' ', len);

	return;
    }

    len = current_unit->max_offset;

    if (!advancing && current_unit->offset < current_unit->max_offset)
	len = current_unit->offset;

    p = current_unit->record;

    q = salloc_w(current_unit->s, len, 0);
    if (q == NULL) {
	generate_error(ERROR_OS, NULL);
	return;
    }

    memcpy(q, p, len);
    sfree(current_unit->s);

    if (advancing)
	terminate_record(current_unit);
}



/* write_record()-- Actually write the current record.  This amounts
 * to simply sfree()-ing the block for a direct access unit or a
 * salloc()/copy/sfree() for a sequential unit. */

static void write_record(void) {

    if (is_internal_unit())
	goto internal;

    switch(current_unit->flags.access) {
    case ACCESS_DIRECT:
    internal:
	memset(current_unit->record + current_unit->max_offset, ' ',
	       current_unit->record_size - current_unit->max_offset);

	if (!is_internal_unit()) {
	    sfree(current_unit->s);
	    current_unit->record = NULL;
	}

	break;

    case ACCESS_SEQUENTIAL:
	if (current_unit->flags.form == FORM_UNFORMATTED)
	    write_unformatted_sequential();
	else
	    write_formatted_sequential(1);

	if (current_unit->s->truncate) {
	    truncate_file(current_unit->s);
	    current_unit->s->truncate = 0;
	}

	break;

    case ACCESS_STREAM:
	write_formatted_sequential(current_unit->flags.form == FORM_FORMATTED);

	if (current_unit->s->truncate) {
	    truncate_file(current_unit->s);
	    current_unit->s->truncate = 0;
	}

	break;

    default:
	internal_error("write_record(): Bad access");
    }

    if (current_unit->flags.access != ACCESS_STREAM &&
	ioparm->advance_status == ADVANCE_YES)
	current_unit->last_record++;

    current_unit->read_bad = 0;
}



/* write_block()-- Allocate a new set of bytes in the current record
 * buffer.  If bytes past the end of the record are requested, we
 * generate an error and return NULL. */

char *write_block(int length) {
char *p;
int m;

    m = current_unit->offset + length;

    if (size_record_buffer(m))
	p = NULL;
    else {
	p = current_unit->record + current_unit->offset;

	current_unit->offset = m;
	if (m > current_unit->max_offset)
	    current_unit->max_offset = m;
    }

    return p;
}



/* move_character_position()-- Move the current character position for
 * the T, TL, TR and X descriptors.  If absolute_flag is nonzero, the
 * column represents the current column number, otherwise it is a
 * difference to add to the current column position.  In write mode,
 * we add spaces to the buffer if the new position is beyond the
 * current active region. */

static void move_character_position(int column, int absolute_flag) {
unsigned u_column;
int new_column;

    if (current_unit->flags.access == ACCESS_STREAM &&
	ioparm->mode == READING) {
	if (!absolute_flag && column >= 0) {
	    u_column = column;
	    read_block(&u_column);
	}

	return;
    }

    new_column = absolute_flag
	? column - 1
	: current_unit->offset + column;

    if (new_column < 0)
	new_column = 0;

    if (new_column > current_unit->record_size)
	new_column = current_unit->record_size;

    size_record_buffer(new_column);

    if (ioparm->mode == WRITING && new_column > current_unit->max_offset)
	memset(current_unit->record + current_unit->max_offset, ' ',
	       new_column - current_unit->max_offset);

    current_unit->offset = new_column;
}



/* init_write()-- Called when we are advancing to the next record in a
 * WRITE statement, prior to data transfer.  This sets up the real
 * buffer for the transfer or a temporary buffer in the case of a
 * sequential write, because the real buffer is most likely smaller
 * than the maximum. */

static void init_write(void) {
int recl;
char *p;

    recl = current_unit->recl; 

    if (is_internal_unit())
	next_internal();

    else if (current_unit->flags.access == ACCESS_STREAM) {
	if (ioparm->pos != NULL) {
	    if (recursive_io())
		generate_error(ERROR_OPTION_CONFLICT,
			       "POS not allowed in child I/O statement");

	    else {
		if (sseek(current_unit->s,
			  extract_mint(ioparm->pos,
				       ioparm->pos_kind)-1) == FAILURE)
		    generate_error(ERROR_OS, NULL);

		if (current_unit->flags.form == FORM_FORMATTED)
		    current_unit->s->truncate = 1;
	    }

	    ioparm->pos = NULL;
	    ioparm->pos_kind = 0;
	}

    } else if (current_unit->flags.access == ACCESS_DIRECT) {
	if (sseek(current_unit->s,
		  (current_unit->last_record - 1)*current_unit->recl)
	    == FAILURE) {
	    generate_error(ERROR_OS, NULL);
	    return;
	}

	p = salloc_w(current_unit->s, recl, 1);
	if (p == NULL)
	    generate_error(ERROR_OS, NULL);

	current_unit->record = p;
	current_unit->record_psize = recl;
    }

    if (!recursive_io()) {
	current_unit->offset = 0;
	current_unit->max_offset = 0;
	current_unit->record_size = current_unit->recl;
    }
}



static void unformatted_write(bt type, void *source, int length) {
void *dest;
int len;

    if (type == BT_REAL)
	len = REAL_SIZE(length);

    else if (type == BT_COMPLEX)
	len = 2*REAL_SIZE(length);

    else
	len = length;

    dest = write_block(len);
    if (dest == NULL)
	return;

    memcpy(dest, source, len);

    if (reverse_flag())
	switch(type) {
	case BT_INTEGER:
	case BT_LOGICAL:
	case BT_REAL:
	    reverse_endian(dest, length);
	    break;

	case BT_COMPLEX:
	    reverse_endian(dest, length);
	    reverse_endian(dest+length, length);
	    break;

	default:
	    break;
	}
}



/* unformatted_read()-- Master function for unformatted reads.  */

static void unformatted_read(bt type, void *dest, int length) {
void *source;
unsigned w;
int len;

    if (type == BT_REAL)
	len = REAL_SIZE(length);

    else if (type == BT_COMPLEX)
	len = 2*REAL_SIZE(length);

    else
	len = length;

    w = len;
    source = read_block(&w);

    if (source == NULL)
	return;

    if (w != len) {
	generate_error(ERROR_END, NULL);
	return;
    }

    memcpy(dest, source, w);
    if (len != w)
	memset(((char *) dest) + w, ' ', len - w);

    if (reverse_flag())
	switch(type) {
	case BT_INTEGER:
	case BT_LOGICAL:
	case BT_REAL:
	    reverse_endian(dest, length);
	    break;

	case BT_COMPLEX:
	    reverse_endian(dest, length);
	    reverse_endian(dest+length, length);
	    break;

	default:
	    break;
	}
}



/* type_name()-- Return a pointer to the name of a type. */

char *type_name(bt type) {
char *p;

    switch(type) {
    case BT_INTEGER:     p = "INTEGER";    break;
    case BT_LOGICAL:     p = "LOGICAL";    break;
    case BT_CHARACTER:   p = "CHARACTER";  break;
    case BT_REAL:        p = "REAL";       break;
    case BT_COMPLEX:     p = "COMPLEX";    break;
    default:
	internal_error("type_name(): Bad type");
	p = NULL;
    }

    return p;
}



/* require_type()-- Given actual and expected types in a formatted
 * data transfer, make sure they agree.  If not, an error message is
 * generated.  Returns nonzero if something went wrong.  */

static int require_type(bt expected, bt actual, fnode *f) {
char buffer[200];

    if (actual == expected)
	return 0;

    st_sprintf(buffer, "Expected %s for item %d in formatted transfer, got %s",
	       type_name(expected), ioparm->item_count, type_name(actual));

    if (expected == BT_CHARACTER)
	strcat(buffer, ".  If you want to make character descriptors "
	       "typeless, compile with -fsloppy-char");

    format_error(f, buffer);
    return 1;
}



/* require_boz_type()-- BOZ types in F2008 are now integer, real or
 * complex.  It was a common extension to allow these before. */

static int require_boz_type(bt actual, fnode *f) {
char buffer[200];

    if (actual == BT_INTEGER || actual == BT_REAL || actual == BT_COMPLEX)
	return 0;

    st_printf(buffer, "Expected INTEGER, REAL or COMPLEX type for B/O/Z "
	      "descriptor, got %s", type_name(actual));

    format_error(f, buffer);
    return 1;
}



/* write_constant_string()-- write a constant string to the output.
 * This is complicated because the string can have doubled delimiters
 * in it.  The length in the format node is the true length. */

static void write_constant_string(fnode *f) {
char c, delimiter, *p, *q;
int length;

    length = f->u.string.length;
    if (length == 0)
	return;

    p = write_block(length);
    if (p == NULL)
	return;

    q = f->u.string.p;
    delimiter = q[-1];

    if (delimiter == 'h' || delimiter == 'H')
	memmove(p, q, length);  /* Hollerith constant */
    else
	for(; length>0; length--) {
	    c = *p++ = *q++;
	    if (c == delimiter) q++;    /* Skip the doubled delimiter */
	}
}



/* formatted_transfer()-- This subroutine is the main loop for a
 * formatted data transfer statement.  It would be natural to
 * implement this as a coroutine with the user program, but C makes
 * that awkward.  We loop, processing format elements.  When we
 * actually have to transfer data instead of just setting flags, we
 * return control to the user program which calls a subroutine that
 * supplies the address and type of the next element, then comes back
 * here to process it.  */

static void formatted_transfer(bt type, void *p, int len) {
int i, n, t;
fnode *f;

    /* Change a complex data item into a pair of reals */

    n = (p == NULL) ? 0 : ((type != BT_COMPLEX) ? 1 : 2);

    if (type == BT_COMPLEX)
	type = BT_REAL;

    for(;;) {
	if (ioparm->library_rc != LIBRARY_OK)
	    break;

	f = next_format();
	if (f == NULL) {
	    if (n > 0)
		format_error(NULL, "Exhausted data descriptors in format");
	    return;
	}

	/* If reversion has occurred and there is another real data item,
	 * then we have to move to the next record */

	if (ioparm->reversion_flag && n > 0) {
	    ioparm->reversion_flag = 0;
	    next_record();
	}

	t = 0;
	switch(f->format) {
	case FMT_I:
	    if (n == 0)
		goto need_data;

	    if (require_type(BT_INTEGER, type, f))
		return;

	    if (ioparm->mode == READING)
		read_decimal(f, p, len);
	    else
		write_i(f, p, len);

	    t = 1;
	    break;
 
	case FMT_A:
	    if (n == 0)
		goto need_data;

	    if (!ioparm->sloppy_char && require_type(BT_CHARACTER, type, f))
		return;

	    if (ioparm->mode == READING)
		read_a(f, p, len);
	    else
		write_a(f, p, len);

	    t = 1;
	    break;

	case FMT_L:
	    if (n == 0)
		goto need_data;

	    if (require_type(BT_LOGICAL, type, f))
		return;

	    if (ioparm->mode == READING)
		read_l(f, p, len);
	    else 
		write_l(f, p, len);

	    t = 1;
	    break;

	case FMT_D:
	    if (n == 0)
		goto need_data;

	    if (require_type(BT_REAL, type, f))
		return;

	    if (ioparm->mode == READING)
		read_f(f, p, len);
	    else 
		write_d(f, p, len);

	    t = 1;
	    break;

	case FMT_B:
	    if (n == 0)
		goto need_data;

	    if (require_boz_type(type, f))
		return;

	    if (ioparm->mode == READING)
		read_radix(f, p, len, 2);
	    else 
		write_b(f, p, len);

	    t = 1;
	    break;
 
	case FMT_O:
	    if (n == 0)
		goto need_data;

	    if (require_boz_type(type, f))
		return;

	    if (ioparm->mode == READING)
		read_radix(f, p, len, 8);
	    else
		write_o(f, p, len);

	    t = 1;
	    break;

	case FMT_Z:
	    if (n == 0)
		goto need_data;

	    if (require_boz_type(type, f))
		return;

	    if (ioparm->mode == READING)
		read_radix(f, p, len, 16);
	    else
		write_z(f, p, len);

	    t = 1;
	    break;

	case FMT_G:
	    if (n == 0)
		goto need_data;

	    if (ioparm->mode == READING)
		switch(type) {
		case BT_INTEGER:     read_decimal(f, p, len);  break;
		case BT_LOGICAL:     read_l(f, p, len);        break;
		case BT_CHARACTER:   read_a(f, p, len);        break;
		case BT_REAL:        read_f(f, p, len);        break;
		default:
		    goto bad_type;
		}

	    else
		switch(type) {
		case BT_INTEGER:     write_gi(f, p, len);  break;
		case BT_LOGICAL:     write_l(f, p, len);   break;
		case BT_CHARACTER:   write_a(f, p, len);   break;
		case BT_REAL:        write_gr(f, p, len);  break;
		default:
		bad_type:
		    internal_error("formatted_transfer(): Bad type");
		}

	    t = 1;
	    break;

	case FMT_F:
	    if (n == 0)
		goto need_data;

	    if (require_type(BT_REAL, type, f))
		return;

	    if (ioparm->mode == READING)
		read_f(f, p, len);
	    else 
		write_f(f, p, len);

	    t = 1;
	    break;

	case FMT_STRING:
	    if (ioparm->mode == READING) {
		format_error(f, "Constant string in input format");
		return;
	    }

	    write_constant_string(f);
	    break;

	case FMT_E:
	    if (n == 0)
		goto need_data;

	    if (require_type(BT_REAL, type, f))
		return;

	    if (ioparm->mode == READING)
		read_f(f, p, len);
	    else 
		write_e(f, p, len);

	    t = 1;
	    break;

	case FMT_ES:
	    if (n == 0)
		goto need_data;

	    if (require_type(BT_REAL, type, f))
		return;

	    if (ioparm->mode == READING)
		read_f(f, p, len);
	    else
		write_es(f, p, len);

	    t = 1;
	    break;

	case FMT_EN:
	    if (n == 0)
		goto need_data;

	    if (require_type(BT_REAL, type, f))
		return;

	    if (ioparm->mode == READING)
		read_f(f, p, len);
	    else
		write_en(f, p, len);

	    t = 1;
	    break;

	    /* Format codes that don't transfer data */

	case FMT_BN:
	    ioparm->blank_status = BLANK_NULL;
	    break;

	case FMT_BZ:
	    ioparm->blank_status = BLANK_ZERO;
	    break;

	case FMT_DC:
	    ioparm->current_decimal = DECIMAL_COMMA;
	    break;

	case FMT_DP:
	    ioparm->current_decimal = DECIMAL_POINT;
	    break;

	case FMT_P:
	    ioparm->scale_factor = f->u.k;
	    break;

	case FMT_DOLLAR:
	    ioparm->seen_dollar = 1;
	    break;

	case FMT_X:
	case FMT_TR:
	    move_character_position(f->u.n, 0);
	    break;

	case FMT_T:
	    move_character_position(f->u.n, 1);
	    break;

	case FMT_TL:
	    move_character_position(-f->u.n, 0);
	    break;

	case FMT_S:
	    ioparm->sign_status = SIGN_S;
	    break;

	case FMT_SS:
	    ioparm->sign_status = SIGN_SS;
	    break;

	case FMT_SP:
	    ioparm->sign_status = SIGN_SP;
	    break;

	case FMT_SLASH:
	    for(i=0; i<f->repeat; i++)
		next_record();

	    f->count = f->repeat;
	    break;

	case FMT_COLON:
	    /* A colon descriptor causes us to exit this loop (in
	     * particular preventing another / descriptor from being
	     * processed) unless there is another data item to be
	     * transferred. */

	    if (n == 0)
		return;

	    break;

	default:
	    internal_error("Bad format node");
	}

	/* Free a buffer that we had to allocate during a sequential
	 * formatted read of a block that was larger than the static
	 * buffer. */

	if (line_buffer != NULL) {
	    free_mem(line_buffer);
	    line_buffer = NULL;
	}

	/* Adjust the item count and data pointer */

	if (t > 0) {
	    n = n - t;

	    p = ((char *) p) + (type == BT_REAL ? REAL_SIZE(len) : len);
	}
    }

    return;

/* Come here when we need a data descriptor but don't have one.  We
 * push the current format node back onto the input, then return and
 * let the user program call us back with the data. */

need_data:
    unget_format(f);
}



/* start_transfer()-- Common code for starting a data transfer */

static void start_transfer(void) {

    if (ioparm->mode == READING && ioparm->library_rc == LIBRARY_OK &&
	current_unit != NULL &&
	current_unit->flags.access == ACCESS_SEQUENTIAL)
	switch(current_unit->endfile) {
	case NO_ENDFILE:
	    break;

	case AT_ENDFILE:
	    generate_error(ERROR_END, NULL);
	    current_unit->endfile = AFTER_ENDFILE;
	    break;

	case AFTER_ENDFILE:
	    if (!options.ignore_endfile)
		generate_error(ERROR_ENDFILE, NULL);
	    break;
	}
}



/* transfer_logical()-- Transfer a logical */

void transfer_logical(void *p, G95_DINT kind) {

    start_transfer();

    ioparm->item_count++;
    if (ioparm->library_rc == LIBRARY_OK)
	ioparm->transfer(BT_LOGICAL, p, kind);
}


/* transfer_character()-- Transfer a character */

void transfer_character(char *p, G95_DINT length) {

    start_transfer();

    ioparm->item_count++;
    if (ioparm->library_rc == LIBRARY_OK)
	ioparm->transfer(BT_CHARACTER, p, length);
}


/* transfer_complex()-- Transfer a complex number */

void transfer_complex(void *p, G95_DINT kind) {

    start_transfer();

    ioparm->item_count++;
    if (ioparm->library_rc == LIBRARY_OK)
	ioparm->transfer(BT_COMPLEX, p, kind);
}


/* transfer_integer()-- Transfer an integer */

void transfer_integer(void *p, G95_DINT kind) {

    start_transfer();

    ioparm->item_count++;
    if (ioparm->library_rc == LIBRARY_OK)
	ioparm->transfer(BT_INTEGER, p, kind);
}


/* transfer_real()-- Transfer a real */

void transfer_real(void *p, G95_DINT kind) {

    start_transfer();

    ioparm->item_count++;
    if (ioparm->library_rc == LIBRARY_OK)
	ioparm->transfer(BT_REAL, p, kind);
}


/* transfer_array()-- Given an array descriptor, transfer the entire
 * array by looping over all of the elements. */

static void transfer_array(g95_array_descriptor *desc, bt type,
			   G95_DINT kind) {
G95_AINT count[G95_MAX_DIMENSIONS];
derived_info *info;
int i, rank;
char *p;

    rank = desc->rank;
    info = current_info;

    /* If the array is not allocated, anything can happen.  Avoid a
     * crash in the case where the section might be zero-sized. */

    if (desc->base == NULL)
	return;

    for(i=0; i<rank; i++) {
	if (desc->info[i].lbound > desc->info[i].ubound)
	    return;

	count[i] = desc->info[i].lbound;
    }

    start_transfer();

    while(ioparm->library_rc == LIBRARY_OK) {
	ioparm->item_count++;

	p = desc->offset;
	for(i=0; i<rank; i++)
	    p += count[i] * desc->info[i].mult;

	switch(type) {
	case BT_DERIVED:
	    transfer_derived(p, info);
	    break;

	case BT_CHARACTER:
	    ioparm->transfer(type, p, desc->element_size);
	    break;

	default:
	    ioparm->transfer(type, p, kind);
	    break;
	}

	if (bump_element(desc, count))
	    break;
    }
}


/* transfer_character_array()-- Transfer a character array */

void transfer_character_array(g95_array_descriptor *desc, G95_DINT kind) {

    transfer_array(desc, BT_CHARACTER, kind);
}


/* transfer_complex_array()-- Transfer a complex array */

void transfer_complex_array(g95_array_descriptor *desc, G95_DINT kind) {

    transfer_array(desc, BT_COMPLEX, kind);
}


/* transfer_integer_array()-- Transfer an integer array */

void transfer_integer_array(g95_array_descriptor *desc, G95_DINT kind) {

    transfer_array(desc, BT_INTEGER, kind);
}


/* transfer_real_array()-- Transfer a real array */

void transfer_real_array(g95_array_descriptor *desc, G95_DINT kind) {

    transfer_array(desc, BT_REAL, kind);
}


/* transfer_logical_array()-- Transfer a logical array */

void transfer_logical_array(g95_array_descriptor *desc, G95_DINT kind) {

    transfer_array(desc, BT_LOGICAL, kind);
}


/* transfer_derived_array()-- Transfer a derived type array.  This
 * subroutine has a slightly difference calling convention than its
 * siblings. */

void transfer_derived_array(g95_array_descriptor *desc, derived_info *info) {

    current_info = info;
    transfer_array(desc, BT_DERIVED, 0);
}


/* transfer_derived()-- Transfer the elements of a derived type. */

void transfer_derived(char *address, derived_info *info) {
g95_array_descriptor *dp, desc;
char *a;
int i;

    start_transfer();

    while(info->name != NULL) {
	a = address + info->offset;

	if (info->rank == 0)
	    switch(info->type) {
	    case 'i':
		transfer_integer(a, info->kind);
		break;

	    case 'r':
		transfer_real(a, info->kind);
		break;

	    case 'z':
		transfer_complex(a, info->kind);
		break;

	    case 'l':
		transfer_logical(a, info->kind);
		break;

	    case 'c':
		transfer_character(a, info->kind);
		break;

	    case 'd':
		transfer_derived(a, info->info);
		break;
	    }

	else {   /* Array transfer */
	    if (info->shape == NULL)
		dp = (g95_array_descriptor *) a;

	    else {  /* Build a descriptor */
		for(i=0; i<info->rank; i++) {
		    desc.info[i].lbound = info->shape[2*i];
		    desc.info[i].ubound = info->shape[2*i+1];
		}

		desc.rank = info->rank;
		desc.base = a;

		switch(info->type) {
		case 'r':
		    desc.element_size = REAL_SIZE(info->kind);
		    break;

		case 'z':
		    desc.element_size = 2*REAL_SIZE(info->kind);
		    break;

		default:
		    desc.element_size = info->kind;
		    break;
		}

		init_multipliers(&desc);
	    }

	    switch(info->type) {
	    case 'i':
		transfer_integer_array(&desc, info->kind);
		break;

	    case 'r':
		transfer_real_array(&desc, info->kind);
		break;

	    case 'z':
		transfer_complex_array(&desc, info->kind);
		break;

	    case 'l':
		transfer_logical_array(&desc, info->kind);
		break;

	    case 'c':
		transfer_character_array(&desc, info->kind);
		break;

	    case 'd':
		transfer_derived_array(&desc, info->info);
		break;
	    }
	}

	info++;
    }
}



/* data_transfer_init()-- Initialize things for a data transfer.  Most
 * of this code is common for both reading and writing. */

static void data_transfer_init(int read_flag) {
G95_MINT m;

    ioparm->mode = read_flag ? READING : WRITING;

    if (ioparm->size != NULL)
	size_count = 0;            /* Initialize the count */

    if (ioparm->unit != NULL) {
	m = extract_mint(ioparm->unit, ioparm->unit_kind);
	if (m == -1) {
	    ioparm->unit = read_flag
		? &options.stdin_unit
		: &options.stdout_unit;

	    ioparm->unit_kind = sizeof(options.stdin_unit);
	}
    }

    current_unit = get_unit();
    if (current_unit == NULL)
	return;

    /* Check the action */

    if (read_flag && current_unit->flags.action == ACTION_WRITE)
	generate_error(ERROR_BAD_ACTION,
		       "Cannot read from file opened for WRITE");

    if (!read_flag && current_unit->flags.action == ACTION_READ)
	generate_error(ERROR_BAD_ACTION,
		       "Cannot write to file opened for READ");

    if (ioparm->library_rc != LIBRARY_OK)
	return;

    /* Check the format */

    if (ioparm->format)
	parse_format();

    if (ioparm->library_rc != LIBRARY_OK)
	return;

    if (current_unit->flags.form == FORM_UNFORMATTED &&
	(ioparm->format != NULL || ioparm->list_format))
	generate_error(ERROR_OPTION_CONFLICT,
		       "Format present for UNFORMATTED data transfer");

    if (current_unit->flags.form == FORM_FORMATTED && ioparm->format == NULL &&
	!ioparm->list_format && ioparm->namelist == NULL)
	generate_error(ERROR_OPTION_CONFLICT,
		       "Missing format for FORMATTED data transfer");

    if (is_internal_unit() && current_unit->flags.form == FORM_UNFORMATTED)
	generate_error(ERROR_OPTION_CONFLICT,
	      "Internal file cannot be accessed by UNFORMATTED data transfer");

  /* Check the record number */

    if (current_unit->flags.access == ACCESS_DIRECT && ioparm->rec == NULL) {
	generate_error(ERROR_MISSING_OPTION,
		       "Direct access data transfer requires record number");
	return;
    }

    if (current_unit->flags.access == ACCESS_SEQUENTIAL &&
	ioparm->rec != NULL) {
	generate_error(ERROR_OPTION_CONFLICT,
	     "Record number not allowed for sequential access data transfer");
	return;
    }

    ioparm->current_decimal = (ioparm->decimal == NULL)
	? current_unit->flags.decimal
	: find_option(ioparm->decimal, ioparm->decimal_len, decimal_opt,
		     "Bad DECIMAL parameter in data transfer statement");

    /* Process the ADVANCE option */

    ioparm->advance_status = (ioparm->advance == NULL) ? ADVANCE_UNSPECIFIED :
	find_option(ioparm->advance, ioparm->advance_len, advance_opt,
		     "Bad ADVANCE parameter in data transfer statement");

    if (ioparm->advance_status != ADVANCE_UNSPECIFIED) {
	if (current_unit->flags.access == ACCESS_DIRECT)
	    generate_error(ERROR_OPTION_CONFLICT,
		     "ADVANCE specification conflicts with sequential access");

    if (is_internal_unit())
      generate_error(ERROR_OPTION_CONFLICT,
		     "ADVANCE specification conflicts with internal file");

    if (ioparm->format == NULL || ioparm->list_format)
      generate_error(ERROR_OPTION_CONFLICT,
		     "ADVANCE specification requires an explicit format");
    }

    if (read_flag) {
	if (ioparm->eor != 0 && ioparm->advance_status != ADVANCE_NO)
	    generate_error(ERROR_MISSING_OPTION,
		  "EOR specification requires an ADVANCE specification of NO");

	if (ioparm->size != NULL && ioparm->advance_status != ADVANCE_NO)
	    generate_error(ERROR_MISSING_OPTION,
		 "SIZE specification requires an ADVANCE specification of NO");

    } else {  /* Write constraints */

	if (ioparm->end != 0)
	    generate_error(ERROR_OPTION_CONFLICT,
		      "END specification cannot appear in a write statement");

	if (ioparm->eor != 0)
	    generate_error(ERROR_OPTION_CONFLICT,
		      "EOR specification cannot appear in a write statement");

	if (ioparm->size != 0)
	    generate_error(ERROR_OPTION_CONFLICT,
		      "SIZE specification cannot appear in a write statement");
    }

    if (ioparm->advance_status == ADVANCE_UNSPECIFIED)
	ioparm->advance_status = ADVANCE_YES;

    if (ioparm->library_rc != LIBRARY_OK)
	return;

    /* Sanity checks on the record number */

    if (ioparm->rec != NULL) {
	m = extract_mint(ioparm->rec, ioparm->rec_kind);
	if (m <= 0) {
	    generate_error(ERROR_BAD_OPTION, "Record number must be positive");
	    return;
	}

	current_unit->last_record = m;
    }

    /* Set the initial value of flags */

    ioparm->blank_status = current_unit->flags.blank;
    ioparm->sign_status = SIGN_S;
    ioparm->scale_factor = 0;
    ioparm->seen_dollar = 0;
    ioparm->first_item = 1;
    ioparm->item_count = 0;

    /* Set up the subroutine that will handle the transfers */

    if (read_flag) {
	ioparm->transfer = (current_unit->flags.form == FORM_UNFORMATTED)
	    ? unformatted_read
	    : (ioparm->list_format
	       ? list_formatted_read
	       : formatted_transfer);

    } else {
	ioparm->transfer = (current_unit->flags.form == FORM_UNFORMATTED)
	    ? unformatted_write
	    : (ioparm->list_format
	       ? list_formatted_write
	       : formatted_transfer);
    }

    /* Make sure that we don't do a read after a nonadvancing write */

    if (read_flag) {
	if (!is_internal_unit() && current_unit->read_bad) {
	    generate_error(ERROR_BAD_OPTION,
			   "Cannot READ after a nonadvancing WRITE");
	    return;
	}
    } else {
	if (ioparm->advance_status == ADVANCE_NO &&
	    current_unit->flags.access == ACCESS_SEQUENTIAL)
	    current_unit->read_bad = 1;
    }

    /* Initialize records for read and write */

    if (!current_unit->previous_noadvance) {
	if (read_flag) {
	    if (current_unit->endfile == NO_ENDFILE)
		read_next_record();
	} else
	    init_write();
    }

    current_unit->previous_noadvance = (ioparm->advance_status == ADVANCE_NO);

    /* Start the data transfer if we are doing a formatted transfer */

    if (current_unit->flags.form == FORM_FORMATTED && !ioparm->list_format &&
	ioparm->namelist == NULL)
	formatted_transfer(0, NULL, 0);
}



/* next_record()-- Position to the next record, which means moving to
 * the end of the current record.  This can happen under several
 * different conditions. */

void next_record(void) {

    current_unit->read_bad = 0;

    if (ioparm->mode == READING)
	read_next_record();

    else {
	write_record();
	init_write();
    }
}



/* finalize_transfer()-- Finalize the current data transfer. */

static void finalize_transfer(void) {

    ioparm->transfer = NULL;

    if (ioparm->list_format && ioparm->mode == READING)
	finish_list_read();

    free_fnodes();
}


/* st_read()-- The READ statement */

void st_read(void) {

    library_start();

    data_transfer_init(1);

    if (ioparm->namelist != NULL)
	namelist_read();
}


/* st_read_done()-- Complete a read statement */

void st_read_done(void) {

    finalize_transfer();

    if (current_unit == NULL) {
	library_end();
	return;
    }

    if (current_unit->read_eof && ioparm->library_rc == LIBRARY_OK &&
	ioparm->item_count == 0) {
	generate_error(ERROR_END, NULL);

	if (!options.ignore_endfile)
	    current_unit->endfile = AFTER_ENDFILE;
    }

    if (is_internal_unit())
	free_mem(current_unit);

    if (ioparm->size != NULL)
	set_integer(size_count, ioparm->size, ioparm->size_kind); 

    library_end();
}



/* st_write()-- Begin a write statement */

void st_write(void) {

    library_start();

    data_transfer_init(0);

    if (current_unit != NULL &&
	(current_unit->unit_number == options.stdout_unit ||
	 current_unit->unit_number == options.stderr_unit))
	lock_stdout();

    if (ioparm->namelist != NULL)
	namelist_write();
}


/* st_write_done()-- Finish a write statement */

void st_write_done(void) {
int n;

    finalize_transfer();

    if (current_unit == NULL) {
	library_end();
	return;
    }

    if (ioparm->library_rc != 0)
	goto done;

    if (ioparm->advance_status == ADVANCE_YES && !ioparm->seen_dollar &&
	!recursive_io()) {
	write_record();
	current_unit->max_offset = 0;

    } else {
	write_formatted_sequential(0);
	ioparm->seen_dollar = 0;

	/* We've written min(max_offset, offset) bytes.  Now shift the
	 * unwritten part of the buffer left. */

	n = current_unit->max_offset - current_unit->offset;

	if (n > 0) {
	    memmove(current_unit->record,
		    current_unit->record + current_unit->offset, n);

	    current_unit->max_offset = n;
	    current_unit->offset = 0;

	} else if (n < 0) {
	    n = -n;
	    memmove(current_unit->record,
		    current_unit->record + current_unit->max_offset, n);

	    current_unit->max_offset = 0;
	    current_unit->offset = n;

	} else {
	    current_unit->max_offset = 0;
	    current_unit->offset = 0;
	}
    }

    /* Deal with endfile conditions associated with sequential files */
 
    if (current_unit->flags.access == ACCESS_SEQUENTIAL)
	switch(current_unit->endfile) {
	case AT_ENDFILE:   /* Remain at the endfile record */
	break;          

	case AFTER_ENDFILE:
	    current_unit->endfile = AT_ENDFILE;  /* Just at it now */
	    break;

	case NO_ENDFILE:   /* Get rid of whatever is after this record */
	    if (!is_internal_unit())
		truncate_file(current_unit->s);

	    current_unit->endfile = AT_ENDFILE;
	    break;
	}

done:
    if (current_unit != NULL &&
	(current_unit->unit_number == options.stdout_unit ||
	 current_unit->unit_number == options.stderr_unit))
	unlock_stdout();

    if (is_internal_unit())
	free_mem(current_unit);

    library_end();
}



/* iolength_transfer()-- Transfer function for the IOLENGTH statement */

static void iolength_transfer(bt type, void *p, int len) {

    switch(type) {
    case BT_COMPLEX:  *iolength_result += 2*len;           break;
    case BT_REAL:     *iolength_result += REAL_SIZE(len);  break;
    default:          *iolength_result += len;             break;
    }
}



/* st_iolength()-- Start an IOLENGTH form of the INQUIRE statement */

#define st_iolength prefix(st_iolength)

void st_iolength(G95_DINT *size) {

    library_start();

    iolength_result = size;
    ioparm->transfer = iolength_transfer;

    *size = 0;
}



/* st_iolength_done()-- Finish an IOLENGTH form of the INQUIRE statement */

#define st_iolength_done prefix(st_iolength_done)

void st_iolength_done(void) {

    library_end();
}


#define unhandled_eof prefix(unhandled_eof)

void unhandled_eof(void) {

    internal_error("EOF condition not handled-- END= tag needed");
}
