
/* Copyright (C) 2005-2008, Free Software Foundation, Inc.
   Contributed by Andy Vaught

  This file is part of g95.

  G95 is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  G95 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with g95; see the file COPYING.  If not, write to
  the Free Software Foundation, 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

  In addition to the permissions in the GNU General Public License, the
  Free Software Foundation gives you unlimited permission to link the
  compiled version of this file into combinations with other programs,
  and to distribute those combinations without any restriction coming
  from the use of this file.  (The General Public License restrictions
  do apply in other respects; for example, they cover modification of
  the file, and distribution when not linked into a combined executable.)
*/


#include "runtime.h"


static char yes[] = "YES", no[] = "NO", unknown[] = "UNKNOWN";


/* standard_desc()-- Return nonzero if the descriptor is a standard
 * descriptor inherited from the operating system. */

int standard_desc(OS_HANDLE fd) {

    return (fd == GetStdHandle(STD_INPUT_HANDLE) ||
	    fd == GetStdHandle(STD_OUTPUT_HANDLE) ||
	    fd == GetStdHandle(STD_ERROR_HANDLE));
}



/* sys_exit()-- Terminate the program with an exit code.
 * ExitProcess() has problems propagating the exit code, so we use
 * exit(). */

void sys_exit(int abend, int code) {

    exit(code);
}



/* setfilepointer()-- Wrapper for SetFilePointer and SetFilePointerEx,
 * depending on what is available. */

static int setfilepointer(OS_HANDLE fd, LARGE_INTEGER pos, LARGE_INTEGER *new,
			  int whence) {
static int check_for_ex=1, (*sfpx)() = NULL;
HANDLE h;
DWORD rc;

    if (check_for_ex) {
	h = GetModuleHandle("kernel32.dll");
	sfpx = GetProcAddress(h, "SetFilePointerEx");
	check_for_ex = 0;
    }

    if (sfpx != NULL)
	return (*sfpx)(fd, pos, new, whence);

    if (new != NULL)
	new->u.HighPart = 0;

    rc = SetFilePointer(fd, pos.u.LowPart, NULL, whence);
    if (rc == INVALID_SET_FILE_POINTER) {
	if (new != NULL)
	    new->u.LowPart = 0;

	return 0;
    }

    if (new != NULL)
	new->u.LowPart = rc;

    return 1;
}



/* fd_length()-- Given a file descriptor, return the length of the
 * file.  This is zero for files that aren't seekable. */

off_t fd_length(OS_HANDLE fd) {
LARGE_INTEGER pos, save, zero;

    zero.QuadPart = 0;

    setfilepointer(fd, zero, &save, FILE_CURRENT);
    setfilepointer(fd, zero, &pos,  FILE_END);
    setfilepointer(fd, save, NULL,  FILE_BEGIN);

    return pos.QuadPart;
}



/* filename_length()-- Return the length of a file for a particular
 * filename. */

off_t filename_length(void) {
char path[PATH_LENGTH+1];
off_t size;
HANDLE h;

    if (unpack_filename(path, ioparm->file, ioparm->file_len))
	return 0;

    h = CreateFile(path, GENERIC_READ, 0, NULL, OPEN_EXISTING,
		   FILE_ATTRIBUTE_NORMAL, NULL);
    if (h == INVALID_HANDLE_VALUE)
	return 0;

    size = fd_length(h);
    CloseHandle(h);

    return size;
}



/* get_oserror()-- Get the most recent operating system error. */

char *get_oserror(void) {
char *message;

    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
		  FORMAT_MESSAGE_FROM_SYSTEM, NULL, GetLastError(),
		  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		  (LPTSTR) &message, 0, NULL);

    return message;
}



/* get_oserrno()-- Get the operating system error number */

int get_oserrno(void) {

    return GetLastError();
}



/* open_tempfile()-- Generate a temporary filename for a scratch file
 * and open it.  The template is pointed to by ioparm->file, which is
 * copied into the unit structure and freed later. */

int open_tempfile(char *path, OS_HANDLE *result) {
char *tempdir, tmpdir0[PATH_LENGTH];
OS_HANDLE h;

    tempdir = getenv("G95_TMPDIR");

    if (tempdir == NULL) {
	GetTempPath(PATH_LENGTH, tmpdir0);
	tempdir = tmpdir0;
    }

    if (GetTempFileName(tempdir, "g95", 0, path) == 0) {
	if (GetTempFileName(".", "g95", 0, path) == 0)
	    return 1;
    }

    h = CreateFile(path, GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_ALWAYS,
		   FILE_FLAG_DELETE_ON_CLOSE, NULL);

    if (h == INVALID_HANDLE_VALUE)
	return 1;

    *result = h;
    return 0;
}



/* win_open()-- The mode is a unix mode that we translate to windows
 * constants. */

int open_file(unit_action action, unit_status status, OS_HANDLE *result,
	      char *path) { 
int mode, create;
HANDLE h;

    if (unpack_filename(path, ioparm->file, ioparm->file_len)) {
	SetLastError(ERROR_FILE_NOT_FOUND);    /* Fake an OS error */
	return -1;
    }
  
    switch(action) {
    case ACTION_READ:
	mode = GENERIC_READ;
	break;

    case ACTION_WRITE:
	mode = GENERIC_WRITE;
	break;

    case ACTION_READWRITE:
	mode = GENERIC_READ | GENERIC_WRITE;
	break;

    default:
	internal_error("open_file(): Bad action");
	mode = 0;
    }

    switch(status) {
    case STATUS_NEW:
	create = CREATE_NEW;
	break;

    case STATUS_OLD:
	create = OPEN_EXISTING;
	break;

    case STATUS_UNKNOWN:
    case STATUS_SCRATCH:
	create = OPEN_ALWAYS;
	break;

    case STATUS_REPLACE:
	create = CREATE_ALWAYS;
	break;

    default:
	internal_error("open_file(): Bad status");
	create = 0;
    }

    h = CreateFile(path, mode, FILE_SHARE_READ | FILE_SHARE_WRITE,
		   NULL, create, FILE_ATTRIBUTE_NORMAL, NULL);

    if (h != INVALID_HANDLE_VALUE)
	*result = h;

    return h == INVALID_HANDLE_VALUE;
}



/* win_read()-- read()-like interface for ReadFile */

int win_read(OS_HANDLE fd, char *buffer, int count) {
DWORD m, n;

    m = count; 
    return ReadFile(fd, buffer, m, &n, NULL) ? n : -1;
}



int win_write(OS_HANDLE fd, char *buffer, int count) {
DWORD m, n;

    m = count;
    return WriteFile(fd, buffer, m, &n, NULL) ? n : -1;
}



int win_lseek(OS_HANDLE fd, off_t position, int flag) {
LARGE_INTEGER pos;

    pos.QuadPart = position;

    return setfilepointer(fd, pos, NULL, flag) ? 0 : -1;
}


int win_close(OS_HANDLE fd) {

    return CloseHandle(fd) ? 0 : -1;
}


/* input_stream()-- Return a stream pointer to the default input stream.
 * Called on initialization. */

stream *input_stream(void) {

    return fd_to_stream(GetStdHandle(STD_INPUT_HANDLE), 1);
}



/* output_stream()-- Return a stream pointer to the default output stream.
 * Called on initialization. */

stream *output_stream(void) {

    return fd_to_stream(GetStdHandle(STD_OUTPUT_HANDLE), 1);
}



/* error_stream()-- Return a stream pointer to the default error stream.
 * Called on initialization. */

stream *error_stream(void) {

    return fd_to_stream(GetStdHandle(STD_ERROR_HANDLE), 1);
}



/* delete_file()-- Given a unit structure, delete the file associated
 * with the unit.  Returns nonzero if something went wrong. */

int delete_file(char *path) {

    return DeleteFile(path) ? 0 : -1;
}



/* win_ftruncate()-- Truncate a file.  The unix version doesn't change
 * the file pointer, causing some gymnastics here. */

int win_ftruncate(OS_HANDLE fd, off_t position) {
LARGE_INTEGER zero, pos, save;
int rc;

    zero.QuadPart = 0;
    pos.QuadPart = position;

    setfilepointer(fd, zero, &save, FILE_CURRENT);
    setfilepointer(fd, pos, NULL, FILE_BEGIN);

    rc = SetEndOfFile(fd) ? 0 : -1;
    setfilepointer(fd, save, NULL, FILE_BEGIN);

    return rc;
}



/* compare_file_filename()-- Given an open stream and a fortran string
 * that is a filename, figure out if the file is the same as the
 * filename. */

int compare_file_filename(iounit_t *u, char *filename, int filename_len) {
char path[PATH_LENGTH+1];

    if (unpack_filename(path, filename, filename_len))
	return 0;

    if (filename_len != u->file_len)
	return 0;

    return memcmp(path, u->file, filename_len) == 0;
}


static iounit_t *find_file0(iounit_t *u) {
iounit_t *v;

    if (u == NULL)
	return NULL;

    if (compare_string(u->file, u->file_len, ioparm->file,
		       ioparm->file_len) == 0)
	return u;

    v = find_file0(u->left);
    if (v != NULL)
	return v;

    v = find_file0(u->right);
    if (v != NULL)
	return v;

    return NULL;
}


/* find_file()-- Take the current filename and see if there is a unit
 * that has the file already open.  Returns a pointer to the unit if so. */

iounit_t *find_file(void) {

    return find_file0(globals.unit_root);
}



/* init_error_stream()-- Return a pointer to the error stream.  This
 * subroutine is called when the stream is needed, rather than at
 * initialization.  We want this to work even if memory has been
 * seriously corrupted. */

stream *init_error_stream(void) {
static stream error;

    memset(&error, '\0', sizeof(error));

    error.fd = options.use_stderr
	? GetStdHandle(STD_ERROR_HANDLE)
	: GetStdHandle(STD_OUTPUT_HANDLE);

    error.unbuffered = 1;
    error.buffer = error.small_buffer;

    return &error;
}



/* terminal_device()-- See if the unit is associated with a terminal.
 * Unlike other files, terminal devices can be opened multiple times. */

int terminal_device(iounit_t *u) {

    return standard_desc(u->s->fd);
}



/* file_exists()-- Returns nonzero if the current filename exists on
 * the system */

int file_exists(void) {
char path[PATH_LENGTH+1];

    return (unpack_filename(path, ioparm->file, ioparm->file_len) ||
	    GetFileAttributes(path) == INVALID_FILE_ATTRIBUTES) ? 0 : 1;
}



/* get_type()-- Return a GetFileType() return value given the path.
 * To get this we have to open the file. */

static int get_type(char *fname, int fname_len) {
char path[PATH_LENGTH+1];
HANDLE h;
int t;

    if (unpack_filename(path, fname, fname_len))
	return FILE_TYPE_UNKNOWN;

    h = CreateFile(path, GENERIC_READ, 0, NULL, OPEN_EXISTING,
		   FILE_ATTRIBUTE_NORMAL, NULL);

    if (h == INVALID_HANDLE_VALUE)
	return FILE_TYPE_UNKNOWN;

    t = GetFileType(h);
    CloseHandle(h);

    return t;
}



/* inquire_sequential()-- Determine if the file is suitable for
 * sequential access. */

static char *sequential(int type) {

    switch(type) {
    case FILE_TYPE_CHAR:
    case FILE_TYPE_DISK:
    case FILE_TYPE_PIPE:
	return yes;

    default:
	break;    
    }

    return unknown;
}



/* inquire_sequential()-- Given a fortran string, determine if the
 * file is suitable for sequential access.  Returns a C-style
 * string. */

char *inquire_sequential(char *fname, int fname_len) {

    return sequential(get_type(fname, fname_len));
}



/* inquire_sequential_fd()-- Determine if an open file is suitable for
 * sequential access. */

char *inquire_sequential_fd(iounit_t *u) {

    if (u == NULL)
	return yes;

    if (u->flags.access == ACCESS_DIRECT)
	return no;

    if (u->flags.access == ACCESS_SEQUENTIAL)
	return yes;

    return sequential(GetFileType(u->s->fd));
}



/* direct()-- Given a fortran string, determine if the file is
 * suitable for direct access.  Returns a C-style string. */

static char *direct(int type) {

    switch(type) {
    case FILE_TYPE_DISK:
	return yes;

    case FILE_TYPE_CHAR:
    case FILE_TYPE_PIPE:
	return no;

    default:
	break;
    }

    return unknown;
}



/* inquire_direct()-- Given a fortran string, determine if the file is
 * suitable for direct access.  Returns a C-style string. */

char *inquire_direct(char *fname, int fname_len) {

    return direct(get_type(fname, fname_len));
}



/* inquire_direct_fd()-- Given an open unit, see if it can be opened
 * in direct mode. */

char *inquire_direct_fd(iounit_t *u) {

    if (u == NULL || u->flags.access == ACCESS_SEQUENTIAL)
	return no;

    if (u->flags.access == ACCESS_DIRECT)
	return yes;

    return direct(GetFileType(u->s->fd));
}


/* formatted()-- Determine if the file is suitable for formatted form.
 * Returns a C-style string. */

static char *formatted(int type) {

    switch(type) {
    case FILE_TYPE_CHAR:
    case FILE_TYPE_DISK:
    case FILE_TYPE_PIPE:
	return yes;

    default:
	break;    
    }

    return unknown;
}



/* inquire_formatted()-- Given a fortran string, determine if the file
 * is suitable for formatted form.  Returns a C-style string. */

char *inquire_formatted(char *fname, int fname_len) {

    return formatted(get_type(fname, fname_len));
}



/* inquire_formatted_fd()-- Determined if an open file can be opened
 * in formatted form. */

char *inquire_formatted_fd(iounit_t *u) {

    if (u == NULL)
	return yes;

    return formatted(GetFileType(u->s->fd));
}



/* inquire_unformatted()-- Given a fortran string, determine if the file
 * is suitable for unformatted form.  Returns a C-style string. */

char *inquire_unformatted(char *fname, int fname_len) {

    return inquire_formatted(fname, fname_len);
}



char *inquire_unformatted_fd(iounit_t *u) {

    return inquire_formatted_fd(u);
}



#define R_OK 1
#define W_OK 2

/* inquire_access()-- Given a fortran string, determine if the file is
 * suitable for access. */

static char *inquire_access(char *fname, int fname_len, int mode) {
char path[PATH_LENGTH+1];

    if (fname == NULL || unpack_filename(path, fname, fname_len))
	return no;

    return yes;
}



static char *inquire_access_fd(iounit_t *u, int mode) {
int flags;
char *p;

    if (u != NULL && u->file_len != 0)
	return inquire_access(u->file, u->file_len, mode);

    flags = (u == NULL)
	? ACTION_READWRITE
	: u->flags.action;

    /* preopened unit */

    switch(flags) {
    case ACTION_READ:       p = ((mode & W_OK) == 0) ? yes : no; break;
    case ACTION_WRITE:      p = ((mode & R_OK) == 0) ? yes : no; break;
    case ACTION_READWRITE:  p = yes;      break;
    default:                p = unknown;  break;
    }

    return p;
}


/* inquire_read()-- Given a fortran string, determine if the file is
 * suitable for READ access. */

char *inquire_read(char *fname, int fname_len) {

    return inquire_access(fname, fname_len, R_OK);
}


char *inquire_read_fd(iounit_t *u) {

    return inquire_access_fd(u, R_OK);
}



/* inquire_write()-- Given a fortran string, determine if the file is
 * suitable for READ access. */

char *inquire_write(char *fname, int fname_len) {

    return inquire_access(fname, fname_len, W_OK);
}


char *inquire_write_fd(iounit_t *u) {

    return inquire_access_fd(u, W_OK);
}


/* inquire_readwrite()-- Given a fortran string, determine if the file is
 * suitable for read and write access. */

char *inquire_readwrite(char *fname, int fname_len) {

    return inquire_access(fname, fname_len, R_OK | W_OK);
}


char *inquire_readwrite_fd(iounit_t *u) {

    return inquire_access_fd(u, R_OK | W_OK);
}


/* default_action()-- Figure out a default action for a file that
 * we're about to open.  */

unit_action default_action(void) {
char path[PATH_LENGTH+1];
int a;

    if (unpack_filename(path, ioparm->file, ioparm->file_len))
	return ACTION_READWRITE;

    a = GetFileAttributes(path);
    if (a != INVALID_FILE_ATTRIBUTES && (a & FILE_ATTRIBUTE_READONLY))
	return ACTION_READ;

    return ACTION_READWRITE;
}
