/*
  setup.c
  
  for GalaXa
  
  by Bill Kendrick
  kendrick@zippy.sonoma.edu
  http://zippy.sonoma.edu/kendrick/
  
  April 6, 1998 - April 21, 1998
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/types.h>
#include <sys/time.h>
#include <math.h>
#include "hints.h"
#include "randnum.h"
#include "setup.h"
#include "text.h"
#include "soundmgr.h"


/* -- Setup -- */

int setup(int argc, char * argv[])
{
  int level;
  
  
  /* Defaults: */
  
  greymode = 0;
  level = 1;
  
  
  /* Check usage: */
  
  if (argc >= 2)
    {
      /* -l# = Level */
      
      if (strcasecmp(argv[argc - 1], "-grey") == 0 ||
	  strcasecmp(argv[argc - 1], "-g") == 0)
        {
	  printf("(Using greyscale)\n");
	  
          greymode = 1;
	  
          argc--;
        }
    }
  
  
  sound_fs = NULL;
  
  if (argc >= 2)
    {
      /* -sound = Use sound */
      
      if (strcasecmp(argv[argc - 1], "-sound") == 0 ||
	  strcasecmp(argv[argc - 1], "-s") == 0)
        {
	  printf("(Using sound)\n");
	  
	  sound_fs = soundmgr(NUM_SOUNDS, sounds);
	  
	  playsound(sound_fs, SND_WELCOME);
	  
          argc--;
        }
    }
    
  
  if (argc >= 2)
    {
      /* -l# = Level */
      
      if (strstr(argv[argc - 1], "-l") == argv[argc - 1] ||
          strstr(argv[argc - 1], "-L") == argv[argc - 1])
        {
          level = atoi(argv[argc - 1] + 2);
          
          if (level < 1 || level > 99)
            level = 1;
          
          argc--;
        }
    }
    
  
  if (argc == 1)
    {
      /* Try to determine display */
      
      if (getenv("DISPLAY") == NULL)
	{
	  fprintf(stderr, "I can't determine your display! Please specify!\n");
	  exit(1);
	}
      else
	{
	  strcpy(server, getenv("DISPLAY"));
	}
    }
  else if (argc == 2)
    {
      /* Get display off of command line: */
      
      strcpy(server, argv[1]);
    }
  else
    {
      /* Tell them how to run the program! */
      
      fprintf(stderr, "Usage: %s [display] [-l#] [-grey]\n", argv[0]);
      exit(1);
    }
  
  xms[0] = 0;
  yms[0] = -2;

  xms[1] = 1;
  yms[1] = -1;

  xms[2] = 2;
  yms[2] = 0;

  xms[3] = 1;
  yms[3] = 1;

  xms[4] = 0;
  yms[4] = 2;

  xms[5] = -1;
  yms[5] = 1;

  xms[6] = -2;
  yms[6] = 0;

  xms[7] = -1;
  yms[7] = -1;
  
  return(level);
}


/* -- Xsetup -- */

void Xsetup(void)
{
  XSetWindowAttributes attributes;
  unsigned long attr_mask;
  int temp_depth, i, z, oz;
  
  
  /* Connect to display: */
  
  display = XOpenDisplay(server);
  
  if (display != NULL)
    {
      screen = DefaultScreen(display);
      root = RootWindow(display, screen);
    }
  else
    {
      fprintf(stderr, "Can't connect to %s!\n", server);
    }
  
  
  /* Determine a good color depth: */
  
  good_depth = DefaultDepthOfScreen(DefaultScreenOfDisplay(display));
  printf("(Server said default color depth is %d bit(s))\n\n", good_depth);
  

  /* Load font: */
  
  font = LoadFont(display, "variable", "fixed");
  fh = FontHeight(font);
  
  
  /* Get primitive colors: */
  
  black = BlackPixel(display, screen);
  white = WhitePixel(display, screen);
  
  
  /* Open the window: */
  
  attributes.event_mask = KeyPressMask | KeyReleaseMask;
  attributes.border_pixel = white;
  attributes.background_pixel = black;
  attr_mask = CWEventMask | CWBackPixel | CWBorderPixel;
  
  window = XCreateWindow(display, root, 10, 10, WIDTH, HEIGHT, 1,
                         CopyFromParent, InputOutput,
			 (Visual *) CopyFromParent, attr_mask,
                         &attributes);
  
#ifdef USE_BACKBUFFER
  backbuffer = 
    XCreatePixmap(display, window, WIDTH, HEIGHT, good_depth);
#else
  backbuffer = window;
#endif
  
  SetStandardHints(display, window, "GalaXa", "GalaXa", 10, 10,
		   WIDTH, HEIGHT);
  
  
  /* Set up visual: */
  
  if (SetUpVisual(display, screen, &temp_visual, &temp_depth))
    {
      if (!SetUpColormap(display, screen, window, temp_visual, &colormap))
	{
	  fprintf(stderr, "Error: Could not set up colormap!\n");
	  exit(1);
	}
    }
  else
    fprintf(stderr, "Warning: Could not find Pseudocolor Visual!\n");
  
  
  /* Create GC's: */
  
  whitegc = CreateGC(white);
  blackgc = CreateGC(black);
  
  greygc = CreateGC(AllocNamedColor(display, colormap, "grey", white));
  
  for (i = 0; i < NUM_EXPLOSION_COLORS; i++)
    explosiongcs[i] = CreateGC(AllocNamedColor(display, colormap,
					       explosioncolors[i], white));
  
  
  /* Put window up: */
  
  XMapWindow(display, window);
  XMapRaised(display, window);
  XFlush(display);
  
  /* Create object pixmaps: */
  
  color_clear();

  XFillRectangle(display, backbuffer, blackgc, 0, 0, WIDTH, HEIGHT);
  
  drawcenteredtext(display, backbuffer, whitegc, 0, WIDTH,
		   (HEIGHT + fh) / 2, "ONE MOMENT", font);
  XFlush(display);
  
  for (i = 0; i < 10; i++)
    {
      loadobject(i);
    }
  
  
  oz = 0;
  for (i = 10; i < NUM_OBJECTS; i++)
    {
      loadobject(i);
      
      z = 10 - ((i * 10) / NUM_OBJECTS);

      /* if (i == OBJ_TITLE)
	{
	  drawshape((WIDTH - object_width[OBJ_TITLE]) / 2, 0, OBJ_TITLE,
		    backbuffer);
	  oz = 0;
	} */
      
      if (z != oz)
	{
	  XFillRectangle(display, backbuffer, blackgc,
			 WIDTH / 2 - 24, HEIGHT - 92, 48, 32);
	  
	  drawshape(WIDTH / 2 - 24, HEIGHT - 92, z / 10, backbuffer);
	  drawshape(WIDTH / 2, HEIGHT - 92, z % 10, backbuffer);
	  
	  oz = z;
	}
      
      XFillRectangle(display, backbuffer, greygc,
		     WIDTH / 4, HEIGHT - 60, WIDTH / 2, 10);
      
      XFillRectangle(display, backbuffer, whitegc,
		     WIDTH / 4, HEIGHT - 60, (i * WIDTH / 2) / NUM_OBJECTS,
		     10);
      
      updatebackbuffer();
      XFlush(display);
    }
}


/* Close everything up: */

void shutdown(void)
{
  XDestroyWindow(display, window);
  XCloseDisplay(display);
  
  printf("\nThank you for playing GalaXa!\n");
  printf("by Bill Kendrick\n");
  printf("kendrick@zippy.sonoma.edu\n\n");
}


/* Creates a CG: */

GC CreateGC(unsigned long fore)
{
  XGCValues xgcv;
  
  xgcv.foreground = fore;
  xgcv.background = black;
  
  return(XCreateGC(display, window, (GCForeground | GCBackground), &xgcv));
}


/* Create a visual: */

int SetUpVisual(Display *display, int screen, Visual **visual, int *depth)
{
  int number_visuals;
  XVisualInfo *visual_array;
  XVisualInfo visual_info_template;
  int status;
  
  status = False;
  
  /* Will the default visual work? */
  
  if (DefaultVisual(display,screen)->class == PseudoColor ||
      DefaultVisual(display,screen)->class == TrueColor)
    {
      *visual = DefaultVisual(display,screen);
      *depth = DefaultDepth(display,screen);
      status = True;
    }
  else
    {
      /* we look for a PseudoColor visual on the current screen */
      visual_info_template.class = PseudoColor;
      visual_info_template.screen = screen;
      visual_array = XGetVisualInfo(display,VisualClassMask | VisualScreenMask,
				    &visual_info_template,&number_visuals);
      
      /* Did we find one? */
      if ((number_visuals>0) && (visual_array != NULL))
        {
          /* Choose the first PseudoColor visual (LAZY!) */
          *visual = visual_array[0].visual;
          *depth = visual_array[0].depth;
          XFree(visual_array);   /* done with this chunk of RAM */
          status = True;
        }
      else
        {
          *visual = CopyFromParent;
          status = False;  /* sorry - live with what's left */
        }
    }
  
  return(status);
}


/* SetUpColormap */

int SetUpColormap(Display *display, int screen, Window window, Visual *visual,
                  Colormap *colormap)
{
  int status = False;
  
  if (visual == DefaultVisual(display,screen))
    {
      *colormap = DefaultColormap(display,screen);
      status = True;
    }
  else
    {
      *colormap = XCreateColormap(display,window,visual,AllocNone);
      if (*colormap != None)
        {
          XSetWindowColormap(display,window,*colormap);
          status = True;
        }
      else
        *colormap = DefaultColormap(display,screen);
    }
  
  return(status);
}


/* Draws a shape (using mask): */

void drawshape(int x, int y, int shape, Drawable d)
{
#ifdef BETTER_GRAPHICS
  XSetClipMask(display, object_gc[shape], object_mask[shape]);
  XSetClipOrigin(display, object_gc[shape], x, y);
#endif

  XCopyArea(display, object_pixmap[shape], d,
	    object_gc[shape], 0, 0,
	    object_width[shape], object_height[shape],
	    x, y);
}


void color_clear(void)
{
  num_seen_colors = 0;
}


int color_seen(unsigned int red, unsigned int green, unsigned int blue)
{
  int i, found;
  
  found = -1;
  
  for (i = 0; i < num_seen_colors; i++)
    {
      if (color_list[i].red == red &&
	  color_list[i].green == green &&
	  color_list[i].blue == blue)
	found = i;
    }
  
  return found;
}


int color_add(unsigned int red, unsigned int green, unsigned int blue,
	       unsigned long pixel, int owner)
{
  color_list[num_seen_colors].red = red;
  color_list[num_seen_colors].green = green;
  color_list[num_seen_colors].blue = blue;
  color_list[num_seen_colors].pixel = pixel;
  color_list[num_seen_colors].owner = owner;
  
  num_seen_colors++;
  
  if (num_seen_colors >= MAX_COLORS)
    {
      fprintf(stderr, "Out of colors!\n");
      exit(1);
    }
  
  return(num_seen_colors - 1);
}


void updatebackbuffer(void)
{
#ifdef USE_BACKBUFFER
  XCopyArea(display, backbuffer, window, whitegc, 0, 0, WIDTH, HEIGHT, 0, 0);
#endif
}


void loadobject(int i)
{
  int x, y, red, green, blue, color_pos, width, height, z;
  char file[512], line[512];
  FILE * fi;
  GC colorgc[MAX_COLORS], color0gc, color1gc;
  XColor xcol;
  XGCValues xgcv, penxgcv;
  unsigned long color_want;

  
  sprintf(file, "images/%s.ppm", object_names[i]);
  
  fi = fopen(file, "r");
  if (fi == NULL)
    {
      perror(file);
      exit(1);
    }
  
  fgets(line, 512, fi);  /* type */
  /* fgets(line, 512, fi); */ /* comment */
  
  fgets(line, 512, fi);  /* size */
  sscanf(line, "%d %d", &width, &height);
  
  fgets(line, 512, fi);  /* maxval */
  
  
  object_width[i] = width;
  object_height[i] = height;
  
  
  object_pixmap[i] =
    XCreatePixmap(display, window, width, height, 
		  good_depth);
  
  XFillRectangle(display, object_pixmap[i], blackgc, 0, 0, width, height);
  
  object_mask[i] = 
    XCreatePixmap(display, window, width, height, 1);
  
  xgcv.foreground = 0;
  color0gc = XCreateGC(display, object_mask[i], GCForeground, &xgcv);
  
  xgcv.foreground = 1;
  color1gc = XCreateGC(display, object_mask[i], GCForeground, &xgcv);
  
  XFillRectangle(display, object_mask[i], color0gc, 0, 0, width, height);
  
  if (!SetUpColormap(display, screen, object_pixmap[i],
		     temp_visual, &colormap))
    {
      fprintf(stderr, "Error: Could not set up colormap!\n");
      exit(1);
    }
  
  
  for (y = 0; y < height; y++)
    {
      for (x = 0; x < width; x++)
	{
	  red = fgetc(fi);
	  green = fgetc(fi);
	  blue = fgetc(fi);
	  
	  if (greymode == 1)
	    {
	      z = (red + green + blue) / 3;
	      red = z;
	      green = z;
	      blue = z;
	    }
	  
	  /* fscanf(fi, "%d %d %d", &red, &green, &blue); */
	  
	  if (red != 255 || green != 255 || blue != 255)
	    {
	      red = (red / 32) * 32 + 15;
	      blue = (blue / 32) * 32 + 15;
	      green = (green / 32) * 32 + 15;
	      
	      color_pos = color_seen(red, green, blue);
	      
	      if (color_pos == -1)
		{
		  xcol.red = red * 256;
		  xcol.green = green * 256;
		  xcol.blue = blue * 256;
		  xcol.flags = DoRed | DoGreen | DoBlue;
		  
		  XAllocColor(display, colormap, &xcol);
		  
		  color_pos = color_add(red, green, blue, 
					xcol.pixel, i);
		  color_want = xcol.pixel;
		  
		  xgcv.foreground = color_want;
		  colorgc[color_pos] = XCreateGC(display, window,
						 GCForeground, &xgcv);
		}
	      else
		{
		  color_want = color_list[color_pos].pixel;
		  
		  if (color_list[color_pos].owner != i)
		    {
		      xgcv.foreground = color_want;
		      
		      XFreeGC(display, colorgc[color_pos]);
		      colorgc[color_pos] =
			XCreateGC(display, object_pixmap[i],
				  GCForeground, &xgcv);
		      
		      color_list[color_pos].owner = i;
		    }
		}
	      
	      XDrawPoint(display, object_pixmap[i], colorgc[color_pos],
			 x, y);
	      XDrawPoint(display, object_mask[i], color1gc, x, y);
	    }
	}
    }
  
  object_gc[i] = CreateGC(white);
  XSetTSOrigin(display, object_gc[i], 0, 0);
  XSetStipple(display, object_gc[i], object_mask[i]);
  
  fclose(fi);
}


unsigned long AllocNamedColor(Display *display, Colormap colormap,
                              char* colorname, unsigned long default_color)
{
  XColor hardwarecolor,exactcolor;
  unsigned long color;
  int status;
  
  status = XAllocNamedColor(display,colormap,colorname,&hardwarecolor,
                          &exactcolor);
  if (status != 0)
    color = hardwarecolor.pixel;
  else
    color = default_color;
  
  return(color);
}
