/* Native support code for HPUX PA-RISC, for GDB the GNU debugger.

   Copyright 2005 Free Software Foundation, Inc.

   Contributed by the Center for Software Science at the
   University of Utah (pa-gdb-bugs@cs.utah.edu).

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */


#include "defs.h"
#include "gdb_string.h"
#include "inferior.h"
#include "regcache.h"
#include "infttrace.h"
#include <sys/ttrace.h>
#include <ia64/sys/uregs.h>
#include "opcode/ia64.h"
#include "gdbcore.h"

static unsigned int u_offsets[] = 
{ /* Static General Registers.  */
  -1,     __r1,   __r2,   __r3,   __r4,   __r5,   __r6,   __r7,
  __r8,   __r9,   __r10,  __r11,  __r12,  __r13,  __r14,  __r15,
  __r16,  __r17,  __r18,  __r19,  __r20,  __r21,  __r22,  __r23,
  __r24,  __r25,  __r26,  __r27,  __r28,  __r29,  __r30,  __r31,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,

  /* Static Floating-Point Registers.  */
  __f0,   __f1,   __f2,   __f3,   __f4,   __f5,   __f6,   __f7,
  __f8,   __f9,   __f10,  __f11,  __f12,  __f13,  __f14,  __f15,
  __f16,  __f17,  __f18,  __f19,  __f20,  __f21,  __f22,  __f23,
  __f24,  __f25,  __f26,  __f27,  __f28,  __f29,  __f30,  __f31,
  __f32,  __f33,  __f34,  __f35,  __f36,  __f37,  __f38,  __f39,
  __f40,  __f41,  __f42,  __f43,  __f44,  __f45,  __f46,  __f47,
  __f48,  __f49,  __f50,  __f51,  __f52,  __f53,  __f54,  __f55,
  __f56,  __f57,  __f58,  __f59,  __f60,  __f61,  __f62,  __f63,
  __f64,  __f65,  __f66,  __f67,  __f68,  __f69,  __f70,  __f71,
  __f72,  __f73,  __f74,  __f75,  __f76,  __f77,  __f78,  __f79,
  __f80,  __f81,  __f82,  __f83,  __f84,  __f85,  __f86,  __f87,
  __f88,  __f89,  __f90,  __f91,  __f92,  __f93,  __f94,  __f95,
  __f96,  __f97,  __f98,  __f99,  __f100, __f101, __f102, __f103,
  __f104, __f105, __f106, __f107, __f108, __f109, __f110, __f111,
  __f112, __f113, __f114, __f115, __f116, __f117, __f118, __f119,
  __f120, __f121, __f122, __f123, __f124, __f125, __f126, __f127,

  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,
  -1,     -1,     -1,     -1,     -1,     -1,     -1,     -1,

  /* Branch Registers.  */
  __b0,   __b1,   __b2,   __b3,   __b4,   __b5,   __b6,   __b7,

  /* Virtual frame pointer and virtual return address pointer.  */
  -1, -1,

  /* Other registers.  */
  __pr, __ip, __cr_ipsr, __cfm,

  /* Kernel registers.  */
  -1,   -1,   -1,   -1,
  -1,   -1,   -1,   -1,

  -1, -1, -1, -1, -1, -1, -1, -1,

  /* Some application registers.  */
  __ar_rsc, __ar_bsp, __ar_bspstore, __ar_rnat,

  -1,
  -1,  /* Not available: FCR, IA32 floating control register.  */
  -1, -1,

  -1,         /* Not available: EFLAG.  */
  -1,         /* Not available: CSD.  */
  -1,         /* Not available: SSD.  */
  -1,         /* Not available: CFLG.  */
  -1,         /* Not available: FSR.  */
  -1,         /* Not available: FIR.  */
  -1,         /* Not available: FDR.  */
  -1,
  __ar_ccv, -1, -1, -1, __ar_unat, -1, -1, -1,
  __ar_fpsr, -1, -1, -1,
  -1,         /* Not available: ITC.  */
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1,
  __ar_pfs, __ar_lc, __ar_ec,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
  -1,

  /* nat bits - not fetched directly; instead we obtain these bits from
     either rnat or unat or from memory.  */
  /* brobecker/2005-04-20: Not implemented yet.  */
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   

  -1,  /* Not available: BOF.  */

  /* Stacked General Registers, r32 - r127, not accessible via ttrace().  */  
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,   
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,
  -1,  -1,  -1,  -1,  -1,  -1,  -1,  -1,

  /* Static Predicate Registers.
     FIXME: brobecker/2005-04-19:  p0 is not accessible.
     It is a read-only register with value always set to 1.
     This explains why the value below is -1, and we need to
     update the fetch/store routine to know about this.  */
    -1,   __p1,   __p2,   __p3,   __p4,   __p5,   __p6,   __p7,
  __p8,   __p9,   __p10,  __p11,  __p12,  __p13,  __p14,  __p15,
  __p16,  __p17,  __p18,  __p19,  __p20,  __p21,  __p22,  __p23,
  __p24,  __p25,  __p26,  __p27,  __p28,  __p29,  __p30,  __p31,

  /* Rotating Predicates Registers.  */
  __p32,  __p33,  __p34,  __p35,  __p36,  __p37,  __p38,  __p39,
  __p40,  __p41,  __p42,  __p43,  __p44,  __p45,  __p46,  __p47,
  __p48,  __p49,  __p50,  __p51,  __p52,  __p53,  __p54,  __p55,
  __p56,  __p57,  __p58,  __p59,  __p60,  __p61,  __p62,  __p63,
};

/* Some register have a fixed value and can not be modified.
   Store their value in static constant buffers that can be used
   later to fill the register cache.  */
static const char r0_value[8] = {0x00, 0x00, 0x00, 0x00,
                                 0x00, 0x00, 0x00, 0x00 };


/*## */
/* Enable HACK for ttrace work.  In
 * infttrace.c/require_notification_of_events,
 * this is set to 0 so that the loop in child_wait
 * won't loop.
 */
int not_same_real_pid = 1;
/*## */

static CORE_ADDR text_end;

void
deprecated_hpux_text_end (struct target_ops *exec_ops)
{
  struct section_table *p;

  /* Set text_end to the highest address of the end of any readonly
     code section.  */
  /* FIXME: The comment above does not match the code.  The code
     checks for sections with are either code *or* readonly.  */
  text_end = (CORE_ADDR) 0;
  for (p = exec_ops->to_sections; p < exec_ops->to_sections_end; p++)
    if (bfd_get_section_flags (p->bfd, p->the_bfd_section)
	& (SEC_CODE | SEC_READONLY))
      {
	if (text_end < p->endaddr)
	  text_end = p->endaddr;
      }
}

/* Fetch a register's value from the process's U area.  */
static void
fetch_register (int regno)
{
  unsigned int addr;
  uint64_t len;
  char buf[MAX_REGISTER_SIZE];
  int tt_status;

  if (regno == IA64_GR0_REGNUM)
    {
      /* r0 is always 0.  */
      regcache_raw_supply (current_regcache, regno, r0_value);
      return;
    }

  if (regno == IA64_RNAT_REGNUM)
    {
      /* FIXME: Fetching this register is a bit more difficult than
         just asking ttrace.  Take care of it later, leave it as zero
         for now.  */
      return;
    }

  /* Get the register location. If the register can not be fetched,
     then return now.  */
  addr = u_offsets[regno];
  if (addr == -1)
    return;

  len = register_size (current_gdbarch, regno);
  
  tt_status =
    read_from_register_save_state (PIDGET (inferior_ptid), addr, buf, len); 
  
  /* FIXME: The following should be an error rather than a warning.
     But I haven't managed to understand why I can't get the value
     of the PSR register yet.  For some reason, this register is not
     documented the book I'm using as an introduction to IA64.  So
     maybe this is a register that's not accessible?  */
  if (tt_status < 0)
    internal_warning (__FILE__, __LINE__, 
                      "Failed to read register value for %s.\n",
                      REGISTER_NAME (regno));

  regcache_raw_supply (current_regcache, regno, buf);
}

void
fetch_inferior_registers (int regno)
{
  if (regno == -1)
    for (regno = 0; regno < NUM_REGS; regno++)
      fetch_register (regno);
  else
    fetch_register (regno);
}

/* Our own version of the offsetof macro, since we can't assume ANSI C.  */
#define HPPAH_OFFSETOF(type, member) ((int) (&((type *) 0)->member))

void
store_register (int regno)
{
  uint64_t addr = u_offsets[regno];
  uint64_t len;
  char buf[MAX_REGISTER_SIZE];
  int tt_status;

  /* If the register can not be stored, then return now.  */
  if (addr == -1)
    return;

  /* FIXME: brobecker/2005-06-10: I don't know how to store that
     register for now.  So just ignore any request to store it, to
     avoid an internal error.  */
  if (regno == IA64_PSR_REGNUM)
    return;

  len = register_size (current_gdbarch, regno);
  regcache_raw_collect (current_regcache, regno, buf);
  
  tt_status =
    write_to_register_save_state (PIDGET (inferior_ptid), addr, buf, len); 

  if (tt_status < 0)
    internal_error (__FILE__, __LINE__, 
                    "Failed to write register value for %s.\n",
                    REGISTER_NAME (regno));
}

/* Store our register values back into the inferior.
   If REGNO is -1, do this for all registers.
   Otherwise, REGNO specifies which register (so we can save time).  */

void
store_inferior_registers (int regno)
{
  if (regno < 0)
    for (regno = 0; regno < NUM_REGS; regno++)
      store_register (regno);
  else
    store_register (regno);
}


/* Format a thread id, given TID.  Be sure to terminate
   this with a null--it's going to be printed via a "%s".

   Note: This is a core-gdb tid, not the actual system tid.
   See infttrace.c for details.  */

/* FIXME: brobecker/2005-04-15: A better name for this function would
   probably be hpux_tid_to_str, since this function is more specific
   to the operating system than to the processor.  It cannot be moved
   and inlined inside infttrace.c because it can also be used in the
   target_tid_to_str macro.  */

char *
hppa_tid_to_str (ptid_t ptid)
{
  /* Static because address returned */
  static char buf[30];
  /* This seems strange, but when I did the ptid conversion, it looked
     as though a pid was always being passed.  - Kevin Buettner  */
  pid_t tid = PIDGET (ptid);

  /* Extra NULLs for paranoia's sake */
  sprintf (buf, "system thread %d%c", tid, '\0');

  return buf;
}

/* Read LEN bytes of inferior memory at MEMADDR into BUFFER.
   return -1 if the read failed.  */

static int
ia64_hpux_read_memory (CORE_ADDR memaddr, char *buffer, int len)
{
  int tt_status;
  
  tt_status = call_ttrace (TT_PROC_RDDATA, PIDGET (inferior_ptid),
                           (TTRACE_ARG_TYPE) memaddr, (TTRACE_ARG_TYPE) len,
                           (TTRACE_ARG_TYPE) buffer);

  if (tt_status < 0)
    tt_status = call_ttrace (TT_PROC_RDTEXT, PIDGET (inferior_ptid),
                             (TTRACE_ARG_TYPE) memaddr,
                             (TTRACE_ARG_TYPE) len,
                             (TTRACE_ARG_TYPE) buffer);

  return tt_status;
}

/* Write LEN bytes from BUFFER into inferior memory at MEMADDR.
   return -1 if the write failed.  */

static int
ia64_hpux_write_memory (CORE_ADDR memaddr, char *buffer, int len)
{
  /* brobecker/2005-06-10: Apparently, the memory writes need to be aligned
     on 16-bytes boundaries, at least when writing in the text section, but
     the buffer size doesn't need to be a multiple of 16bytes.  */
  const CORE_ADDR buf_addr = memaddr & ~0x0f;
  const int buf_len = len + (memaddr - buf_addr);
  char *buf;
  int tt_status;

  buf = (char *) alloca (buf_len * sizeof (char));
  
  tt_status = ia64_hpux_read_memory (buf_addr, buf, buf_len);
  if (tt_status < 0)
    return 0;

  memcpy (buf + (memaddr - buf_addr), buffer, len);
  
  tt_status = call_ttrace (TT_PROC_WRDATA, PIDGET (inferior_ptid),
                           (TTRACE_ARG_TYPE) memaddr, (TTRACE_ARG_TYPE) len,
                           (TTRACE_ARG_TYPE) buffer);
  if (tt_status < 0)
    tt_status = call_ttrace (TT_PROC_WRTEXT, PIDGET (inferior_ptid),
                             (TTRACE_ARG_TYPE) memaddr, 
                             (TTRACE_ARG_TYPE) len,
                             (TTRACE_ARG_TYPE) buffer);

  return tt_status;
}

/* Copy LEN bytes to or from inferior's memory starting at MEMADDR
   to debugger memory starting at MYADDR.   Copy to inferior if
   WRITE is nonzero.

   Returns the length copied, which is either the LEN argument or
   zero.  This xfer function does not do partial moves, since
   deprecated_child_ops doesn't allow memory operations to cross below
   us in the target stack anyway.  TARGET is ignored.  */

int
child_xfer_memory (CORE_ADDR memaddr, char *myaddr, int len, int write,
		   struct mem_attrib *mem,
		   struct target_ops *target)
{
  int status;

  if (write)
    status = ia64_hpux_write_memory (memaddr, myaddr, len);
  else
    status = ia64_hpux_read_memory (memaddr, myaddr, len);

  if (status < 0)
    return 0;

  return len;
}

char *saved_child_execd_pathname = NULL;
int saved_vfork_pid;
enum {
  STATE_NONE,
  STATE_GOT_CHILD,
  STATE_GOT_EXEC,
  STATE_GOT_PARENT,
  STATE_FAKE_EXEC
} saved_vfork_state = STATE_NONE;

/* Return non-zero if the instruction at the current PC is a breakpoint
   part of the dynamic loading process.

   We identify such instructions by checking that the instruction at
   the current pc is a break insn where no software breakpoint has been
   inserted by us.  */

int
dld_breakpoint (void)
{
  /* FIXME: brobecker/2005-06-08: The following is almost a copy/paste
     of the first half of ia64-dis.c:print_insn_ia64().  Can we extract
     that part off print_insn_ia64(), and then share the extracted
     function?  This is good enough for now.  */
  ia64_insn t0, t1, slot[3], template, insn;
  int slotnum;
  bfd_byte bundle[16];
  CORE_ADDR pc = read_pc ();

  /* If this is a regular breakpoint, then it can not be a dld one.  */
  if (breakpoint_inserted_here_p (pc))
    return 0;

  slotnum = ((long) pc) & 0xf;
  if (slotnum > 2)
    internal_error (__FILE__, __LINE__,
                    "invalid slot (%d) for address 0x%lx", slotnum, pc);

  pc -= (pc & 0xf);
  read_memory (pc, bundle, sizeof (bundle));

  /* bundles are always in little-endian byte order */
  t0 = bfd_getl64 (bundle);
  t1 = bfd_getl64 (bundle + 8);
  template = (t0 >> 1) & 0xf;
  slot[0] = (t0 >>  5) & 0x1ffffffffffLL;
  slot[1] = ((t0 >> 46) & 0x3ffff) | ((t1 & 0x7fffff) << 18);
  slot[2] = (t1 >> 23) & 0x1ffffffffffLL;

  if (template == 2 && slotnum == 1)
    {
      /* skip L slot in MLI template: */
      slotnum = 2;
    }

  insn = slot[slotnum];

  return ((insn & 0x01eff8000000) == 0);
}

/* Wait for child to do something.  Return pid of child, or -1 in case
   of error; store status through argument pointer OURSTATUS.  */

ptid_t
child_wait (ptid_t ptid, struct target_waitstatus *ourstatus)
{
  int save_errno;
  int status;
  char *execd_pathname = NULL;
  int exit_status;
  int related_pid;
  int syscall_id;
  enum target_waitkind kind;
  int pid;

  if (saved_vfork_state == STATE_FAKE_EXEC)
    {
      saved_vfork_state = STATE_NONE;
      ourstatus->kind = TARGET_WAITKIND_EXECD;
      ourstatus->value.execd_pathname = saved_child_execd_pathname;
      return inferior_ptid;
    }

  do
    {
      set_sigint_trap ();	/* Causes SIGINT to be passed on to the
				   attached process. */
      set_sigio_trap ();

      pid = ptrace_wait (inferior_ptid, &status);

      save_errno = errno;

      clear_sigio_trap ();

      clear_sigint_trap ();

      if (pid == -1)
	{
	  if (save_errno == EINTR)
	    continue;

	  fprintf_unfiltered (gdb_stderr, "Child process unexpectedly missing: %s.\n",
			      safe_strerror (save_errno));

	  /* Claim it exited with unknown signal.  */
	  ourstatus->kind = TARGET_WAITKIND_SIGNALLED;
	  ourstatus->value.sig = TARGET_SIGNAL_UNKNOWN;
	  return pid_to_ptid (-1);
	}

      /* Did it exit?
       */
      if (target_has_exited (pid, status, &exit_status))
	{
	  /* ??rehrauer: For now, ignore this. */
	  continue;
	}

      if (!target_thread_alive (pid_to_ptid (pid)))
	{
	  ourstatus->kind = TARGET_WAITKIND_SPURIOUS;
	  return pid_to_ptid (pid);
	}

      if (hpux_has_forked (pid, &related_pid))
	{
	  /* Ignore the parent's fork event.  */
	  if (pid == PIDGET (inferior_ptid))
	    {
	      ourstatus->kind = TARGET_WAITKIND_IGNORE;
	      return inferior_ptid;
	    }

	  /* If this is the child's fork event, report that the
	     process has forked.  */
	  if (related_pid == PIDGET (inferior_ptid))
	    {
	      ourstatus->kind = TARGET_WAITKIND_FORKED;
	      ourstatus->value.related_pid = pid;
	      return inferior_ptid;
	    }
	}

      if (hpux_has_vforked (pid, &related_pid))
	{
	  if (pid == PIDGET (inferior_ptid))
	    {
	      if (saved_vfork_state == STATE_GOT_CHILD)
		saved_vfork_state = STATE_GOT_PARENT;
	      else if (saved_vfork_state == STATE_GOT_EXEC)
		saved_vfork_state = STATE_FAKE_EXEC;
	      else
		fprintf_unfiltered (gdb_stdout,
				    "hppah: parent vfork: confused\n");
	    }
	  else if (related_pid == PIDGET (inferior_ptid))
	    {
	      if (saved_vfork_state == STATE_NONE)
		saved_vfork_state = STATE_GOT_CHILD;
	      else
		fprintf_unfiltered (gdb_stdout,
				    "hppah: child vfork: confused\n");
	    }
	  else
	    fprintf_unfiltered (gdb_stdout,
				"hppah: unknown vfork: confused\n");

	  if (saved_vfork_state == STATE_GOT_CHILD)
	    {
	      child_post_startup_inferior (pid_to_ptid (pid));
	      detach_breakpoints (pid);
#ifdef SOLIB_REMOVE_INFERIOR_HOOK
	      SOLIB_REMOVE_INFERIOR_HOOK (pid);
#endif
	      child_resume (pid_to_ptid (pid), 0, TARGET_SIGNAL_0);
	      ourstatus->kind = TARGET_WAITKIND_IGNORE;
	      return pid_to_ptid (related_pid);
	    }
	  else if (saved_vfork_state == STATE_FAKE_EXEC)
	    {
	      ourstatus->kind = TARGET_WAITKIND_VFORKED;
	      ourstatus->value.related_pid = related_pid;
	      return pid_to_ptid (pid);
	    }
	  else
	    {
	      /* We saw the parent's vfork, but we haven't seen the exec yet.
		 Wait for it, for simplicity's sake.  It should be pending.  */
	      saved_vfork_pid = related_pid;
	      ourstatus->kind = TARGET_WAITKIND_IGNORE;
	      return pid_to_ptid (pid);
	    }
	}

      if (hpux_has_execd (pid, &execd_pathname))
	{
	  /* On HP-UX, events associated with a vforking inferior come in
	     threes: a vfork event for the child (always first), followed
	     a vfork event for the parent and an exec event for the child.
	     The latter two can come in either order.  Make sure we get
	     both.  */
	  if (saved_vfork_state != STATE_NONE)
	    {
	      if (saved_vfork_state == STATE_GOT_CHILD)
		{
		  saved_vfork_state = STATE_GOT_EXEC;
		  /* On HP/UX with ptrace, the child must be resumed before
		     the parent vfork event is delivered.  A single-step
		     suffices.  */
		  if (RESUME_EXECD_VFORKING_CHILD_TO_GET_PARENT_VFORK ())
		    target_resume (pid_to_ptid (pid), 1, TARGET_SIGNAL_0);
		  ourstatus->kind = TARGET_WAITKIND_IGNORE;
		}
	      else if (saved_vfork_state == STATE_GOT_PARENT)
		{
		  saved_vfork_state = STATE_FAKE_EXEC;
		  ourstatus->kind = TARGET_WAITKIND_VFORKED;
		  ourstatus->value.related_pid = saved_vfork_pid;
		}
	      else
		fprintf_unfiltered (gdb_stdout,
				    "hppa: exec: unexpected state\n");

	      saved_child_execd_pathname = execd_pathname;

	      return inferior_ptid;
	    }
	  
	  /* Are we ignoring initial exec events?  (This is likely because
	     we're in the process of starting up the inferior, and another
	     (older) mechanism handles those.)  If so, we'll report this
	     as a regular stop, not an exec.
	   */
	  if (inferior_ignoring_startup_exec_events)
	    {
	      inferior_ignoring_startup_exec_events--;
	    }
	  else
	    {
	      ourstatus->kind = TARGET_WAITKIND_EXECD;
	      ourstatus->value.execd_pathname = execd_pathname;
	      return pid_to_ptid (pid);
	    }
	}

      /* All we must do with these is communicate their occurrence
         to wait_for_inferior...
       */
      if (hpux_has_syscall_event (pid, &kind, &syscall_id))
	{
	  ourstatus->kind = kind;
	  ourstatus->value.syscall_id = syscall_id;
	  return pid_to_ptid (pid);
	}

      if (dld_breakpoint ())
        {
          /* FIXME: I think this part will eventually go to the
             file where solib support is implemented.  But for now,
             we handle it here.  We don't support shared libraries
             auto-loading at the moment, so just move the PC to the
             next instruction bundle, and continue.  */
          CORE_ADDR pc = read_pc ();

          pc -= pc & 0xf;
          pc += 16;
          write_pc (pc);

          target_resume (pid_to_ptid (pid), 0, TARGET_SIGNAL_0);
          ourstatus->kind = TARGET_WAITKIND_IGNORE;
          return pid_to_ptid (pid);
        }

      /*##  } while (pid != PIDGET (inferior_ptid)); ## *//* Some other child died or stopped */
/* hack for thread testing */
    }
  while ((pid != PIDGET (inferior_ptid)) && not_same_real_pid);
/*## */

  store_waitstatus (ourstatus, status);
  return pid_to_ptid (pid);
}
