/*
  gdbm-clear.c
  
  Jonathan B. Leffert <jonathan@leffert.net>
  24 April 2000
  
  $Id: gdbm-clear.c,v 1.16 2000/05/15 21:37:37 jonathan Exp $
  
  Copyright (c) 2000 Jonathan B. Leffert <jonathan@leffert.net>
  All rights reserved.
*/

#include "common.h"

/* prototypes */
void parse_args(int *, char ***, char **);
void usage(char *);

/* function defs */

void parse_args(int *argc, char ***argv, char **dbName)
{
  char *options = "d:hv";
  int c;

  while ( (c = getopt(*argc, *argv, options)) != EOF )
    {
      switch (c)
	{
	case 'd':
	  /* database name */
	  *dbName = optarg;
	  break;
	case 'h':
	  /* request help */
	  usage((*argv)[0]);
	  exit(0);
	  break;
	case 'v':
	  /* get version info */
	  printf("Version %f\n", VERSION);
	  exit(0);
	  break;
	case '?':
	  /* unknown option */
	  usage((*argv)[0]);
	  exit(1);
	  break;
	default:
	  /* default: do nothing */
	  break;
	}
    }
}

void usage(char *program_name)
{
  printf("Usage:\n");
  printf("\n");
  printf("%s [-hv] [-d DB]\n", program_name);
  printf("\n");
  printf("-d DB     = Use database with specified name.\n");
  printf("            Must be supplied.\n");
  printf("-h        = Prints this help message.\n");
  printf("-v        = Prints version information.\n");
  printf("\n");
  printf("This program clears the given database of all its contents.\n");
  printf("\n");
  printf("Error Codes:\n");
  printf("  0 on success\n");
  printf("  1 on non-database failure\n");
  printf("  2 on database failure\n");
}

int main(int argc, char **argv)
{
  char *name = NULL;
  GDBM_FILE db = NULL;
  datum key;

  parse_args(&argc, &argv, &name);

  if ( name == NULL )
    {
      /* somebody was bad and didn't specify the -d flag */
      usage(argv[0]);
      exit(1);
    }

  /* open the database read-write */
  db = open_gdbm_database_rw(name);
  
  if ( db != NULL )
    {
      /* open succeeded: now process key(s) */
      
      key = gdbm_firstkey(db);
      while ( key.dptr != NULL )
	{
	  gdbm_delete(db, key);

	  /* free memory */
	  free(key.dptr);
	  key.dptr = NULL;
	  key.dsize = 0;

	  key = gdbm_firstkey(db);
	}
      
      /* close the database */
      gdbm_close(db);
    }
  else
    {
      /* open failed */
      perror(argv[0]);
      exit(2);
    }

  return 0;
}
