/*
  gdbm-load.c
  
  Jonathan B. Leffert <jonathan@leffert.net>
  28 April 2000
  
  $Id: gdbm-load.c,v 1.10 2000/05/15 21:37:37 jonathan Exp $
  
  Copyright (c) 2000 Jonathan B. Leffert <jonathan@leffert.net>
  All rights reserved.
*/

#include "common.h"
#include "fileio.h"
#include "parse.h"

/* prototypes */
void parse_args(int *, char ***, char **, char **, char **);
void usage(char *);

/* function defs */

void parse_args(int *argc, char ***argv, char **dbName,
		char **sep, char **filename)
{
  char *options = "d:f:hs:v";
  int c;

  while ( (c = getopt(*argc, *argv, options)) != EOF )
    {
      switch (c)
	{
	case 'd':
	  /* database name */
	  *dbName = optarg;
	  break;
	case 'f':
	  /* filename name */
	  *filename = optarg;
	  break;
	case 'h':
	  /* request help */
	  usage((*argv)[0]);
	  exit(0);
	  break;
	case 's':
	  /* separator set */
	  *sep = optarg;
	  break;
	case 'v':
	  /* version info */
	  printf("Version %f\n", VERSION);
	  exit(0);
	  break;
	case '?':
	  /* unknown option */
	  usage((*argv)[0]);
	  exit(1);
	  break;
	default:
	  /* default: do nothing */
	  break;
	}
    }
}

void usage(char *program_name)
{
  printf("Usage:\n");
  printf("\n");
  printf("%s [-hv] [-d DB] [-f FILE] [-s SEP]\n", program_name);
  printf("\n");
  printf("-d DB   = Use database with specified name.\n");
  printf("          Must be supplied.\n");
  printf("-f FILE = Load key-value pairs from specified file.\n");
  printf("          If this is not supplied, STDIN is used as input.\n");
  printf("-h      = This help message.\n");
  printf("-s SEP  = Field separator for FILE.  A string that is\n");
  printf("          passed to strtok(3).  Defaults to '\\t'.\n");
  printf("-v      = Prints version information.\n");
  printf("\n");
  printf("This program loads the contents of the given file into the\n");
  printf("given database.  Keys and values are separated by the SEP token\n");
  printf("and key-value pairs are separated by newlines.  All inserts into\n");
  printf("the database are destructive (i.e., they overwrite any\n");
  printf("current values).\n");
  printf("\n");
  printf("Error Codes:\n");
  printf("  0 on success\n");
  printf("  1 on non-database failure\n");
  printf("  2 on database failure\n");
}

int main(int argc, char **argv)
{
  char *dbname = NULL;
  char *filename = NULL;
  char *sep = "\t";
  GDBM_FILE db = NULL;
  FILE *datafile = NULL;
  datum key;
  datum value;
  char *line = NULL;
  char *key_str = NULL;
  char *val_str = NULL;
  int i;

  parse_args(&argc, &argv, &dbname, &sep, &filename);
  
  if ( (dbname == NULL) || (sep == NULL) )
    {
      usage(argv[0]);
      exit(1);
    }

  /* open database rw */
  db = open_gdbm_database_rw(dbname);
  
  if ( db != NULL )
    {
      /* database open succeeded */
      
      /* open the datafile */
      if ( filename != NULL )
	{
	  /* file is a real file */
	  datafile = fopen(filename, "r");
	}
      else
	{
	  /* file is stdin */
	  datafile = stdin;
	}
	  

      if ( datafile != NULL )
	{
	  /* datafile open succeeded */
      
	  while ( (line = readline(datafile)) != NULL )
	    {
	      /* grab the key */
	      key_str = strtok(line,sep);
	      key_str = unescape_string(key_str);
	      /* grab the value: the rest of the line */
	      val_str = line + strlen(key_str) + strlen(sep);
	      val_str = unescape_string(val_str);

	      if ( (key_str != NULL) && (val_str != NULL) )
		{
		  /* build the key-value pair */
		  key.dptr = key_str;
		  key.dsize = (strlen(key_str) + 1) * sizeof(char);
		  value.dptr = val_str;
		  value.dsize = (strlen(val_str) + 1) * sizeof(char);

		  /* store the pair */
		  i = gdbm_store(db, key, value, GDBM_REPLACE);
		  if ( i == 0 )
		    {
		      /* success: do nothing */
		    }
		  else if ( i == -1 )
		    {
		      /* db is read-only */
		      perror(argv[0]);
		    }
		  else if ( i == 1 )
		    {
		      /* key collision (should never happen because
			 gdbm_store(3) is called with GDBM_REPLACE) */
		      perror(argv[0]);
		    }
		  /* free allocated memory */
		  free(key_str);
		  key_str = NULL;
		  free(val_str);
		  val_str = NULL;
		  
		}
	      
	      /* free the allocated memory */
	      free(line);
	      line = NULL;
	    }
	  
	  /* close the datafile */
	  fclose(datafile);
	}
      else
	{
	  /* error occurred */
	  perror(argv[0]);
	  /* close the database */
	  gdbm_close(db);
	  exit(1);
	}
	  
      /* close the database */
      gdbm_close(db);
    }
  else
    {
      /* error occurred */
      perror(argv[0]);
      exit(2);
    }
  
  return 0;
}
