/*
  gdbm-revget.c
  
  Jonathan B. Leffert <jonathan@leffert.net>
  24 April 2000
  
  $Id: gdbm-revget.c,v 1.20 2000/05/15 21:37:37 jonathan Exp $
  
  Copyright (c) 2000 Jonathan B. Leffert <jonathan@leffert.net>
  All rights reserved.
*/

#include "common.h"

/* prototypes */
void parse_args(int *, char ***, char **, char **);
void usage(char *);

/* function defs */

void parse_args(int *argc, char ***argv, char **dbName, char **format)
{
  char *options = "d:f:hv";
  int c;

  while ( (c = getopt(*argc, *argv, options)) != EOF )
    {
      switch (c)
	{
	case 'd':
	  /* database name */
	  *dbName = optarg;
	  break;
	case 'f':
	  /* format string */
	  *format = optarg;
	  break;
	case 'h':
	  /* request help */
	  usage((*argv)[0]);
	  exit(0);
	  break;
	case 'v':
	  /* version info */
	  printf("Version %f\n", VERSION);
	  exit(0);
	  break;
	case '?':
	  /* unknown option */
	  usage((*argv)[0]);
	  exit(1);
	  break;
	default:
	  /* default: do nothing */
	  break;
	}
    }
}

void usage(char *program_name)
{
  printf("Usage:\n");
  printf("\n");
  printf("%s [-hv] [-d DB] [-f FORMAT] value(s)\n", program_name);
  printf("\n");
  printf("-d DB     = Use database with specified name.\n");
  printf("            Must be supplied.\n");
  printf("-f FORMAT = Format string for printf of key.\n");
  printf("            Default is \"%%s\"");
  printf(" (trailing newline is implicit).\n");
  printf("-h        = Prints this help message.\n");
  printf("-v        = Prints version information.\n");
  printf("\n");
  printf("This program performs a reverse lookup on the given database.\n");
  printf("All keys with the given values are printed.  If no values\n");
  printf("are given or if an invalid value is given, 1 is returned.\n");
  printf("\n");
  printf("Error Codes:\n");
  printf("  0 on success\n");
  printf("  1 on non-database failure\n");
  printf("  2 on database failure\n");
  printf("  139 on segmentation fault.  Usually happens if -f string\n");
  printf("      is not valid.\n");
	 
}

int main(int argc, char **argv)
{
  char *name = NULL;
  char *format = "%s";
  GDBM_FILE db = NULL;
  datum key;
  datum oldkey;
  datum value;
  int i;
  int returnValue = 0;

  parse_args(&argc, &argv, &name, &format);

  if ( name == NULL )
    {
      /* somebody was bad and didn't specify the -d flag */
      usage(argv[0]);
      exit(1);
    }

  /* open the database read-only */
  db = open_gdbm_database_ro(name);
  
  if ( db != NULL )
    {
      /* open succeeded: now process key(s) */
      if ( optind == argc )
	{
	  returnValue = 1;
	}
      else
	{
	  for ( i = optind; i < argc; i++ )
	    {
	      key = gdbm_firstkey(db);
	      while ( key.dptr != NULL )
		{
		  value = gdbm_fetch(db, key);
		  
		  if ( strcmp(value.dptr, argv[i]) == 0 )
		    {
		      /* print the key and value */
		      printf(format, key.dptr);
		      printf("\n");
		    }
		  
		  oldkey = key;
		  key = gdbm_nextkey(db,key);
		  
		  /* free memory */
		  free(oldkey.dptr);
		  oldkey.dptr = NULL;
		  oldkey.dsize = 0;
		  free(value.dptr);
		  value.dptr = NULL;
		  value.dsize = 0;
		}
	    }
	}
      
      /* close the database */
      gdbm_close(db);
    }
  else
    {
      /* open failed */
      perror(argv[0]);
      exit(2);
    }

  return returnValue;
}
