#include "hre.h"
#include "hre_core.h"
#include <stdio.h>
#include <string.h>

#define REG_BASIC       0000
#define REG_EXTENDED    0001
#define REG_ICASE       0002
#define REG_NOSUB       0004
#define REG_NEWLINE     0010
#define REG_NOSPEC      0020
#define REG_PEND        0040
#define REG_DUMP        0200

#define REG_NOTBOL      00001
#define REG_NOTEOL      00002
#define REG_STARTEND    00004
#define REG_TRACE       00400   /* tracing of execution */
#define REG_LARGE       01000   /* force large representation */
#define REG_BACKR       02000   /* force use of backref code */

int number = 1;
int line = 0;
int copts = REG_EXTENDED;
int eopts = 0;

static struct hre_err {
  int code;
  char *name;
  char *explain;
} hre_errs[] = {
  { REG_NOMATCH,  "REG_NOMATCH",  "hre_search() failed to match" },
  { REG_BADPAT,   "REG_BADPAT",   "invalid regular expression" },
  { REG_ECOLLATE, "REG_ECOLLATE", "invalid collating element" },
  { REG_ECTYPE,   "REG_ECTYPE",   "invalid character class" },
  { REG_EESCAPE,  "REG_EESCAPE",  "trailing backslash (\\)" },
  { REG_ESUBREG,  "REG_ESUBREG",  "invalid backreference number" },
  { REG_EBRACK,   "REG_EBRACK",   "brackets ([ ]) not balanced" },
  { REG_EPAREN,   "REG_EPAREN",   "parentheses not balanced" },
  { REG_EBRACE,   "REG_EBRACE",   "braces not balanced" },
  { REG_BADBR,    "REG_BADBR",    "invalid repetition count(s)" },
  { REG_ERANGE,   "REG_ERANGE",   "invalid character range" },
  { REG_ESPACE,   "REG_ESPACE",   "out of memory" },
  { REG_BADRPT,   "REG_BADRPT",   "repetition-operator operand invalid" },
  { REG_EMPTY,    "REG_EMPTY",    "empty (sub)expression" },
  { REG_ASSERT,   "REG_ASSERT",   "\"can't happen\" -- you found a bug" },
  { REG_INVARG,   "REG_INVARG",   "invalid argument to regex routine" },
  { 0,            "",             "*** unknown regexp error code ***" }
};

/*
 - split - divide a string into fields, like awk split()
 = int split(char *string, char *fields[], int nfields, char *sep);
 */
int                             /* number of fields, including overflow */
split(string, fields, nfields, sep)
     char *string;
     char *fields[];                 /* list is not NULL-terminated */
     int nfields;                    /* number of entries available in fields[] */
     char *sep;                      /* "" white, "c" single char, "ab" [ab]+ */
{
  register char *p = string;
  register char c;                        /* latest character */
  register char sepc = sep[0];
  register char sepc2;
  register int fn;
  register char **fp = fields;
  register char *sepp;
  register int trimtrail;

  /* white space */
  if (sepc == '\0')
    {
      while ((c = *p++) == ' ' || c == '\t')
        continue;
      p--;
      trimtrail = 1;
      sep = " \t";    /* note, code below knows this is 2 long */
      sepc = ' ';
    }
  else
    trimtrail = 0;
  sepc2 = sep[1];         /* now we can safely pick this up */

  /* catch empties */
  if (*p == '\0')
    return(0);

  /* single separator */
  if (sepc2 == '\0')
    {
      fn = nfields;
      for (;;)
        {
          *fp++ = p;
          fn--;
          if (fn == 0)
            break;
          while ((c = *p++) != sepc)
            if (c == '\0')
              return(nfields - fn);
          *(p-1) = '\0';
        }
      /* we have overflowed the fields vector -- just count them */
      fn = nfields;
      for (;;)
        {
          while ((c = *p++) != sepc)
            if (c == '\0')
              return(fn);
          fn++;
        }
      /* not reached */
    }

  /* two separators */
  if (sep[2] == '\0')
    {
      fn = nfields;
      for (;;)
        {
          *fp++ = p;
          fn--;
          while ((c = *p++) != sepc && c != sepc2)
            if (c == '\0')
              {
                if (trimtrail && **(fp-1) == '\0')
                  fn++;
                return(nfields - fn);
              }
          if (fn == 0)
            break;
          *(p-1) = '\0';
          while ((c = *p++) == sepc || c == sepc2)
            continue;
          p--;
        }
      /* we have overflowed the fields vector -- just count them */
      fn = nfields;
      while (c != '\0')
        {
          while ((c = *p++) == sepc || c == sepc2)
            continue;
          p--;
          fn++;
          while ((c = *p++) != '\0' && c != sepc && c != sepc2)
            continue;
        }
      /* might have to trim trailing white space */
      if (trimtrail)
        {
          p--;
          while ((c = *--p) == sepc || c == sepc2)
            continue;
          p++;
          if (*p != '\0')
            {
              if (fn == nfields+1)
                *p = '\0';
              fn--;
            }
        }
      return(fn);
    }

  /* n separators */
  fn = 0;
  for (;;)
    {
      if (fn < nfields)
        *fp++ = p;
      fn++;
      for (;;)
        {
          c = *p++;
          if (c == '\0')
            return(fn);
          sepp = sep;
          while ((sepc = *sepp++) != '\0' && sepc != c)
            continue;
          if (sepc != '\0')       /* it was a separator */
            break;
        }
      if (fn < nfields)
        *(p-1) = '\0';
      for (;;) {
        c = *p++;
        sepp = sep;
        while ((sepc = *sepp++) != '\0' && sepc != c)
          continue;
        if (sepc == '\0')       /* it wasn't a separator */
          break;
      }
      p--;
    }

  /* not reached */
}

/*
 - options - pick options out of a regression-test string
 == int options(int type, char *s);
 */
int
options(type, s)
     int type;                       /* 'c' compile, 'e' exec */
     char *s;
{
  register char *p;
  register int o = (type == 'c') ? copts : eopts;
  register char *legal = (type == 'c') ? "bisnmp" : "^$#tl";

  for (p = s; *p != '\0'; p++)
    if (strchr(legal, *p) != NULL)
      switch (*p)
        {
        case 'b':
          o &= ~REG_EXTENDED;
          break;
        case 'i':
          o |= REG_ICASE;
          break;
        case 's':
          o |= REG_NOSUB;
          break;
        case 'n':
          o |= REG_NEWLINE;
          break;
        case 'm':
          o &= ~REG_EXTENDED;
          o |= REG_NOSPEC;
          break;
        case 'p':
          o |= REG_PEND;
          break;
        case '^':
          o |= REG_NOTBOL;
          break;
        case '$':
          o |= REG_NOTEOL;
          break;
        case '#':
          o |= REG_STARTEND;
          break;
        case 't':       /* trace */
          o |= REG_TRACE;
          break;
        case 'l':       /* force long representation */
          o |= REG_LARGE;
          break;
        case 'r':       /* force backref use */
          o |= REG_BACKR;
          break;
        }
  return(o);
}

/*
 - opt - is a particular option in a regression string?
 == int opt(int c, char *s);
 */
int                             /* predicate */
opt(c, s)
     int c;
     char *s;
{
  return(strchr(s, c) != NULL);
}

/*
 - efind - convert error name to number
 == static int efind(char *name);
 */
int
efind(name)
     char *name;
{
  static char efbuf[100];
  register struct hre_err *r;

  sprintf(efbuf, "REG_%s", name);

  for (r = hre_errs; r->code != 0; r++)
    {
      if (!strcmp (r->name, efbuf))
	break;
    }

  if (r)
    return r->code;
  return 0;
}

void
try(f0, f1, f2, f3, f4, opts)
     char *f0;
     char *f1;
     char *f2;
     char *f3;
     char *f4;
     int opts;                       /* may not match f1 */
{
  int p, i;
  hre_t test;
  hre_option_t option;
  char *type = (opts & REG_EXTENDED) ? "ERE" : "BRE";

  fprintf (stderr, "Testing %3d F:%2d RE: %-20s", number++, opts, f0);
  fprintf (stderr, " STR: %-30s", f2);

#if 0
  fprintf (stderr, "%s\n", f1);
  fprintf (stderr, "%s\n", f3);
  fprintf (stderr, "%s\n", f4);
  fprintf (stderr, "%d\n", opts);
#endif

  HRE_OPTIONS_MODE (&option) = HRE_MODE_ERE;

  test = hre_compile (f0, &option);

  if (HDFA_ERRNO (test) != 0 && (!opt('C', f1) || HDFA_ERRNO (test) != efind(f2)))
    {
      /* unexpected error or wrong error */
      fprintf (stderr, "ERROR NUMBER = %d\n", HDFA_ERRNO (test));
      hre_free (test);
    }
  else if (HDFA_ERRNO (test) == 0 && opt('C', f1))
    {
      /* unexpected success */
      fprintf(stderr, "%d: %s should have given REG_%s\n",
	      line, type, f2);
    }
  else
    {
      if (HDFA_ERRNO (test) != 0 && opt('C', f1))
	{
	  if (HDFA_ERRNO (test) == efind(f2))
	    fprintf (stderr, "   OK\n");
	  else
	    fprintf (stderr, " ERROR (Line: %d) [%d/%d]\n", line, i, p);
	}
      else
	{
	  i = hre_search (test, f2);
	  hre_free (test);
	  
	  p = atoi (f3);
	  if (i != p)
	    fprintf (stderr, " ERROR (Line: %d) [%d/%d]\n", line, i, p);
	  else
	    fprintf (stderr, "%2d OK\n", p);
	}
    }
}

#define MAXF    10

int
main (int argc, char *argv[])
{
  char inbuf[1000];
  FILE *in = stdin;
  int nf;
  char *f[MAXF];
  int i;

  while (fgets(inbuf, sizeof(inbuf), in) != NULL)
    {
      line++;
      if (inbuf[0] == '#' || inbuf[0] == '\n')
        continue;                       /* NOTE CONTINUE */
      inbuf[strlen(inbuf)-1] = '\0';  /* get rid of stupid \n */
      nf = split(inbuf, f, MAXF, "\t\t");
      if (nf < 3)
        {
          fprintf(stderr, "bad input, line %d\n", line);
          exit(1);
        }
      for (i = 0; i < nf; i++)
        if (strcmp(f[i], "\"\"") == 0)
          f[i] = "";
      if (nf <= 3)
        f[3] = NULL;
      if (nf <= 4)
        f[4] = NULL;
      try(f[0], f[1], f[2], f[3], f[4], options('c', f[1]));
    }
}
