#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <string.h>
#include <time.h>
#include <stdarg.h>
#include "icd.h"
#include "pic16f87x.h"

#define BAUDRATE B57600

int icd_fd;  /* file descriptor for serial port */
int controller_type=-1;

void udelay(unsigned usec)
{
	/* wait for msec milliseconds or more ... */

	struct timespec time;

	time.tv_sec  = usec / 1000000;
	time.tv_nsec = ( usec % 1000000) * 1000;

	nanosleep(&time,NULL);
}

void dtr_set()
{
	int flag = TIOCM_DTR;

	if(ioctl(icd_fd, TIOCMBIS, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}

void dtr_clear()
{
	int flag  = TIOCM_DTR;

	if(ioctl(icd_fd, TIOCMBIC, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}

void rts_set()
{
	int flag = TIOCM_RTS;

	if(ioctl(icd_fd, TIOCMBIS, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}

void rts_clear()
{
	int flag  = TIOCM_RTS;

	if(ioctl(icd_fd, TIOCMBIC, &flag)) {
		perror("ioctl");
		exit(-1);
	}
}

void icd_hw_reset()
{
	rts_clear();
	dtr_clear();   /* reset */
	udelay(10000);
	dtr_set();     /* remove reset */
}


#define MAX_CMD_LEN 100

int icd_cmd(const char *cmd, ...)
{
	char command[MAX_CMD_LEN];
	unsigned char resp[3];
	int n_read;
	va_list ap;

	va_start(ap, cmd);
	(void) vsnprintf(command,MAX_CMD_LEN, cmd, ap);
	va_end(ap);

	write(icd_fd,command, strlen(command));   /* Error checking ... */

	n_read=read(icd_fd,resp,1);

	rts_clear();
	udelay(1);
	rts_set();
	
	if(n_read != 1) {
		printf("Error \n");
		return -1;
	}

	n_read=read(icd_fd,resp+1,1);

	rts_clear();
	udelay(1);
	rts_set();

	if(n_read != 1) {
		printf("Ouch, got only one byte ...\n");
		return -1;
	}

	return (resp[0] << 8 ) |  resp[1];
}


int icd_baudrate_init()
{
	int tries=3;
	char ch;

	while(tries) {
		write(icd_fd,"U",1);

		if(read(icd_fd,&ch,1) > 0) {
			rts_clear();
			udelay(10);
			rts_set();
			if(ch=='u') {
				fprintf(stderr,"baud rate init ok\n");
				return 1;
			}
		}
		tries--;
	}

	fprintf(stderr,"baud init failed\n");
	return 0;
}

void icd_firmware_version()
{
	unsigned int ver1,ver2;
	ver1 = icd_cmd("$7F00\r");
	ver2 = icd_cmd("$7021\r");

	printf("ICD firmware version: %X.%02X.%02X\n", ver1>>8, ver1&0xFF, ver2);
}

void icd_voltages()
{
	unsigned int vcc,vpp;

	icd_cmd("$7000\r");     /* enable Vpp */
	vcc=icd_cmd("$701C\r");
	vpp=icd_cmd("$701D\r")  & 0xFF;  /* What the heck does the high byte contain ? */
	icd_cmd("$7001\r");     /* disable Vpp */

	printf("Vcc = %4.2lfV, ", ((double)vcc) / 40.0);
	printf("Vpp = %5.2lfV\n", ((double)vpp) / 11.25);
}

int icd_controller_type()
{
	unsigned int dev_id, type,rev;

	dev_id=icd_cmd("$7020\r");
	type = (dev_id>>5) & 0x1FF;
	rev = type & 0x1F;

	if(dev_id == 0x3FFF) {
		printf("no target\n");
		return 0;
	} else {
		switch(type) {
		case 0x68:
			printf("Controller: PIC 16F870 rev %d\n",rev);
			return 870;
			break;
		case 0x69:
			printf("Controller: PIC 16F871 rev %d\n",rev);
			return 871;
			break;
		case 0x47:
			printf("Controller: PIC 16F872 rev %d\n",rev);
			return 872;
			break;
		case 0x4B:
			printf("Controller: PIC 16F873 rev %d\n",rev);
			return 873;
			break;
		case 0x49:
			printf("Controller: PIC 16F874 rev %d\n",rev);
			return 874;
			break;
		case 0x4F:
			printf("Controller: PIC 16F876 rev %d\n",rev);
			return 876;
			break;
		case 0x4D:
			printf("Controller: PIC 16F877 rev %d\n",rev);
			return 877;
			break;

		default:
			printf("Unknown controller, device id = %02X\n",dev_id);
			return -1;
			break;
		}
	}
}

void icd_erase()
{
	icd_cmd("$7000\r");	
	printf("Erasing device ...\n");
	if(icd_cmd("$7007\r") != 0x3FFF) {
		printf("Error erasing device\n");
	}
}

void icd_prog(int *mem)
{
	int addr,last_addr=-10;

	printf("Programming flash memory\n");

	icd_cmd("$7000\r");
	for(addr=0;addr < MAX_PROG_MEM_SIZE;addr++) {
		if(mem[addr] != UNINITIALIZED) {
			if(addr != last_addr+1) {
				if(icd_cmd("$7005\r") != 0x7005) {
					printf("programming error (address init)\n");
				}

				if(icd_cmd("$%04X\r", addr | 0x2000 ) !=  (addr | 0x2000)) {
					printf("programming error (address)\n");
				}
			}

			printf("programming: 0x%04X = 0x%04X\n",addr,mem[addr]);
			if(icd_cmd("$%04X\r", mem[addr] | 0x8000 ) != mem[addr]) {
				printf("programming error (data)\n");
			}
			last_addr = addr;
		}
	}

	printf("Programming id locations and configuration word\n");

	if(icd_cmd("$7005\r") != 0x7005) {
		printf("programming error (id_loc address init)\n");
	}
	
	if(icd_cmd("$C000\r") != 0x0000) {
		printf("programming error (id_loc address)\n");
	}

	for(addr = ID_LOC_ADDR; addr < ID_LOC_ADDR + ID_LOC_LEN;addr++) {
		if(mem[addr] == UNINITIALIZED) {
			if(icd_cmd("$7003\r") != 0x7003) {   /* increment address */
				printf("programming error (increment address)\n");
			}
		} else {
			if(icd_cmd("$%04X\r", mem[addr] | 0x8000 ) != mem[addr]) {
				printf("programming error (data)\n");
			}
		}
	}

	icd_cmd("$7001\r");
}

int icd_init(char *port)
{
	struct termios oldtio, newtio;

	if((icd_fd=open(port, O_NOCTTY | O_RDWR | O_SYNC)) == -1) {
		perror("Error opening device:");
		return -1;
	}

	tcgetattr(icd_fd, &oldtio);

	memset(&newtio,0, sizeof(newtio));
	newtio.c_cflag = BAUDRATE | CS8 | CLOCAL | CREAD;
	newtio.c_iflag = IGNPAR;
	newtio.c_oflag =  0;
	newtio.c_lflag =  0;

	newtio.c_cc[VTIME] = 20;
        newtio.c_cc[VMIN] = 0;

	tcflush(icd_fd, TCIFLUSH);
	tcsetattr(icd_fd, TCSANOW, &newtio);

	icd_hw_reset();

	rts_set();

	if(!icd_baudrate_init()) {
		fprintf(stderr,"Can't initialize the ICD\n");
	}

	icd_cmd("$6300\r");     /* I really don't know what this is, but MPLAB does
				   this. The program works ok without this though ..*/

	icd_firmware_version();

	icd_voltages();
	
	controller_type=icd_controller_type();
	
      	return 0;
}


void icd_close()
{
	rts_clear();
}
