#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "intel.h"
#include "icd.h"
#include "pic16f87x.h"

void usage(char *argv_0)
{
	printf("Usage: %s [-p serial_port_device] [-v] <hexfile>\n\n", argv_0);
	printf("Programs a Microchip PIC controller with the Microchip ICD programmer\n\n");
	printf("  options:\n");
	printf("   -p serial_port_device, default /dev/ttyS0\n");
	printf("   -v verify only\n");
}

int read_file(char *filename, int *mem)
{
	int i;
	int dat, datlow, dathigh, err;
	long addr;

	HEXFILE *hfp;

	for(i=0;i<MAX_MEM_SIZE;i++) {  /* fill array with -1 -> no data */
		mem[i] = UNINITIALIZED;
	}

	if ((hfp = open_hexfile(filename, "r")) == NULL) {
		printf("Error: Could not open %s\n", filename);
		return -1;
	}

	while ((err = read_hexfile(hfp, &datlow, &addr)) == HEXOK) {
		if(((addr/2) < MAX_PROG_MEM_SIZE) ||
		  (((addr/2)>=ID_LOC_ADDR) && ((addr/2)<ID_LOC_ADDR + ID_LOC_LEN))) {

			err = read_hexfile(hfp, &dathigh, &addr);  /* FIXME: error handling .. */
			dat = datlow + (dathigh << 8);
			addr = addr / 2;

			if(mem[addr] != UNINITIALIZED) {
				printf("Warning, hex file contains duplicate data\n");
			}

			if(addr==DEVICE_ID_ADDR) {
				printf("Waring, hex file contains a device ID (address = %04X)\n",DEVICE_ID_ADDR);
			}


			if(addr==CONF_WORD_ADDR) {
				printf("----- Device configuration: -----\n");
				printf("Oscillator selection bits: ");
				switch(dat & 0x3) {
				case 0:
					printf("LP\n");
					break;
				case 1:
					printf("XT\n");
					break;
				case 2:
					printf("HS\n");
					break;
				case 3:
					printf("RC\n");
					break;
				}
				printf("Watchdog timer:  %s\n", (dat & 0x04) ? "enabled"  : "disabled");
				printf("Power up timer:  %s\n", (dat & 0x08) ? "disabled" : "enabled");
				printf("Brown out reset: %s\n", (dat & 0x40) ? "enabled"  : "disabled");
				printf("Low voltage programming: %s\n", (dat & 0x80) ? "enabled"  : "disabled");
				printf("EE data code protection: %s\n", (dat & 0x100) ? "off"  : "on");
				printf("Flash memory write: %s\n", (dat & 0x200) ? "ok"  : "protected");

				if(((dat >>4)&0x3) != ((dat >> 12) & 0x3)) {
					fprintf(stderr,"Waring, funny code protection bits.\n");
				} else {
					switch((dat >> 4)&0x3) {
					case 0:
						printf("0x0000-0x0FFF (4K devices) or 0x0000-0x1FFF (8K devices) protected\n");
						break;
					case 1:
						printf("0x0800-0x0FFF (4K devices) or 0x1000-0x1FFF (8K devices) protected\n");
						break;
					case 2:
						printf("0x0F00-0x0FFF (4K devices) or 0x1F00-0x1FFF (8K devices) protected\n");
						break;
					case 3:
						printf("Code protection: off\n");
						break;
					}
				}

				if(!(dat & 0x400)) {
					printf("Warning, bit 10 in config word should be 1\n");
				}

				if(!(dat & 0x800)) {
					printf("Warning, bit 11 in config word should be 1\n");
				}

				dat |= 0xC00;  /* The ICD refuses to program if these bits are wrong */
			}
			mem[addr] = dat;
		}
	}

	if (err == HEXERR) {
		printf("Error reading %s (Is this a valid Intel hex file ?)\n", filename);
		return -1;
	}

	close_hexfile(hfp);

	return 0;
}

int main(int argc, char *argv[])
{
	char *filename;
	char *icd_device = "/dev/ttyS0";
	int mem[MAX_MEM_SIZE];

	int option;
	int verify_only = 0;

	while ((option = getopt(argc, argv, "vp:")) != -1) {
		switch (option) {
		case 'v':
			verify_only = 1;
			printf("Verify not implemented ...\n");
			break;

		case 'p':
			icd_device=optarg;
			break;

		case '?':
		default:
			usage(argv[0]);
			exit(1);
			break;
		}
	}

	if (optind == argc - 1) {	/* There should be exactly one arg left, the file name */
		filename = argv[optind];
	} else {
		usage(argv[0]);
		exit(1);
	}

	read_file(filename,mem);

	if(icd_init(icd_device) != 0) {
		exit(-1);
	}

	if (!verify_only) {
		icd_erase();
		icd_prog(mem);
	}

	if (verify_only) {
		printf("Not yet implemeted\n");
	}
	return 0;
}
