/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_log.c,v 1.3.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>

#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

/*
 * To prevent possible recursion from mem_*() functions,
 * libc memory allocation function is used in this file.
 */

/*
 * open_log(), logmsg() and mod_logmsg() save errno on enter
 * and restore errno on exit, to allow external function to check
 * value of errno, changed by some function which caused error.
 */

#define LOG_STREAM_BUF_SIZE	1024

/*
 * Do not call read() syscall for caught standard output and
 * error stream more than LOG_STREAM_READ_CALLS, to get other code
 * a chance to be executed.
 */
#define LOG_STREAM_READ_CALLS	10

/* vlogmsgx() or something like this. */
void		(*xvlogmsgx)(int, const char *, va_list);

const char	*log_ident = IPA_LOG_IDENT; /* -i log_ident */
const char	*log_file = NULL;	/* -o log_file */

static int	syslog_logopt;		/* Syslog log options. */

static long	mypid;			/* PID of current process. */
static int	log_fd = -1;		/* Log file descriptor. */

static int	bt_indent = 0;		/* Indent for logbt(). */

#ifndef LOG_FILE_PERM
# define LOG_FILE_PERM (S_IRUSR|S_IWUSR|S_IRGRP) /* 0640 */
#endif

/* IPA_LOG_xxx -> Message */
static const char *const priomsg[] = {
	"",			/* 0 */
	"W: ",			/* 1 */
	"E: "			/* 2 */
};

/* IPA_LOG_xxx -> LOG_xxx */
static const int log_priority[] = {
	LOG_INFO,		/* 0 */
	LOG_WARNING,		/* 1 */
	LOG_ERR			/* 2 */
};

#ifdef WITH_PTHREAD
# define STRERRBUF_SIZE		128	/* Size of buffer for strerror_r(). */
#endif

/*
 * Initialize settings for syslog only.
 */
void
init_log(void)
{
	if (log_file == NULL)
		syslog_logopt = LOG_PID |
#ifdef LOG_ODELAY
		    LOG_ODELAY |
#endif
#ifdef LOG_NOWAIT
		    LOG_NOWAIT |
#endif
		    (debug ?
#ifdef LOG_PERROR
			LOG_PERROR
#else
			0
#endif
			: LOG_CONS);
}

/*
 * This function is called by modules to output their log messages.
 */
void
mod_logmsg(const char *mod_name, int priority, int code, const char *format,
    va_list ap)
{
	char buf[LOG_BUF_SIZE];
	char *msg;
	int rv, errno_save, msg_alloced;

	errno_save = errno;

	msg_alloced = 0;
	rv = vsnprintf(buf, sizeof(buf), format, ap);
	if (rv < 0)
		msg = "(mod_logmsg: vsnprintf failed)";
	else if (rv < sizeof(buf))
		msg = buf;
	else if (rv >= sizeof(buf)) {
		msg = malloc(++rv);
		if (msg == NULL)
			msg = "(mod_logmsg: malloc failed)";
		else if (vsnprintf(msg, rv, format, ap) < 0) {
			free(msg);
			msg = "(mod_logmsg: vsnprintf failed)";
		} else
			msg_alloced = 1;
	}

	if (code != 0) {
		errno = code;
		logmsg(priority, "MOD %s: %s", mod_name, msg);
	} else
		logmsgx(priority, "MOD %s: %s", mod_name, msg);

	if (msg_alloced)
		free(msg);
	errno = errno_save;
}

/*
 * Write a message with a timestamp to the log file and to
 * standard error if needed.  If stat(2) for the opened
 * log file failed, then try to open/create it again.
 * If log_file is NULL, then output message only to stderr.
 */
static void
log_message(const char *message)
{
	struct iovec iov[3];
	struct stat statbuf;
	time_t t;
	int fd;
#ifdef WITH_PTHREAD
	char strtimebuf[26];
	char strerrbuf[STRERRBUF_SIZE];
#endif

	iov[2].iov_base = "\n";
	iov[2].iov_len = 1;

	while (log_file != NULL) {
		if (log_fd >= 0) {
			/* Log file is opened, check if it still exist. */
			if (stat(log_file, &statbuf) == 0)
				break;
			/* File does not exist. */
			(void)close(log_fd);
		}
		/* Create/open log file. */
		log_fd = open(log_file, O_WRONLY|O_APPEND|O_CREAT,
		    LOG_FILE_PERM);
		if (log_fd >= 0)
			break;
		if (!debug) {
			/*
			 * Cannot open/create log file and
			 * cannot log to stderr.
			 */
			return;
		}
		iov[0].iov_base = "log_message: cannot open log file: ";
		iov[0].iov_len = 35;
#ifdef WITH_PTHREAD
		iov[1].iov_base =
		    strerror_r(errno, strerrbuf, sizeof(strerrbuf)) == 0 ?
		    strerrbuf : "(strerror_r failed)";
#else
		iov[1].iov_base = strerror(errno);
#endif
		iov[1].iov_len = strlen(iov[1].iov_base);
		(void)writev(STDERR_FILENO, iov, 3);
		break;
	}

	fd = debug ? STDERR_FILENO : log_fd;

	if (time(&t) == (time_t)-1) {
		iov[0].iov_base = "log_message: time failed: ";
		iov[0].iov_len = 26;
#ifdef WITH_PTHREAD
		iov[1].iov_base =
		    (strerror_r(errno, strerrbuf, sizeof(strerrbuf)) == 0) ?
		    strerrbuf : "(strerror_r failed)";
#else
		iov[1].iov_base = strerror(errno);
#endif
		iov[1].iov_len = strlen(iov[1].iov_base);
	} else {
		/* From SUSv3: "DDD MMM dd hh:mm:ss YYYY\n\0" */
#ifdef WITH_PTHREAD
		iov[0].iov_base = ctime_r(&t, strtimebuf) + 4;
#else
		iov[0].iov_base = ctime(&t) + 4;
#endif
		iov[0].iov_len = strlen(iov[0].iov_base) - 1;
		iov[1].iov_base = (char *)message;
		iov[1].iov_len = strlen(message);
	}

	for (;;) {
		if (writev(fd, iov, 3) < 0)
			if (debug && fd != STDERR_FILENO) {
				(void)close(log_fd);
				log_fd = -1;
				break;
			}
		if (debug && fd == STDERR_FILENO) {
			fd = log_fd;
			if (fd < 0)
				break;
		} else
			break;
	}
}

/*
 * Call openlog() for syslog (real open for log descriptor is delayed).
 */
void
open_log(void)
{
	if (log_file == NULL) {
		int errno_save;

		errno_save = errno;
		openlog(log_ident, syslog_logopt, LOG_USER);
		errno = errno_save;
	} else {
		mypid = (long)getpid();
		log_fd = -1;
	}
}

/*
 * Close log descriptor.
 */
void
close_log(void)
{
	if (log_file == NULL)
		closelog();
	else if (log_fd >= 0) {
		(void)close(log_fd);
		log_fd = -1;
	}
}

/*
 * Wrapper for log function from memfunc.c.
 */
void
mvlogmsgx_wrapper(const char *format, va_list ap)
{
	/*
	 * Since mxxx() functions log messages when errors
	 * occurred, we simply use fixed log priority here.
	 */
	vlogmsgx(IPA_LOG_ERR, format, ap);
}

/*
 * vprintf-like function, do not output message for errno.
 */
void
vlogmsgx(int priority, const char *format, va_list ap)
{
	bt_indent = 0;
	if (log_file == NULL) {
#ifdef HAVE_VSYSLOG
		vsyslog(log_priority[priority], format, ap);
#else
		char buf[LOG_BUF_SIZE];

		if (vsnprintf(buf, sizeof(buf), format, ap) >= 0)
			syslog(log_priority[priority], buf);
#endif
#ifdef LOG_PERROR
		return;
#endif
	}

	{
		char buf1[LOG_BUF_SIZE];
		char buf2[LOG_BUF_SIZE + 128];
		char *msg1, *msg2;
		int rv;
		char msg1_alloced, msg2_alloced;

		msg1_alloced = msg2_alloced = 0;
		rv = vsnprintf(buf1, sizeof(buf1), format, ap);
		if (rv < 0)
			msg1 = "(vlogmsgx: vsnprintf failed for msg1)";
		else if (rv < sizeof(buf1))
			msg1 = buf1;
		else if (rv >= sizeof(buf1)) {
			msg1 = malloc(++rv);
			if (msg1 == NULL)
				msg1 = "(vlogmsgx: malloc failed for msg1)";
			else if (vsnprintf(msg1, rv, format, ap) < 0) {
				free(msg1);
				msg1 = "(vlogmsgx: vsnprintf failed for msg1)";
			} else
				msg1_alloced = 1;
		}
		rv = snprintf(buf2, sizeof(buf2), " %s[%ld]: %s%s", log_ident,
		    mypid, priomsg[priority], msg1);
		if (rv < 0)
			msg2 = "(vlogmsgx: vsnprintf failed for msg2)";
		else if (rv < sizeof(buf2))
			msg2 = buf2;
		else if (rv >= sizeof(buf2)) {
			msg2 = malloc(++rv);
			if (msg2 == NULL)
				msg2 = "(vlogmsgx: malloc failed for msg2)";
			else if (vsnprintf(msg2, rv, format, ap) < 0) {
				free(msg2);
				msg2 = "(vlogmsgx: vsnprintf failed for msg2)";
			} else
				msg2_alloced = 1;
		}

		log_message(msg2);

		if (msg1_alloced)
			free(msg1);
		if (msg2_alloced)
			free(msg2);
	}
}

/*
 * The same as vlogmsgx(), but output everything on stderr and
 * output is not atomic, since this is not required.
 */
void
vlogmsgx_stderr(int priority, const char *format, va_list ap)
{
	(void)fflush(stdout);
	switch (priority) {
	case IPA_LOG_INFO:
		fprintf(stderr, "Info: ");
		break;
	case IPA_LOG_WARNING:
		fprintf(stderr, "Warning: ");
		break;
	case IPA_LOG_ERR:
		fprintf(stderr, "Error: ");
		break;
	}
	vfprintf(stderr, format, ap);
	fprintf(stderr, "\n");
}

/*
 * printf-like function, do not output message for errno.
 */
void
logmsgx(int priority, const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vlogmsgx(priority, format, ap);
	va_end(ap);
}

/*
 * printf-like function, also output message for errno.
 */
void
logmsg(int priority, const char *format, ...)
{
	va_list ap;

	if (errno == 0) {
		va_start(ap, format);
		vlogmsgx(priority, format, ap);
		va_end(ap);
		return;
	}

	{
		char buf[LOG_BUF_SIZE];
		char *msg;
		int rv, errno_save, msg_alloced;
#ifdef WITH_PTHREAD
		char strerrbuf[STRERRBUF_SIZE];
#endif

		errno_save = errno;
		msg_alloced = 0;
		va_start(ap, format);
		rv = vsnprintf(buf, sizeof(buf), format, ap);
		if (rv < 0)
			msg = "(logmsg: vsnprintf failed)";
		else if (rv < sizeof(buf))
			msg = buf;
		else if (rv >= sizeof(buf)) {
			msg = malloc(++rv);
			if (msg == NULL)
				msg = "(logmsg: malloc failed)";
			else if (vsnprintf(msg, rv, format, ap) < 0) {
				free(msg);
				msg = "(logmsg: vsnprintf failed)";
			} else
				msg_alloced = 1;
		}

		va_end(ap);
#ifdef WITH_PTHREAD
		if (strerror_r(errno_save, strerrbuf, sizeof(strerrbuf)) == 0)
			logmsgx(priority, "%s: %s", msg, strerrbuf);
		else
			logmsgx(priority, "%s: error code %d", msg, errno_save);
#else
		logmsgx(priority, "%s: %s", msg, strerror(errno_save));
#endif
		if (msg_alloced)
			free(msg);
		errno = errno_save;
	}
}

/*
 * printf-like function for debug messages.
 */
void
logdbg(const char *format, ...)
{
	char buf[LOG_BUF_SIZE];
	va_list ap;
	int rv;

	va_start(ap, format);
	rv = vsnprintf(buf, sizeof(buf), format, ap);
	va_end(ap);
	if (rv < 0)
		logmsg(IPA_LOG_ERR, "logdbg: vsnprintf failed");
	else
		logmsgx(IPA_LOG_INFO, "DEBUG: %s", buf);
}

/*
 * Call xvlomsgx() which is a pointer to some vlogmsgx-like function.
 */
void
xlogmsgx(int priority, const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	xvlogmsgx(priority, format, ap);
	va_end(ap);
}

/*
 * Log backtrace information.
 */
void
logbt(const char *msg)
{
	int indent;

	indent = bt_indent;
	logmsgx(log_file == NULL ? IPA_LOG_INFO : IPA_LOG_ERR, " %*s`- %s",
	    indent, "", msg);
	bt_indent = indent + 2;
}

/*
 * Read data from non-blocking fd and log them prepending with
 * stream name string, interpret '\0' and '\n' as new lines, call
 * read() syscall for fd no more than LOG_STREAM_READ_CALLS times.
 * This function is used only by log_stdout() and log_stderr().
 */
void
log_stream(int fd, const char *stream_name)
{
	char *ptr1, *ptr2;
	ssize_t nread;		/* Number of read bytes from last read(). */
	size_t ngot;		/* Number of valid bytes in string. */
	size_t nleft;		/* Number of not used bytes in string. */
	unsigned int ncalls;	/* Number of read() invocations. */
	int i;
	char string[LOG_STREAM_BUF_SIZE];

	ncalls = 0;
	ptr1 = string;
	nleft = sizeof(string) - 1;
	ngot = 0;
	(void)fflush(stdout);
	for (;;) {
		nread = read(fd, ptr1, nleft);
		if (nread < 0) {
			if (errno == EAGAIN
#if EAGAIN != EWOULDBLOCK
			    || errno == EWOULDBLOCK
#endif
			    ) {
				if (ptr1 != string)
					logmsgx(IPA_LOG_WARNING, "*%s: %s",
					    stream_name, string);
			} else
				logmsg(IPA_LOG_ERR, "log_stream(%s): read",
				    stream_name);
			break;
		}
		ngot += nread;
		for (i = 0, ptr1 = ptr2 = string; i < ngot; ++ptr2, ++i)
			switch (*ptr2) {
			case '\n':
				*ptr2 = '\0';
				/* FALLTHROUGH */
			case '\0':
				logmsgx(IPA_LOG_WARNING, "*%s: %s",
				    stream_name, ptr1);
				ptr1 = ptr2 + 1;
			}
		if (ptr1 == string) {
			ptr1[ngot] = '\0';
			ngot = 0;
			nleft = sizeof(string) - 1;
			logmsgx(IPA_LOG_WARNING, "*%s: %s", stream_name, ptr1);
		} else if (ptr1 != ptr2) {
			ngot = ptr2 - ptr1;
			memcpy(string, ptr1, ngot);
			ptr1 = string + ngot;
			*ptr1 = '\0';
			nleft = sizeof(string) - ngot - 1;
		} else {
			ptr1 = string;
			ngot = 0;
			nleft = sizeof(string) - 1;
		}
		if (++ncalls == LOG_STREAM_READ_CALLS) {
			if (ngot != 0) /* and has '\0' in string. */
				logmsgx(IPA_LOG_WARNING, "*%s: %s",
				    stream_name, string);
			break;
		}
	}
}
