#include "isectd.h"
#include <errno.h>
#include <time.h>
#include <stdlib.h>

#define ISMAXWORDS 10

#ifndef strupr
extern char *strupr(char *);
#endif

static char
	*blockclient(char *, int, char **), 
	*buckets(char *, int, char **), 
	*examineclient(char *, int, char **),
	*changepriority(char *, int, char **), 
	*resetall(char *, int, char **), 
	*shutdownfd(char *, int, char **), 
	*idleworker(char *, int, char **), 
	*startservice(char *, int, char **), 
	*stopservice(char *, int, char **), 
	*pauseworkers(char *, int, char **), 
	*readyworkers(char *, int, char **), 
	*examineworker(char *, int, char **), 
	*cloneworker(char *, int, char **),
	*listevents(char *, int, char **), 
	*listservices(char *, int, char **),
	*deleteid(char *, int, char **), 
	*changeservice(char *, int, char **), 
	*listworkers(char *, int, char **),
	*configme(char *, int, char **);

static char *
suicide(char *cp, int argc, char **argv)
{
	isdLog("operator requested HALT\n");
	exit(1);

	return NULL;
}

static char *
isdAbort(char *cp, int argc, char **argv)
{
	isdLog("operator requested ABORT\n");
	abort();

	return NULL;
}

struct isdActions {
	char *action;
	char *(*routine) (char *, int, char **);
} isdActions[] = {

	{
		"ABORT", isdAbort
	},
	{
		"BLOCK", blockclient
	},
	{
		"BUCKETS", buckets
	},
	{
		"CONFIG", configme
	},
	{
		"CLIENT", examineclient
	},
	{
		"CLIENTS", isdStatClientQueue
	},
	{
		"CLONE", cloneworker
	},
	{
		"DELETE", deleteid
	},
	{
		"HALT", suicide
	},
	{
		"IDLE", idleworker
	},
	{
		"LISTEVENTS", listevents
	},
	{
		"PAUSE", pauseworkers
	},
	{
		"SERVICES", listservices
	},
	{
		"PRI", changepriority
	},
	{
		"READY", readyworkers
	},
	{
		"RESET", resetall
	},
	{
		"SERVICE", changeservice
	},
	{
		"SHUTDOWN", shutdownfd
	},
	{
		"START", startservice
	},
	{
		"STOP", stopservice
	},
	{
		"UNBLOCK", blockclient
	},
	{
		"WORKER", examineworker
	},
	{
		"WORKERS", listworkers
	},
	{
		NULL, NULL
	}
};

char *
isdCommands(char *request)
{
	int i, argc;
	char *cp, *argv[ISMAXWORDS];
	char arg[ISMAXWORDS][128];
	static char *reply = NULL;

	if (reply == NULL)
		reply = malloc(2 * 8192);

	for (i = 0; i < ISMAXWORDS; i++)
		argv[i] = arg[i];

	argc = sscanf(request, "%s%s%s%s%s%s%s%s%s%s", arg[0], arg[1], arg[2], arg[3], arg[4], arg[5], arg[6], arg[7], arg[8], arg[9]);

	*(cp = reply) = 0;

	if (argc > 0 && (*argv[0] != '#')) {
		strupr(argv[0]);

		for (i = 0; isdActions[i].action; i++)
			if (strcmp(argv[0], isdActions[i].action) == 0) {
				isdActions[i].routine(cp, argc, argv);
				break;
			}
		if (isdActions[i].action == NULL) {
			strcat(reply,
				   "abort                    isectd calls abort() - creates a core file\n"
				   "[un]block clientid       [un]blocks incoming transactions from clientid\n"
				   "buckets num secs         set the number of buckets to 'num' and the bucket\n"
				   "                         size to 'secs' seconds\n"
				   "client id                examine the client structure for client id id\n"
				   "clients                  displays the client list\n"
				   "clone workerid           clones a worker id\n"
				   "config events n          remember n events (listevents)\n"
				   "delete {worker|client} id\n"
				   "                         deletes a client or worker\n"
				   "halt                     isectd calls exit(1)\n"
				   "idle id                  forcibly idles the busy worker id\n"
				   "listevents [n]           list interesting events\n"
				   "pause service | worker | all\n"
				   "                         pauses a worker (makes lazy) without killing it\n"
				   "services                 lists isectd's services\n"
				   "preempt [on|off]         sets or reports preempting option\n"
				   "pri id n                 changes a client's priority to n\n"
				   "ready service | worker | all\n"
				   "                         unpauses a worker\n"
				   "reset                    resets statistics\n"
				   "send worker_id|service_name|all command:(reinit, status(TP workers only))\n"
				   "                         sends command to a worker, service or all services\n"
				   "service id newservice    changes a client's service to newservice\n"
				   "shutdown fd | all        shutdown the fd for sending\n"
				   "start service | all      start processing messages for service\n"
				   "stop service | all       stop processing messages for service\n"
				   "unblock clientid         reverses \"block clientid\"\n"
				   "worker id                examine the worker structure for worker id id\n"
				   "workers [lazy | idle | busy | all]\n"
				   "                         displays the worker queues\n"
				   "<nil>                    examine isdData\n"
				);
		}
	}
	else if (*argv[0] != '#')
		isdStatistics(cp);
	else
		;

	return strdup(reply);
}

static char *
listworkers(char *cp, int argc, char **argv)
{
	char *option = "ALL";

	if (argc > 1)
		option = argv[1];

	/* sae 08/11/1999 */
	strupr(option);
	if (strstr("~LAZY~IDLE~BUSY~ALL~", option) == NULL)
		cp += ADDBYTES(sprintf(cp, "\"%s\" is an unrecognized option - must be LAZY, IDLE, BUSY, ALL, or omitted\n", option));

	else {
		if ((strcmp(option, "LAZY") == 0) || (strcmp(option, "ALL") == 0)) {
			cp += ADDBYTES(sprintf(cp, "LAZY WORKERS\n"));
			cp = isdStatWorkerQueue(cp, &isdData.LazyWorkers);
		}

		if ((strcmp(option, "IDLE") == 0) || (strcmp(option, "ALL") == 0)) {
			cp += ADDBYTES(sprintf(cp, "IDLE WORKERS\n"));
			cp = isdStatWorkerQueue(cp, &isdData.IdleWorkers);
		}

		if ((strcmp(option, "BUSY") == 0) || (strcmp(option, "ALL") == 0)) {
			cp += ADDBYTES(sprintf(cp, "BUSY WORKERS\n"));
			cp = isdStatWorkerQueue(cp, &isdData.BusyWorkers);
		}
	}

	return cp;
}

static char *
changeservice(char *cp, int argc, char **argv)
{
	int id;
	isdClient *client;
	isdService *service;

	if (argc == 3) {
		client = (isdClient *) QueueSearch(&isdData.PriList, QEQ, offsetof(isdClient, id), id = atoi(argv[1]), QNULL);
		if (client != (isdClient *) QNULL) {
			service = (isdService *) QueueByteCmp(&isdData.ServiceList, QEQ, offsetof(isdService, name[0]), argv[2], strlen(argv[2]), QNULL);
			if (service != (isdService *) QNULL) {
				client->service = service;
				client->service_id = service->id;
				cp += ADDBYTES(sprintf(cp, "Ok."));
			}
			else
				cp += ADDBYTES(sprintf(cp, "WARNING: couldn't find service %s\n", argv[2]));
		}
		else
			cp += ADDBYTES(sprintf(cp, "WARNING: couldn't find client id %s\n", argv[1]));
	}
	else
		cp += ADDBYTES(sprintf(cp, "too many or too few arguments - use 'help'\n", argv[1]));

	return cp;
}

static char *
changepriority(char *cp, int argc, char **argv)
{
	int id;
	isdClient *client;

	if (argc == 3) {
		client = (isdClient *) QueueSearch(&isdData.PriList, QEQ, offsetof(isdClient, id), id = atoi(argv[1]), QNULL);
		if (client != (isdClient *) QNULL) {
			client->priority = atol(argv[2]);
			isdPriList((isdClient *) RemoveNode((Node *) client));
			cp += ADDBYTES(sprintf(cp, "Ok."));
		}
		else
			cp += ADDBYTES(sprintf(cp, "WARNING: couldn't find client id %s\n", argv[1]));
	}
	else
		cp += ADDBYTES(sprintf(cp, "too many or too few arguments - use 'help'\n", argv[1]));

	return cp;
}

static char *
resetall(char *cp, int argc, char **argv)
{
	isdZeroWorkerQueue(&isdData.LazyWorkers);
	isdZeroWorkerQueue(&isdData.IdleWorkers);
	/* isdZeroWorkerQueue(&isdData.BusyWorkers);              DANGEROUS to attempt against the Busy queue */
	isdZeroClientQueue();
	isdData.PeakBucket = 0;
	if (QueueItems(&isdData.BusyWorkers) > 1)
		cp += ADDBYTES(sprintf(cp, "WARNING: busy workers were not reset\n"));
	else
		cp += ADDBYTES(sprintf(cp, "REMINDER: %s can not be reset\n", isdData.command_worker));

	return cp;
}

static char *
shutdownfd(char *cp, int argc, char **argv)
{
	int i;
	Node *node;
	isdWorker *worker;
	Queue *queue[2];

	if (argc == 2) {
		queue[0] = &isdData.IdleWorkers;
		queue[1] = &isdData.BusyWorkers;
		strupr(argv[1]);
		if (strcmp(argv[1], "ALL") == 0) {
			for (i = 0; i < 2; i++) {
				for (node = queue[i]->head; node != QNULL; node = node->next) {
					if (strcmp((worker = (isdWorker *) node)->name, isdData.command_worker) != 0) {
						worker = (isdWorker *) node;
						cp += ADDBYTES(sprintf(cp, "%d ", worker->socket));
						if (shutdown(worker->socket, 1) == 0)
							cp += ADDBYTES(sprintf(cp, "Ok.\n"));
						else
							cp += ADDBYTES(sprintf(cp, "Error %d : %s\n", errno, strerror(errno)));
					}
				}
			}
		}
		else {
			if (shutdown(atoi(argv[1]), 1) == 0)
				cp += ADDBYTES(sprintf(cp, "Ok.\n"));
			else
				cp += ADDBYTES(sprintf(cp, "Error %d : %s\n", errno, strerror(errno)));
		}
	}
	else
		cp += ADDBYTES(sprintf(cp, "usage: shutdown fd\n"));

	return cp;
}

static char *
idleworker(cp, argc, argv)
char *cp;
int argc;
char **argv;
{
	int id;
	isdWorker *worker;

	if (argc == 2) {
		id = atol(argv[1]);
		worker = (isdWorker *) QueueSearch(&isdData.BusyWorkers, QEQ, offsetof(isdWorker, id), id, QNULL);
		if (worker != (isdWorker *) QNULL) {
			isdTranStats(worker);
			if (worker->client->socket != INVALID_SOCKET) {		/* if client still lives */
				FD_SET(worker->client->socket, &isdData.allfds);
				if (worker->header.reply) {
					worker->header.error = -1;
					isdSend(worker->client->socket, "PROTOCOL INTERVENTION", -1, &worker->header);
					cp += ADDBYTES(sprintf(cp, "sent reply to client %d\n", worker->client->id));
				}
			}
			isdIdleWorker(worker);
			cp += ADDBYTES(sprintf(cp, "idled worker %d\n", worker->id));
		}
		else
			cp += ADDBYTES(sprintf(cp, "worker %d not found: command only valid for BUSY workers\n", id));
	}
	else
		cp += ADDBYTES(sprintf(cp, "usage: 'idle id' - only valid for BUSY workers\n"));

	return cp;
}

static char *
stopservice(char *cp, int argc, char **argv)
{
	int i;
	Node *node;
	isdWorker *worker;
	Queue *queue[3];
	char service[MAXSERVICENAMELEN];

	if (argc == 2) {
		strupr(strncpy(service, argv[1], sizeof service));
		queue[0] = &isdData.LazyWorkers;
		queue[1] = &isdData.IdleWorkers;
		queue[2] = &isdData.BusyWorkers;

		for (i = 0; i < 3; i++) {
			for (node = queue[i]->head; node != QNULL; node = node->next) {
				worker = (isdWorker *) node;
				if (strcmp(service, "ALL") == 0 || strcmp(worker->name, argv[1]) == 0) {
					if (strcmp(worker->name, isdData.command_worker) != 0) {
						worker->stopped = 1;
						if (shutdown(worker->socket, 1) == 0)
							cp += ADDBYTES(sprintf(cp, "%d Ok.\n", worker->socket));
						else
							cp += ADDBYTES(sprintf(cp, " %d Error %d : %s\n", worker->socket, errno, strerror(errno)));
					}
					else
						cp += ADDBYTES(sprintf(cp, "WARNING: you cannot stop %s\n", isdData.command_worker));
				}
			}
		}
	}
	else
		cp += ADDBYTES(sprintf(cp, "usage: stop service | all\n"));

	return cp;
}

static char *
startservice(cp, argc, argv)
char *cp;
int argc;
char **argv;
{
	int i;
	Node *node;
	isdWorker *worker;
	Queue *queue[3];
	char service[MAXSERVICENAMELEN];

	if (argc == 2) {
		strupr(strncpy(service, argv[1], sizeof service));
		queue[0] = &isdData.LazyWorkers;
		queue[1] = &isdData.IdleWorkers;
		queue[2] = &isdData.BusyWorkers;

		for (i = 0; i < 3; i++) {
			for (node = queue[i]->head; node != QNULL; node = node->next) {
				worker = (isdWorker *) node;
				if (strcmp(service, "ALL") == 0 || strcmp(worker->name, argv[1]) == 0) {
					worker->up_not_log_in = 0;
					worker->stopped = 0;
				}
			}
		}
		cp += ADDBYTES(sprintf(cp, "%s\n", isdStartWorkers() == 0 ? "Ok." : "Error"));
	}
	else
		cp += ADDBYTES(sprintf(cp, "usage: start service | all\n"));

	return cp;
}

static char *
pauseworkers(char *cp, int argc, char **argv)
{
	int i;
	Node *node;
	isdWorker *worker;
	Queue *queue[3];
	char service[MAXSERVICENAMELEN];
	Node **topause, **nexttopause;

	if (argc == 2) {
		strupr(strncpy(service, argv[1], sizeof service));
		queue[0] = &isdData.LazyWorkers;
		queue[1] = &isdData.IdleWorkers;
		queue[2] = &isdData.BusyWorkers;

		nexttopause = topause = (Node **) calloc(QueueItems(queue[0]) + QueueItems(queue[1]) + QueueItems(queue[2]), sizeof(Node *));
		for (i = 0; i < 3; i++) {
			for (node = queue[i]->head; node != QNULL; node = node->next) {
				worker = (isdWorker *) node;
				if (strcmp(service, "ALL") == 0 || strcmp(worker->name, argv[1]) == 0 || worker->id == atol(argv[1])) {
					if (strcmp(worker->name, isdData.command_worker) != 0) {
						worker->stopped = 1;
						*nexttopause++ = node;
						cp += ADDBYTES(sprintf(cp, "paused worker->id = %d, worker->name = %s\n", worker->id, worker->name));
					}
					else
						cp += ADDBYTES(sprintf(cp, "WARNING: you cannot pause %s\n", isdData.command_worker));
				}
			}
			for (nexttopause = topause; *nexttopause; nexttopause++)
				if ((*nexttopause)->queue != &isdData.BusyWorkers)
					InsertHead(&isdData.LazyWorkers, RemoveNode(*nexttopause));
		}
		free((char *) topause);
	}
	else
		cp += ADDBYTES(sprintf(cp, "usage: stop service | all\n"));

	return cp;
}

static char *
readyworkers(cp, argc, argv)
char *cp;
int argc;
char **argv;
{
	int i;
	Node *node;
	isdWorker *worker;
	Queue *queue[3];
	char service[MAXSERVICENAMELEN];
	Node **toidle, **nexttoidle;

	if (argc == 2) {
		strupr(strncpy(service, argv[1], sizeof service));
		queue[0] = &isdData.LazyWorkers;
		queue[1] = &isdData.IdleWorkers;
		queue[2] = &isdData.BusyWorkers;

		nexttoidle = toidle = (Node **) calloc(QueueItems(queue[0]) + QueueItems(queue[1]) + QueueItems(queue[2]), sizeof(Node *));
		for (i = 0; i < 3; i++) {
			for (node = queue[i]->head; node != QNULL; node = node->next) {
				worker = (isdWorker *) node;
				if (strcmp(service, "ALL") == 0 || strcmp(worker->name, argv[1]) == 0 || worker->id == atol(argv[1])) {
					worker->stopped = 0;
					if ((worker->socket >= 0) && (queue[i] == &isdData.LazyWorkers))
						*nexttoidle++ = node;
					cp += ADDBYTES(sprintf(cp, "readied worker->id = %d, worker->name = %s\n", worker->id, worker->name));
				}
			}
		}
		for (nexttoidle = toidle; *nexttoidle; nexttoidle++)
			if ((*nexttoidle)->queue != &isdData.BusyWorkers)
				isdIdleWorker((isdWorker *) * nexttoidle);
	}
	else
		cp += ADDBYTES(sprintf(cp, "usage: stop service | all\n"));

	free((char *) toidle);
	return cp;
}

static char *zapBuffer(char *cp, char *buffer, int size)
{
	int i;

	if (size > 14000)
		size = 14000;
		
	for (i = 0; i < size; i += 64)
		cp += ADDBYTES(sprintf(cp, "% 8x  %s\n", i, zapLine(buffer + i, i + 64 < size ? 64:size - i, 64)));

	return cp;
}

static char *
examineworker(char *cp, int argc, char **argv)
{
	int i;
	isdWorker *worker;

	if (argc == 2) {
		if (worker = isdVldWrkrId(atol(argv[1]))) {
			cp += ADDBYTES(sprintf(cp, "worker->node.next     = 0x%08x\n", worker->node.next));
			cp += ADDBYTES(sprintf(cp, "worker->node.previous = 0x%08x\n", worker->node.previous));
			cp += ADDBYTES(sprintf(cp, "worker->node.queue    = 0x%08x\n", worker->node.queue));

			cp += ADDBYTES(sprintf(cp, "worker->id            = %d\n", worker->id));
			cp += ADDBYTES(sprintf(cp, "worker->service_id    = %d\n", worker->service_id));
			cp += ADDBYTES(sprintf(cp, "worker->hostpid       = %d\n", worker->hostpid));

			cp += ADDBYTES(sprintf(cp, "worker->name          = \"%s\"\n", worker->name));
			cp += ADDBYTES(sprintf(cp, "worker->host          = \"%s\"\n", worker->host));
			cp += ADDBYTES(sprintf(cp, "worker->path          = \"%s\"\n", worker->path));
			cp += ADDBYTES(sprintf(cp, "worker->args          = \"%s\"\n", worker->args));

			cp += ADDBYTES(sprintf(cp, "worker->Transactions  = %d\n", worker->Transactions));
			cp += ADDBYTES(sprintf(cp, "worker->fromCount     = %d\n", worker->fromCount));
			cp += ADDBYTES(sprintf(cp, "worker->toCount       = %d\n", worker->toCount));
			cp += ADDBYTES(sprintf(cp, "worker->restarts      = %d\n", worker->restarts));
			cp += ADDBYTES(sprintf(cp, "worker->stopped       = %d\n", worker->stopped));
			cp += ADDBYTES(sprintf(cp, "worker->up_not_log_in = %d\n", worker->up_not_log_in));

			if (worker->msgBuffer) 
				cp = zapBuffer(cp, worker->msgBuffer, worker->header.len);
		}
		else
			cp += ADDBYTES(sprintf(cp, "\"%s\" is not a valid workerid\n", argv[1]));
	}
	else
		cp += ADDBYTES(sprintf(cp, "\"%s\" requires a workerid argument\n", argv[0]));
	return cp;
}

static char *
cloneworker(char *cp, int argc, char **argv)
{
	isdWorker *oldworker, *newworker;

	if (argc == 2) {
		if (oldworker = isdVldWrkrId(atol(argv[1]))) {
			newworker = (isdWorker *) calloc(1, sizeof *newworker);

			newworker->id = ++isdData.NextWorkerID;
			newworker->socket = INVALID_SOCKET;
			newworker->service = oldworker->service;
			newworker->service_id = oldworker->service_id;
			newworker->stopped = oldworker->stopped;

			strcpy(newworker->name, oldworker->name);
			strcpy(newworker->host, oldworker->host);
			strcpy(newworker->path, oldworker->path);
			strcpy(newworker->args, oldworker->args);

			InsertHead((Queue *) & isdData.LazyWorkers, (Node *) newworker);
			isdStartWorker(newworker);
			cp += ADDBYTES(sprintf(cp, "cloned %d from %d\n", newworker->id, oldworker->id));
		}
		else
			cp += ADDBYTES(sprintf(cp, "\"%s\" is not a valid workerid\n", argv[1]));
	}
	else
		cp += ADDBYTES(sprintf(cp, "clone requires a workerid\n"));

	return cp;
}

static char *
listevents(cp, argc, argv)
char *cp;
int argc;
char **argv;
{
	int i, num_to_show = isdData.NumEvents;
	isdEvent *event;

	if (argc == 2)
		num_to_show = atoi(argv[1]);

	for (i = 0, event = (isdEvent *) isdData.Events.head;
		 event != (isdEvent *) QNULL && i < num_to_show; event = (isdEvent *) event->node.next, i++) {
		if (event->text[0])
			cp += ADDBYTES(sprintf(cp, "%s\n", event->text));
		else
			break;
	}

	return cp;
}

static char *
listservices(cp, argc, argv)
char *cp;
int argc;
char **argv;
{
	Node *node;
	isdService *service;

	cp += ADDBYTES(sprintf(cp, "%*s  %10s\n", MAXSERVICENAMELEN, "service", "id"));

	for (node = isdData.ServiceList.head; node != QNULL; node = node->next) {
		service = (isdService *) node;
		cp += ADDBYTES(sprintf(cp, "%*s  %10d\n", MAXSERVICENAMELEN, service->name, service->id));
	}
	return cp;
}

static char *
examineclient(cp, argc, argv)
char *cp;
int argc;
char **argv;
{
	Node *node;
	isdClient *client;
	double seconds;

	if (argc == 2) {
		if ((node = QueueSearch(&isdData.PriList, QEQ, offsetof(isdClient, id), atol(argv[1]), QNULL)) != QNULL) {
			client = (isdClient *) node;

			cp += ADDBYTES(sprintf(cp, "client->node.next     = 0x%08x\n", client->node.next));
			cp += ADDBYTES(sprintf(cp, "client->node.previous = 0x%08x\n", client->node.previous));
			cp += ADDBYTES(sprintf(cp, "client->node.queue    = 0x%08x\n", client->node.queue));

			cp += ADDBYTES(sprintf(cp, "client->id            = %d\n", client->id));
			cp += ADDBYTES(sprintf(cp, "client->service_id    = %d\n", client->service_id));
			cp += ADDBYTES(sprintf(cp, "client->socket        = %d\n", client->socket));
			cp += ADDBYTES(sprintf(cp, "client->priority      = %d\n", client->priority));
			cp += ADDBYTES(sprintf(cp, "client->loghost       = %s\n", client->loghost));
			cp += ADDBYTES(sprintf(cp, "client->hostpid       = %d\n", client->hostpid));
			cp += ADDBYTES(sprintf(cp, "client->ready         = %d\n", client->ready));
			cp += ADDBYTES(sprintf(cp, "client->Transactions  = %d\n", client->Transactions));
			cp += ADDBYTES(sprintf(cp, "client->fromCount     = %d\n", client->fromCount));
			cp += ADDBYTES(sprintf(cp, "client->toCount       = %d\n\n", client->toCount));

			cp += ADDBYTES(sprintf(cp, "Login Seconds         = %d (+ -time(0))\n", (int) (seconds = (double) (time(0) - client->login_time))));
			cp += ADDBYTES(sprintf(cp, "Average Response      = %6.3f\n", client->AverageTime));
			cp += ADDBYTES(sprintf(cp, "Last Response         = %6.3f\n", client->LastTime));
			cp += ADDBYTES(sprintf(cp, "Messages Per Second   = %6.3f\n", (double) (double) client->Transactions / seconds));
		}
		else
			cp += ADDBYTES(sprintf(cp, "\"%s\" is not a valid clientid\n", argv[1]));
	}
	else
		cp += ADDBYTES(sprintf(cp, "\"%s\" requires a clientid argument\n", argv[0]));

	return cp;
}

static char *
deleteid(char *cp, int argc, char **argv)
{
	Queue *queue;
	isdWorker *worker;
	isdClient *client;
	int id;

	if (argc == 3) {
		char *type = strupr(strdup(argv[1]));
		id = atol(argv[2]);
		if (*type == 'W') {
			worker = isdVldWrkrId(id);
			if (worker) {
				queue = worker->node.queue;
				if (queue == &isdData.LazyWorkers && worker->socket == INVALID_SOCKET) {
					RemoveNode((Node *) worker);
					free(worker);
					cp += ADDBYTES(sprintf(cp, "worker %d deleted\n", id));
				}
				else
					cp += ADDBYTES(sprintf(cp, "worker must be stopped before it can be deleted\n"));
			}
			else
				cp += ADDBYTES(sprintf(cp, "\"%s\" is an unknown worker id\n", argv[2]));
		}
		else if (*type == 'C') {
			client = (isdClient *) QueueSearch(&isdData.PriList, QEQ, offsetof(isdClient, id), id, QNULL);
			if (client) {
				if (client->socket != INVALID_SOCKET) {
					if (shutdown(client->socket, 0) == -1)
						cp += ADDBYTES(sprintf(cp, "Error %d : %s\n", errno, strerror(errno)));
					else
						cp += ADDBYTES(sprintf(cp, "client %d delete-ready\n", id));
				}
				else
					cp += ADDBYTES(sprintf(cp, "client %d's fd is already -1\n", id));
			}
			else
				cp += ADDBYTES(sprintf(cp, "\"%s\" is an unknown client id\n", argv[2]));
		}
		else
			cp += ADDBYTES(sprintf(cp, "\"%s\" is an unknown type\n", argv[1]));

		free(type);
	}
	else
		cp += ADDBYTES(sprintf(cp, "\"%s\" requires two arguments\n", argv[0]));

	return cp;
}

static char *
blockclient(char *cp, int argc, char **argv)
{
	isdClient *client;

	if (argc == 2) {
		client = (isdClient *) QueueSearch(&isdData.PriList, QEQ, offsetof(isdClient, id), atoi(argv[1]), QNULL);
		if (client != (isdClient *) QNULL) {
			if (strcmp("BLOCK", argv[0]) == 0) {
				client->blocked = 1;
				FD_CLR(client->socket, &isdData.allfds);
			}
			else {
				client->blocked = 0;
				FD_SET(client->socket, &isdData.allfds);
			}
			cp += ADDBYTES(sprintf(cp, "Ok."));
		}
		else
			cp += ADDBYTES(sprintf(cp, "WARNING: couldn't find client id %s\n", argv[1]));
	}
	else
		cp += ADDBYTES(sprintf(cp, "\"%s\" requires a clientid\n", argv[0]));

	return cp;
}

static char *
buckets(cp, argc, argv)
char *cp;
int argc;
char **argv;
{
	int num, size;

	if (argc == 3) {
		if ((num = atoi(argv[1])) > 0)
			if ((size = atoi(argv[2])) > 0) {
				isdBucketParams(num, size);
				cp += ADDBYTES(sprintf(cp, "Ok.\n", argv[2]));
			}
			else
				cp += ADDBYTES(sprintf(cp, "\"%s\" is an invalid bucket size argument\n", argv[2]));
		else
			cp += ADDBYTES(sprintf(cp, "\"%s\" is an invalid bucket num argument\n", argv[1]));
	}
	else
		cp += ADDBYTES(sprintf(cp, "\"%s\" requires two arguments\n", argv[0]));

	return cp;
}

static char *
configme(char *cp, int argc, char **argv)
{
	if (argc < 2)
		strcat(cp, "Not enough arguments for CONFIG\n");
	else
		cp = isdConfig(cp, argc, argv);

	return strchr(cp, 0);
}
