/* -*- Mode: C; c-file-style: "gnu" -*-
   addr.c -- native methods for internet address stuff.
   Created: Chris Toshok <toshok@hungry.com>, 2-Aug-97
 */
/*
  This file is part of Japhar, the GNU Virtual Machine for Java Bytecodes.
  Japhar is a project of The Hungry Programmers, GNU, and OryxSoft.

  Copyright (C) 1997, 1998, 1999 The Hungry Programmers

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "jni.h"
#include "exceptions.h" /* for throw_Exception() */
#include "log.h"
#include "compat.h"

#include <string.h>
#include <stdlib.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#include <assert.h>
#ifdef HAVE_UTSNAME_H
#include <sys/utsname.h>
#endif
#ifdef HAVE_WINSOCK2_H
#include <winsock2.h>
#endif

#define MYLOG "Native"

/* XXX This function is not thread safe! */
static char *
my_hostname(void)
{
#ifdef HAVE_UTSNAME_H
  static struct utsname name;

  if (uname(&name) == 0)
    return name.nodename;
  else
#endif
    {
      return "localhost"; /* XXXX */
    }
}

JNIEXPORT jstring JNICALL
Java_java_net_InetAddressImpl_getLocalHostName(JNIEnv *env,
					       jobject obj)
{
  jstring str = (*env)->NewStringUTF(env, my_hostname());
  str = (*env)->NewGlobalRef(env, str);

  return str;
}

JNIEXPORT void JNICALL
Java_java_net_InetAddressImpl_makeAnyLocalAddress(JNIEnv *env,
						  jobject obj,
						  jobject addr)
{
  jclass clazz = (*env)->FindClass(env, "java/net/InetAddress");
  jfieldID addr_field = (*env)->GetFieldID(env, clazz, "address", "I");
  /*  jfieldID family_field = (*env)->GetFieldID(env, clazz, "family", "I"); */
  assert(NULL != addr_field);
  /*  assert(NULL != family_field); */
  (*env)->SetIntField(env, addr, addr_field, htonl(INADDR_ANY));
  /*  (*env)->SetIntField(env, addr, family_field, AF_INET); */
}

JNIEXPORT jobjectArray JNICALL
Java_java_net_InetAddressImpl_lookupAllHostAddr(JNIEnv *env,
						jobject obj,
						jstring hostname)
{
  struct hostent *host;
  const jbyte *hostname_bytes;
  jint hostname_length;
  char *hostname_str;
  jobject ret_array;
  jclass ret_array_clazz = (*env)->FindClass(env, "[B");
  int num_addresses;
  int index;

  hostname_bytes = (*env)->GetStringUTFChars(env, hostname, NULL);
  hostname_length = (*env)->GetStringUTFLength(env, hostname) + 1;
  hostname_str = (char*)malloc(hostname_length * sizeof(char));

  strncpy(hostname_str, (char*)hostname_bytes, hostname_length);

  (*env)->ReleaseStringUTFChars(env, hostname, hostname_bytes);

  JAVARLOG1(MYLOG, 3, "lookupAllHostAddr(%s)\n", hostname_str);

  host = gethostbyname(hostname_str);

  if (host == NULL)
    {
      throw_Exception(env, "java/net/UnknownHostException", hostname_str);
      free(hostname_str);
      return NULL;
    }
  free(hostname_str);

  num_addresses = 0;
  while (host->h_addr_list[num_addresses])
    {
      num_addresses ++;
    }

  ret_array = (*env)->NewObjectArray(env, num_addresses, ret_array_clazz, NULL);
  index = 0;
  while (index < num_addresses)
    {
      jobject sub_array = (*env)->NewByteArray(env, 4);
      jbyte *bytes;
      int i;

      bytes = (*env)->GetByteArrayElements(env, sub_array, NULL);

      for (i = 0; i < 4; i ++)
	bytes[i] = host->h_addr_list[index][i];

      (*env)->ReleaseByteArrayElements(env, sub_array, bytes, 0);

      sub_array = (*env)->NewGlobalRef(env, sub_array);
      (*env)->SetObjectArrayElement(env, ret_array, index, sub_array);

      index ++;
    }

  ret_array = (*env)->NewGlobalRef(env, ret_array);

  return ret_array;
}

JNIEXPORT jstring JNICALL
Java_java_net_InetAddressImpl_getHostByAddr(JNIEnv *env,
					    jobject obj,
					    jint addr)
{
  /*
   * This is modeled after kaffe 0.9.2.  I am not sure what this
   * function is supposed to do, or which params it take.
   * XXX This implementation should be verified.
   */
  struct hostent *entry = NULL;
  jstring str;
  jint nladdr = htonl(addr);

  /* XXX This method is not thread safe! */
  entry = gethostbyaddr((char*)&nladdr, sizeof(jint), AF_INET);

  if (NULL == entry)
    {
      char strbuffer[4*3+6]; /* Space for [NNN.NNN.NNN.NNN] */
      snprintf(strbuffer, sizeof(strbuffer), "[%o.%o.%o.%o]", 
	       (nladdr >> 24) & 0xf,
	       (nladdr >> 16) & 0xf,
	       (nladdr >> 8) & 0xf,
	       nladdr & 0xf);
      throw_Exception(env, "java/net/UnknownHostException", strbuffer);
      return NULL;
    }

  str = (*env)->NewStringUTF(env, entry->h_name);
  str = (*env)->NewGlobalRef(env, str);
  return str;
}

JNIEXPORT jint JNICALL
Java_java_net_InetAddressImpl_getInetFamily(JNIEnv *env,
					    jobject obj)
{
  return AF_INET;
}

/* JDK1.2 */
JNIEXPORT void JNICALL
Java_java_net_InetAddress_init(JNIEnv *env,
			       jclass cls)
{
  /* nothing to do */
}
