/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef XEN_HOST_H
#define XEN_HOST_H

#include <xen/api/xen_blob_decl.h>
#include <xen/api/xen_common.h>
#include <xen/api/xen_host_allowed_operations.h>
#include <xen/api/xen_host_cpu_decl.h>
#include <xen/api/xen_host_crashdump_decl.h>
#include <xen/api/xen_host_decl.h>
#include <xen/api/xen_host_metrics_decl.h>
#include <xen/api/xen_host_patch_decl.h>
#include <xen/api/xen_pbd_decl.h>
#include <xen/api/xen_pif_decl.h>
#include <xen/api/xen_sr_decl.h>
#include <xen/api/xen_string_blob_map.h>
#include <xen/api/xen_string_host_allowed_operations_map.h>
#include <xen/api/xen_string_set.h>
#include <xen/api/xen_string_string_map.h>
#include <xen/api/xen_vdi_decl.h>
#include <xen/api/xen_vdi_string_map.h>
#include <xen/api/xen_vm_decl.h>
#include <xen/api/xen_vm_string_set_map.h>


/*
 * The host class.
 * 
 * A physical host.
 */


/**
 * Free the given xen_host.  The given handle must have been allocated
 * by this library.
 */
extern void
xen_host_free(xen_host host);


typedef struct xen_host_set
{
    size_t size;
    xen_host *contents[];
} xen_host_set;

/**
 * Allocate a xen_host_set of the given size.
 */
extern xen_host_set *
xen_host_set_alloc(size_t size);

/**
 * Free the given xen_host_set.  The given set must have been allocated
 * by this library.
 */
extern void
xen_host_set_free(xen_host_set *set);


typedef struct xen_host_record
{
    xen_host handle;
    char *uuid;
    char *name_label;
    char *name_description;
    int64_t memory_overhead;
    struct xen_host_allowed_operations_set *allowed_operations;
    xen_string_host_allowed_operations_map *current_operations;
    int64_t api_version_major;
    int64_t api_version_minor;
    char *api_version_vendor;
    xen_string_string_map *api_version_vendor_implementation;
    bool enabled;
    xen_string_string_map *software_version;
    xen_string_string_map *other_config;
    struct xen_string_set *capabilities;
    xen_string_string_map *cpu_configuration;
    char *sched_policy;
    struct xen_string_set *supported_bootloaders;
    struct xen_vm_record_opt_set *resident_vms;
    xen_string_string_map *logging;
    struct xen_pif_record_opt_set *pifs;
    struct xen_sr_record_opt *suspend_image_sr;
    struct xen_sr_record_opt *crash_dump_sr;
    struct xen_host_crashdump_record_opt_set *crashdumps;
    struct xen_host_patch_record_opt_set *patches;
    struct xen_pbd_record_opt_set *pbds;
    struct xen_host_cpu_record_opt_set *host_cpus;
    xen_string_string_map *cpu_info;
    char *hostname;
    char *address;
    struct xen_host_metrics_record_opt *metrics;
    xen_string_string_map *license_params;
    struct xen_string_set *ha_statefiles;
    struct xen_string_set *ha_network_peers;
    xen_string_blob_map *blobs;
    struct xen_string_set *tags;
    char *external_auth_type;
    char *external_auth_service_name;
    xen_string_string_map *external_auth_configuration;
    char *edition;
    xen_string_string_map *license_server;
    xen_string_string_map *bios_strings;
    char *power_on_mode;
    xen_string_string_map *power_on_config;
    struct xen_sr_record_opt *local_cache_sr;
} xen_host_record;

/**
 * Allocate a xen_host_record.
 */
extern xen_host_record *
xen_host_record_alloc(void);

/**
 * Free the given xen_host_record, and all referenced values.  The
 * given record must have been allocated by this library.
 */
extern void
xen_host_record_free(xen_host_record *record);


typedef struct xen_host_record_opt
{
    bool is_record;
    union
    {
        xen_host handle;
        xen_host_record *record;
    } u;
} xen_host_record_opt;

/**
 * Allocate a xen_host_record_opt.
 */
extern xen_host_record_opt *
xen_host_record_opt_alloc(void);

/**
 * Free the given xen_host_record_opt, and all referenced values.  The
 * given record_opt must have been allocated by this library.
 */
extern void
xen_host_record_opt_free(xen_host_record_opt *record_opt);


typedef struct xen_host_record_set
{
    size_t size;
    xen_host_record *contents[];
} xen_host_record_set;

/**
 * Allocate a xen_host_record_set of the given size.
 */
extern xen_host_record_set *
xen_host_record_set_alloc(size_t size);

/**
 * Free the given xen_host_record_set, and all referenced values.  The
 * given set must have been allocated by this library.
 */
extern void
xen_host_record_set_free(xen_host_record_set *set);



typedef struct xen_host_record_opt_set
{
    size_t size;
    xen_host_record_opt *contents[];
} xen_host_record_opt_set;

/**
 * Allocate a xen_host_record_opt_set of the given size.
 */
extern xen_host_record_opt_set *
xen_host_record_opt_set_alloc(size_t size);

/**
 * Free the given xen_host_record_opt_set, and all referenced values. 
 * The given set must have been allocated by this library.
 */
extern void
xen_host_record_opt_set_free(xen_host_record_opt_set *set);


/**
 * Get a record containing the current state of the given host.
 */
extern bool
xen_host_get_record(xen_session *session, xen_host_record **result, xen_host host);


/**
 * Get a reference to the host instance with the specified UUID.
 */
extern bool
xen_host_get_by_uuid(xen_session *session, xen_host *result, char *uuid);


/**
 * Get all the host instances with the given label.
 */
extern bool
xen_host_get_by_name_label(xen_session *session, struct xen_host_set **result, char *label);


/**
 * Get the uuid field of the given host.
 */
extern bool
xen_host_get_uuid(xen_session *session, char **result, xen_host host);


/**
 * Get the name/label field of the given host.
 */
extern bool
xen_host_get_name_label(xen_session *session, char **result, xen_host host);


/**
 * Get the name/description field of the given host.
 */
extern bool
xen_host_get_name_description(xen_session *session, char **result, xen_host host);


/**
 * Get the memory/overhead field of the given host.
 */
extern bool
xen_host_get_memory_overhead(xen_session *session, int64_t *result, xen_host host);


/**
 * Get the allowed_operations field of the given host.
 */
extern bool
xen_host_get_allowed_operations(xen_session *session, struct xen_host_allowed_operations_set **result, xen_host host);


/**
 * Get the current_operations field of the given host.
 */
extern bool
xen_host_get_current_operations(xen_session *session, xen_string_host_allowed_operations_map **result, xen_host host);


/**
 * Get the API_version/major field of the given host.
 */
extern bool
xen_host_get_api_version_major(xen_session *session, int64_t *result, xen_host host);


/**
 * Get the API_version/minor field of the given host.
 */
extern bool
xen_host_get_api_version_minor(xen_session *session, int64_t *result, xen_host host);


/**
 * Get the API_version/vendor field of the given host.
 */
extern bool
xen_host_get_api_version_vendor(xen_session *session, char **result, xen_host host);


/**
 * Get the API_version/vendor_implementation field of the given host.
 */
extern bool
xen_host_get_api_version_vendor_implementation(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the enabled field of the given host.
 */
extern bool
xen_host_get_enabled(xen_session *session, bool *result, xen_host host);


/**
 * Get the software_version field of the given host.
 */
extern bool
xen_host_get_software_version(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the other_config field of the given host.
 */
extern bool
xen_host_get_other_config(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the capabilities field of the given host.
 */
extern bool
xen_host_get_capabilities(xen_session *session, struct xen_string_set **result, xen_host host);


/**
 * Get the cpu_configuration field of the given host.
 */
extern bool
xen_host_get_cpu_configuration(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the sched_policy field of the given host.
 */
extern bool
xen_host_get_sched_policy(xen_session *session, char **result, xen_host host);


/**
 * Get the supported_bootloaders field of the given host.
 */
extern bool
xen_host_get_supported_bootloaders(xen_session *session, struct xen_string_set **result, xen_host host);


/**
 * Get the resident_VMs field of the given host.
 */
extern bool
xen_host_get_resident_vms(xen_session *session, struct xen_vm_set **result, xen_host host);


/**
 * Get the logging field of the given host.
 */
extern bool
xen_host_get_logging(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the PIFs field of the given host.
 */
extern bool
xen_host_get_pifs(xen_session *session, struct xen_pif_set **result, xen_host host);


/**
 * Get the suspend_image_sr field of the given host.
 */
extern bool
xen_host_get_suspend_image_sr(xen_session *session, xen_sr *result, xen_host host);


/**
 * Get the crash_dump_sr field of the given host.
 */
extern bool
xen_host_get_crash_dump_sr(xen_session *session, xen_sr *result, xen_host host);


/**
 * Get the crashdumps field of the given host.
 */
extern bool
xen_host_get_crashdumps(xen_session *session, struct xen_host_crashdump_set **result, xen_host host);


/**
 * Get the patches field of the given host.
 */
extern bool
xen_host_get_patches(xen_session *session, struct xen_host_patch_set **result, xen_host host);


/**
 * Get the PBDs field of the given host.
 */
extern bool
xen_host_get_pbds(xen_session *session, struct xen_pbd_set **result, xen_host host);


/**
 * Get the host_CPUs field of the given host.
 */
extern bool
xen_host_get_host_cpus(xen_session *session, struct xen_host_cpu_set **result, xen_host host);


/**
 * Get the cpu_info field of the given host.
 */
extern bool
xen_host_get_cpu_info(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the hostname field of the given host.
 */
extern bool
xen_host_get_hostname(xen_session *session, char **result, xen_host host);


/**
 * Get the address field of the given host.
 */
extern bool
xen_host_get_address(xen_session *session, char **result, xen_host host);


/**
 * Get the metrics field of the given host.
 */
extern bool
xen_host_get_metrics(xen_session *session, xen_host_metrics *result, xen_host host);


/**
 * Get the license_params field of the given host.
 */
extern bool
xen_host_get_license_params(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the ha_statefiles field of the given host.
 */
extern bool
xen_host_get_ha_statefiles(xen_session *session, struct xen_string_set **result, xen_host host);


/**
 * Get the ha_network_peers field of the given host.
 */
extern bool
xen_host_get_ha_network_peers(xen_session *session, struct xen_string_set **result, xen_host host);


/**
 * Get the blobs field of the given host.
 */
extern bool
xen_host_get_blobs(xen_session *session, xen_string_blob_map **result, xen_host host);


/**
 * Get the tags field of the given host.
 */
extern bool
xen_host_get_tags(xen_session *session, struct xen_string_set **result, xen_host host);


/**
 * Get the external_auth_type field of the given host.
 */
extern bool
xen_host_get_external_auth_type(xen_session *session, char **result, xen_host host);


/**
 * Get the external_auth_service_name field of the given host.
 */
extern bool
xen_host_get_external_auth_service_name(xen_session *session, char **result, xen_host host);


/**
 * Get the external_auth_configuration field of the given host.
 */
extern bool
xen_host_get_external_auth_configuration(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the edition field of the given host.
 */
extern bool
xen_host_get_edition(xen_session *session, char **result, xen_host host);


/**
 * Get the license_server field of the given host.
 */
extern bool
xen_host_get_license_server(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the bios_strings field of the given host.
 */
extern bool
xen_host_get_bios_strings(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the power_on_mode field of the given host.
 */
extern bool
xen_host_get_power_on_mode(xen_session *session, char **result, xen_host host);


/**
 * Get the power_on_config field of the given host.
 */
extern bool
xen_host_get_power_on_config(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Get the local_cache_sr field of the given host.
 */
extern bool
xen_host_get_local_cache_sr(xen_session *session, xen_sr *result, xen_host host);


/**
 * Set the name/label field of the given host.
 */
extern bool
xen_host_set_name_label(xen_session *session, xen_host host, char *label);


/**
 * Set the name/description field of the given host.
 */
extern bool
xen_host_set_name_description(xen_session *session, xen_host host, char *description);


/**
 * Set the other_config field of the given host.
 */
extern bool
xen_host_set_other_config(xen_session *session, xen_host host, xen_string_string_map *other_config);


/**
 * Add the given key-value pair to the other_config field of the given
 * host.
 */
extern bool
xen_host_add_to_other_config(xen_session *session, xen_host host, char *key, char *value);


/**
 * Remove the given key and its corresponding value from the
 * other_config field of the given host.  If the key is not in that Map, then
 * do nothing.
 */
extern bool
xen_host_remove_from_other_config(xen_session *session, xen_host host, char *key);


/**
 * Set the logging field of the given host.
 */
extern bool
xen_host_set_logging(xen_session *session, xen_host host, xen_string_string_map *logging);


/**
 * Add the given key-value pair to the logging field of the given host.
 */
extern bool
xen_host_add_to_logging(xen_session *session, xen_host host, char *key, char *value);


/**
 * Remove the given key and its corresponding value from the logging
 * field of the given host.  If the key is not in that Map, then do nothing.
 */
extern bool
xen_host_remove_from_logging(xen_session *session, xen_host host, char *key);


/**
 * Set the suspend_image_sr field of the given host.
 */
extern bool
xen_host_set_suspend_image_sr(xen_session *session, xen_host host, xen_sr suspend_image_sr);


/**
 * Set the crash_dump_sr field of the given host.
 */
extern bool
xen_host_set_crash_dump_sr(xen_session *session, xen_host host, xen_sr crash_dump_sr);


/**
 * Set the hostname field of the given host.
 */
extern bool
xen_host_set_hostname(xen_session *session, xen_host host, char *hostname);


/**
 * Set the address field of the given host.
 */
extern bool
xen_host_set_address(xen_session *session, xen_host host, char *address);


/**
 * Set the tags field of the given host.
 */
extern bool
xen_host_set_tags(xen_session *session, xen_host host, struct xen_string_set *tags);


/**
 * Add the given value to the tags field of the given host.  If the
 * value is already in that Set, then do nothing.
 */
extern bool
xen_host_add_tags(xen_session *session, xen_host host, char *value);


/**
 * Remove the given value from the tags field of the given host.  If
 * the value is not in that Set, then do nothing.
 */
extern bool
xen_host_remove_tags(xen_session *session, xen_host host, char *value);


/**
 * Set the license_server field of the given host.
 */
extern bool
xen_host_set_license_server(xen_session *session, xen_host host, xen_string_string_map *license_server);


/**
 * Add the given key-value pair to the license_server field of the
 * given host.
 */
extern bool
xen_host_add_to_license_server(xen_session *session, xen_host host, char *key, char *value);


/**
 * Remove the given key and its corresponding value from the
 * license_server field of the given host.  If the key is not in that Map,
 * then do nothing.
 */
extern bool
xen_host_remove_from_license_server(xen_session *session, xen_host host, char *key);


/**
 * Puts the host into a state in which no new VMs can be started.
 * Currently active VMs on the host continue to execute.
 */
extern bool
xen_host_disable(xen_session *session, xen_host host);


/**
 * Puts the host into a state in which new VMs can be started.
 */
extern bool
xen_host_enable(xen_session *session, xen_host host);


/**
 * Shutdown the host. (This function can only be called if there are no
 * currently running VMs on the host and it is disabled.).
 */
extern bool
xen_host_shutdown(xen_session *session, xen_host host);


/**
 * Reboot the host. (This function can only be called if there are no
 * currently running VMs on the host and it is disabled.).
 */
extern bool
xen_host_reboot(xen_session *session, xen_host host);


/**
 * Get the host xen dmesg.
 */
extern bool
xen_host_dmesg(xen_session *session, char **result, xen_host host);


/**
 * Get the host xen dmesg, and clear the buffer.
 */
extern bool
xen_host_dmesg_clear(xen_session *session, char **result, xen_host host);


/**
 * Get the host's log file.
 */
extern bool
xen_host_get_log(xen_session *session, char **result, xen_host host);


/**
 * Inject the given string as debugging keys into Xen.
 */
extern bool
xen_host_send_debug_keys(xen_session *session, xen_host host, char *keys);


/**
 * Run xen-bugtool --yestoall and upload the output to Citrix support.
 */
extern bool
xen_host_bugreport_upload(xen_session *session, xen_host host, char *url, xen_string_string_map *options);


/**
 * List all supported methods.
 */
extern bool
xen_host_list_methods(xen_session *session, struct xen_string_set **result);


/**
 * Apply a new license to a host.
 */
extern bool
xen_host_license_apply(xen_session *session, xen_host host, char *contents);


/**
 * Create a new host record.
 */
extern bool
xen_host_create(xen_session *session, xen_host *result, char *uuid, char *name_label, char *name_description, char *hostname, char *address, char *external_auth_type, char *external_auth_service_name, xen_string_string_map *external_auth_configuration, xen_string_string_map *license_params, char *edition, xen_string_string_map *license_server);


/**
 * Destroy specified host record in database.
 */
extern bool
xen_host_destroy(xen_session *session, xen_host self);


/**
 * Attempt to power-on the host (if the capability exists).
 */
extern bool
xen_host_power_on(xen_session *session, xen_host host);


/**
 * Set the new license details in the database, trigger a recomputation
 * of the pool SKU.
 */
extern bool
xen_host_set_license_params(xen_session *session, xen_host self, xen_string_string_map *value);


/**
 * This call disables HA on the local host. This should only be used
 * with extreme care.
 */
extern bool
xen_host_emergency_ha_disable(xen_session *session);


/**
 * Disarms the fencing function of the HA subsystem. This function is
 * extremely dangerous and should only be used as part of a controlled
 * shutdown of the HA system.
 */
extern bool
xen_host_ha_disarm_fencing(xen_session *session, xen_host host);


/**
 * Attach statefiles, generate config files but do not start the xHA
 * daemon.
 */
extern bool
xen_host_preconfigure_ha(xen_session *session, xen_host host, struct xen_vdi_set *statefiles, xen_vdi metadata_vdi, char *generation);


/**
 * Block until this host joins the liveset.
 */
extern bool
xen_host_ha_join_liveset(xen_session *session, xen_host host);


/**
 * Prevents future failover decisions happening on this node. This
 * function should only be used as part of a controlled shutdown of the HA
 * system.
 */
extern bool
xen_host_ha_disable_failover_decisions(xen_session *session, xen_host host);


/**
 * Block until this host xHA daemon exits after having seen the invalid
 * statefile. If the host loses statefile access then throw an exception.
 */
extern bool
xen_host_ha_wait_for_shutdown_via_statefile(xen_session *session, xen_host host);


/**
 * Stops the HA daemon. This function is extremely dangerous and should
 * only be used as part of a controlled shutdown of the HA system.
 */
extern bool
xen_host_ha_stop_daemon(xen_session *session, xen_host host);


/**
 * Cleans up any resources on the host associated with this HA
 * instance.
 */
extern bool
xen_host_ha_release_resources(xen_session *session, xen_host host);


/**
 * Returns true if xapi appears to be functioning normally.
 */
extern bool
xen_host_ha_xapi_healthcheck(xen_session *session, bool *result);


/**
 * Returns nothing if this host is healthy, otherwise it throws an
 * error explaining why the host is unhealthy.
 */
extern bool
xen_host_local_assert_healthy(xen_session *session);


/**
 * Request this host performs a database backup.
 */
extern bool
xen_host_request_backup(xen_session *session, xen_host host, int64_t generation, bool force);


/**
 * Request this host syncs dom0 config files.
 */
extern bool
xen_host_request_config_file_sync(xen_session *session, xen_host host, char *hash);


/**
 * First phase of a two-phase commit protocol to set the new master. If
 * the host has already committed to another configuration or if the proposed
 * new master is not in this node's membership set then the call will return
 * an exception.
 */
extern bool
xen_host_propose_new_master(xen_session *session, char *address, bool manual);


/**
 * Second phase of a two-phase commit protocol to set the new master.
 */
extern bool
xen_host_commit_new_master(xen_session *session, char *address);


/**
 * Causes the new master transaction to abort.
 */
extern bool
xen_host_abort_new_master(xen_session *session, char *address);


/**
 * Start recording the specified data source.
 */
extern bool
xen_host_record_data_source(xen_session *session, xen_host host, char *data_source);


/**
 * Query the latest value of the specified data source.
 */
extern bool
xen_host_query_data_source(xen_session *session, double *result, xen_host host, char *data_source);


/**
 * Forget the recorded statistics related to the specified data source.
 */
extern bool
xen_host_forget_data_source_archives(xen_session *session, xen_host host, char *data_source);


/**
 * Check this host can be evacuated.
 */
extern bool
xen_host_assert_can_evacuate(xen_session *session, xen_host host);


/**
 * Return a set of VMs which prevent the host being evacuated, with
 * per-VM error codes.
 */
extern bool
xen_host_get_vms_which_prevent_evacuation(xen_session *session, xen_vm_string_set_map **result, xen_host self);


/**
 * Return a set of VMs which are not co-operating with the host's
 * memory control system.
 */
extern bool
xen_host_get_uncooperative_resident_vms(xen_session *session, struct xen_vm_set **result, xen_host self);


/**
 * Return the set of domain uuids which are not co-operating with the
 * host's memory control system.
 */
extern bool
xen_host_get_uncooperative_domains(xen_session *session, struct xen_string_set **result, xen_host self);


/**
 * Migrate all VMs off of this host, where possible.
 */
extern bool
xen_host_evacuate(xen_session *session, xen_host host);


/**
 * Signals that the management IP address or hostname has been changed
 * beneath us.
 */
extern bool
xen_host_signal_networking_change(xen_session *session);


/**
 * Notify an event.
 */
extern bool
xen_host_notify(xen_session *session, char *ty, char *params);


/**
 * Re-configure syslog logging.
 */
extern bool
xen_host_syslog_reconfigure(xen_session *session, xen_host host);


/**
 * Reconfigure the management network interface.
 */
extern bool
xen_host_management_reconfigure(xen_session *session, xen_pif pif);


/**
 * Reconfigure the management network interface. Should only be used if
 * Host.management_reconfigure is impossible because the network configuration
 * is broken.
 */
extern bool
xen_host_local_management_reconfigure(xen_session *session, char *interface);


/**
 * Disable the management network interface.
 */
extern bool
xen_host_management_disable(xen_session *session);


/**
 * .
 */
extern bool
xen_host_get_system_status_capabilities(xen_session *session, char **result, xen_host host);


/**
 * Return timing statistics for diagnostic purposes.
 */
extern bool
xen_host_get_diagnostic_timing_stats(xen_session *session, xen_string_string_map **result, xen_host host);


/**
 * Restarts the agent after a 10 second pause. WARNING: this is a
 * dangerous operation. Any operations in progress will be aborted, and
 * unrecoverable data loss may occur. The caller is responsible for ensuring
 * that there are no operations in progress when this method is called.
 */
extern bool
xen_host_restart_agent(xen_session *session, xen_host host);


/**
 * Shuts the agent down after a 10 second pause. WARNING: this is a
 * dangerous operation. Any operations in progress will be aborted, and
 * unrecoverable data loss may occur. The caller is responsible for ensuring
 * that there are no operations in progress when this method is called.
 */
extern bool
xen_host_shutdown_agent(xen_session *session);


/**
 * Sets the host name to the specified string.  Both the API and
 * lower-level system hostname are changed immediately.
 */
extern bool
xen_host_set_hostname_live(xen_session *session, xen_host host, char *hostname);


/**
 * Diagnostics call to discover if host is in emergency mode.
 */
extern bool
xen_host_is_in_emergency_mode(xen_session *session, bool *result);


/**
 * Computes the amount of free memory on the host.
 */
extern bool
xen_host_compute_free_memory(xen_session *session, int64_t *result, xen_host host);


/**
 * Computes the virtualization memory overhead of a host.
 */
extern bool
xen_host_compute_memory_overhead(xen_session *session, int64_t *result, xen_host host);


/**
 * Needs to be called every 30 seconds for the master to believe the
 * host is alive.
 */
extern bool
xen_host_tickle_heartbeat(xen_session *session, xen_string_string_map **result, xen_host host, xen_string_string_map *stuff);


/**
 * This causes the synchronisation of the non-database data (messages,
 * RRDs and so on) stored on the master to be synchronised with the host.
 */
extern bool
xen_host_sync_data(xen_session *session, xen_host host);


/**
 * This causes the RRDs to be backed up to the master.
 */
extern bool
xen_host_backup_rrds(xen_session *session, xen_host host, double delay);


/**
 * Create a placeholder for a named binary blob of data that is
 * associated with this host.
 */
extern bool
xen_host_create_new_blob(xen_session *session, xen_blob *result, xen_host host, char *name, char *mime_type);


/**
 * Call a XenAPI plugin on this host.
 */
extern bool
xen_host_call_plugin(xen_session *session, char **result, xen_host host, char *plugin, char *fn, xen_string_string_map *args);


/**
 * This call queries the host's clock for the current time.
 */
extern bool
xen_host_get_servertime(xen_session *session, time_t *result, xen_host host);


/**
 * This call queries the host's clock for the current time in the
 * host's local timezone.
 */
extern bool
xen_host_get_server_localtime(xen_session *session, time_t *result, xen_host host);


/**
 * Enable binary storage on a particular host, for storing RRDs,
 * messages and blobs.
 */
extern bool
xen_host_enable_binary_storage(xen_session *session, xen_host host);


/**
 * Disable binary storage on a particular host, deleting stored RRDs,
 * messages and blobs.
 */
extern bool
xen_host_disable_binary_storage(xen_session *session, xen_host host);


/**
 * This call enables external authentication on a host.
 */
extern bool
xen_host_enable_external_auth(xen_session *session, xen_host host, xen_string_string_map *config, char *service_name, char *auth_type);


/**
 * This call disables external authentication on the local host.
 */
extern bool
xen_host_disable_external_auth(xen_session *session, xen_host host, xen_string_string_map *config);


/**
 * Retrieves recommended host migrations to perform when evacuating the
 * host from the wlb server. If a VM cannot be migrated from the host the
 * reason is listed instead of a recommendation.
 */
extern bool
xen_host_retrieve_wlb_evacuate_recommendations(xen_session *session, xen_vm_string_set_map **result, xen_host self);


/**
 * Install an SSL certificate to this host.
 */
extern bool
xen_host_certificate_install(xen_session *session, xen_host host, char *name, char *cert);


/**
 * Remove an SSL certificate from this host.
 */
extern bool
xen_host_certificate_uninstall(xen_session *session, xen_host host, char *name);


/**
 * List all installed SSL certificates.
 */
extern bool
xen_host_certificate_list(xen_session *session, struct xen_string_set **result, xen_host host);


/**
 * Install an SSL certificate revocation list to this host.
 */
extern bool
xen_host_crl_install(xen_session *session, xen_host host, char *name, char *crl);


/**
 * Remove an SSL certificate revocation list from this host.
 */
extern bool
xen_host_crl_uninstall(xen_session *session, xen_host host, char *name);


/**
 * List all installed SSL certificate revocation lists.
 */
extern bool
xen_host_crl_list(xen_session *session, struct xen_string_set **result, xen_host host);


/**
 * Resync installed SSL certificates and CRLs.
 */
extern bool
xen_host_certificate_sync(xen_session *session, xen_host host);


/**
 * Get the installed server SSL certificate.
 */
extern bool
xen_host_get_server_certificate(xen_session *session, char **result, xen_host host);


/**
 * .
 */
extern bool
xen_host_update_pool_secret(xen_session *session, xen_host host, char *pool_secret);


/**
 * .
 */
extern bool
xen_host_update_master(xen_session *session, xen_host host, char *master_address);


/**
 * Statically attach VDIs on a host.
 */
extern bool
xen_host_attach_static_vdis(xen_session *session, xen_host host, xen_vdi_string_map *vdi_reason_map);


/**
 * Detach static VDIs from a host.
 */
extern bool
xen_host_detach_static_vdis(xen_session *session, xen_host host, struct xen_vdi_set *vdis);


/**
 * Set a key in the local DB of the host.
 */
extern bool
xen_host_set_localdb_key(xen_session *session, xen_host host, char *key, char *value);


/**
 * Change to another edition, or reactivate the current edition after a
 * license has expired. This may be subject to the successful checkout of an
 * appropriate license.
 */
extern bool
xen_host_apply_edition(xen_session *session, xen_host host, char *edition);


/**
 * Refresh the list of installed Supplemental Packs.
 */
extern bool
xen_host_refresh_pack_info(xen_session *session, xen_host host);


/**
 * Set the power-on-mode, host, user and password .
 */
extern bool
xen_host_set_power_on_mode(xen_session *session, xen_host self, char *power_on_mode, xen_string_string_map *power_on_config);


/**
 * Set the CPU features to be used after a reboot, if the given
 * features string is valid.
 */
extern bool
xen_host_set_cpu_features(xen_session *session, xen_host host, char *features);


/**
 * Remove the feature mask, such that after a reboot all features of
 * the CPU are enabled.
 */
extern bool
xen_host_reset_cpu_features(xen_session *session, xen_host host);


/**
 * Purge all network-related metadata associated with the given host.
 */
extern bool
xen_host_reset_networking(xen_session *session, xen_host host);


/**
 * Enable the use of a local SR for caching purposes.
 */
extern bool
xen_host_enable_local_storage_caching(xen_session *session, xen_host host, xen_sr sr);


/**
 * Disable the use of a local SR for caching purposes.
 */
extern bool
xen_host_disable_local_storage_caching(xen_session *session, xen_host host);


/**
 * Return a list of all the hosts known to the system.
 */
extern bool
xen_host_get_all(xen_session *session, struct xen_host_set **result);


#endif
