/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */


#include <stddef.h>
#include <stdlib.h>

#include "xen_internal.h"
#include <xen/api/xen_common.h>
#include <xen/api/xen_string_string_map.h>
#include <xen/api/xen_vif_metrics.h>


XEN_FREE(xen_vif_metrics)
XEN_SET_ALLOC_FREE(xen_vif_metrics)
XEN_ALLOC(xen_vif_metrics_record)
XEN_SET_ALLOC_FREE(xen_vif_metrics_record)
XEN_ALLOC(xen_vif_metrics_record_opt)
XEN_RECORD_OPT_FREE(xen_vif_metrics)
XEN_SET_ALLOC_FREE(xen_vif_metrics_record_opt)


static const struct_member xen_vif_metrics_record_struct_members[] =
    {
        { .key = "uuid",
          .type = &abstract_type_string,
          .offset = offsetof(xen_vif_metrics_record, uuid) },
        { .key = "io_read_kbs",
          .type = &abstract_type_float,
          .offset = offsetof(xen_vif_metrics_record, io_read_kbs) },
        { .key = "io_write_kbs",
          .type = &abstract_type_float,
          .offset = offsetof(xen_vif_metrics_record, io_write_kbs) },
        { .key = "last_updated",
          .type = &abstract_type_datetime,
          .offset = offsetof(xen_vif_metrics_record, last_updated) },
        { .key = "other_config",
          .type = &abstract_type_string_string_map,
          .offset = offsetof(xen_vif_metrics_record, other_config) }
    };

const abstract_type xen_vif_metrics_record_abstract_type_ =
    {
       .typename = STRUCT,
       .struct_size = sizeof(xen_vif_metrics_record),
       .member_count =
           sizeof(xen_vif_metrics_record_struct_members) / sizeof(struct_member),
       .members = xen_vif_metrics_record_struct_members
    };


void
xen_vif_metrics_record_free(xen_vif_metrics_record *record)
{
    if (record == NULL)
    {
        return;
    }
    free(record->handle);
    free(record->uuid);
    xen_string_string_map_free(record->other_config);
    free(record);
}


bool
xen_vif_metrics_get_record(xen_session *session, xen_vif_metrics_record **result, xen_vif_metrics vif_metrics)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics }
        };

    abstract_type result_type = xen_vif_metrics_record_abstract_type_;

    *result = NULL;
    XEN_CALL_("VIF_metrics.get_record");

    if (session->ok)
    {
       (*result)->handle = xen_strdup_((*result)->uuid);
    }

    return session->ok;
}


bool
xen_vif_metrics_get_by_uuid(xen_session *session, xen_vif_metrics *result, char *uuid)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = uuid }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF_metrics.get_by_uuid");
    return session->ok;
}


bool
xen_vif_metrics_get_io_read_kbs(xen_session *session, double *result, xen_vif_metrics vif_metrics)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics }
        };

    abstract_type result_type = abstract_type_float;

    XEN_CALL_("VIF_metrics.get_io_read_kbs");
    return session->ok;
}


bool
xen_vif_metrics_get_io_write_kbs(xen_session *session, double *result, xen_vif_metrics vif_metrics)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics }
        };

    abstract_type result_type = abstract_type_float;

    XEN_CALL_("VIF_metrics.get_io_write_kbs");
    return session->ok;
}


bool
xen_vif_metrics_get_last_updated(xen_session *session, time_t *result, xen_vif_metrics vif_metrics)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics }
        };

    abstract_type result_type = abstract_type_datetime;

    XEN_CALL_("VIF_metrics.get_last_updated");
    return session->ok;
}


bool
xen_vif_metrics_get_other_config(xen_session *session, xen_string_string_map **result, xen_vif_metrics vif_metrics)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics }
        };

    abstract_type result_type = abstract_type_string_string_map;

    *result = NULL;
    XEN_CALL_("VIF_metrics.get_other_config");
    return session->ok;
}


bool
xen_vif_metrics_set_other_config(xen_session *session, xen_vif_metrics vif_metrics, xen_string_string_map *other_config)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics },
            { .type = &abstract_type_string_string_map,
              .u.set_val = (arbitrary_set *)other_config }
        };

    xen_call_(session, "VIF_metrics.set_other_config", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_vif_metrics_add_to_other_config(xen_session *session, xen_vif_metrics vif_metrics, char *key, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics },
            { .type = &abstract_type_string,
              .u.string_val = key },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "VIF_metrics.add_to_other_config", param_values, 3, NULL, NULL);
    return session->ok;
}


bool
xen_vif_metrics_remove_from_other_config(xen_session *session, xen_vif_metrics vif_metrics, char *key)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics },
            { .type = &abstract_type_string,
              .u.string_val = key }
        };

    xen_call_(session, "VIF_metrics.remove_from_other_config", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_vif_metrics_get_all(xen_session *session, struct xen_vif_metrics_set **result)
{

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    xen_call_(session, "VIF_metrics.get_all", NULL, 0, &result_type, result);
    return session->ok;
}


bool
xen_vif_metrics_get_uuid(xen_session *session, char **result, xen_vif_metrics vif_metrics)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = vif_metrics }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("VIF_metrics.get_uuid");
    return session->ok;
}
