/*
 * Copyright (c) 2004, 2005 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>
#include <time.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>

#include <arpa/inet.h>
#include <netinet/in.h>

#include "mdd.h"
#include "config.h"
#include "packets.h"

struct mdd_stat {
	in_addr_t rcv;
	long long resend;
};

void
sniffer(group_addr, stat)
	in_addr_t group_addr;
	int stat;
{
	int rd;
	struct sockaddr_in recv_addr;
	struct mdd_packet packet;
	in_addr_t src_addr;
	struct ip_mreq m;
	int water_mark;
	int on = 1;
	struct mdd_stat *mdd_stat = NULL;
	struct mdd_stat *ms = NULL;
	struct mdd_stat *sms = NULL;
	int statcount = 1;
	int count = 0;
	long long total = 0;
	time_t finish = 0;


	if ((rd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) {
		printf("Cannot create UDP/IP socket: %s\n", strerror(errno));
		exit(-1);
	}

	if ((setsockopt(rd, SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on))) == -1) {
		perror("cannot set SO_REUSEPORT");
		exit(-1);
	}

	src_addr = local_addr(group_addr);
	if (debug) {
		struct in_addr sa;

		sa.s_addr = src_addr;
		printf("setting IP_MULTICAST_IF for %s\n", inet_ntoa(sa));
	}

	if ((setsockopt(rd, IPPROTO_IP, IP_MULTICAST_IF, &src_addr, 
	    sizeof(src_addr))) == -1) {
		perror("cannot set IP_MULTICAST_IF");
		exit(-1);
	}
	bzero(&m, sizeof(m));
	m.imr_multiaddr.s_addr = group_addr;
	if ((setsockopt(rd, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
	    &m, sizeof(m))) == -1) {
		perror("cannot set IP_ADD_MEMBERSHIP");
		exit(-1);
	}

	(void)memset(&recv_addr, 0, sizeof(recv_addr));
	recv_addr.sin_family = AF_INET;
	recv_addr.sin_len = sizeof(recv_addr);
	recv_addr.sin_addr.s_addr = INADDR_ANY;
	recv_addr.sin_port = htons(RESEND_REQ_PORT);
	if ((bind(rd, (struct sockaddr *)&recv_addr, 
	    sizeof(recv_addr))) == -1) {
		printf("Cannot bind to UDP port %d: %s\n",
		    RESEND_REQ_PORT, strerror(errno));
		exit(-1);
	}

	water_mark = sizeof(struct mdd_packet);
	if ((setsockopt(rd, SOL_SOCKET, SO_RCVLOWAT, 
	    &water_mark, sizeof(water_mark))) == -1) {
		perror("cannot set recveive low water mark");
		exit(-1);
	}

	if (stat) {
		finish = time(NULL) + 1;
		mdd_stat = malloc(statcount * sizeof(*mdd_stat));
		if (mdd_stat == NULL) {
			perror("malloc failed");
			exit(-1);
		}
		bzero(mdd_stat, statcount * sizeof(*mdd_stat));
	}

	while(1) {
		struct sockaddr_in from_sin;
		char from_str[20];
		size_t from_salen;

		from_salen = sizeof(from_sin);
		if (recvfrom(rd, &packet, sizeof(packet), 
		    0, (struct sockaddr *)&from_sin, &from_salen) > 0) {
			inet_ntop(AF_INET, &from_sin.sin_addr,
			    from_str, sizeof(from_str));

			if (debug) {
				printf("Packet received from %s\n", from_str);
				printf("operation = %d\n", 
				    packet.mdd_operation);
				printf("offset = %lld\n", packet.mdd_offset);
			}

			if (stat) {
				in_addr_t msa;

				msa = from_sin.sin_addr.s_addr;
				for (ms = mdd_stat; ms->rcv; ms++) {
					if (ms->rcv == msa) {
						ms->resend++;
						break;
					}
				}

				if (ms->rcv == 0x000000) { /* not found */
					size_t len;

					statcount++;
					len = sizeof(*ms) * statcount;
					mdd_stat = realloc(mdd_stat, len);
					if (mdd_stat == NULL) {
						perror("malloc failed");
						exit(-1);
					}
					
					mdd_stat[statcount - 2].rcv = msa;
					mdd_stat[statcount - 2].resend = 1;

					mdd_stat[statcount - 1].rcv = 0x000000;
					mdd_stat[statcount - 1].resend = 0;
				}

				if (time(NULL) > finish) 
					goto display;
			} else {
				printf("resend request from %s -> %lld\n", 
				    from_str, packet.mdd_offset);
			}
		}
	}

display:
	/* Only stat reaches there */
	if ((sms = malloc(statcount * sizeof(*sms))) == NULL) {
		perror("malloc failed");
		exit(1);
	}

	for (count = 0; count < statcount; count++) {
		struct mdd_stat top_ms_rec = { 0x00000000, 0 };
		struct mdd_stat *top_ms = &top_ms_rec;

		for (ms = mdd_stat; ms->rcv; ms++)
			if (ms->resend > top_ms->resend)
				top_ms = ms;

		memcpy(&sms[count], top_ms, sizeof(*top_ms));
		total += top_ms->resend;
		top_ms->resend = 0;
		top_ms->rcv = 0x000000;
	}
	
	if (total == 0) {
		printf("No resend request received\n");
		return;
	}

	for (count = 0; count < statcount; count++) {
		struct in_addr addr;

		if (sms[count].rcv == 0x0000)
			continue;

		addr.s_addr = sms[count].rcv;
		printf("%s: %lld/%lld (%lld%%)\n", inet_ntoa(addr),
		    sms[count].resend, total, 
		    (100 * sms[count].resend / total));
	}
	return;
}
