/* Copleft@2004,2005 - pancake@phreaker.net */

#include "main.h"
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "iface.h"

#ifndef VERSION
#error VERSION variable not defined
#endif

enum {
	MODE_NULL = 0,
	MODE_FILE,
	MODE_DISK,
	MODE_IFACE,
	MODE_OFACE,
	MODE_EFACE,
	MODE_MEM,
	MODE_STDIN,
	MODE_SNIFF,
	MODE_INVAL
};

int  mode     = MODE_NULL;
int  count    = -1;
bool verbose  = false;
char *target  = NULL;
char *iface   = NULL;
bool promisc  = false;
bool transfer = false;
bool decimal  = false;
bool average  = false;
bool stdinvals= false;
int  mysleep  = 1;

char vstring[10]; /* KB/s GB/s strings.. */
unsigned int kilobah = 1;
int n = 0;
FILE *fd;

int marked=0;
#if HAVE_GMP
mpz_t one, two, foo, tmp;
#else
long one, two, foo, tmp;
#endif

void mesure_signal(int sig)
{
	switch(mode)
	{
	case MODE_STDIN:
		stdin_finalize();
		break;
	case MODE_SNIFF:
		sniff_finalize();
		break;
	}

	fcntl(0, F_SETFL, O_RDWR);

	exit(0);
}
void mesure_atexit()
{
	if (!n) printf("\n");
	mesure_signal(0);
}

int show_usage(char *argv0)
{
	printf(
		"Usage '%s' [options] v%.1f   // pancake AT phreaker DOT net\n"
		" >sources:\n"
		"  -i iface     - Stats of input iface.\n"
		"  -e iface     - Stats of errors on iface.\n"
		"  -o iface     - Stats of output iface.\n"
		"  -f file      - Stats of file.\n"
		"  -d diskmp    - Stats of target mountpoint.\n"
#if HAVE_PCAP
		"  -p pcapexp   - Stats of matching packets. (use -i to specify the iface).\n"
#endif
		"  -m           - Stats of memory.\n"
		"  -z           - Measure stdin transfers.\n"
		"  -Z           - Read values from stdin.\n"
		"  -ZZ          - Read values from stdin and adds the result to other mode.\n"
		" >options:\n"
		"  -c count     - Repeat /count/ times. (default=-1)\n"
		"  -s sleep     - Sleep between intervals. (default=0)\n"
		"  -t           - Show transfer instead of bitrate\n"
		"  -a           - Measure average (implies -t)\n"
		"  -v           - Verbose (append KB/s MB/s...)\n"
		"  -n           - Use \\n instead of \\r\n"
		"  -D           - Show large numbers in dot format.\n"
		"  -P           - Promiscuous mode (sniffer mode (-p))\n"
		"  -K           - Show sizes in KB\n"
		"  -M           - Show sizes in MB\n"
		"  -G           - Show sizes in GB\n"
	,argv0,VERSION);

	return 1;
}


int
do_work()
{
	if (count != -1)
		count--;

	alarm(mysleep);

#if HAVE_GMP
	if (mpz_cmp_d(two,0) != 0)
	{
		if (average)
		{
			mpz_sub(tmp,one,two); // (one-two)
			mpz_fdiv_q_ui(tmp,tmp,mysleep); // (one-two)/mysleep
			mpz_add(tmp,foo,tmp);
			mpz_fdiv_q_ui(foo,tmp,kilobah); // ((one-two)/mysleep)/kilobah)
			mpz_set(tmp,foo);
			mpz_fdiv_q_ui(foo,tmp,2.0f);
		} else {
			mpz_sub(tmp,one,two); // (one-two)
			mpz_fdiv_q_ui(foo,tmp,mysleep); // (one-two)/mysleep
			mpz_fdiv_q_ui(foo,tmp,kilobah); // ((one-two)/mysleep)/kilobah)
		}		
#else
	if (two)
	{
		if (average)
		{
			foo = (foo+((one-two)/mysleep)/(kilobah))/2;
		} else {
			foo = ((one-two)/mysleep)/(kilobah);
		}
#endif
	}


#if HAVE_GMP
	mpz_set(two,one);
#else
	two = one;
#endif

	switch(mode)
	{
	case MODE_IFACE:
#if HAVE_GMP
		iface_get_value(target,IFACE_IN,&one);
#else
		one = iface_get_value(target,IFACE_IN);
#endif
		break;
	case MODE_OFACE:
#if HAVE_GMP
		iface_get_value(target,IFACE_OUT,&one);
#else
		one = iface_get_value(target,IFACE_OUT);
#endif
		break;
	case MODE_EFACE:
#if HAVE_GMP
		iface_get_value(target,IFACE_ERR,&one);
#else
		one = iface_get_value(target,IFACE_ERR);
#endif
		break;
	case MODE_FILE:
#if HAVE_GMP
		file_get_value(target,&one);
#else
		one = file_get_value(target);
#endif
		break;
	case MODE_MEM:
#if HAVE_GMP
		mpz_set_d(one, mem_get_value());
#else
		one = mem_get_value();
#endif
		break;
	case MODE_DISK:
#if HAVE_GMP
		disk_get_value(target,&one);
#else
		one = disk_get_value(target);
#endif
		break;
	case MODE_STDIN:
#if HAVE_GMP
		mpz_set_d(one,stdin_get_value());
#else
		one = stdin_get_value();
#endif
		break;
	case MODE_INVAL:
#if HAVE_GMP
#else
		scanf("%ld",&one);
#endif
		break;
	case MODE_SNIFF:
#if HAVE_PCAP
#if HAVE_GMP
		sniff_get_value(&one);
#else
		one = sniff_get_value();
#endif
#else
		fprintf(stderr,"This binary was compiled without sniffer mode\n");
		exit(1);
#endif
		break;
	default:
		fprintf(stderr,"Mode not yet supported. Try next release.\n");
		exit(1);
	}

if (stdinvals)
{
	fcntl(0, F_SETFL, O_RDWR|O_NONBLOCK);
#if HAVE_GMP
	mpz_set_d(tmp,0); //?
	gmp_scanf("%Zd",&tmp);
	mpz_add(one,one,tmp);
#else
	tmp = 0; //?
	scanf("%ld",&tmp);
	one += tmp;
#endif
}

// exits with ^D
if (feof(stdin))
{
	exit(0);
}

#if HAVE_GMP
	if (transfer)
	{
		mpz_set(tmp,foo);
	} else {
		mpz_fdiv_q_ui(tmp,one,kilobah);
	}

	if (decimal)
	{
		char string[1024];
		int i,j,dot;

		gmp_snprintf(string,1023,"%Zd",tmp);
		string[1023]='\0';
		dot = strlen(string)%3;
		if (dot == 0) dot = 3;

		for (i=0;string[i]!='\0';i++,dot--)
		{
			if (dot <= 0)
			{
				dot = 4;
				for(j=strlen(string);j>=i;j--)
					string[j+1] = string[j];
				string[i] = '.';
			}
		}
		
		fprintf(fd,"                    \r%s %s ",string,vstring);
	} else {
		gmp_fprintf(fd,"                    \r%Zd %s ",tmp,vstring);
	}

#else
	if (decimal)
	{
		char string[1024];
		int i,j,dot;

		sprintf(string,"%ld", (transfer)?foo:one/kilobah);
		string[1023]='\0';
		dot = strlen(string)%3;
		if (dot == 0) dot = 3;

		for (i=0;string[i]!='\0';i++,dot--)
		{
			if (dot <= 0)
			{
				dot = 4;
				for(j=strlen(string);j>=i;j--)
					string[j+1] = string[j];
				string[i] = '.';
			}
		}
		
		fprintf(fd,"                    \r%s %s ",string,vstring);
	} else {
		fprintf(fd,"                    \r%ld %s ",
			(transfer)?foo:one/kilobah,vstring);
	}
#endif

	if (n) fprintf(fd,"\n");
	fflush(fd);

	return;
}

int
main(int argc,char **argv)
{
	int c;
	off_t o,t;
	int sec=1;
	fd = stdout;

	/* final hooks */
	atexit(mesure_atexit);
	signal(SIGINT,mesure_signal);

	while((c = getopt(argc,argv,"i:o:e:f:d:c:s:zZp:tKMGDavPmn")) != -1)
	{
		switch(c)
		{
		case 'v':
			verbose = true;
			break;
		case 'i':
			if (mode == MODE_SNIFF)
			{
				iface = optarg;
			} else {
				mode = MODE_IFACE;
				target = optarg;
			}
			break;
		case 'o':
			mode   = MODE_OFACE;
			target = optarg;
			break;
		case 'e':
			mode   = MODE_EFACE;
			target = optarg;
			break;
		case 'f':
			mode   = MODE_FILE;
			target = optarg;
			break;
		case 'd':
			mode   = MODE_DISK;
			target = optarg;
			break;
		case 'm':
			mode   = MODE_MEM;
			break;
		case 'p':
			if (mode!=MODE_NULL)
			{
				iface = target;
			}
			mode   = MODE_SNIFF;
			target = optarg;
			break;
		case 'z':
			mode = MODE_STDIN;
			if (stdinvals)
			{
				fprintf(stderr,"Cannot use z and ZZ modes together.\n");
				return (1);
			}
			fcntl(0, F_SETFL, O_RDWR);
			fd = stderr;
			break;
		case 'Z':
			if (mode == MODE_STDIN)
			{
				fprintf(stderr,"Cannot use z and ZZ modes together.\n");
				return (1);
			}
			if (mode == MODE_INVAL)
			{
				mode = MODE_NULL;
				fcntl(0, F_SETFL, O_RDWR|O_NONBLOCK);
				stdinvals = true;
			} else {
				mode = MODE_INVAL;
			}
			break;
		case 'c':
			count = atoi(optarg);
			break;
		case 'a':
			average = true;
			transfer = true;
			break;
		case 's':
			// TODO allow runtime
			mysleep=atoi(optarg);
			break;
		case 'D':
			decimal = 1;
			break;
		case 'P':
			promisc = true;
			break;
		case 't':
			transfer = true;
			break;
		case 'n':
			n = true;
			break;
		case 'K':
			kilobah = 1024;
			break;
		case 'M':
			kilobah = 1024*1024;
			break;
		case 'G':
			kilobah = 1024*1024*1024;
			break;
		default: // error found
			mode = MODE_NULL;
			break;
		}
	}

	if (mode == MODE_NULL)
	{
		return
		show_usage(argv[0]);
	}

	if (verbose)
	{
		switch (kilobah)
		{
		case 1:
			strcpy(vstring,"B");
			break;
		case 1024:
			strcpy(vstring,"KB");
			break;
		case 1024*1024:
			strcpy(vstring,"MB");
			break;
		case 1024*1024*1024:
			strcpy(vstring,"GB");
			break;
		}

		if (transfer)
			strcat(vstring,"/s");

	} else {
		vstring[0]='\0';
	}

	signal(SIGALRM,(void *)do_work);

	/* init counters */
#if HAVE_GMP
	mpz_init(one);
	mpz_init(two);
	mpz_init(foo);
	mpz_init(tmp);
#else
	one = 0;
	two = 0;
	foo = 0;
	tmp = 0;
#endif

	do_work();

	while(count!=0)
	{
		usleep(100);
	}

#if HAVE_GMP
	mpz_clear(one);
	mpz_clear(two);
	mpz_clear(foo);
	mpz_clear(tmp);
#endif
	return 0;
}

