/*
 * ninjacmd.c:
 *
 * this file contains code dealing with /commands for ninja irc
 *
 * written by Kraig Amador and Joshua J. Drake
 *
 */

#include "irc.h"
#include "dma.h"
#include "ircterm.h"
#include "vars.h"
#include "ircaux.h"
#include "server.h"
#include "hook.h"
#include "window.h"
#include "screen.h"
#include "output.h"
#include "whois.h"
#include "parse.h"
#include "log.h"
#include "notify.h"

#include "channels.h"
#include "bans.h"
#include "info.h"
#include "friends.h"
#include "enemies.h"
#include "hosts.h"
#include "how_many.h"
#include "grep.h"
#include "orignick.h"
#include "ninja.h"
#include "ckey.h"
#include "mileston.h"
#include "irc_comm.h"
#include "ninjacmd.h"

/* little thingy to help /mdop out */
static int nodeop _((u_char **, u_char *));

/* stuff in edit.c */
extern void send_action _((u_char *, u_char *));

/* nasty global var!! */
int in_update_cmd = 0;

/*
 * does /whois but with the nickname twice
 */
void
whois2(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *nick;

   if (args && *args)
      nick = args;
   else
      nick = get_server_nickname(from_server);
   send_to_server("%s %s %s", command, nick, nick);
}

/*
 * chhost() changes the source host name and tells the user
 * to reconnect to the server..
 */
void
chhost(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *nhn;

   if ((nhn = next_arg(args, &args)) == NULL)
     {
	usage("chhost", "<new host name>");
	return;
     }
   set_var_value(IRCHOST_VAR, nhn);
   put_info("Use /reconnect to make changes effective.");
}

/*
 *  chuname() changes the username (optionally the hostname and realname field)
 */
void
chuname(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *nun, *nhn = UNULL;
   u_char *identfn;

   nun = next_arg(args, &args);
   if (nun)
     {
	nhn = my_rindex(nun, '@');
	if (nhn)
	   *nhn++ = '\0';
     }

   /* check for a username */
   if (!nun || my_strlen(nun) < 1)
     {
	usage(command, "<new user name>[@<host>] [<real name field>]");
	return;
     }

   /* is there a hostname too? */
   if (nhn)
     {
	if (my_strlen(nhn) > 0)
	   set_var_value(IRCHOST_VAR, nhn);
	else
	  {
	     /* zero length host? */
	     put_error("Please specify a hostname after the @.");
	     return;
	  }
     }

   /* make the username active */
   my_strmcpy(username, nun, NAME_LEN - 1);

   /* if we have an ident file name, update it */
   if ((identfn = get_string_var(HACKED_IDENTD_VAR)) != NULL)
     {
	FILE *identf;
	u_char *ex_idfn = NULL;

	if (*identfn == '~')
	   ex_idfn = expand_twiddle(identfn);
	if (!ex_idfn)
	   ex_idfn = identfn;

	/* update it */
	identf = fopen(ex_idfn, "w");
	if (identf == NULL)
	   put_error("Unable to write to your HACKED_IDENTD file: %s", strerror(errno));
	else
	  {
	     fprintf(identf, "%s\n", nun);
	     fflush(identf);
	     fclose(identf);
	     put_info("Identity file, \"%s\", updated.", identfn);
	  }
	if (ex_idfn != identfn)
	   dma_Free(&ex_idfn);
     }
   /* just a username change? */
   if (nhn == NULL)
      put_info("Changed your user name to: %s", nun);
   else				/* both user and host */
      put_info("Changed your user@host to: %s@%s", nun, nhn);

   /* do we have a realname too?! */
   if (args && *args)
     {
	set_string_var(REALNAME_VAR, realname);
	put_info("Changed real name field to: %s", realname);
     }

   put_info("Use /reconnect to make changes effective.");
}

/*
 * /about command
 */
void
ninja_about(command, args, subargs)
   u_char *command, *args, *subargs;
{
   put_info("Ninja IRC v%s, build #%s by %s@%s on %s [%s]",
	    irc_version, 
	    compile_num, compile_user, compile_host, compile_time,
	    ninja_release);
   put_info("");
   put_info("   started in 1997 with ircII 2.8.2 to");
   put_info("   pass some time.  i wanted to make");
   put_info("   ircII look better and have extra features");
   put_info("   without the slowness of the scripts...");
   put_info("");
   put_info("ninja contributors");
   put_info("");
   put_info("   Joshua J. Drake         Kraig Amador");
   put_info("");
   put_info("bug reports and feedback should be aimed at");
   put_info("IRC: jduck@EFNet or EMAIL: ninja@qoop.org");
}

/*
 * /uptime command
 */
void
uptime(command, args, subargs)
   u_char *command, *args, *subargs;
{
   time_t uptime_val = time(NULL) - start_time;

   put_info("Your client has been running %s.", ninja_etime(uptime_val));
   return;
}

/*
 * /update command
 */
void
update_cmd(command, args, subargs)
   u_char *command, *args, *subargs;
{
   extern int checked_for_update;

   checked_for_update = 1;
   in_update_cmd = 1;
   ninja_check_update(0);
#ifndef NON_BLOCKING_CONNECTS
   in_update_cmd = 0;
#endif
}

#ifdef ALLOW_CHAN_RESYNC
/*
 * re-sync a channel
 *
 * this is mostly just a debug command..
 * if you thing its usefull enough, compile it in...
 */
void
resync(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *chan;
   Channel *tmp;

   if ((chan = next_arg(args, &args)) == NULL)
     {
	usage(command, "<channel>");
	return;
     }
   if ((tmp = lookup_channel(make_chan(chan), from_server, 0)) == (Channel *) NULL)
     {
	put_error("You're not on %s", chan);
	return;
     }
   tmp->status = 0;
   send_to_server("NAMES %s", tmp->channel);
   send_to_server("WHO %s", tmp->channel);
   send_to_server("MODE %s", tmp->channel);
   send_to_server("MODE %s b", tmp->channel);
   /*
      if (my_index(server_list[from_server].cmodes, 'e'))
      send_to_server("MODE %s e", tmp->channel);
      if (my_index(server_list[from_server].cmodes, 'd'))
      send_to_server("MODE %s d", tmp->channel);
    */
}
#endif

/*
 * /sc command..
 */
void
ninja_scan(command, args, subargs)
   u_char *command, *args, *subargs;
{
   Channel *chan;
   Nick *nick;
   u_char *channel = NULL, ltb[64], tmpbuf1[1024];
   int count = 0, vcount = 0, ocount = 0;
   float op_pc, voi_pc;

   if (args && *args)
      channel = make_chan(next_arg(args, &args));
   else if (!get_channel_by_refnum(0))
     {
	put_info("You are not on a channel!");
	return;
     }
   else
      channel = get_channel_by_refnum(0);
   if ((chan = lookup_channel(channel, from_server, 0)))
     {
	u_char *nickstr = NULL, *space = " ", *at = "@", *plus = "+";

	if (!(chan->status & CHAN_WHO))
	  {
	     put_info("Sorry, %s is not totally synched yet, please wait...", chan->channel);
	     return;
	  }
	for (nick = chan->nicks; nick; nick = nick->next)
	  {
	     if ((nick->status & NICK_CHOP))
		dma_strcat(&nickstr, at);
	     else if ((nick->status & NICK_VOICE))
		dma_strcat(&nickstr, plus);
	     dma_strcat(&nickstr, nick->nick);
	     if (nick->next)
		dma_strcat(&nickstr, space);
	  }
	if (do_hook(NAMES_LIST, "%s %s", chan->channel, nickstr))
	  {
	     for (nick = chan->nicks; nick; nick = nick->next)
	       {
		  if ((nick->status & NICK_VOICE) && !(nick->status & NICK_CHOP))
		     vcount++;
		  if (nick->status & NICK_CHOP)
		     ocount++;
		  count++;
	       }
	     op_pc = ((float) ocount * 100.0) / (float) count;
	     voi_pc = ((float) vcount * 100.0) / (float) count;

	     snprintf(tmpbuf1, sizeof(tmpbuf1) - 1,
		      "--[ Users(%d) on %s ]-[ %.1f%% ops(%d), %.1f%% voiced(%d) ]--",
		      count, chan->channel, op_pc, ocount, voi_pc, vcount);
	     tmpbuf1[sizeof(tmpbuf1)-1] = '\0';
	     put_raw("%s%s", tmpbuf1, strfill('-', current_screen->co - my_strlen(tmpbuf1) - 2));

	     tmpbuf1[0] = '\0';
	     if (count)
	       {
		  count = 0;
		  for (nick = chan->nicks; nick; nick = nick->next)
		    {
		       if ((nick->status & NICK_CHOP))
			  snprintf(ltb, sizeof(ltb) - 1, "@%-9.9s ", nick->nick);
		       else if ((nick->status & NICK_VOICE))
			  snprintf(ltb, sizeof(ltb) - 1, "+%-9.9s ", nick->nick);
		       else
			  snprintf(ltb, sizeof(ltb) - 1, " %-9.9s ", nick->nick);
		       ltb[sizeof(ltb)-1] = '\0';
		       strmcat(tmpbuf1, ltb, sizeof(tmpbuf1) - 1);
		       if (count++ == (((current_screen->co - 2) / 11) - 1))
			 {
			    put_raw("%s", tmpbuf1);
			    *tmpbuf1 = '\0';
			    count = 0;
			 }
		    }
		  if (count && tmpbuf1[0])
		     put_raw("%s", tmpbuf1);
	       }
	  }
	dma_Free(&nickstr);
     }
   else
     {
	put_info("You're not on %s!", channel);
	return;
     }
}

/*
 * /users command
 *
 * this command attempts to mimic /who but with cached information
 */
void
users(command, args, subargs)
   u_char *command, *args, *subargs;
{
   Channel *chan;
   Ban *banptr;
   Nick *nicks;
   Enemy *e;
   EChan *ec;
   Friend *f;
   FChan *fc;
   u_char *channel = NULL, *foo, *foo2;
   u_char feb[16], format[128], tmpbuf1[384];
   u_int cwidth, showhops;
   /* this stuff is for /who comapatability.. */
   u_char *arg, *nam, *hst, *srv, *file, *nick, *real, *no_open = UNULL;
   int noargs = 1, len, mask = 0, ok = 1;
   u_char buf_data[BUFSIZ];
   FILE *fip;

   hst = srv = nam = file = nick = real = UNULL;
   /* parse for [<flags>] */
   while ((arg = next_arg(args, &args)))
     {
	noargs = 0;
	if (*arg == '-' && !isdigit(*(arg + 1)))
	  {
	     u_char *cmd = UNULL;

	     arg++;
	     if ((len = my_strlen(arg)) == 0)
	       {
		  usage(command, "[<flags>] [<channel>]");
		  goto out;
	       }
	     dma_strcpy(&cmd, arg);
	     lower(cmd);
	     if (my_strncmp(cmd, "operators", len) == 0)
		mask |= WHO_OPS;
	     else if (my_strncmp(cmd, "lusers", len) == 0)
		mask |= WHO_LUSERS;
	     else if (my_strncmp(cmd, "chops", len) == 0)
		mask |= WHO_CHOPS;
	     else if (my_strncmp(cmd, "hosts", len) == 0)
	       {
		  if ((arg = next_arg(args, &args)) != NULL)
		    {
		       mask |= WHO_HOST;
		       malloc_strcpy(&hst, arg);
		       channel = hst;
		    }
		  else
		    {
		       usage(command, "-host <hostmask> [<channel>]");
		       new_free(&cmd);
		       goto out;
		    }
	       }
	     else if (my_strncmp(cmd, "here", len) == 0)
		mask |= WHO_HERE;
	     else if (my_strncmp(cmd, "away", len) == 0)
		mask |= WHO_AWAY;
	     else if (my_strncmp(cmd, "servers", len) == 0)
	       {
		  if ((arg = next_arg(args, &args)) != NULL)
		    {
		       mask |= WHO_SERVER;
		       malloc_strcpy(&srv, arg);
		       channel = srv;
		    }
		  else
		    {
		       usage(command, "-server <server mask> [<channel>]");
		       new_free(&cmd);
		       goto out;
		    }
	       }
	     else if (my_strncmp(cmd, "name", len) == 0 || my_strncmp(cmd, "user", len) == 0)
	       {
		  if ((arg = next_arg(args, &args)) != NULL)
		    {
		       mask |= WHO_NAME;
		       malloc_strcpy(&nam, arg);
		       channel = nam;
		    }
		  else
		    {
		       usage(command, "-name <username mask> [<channel>]");
		       new_free(&cmd);
		       goto out;
		    }
	       }
	     else if (my_strncmp(cmd, "realname", len) == 0)
	       {
		  if ((arg = next_arg(args, &args)) != NULL)
		    {
		       mask |= WHO_REAL;
		       malloc_strcpy(&real, arg);
		       channel = real;
		    }
		  else
		    {
		       usage(command, "-realname <real mask> [<channel>]");
		       new_free(&cmd);
		       goto out;
		    }
	       }
	     else if (my_strncmp(cmd, "nick", len) == 0)
	       {
		  if ((arg = next_arg(args, &args)) != NULL)
		    {
		       mask |= WHO_NICK;
		       malloc_strcpy(&nick, arg);
		       channel = who_nick;
		    }
		  else
		    {
		       usage(command, "-nick <nick name mask> [<channel>]");
		       new_free(&cmd);
		       goto out;
		    }
		  /* WHO -FILE by Martin 'Efchen' Friedrich */
	       }
	     else if (my_strncmp(cmd, "file", len) == 0)
	       {
		  mask |= WHO_FILE;
		  if ((arg = next_arg(args, &args)) != NULL)
		     malloc_strcpy(&file, arg);
		  else
		    {
		       usage(command, "-file <mask file> [<channel>]");
		       new_free(&cmd);
		       goto out;
		    }
	       }
	     else
	       {
		  usage(command, "[<flags>] [<channel>]");
		  new_free(&cmd);
		  goto out;
	       }
	     new_free(&cmd);
	  }
	else if (my_strcmp(arg, "*") == 0)
	  {
	     channel = get_channel_by_refnum(0);
	     if (!channel || *channel == '0')
	       {
		  say("I wouldn't do that if I were you");
		  goto out;
	       }
	  }
	else
	   channel = arg;
     }
   /* get the channel to list */
   if (noargs)
      channel = get_channel_by_refnum(0);
   if (channel && !is_channel(channel))
      channel = make_chan(channel);
   if (!channel)
     {
	usage(command, "[<flags>] [<channel>]");
	goto out;;
     }

   /* are we on the channel? */
   if ((chan = lookup_channel(channel, from_server, 0)) == NULL)
     {
	put_info("You're not on %s!", channel);
	goto out;
     }

   /* is the channel fully cached? */
   if (!(chan->status & CHAN_WHO))
     {
#ifdef SHOW_JOIN_TIME
	put_info("Sorry, %s is not cached yet, please wait for the cache message.", chan->channel);
#else
	put_info("Sorry, %s is not cached yet, please wait..", chan->channel);
#endif
	goto out;
     }

   /* create the format */
   showhops = get_int_var(SHOW_WHO_HOPCOUNT_VAR);
   if ((cwidth = get_int_var(CHANNEL_NAME_WIDTH_VAR)) != 0)
     {
	snprintf(format, sizeof(format) - 1, "%%-%u.%us %%-9s |%%-3s", (u_char) cwidth, (u_char) cwidth);
	format[sizeof(format)-1] = '\0';
     }
   else
      my_strmcpy(format, "%s\t%-9s |%-3s", sizeof(format) - 1);
   if (showhops)
      my_strmcat(format, " %2d", sizeof(format) - 1);
   my_strmcat(format, "|", sizeof(format) - 1);
   my_strmcat(format, " %s@%s (%s)", sizeof(format) - 1);

   /* show them.. */
   for (nicks = chan->nicks; nicks; nicks = nicks->next)
     {
	ok = 1;

	memset(feb, 0, sizeof(feb));
	foo = feb;
	/* here or away? */
	if (nicks->status & NICK_HERE)
	   *foo++ = 'H';
	if (nicks->status & NICK_AWAY)
	   *foo++ = 'G';
	/* operator? */
	if (nicks->status & NICK_OPER)
	   *foo++ = '*';
	/* opped? */
	if (nicks->status & NICK_CHOP)
	   *foo++ = '@';
	/* voiced? */
	if (nicks->status & NICK_VOICE)
	   *foo++ = '+';

/* enemy? *//* XXX: NEED TO CREATE GET_ENEMY_BY_NUH() */
	foo2 = nicks->user;
	if (foo2 && *foo2 == '~')
	   foo2++;
	snprintf(tmpbuf1, sizeof(tmpbuf1) - 1, "%s!%s@%s", nicks->nick, foo2 ? foo2 : empty_string, nicks->host);
	tmpbuf1[sizeof(tmpbuf1)-1] = '\0';

	if ((e = get_enemy(tmpbuf1)) != NULL && (ec = get_enemy_chan(e, chan->channel, 0)) != NULL)
	   *foo++ = 'E';

	/* banned? */
	for (banptr = chan->bans; banptr; banptr = banptr->next)
	  {
	     if (wild_match(banptr->ban, tmpbuf1))
	       {
		  *foo++ = 'B';
		  break;
	       }
	  }

	/* friend? */
	f = get_friend_by_mask(tmpbuf1);
	if (f)
	  {
	     *foo++ = 'F';
	     if ((fc = get_fchan(f, chan->channel, 0)))
		*foo++ = 'C';
	  }

	/* realname cached? */
	if (nicks->realname)
	   foo = nicks->realname;
	else
	   foo = empty_string;

	/* check for mask flags */
	if (mask)
	  {
	     if (mask & WHO_HERE)
		ok = ok && (nicks->status & NICK_HERE);
	     if (mask & WHO_AWAY)
		ok = ok && (nicks->status & NICK_AWAY);
	     if (mask & WHO_OPS)
		ok = ok && (nicks->status & NICK_OPER);
	     if (mask & WHO_LUSERS)
		ok = ok && !(nicks->status & NICK_OPER);
	     if (mask & WHO_CHOPS)
		ok = ok && (nicks->status & NICK_CHOP);
	     if (mask & WHO_NAME)
		ok = ok && wild_match(nam, nicks->user);
	     if (mask & WHO_NICK)
		ok = ok && wild_match(nick, nicks->nick);
	     if (mask & WHO_HOST)
		ok = ok && wild_match(hst, nicks->host);
	     if (mask & WHO_REAL)
		ok = ok && wild_match(real, nicks->realname);
	     if (mask & WHO_SERVER)
		ok = ok && wild_match(srv, nicks->server);
	     if (mask & WHO_FILE)
	       {
		  ok = 0;
		  if ((fip = fopen(CP(file), "r")) != (FILE *) 0)
		    {
		       while (fgets((char *) buf_data, BUFSIZ, fip) != (char *) 0)
			 {
			    buf_data[my_strlen(buf_data) - 1] = '\0';
			    ok = ok || wild_match(buf_data, nicks->nick);
			 }
		       fclose(fip);
		    }
		  else
		     no_open = file;
	       }
	  }

	/* show the line! */
	if (ok && do_hook(WHO_LIST, "%s %s %s %s %s %s", chan->channel, nicks->nick, feb, nicks->user, nicks->host, foo))
	  {
	     if (showhops)
		put_raw(format, chan->channel, nicks->nick, feb, nicks->hops, nicks->user, nicks->host, foo);
	     else
		put_raw(format, chan->channel, nicks->nick, feb, nicks->user, nicks->host, foo);
	  }
     }
   if (no_open)
      put_error("Cannot open: %s", no_open);
 out:
   dma_Free(&hst);
   dma_Free(&srv);
   dma_Free(&nam);
   dma_Free(&real);
   dma_Free(&nick);
   dma_Free(&file);
}

/*
 * just leave a channel and join it
 */
void
cycle(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *tmpchannel = UNULL, tmpbuf[1024];
   Channel *chan;

   if (args && *args)
      tmpchannel = make_chan(next_arg(args, &args));
   else if (!get_channel_by_refnum(0))
     {
	put_info("You are not on a channel!");
	return;
     }
   else
      tmpchannel = get_channel_by_refnum(0);

   if ((chan = lookup_channel(tmpchannel, from_server, 0)) == NULL)
     {
	put_info("You are not on %s!", tmpchannel);
	return;
     }

   my_strmcpy(tmpbuf, chan->channel, sizeof(tmpbuf) - 1);
   if (chan->key)
     {
	my_strmcat(tmpbuf, " ", sizeof(tmpbuf) - 1);
	my_strmcat(tmpbuf, chan->key, sizeof(tmpbuf) - 1);
     }
   send_to_server("PART %s", tmpchannel);
   send_to_server("JOIN %s", tmpbuf);
}

/*
 * send a message to all ops on a channel
 *
 * note, this is currently broken due to the nick,nick,nick support
 * in most servers being broken.
 */
void
do_wall(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *channel;
   Nick *nickl;
   Channel *chanl;

   if (!args || !*args)
     {
	usage(command, "[<channel>] <message>");
	return;
     }

   if (is_channel(args))
      channel = next_arg(args, &args);
   else if ((channel = get_channel_by_refnum(0)) == NULL)
     {
	put_info("You're not on a channel.");
	return;
     }

   /* am i on the channel? */
   chanl = lookup_channel(channel, from_server, 0);
   if (!chanl)
     {
	put_error("You're not on %s.", channel);
	return;
     }
   else
     {
	u_char wallline[1024], ops[1024];
	int count = 0, opcount = 0;
	int first = 1;

	/* what are we going to send everyone? */
	snprintf(wallline, sizeof(wallline) - 1, "(%s WallOp) %s", channel, args);
	wallline[sizeof(wallline)-1] = '\0';

	/* who are we going to send it to? */
	ops[0] = '\0';
	for (nickl = chanl->nicks; nickl; nickl = nickl->next)
	   if ((nickl->status & NICK_CHOP) && (strcmp(nickl->nick, nickname) != 0))
	     {
		if (first)
		  {
		     snprintf(ops, sizeof(ops) - 1, "%s,", nickl->nick);
		     ops[sizeof(ops)-1] = '\0';
		     first = 0;
		  }
		else
		  {
		     strmcat(ops, nickl->nick, sizeof(ops) - 1);
		     strmcat(ops, ",", sizeof(ops) - 1);
		  }
		count++;
		opcount++;
		if (count == 10)
		  {
		     count = 0;
		     first = 1;
		     send_to_server("NOTICE %s :%s", ops, wallline);
		     ops[0] = '\0';
		  }
	     }
	send_to_server("NOTICE %s :%s", ops, wallline);
	put_raw("-> (sent %d) %s", opcount, wallline);
     }
}

/*
 * this handles kick and bankick
 *
 * 09/27/01 cleaned up/optimized by jjd
 */
void
do_special_k(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *person, *channel, *reason = NULL;
   Channel *chanl;

   if (!args || !*args)
     {
	usage(command, "[<channel>] <nick> [<reason>]");
	return;
     }

   if (is_channel(args))
      channel = next_arg(args, &args);
   else if ((channel = get_channel_by_refnum(0)) == NULL)
     {
	put_info("You're not on a channel.");
	return;
     }

   /* am i on the channel? */
   chanl = lookup_channel(channel, from_server, 0);
   if (!chanl)
     {
	put_error("You're not on %s.", channel);
	return;
     }

   if (!(chanl->status & CHAN_CHOP))
     {
	put_error("You're not opped on %s.", channel);
	return;
     }
   else
     {
	person = next_arg(args, &args);
	if (!is_on_channel(channel, from_server, person))
	  {
	     put_error("Hey! %s is not on %s!", person, channel);
	     return;
	  }
	/* do i have a reason? */
	if (args && *args)
	   reason = args;
	else
	   reason = UP("out!");
	/* should we ban them too? */
	if (*command == 'B' || my_strcmp(command, "KICKBAN") == 0 || my_strcmp(command, "KB") == 0)
	   do_ban_stuff("BAN", person, channel);
	/* kick them */
	send_to_server("KICK %s %s :%s", chanl->channel, person, reason);
     }
}

/*
 * this handles /fk and /fbk
 *
 * 09/27/01 cleaned up/optimized by jjd
 */
void
filter_kick(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *channel, *nuhmask, reason[128];
   int ban = 0, old_limit, count = 0;
   Channel *chanl;
   Nick *nickl;

   if (my_strcmp(command, "FBK") == 0)
      ban = 1;

   /* check for a mask */
   nuhmask = next_arg(args, &args);
   if (!nuhmask)
     {
	usage(command, "<filter mask> [<channel> [<reason>]]");
	return;
     }
   if (!my_index(nuhmask, '!') || !my_index(nuhmask, '@'))
     {
	put_error("You must specify a valid nick!user@host mask.");
	return;
     }

   /* check out the channel situation */
   if (args && is_channel(args))
      channel = next_arg(args, &args);
   else
      channel = get_channel_by_refnum(0);
   if (!channel)
     {
	put_info("You're not on a channel.");
	return;
     }

   /* see if i'm on the channel */
   chanl = lookup_channel(channel, from_server, 0);
   if (!chanl)
     {
	put_error("You're not on %s.", channel);
	return;
     }
   if (!(chanl->status & CHAN_CHOP))
     {
	put_error("You're not opped on %s.", channel);
	return;
     }

   /* do we have a reason? */
   if (args && *args)
      my_strmcpy(reason, args, sizeof(reason) - 1);
   else
     {
	my_strmcpy(reason, "You match ", sizeof(reason) - 1);
	my_strmcat(reason, nuhmask, sizeof(reason) - 1);
     }

   /* initial pass to see if anyone matches.. */
   for (nickl = chanl->nicks; nickl; nickl = nickl->next)
     {
	u_char nuh[384], *foo;
	
	/* don't kick myself!! */
	if (my_stricmp(nickl->nick, get_server_nickname(chanl->server)) == 0)
	  continue;
	foo = nickl->user;
	if (*foo == '~')
	  foo++;
	snprintf(nuh, sizeof(nuh) - 1, "%s!*%s@%s", nickl->nick, foo, nickl->host);
	nuh[sizeof(nuh)-1] = '\0';
	if (wild_match(nuhmask, nuh))
	  count++;
     }
   
   if (count < 1)
     {
	put_error("Nobody on %s matches %s!", channel, nuhmask);
	return;
     }

   /* if we are supposed to ban them, do it last, but for now, +l 1 */
   old_limit = chanl->limit;
   if (ban)
     send_to_server("MODE %s +l 1", channel);

   /* kick those foolz */
   for (nickl = chanl->nicks; nickl; nickl = nickl->next)
     {
	u_char nuh[384], *foo;

	/* don't kick myself!! */
	if (my_stricmp(nickl->nick, get_server_nickname(chanl->server)) == 0)
	   continue;
	foo = nickl->user;
	if (*foo == '~')
	   foo++;
	snprintf(nuh, sizeof(nuh) - 1, "%s!*%s@%s", nickl->nick, foo, nickl->host);
	nuh[sizeof(nuh)-1] = '\0';
	if (wild_match(nuhmask, nuh))
	   send_to_server("KICK %s %s :%s", chanl->channel, nickl->nick, reason);
     }

   /* add the ban and restore the limit if necessary */
   if (ban)
     send_to_server("MODE %s +b %s", channel, nuhmask);
   if (old_limit > 0)
     send_to_server("MODE %s +l %u", channel, old_limit);
   else
     send_to_server("MODE %s -l", channel);
}

/*
 * kick all non-ops out of a channel
 */
void
lamer_kick(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *chan, reason[128];
   Nick *nickl;
   Channel *chanl;

   chan = next_arg(args, &args);
   if (!chan)
     {
	usage(command, "<channel> [<reason>]");
	return;
     }
   chan = make_chan(chan);

   /* make sure i'm on the channel */
   chanl = lookup_channel(chan, from_server, 0);
   if (!chanl)
     {
	put_error("You're not on %s.", chan);
	return;
     }

   /* make sure i can kick */
   if (!(chanl->status & CHAN_CHOP))
     {
	put_error("You're not opped on %s.", chan);
	return;
     }

   /* setup the reason */
   if (args && *args)
     {
	my_strmcpy(reason, args, sizeof(reason) - 1);
	my_strmcat(reason, " (", sizeof(reason) - 1);
	my_strmcat(reason, "non-op", sizeof(reason) - 1);
	my_strmcat(reason, ")", sizeof(reason) - 1);
     }
   else
      my_strmcpy(reason, "non-op", sizeof(reason) - 1);

   /* kick those lamers! */
   for (nickl = chanl->nicks; nickl; nickl = nickl->next)
      if (!(nickl->status & NICK_CHOP))
	 send_to_server("KICK %s %s :%s", chan, nickl->nick, reason);
}

/*
 * give_or_take_ops() does a standard:
 * /mode [<channel>] +/-o[ooo] <nick> [.. .. ..]
 *
 * it gives/takes ops to/from all people in the args that are
 * on the specified channel (or current if no channel is specified)
 *
 * 07/05/97 does voice too. FUNCTION OVERLOAD!@#
 * 09/27/01 cleanup/optimization by jjd
 */
void
give_or_take_ops(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *opchan = UNULL;
   char *mstr = NULL;
   u_char sign, mch;
   int num_nicks = 0;

   /* do we have a channel? */
   if (is_channel(args))
     {
	opchan = next_arg(args, &args);
	if (!(args && *args))
	  {
	     usage(command, "[#/&<channel>] <nick1> ... <nickn>");
	     return;
	  }
     }
   else if ((opchan = get_channel_by_refnum(0)) == NULL)
     {
	put_info("You are not on any channels!");
	return;
     }

   /* plus or minus? */
   if ((my_strncmp(command, "OP", 2) == 0) || (my_strncmp(command, "VOICE", 5) == 0))
      sign = '+';
   else
      sign = '-';

   /* voice or ops? */
   if (!my_strncmp(command, "VOICE", 5) || !my_strncmp(command, "DVOICE", 6))
      mch = 'v';
   else
      mch = 'o';

   /* how many people were specified? */
   num_nicks = how_many(args, ' ') + 1;
   if (num_nicks >= 4)
      mstr = strfill(mch, 4);
   while (num_nicks >= 4)
     {
	u_char *nick1 = NULL, *nick2 = NULL, *nick3 = NULL, *nick4 = NULL;

	nick1 = next_arg(args, &args);
	nick2 = next_arg(args, &args);
	nick3 = next_arg(args, &args);
	nick4 = next_arg(args, &args);
	send_to_server("MODE %s %c%s %s %s %s %s", opchan, sign, mstr, nick1, nick2, nick3, nick4);
	num_nicks -= 4;
     }
   if (num_nicks > 0)
     {
	mstr = strfill(mch, num_nicks);
	send_to_server("MODE %s %c%s %s", opchan, sign, mstr, args);
     }
}

#ifdef ALLOW_OJ			/* lets see if anyone misses, i won't */
/*
 * on join ?
 * for /oj (op the last person to join)
 * this is bloat.
 */
void
oj(char *command, char *args, char *subargs)
{
   extern *joined_chan;

   if (!joined_nick)
      return;
   else
      send_to_server("MODE %s +o %s", joined_chan, joined_nick);
}
#endif

/*
 * change a channel's key
 *
 * 09/27/01 cleaned up/optimized by jjd
 */
void
chkey(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *mychan = NULL, *newkey = NULL;
   Channel *chan;

   if (!(args && *args))
     {
	usage("chkey", "[<channel>] -|<new key>");
	return;
     }
   newkey = next_arg(args, &args);
   if (args && *args)
     {
	mychan = make_chan(newkey);
	newkey = next_arg(args, &args);
     }
   else if (!get_channel_by_refnum(0))
     {
	put_info("You are not on any channels!");
	return;
     }
   else
      mychan = get_channel_by_refnum(0);

   /* am i on the channel? */
   chan = lookup_channel(mychan, from_server, 0);
   if (!chan)
     {
	put_info("You are not on %s.", mychan);
	return;
     }

   /* check for ops */
   if (!(chan->status & CHAN_CHOP))
     {
	put_info("Changing the key requires ops.");
	return;
     }

   /* change the key */
   if (chan->key)
      send_to_server("MODE %s -k %s", mychan, chan->key);
   if (my_strcmp(newkey, "-") != 0)
      send_to_server("MODE %s +k %s", mychan, newkey);
}

/*
 * change the channel limit
 */
void
change_channel_limit(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *limit, *chann;
   long ldlimit;

   limit = next_arg(args, &args);
   if (!limit)
     {
	usage(command, "<limit/-> [<channel>]");
	return;
     }

   /* check out the channel situation */
   if (((chann = next_arg(args, &args)) != NULL) && (my_strcmp(chann, "*") != 0))
      chann = (u_char *) make_chan(chann);
   else if ((chann = get_channel_by_refnum(0)) == NULL)
     {
	put_info("You are not on a channel.");
	return;
     }

   ldlimit = my_atol(limit);
   if (ldlimit > 0)
      send_to_server("MODE %s +l %ld", chann, ldlimit);
   else if (my_strcmp(limit, "-") == 0)
      send_to_server("MODE %s -l", chann);
   else
      put_error("Invalid channel limit: %s", limit);
}

/*
 * mass modes!
 */
void
mass(command, args, subargs)
   u_char *command, *args, *subargs;
{
   Channel *chanl;
   Nick *nickl;
   u_char *type, *chann;
   u_char mode;
   int dickmode = 0, count = 0;
   u_char tmpbuf1[1024];

   if (!args || !*args)
     {
	usage(command, "[+|-]<mode> [<channel>]");
	return;
     }
   type = next_arg(args, &args);
   if (*type == '-')
      dickmode = 1;
   if (!isalpha(*type))
      type++;
   switch (*type)
     {
	case 'v':
	   mode = 'v';
	   break;
	case 'o':
	   mode = 'o';
	   break;
	default:
	   usage(command, "[+|-]<mode> [<channel>]");
	   return;
     }
   if ((chann = next_arg(args, &args)) == NULL)
      if ((chann = get_channel_by_refnum(0)) == NULL)
	{
	   put_error("You must by on a channel to do mass modes..");
	   return;
	}
   chann = (u_char *) make_chan(chann);
   if ((chanl = lookup_channel(chann, from_server, 0)) == NULL)
     {
	put_error("You're not on %s!", chann);
	return;
     }
   if (!is_chanop(chann, get_server_nickname(from_server)))
     {
	put_error("You must be opped to do mass modes.", chann);
	return;
     }
   put_info("Executing a mass %s%c on %s.", dickmode ? "-" : "+", mode, chanl->channel);
   if ((mode == 'o') && dickmode)
     {
	mdop("MDOP", chanl->channel, NULL);
	return;
     }
   bzero(tmpbuf1, sizeof(tmpbuf1));
   for (nickl = chanl->nicks; nickl; nickl = nickl->next)
     {
	if (my_strcmp(nickl->nick, get_server_nickname(from_server)) != 0)
	  {
	     if ((dickmode && mode == 'v' && (nickl->status & NICK_VOICE)) ||
		 (!dickmode && mode == 'v' && !(nickl->status & NICK_VOICE) && !(nickl->status & NICK_CHOP)) ||
		 (!dickmode && mode == 'o' && !(nickl->status & NICK_CHOP)))
	       {
		  strmcat(tmpbuf1, " ", sizeof(tmpbuf1) - 1);
		  strmcat(tmpbuf1, nickl->nick, sizeof(tmpbuf1) - 1);
		  count++;
	       }
	  }
	if (count == 4)
	  {
	     count = 0;
	     send_to_server("MODE %s %s%s %s", chanl->channel, dickmode ? "-" : "+", strfill(mode, 4), tmpbuf1);
	     bzero(tmpbuf1, sizeof(tmpbuf1));
	  }
     }
   send_to_server("MODE %s %s%s %s", chanl->channel, dickmode ? "-" : "+", strfill(mode, count), tmpbuf1);
   return;
}

/* dispatcher for abbreviated /mass commands:
 *  /mop, /mvoice, /mdvoice
 */
void
mass_abbrev(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *tmp;

   tmp = (u_char *) dma_Malloc(strlen(args) + 4);
   if (strcmp(command, "MOP") == 0)
     {
	sprintf(tmp, "+o %s", args);	/* safe, malloc()'d */
	mass("MOP", tmp, NULL);
     }
   else if (strcmp(command, "MVOICE") == 0)
     {
	sprintf(tmp, "+v %s", args);	/* safe, malloc()'d */
	mass("MVOICE", tmp, NULL);
     }
   else if (strcmp(command, "MDVOICE") == 0)
     {
	sprintf(tmp, "-v %s", args);	/* safe, malloc()'d */
	mass("MDVOICE", tmp, NULL);
     }
   dma_Free(&tmp);
}

/*
 * super k-rad elite MASS DEOP code
 */
void
mdop(command, args, subargs)
   u_char *command, *args, *subargs;
{
#define DEADPPL_SIZE 1024
#define MAX_DEADPPL 200
#define MAX_SAFEPPL 20
#define NINJA_BAN_MASK "ninja!irc@rulez"
   u_char *chann, deadppl[DEADPPL_SIZE][MAX_DEADPPL], *safeppl[MAX_SAFEPPL];
   u_char tmpbuf1[1024];
   Channel *chanl;
   Nick *nicks;
   int count = 0, lines = 0, i = 0, safecnt = 0, test = 0;

   chann = next_arg(args, &args);
   if (chann && my_strcmp(chann, "-t") == 0)
     {
	test = 1;
	chann = next_arg(args, &args);
     }
   if (!chann)
     {
	usage(command, "[-t] <channel> [<nicks not to deop>]");
	return;
     }
   /* zero out deadppl and safeppl.. */
   for (i = 0; i < MAX_DEADPPL; i++)
     *deadppl[i] = 0;
   for (i = 0; i < MAX_SAFEPPL; i++)
     safeppl[i] = NULL;

   if (my_strlen(args) > 0)
     {
	put_info("I promise not to deop: %s", args);
	safeppl[safecnt] = next_arg(args, &args);
	while (safeppl[safecnt] != NULL)
	  {
	     safecnt++;
	     safeppl[safecnt] = next_arg(args, &args);
	  }
     }
   if (my_strcmp(chann, asterik) == 0)
      chann = get_channel_by_refnum(0);
   else
      chann = (u_char *) make_chan(chann);
   if (!chann || !(chanl = lookup_channel(chann, from_server, CHAN_NOUNLINK)))
     {
	put_error("You're not on %s!", chann);
	return;
     }
   if (!test && !(chanl->status & CHAN_CHOP))
     {
	put_error("You must be opped to mass de-op!");
	return;
     }
   bzero(tmpbuf1, sizeof(tmpbuf1));
   for (nicks = chanl->nicks; nicks; nicks = nicks->next)
     {
	if ((strcmp(get_server_nickname(from_server), nicks->nick) != 0) &&
	    (nicks->status & NICK_CHOP) && !nodeop(safeppl, nicks->nick))
	  {
	     if (*tmpbuf1)
	       {
		  strmcat(tmpbuf1, " ", sizeof(tmpbuf1) - 1);
		  strmcat(tmpbuf1, nicks->nick, sizeof(tmpbuf1) - 1);
	       }
	     else
		snprintf(tmpbuf1, sizeof(tmpbuf1), "%s", nicks->nick);
	     count++;
	  }
	if (count == 3)
	  {
	     snprintf(deadppl[lines], DEADPPL_SIZE - 1, "MODE %s -b%s %s %s", chanl->channel,
		      strfill('o', count), NINJA_BAN_MASK, tmpbuf1);
	     *(deadppl[lines] + DEADPPL_SIZE - 1) = '\0';
	     lines++;
	     count = 0;
	     *tmpbuf1 = '\0';
	  }
     }
   if (count)
     {
	snprintf(deadppl[lines], DEADPPL_SIZE - 1, "MODE %s -b%s %s %s", chanl->channel,
		 strfill('o', count), NINJA_BAN_MASK, tmpbuf1);
	*(deadppl[lines] + DEADPPL_SIZE - 1) = '\0';
	lines++;
     }
   if (lines == 0)
     {
	put_error("Nobody on %s is opped.", chanl->channel);
	return;
     }
   for (i = 0; i < lines; i++)
     {
	if (test)
	   put_raw("%s", deadppl[i]);
	else
	   send_to_server("%s", deadppl[i]);
     }
}

/* this goes with /mdop */
static int
nodeop(u_char * ppl[], u_char * user)
{
   int blehcnt = 0;

   if (ppl[blehcnt] == NULL)
      return 0;
   do
     {
	if (!my_stricmp(ppl[blehcnt], user))
	   return 1;
	blehcnt++;
     }
   while (ppl[blehcnt] != NULL);
   return 0;
}

/*
 * ban/unban masks/nicks
 */
void
do_ban_stuff(command, args, subargs)
   u_char *command, *args, *subargs;
{
   int meany;
   u_char *person, *chann, *evil = NULL;
   u_char *ex, *at;
   Nick *nicks;
   Channel *c;

   if (command[0] == 'B')
      meany = 1;
   else
      meany = 0;
   if ((person = next_arg(args, &args)) != NULL)
     {
	chann = next_arg(args, &args);
	if (!chann)
	   chann = get_channel_by_refnum(0);
	if (!chann || !*chann)
	  {
	     put_info("You're not on a channel.");
	     return;
	  }
	if (!is_channel(chann))
	   chann = make_chan(chann);
	/* now we should have a valid channel */
	c = lookup_channel(chann, from_server, CHAN_NOUNLINK);
	if (!c)
	   return;

	if (!(c->status & CHAN_CHOP))
	  {
	     put_info("You have no ops to add/remove bans with!");
	     return;
	  }

	/* if we are specifying a mask instead of a nickname */
	ex = my_index(person, '!');
	at = my_index(person, '@');
	if (ex || at)
	  {
	     u_char theshiz[512];

	     if (!at && ex)
		snprintf(theshiz, sizeof(theshiz) - 1, "%s@*", person);
	     else if (at && !ex)
		snprintf(theshiz, sizeof(theshiz) - 1, "*!%s", person);
	     else
		my_strncpy(theshiz, person, sizeof(theshiz) - 1);
	     theshiz[sizeof(theshiz)-1] = '\0';
	     if (meany)
		send_to_server("MODE %s +b %s", chann, theshiz);
	     else
		remove_matching_bans(theshiz, NULL, NULL, chann, 0);
	     return;
	  }
	else if ((nicks = find_nick(person, UNULL, from_server, c)) != NULL)
	  {
	     u_char *user = nicks->user;

	     if (user && *user == '~')
		user++;
	     if (user && *user)
		user++;
	     if (meany)
	       {
		  if (nicks->status & NICK_CHOP)
		     send_to_server("MODE %s -o+b %s *!*%s@%s", chann, nicks->nick, user, cluster(nicks->host));
		  else
		     send_to_server("MODE %s +b *!*%s@%s", chann, user, cluster(nicks->host));
	       }
	     else
		remove_matching_bans(nicks->nick, user, nicks->host, chann, 0);
	     evil--;
	     return;
	  }
	if (meany)
	   add_to_whois_queue(person, ban_queue, "%s", chann);
	else
	   add_to_whois_queue(person, unban_queue, "%s", chann);
     }
   else
      usage(command, "<nick/mask> [<channel>]");
}

/*
 * /adds
 */
void
addserver(command, args, subargs)
   u_char *command, *args, *subargs;
{
   int port_num = 0;
   int old_server = 0;
   u_char *password, *port, *nick, *server, *extra, *umode;

   if (!(args && *args))
     {
	usage("adds", "<server>[:<port>:<password>:<nick>:<umode>]");
	return;
     }
   password = port = nick = extra = umode = UNULL;
   server = next_arg(args, &args);
   parse_server_info(&server, &port, &password, &nick, &umode, &extra);
   if (port && *port)
     {
	port_num = my_atoi(port);
	if (!port_num)
	   port_num = irc_port;
     }
   else
      port_num = irc_port;
   old_server = from_server;
   add_to_server_list(server, port_num, password, nick, umode, 0);
   from_server = old_server;
   put_info("%s %d added to your server list.", server, port_num);
}

/*
 * /rems
 */
void
removeserver(command, args, subargs)
   u_char *command, *args, *subargs;
{
   int port_num = 0;
   int old_server = 0;
   int servernum = 0;
   u_char *password, *port, *nick, *server, *extra, *umode;

   if (!(args && *args))
     {
	usage("rems", "<#> or <server>[:<port>]");
	return;
     }
   password = port = nick = extra = umode = UNULL;
   server = next_arg(args, &args);
   parse_server_info(&server, &port, &password, &nick, &umode, &extra);
   if (port && *port)
     {
	port_num = my_atoi(port);
	if (!port_num)
	   port_num = irc_port;
     }
   else
      port_num = irc_port;

   old_server = from_server;
   if ((servernum = atoi(server)) > 0 && servernum <= number_of_servers)
      servernum--;
   else
      servernum = find_in_server_list(server, port_num, NULL);

   if (servernum < 0 || servernum >= number_of_servers)
     {
	put_info("Unable to locate server \"%s\" in the server list.", server);
	return;
     }

   if (server_list[servernum].connected || server_list[servernum].read != -1)
      put_info("Server #%d (%s:%d) is connected, not removing.", servernum + 1, server_list[servernum].name,
	       server_list[servernum].port);
   else
     {
	put_info("Removed server #%d (%s:%d) from your list.", servernum + 1, server_list[servernum].name,
		 server_list[servernum].port);
	remove_from_server_list(servernum);
     }
   from_server = old_server;
}

/* this is an input prompt */
void
short_playlog(u_char * data, u_char * line)
{
   if (!(strcasecmp(line, "Y")))
      playlog(NULL, NULL, NULL);
   else
      add_wait_prompt("Erase your stored messages? [y/N] ", short_eraselog, "", WAIT_PROMPT_LINE);
}

/*
 * away: the /AWAY command.  Keeps track of the away message locally, and
 * sends the command on to the server.
 * Only sets the current server away unless -all is specified
 * vars used: EMAIL_ADDRESS, SILENT_AWAY, DEFAULT_AWAY_MSG
 * if EMAIL_ADDRESS is non-NULL it is appened to the away message as
 * [email: blah@lemae.com], else ignored.  if SILENT_AWAY is set, the away
 * messages will not be sent to the channel(s)
 */
void
away(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char tmpu_char[BIG_BUFFER_SIZE], *defaway = NULL;
   u_char awaymsg[1024], *arg1, *email;
   int old_server = from_server;

   email = get_string_var(EMAIL_ADDRESS_VAR);
   if (args && *args)
     {
	if (*args == '-')
	  {
	     arg1 = next_arg(args, &args);
	     if (my_strnicmp(arg1 + 1, "ALL", 3) == 0)
	       {
		  if ((args && *args) || ((defaway = get_string_var(DEFAULT_AWAY_MSG_VAR)) != NULL))
		     snprintf(awaymsg, sizeof(awaymsg) - 1, "(%s)", args && *args ? args : defaway);
		  else
		    {
		       put_info("Either provide a reason or set DEFAULT_AWAY_MSG to something.");
		       return;
		    }
		  if (email)
		    {
		       snprintf(tmpu_char, sizeof(tmpu_char) - 1, " [email: %s]", email);
		       strmcat(awaymsg, tmpu_char, sizeof(awaymsg) - 1);
		    }
		  for (from_server = 0; from_server < number_of_servers; from_server++)
		    {
		       if (server_list[from_server].away_set == 0)
			 {
			    server_list[from_server].away_set = time(NULL);
			    server_list[from_server].away_count = 0;
			    dma_strcpy(&server_list[from_server].away, awaymsg);
			    if (server_list[from_server].connected)
			      {
				 if (!get_int_var(SILENT_AWAY_VAR))
				   {
				      snprintf(tmpu_char, sizeof(tmpu_char), "is out. %s", awaymsg);
				      ame("AME", tmpu_char, NULL);
				   }
				 send_to_server("%s :%s", command, awaymsg);
			      }
			 }
		    }
		  from_server = old_server;
		  open_away_log(awaymsg);
		  update_all_status();
		  return;
	       }
	     else
	       {
		  put_error("Unknown option: %s", arg1);
		  return;
	       }
	  }
	else
	   snprintf(awaymsg, sizeof(awaymsg) - 1, "(%s)", args);
     }
   else if ((defaway = get_string_var(DEFAULT_AWAY_MSG_VAR)) != NULL)
      snprintf(awaymsg, sizeof(awaymsg) - 1, "(%s)", defaway);
   else
     {
	put_info("Either provide a reason or set DEFAULT_AWAY_MSG to something.");
	return;
     }
   if (server_list[from_server].away_set != 0)
     {
	put_error("You're already set away on this server. Use /back.");
	return;
     }
   if (email)
     {
	snprintf(tmpu_char, sizeof(tmpu_char) - 1, " [email: %s]", email);
	strmcat(awaymsg, tmpu_char, sizeof(awaymsg) - 1);
     }
   if (!get_int_var(SILENT_AWAY_VAR))
     {
	snprintf(tmpu_char, sizeof(tmpu_char), "is out. %s", awaymsg);
	ame("AME", tmpu_char, NULL);
     }
   server_list[from_server].away_set = time(NULL);
   server_list[from_server].away_count = 0;
   dma_strcpy(&server_list[from_server].away, awaymsg);
   open_away_log(awaymsg);
   send_to_server("%s :%s", command, awaymsg);
   update_all_status();
}

/*
 * same algorithm as away, but messages reflect being back and away gets unset.
 * -ALL is honored, if not set, only the current server is set back..
 * vars used: SILENT_AWAY, DEFAULT_BACK_MSG
 * they are fairly self explanatory
 * asks the user if they want to view their away messages if they get set back
 */
void
back(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *defback = NULL, *arg1;
   u_char backmsg[1024], tmpu_char[BIG_BUFFER_SIZE];
   time_t atime;
   int old_server = from_server;

   if (args && *args)
     {
	if (*args == '-')
	  {
	     arg1 = next_arg(args, &args);
	     if (my_strnicmp(arg1 + 1, "ALL", 3) == 0)
	       {
		  if ((args && *args) || ((defback = get_string_var(DEFAULT_BACK_MSG_VAR)) != NULL))
		     snprintf(backmsg, sizeof(backmsg), "(%s)", args && *args ? args : defback);
		  else
		    {
		       put_info("Either provide a reason or set DEFAULT_BACK_MSG to something.");
		       return;
		    }
		  for (from_server = 0; from_server < number_of_servers; from_server++)
		     if (server_list[from_server].away_set != 0)
		       {
			  atime = time(NULL) - server_list[from_server].away_set;
			  server_list[from_server].away_set = 0;
			  server_list[from_server].away_count = 0;
			  dma_Free(&server_list[from_server].away);
			  if (server_list[from_server].connected)
			    {
			       if (!get_int_var(SILENT_AWAY_VAR))
				 {
				    snprintf(tmpu_char, sizeof(tmpu_char), "is back after %s. %s", ninja_etime(atime), backmsg);
				    ame("AME", tmpu_char, NULL);
				 }
			       send_to_server("%s :", command);
			    }
		       }
		  from_server = old_server;
		  update_all_status();
		  close_away_log(backmsg);
		  add_wait_prompt("Play your stored messages? [y/N] ", short_playlog, "", WAIT_PROMPT_LINE);
		  return;
	       }
	     else
	       {
		  put_error("Unknown option: %s", arg1);
		  return;
	       }
	  }
	else
	   snprintf(backmsg, sizeof(backmsg), "(%s)", args);
     }
   else if ((defback = get_string_var(DEFAULT_BACK_MSG_VAR)) != NULL)
      snprintf(backmsg, sizeof(backmsg), "(%s)", defback);
   else
     {
	put_info("Either provide a reason or set DEFAULT_BACK_MSG to something.");
	return;
     }
   if (server_list[from_server].away_set == 0)
     {
	put_error("You're not marked away on this server.");
	return;
     }
   atime = time(NULL) - server_list[from_server].away_set;
   if (!get_int_var(SILENT_AWAY_VAR))
     {
	snprintf(tmpu_char, sizeof(tmpu_char), "has returned after %s. %s", ninja_etime(atime), backmsg);
	ame("AME", tmpu_char, NULL);
     }
   server_list[from_server].away_set = 0;
   server_list[from_server].away_count = 0;
   dma_Free(&server_list[from_server].away);
   send_to_server("%s :", command);
   update_all_status();
   close_away_log(backmsg);
   add_wait_prompt("Play your stored messages? [y/N] ", short_playlog, "", WAIT_PROMPT_LINE);
}

/*
 * /me on all channels
 */
void
ame(command, args, subargs)
   u_char *command, *args, *subargs;
{
   Channel *tmp;

   if (!args || !*args)
     {
	usage(command, "<message>");
	return;
     }
   for (tmp = server_list[from_server].chan_list; tmp; tmp = tmp->next)
     {
	int old = set_lastlog_msg_level(LOG_ACTION);

	save_message_from();
	send_action(tmp->channel, args);
	message_from(tmp->channel, LOG_ACTION);
	if (do_hook(SEND_ACTION_LIST, "%s %s", tmp->channel, args))
	   put_raw("* %s %s", get_server_nickname(from_server), args);
	set_lastlog_msg_level(old);
	restore_message_from();
     }
}

/*
 *
 * XXX: at some point i will merge this with IRCII /save
 */
void
ninja_save(command, args, subargs)
   u_char *command, *args, *subargs;
{
   ChanGrep *channel;
   u_char *filename = NULL, tmpoutbuf[1024], ltb[512];
   FILE *outfile;
   int vcount = 0, scount = 0, ncount = 0, gcount = 0;
   int fcount = 0, ecount = 0, kcount = 0;
   int first = 1;
   u_char *na, *pa, *ni, *um;

   if (qflag)
     {
	put_error("You started Ninja IRC with the -q flag, saving is disabled.");
	return;
     }

   filename = NINJA_SAVE_FILE;
   if (*filename == '~')
      filename = expand_twiddle(filename);
   outfile = fopen(filename, "w");
   if (outfile == NULL)
     {
	put_error("Cannot open save file(%s), aborting save..", filename);
	if (CP(filename) != NINJA_SAVE_FILE)
	   dma_Free(&filename);
	return;
     }
   if (CP(filename) != NINJA_SAVE_FILE)
      dma_Free(&filename);
   if (orignickname)
      fprintf(outfile, "ORIGNICK ON %s\n", orignickname);
   vcount = save_variables(outfile, 1);
   if (server_list)
      for (scount = 0; scount < number_of_servers; scount++)
	{
	   na = server_list[scount].name;
	   first = server_list[scount].port;
	   pa = server_list[scount].password;
	   ni = server_list[scount].nickname;
	   um = server_list[scount].usermode;
	   fprintf(outfile, "ADDS %s:%d:%s:%s:%s\n",
		   na ? na : empty_string, first, pa ? pa : empty_string, ni ? ni : empty_string, um ? um : empty_string);
	}
   for (channel = grep_list; channel; channel = channel->next)
     {
	fprintf(outfile, "GREP %s %s\n", channel->channel, channel->words);
	gcount++;
     }

   /* save the notify list... */
   ncount = save_notify(outfile);
   
   /* friends, enemies, channel keys */
   fcount = save_friends(1);
   ecount = save_enemies(1);
   kcount = save_ckeys(1);
   
   /* build the output string.. */
   first = 1;
   strcpy(tmpoutbuf, "Saved:");
   if (vcount)
     {
	if (!first)
	  strmcat(tmpoutbuf, ",", sizeof(tmpoutbuf)-1);
	first = 0;
	snprintf(ltb, sizeof(ltb)-1, 
		 " %d variable%s", vcount, PLURAL(vcount));
	ltb[sizeof(ltb)-1] = '\0';
	strmcat(tmpoutbuf, ltb, sizeof(tmpoutbuf)-1);
     }
   if (scount)
     {
	if (!first)
	  strmcat(tmpoutbuf, ",", sizeof(tmpoutbuf)-1);
	first = 0;
	snprintf(ltb, sizeof(ltb)-1,
		 " %d server%s", scount, PLURAL(scount));
	ltb[sizeof(ltb)-1] = '\0';
	strmcat(tmpoutbuf, ltb, sizeof(tmpoutbuf)-1);
     }
   if (gcount)
     {
	if (!first)
	  strmcat(tmpoutbuf, ",", sizeof(tmpoutbuf)-1);
	first = 0;
	snprintf(ltb, sizeof(ltb)-1, 
		 " %d grep entr%s", gcount, Y_PLURAL(gcount));
	ltb[sizeof(ltb)-1] = '\0';
	strmcat(tmpoutbuf, ltb, sizeof(tmpoutbuf)-1);
     }
   if (ncount)
     {
	if (!first)
	  strmcat(tmpoutbuf, ",", sizeof(tmpoutbuf)-1);
	first = 0;
	snprintf(ltb, sizeof(ltb)-1, 
		 " %d notify entr%s", ncount, Y_PLURAL(ncount));
	ltb[sizeof(ltb)-1] = '\0';
	strmcat(tmpoutbuf, ltb, sizeof(tmpoutbuf)-1);
     }
   if (fcount)
     {
	if (!first)
	  strmcat(tmpoutbuf, ",", sizeof(tmpoutbuf)-1);
	first = 0;
	snprintf(ltb, sizeof(ltb)-1, 
		 " %d friend%s", fcount, PLURAL(fcount));
	ltb[sizeof(ltb)-1] = '\0';
	strmcat(tmpoutbuf, ltb, sizeof(tmpoutbuf)-1);
     }
   if (ecount)
     {
	if (!first)
	  strmcat(tmpoutbuf, ",", sizeof(tmpoutbuf)-1);
	first = 0;
	snprintf(ltb, sizeof(ltb)-1, 
		 " %d enem%s", ecount, Y_PLURAL(ecount));
	ltb[sizeof(ltb)-1] = '\0';
	strmcat(tmpoutbuf, ltb, sizeof(tmpoutbuf)-1);
     }
   if (kcount)
     {
	if (!first)
	  strmcat(tmpoutbuf, ",", sizeof(tmpoutbuf)-1);
	first = 0;
	snprintf(ltb, sizeof(ltb)-1, 
		 " %d channel key%s", kcount, PLURAL(kcount));
	ltb[sizeof(ltb)-1] = '\0';
	strmcat(tmpoutbuf, ltb, sizeof(tmpoutbuf)-1);
     }
   if (!first)
     {
	strmcat(tmpoutbuf, ".", sizeof(tmpoutbuf) - 1);
	put_info("%s", tmpoutbuf);
     }
   fclose(outfile);
}

/*
 * show the bans from cache..
 */
void
show_ban_cache(command, args, subargs)
   u_char *command, *args, *subargs;
{
   put_error("Not implemented yet.");
}
