/*
 * notify.c: a few handy routines to notify you when people enter and leave irc 
 *
 * Written By Michael Sandrof
 *
 * Copyright (c) 1990 Michael Sandrof.
 * Copyright (c) 1991, 1992 Troy Rollo.
 * Copyright (c) 1992-2000 Matthew R. Green.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Revamped by lynX - Dec '91
 */

#include "irc.h"
#include "dma.h"
IRCII_RCSID("@(#)$Id: notify.c,v 1.34 2000/05/11 03:35:16 mrg Exp $");

#include "list.h"
#include "notify.h"
#include "ircaux.h"
#include "whois.h"
#include "hook.h"
#include "server.h"
#include "output.h"
#include "vars.h"
#include "status.h"

#include "ninja.h"
#include "friends.h"

/* NotifyList: the structure for the notify stuff */
typedef	struct	notify_stru
{
	struct	notify_stru	*next;	/* pointer to next notify person */
	u_char	*nick;			/* nickname of person to notify about */
	u_char	*user;			/* username of person to notify about */
	u_char	*host;			/* hostname of person to notify about */
	u_char	*mask;			/* host mask to check of person to notify about */
	int	flags;			/* based on the following flags  WAS: 1=person on irc, 0=person not on irc */
#define NOTIFY_ONLINE	0x01   
#define NOTIFY_AWAY	0x01
#define NOTIFY_LOG	0x01   
}	NotifyList;

static	NotifyList	*notify_list = (NotifyList *) 0;

static	void	free_notify _((NotifyList **));

/* Rewritten, -lynx */
void
show_notify_list(all)
	int	all;
{
   NotifyList	*tmp;
   u_char	*list = (u_char *) 0;
   int		first = 1;
   
   for (tmp = notify_list; tmp; tmp = tmp->next)
     {
	if (tmp->flags & NOTIFY_ONLINE)
	  {
	     if (first)
	       {
		  first = 0;
		  say("Currently present:");
	       }
	     put_raw("   %-15s %12s@%s", tmp->nick, tmp->user, tmp->host);
	  }
     }
   if (all)
     {
	malloc_strcpy(&list, empty_string);
	for (tmp = notify_list; tmp; tmp = tmp->next)
	  {
	     if (!(tmp->flags & NOTIFY_ONLINE))
	       {
		  malloc_strcat(&list, UP(" "));
		  malloc_strcat(&list, tmp->nick);
	       }
	  }
	if (*list) say("Currently absent:%s", list);
	new_free(&list);
     }
}

/* notify: the NOTIFY command.  Does the whole ball-o-wax */
/*ARGSUSED*/
void
notify(command, args, subargs)
	u_char	*command,
		*args,
		*subargs;
{
   u_char	*nick,
     *list = (u_char *) 0,
     *ptr;
   int	no_nicks = 1;
   int	do_ison = 0;
   int	old_server = from_server;
   NotifyList	*new;
   u_char *mask = UNULL, *tmp = UNULL;
   int count = 0;
   
   malloc_strcpy(&list, empty_string);
   while ((nick = next_arg(args, &args)) != NULL)
     {
	no_nicks = 0;
	while (nick)
	  {
	     if ((ptr = my_index(nick, ',')) != NULL)
	       *ptr++ = '\0';
	     if (*nick == '-')
	       {
		  nick++;
		  if (*nick)
		    {
		       if ((new = (NotifyList *) remove_from_list((List **) &notify_list, nick)) != NULL)
			 {
			    free_notify(&new);
			    say("%s removed from notification list", nick);
			 }
		       else
			 say("%s is not on the notification list", nick);
		    }
		  else
		    {
		       while ((new = notify_list))
			 {
			    notify_list = new->next;
			    free_notify(&new);
			 }
		       say("Notify list cleared");
		    }
	       }
	     else
	       {
		  /* compatibility */
		  if (*nick == '+')
		    nick++;
		  if (*nick)
		    {
		       do_ison = 1;
		       /* ninja extension, get the mask */
		       if (match("*!*@*", nick))
			 {
			    dma_strcpy(&tmp, nick);
			    mask = my_index(tmp, '!');
			    *mask++ = '\0';
			    nick = tmp;
			 }
		       if (my_index(nick, '*'))
			 say("Wildcards not allowed in NOTIFY nicknames!");
		       else
			 {
			    
			    if ((new = (NotifyList *) remove_from_list((List **) &notify_list, nick)) != NULL)
			      free_notify(&new);
			    new = (NotifyList *) new_malloc(sizeof(NotifyList));
			    /*
			     * dma_malloc bzero's
			    new->nick = (u_char *) 0;
			    new->flag = 0;
			     */
			    if (*nick == '%')
			      {
				 new->flags |= NOTIFY_LOG;
				 nick++;
			      }
			    malloc_strcpy(&(new->nick), nick);
			    if (mask)
			      malloc_strcpy(&(new->mask), mask);
			    add_to_list((List **) &notify_list, (List *) new);
			    from_server = primary_server;
			    if (get_server_2_6_2(from_server))
			      {
				 if (count == 5)
				   {
				      add_userhost_to_whois(list, userhost_notify);
				      new_free(&list);
				      count = 0;
				   }
				 count++;
				 if (list)
				   malloc_strcat(&list, UP(" "));
				 malloc_strcat(&list, new->nick);
			      }
			    else
			      add_userhost_to_whois(new->nick, userhost_notify);
			    say("%s added to the notification list", nick);
			    from_server = old_server;
			 }
		    }
		  else
		    show_notify_list(0);
	       }
	     nick = ptr;
	  }
     }
   if (count && list) {
      from_server = primary_server;
      add_userhost_to_whois(list, userhost_notify);
      from_server = old_server;
   }
   new_free(&list);
   if (no_nicks)
     show_notify_list(1);
}

/*
 * do_notify: This simply goes through the notify list, sending out a WHOIS
 * for each person on it.  This uses the fancy whois stuff in whois.c to
 * figure things out.  Look there for more details, if you can figure it out.
 * I wrote it and I can't figure it out.
 *
 * Thank you Michael... leaving me bugs to fix :) Well I fixed them!
 */
void
do_notify()
{
   int	count = 0,
     old_server = from_server;
   u_char	buf[BIG_BUFFER_SIZE+1], *p = buf;
   u_int	nlen;
   NotifyList	*tmp;
   
   /* ICB doesn't support /notify */
#ifdef SUPPORT_ICB
   if (get_server_version(primary_server) == ServerICB)
     return;
#endif
   from_server = primary_server;
   *p = '\0';
   for (tmp = notify_list; tmp; tmp = tmp->next)
     {
	nlen = strlen(tmp->nick);
	if ((sizeof(buf) - (p - buf) < nlen + 2)
	    || count == 5)
	  {
	     /* kludge */
	     if (count == 0)
	       break;
	     /* terminate/add/reset */
	     *p = '\0';
	     p = buf;
	     add_userhost_to_whois(p, userhost_notify);
	     *p = '\0';
	     count = 0;
	  }
	
	/* add one to the list.. */
	if (*buf != '\0')
	  {
	     my_strcat(p, " ");
	     p++;
	  }
	my_strncat(p, tmp->nick, nlen);
	p += nlen;
	count++;
     }
   if (count)
     {
	*p = '\0';
	add_userhost_to_whois(buf, userhost_notify);
     }
   from_server = old_server;
}

/*
 * notify_mark: This marks a given person on the notify list as either on irc
 * (if flag is 1), or not on irc (if flag is 0).  If the person's status has
 * changed since the last check, a message is displayed to that effect.  If
 * the person is not on the notify list, this call is ignored 
 * doit if passed as 0 means it comes from a join, or a msg, etc, not from
 * an ison reply.  1 is the other..
 * we also bail out if the primary_server != from_server and the from-server
 * is valid.
 *
 * NOTE:  this function should be called with no particular to_window or other
 * variables set that would affect what window it would be displayed in.
 * ideally, a message_from((u_char *) 0, LOG_CURRENT) should be the what is the
 * current window level.
 */
void
notify_mark(nick, flag, doit, user, host, away)
	u_char	*nick;
	int	flag;
	int	doit;
	u_char	*user;
	u_char	*host;
	int	away;
{
   NotifyList	*tmp;
   u_char	*s = get_string_var(NOTIFY_HANDLER_VAR);
   int		matched = 1;
   
   if (from_server != primary_server && from_server != -1)
     return;
   /* if (!s || (!doit && 'O' == *s)) - this broke notify -gkm *//* old notify */
   if (!doit && 'O' == *s)		/* old notify */
     return;	
   if ('N' == *s)			/* noisy notify */
     doit = 1;
   if ((tmp = (NotifyList *) list_lookup((List **) &notify_list, nick,
					 !USE_WILDCARDS, !REMOVE_FROM_LIST)) != NULL)
     {
	if (tmp->mask)
	  {
	     u_char tbuf[384];
	     snprintf(tbuf, sizeof(tbuf)-1, "%s@%s", user, host);
	     tbuf[sizeof(tbuf)-1] = '\0';
	     matched = match(tmp->mask, tbuf);
	  }
	if (flag)
	  {
	     if (!(tmp->flags & NOTIFY_ONLINE))
	       {
		  u_char *away_str = UP(" (away)");
		  if (doit && matched && do_hook(NOTIFY_SIGNON_LIST, "%s %s %s %s", nick, user, host, away ? "G" : "H"))
		    say("Notify %s: %s (%s@%s) is now online.%s", update_clock(0, 0, GET_TIME), nick, user, host, away ? away_str : empty_string);
		  if ((tmp->flags & NOTIFY_LOG) && doit && matched && server_list[from_server].away_set)
		    add_to_awaylog("Notify %s: %s (%s@%s) is now online.%s", update_clock(0, 0, GET_TIME), nick, user, host, away ? away_str : empty_string);
		  /*
		   * copy the correct case of the nick
		   * into our array  ;)
		   */
		  malloc_strcpy(&(tmp->nick), nick);
		  malloc_strcpy(&(tmp->user), user);
		  malloc_strcpy(&(tmp->host), host);
		  malloc_strcpy(&last_notify_nick, nick);
		  tmp->flags |= NOTIFY_ONLINE;
		  
		  /* see if we do something for this friend upon notify */
		  check_friend_notify(nick, user, host);
	       }
	  }
	else
	  {
	     if ((tmp->flags & NOTIFY_ONLINE) && doit && matched && do_hook(NOTIFY_SIGNOFF_LIST, "%s %s %s", nick, tmp->user, tmp->host))
	       say("Notify %s: %s (%s@%s) has left irc.", update_clock(0, 0, GET_TIME), nick, tmp->user, tmp->host);
	     tmp->flags &= ~NOTIFY_ONLINE;
	  }
     }
}

int
save_notify(fp)
	FILE	*fp;
{
   NotifyList	*tmp;
   int		count = 0, tc = 0;
   u_char	*list = UNULL;
   
   if (notify_list)
     {
	for (tmp = notify_list; tmp; tmp = tmp->next)
	  {
	     if (!list)
	       malloc_strcat(&list, " ");
	     if (tmp->flags & NOTIFY_LOG)
	       malloc_strcat(&list, "%");
	     malloc_strcat(&list, tmp->nick);
	     if (tmp->mask)
	       {
		  malloc_strcat(&list, "!");
		  malloc_strcat(&list, tmp->mask);
	       }
	     
	     count++;
	     tc++;
	     
	     if (count == 5)
	       {
		  fprintf(fp, "NOTIFY %s\n", list);
		  new_free(&list);
		  count = 0;
	       }
	  }
	if (count > 0 && list)
	  {
	     fprintf(fp, "NOTIFY %s\n", list);
	     new_free(&list);
	  }
     }
   return tc;
}

/* I hate broken compilers -mrg */
static	char	*vals[] = { "NOISY", "QUIET", "OLD", (char *) 0 };

void
set_notify_handler(value)
	u_char	*value;
{
   size_t	len;
   int	i;
   char	*s;
   
   if (!value)
     value = empty_string;
   for (i = 0, len = my_strlen(value); (s = vals[i]); i++)
     if (0 == my_strnicmp(value, UP(s), len))
       break;
   set_string_var(NOTIFY_HANDLER_VAR, UP(s));
   return;
}

static void
free_notify(nent)
    NotifyList **nent;
{
   NotifyList *new = *nent;
   
   new_free(&(new->nick));
   new_free(&(new->user));
   new_free(&(new->host));
   new_free(&(new->mask));
   new_free(&new);
}
