/*
 * ntesla - a program for designing Tesla Coils
 * 
 * Copyright (C) 1997  Steven A. Falco
 * 
 * Email contact: sfalco@worldnet.att.net
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define EXTERN extern
#include "protos.h"

/* Find self capacitance via Medhurst formula - see interp.c for reference. */
double
selfCapMedhurst(double length, double diameter)
{
	double metricDiameter = diameter * 2.54;
	double kFactor = findMedhurstF(length / diameter);

	return kFactor * metricDiameter;
}

/* Resonant frequency */
double
res_freq(double ind, double cap)
{
	return(1e-3 / (2.0 * PI * sqrt(ind * cap / 1e15)));
}

/* Calculate all the secondary solonoid parameters */
void
solonoidParms(double wire, double diameter, double length)
{
	double radius, ratio, iratio, solFactor, turns, wirelength, parmV;
	double distCap, induct, resonate, reslower, react, rdc;
	double qideal, qterm, qun, qload, total_cap;

	radius	   = diameter / 2.0;
	ratio	   = diameter / length;
	iratio	   = length / diameter;	/* inverted ratio */
	solFactor  = findSolFF(ratio);
	turns	   = SPACEFACTOR * length / wire;
	wirelength = PI * diameter * turns;
	parmV	   = length / radius;
	distCap	   = selfCapMedhurst(length, diameter);
	induct	   = solFactor * turns * turns * diameter / 1e3;
	resonate   = res_freq(induct, distCap);
	total_cap  = distCap + Tcparms.terminal_cap;
	reslower   = res_freq(induct, total_cap);
	react	   = 2.0 * PI * resonate * induct;
	rdc	   = inch_to_ohm(wire) * wirelength;
	qideal	   = findQFactor(iratio) * diameter * sqrt(resonate * 1e-3);
	qterm	   = findQFactor(iratio) * diameter * sqrt(reslower * 1e-3);

	/* Derated versions of Q based on TESLAC.EXE.  I don't have any
	 * justification for this.  If anyone has a more realistic formula
	 * for Q, let me know.  This will do for now...
	 */
	qun	   = qideal / 2.71;
	qload	   = qterm / 6.0;

	Tcparms.secondary_lfreq = reslower;

S "               Number of turns = %g (closewound)\n", turns);
S "            Aspect ratio (L/D) = %g\n", length / diameter);
S "            Resonant frequency = %g kHz\n", resonate);
S "Resonant frequency with toroid = %g kHz\n", reslower);
S "                   Wire length = %s\n", UnitOut((void *)&wirelength,
					Tcparms.units ? TOFEET : TOMETER, Ub0));
S "                 DC resistance = %g ohms\n", rdc);
S "                 AC resistance = %g ohms\n", react / qun);
S "      Inductive reactance (XL) = %g ohms\n", react);
S "                    Inductance = %g millihenry\n", induct);
S "                    Unloaded Q = %g\n", qun);
S "                 Q with toroid = %g\n", qload);
S "       Distributed capacitance = %g pF\n\n", distCap);
}

/* Build the screen image */
void
secgen(int mode)
{
	Row = 0;
S "         *Gauge: %s  *Diameter: %s  *Length: %s\n",
   UnitOut((void *)&Tcparms.secondary_wire, Tcparms.units ? TOBS : TOWMM, Ub0),
   UnitOut((void *)&Tcparms.secondary_diam, Tcparms.units ? TOINCH : TOCM, Ub1),
   UnitOut((void *)&Tcparms.secondary_leng, Tcparms.units ? TOINCH : TOCM, Ub2));
S "         *Toroid capacitance: %g pF\n", Tcparms.terminal_cap);
	if(mode) {
S "(You can change the toroid capacitance here, but you must change it on the\n");
S "Toroid screen for the change to be permanent.)\n");
S "\n");
	}
	solonoidParms(Tcparms.secondary_wire.thick, Tcparms.secondary_diam,
	  Tcparms.secondary_leng);
}

/* Secondary menu */
void
secondary()
{
	char buf[BS];
	double tmpval;
	int iii;

	while(1) {
		secgen(1);	

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up general instructions */
		general_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'g':
			UnitIn(atof(buf + 1), (void *)&Tcparms.secondary_wire,
			  Tcparms.units ? TOBS : TOWMM);
		  	break;

		  case 'd':
			UnitIn(atof(buf + 1), (void *)&tmpval,
			  Tcparms.units ? TOINCH : TOCM);
			if((Tcparms.secondary_leng / tmpval <= 10.0)
			   &&
			   (Tcparms.secondary_leng / tmpval >= 0.1)) {
				if(tmpval < Tcparms.primary_inner) {
					Tcparms.secondary_diam = tmpval;
				} else {
					E "\007Secondary diameter must be less than primary inner diameter ");
				}
			} else {
				restrict();
			}
		  	break;

		  case 'l':
			UnitIn(atof(buf + 1), (void *)&tmpval,
			  Tcparms.units ? TOINCH : TOCM);
			if((tmpval / Tcparms.secondary_diam <= 10.0)
			   &&
			   (tmpval / Tcparms.secondary_diam >= 0.1)) {
				Tcparms.secondary_leng = tmpval;
			} else {
				restrict();
			}
		  	break;

		  case 't':
		  	Tcparms.terminal_cap = atof(buf + 1);
		  	break;

		  case 'h':
		  	sec_help();
			break;

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

void
sec_help()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;
S "Your secondary should be wound using magnet (enamelled) wire on a low-\n");
S "loss coil form.  Most coils should use approximately #22 gauge wire.  You\n");
S "want around 500 to 1000 turns of wire for best results.\n");
S "\n");
S "Please note that PVC pipe is probably the most commonly used coil-form, but\n");
S "PVC is somewhat porous, and the trapped moisture will degrade the performance\n");
S "of your coil.  Some folks advocate baking the coil form at perhaps 120 degrees\n");
S "Fahr. for a few days, then coating with a water-proof varnish or epoxy\n");
S "paint.  If you can find a low loss plastic, you can skip this step.\n");
S "\n");
S "Also, you do NOT want the wires of your coil to go inside the coil-form.\n");
S "Keep all the wire on the outside of the form; use varnish to hold the turns\n");
S "in place.  If wires enter the form, there is a risk of arcing which can\n");
S "destroy the form.  For the same reason, don't use metal hardware in the\n");
S "construction of the secondary.\n");
S "\n");
S "Your secondary needs a good RF ground to function properly.  Read the\n");
S "recommendations on the net for more information on constructing an RF\n");
S "ground.\n");
S "\n");


		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}
