/*
 * ntesla - a program for designing Tesla Coils
 * 
 * Copyright (C) 1997  Steven A. Falco
 * 
 * Email contact: sfalco@worldnet.att.net
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define EXTERN extern
#include "protos.h"

#define OL	3.25	/* overload factor for pole pigs */

/* Build the screen image */
void
transgen(int mode)
{
	int gaps;
	double w;	/* frequency in radians */

	Row = 0;
	w = 2.0 * PI * Tcparms.xf_freq;
	if(Tcparms.primary_type == SHUNTED) {
S "  *Type: %s  *Gap: %s\n",
   Tcparms.primary_type == SHUNTED ? "shunted (neon)" : "non-shunted (pole pig)",
   Tcparms.primary_gap == ROTARY ? "fast rotary" : "slow rotary or fixed");
S "  *Voltage: %g Volts  *Amperage: %g mA  *Frequency: %g Hz\n",
   Tcparms.primary_volts, Tcparms.primary_amps * 1e3, Tcparms.xf_freq);

		Tcparms.primary_va =
			Tcparms.primary_volts * Tcparms.primary_amps;

		/* Magic fudge factors from TESLAC.EXE */
		if(Tcparms.primary_gap == ROTARY) {
			Tcparms.primary_cap =
			  (0.8 * 1e6 * Tcparms.primary_amps)
			  /
			  (w * M_SQRT2 * Tcparms.primary_volts);
		} else {
			Tcparms.primary_cap =
			  (1e6 * Tcparms.primary_amps)
			  /
			  (w * M_SQRT2 * Tcparms.primary_volts);
		}
	} else {
S "  *Type: %s  *Gap: %s\n",
   Tcparms.primary_type == SHUNTED ? "shunted (neon)" : "non-shunted (pole pig)",
   Tcparms.primary_gap == ROTARY ? "fast rotary" : "slow rotary or fixed");
S "  *Voltage: %g Volts  *Wattage: %g VA  *Frequency: %g Hz\n",
   Tcparms.primary_volts, Tcparms.primary_va, Tcparms.xf_freq);
S "\n");
S "(Capacitor recommendation assumes short-duration overload of transformer to\n");
S "about %g watts (10%%%% duty cycle).\n", Tcparms.primary_va * OL);

		Tcparms.primary_amps =
			Tcparms.primary_va / Tcparms.primary_volts;

		/* Magic fudge factors from TESLAC.EXE */
		if(Tcparms.primary_gap == ROTARY) {
			Tcparms.primary_cap =
			  (0.8 * OL * 1e6 * Tcparms.primary_va)
			  /
			  (w * pow(M_SQRT2 * Tcparms.primary_volts, 2.0));
		} else {
			Tcparms.primary_cap =
			  (OL * 1e6 * Tcparms.primary_va)
			  /
			  (w * pow(M_SQRT2 * Tcparms.primary_volts, 2.0));
		}
	}
S "\n");
	if(mode) {
S "Set type to 's' for a shunted (neon) transformer, or 'n' for a non-shunted\n");
S "(pole pig) transformer.  Set gap to 'r' for a fast rotary gap (above 360 BPS),\n");
S "or 'f' for a fixed or slow rotary gap (below 360 BPS).\n");
S "\n");
	}
S "        Maximum capacitance = %.3g uF\n", Tcparms.primary_cap);
S "   Capacitor voltage rating = %g V\n", Tcparms.primary_volts * 2.5);

	/* Magic fudge factor from TESLAC.EXE */
	gaps = (int)((Tcparms.primary_volts / 1786.0) + 0.5);
S "Number of series spark gaps = %d\n", gaps);
S "\n");
}

/* Transformer menu */
void
transformer()
{
	char buf[BS], *bp;
	int iii;

	while(1) {
		transgen(1);	

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up general instructions */
		general_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'v':
		  	Tcparms.primary_volts = atof(buf + 1);
		  	break;

		  case 'a':
		  	Tcparms.primary_amps = atof(buf + 1) * 1e-3;
		  	break;

		  case 'w':
		  	Tcparms.primary_va = atof(buf + 1);
		  	break;

		  case 'f':
		  	Tcparms.xf_freq = atof(buf + 1);
		  	break;

		  case 't':
			bp = buf + 1;
			while(*bp != 0 && *bp == ' ') bp++;
			if(tolower(*bp) == 's') {
				Tcparms.primary_type = SHUNTED;
			} else if(tolower(*bp) == 'n') {
				Tcparms.primary_type = NONSHUNTED;
			} else {
				E "\007Sorry - type cannot be set to %c - try again ", *bp);
			}
		  	break;

		  case 'g':
			bp = buf + 1;
			while(*bp != 0 && *bp == ' ') bp++;
			if(tolower(*bp) == 'r') {
				Tcparms.primary_gap = ROTARY;
			} else if(tolower(*bp) == 'f') {
				Tcparms.primary_gap = FIXED;
			} else {
				E "\007Sorry - gap cannot be set to %c - try again ", *bp);
			}
		  	break;

		  case 'h':
		  	trans_help();
			break;

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

void
trans_help()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;
S "Transformers come in two varieties: shunted and non-shunted.  The shunted\n");
S "variety is recommended for beginners because it is simpler to work with, less\n");
S "dangerous, and less expensive.  Choices are: neon transformers, often\n");
S "available used from a neon sign shop, and oil-burner ignition transformers,\n");
S "often available used from a heating service shop.\n");
S "\n");
S "You probably want a voltage of at least 5000 volts, and a current of about\n");
S "30 mA (0.03 amps) or more.  If you can find a 12000 or 15000 volt neon\n");
S "transformer, you are set.  Whatever transformer you pick, you should use\n");
S "a protective spark gap and choke coils to protect it from excessive voltages\n");
S "from your main coil.  Look on the Tesla lists and the net for more details\n");
S "about this.\n");
S "\n");
S "The best spark gap for beginners is a static gap.  You want several gaps of\n");
S "perhaps 0.030\" in series.  A single gap will have poor performance.  The\n");
S "actual number of series gaps is determined by your transformer.  A typical\n");
S "number might be 6 or 7 series gaps for the 12000 volt neon transformer\n");
S "recommended above.  Look for plans for the \"Richard Quick\" spark gap.\n");
S "It is pretty easy to build and has good performance.\n");
S "\n");


		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}
