#include <unistd.h>
#include <sys/types.h>
#include <time.h>
#include <regex.h>
#include <signal.h>
#include <sys/wait.h>
#include "config.h"

#define LINE 2048
#define DEBUG 0

#define OAK_VERSION 1.5.0
#define OAK_VERSION_STRING "1.5.0"

typedef struct _oak_list {
  int size;
  int avail;
  void **list;
} oak_list;

typedef struct _oak_string {
  char *buff;
  int avail;
  int len;
} oak_string;

typedef struct _oak_fire {
  time_t next;       /* the unix time at which to fire next */
  time_t interval;   /* the number of seconds between intervals */
} oak_fire;

typedef struct _oak_queue_entry {
  char *msg;
  int repcount;
  char *host;
} oak_queue_entry;

typedef struct _oak_queue {
  char *name;
  oak_list entrylist;
  oak_fire *fire;
  oak_list *actionlist;
  int locktime;
  char *header;
  int trashqueue;
  int prescan;
} oak_queue;

typedef struct _oak_action {
  char *name;
  int argc;
  char **argv;
  int haslimits;
  int limit_lines;
  int limit_linelen;
  int limit_numhosts;
  int limit_hostents;
} oak_action;

typedef oak_list oak_matchlist;
typedef oak_list oak_queuelist;
typedef oak_list oak_timelist;

typedef struct _oak_matchline {
  regex_t regex;
  oak_queuelist ql;
  oak_queuelist locked_queuelist;
  oak_timelist  lock_times;
} oak_matchline;

typedef struct _oak_global {
  char logfile[1024];
  oak_queuelist queuelist;
  oak_matchlist matchlist;
  int quit;
  int nukepid;
  int nukeciscoid;
  int nukesmqid;
  int onlyhosts;
  int ignorehosts;
  int debug;
  regex_t soljunk_regex;
  regex_t pid_regex;
  regex_t ciscopid_regex;
  regex_t smqid_regex;
  oak_list hostlist;
  int hostsort;
  oak_list hostsortlist;
  char replacestr[100];
} oak_global;

oak_global g;

/* -------------------------------- action.c -------------------------------- */
extern oak_action *oak_action_create(char *name, int argc, char **argv);
extern oak_action *oak_action_create_from_argstring(char *name, char *args);
extern void oak_action_set_limits(oak_action *act, int lines, int linelen, int numhosts, int hostents);
extern void oak_action_execute(oak_action *act, oak_queue *q);
extern char *oak_action_get_name(oak_action *act);

/* -------------------------------- fire.c -------------------------------- */
extern oak_fire *oak_fire_create(char *in);
extern int oak_fire_is_time(oak_fire *f);

/* -------------------------------- global.c -------------------------------- */
extern void oak_global_init(oak_global *g);
extern int oak_global_is_debug(oak_global *g);
extern void oak_global_set_debug(oak_global *g);
extern void oak_global_set_nodebug(oak_global *g);
extern char *oak_global_get_logfile(oak_global *g);
extern void oak_global_set_logfile(oak_global *g, char *filename);
extern oak_queuelist *oak_global_get_queuelist(oak_global *g);
extern oak_matchlist *oak_global_get_matchlist(oak_global *g);
extern int oak_global_is_nukepid(oak_global *g);
extern void oak_global_set_nukepid(oak_global *g);
extern void oak_global_set_no_nukepid(oak_global *g);
extern int oak_global_is_nukeciscoid(oak_global *g);
extern void oak_global_set_nukeciscoid(oak_global *g);
extern void oak_global_set_no_nukeciscoid(oak_global *g);
extern int oak_global_is_nukesmqid(oak_global *g);
extern void oak_global_set_nukesmqid(oak_global *g);
extern void oak_global_set_no_nukesmqid(oak_global *g);
extern char *oak_global_get_replacestr(oak_global *g);
extern void oak_global_set_replacestr(oak_global *g, char *in);
extern int oak_global_host_is_ok(oak_global *g, char *host);
extern int oak_global_set_ignorehosts(oak_global *g, char *hosts);
extern int oak_global_set_onlyhosts(oak_global *g, char *hosts);
extern void _oak_global_set_hostlist(oak_global *g, char *hosts);
extern regex_t *oak_global_get_regex_soljunk(oak_global *g);
extern regex_t *oak_global_get_regex_pid(oak_global *g);
extern regex_t *oak_global_get_regex_ciscopid(oak_global *g);
extern regex_t *oak_global_get_regex_smqid(oak_global *g);
extern void oak_global_set_quit(oak_global *g);
extern int oak_global_is_quit(oak_global *g);

/* -------------------------------- list.c -------------------------------- */
extern int oak_list_create(oak_list *l);
extern int oak_list_get_size(oak_list *l);
extern void *oak_list_get_element(oak_list *l, int n);
extern int oak_list_append_element(oak_list *l, void *element);
extern int oak_list_remove_element(oak_list *l, int n);
extern void oak_list_free_all(oak_list *l, void (*elefree)(void *));
extern void oak_list_free_simple(oak_list *l);

/* -------------------------------- matchline.c -------------------------------- */
extern oak_matchline *oak_matchline_create(char *regex, char *queuenames);
extern int oak_matchline_set_queuelist_from_string(oak_matchline *m, char *queuenames);
extern int oak_matchline_string_matches(oak_matchline *m, char *instr, char *out);
extern oak_queuelist *oak_matchline_get_queuelist(oak_matchline *m);
extern int oak_matchline_locked_for_queue(oak_matchline *m, oak_queue *q);
extern void oak_matchline_set_lock(oak_matchline *m, oak_queue *q);

/* -------------------------------- oak.c -------------------------------- */
extern int main(int argc, char **argv);
extern void oak_print_usage(int err);
extern void sig_handler(int sig, siginfo_t *si, void *data);

/* -------------------------------- queue.c -------------------------------- */
extern oak_queue *oak_queue_create(char *name, oak_list *actionlist, oak_fire *fire);
extern char *oak_queue_get_name(oak_queue *q);
extern int oak_queue_is_prescan(oak_queue *q);
extern void oak_queue_set_prescan(oak_queue *q);
extern int oak_queue_get_nummsgs(oak_queue *q);
extern char *oak_queue_get_msg_n(oak_queue *q, int n);
extern char *oak_queue_get_host_n(oak_queue *q, int n);
extern int oak_queue_get_msgcount_n(oak_queue *q, int n);
extern void oak_queue_set_header(oak_queue *q, char *header);
extern char *oak_queue_get_header(oak_queue *q);
extern int oak_queue_is_locking(oak_queue *q);
extern void oak_queue_set_locktime_fromstring(oak_queue *q, char *in);
extern int oak_queue_get_locktime(oak_queue *q);
extern void oak_queue_add_msg(oak_queue *q, char *msg, char *host);
extern void oak_queue_delete_msgs(oak_queue *q);
extern oak_fire *oak_queue_get_fire(oak_queue *q);
extern oak_list *oak_queue_get_actionlist(oak_queue *q);
extern void oak_queue_set_fire(oak_queue *q, oak_fire *fire);
extern void oak_queue_set_actionlist(oak_queue *q, oak_list *actionlist);
extern char *oak_queue_to_string(oak_queue *q, int lines, int linelen, int numhosts, int hostents);

/* -------------------------------- queue_entry.c -------------------------------- */
extern void oak_queue_entry_create(oak_queue_entry *qe, char *host, char *msg);
extern int oak_queue_entry_matches(oak_queue_entry *qe, char *host, char *str);
extern void oak_queue_entry_up_repcount(oak_queue_entry *qe);
extern int oak_queue_entry_get_repcount(oak_queue_entry *qe);
extern char *oak_queue_entry_get_msg(oak_queue_entry *qe);
extern char *oak_queue_entry_get_host(oak_queue_entry *qe);
extern void oak_queue_entry_free(oak_queue_entry *qe);

/* -------------------------------- queuelist.c -------------------------------- */
extern oak_queue *oak_queuelist_get_queue_by_name(oak_queuelist *ql, char *name);

/* -------------------------------- readconfig.c -------------------------------- */
extern void oak_readconfig(char *file);

/* -------------------------------- string.c -------------------------------- */
extern int oak_string_create(oak_string *s, char *in);
extern int oak_string_create_with_size(oak_string *s, int size);
extern char *oak_string_to_str(oak_string *s);
extern void oak_string_strcat(oak_string *s, char *in);
extern int oak_string_get_strlen(oak_string *s);
extern void oak_string_free(oak_string *s);

/* -------------------------------- text.c -------------------------------- */
extern int oak_text_trim_string_by_lines(char *instr, char *outstr, int lines);
extern void oak_text_replace_string_sections(char *in, regmatch_t pmatch[], char *out, char *newstr);
extern char *oak_text_get_line_from_log(char *log);
extern char *oak_text_get_host_from_log(char *log);

/* -------------------------------- util.c -------------------------------- */
extern void oak_util_debugmsg(char *fmt, ...);
extern char **argstring_to_argv(char *in, int *outargc);

