/*
	XHelperClasses.h for OBPager, a pager dockapp designed to work with OpenBox or any netwm-compliant window manager.
	
	Copyright (c) 2004 - Roy Wood
	
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	See the COPYING file for a copy of the GNU General Public License.
*/


#ifndef __X_HELPER_CLASSES__
#define __X_HELPER_CLASSES__

#include <iostream>
#include <string>

#include <Xlib.h>
#include <Xutil.h>
#include <Xos.h>
#include <Xatom.h>
#include <shape.h>


//#define __DEBUG__


// The following are a collection of classes used to manage the ownership of Xlib resources.
// Technically, these resources are all released when an X app exits, so these classes might not be necessary, depending on how your app is written.



/**
 * This is a template class to manage a generic X resource.  
 * In particular, it handles releasing the resource via XFree() upon object destruction.
 */
 
template <class T> class XResourceKeeper
{
	public:
		
		/// Default constructor
		
		XResourceKeeper() : mResource(NULL)
		{
			
		}
		
		
		/// Constructor with X resource/pointer specified
		
		XResourceKeeper(T *xResource) : mResource(NULL)
		{
			acquire(xResource);
		}
		
		
		/// Destructor that releases the owned resource
		
		~XResourceKeeper()
		{
			release();
		}
		
		
		/// Take ownership of a resource after releasing the previously-owned resource (if appropriate)
		
		void acquire(T* xResource)
		{
			release();
			
			mResource = xResource;
			
			#ifdef __DEBUG__
				std::cout << "XResourceKeeper::acquire(): Taking ownership of X resource 0x" << std::hex << (unsigned long) mResource << std::dec << std::endl;
			#endif
		}
		
		
		/// Release the owned resource (if any)
		
		void release()
		{
			T* oldResource = mResource;
			
			mResource = NULL;
			
			if (oldResource != NULL) {
				#ifdef __DEBUG__
					std::cout << "XResourceKeeper::release(): Releasing ownership of X resource 0x" << std::hex << (unsigned long) oldResource << std::dec << std::endl;
				#endif
					
				XFree(oldResource);
			}
		}
		
		
		/// Get the resource pointer
		
		T* get() const
		{
			return mResource;
		}
		
		
		/// Get the resource pointer
		
		T* operator() (void) const
		{
			return mResource;
		}
		
		
		/// Access the resource as a pointer
		
		T* operator->() const
		{
			return mResource;
		}
		
		
		/// Access the resource as a pointer
		
		T* operator*() const
		{
			return *mResource;
		}
		
		
		private:
			
			// Hide these
			XResourceKeeper(XResourceKeeper&);
			XResourceKeeper(const XResourceKeeper&);
			XResourceKeeper& operator=(XResourceKeeper&);
			XResourceKeeper& operator=(const XResourceKeeper&);
			
			
			// This is the pointer we manage
			
			T *mResource;
};




// The Xlib "Display" struct is opaque and not publicly defined, so we fake things....

struct _XDisplay
{
	// Nothing here!
};


/**
 * This is a class to manage an X "Display" reference.  In particular, it handles closing the display upon object destruction.
 */

class XDisplayKeeper
{
	public:
		
		/// Default constructor
		
		XDisplayKeeper() : mDisplay(NULL)
		{
			
		}
		
		
		/// Constructor with the Display specified
		
		XDisplayKeeper(Display* xDisplay) : mDisplay(NULL)
		{
			acquire(xDisplay);
		}
		
		
		/// Destructor closes the display
		
		~XDisplayKeeper()
		{
			release();
		}
		
		
		/// Take ownership of a Display pointer; release the previously-owned Display, if any
		
		void acquire(Display* xDisplay)
		{
			release();
			
			mDisplay = xDisplay;
			
			#ifdef __DEBUG__
				std::cout << "XDisplayKeeper::acquire(): Taking ownership of X Display 0x" << std::hex << (unsigned long) mDisplay << std::dec << std::endl;
			#endif
		}
		
		
		/// Release ownership of the Display pointer
		
		void release()
		{
			Display* oldDisplay = mDisplay;
			
			mDisplay = NULL;
			
			if (oldDisplay != NULL) {
				#ifdef __DEBUG__
					std::cout << "XDisplayKeeper::release(): Releasing ownership of X Display 0x" << std::hex << (unsigned long) oldDisplay << std::dec << std::endl;
				#endif
					
				XCloseDisplay(oldDisplay);
			}
		}
		
		
		/// Get the Display pointer
		
		Display* get() const
		{
			return mDisplay;
		}
		
		/// Get the Display pointer
		
		Display* operator() (void) const
		{
			return mDisplay;
		}
		
		
		private:
			
			// Hide these
			XDisplayKeeper(XDisplayKeeper&);
			XDisplayKeeper(const XDisplayKeeper&);
			XDisplayKeeper& operator=(XDisplayKeeper&);
			XDisplayKeeper& operator=(const XDisplayKeeper&);
			
			
			/// The Display we manage
			
			Display *mDisplay;
};



/**
 * This is a class to manage an X "Window" reference.  In particular, it handles destroying the window upon object destruction.
 *
 * Note that this class also requires a copy of the X Display pointer, since that is required for the call to XDestroyWindow (dunno why,
 * but it is).  The Display pointer is NOT managed though, so the user of this class is still required to look after closing the Display
 * when appropriate (perhaps by using an XDisplayKeeper, though you must be careful about the creation/destruction order of the XDisplayKeeper
 * and the XWindowKeeper to avoid accidentally closing the Display prematurely).
 */

class XWindowKeeper
{
	public:
		
		/// Default constructor
		
		XWindowKeeper() : mDisplay(NULL), mWindow(0)
		{
			
		}
		
		
		/// Constructor with the Window specified)
		
		XWindowKeeper(Display* xDisplay, Window xWindow) : mDisplay(NULL), mWindow(0)
		{
			acquire(xDisplay, xWindow);
		}
		
		
		/// Destructor takes care of destroying the Window
		
		~XWindowKeeper()
		{
			release();
		}
		
		
		/// Take ownership of the Window; release any previously-owned Window first
		
		void acquire(Display* xDisplay, Window xWindow)
		{
			release();
			
			mDisplay = xDisplay;
			mWindow = xWindow;
			
			#ifdef __DEBUG__
				std::cout << "XWindowKeeper::acquire(): Taking ownership of X Window 0x" << std::hex << (unsigned long) mWindow << std::dec << std::endl;
			#endif
		}
		
		
		/// Release ownership of the Window
		
		void release()
		{
			Display* oldDisplay = mDisplay;
			Window oldWindow = mWindow;
			
			mDisplay = NULL;
			mWindow = 0;
			
			if (oldWindow != 0) {
				#ifdef __DEBUG__
					std::cout << "XWindowKeeper::release(): Releasing ownership of X Window 0x" << std::hex << (unsigned long) oldWindow << std::dec << std::endl;
				#endif
					
				XDestroyWindow(oldDisplay, oldWindow);
			}
		}
		
		
		/// Get the Window reference
		
		Window get() const
		{
			return mWindow;
		}
		
		/// Get the Window reference
		
		Window operator() (void) const
		{
			return mWindow;
		}
		
		
	private:
		
		// Hide these
		XWindowKeeper(XWindowKeeper&);
		XWindowKeeper(const XWindowKeeper&);
		XWindowKeeper& operator=(XWindowKeeper&);
		XWindowKeeper& operator=(const XWindowKeeper&);
		
		/// The X Display pointer (we do NOT manage it!)
		Display *mDisplay;
		
		/// The X Window reference we manage
		Window mWindow;
};



/**
 * This is a class to manage an X "XFontStruct" pointer.  In particular, it handles freeing the XFontStruct upon object destruction.
 *
 * Note that this class also requires a copy of the X Display pointer, since that is required for the call to XFreeFont (dunno why,
 * but it is).  The Display pointer is NOT managed though, so the user of this class is still required to look after closing the Display
 * when appropriate (perhaps by using an XDisplayKeeper, though you must be careful about the creation/destruction order of the XDisplayKeeper
 * and the XWindowKeeper to avoid accidentally closing the Display prematurely).
 */

class XFontStructKeeper
{
	public:
		
		/// Default constructor
		
		XFontStructKeeper() : mDisplay(NULL), mFontStruct(0)
		{
			
		}
		
		
		/// Constructor with XFontStruct* specified
		
		XFontStructKeeper(Display* xDisplay, XFontStruct* xFontStruct) : mDisplay(NULL), mFontStruct(0)
		{
			acquire(xDisplay, xFontStruct);
		}
		
		
		/// Destructor takes care of freeing the font
		
		~XFontStructKeeper()
		{
			release();
		}
		
		
		/// Take ownership of the XFontStruct
		
		void acquire(Display* xDisplay, XFontStruct* xFontStruct)
		{
			release();
			
			mDisplay = xDisplay;
			mFontStruct = xFontStruct;
			
			#ifdef __DEBUG__
				std::cout << "XFontStructKeeper::acquire(): Taking ownership of X FontStruct 0x" << std::hex << (unsigned long) mFontStruct << std::dec << std::endl;
			#endif
		}
		
		
		/// Release the managed XFontStruct by calling XFreeFont
		
		void release()
		{
			Display* oldDisplay = mDisplay;
			XFontStruct* oldFontStruct = mFontStruct;
			
			mDisplay = NULL;
			mFontStruct = NULL;
			
			if (oldFontStruct != NULL) {
				#ifdef __DEBUG__
					std::cout << "XFontStructKeeper::release(): Releasing ownership of X FontStruct 0x" << std::hex << (unsigned long) oldFontStruct << std::dec << std::endl;
				#endif
				
				XFreeFont(oldDisplay, oldFontStruct);
			}
		}
		
		
		/// Get the XFontStruct pointer
		
		XFontStruct* get() const
		{
			return mFontStruct;
		}
		
		
		/// Get the XFontStruct pointer
		
		XFontStruct* operator() (void) const
		{
			return mFontStruct;
		}
		
		
	private:
		
		// Hide these
		XFontStructKeeper(XFontStructKeeper&);
		XFontStructKeeper(const XFontStructKeeper&);
		XFontStructKeeper& operator=(XFontStructKeeper&);
		XFontStructKeeper& operator=(const XFontStructKeeper&);
		
		
		/// We need the Display*
		
		Display *mDisplay;
		
		/// The XFontStruct pointer we manage
		
		XFontStruct* mFontStruct;
};



/**
 * This is a class to manage an X "GC" reference.  In particular, it handles freeing the GC upon object destruction.
 *
 * Note that this class also requires a copy of the X Display pointer, since that is required for the call to XFreeGC (dunno why,
 * but it is).  The Display pointer is NOT managed though, so the user of this class is still required to look after closing the Display
 * when appropriate (perhaps by using an XDisplayKeeper, though you must be careful about the creation/destruction order of the XDisplayKeeper
 * and the XWindowKeeper to avoid accidentally closing the Display prematurely).
 */

class XGCKeeper
{
	public:
		
		/// Default constructor
		
		XGCKeeper() : mDisplay(NULL), mGC(0)
		{
			
		}
		
		
		/// Constructor with the GC specified
		
		XGCKeeper(Display* xDisplay, GC xGC) : mDisplay(NULL), mGC(0)
		{
			acquire(xDisplay, xGC);
		}
		
		
		/// Destructor releases the GC
		
		~XGCKeeper()
		{
			release();
		}
		
		
		/// Take ownership of a GC
		
		void acquire(Display* xDisplay, GC xGC)
		{
			release();
			
			mDisplay = xDisplay;
			mGC = xGC;
			
			#ifdef __DEBUG__
				std::cout << "XGCKeeper::acquire(): Taking ownership of X 'GC' graphics context 0x" << std::hex << (unsigned long) mGC << std::dec << std::endl;
			#endif
		}
		
		
		/// Release ownership of the GC by calling XFreeGC
		
		void release()
		{
			Display* oldDisplay = mDisplay;
			GC oldGC = mGC;
			
			mDisplay = NULL;
			mGC = 0;
			
			if (oldGC != 0) {
				#ifdef __DEBUG__
					std::cout << "XGCKeeper::release(): Releasing ownership of X 'GC' graphics context 0x" << std::hex << (unsigned long) oldGC << std::dec << std::endl;
				#endif
				
				XFreeGC(oldDisplay, oldGC);
			}
		}
		
		
		/// Get the GC reference
		
		GC get() const
		{
			return mGC;
		}
		
		
		/// Get the GC reference
		
		GC operator() (void) const
		{
			return mGC;
		}
		
		
	private:
		
		// Hide these
		XGCKeeper(XGCKeeper&);
		XGCKeeper(const XGCKeeper&);
		XGCKeeper& operator=(XGCKeeper&);
		XGCKeeper& operator=(const XGCKeeper&);
		
		
		/// We need the Display pointer to call XFreeGC
		
		Display *mDisplay;
		
		
		/// The GC we manage
		
		GC mGC;
};



/**
 * This is a class to manage an X "Pixmap" reference.  In particular, it handles freeing the PixMap upon object destruction.
 *
 * Note that this class also requires a copy of the X Display pointer, since that is required for the call to XFreePixmap (dunno why,
 * but it is).  The Display pointer is NOT managed though, so the user of this class is still required to look after closing the Display
 * when appropriate (perhaps by using an XDisplayKeeper, though you must be careful about the creation/destruction order of the XDisplayKeeper
 * and the XWindowKeeper to avoid accidentally closing the Display prematurely).
 */

class XPixmapKeeper
{
	public:
		
		/// Default constructor	
		
		XPixmapKeeper() : mDisplay(NULL), mPixmap(0)
		{
			
		}
		
		
		/// Constructor with Pixmap specified
		
		XPixmapKeeper(Display* xDisplay, Pixmap xPixmap) : mDisplay(NULL), mPixmap(0)
		{
			acquire(xDisplay, xPixmap);
		}
		
		
		/// Destructor releases the Pixmap
		
		~XPixmapKeeper()
		{
			release();
		}
		
		
		/// Take ownership of the Pixmap
		
		void acquire(Display* xDisplay, Pixmap xPixmap)
		{
			release();
			
			mDisplay = xDisplay;
			mPixmap = xPixmap;
			
			#ifdef __DEBUG__
				std::cout << "XPixmapKeeper::acquire(): Taking ownership of X 'Pixmap' graphics context 0x" << std::hex << (unsigned long) mPixmap << std::dec << std::endl;
			#endif
		}
		
		
		/// Release the Pixmap by calling XFreePixmap
		
		void release()
		{
			Display* oldDisplay = mDisplay;
			Pixmap oldPixmap = mPixmap;
			
			mDisplay = NULL;
			mPixmap = 0;
			
			if (oldPixmap != 0) {
				#ifdef __DEBUG__
					std::cout << "XPixmapKeeper::release(): Releasing ownership of X 'Pixmap' graphics context 0x" << std::hex << (unsigned long) oldPixmap << std::dec << std::endl;
				#endif
				
				XFreePixmap(oldDisplay, oldPixmap);
			}
		}
		
		
		/// Get the Pixmap reference
		
		Pixmap get() const
		{
			return mPixmap;
		}
		
		
		/// Get the Pixmap reference
		
		Pixmap operator() (void) const
		{
			return mPixmap;
		}
		
		
	private:
		
		// Hide these
		XPixmapKeeper(XPixmapKeeper&);
		XPixmapKeeper(const XPixmapKeeper&);
		XPixmapKeeper& operator=(XPixmapKeeper&);
		XPixmapKeeper& operator=(const XPixmapKeeper&);
		
		
		/// We need the Display pointer to call XFreePixmap
		
		Display *mDisplay;
		
		
		/// The Pixmap we manage
		
		Pixmap mPixmap;
};

#endif
