/*******************************************************************************
* Copyright (C) 2003-2004 Vintela, Inc. All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of source code must retain the above copyright notice,
*    this list of conditions and the following disclaimer.
*
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this list of conditions and the following disclaimer in the documentation
*    and/or other materials provided with the distribution.
*
*  - Neither the name of Vintela, Inc. nor the names of its
*    contributors may be used to endorse or promote products derived from this
*    software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL Vintela, Inc. OR THE CONTRIBUTORS
* BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/**
 * @author Dan Nuffer
 */

// Portions of this file based on utf/detail/utf8_algo.hpp
//
//  Copyright (c) 2003 Alberto Barbati
//
//  Permission to copy, use, modify, sell and distribute this software
//  is granted provided this copyright notice appears in all copies.
//  This software is provided "as is" without express or implied
//  warranty, and with no claim as to its suitability for any purpose.
#include "OW_config.h"
#include "OW_UTF8Utils.hpp"
#include "OW_String.hpp"
#include "OW_Assertion.hpp"
#include "OW_Array.hpp"
#include "OW_Format.hpp"
#include "OW_ExceptionIds.hpp"

#include <string.h> // for strlen
#include <algorithm> // for std::lower_bound

namespace OW_NAMESPACE
{

OW_DEFINE_EXCEPTION_WITH_ID(InvalidUTF8);

namespace UTF8Utils
{
namespace {
/*
   UTF-8 sequence length table.
   This table gives the sequence length according to the value of the
   sequence leading character.
   Notice: this table is different from the one found in the official
   UTF-8 conversion program, found here
   http://www.unicode.org/Public/PROGRAMS/CVTUTF/
   This table encodes only the sequence lenghts of UTF-8 sequences
   that can encode (or possibly encode) a Unicode character according
   to Unicode 3.2 requirements.
   In particular:
   - lines from 0x80 to 0xb0 (inclusive) contains "0" because those
   values do not represent a valid leading character
   - line 0xc0 begins with two "0" because values 0xc0 and 0xc1 lead
   non-shortest sequences, that are illegal since Unicode 3.1
   - line 0xf0 has only five "4" instead of eight and lacks values
   "5" and "6" because values above 0xf4 lead sequences that cannot
   encode a Unicode character.
*/
UInt8 SequenceLengthTable[256] =
{
	/* 0x00 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x0f */
	/* 0x10 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x1f */
	/* 0x20 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x2f */
	/* 0x30 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x3f */
	/* 0x40 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x4f */
	/* 0x50 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x5f */
	/* 0x60 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x6f */
	/* 0x70 */ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* 0x7f */
	/* 0x80 */ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0x8f */
	/* 0x90 */ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0x9f */
	/* 0xa0 */ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0xaf */
	/* 0xb0 */ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* 0xbf */
	/* 0xc0 */ 0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, /* 0xcf */
	/* 0xd0 */ 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, /* 0xdf */
	/* 0xe0 */ 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, /* 0xef */
	/* 0xf0 */ 4, 4, 4, 4, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0  /* 0xff */
};
} // end unnamed namespace
size_t charCount(const char* utf8str)
{
	OW_ASSERT(utf8str != 0);
	const char* p = utf8str;
	size_t count = 0;
	while (*p)
	{
		// any chars 0x80-0xBF are extension bytes.  Anything else signals a new char
		UInt8 c = static_cast<UInt8>(*p);
		if (c < 0x80 || c > 0xBF)
		{
			++count;
		}
		++p;
	}
	return count;
}
/////////////////////////////////////////////////////////////////////////////
UInt16 UTF8toUCS2(const char* utf8char)
{
	UInt32 c = UTF8toUCS4(utf8char);
	if (c > 0xFFFF)
	{
		return 0xFFFF;
	}
	else
	{
		return static_cast<UInt16>(c);
	}
}
/////////////////////////////////////////////////////////////////////////////
String UCS2toUTF8(UInt16 ucs2char)
{
	// UCS2 and UCS4 are the same, only different sizes.
	return UCS4toUTF8(ucs2char);
}
/////////////////////////////////////////////////////////////////////////////
UInt32 UTF8toUCS4(const char* utf8char)
{
	OW_ASSERT(utf8char != 0);
	OW_ASSERT(utf8char[0] != '\0');
	const char* p = utf8char;
	const UInt32 c0 = static_cast<UInt8>(p[0]);
	const UInt32 bad = 0xFFFFFFFF;
	switch (SequenceLengthTable[c0])
	{
		case 1:
		{
			return c0;
		}
		case 2:
		{
			// check for short (invalid) utf8 sequence
			if (p[1] == '\0')
			{
				return bad;
			}
			const UInt32 c1 = static_cast<UInt8>(p[1]);
			return ((c0 & 0x1fu) << 6) | (c1 & 0x3fu);
		}
		case 3:
		{
			// check for short (invalid) utf8 sequence
			if (p[1] == '\0' || p[2] == '\0')
			{
				return bad;
			}
			const UInt32 c1 = static_cast<UInt8>(p[1]);
			const UInt32 c2 = static_cast<UInt8>(p[2]);
			return ((c0 & 0x0fu) << 12) | ((c1 & 0x3fu) << 6) | (c2 & 0x3fu);
		}
		case 4:
		{
			// check for short (invalid) utf8 sequence
			if (p[1] == '\0' || p[2] == '\0' || p[3] == '\0')
			{
				return bad;
			}
			
			const UInt32 c1 = static_cast<UInt8>(p[1]);
			const UInt32 c2 = static_cast<UInt8>(p[2]);
			const UInt32 c3 = static_cast<UInt8>(p[3]);
			
			return ((c0 & 0x03u) << 18) | ((c1 & 0x3fu) << 12) | ((c2 & 0x3fu) << 6) | (c3 & 0x3fu);
		}
		default:
		{
			// invalid, just skip it
			break;
		}
	}
	return bad;
}

/////////////////////////////////////////////////////////////////////////////
String UCS4toUTF8(UInt32 ucs4char)
{
	StringBuffer sb(5); // max 4 chars + null
	UCS4toUTF8(ucs4char, sb);
	return sb.releaseString();
}

/////////////////////////////////////////////////////////////////////////////
void UCS4toUTF8(UInt32 ucs4char, StringBuffer& sb)
{
	if (ucs4char < 0x80u)
	{
		// one byte
		sb += static_cast<char>(static_cast<UInt8>(ucs4char));
	}
	else if (ucs4char < 0x800u)
	{
		sb += static_cast<char>(static_cast<UInt8>(0xc0u | (ucs4char >> 6)));
		sb += static_cast<char>(static_cast<UInt8>(0x80u | (ucs4char & 0x3fu)));
	}
	else if (ucs4char < 0x10000u)
	{
		sb += static_cast<char>(static_cast<UInt8>(0xe0u | (ucs4char >> 12)));
		sb += static_cast<char>(static_cast<UInt8>(0x80u | ((ucs4char >> 6) & 0x3fu)));
		sb += static_cast<char>(static_cast<UInt8>(0x80u | (ucs4char & 0x3fu)));
	}
	else
	{
		sb += static_cast<char>(static_cast<UInt8>(0xf0u | (ucs4char >> 18)));
		sb += static_cast<char>(static_cast<UInt8>(0x80u | ((ucs4char >> 12) & 0x3fu)));
		sb += static_cast<char>(static_cast<UInt8>(0x80u | ((ucs4char >> 6) & 0x3fu)));
		sb += static_cast<char>(static_cast<UInt8>(0x80u | (ucs4char & 0x3fu)));
	}
}

namespace
{
/////////////////////////////////////////////////////////////////////////////
void UCS4toUTF8(UInt32 ucs4char, char* p)
{
	if (ucs4char < 0x80u)
	{
		// one byte
		p[0] = static_cast<char>(static_cast<UInt8>(ucs4char));
	}
	else if (ucs4char < 0x800u)
	{
		p[0] = static_cast<char>(static_cast<UInt8>(0xc0u | (ucs4char >> 6)));
		p[1] = static_cast<char>(static_cast<UInt8>(0x80u | (ucs4char & 0x3fu)));
	}
	else if (ucs4char < 0x10000u)
	{
		p[0] = static_cast<char>(static_cast<UInt8>(0xe0u | (ucs4char >> 12)));
		p[1] = static_cast<char>(static_cast<UInt8>(0x80u | ((ucs4char >> 6) & 0x3fu)));
		p[2] = static_cast<char>(static_cast<UInt8>(0x80u | (ucs4char & 0x3fu)));
	}
	else
	{
		p[0] = static_cast<char>(static_cast<UInt8>(0xf0u | (ucs4char >> 18)));
		p[1] = static_cast<char>(static_cast<UInt8>(0x80u | ((ucs4char >> 12) & 0x3fu)));
		p[2] = static_cast<char>(static_cast<UInt8>(0x80u | ((ucs4char >> 6) & 0x3fu)));
		p[3] = static_cast<char>(static_cast<UInt8>(0x80u | (ucs4char & 0x3fu)));
	}
}

/////////////////////////////////////////////////////////////////////////////
Array<UInt16> StringToUCS2Common(const String& input, bool throwException)
{
	// TODO: Remove the duplication between this function and UTF8toUCS2()
	Array<UInt16> rval;
	OW_ASSERT(input.length() == ::strlen(input.c_str()));
	const UInt16 UCS2ReplacementChar = 0xFFFD;
	const char* begin = input.c_str();
	const char* end = begin + input.length();

	const char* p = begin;
	while (p < end)
	{
		const UInt32 c0 = static_cast<UInt8>(p[0]);
		switch (SequenceLengthTable[c0])
		{
			case 1:
			{
				rval.push_back(c0);
				++p;
			}
			break;
			case 2:
			{
				// check for short (invalid) utf8 sequence
				if (p[1] == '\0')
				{
					if (throwException)
					{
						OW_THROW(InvalidUTF8Exception, Format("Length: %1, input = %2, p = %3", 
							static_cast<int>(SequenceLengthTable[c0]), input.c_str(), p).c_str());
					}
					else
					{
						rval.push_back(UCS2ReplacementChar);
						p += 2;
					}
				}
				const UInt32 c1 = static_cast<UInt8>(p[1]);
				rval.push_back(((c0 & 0x1fu) << 6) | (c1 & 0x3fu));
				p += 2;
			}
			break;
			case 3:
			{
				// check for short (invalid) utf8 sequence
				if (p[1] == '\0' || p[2] == '\0')
				{
					if (throwException)
					{
						OW_THROW(InvalidUTF8Exception, Format("Length: %1, input = %2, p = %3", 
							static_cast<int>(SequenceLengthTable[c0]), input.c_str(), p).c_str());
					}
					else
					{
						rval.push_back(UCS2ReplacementChar);
						p += 3;
					}

				}
				const UInt32 c1 = static_cast<UInt8>(p[1]);
				const UInt32 c2 = static_cast<UInt8>(p[2]);
				rval.push_back(((c0 & 0x0fu) << 12) | ((c1 & 0x3fu) << 6) | (c2 & 0x3fu));
				p += 3;
			}
			break;
			case 4:
			{
				// UCS2 can't hold a value this big
				if (throwException)
				{
					OW_THROW(InvalidUTF8Exception, Format("Length: %1, input = %2, p = %3", 
						static_cast<int>(SequenceLengthTable[c0]), input.c_str(), p).c_str());
				}
				else
				{
					rval.push_back(UCS2ReplacementChar);
					p += 4;
				}

			}
			break;
			default:
			{
				if (throwException)
				{
					OW_THROW(InvalidUTF8Exception, Format("Length: %1, input = %2, p = %3", 
						static_cast<int>(SequenceLengthTable[c0]), input.c_str(), p).c_str());
				}
				else
				{
					rval.push_back(UCS2ReplacementChar);
					++p;
				}
			}
		}
	}
	return rval;
}

} // end anonymous namespace

/////////////////////////////////////////////////////////////////////////////
Array<UInt16> StringToUCS2ReplaceInvalid(const String& input)
{
	return StringToUCS2Common(input, false);
}

/////////////////////////////////////////////////////////////////////////////
Array<UInt16> StringToUCS2(const String& input)
{
	return StringToUCS2Common(input, true);
}

/////////////////////////////////////////////////////////////////////////////
String UCS2ToString(const void* input, size_t inputLength)
{
	// start out with 1 byte/char in input, this is just big enough for a
	// standard ASCII string.  If any chars are bigger, we'll only incur 1 or
	// 2 (worse case) reallocations of the buffer.
	size_t numchars = inputLength/2;
	StringBuffer sb(numchars + 1);
	for (size_t i = 0; i < numchars; ++i)
	{
		UCS4toUTF8(reinterpret_cast<const UInt16*>(input)[i], sb);
	}
	return sb.releaseString();
}

/////////////////////////////////////////////////////////////////////////////
String UCS2ToString(const Array<UInt16>& input)
{
	return UCS2ToString(&input[0], input.size() * sizeof(UInt16));
}
/////////////////////////////////////////////////////////////////////////////
String UCS2ToString(const Array<char>& input)
{
	return UCS2ToString(&input[0], input.size());
}

/////////////////////////////////////////////////////////////////////////////
namespace
{

int UTF8CharLen(UInt32 ucs4char)
{
	if (ucs4char < 0x80u)
	{
		return 1;
	}
	else if (ucs4char < 0x800u)
	{
		return 2;
	}
	else if (ucs4char < 0x10000u)
	{
		return 3;
	}
	else
	{
		return 4;
	}
}

template <typename TransformT>
bool transformInPlace(char* input, TransformT transformer)
{
	char* p = input;
	char* output = input;
	while (*p)
	{
		UInt32 ucs4char = UTF8toUCS4(p);
		if (ucs4char == 0xFFFFFFFF)
		{
			++p;
			++output;
			continue;
		}
		UInt32 newUcs4Char = transformer(ucs4char);

		const UInt32 c0 = static_cast<UInt8>(p[0]);
		int prevCharLen = SequenceLengthTable[c0];
		int newCharLen = UTF8CharLen(newUcs4Char);
		
		// can't grow the string, only shrink it. This can't happen with valid UTF8, but with invalid stuff it could.
		if (p > output)
		{
			return false;
		}

		// This is commented out, because, given the current set of data from Unicode 4.0.1, there are no
		// chars that grow when either upper or lower-cased.
		//if ((p + prevCharLen) < (output + newCharLen))
		//{
		//	return false;
		//}
		UCS4toUTF8(newUcs4Char, output);
		p += prevCharLen;
		output += newCharLen;
	}
	*output = '\0'; // null terminate in case the string shrunk
	return true;
}

template <typename TransformT>
String transform(const char* input, TransformT transformer)
{
	StringBuffer rval(strlen(input));
	const char* p = input;
	while (*p)
	{
		UInt32 ucs4char = UTF8toUCS4(p);
		if (ucs4char == 0xFFFFFFFF)
		{
			rval += *p;
			++p;
			continue;
		}

		UTF8Utils::UCS4toUTF8(transformer(ucs4char), rval);

		const UInt32 c0 = static_cast<UInt8>(p[0]);
		int prevCharLen = SequenceLengthTable[c0];
		p += prevCharLen;
	}
	return rval.releaseString();
}

// These mappings were generated by OW_GenCaseMappings.cpp with UnicodeData.txt from unicode 4.0.1 and downloaded from http://www.unicode.org/Public/UNIDATA/
// Don't modify it by hand, re-run the generator if something needs to change.
struct CaseMapping
{
	UInt32 codePoint;
	UInt32 mapping;
};
const CaseMapping lowerMappings[] =
{
	{0x41, 0x61},
	{0x42, 0x62},
	{0x43, 0x63},
	{0x44, 0x64},
	{0x45, 0x65},
	{0x46, 0x66},
	{0x47, 0x67},
	{0x48, 0x68},
	{0x49, 0x69},
	{0x4a, 0x6a},
	{0x4b, 0x6b},
	{0x4c, 0x6c},
	{0x4d, 0x6d},
	{0x4e, 0x6e},
	{0x4f, 0x6f},
	{0x50, 0x70},
	{0x51, 0x71},
	{0x52, 0x72},
	{0x53, 0x73},
	{0x54, 0x74},
	{0x55, 0x75},
	{0x56, 0x76},
	{0x57, 0x77},
	{0x58, 0x78},
	{0x59, 0x79},
	{0x5a, 0x7a},
	{0xc0, 0xe0},
	{0xc1, 0xe1},
	{0xc2, 0xe2},
	{0xc3, 0xe3},
	{0xc4, 0xe4},
	{0xc5, 0xe5},
	{0xc6, 0xe6},
	{0xc7, 0xe7},
	{0xc8, 0xe8},
	{0xc9, 0xe9},
	{0xca, 0xea},
	{0xcb, 0xeb},
	{0xcc, 0xec},
	{0xcd, 0xed},
	{0xce, 0xee},
	{0xcf, 0xef},
	{0xd0, 0xf0},
	{0xd1, 0xf1},
	{0xd2, 0xf2},
	{0xd3, 0xf3},
	{0xd4, 0xf4},
	{0xd5, 0xf5},
	{0xd6, 0xf6},
	{0xd8, 0xf8},
	{0xd9, 0xf9},
	{0xda, 0xfa},
	{0xdb, 0xfb},
	{0xdc, 0xfc},
	{0xdd, 0xfd},
	{0xde, 0xfe},
	{0x100, 0x101},
	{0x102, 0x103},
	{0x104, 0x105},
	{0x106, 0x107},
	{0x108, 0x109},
	{0x10a, 0x10b},
	{0x10c, 0x10d},
	{0x10e, 0x10f},
	{0x110, 0x111},
	{0x112, 0x113},
	{0x114, 0x115},
	{0x116, 0x117},
	{0x118, 0x119},
	{0x11a, 0x11b},
	{0x11c, 0x11d},
	{0x11e, 0x11f},
	{0x120, 0x121},
	{0x122, 0x123},
	{0x124, 0x125},
	{0x126, 0x127},
	{0x128, 0x129},
	{0x12a, 0x12b},
	{0x12c, 0x12d},
	{0x12e, 0x12f},
	{0x130, 0x69},
	{0x132, 0x133},
	{0x134, 0x135},
	{0x136, 0x137},
	{0x139, 0x13a},
	{0x13b, 0x13c},
	{0x13d, 0x13e},
	{0x13f, 0x140},
	{0x141, 0x142},
	{0x143, 0x144},
	{0x145, 0x146},
	{0x147, 0x148},
	{0x14a, 0x14b},
	{0x14c, 0x14d},
	{0x14e, 0x14f},
	{0x150, 0x151},
	{0x152, 0x153},
	{0x154, 0x155},
	{0x156, 0x157},
	{0x158, 0x159},
	{0x15a, 0x15b},
	{0x15c, 0x15d},
	{0x15e, 0x15f},
	{0x160, 0x161},
	{0x162, 0x163},
	{0x164, 0x165},
	{0x166, 0x167},
	{0x168, 0x169},
	{0x16a, 0x16b},
	{0x16c, 0x16d},
	{0x16e, 0x16f},
	{0x170, 0x171},
	{0x172, 0x173},
	{0x174, 0x175},
	{0x176, 0x177},
	{0x178, 0xff},
	{0x179, 0x17a},
	{0x17b, 0x17c},
	{0x17d, 0x17e},
	{0x181, 0x253},
	{0x182, 0x183},
	{0x184, 0x185},
	{0x186, 0x254},
	{0x187, 0x188},
	{0x189, 0x256},
	{0x18a, 0x257},
	{0x18b, 0x18c},
	{0x18e, 0x1dd},
	{0x18f, 0x259},
	{0x190, 0x25b},
	{0x191, 0x192},
	{0x193, 0x260},
	{0x194, 0x263},
	{0x196, 0x269},
	{0x197, 0x268},
	{0x198, 0x199},
	{0x19c, 0x26f},
	{0x19d, 0x272},
	{0x19f, 0x275},
	{0x1a0, 0x1a1},
	{0x1a2, 0x1a3},
	{0x1a4, 0x1a5},
	{0x1a6, 0x280},
	{0x1a7, 0x1a8},
	{0x1a9, 0x283},
	{0x1ac, 0x1ad},
	{0x1ae, 0x288},
	{0x1af, 0x1b0},
	{0x1b1, 0x28a},
	{0x1b2, 0x28b},
	{0x1b3, 0x1b4},
	{0x1b5, 0x1b6},
	{0x1b7, 0x292},
	{0x1b8, 0x1b9},
	{0x1bc, 0x1bd},
	{0x1c4, 0x1c6},
	{0x1c5, 0x1c6},
	{0x1c7, 0x1c9},
	{0x1c8, 0x1c9},
	{0x1ca, 0x1cc},
	{0x1cb, 0x1cc},
	{0x1cd, 0x1ce},
	{0x1cf, 0x1d0},
	{0x1d1, 0x1d2},
	{0x1d3, 0x1d4},
	{0x1d5, 0x1d6},
	{0x1d7, 0x1d8},
	{0x1d9, 0x1da},
	{0x1db, 0x1dc},
	{0x1de, 0x1df},
	{0x1e0, 0x1e1},
	{0x1e2, 0x1e3},
	{0x1e4, 0x1e5},
	{0x1e6, 0x1e7},
	{0x1e8, 0x1e9},
	{0x1ea, 0x1eb},
	{0x1ec, 0x1ed},
	{0x1ee, 0x1ef},
	{0x1f1, 0x1f3},
	{0x1f2, 0x1f3},
	{0x1f4, 0x1f5},
	{0x1f6, 0x195},
	{0x1f7, 0x1bf},
	{0x1f8, 0x1f9},
	{0x1fa, 0x1fb},
	{0x1fc, 0x1fd},
	{0x1fe, 0x1ff},
	{0x200, 0x201},
	{0x202, 0x203},
	{0x204, 0x205},
	{0x206, 0x207},
	{0x208, 0x209},
	{0x20a, 0x20b},
	{0x20c, 0x20d},
	{0x20e, 0x20f},
	{0x210, 0x211},
	{0x212, 0x213},
	{0x214, 0x215},
	{0x216, 0x217},
	{0x218, 0x219},
	{0x21a, 0x21b},
	{0x21c, 0x21d},
	{0x21e, 0x21f},
	{0x220, 0x19e},
	{0x222, 0x223},
	{0x224, 0x225},
	{0x226, 0x227},
	{0x228, 0x229},
	{0x22a, 0x22b},
	{0x22c, 0x22d},
	{0x22e, 0x22f},
	{0x230, 0x231},
	{0x232, 0x233},
	{0x386, 0x3ac},
	{0x388, 0x3ad},
	{0x389, 0x3ae},
	{0x38a, 0x3af},
	{0x38c, 0x3cc},
	{0x38e, 0x3cd},
	{0x38f, 0x3ce},
	{0x391, 0x3b1},
	{0x392, 0x3b2},
	{0x393, 0x3b3},
	{0x394, 0x3b4},
	{0x395, 0x3b5},
	{0x396, 0x3b6},
	{0x397, 0x3b7},
	{0x398, 0x3b8},
	{0x399, 0x3b9},
	{0x39a, 0x3ba},
	{0x39b, 0x3bb},
	{0x39c, 0x3bc},
	{0x39d, 0x3bd},
	{0x39e, 0x3be},
	{0x39f, 0x3bf},
	{0x3a0, 0x3c0},
	{0x3a1, 0x3c1},
	{0x3a3, 0x3c3},
	{0x3a4, 0x3c4},
	{0x3a5, 0x3c5},
	{0x3a6, 0x3c6},
	{0x3a7, 0x3c7},
	{0x3a8, 0x3c8},
	{0x3a9, 0x3c9},
	{0x3aa, 0x3ca},
	{0x3ab, 0x3cb},
	{0x3d8, 0x3d9},
	{0x3da, 0x3db},
	{0x3dc, 0x3dd},
	{0x3de, 0x3df},
	{0x3e0, 0x3e1},
	{0x3e2, 0x3e3},
	{0x3e4, 0x3e5},
	{0x3e6, 0x3e7},
	{0x3e8, 0x3e9},
	{0x3ea, 0x3eb},
	{0x3ec, 0x3ed},
	{0x3ee, 0x3ef},
	{0x3f4, 0x3b8},
	{0x3f7, 0x3f8},
	{0x3f9, 0x3f2},
	{0x3fa, 0x3fb},
	{0x400, 0x450},
	{0x401, 0x451},
	{0x402, 0x452},
	{0x403, 0x453},
	{0x404, 0x454},
	{0x405, 0x455},
	{0x406, 0x456},
	{0x407, 0x457},
	{0x408, 0x458},
	{0x409, 0x459},
	{0x40a, 0x45a},
	{0x40b, 0x45b},
	{0x40c, 0x45c},
	{0x40d, 0x45d},
	{0x40e, 0x45e},
	{0x40f, 0x45f},
	{0x410, 0x430},
	{0x411, 0x431},
	{0x412, 0x432},
	{0x413, 0x433},
	{0x414, 0x434},
	{0x415, 0x435},
	{0x416, 0x436},
	{0x417, 0x437},
	{0x418, 0x438},
	{0x419, 0x439},
	{0x41a, 0x43a},
	{0x41b, 0x43b},
	{0x41c, 0x43c},
	{0x41d, 0x43d},
	{0x41e, 0x43e},
	{0x41f, 0x43f},
	{0x420, 0x440},
	{0x421, 0x441},
	{0x422, 0x442},
	{0x423, 0x443},
	{0x424, 0x444},
	{0x425, 0x445},
	{0x426, 0x446},
	{0x427, 0x447},
	{0x428, 0x448},
	{0x429, 0x449},
	{0x42a, 0x44a},
	{0x42b, 0x44b},
	{0x42c, 0x44c},
	{0x42d, 0x44d},
	{0x42e, 0x44e},
	{0x42f, 0x44f},
	{0x460, 0x461},
	{0x462, 0x463},
	{0x464, 0x465},
	{0x466, 0x467},
	{0x468, 0x469},
	{0x46a, 0x46b},
	{0x46c, 0x46d},
	{0x46e, 0x46f},
	{0x470, 0x471},
	{0x472, 0x473},
	{0x474, 0x475},
	{0x476, 0x477},
	{0x478, 0x479},
	{0x47a, 0x47b},
	{0x47c, 0x47d},
	{0x47e, 0x47f},
	{0x480, 0x481},
	{0x48a, 0x48b},
	{0x48c, 0x48d},
	{0x48e, 0x48f},
	{0x490, 0x491},
	{0x492, 0x493},
	{0x494, 0x495},
	{0x496, 0x497},
	{0x498, 0x499},
	{0x49a, 0x49b},
	{0x49c, 0x49d},
	{0x49e, 0x49f},
	{0x4a0, 0x4a1},
	{0x4a2, 0x4a3},
	{0x4a4, 0x4a5},
	{0x4a6, 0x4a7},
	{0x4a8, 0x4a9},
	{0x4aa, 0x4ab},
	{0x4ac, 0x4ad},
	{0x4ae, 0x4af},
	{0x4b0, 0x4b1},
	{0x4b2, 0x4b3},
	{0x4b4, 0x4b5},
	{0x4b6, 0x4b7},
	{0x4b8, 0x4b9},
	{0x4ba, 0x4bb},
	{0x4bc, 0x4bd},
	{0x4be, 0x4bf},
	{0x4c1, 0x4c2},
	{0x4c3, 0x4c4},
	{0x4c5, 0x4c6},
	{0x4c7, 0x4c8},
	{0x4c9, 0x4ca},
	{0x4cb, 0x4cc},
	{0x4cd, 0x4ce},
	{0x4d0, 0x4d1},
	{0x4d2, 0x4d3},
	{0x4d4, 0x4d5},
	{0x4d6, 0x4d7},
	{0x4d8, 0x4d9},
	{0x4da, 0x4db},
	{0x4dc, 0x4dd},
	{0x4de, 0x4df},
	{0x4e0, 0x4e1},
	{0x4e2, 0x4e3},
	{0x4e4, 0x4e5},
	{0x4e6, 0x4e7},
	{0x4e8, 0x4e9},
	{0x4ea, 0x4eb},
	{0x4ec, 0x4ed},
	{0x4ee, 0x4ef},
	{0x4f0, 0x4f1},
	{0x4f2, 0x4f3},
	{0x4f4, 0x4f5},
	{0x4f8, 0x4f9},
	{0x500, 0x501},
	{0x502, 0x503},
	{0x504, 0x505},
	{0x506, 0x507},
	{0x508, 0x509},
	{0x50a, 0x50b},
	{0x50c, 0x50d},
	{0x50e, 0x50f},
	{0x531, 0x561},
	{0x532, 0x562},
	{0x533, 0x563},
	{0x534, 0x564},
	{0x535, 0x565},
	{0x536, 0x566},
	{0x537, 0x567},
	{0x538, 0x568},
	{0x539, 0x569},
	{0x53a, 0x56a},
	{0x53b, 0x56b},
	{0x53c, 0x56c},
	{0x53d, 0x56d},
	{0x53e, 0x56e},
	{0x53f, 0x56f},
	{0x540, 0x570},
	{0x541, 0x571},
	{0x542, 0x572},
	{0x543, 0x573},
	{0x544, 0x574},
	{0x545, 0x575},
	{0x546, 0x576},
	{0x547, 0x577},
	{0x548, 0x578},
	{0x549, 0x579},
	{0x54a, 0x57a},
	{0x54b, 0x57b},
	{0x54c, 0x57c},
	{0x54d, 0x57d},
	{0x54e, 0x57e},
	{0x54f, 0x57f},
	{0x550, 0x580},
	{0x551, 0x581},
	{0x552, 0x582},
	{0x553, 0x583},
	{0x554, 0x584},
	{0x555, 0x585},
	{0x556, 0x586},
	{0x1e00, 0x1e01},
	{0x1e02, 0x1e03},
	{0x1e04, 0x1e05},
	{0x1e06, 0x1e07},
	{0x1e08, 0x1e09},
	{0x1e0a, 0x1e0b},
	{0x1e0c, 0x1e0d},
	{0x1e0e, 0x1e0f},
	{0x1e10, 0x1e11},
	{0x1e12, 0x1e13},
	{0x1e14, 0x1e15},
	{0x1e16, 0x1e17},
	{0x1e18, 0x1e19},
	{0x1e1a, 0x1e1b},
	{0x1e1c, 0x1e1d},
	{0x1e1e, 0x1e1f},
	{0x1e20, 0x1e21},
	{0x1e22, 0x1e23},
	{0x1e24, 0x1e25},
	{0x1e26, 0x1e27},
	{0x1e28, 0x1e29},
	{0x1e2a, 0x1e2b},
	{0x1e2c, 0x1e2d},
	{0x1e2e, 0x1e2f},
	{0x1e30, 0x1e31},
	{0x1e32, 0x1e33},
	{0x1e34, 0x1e35},
	{0x1e36, 0x1e37},
	{0x1e38, 0x1e39},
	{0x1e3a, 0x1e3b},
	{0x1e3c, 0x1e3d},
	{0x1e3e, 0x1e3f},
	{0x1e40, 0x1e41},
	{0x1e42, 0x1e43},
	{0x1e44, 0x1e45},
	{0x1e46, 0x1e47},
	{0x1e48, 0x1e49},
	{0x1e4a, 0x1e4b},
	{0x1e4c, 0x1e4d},
	{0x1e4e, 0x1e4f},
	{0x1e50, 0x1e51},
	{0x1e52, 0x1e53},
	{0x1e54, 0x1e55},
	{0x1e56, 0x1e57},
	{0x1e58, 0x1e59},
	{0x1e5a, 0x1e5b},
	{0x1e5c, 0x1e5d},
	{0x1e5e, 0x1e5f},
	{0x1e60, 0x1e61},
	{0x1e62, 0x1e63},
	{0x1e64, 0x1e65},
	{0x1e66, 0x1e67},
	{0x1e68, 0x1e69},
	{0x1e6a, 0x1e6b},
	{0x1e6c, 0x1e6d},
	{0x1e6e, 0x1e6f},
	{0x1e70, 0x1e71},
	{0x1e72, 0x1e73},
	{0x1e74, 0x1e75},
	{0x1e76, 0x1e77},
	{0x1e78, 0x1e79},
	{0x1e7a, 0x1e7b},
	{0x1e7c, 0x1e7d},
	{0x1e7e, 0x1e7f},
	{0x1e80, 0x1e81},
	{0x1e82, 0x1e83},
	{0x1e84, 0x1e85},
	{0x1e86, 0x1e87},
	{0x1e88, 0x1e89},
	{0x1e8a, 0x1e8b},
	{0x1e8c, 0x1e8d},
	{0x1e8e, 0x1e8f},
	{0x1e90, 0x1e91},
	{0x1e92, 0x1e93},
	{0x1e94, 0x1e95},
	{0x1ea0, 0x1ea1},
	{0x1ea2, 0x1ea3},
	{0x1ea4, 0x1ea5},
	{0x1ea6, 0x1ea7},
	{0x1ea8, 0x1ea9},
	{0x1eaa, 0x1eab},
	{0x1eac, 0x1ead},
	{0x1eae, 0x1eaf},
	{0x1eb0, 0x1eb1},
	{0x1eb2, 0x1eb3},
	{0x1eb4, 0x1eb5},
	{0x1eb6, 0x1eb7},
	{0x1eb8, 0x1eb9},
	{0x1eba, 0x1ebb},
	{0x1ebc, 0x1ebd},
	{0x1ebe, 0x1ebf},
	{0x1ec0, 0x1ec1},
	{0x1ec2, 0x1ec3},
	{0x1ec4, 0x1ec5},
	{0x1ec6, 0x1ec7},
	{0x1ec8, 0x1ec9},
	{0x1eca, 0x1ecb},
	{0x1ecc, 0x1ecd},
	{0x1ece, 0x1ecf},
	{0x1ed0, 0x1ed1},
	{0x1ed2, 0x1ed3},
	{0x1ed4, 0x1ed5},
	{0x1ed6, 0x1ed7},
	{0x1ed8, 0x1ed9},
	{0x1eda, 0x1edb},
	{0x1edc, 0x1edd},
	{0x1ede, 0x1edf},
	{0x1ee0, 0x1ee1},
	{0x1ee2, 0x1ee3},
	{0x1ee4, 0x1ee5},
	{0x1ee6, 0x1ee7},
	{0x1ee8, 0x1ee9},
	{0x1eea, 0x1eeb},
	{0x1eec, 0x1eed},
	{0x1eee, 0x1eef},
	{0x1ef0, 0x1ef1},
	{0x1ef2, 0x1ef3},
	{0x1ef4, 0x1ef5},
	{0x1ef6, 0x1ef7},
	{0x1ef8, 0x1ef9},
	{0x1f08, 0x1f00},
	{0x1f09, 0x1f01},
	{0x1f0a, 0x1f02},
	{0x1f0b, 0x1f03},
	{0x1f0c, 0x1f04},
	{0x1f0d, 0x1f05},
	{0x1f0e, 0x1f06},
	{0x1f0f, 0x1f07},
	{0x1f18, 0x1f10},
	{0x1f19, 0x1f11},
	{0x1f1a, 0x1f12},
	{0x1f1b, 0x1f13},
	{0x1f1c, 0x1f14},
	{0x1f1d, 0x1f15},
	{0x1f28, 0x1f20},
	{0x1f29, 0x1f21},
	{0x1f2a, 0x1f22},
	{0x1f2b, 0x1f23},
	{0x1f2c, 0x1f24},
	{0x1f2d, 0x1f25},
	{0x1f2e, 0x1f26},
	{0x1f2f, 0x1f27},
	{0x1f38, 0x1f30},
	{0x1f39, 0x1f31},
	{0x1f3a, 0x1f32},
	{0x1f3b, 0x1f33},
	{0x1f3c, 0x1f34},
	{0x1f3d, 0x1f35},
	{0x1f3e, 0x1f36},
	{0x1f3f, 0x1f37},
	{0x1f48, 0x1f40},
	{0x1f49, 0x1f41},
	{0x1f4a, 0x1f42},
	{0x1f4b, 0x1f43},
	{0x1f4c, 0x1f44},
	{0x1f4d, 0x1f45},
	{0x1f59, 0x1f51},
	{0x1f5b, 0x1f53},
	{0x1f5d, 0x1f55},
	{0x1f5f, 0x1f57},
	{0x1f68, 0x1f60},
	{0x1f69, 0x1f61},
	{0x1f6a, 0x1f62},
	{0x1f6b, 0x1f63},
	{0x1f6c, 0x1f64},
	{0x1f6d, 0x1f65},
	{0x1f6e, 0x1f66},
	{0x1f6f, 0x1f67},
	{0x1f88, 0x1f80},
	{0x1f89, 0x1f81},
	{0x1f8a, 0x1f82},
	{0x1f8b, 0x1f83},
	{0x1f8c, 0x1f84},
	{0x1f8d, 0x1f85},
	{0x1f8e, 0x1f86},
	{0x1f8f, 0x1f87},
	{0x1f98, 0x1f90},
	{0x1f99, 0x1f91},
	{0x1f9a, 0x1f92},
	{0x1f9b, 0x1f93},
	{0x1f9c, 0x1f94},
	{0x1f9d, 0x1f95},
	{0x1f9e, 0x1f96},
	{0x1f9f, 0x1f97},
	{0x1fa8, 0x1fa0},
	{0x1fa9, 0x1fa1},
	{0x1faa, 0x1fa2},
	{0x1fab, 0x1fa3},
	{0x1fac, 0x1fa4},
	{0x1fad, 0x1fa5},
	{0x1fae, 0x1fa6},
	{0x1faf, 0x1fa7},
	{0x1fb8, 0x1fb0},
	{0x1fb9, 0x1fb1},
	{0x1fba, 0x1f70},
	{0x1fbb, 0x1f71},
	{0x1fbc, 0x1fb3},
	{0x1fc8, 0x1f72},
	{0x1fc9, 0x1f73},
	{0x1fca, 0x1f74},
	{0x1fcb, 0x1f75},
	{0x1fcc, 0x1fc3},
	{0x1fd8, 0x1fd0},
	{0x1fd9, 0x1fd1},
	{0x1fda, 0x1f76},
	{0x1fdb, 0x1f77},
	{0x1fe8, 0x1fe0},
	{0x1fe9, 0x1fe1},
	{0x1fea, 0x1f7a},
	{0x1feb, 0x1f7b},
	{0x1fec, 0x1fe5},
	{0x1ff8, 0x1f78},
	{0x1ff9, 0x1f79},
	{0x1ffa, 0x1f7c},
	{0x1ffb, 0x1f7d},
	{0x1ffc, 0x1ff3},
	{0x2126, 0x3c9},
	{0x212a, 0x6b},
	{0x212b, 0xe5},
	{0x2160, 0x2170},
	{0x2161, 0x2171},
	{0x2162, 0x2172},
	{0x2163, 0x2173},
	{0x2164, 0x2174},
	{0x2165, 0x2175},
	{0x2166, 0x2176},
	{0x2167, 0x2177},
	{0x2168, 0x2178},
	{0x2169, 0x2179},
	{0x216a, 0x217a},
	{0x216b, 0x217b},
	{0x216c, 0x217c},
	{0x216d, 0x217d},
	{0x216e, 0x217e},
	{0x216f, 0x217f},
	{0x24b6, 0x24d0},
	{0x24b7, 0x24d1},
	{0x24b8, 0x24d2},
	{0x24b9, 0x24d3},
	{0x24ba, 0x24d4},
	{0x24bb, 0x24d5},
	{0x24bc, 0x24d6},
	{0x24bd, 0x24d7},
	{0x24be, 0x24d8},
	{0x24bf, 0x24d9},
	{0x24c0, 0x24da},
	{0x24c1, 0x24db},
	{0x24c2, 0x24dc},
	{0x24c3, 0x24dd},
	{0x24c4, 0x24de},
	{0x24c5, 0x24df},
	{0x24c6, 0x24e0},
	{0x24c7, 0x24e1},
	{0x24c8, 0x24e2},
	{0x24c9, 0x24e3},
	{0x24ca, 0x24e4},
	{0x24cb, 0x24e5},
	{0x24cc, 0x24e6},
	{0x24cd, 0x24e7},
	{0x24ce, 0x24e8},
	{0x24cf, 0x24e9},
	{0xff21, 0xff41},
	{0xff22, 0xff42},
	{0xff23, 0xff43},
	{0xff24, 0xff44},
	{0xff25, 0xff45},
	{0xff26, 0xff46},
	{0xff27, 0xff47},
	{0xff28, 0xff48},
	{0xff29, 0xff49},
	{0xff2a, 0xff4a},
	{0xff2b, 0xff4b},
	{0xff2c, 0xff4c},
	{0xff2d, 0xff4d},
	{0xff2e, 0xff4e},
	{0xff2f, 0xff4f},
	{0xff30, 0xff50},
	{0xff31, 0xff51},
	{0xff32, 0xff52},
	{0xff33, 0xff53},
	{0xff34, 0xff54},
	{0xff35, 0xff55},
	{0xff36, 0xff56},
	{0xff37, 0xff57},
	{0xff38, 0xff58},
	{0xff39, 0xff59},
	{0xff3a, 0xff5a},
	{0x10400, 0x10428},
	{0x10401, 0x10429},
	{0x10402, 0x1042a},
	{0x10403, 0x1042b},
	{0x10404, 0x1042c},
	{0x10405, 0x1042d},
	{0x10406, 0x1042e},
	{0x10407, 0x1042f},
	{0x10408, 0x10430},
	{0x10409, 0x10431},
	{0x1040a, 0x10432},
	{0x1040b, 0x10433},
	{0x1040c, 0x10434},
	{0x1040d, 0x10435},
	{0x1040e, 0x10436},
	{0x1040f, 0x10437},
	{0x10410, 0x10438},
	{0x10411, 0x10439},
	{0x10412, 0x1043a},
	{0x10413, 0x1043b},
	{0x10414, 0x1043c},
	{0x10415, 0x1043d},
	{0x10416, 0x1043e},
	{0x10417, 0x1043f},
	{0x10418, 0x10440},
	{0x10419, 0x10441},
	{0x1041a, 0x10442},
	{0x1041b, 0x10443},
	{0x1041c, 0x10444},
	{0x1041d, 0x10445},
	{0x1041e, 0x10446},
	{0x1041f, 0x10447},
	{0x10420, 0x10448},
	{0x10421, 0x10449},
	{0x10422, 0x1044a},
	{0x10423, 0x1044b},
	{0x10424, 0x1044c},
	{0x10425, 0x1044d},
	{0x10426, 0x1044e},
	{0x10427, 0x1044f},
};

const CaseMapping upperMappings[] =
{
	{0x61, 0x41},
	{0x62, 0x42},
	{0x63, 0x43},
	{0x64, 0x44},
	{0x65, 0x45},
	{0x66, 0x46},
	{0x67, 0x47},
	{0x68, 0x48},
	{0x69, 0x49},
	{0x6a, 0x4a},
	{0x6b, 0x4b},
	{0x6c, 0x4c},
	{0x6d, 0x4d},
	{0x6e, 0x4e},
	{0x6f, 0x4f},
	{0x70, 0x50},
	{0x71, 0x51},
	{0x72, 0x52},
	{0x73, 0x53},
	{0x74, 0x54},
	{0x75, 0x55},
	{0x76, 0x56},
	{0x77, 0x57},
	{0x78, 0x58},
	{0x79, 0x59},
	{0x7a, 0x5a},
	{0xb5, 0x39c},
	{0xe0, 0xc0},
	{0xe1, 0xc1},
	{0xe2, 0xc2},
	{0xe3, 0xc3},
	{0xe4, 0xc4},
	{0xe5, 0xc5},
	{0xe6, 0xc6},
	{0xe7, 0xc7},
	{0xe8, 0xc8},
	{0xe9, 0xc9},
	{0xea, 0xca},
	{0xeb, 0xcb},
	{0xec, 0xcc},
	{0xed, 0xcd},
	{0xee, 0xce},
	{0xef, 0xcf},
	{0xf0, 0xd0},
	{0xf1, 0xd1},
	{0xf2, 0xd2},
	{0xf3, 0xd3},
	{0xf4, 0xd4},
	{0xf5, 0xd5},
	{0xf6, 0xd6},
	{0xf8, 0xd8},
	{0xf9, 0xd9},
	{0xfa, 0xda},
	{0xfb, 0xdb},
	{0xfc, 0xdc},
	{0xfd, 0xdd},
	{0xfe, 0xde},
	{0xff, 0x178},
	{0x101, 0x100},
	{0x103, 0x102},
	{0x105, 0x104},
	{0x107, 0x106},
	{0x109, 0x108},
	{0x10b, 0x10a},
	{0x10d, 0x10c},
	{0x10f, 0x10e},
	{0x111, 0x110},
	{0x113, 0x112},
	{0x115, 0x114},
	{0x117, 0x116},
	{0x119, 0x118},
	{0x11b, 0x11a},
	{0x11d, 0x11c},
	{0x11f, 0x11e},
	{0x121, 0x120},
	{0x123, 0x122},
	{0x125, 0x124},
	{0x127, 0x126},
	{0x129, 0x128},
	{0x12b, 0x12a},
	{0x12d, 0x12c},
	{0x12f, 0x12e},
	{0x131, 0x49},
	{0x133, 0x132},
	{0x135, 0x134},
	{0x137, 0x136},
	{0x13a, 0x139},
	{0x13c, 0x13b},
	{0x13e, 0x13d},
	{0x140, 0x13f},
	{0x142, 0x141},
	{0x144, 0x143},
	{0x146, 0x145},
	{0x148, 0x147},
	{0x14b, 0x14a},
	{0x14d, 0x14c},
	{0x14f, 0x14e},
	{0x151, 0x150},
	{0x153, 0x152},
	{0x155, 0x154},
	{0x157, 0x156},
	{0x159, 0x158},
	{0x15b, 0x15a},
	{0x15d, 0x15c},
	{0x15f, 0x15e},
	{0x161, 0x160},
	{0x163, 0x162},
	{0x165, 0x164},
	{0x167, 0x166},
	{0x169, 0x168},
	{0x16b, 0x16a},
	{0x16d, 0x16c},
	{0x16f, 0x16e},
	{0x171, 0x170},
	{0x173, 0x172},
	{0x175, 0x174},
	{0x177, 0x176},
	{0x17a, 0x179},
	{0x17c, 0x17b},
	{0x17e, 0x17d},
	{0x17f, 0x53},
	{0x183, 0x182},
	{0x185, 0x184},
	{0x188, 0x187},
	{0x18c, 0x18b},
	{0x192, 0x191},
	{0x195, 0x1f6},
	{0x199, 0x198},
	{0x19e, 0x220},
	{0x1a1, 0x1a0},
	{0x1a3, 0x1a2},
	{0x1a5, 0x1a4},
	{0x1a8, 0x1a7},
	{0x1ad, 0x1ac},
	{0x1b0, 0x1af},
	{0x1b4, 0x1b3},
	{0x1b6, 0x1b5},
	{0x1b9, 0x1b8},
	{0x1bd, 0x1bc},
	{0x1bf, 0x1f7},
	{0x1c5, 0x1c4},
	{0x1c6, 0x1c4},
	{0x1c8, 0x1c7},
	{0x1c9, 0x1c7},
	{0x1cb, 0x1ca},
	{0x1cc, 0x1ca},
	{0x1ce, 0x1cd},
	{0x1d0, 0x1cf},
	{0x1d2, 0x1d1},
	{0x1d4, 0x1d3},
	{0x1d6, 0x1d5},
	{0x1d8, 0x1d7},
	{0x1da, 0x1d9},
	{0x1dc, 0x1db},
	{0x1dd, 0x18e},
	{0x1df, 0x1de},
	{0x1e1, 0x1e0},
	{0x1e3, 0x1e2},
	{0x1e5, 0x1e4},
	{0x1e7, 0x1e6},
	{0x1e9, 0x1e8},
	{0x1eb, 0x1ea},
	{0x1ed, 0x1ec},
	{0x1ef, 0x1ee},
	{0x1f2, 0x1f1},
	{0x1f3, 0x1f1},
	{0x1f5, 0x1f4},
	{0x1f9, 0x1f8},
	{0x1fb, 0x1fa},
	{0x1fd, 0x1fc},
	{0x1ff, 0x1fe},
	{0x201, 0x200},
	{0x203, 0x202},
	{0x205, 0x204},
	{0x207, 0x206},
	{0x209, 0x208},
	{0x20b, 0x20a},
	{0x20d, 0x20c},
	{0x20f, 0x20e},
	{0x211, 0x210},
	{0x213, 0x212},
	{0x215, 0x214},
	{0x217, 0x216},
	{0x219, 0x218},
	{0x21b, 0x21a},
	{0x21d, 0x21c},
	{0x21f, 0x21e},
	{0x223, 0x222},
	{0x225, 0x224},
	{0x227, 0x226},
	{0x229, 0x228},
	{0x22b, 0x22a},
	{0x22d, 0x22c},
	{0x22f, 0x22e},
	{0x231, 0x230},
	{0x233, 0x232},
	{0x253, 0x181},
	{0x254, 0x186},
	{0x256, 0x189},
	{0x257, 0x18a},
	{0x259, 0x18f},
	{0x25b, 0x190},
	{0x260, 0x193},
	{0x263, 0x194},
	{0x268, 0x197},
	{0x269, 0x196},
	{0x26f, 0x19c},
	{0x272, 0x19d},
	{0x275, 0x19f},
	{0x280, 0x1a6},
	{0x283, 0x1a9},
	{0x288, 0x1ae},
	{0x28a, 0x1b1},
	{0x28b, 0x1b2},
	{0x292, 0x1b7},
	{0x345, 0x399},
	{0x3ac, 0x386},
	{0x3ad, 0x388},
	{0x3ae, 0x389},
	{0x3af, 0x38a},
	{0x3b1, 0x391},
	{0x3b2, 0x392},
	{0x3b3, 0x393},
	{0x3b4, 0x394},
	{0x3b5, 0x395},
	{0x3b6, 0x396},
	{0x3b7, 0x397},
	{0x3b8, 0x398},
	{0x3b9, 0x399},
	{0x3ba, 0x39a},
	{0x3bb, 0x39b},
	{0x3bc, 0x39c},
	{0x3bd, 0x39d},
	{0x3be, 0x39e},
	{0x3bf, 0x39f},
	{0x3c0, 0x3a0},
	{0x3c1, 0x3a1},
	{0x3c2, 0x3a3},
	{0x3c3, 0x3a3},
	{0x3c4, 0x3a4},
	{0x3c5, 0x3a5},
	{0x3c6, 0x3a6},
	{0x3c7, 0x3a7},
	{0x3c8, 0x3a8},
	{0x3c9, 0x3a9},
	{0x3ca, 0x3aa},
	{0x3cb, 0x3ab},
	{0x3cc, 0x38c},
	{0x3cd, 0x38e},
	{0x3ce, 0x38f},
	{0x3d0, 0x392},
	{0x3d1, 0x398},
	{0x3d5, 0x3a6},
	{0x3d6, 0x3a0},
	{0x3d9, 0x3d8},
	{0x3db, 0x3da},
	{0x3dd, 0x3dc},
	{0x3df, 0x3de},
	{0x3e1, 0x3e0},
	{0x3e3, 0x3e2},
	{0x3e5, 0x3e4},
	{0x3e7, 0x3e6},
	{0x3e9, 0x3e8},
	{0x3eb, 0x3ea},
	{0x3ed, 0x3ec},
	{0x3ef, 0x3ee},
	{0x3f0, 0x39a},
	{0x3f1, 0x3a1},
	{0x3f2, 0x3f9},
	{0x3f5, 0x395},
	{0x3f8, 0x3f7},
	{0x3fb, 0x3fa},
	{0x430, 0x410},
	{0x431, 0x411},
	{0x432, 0x412},
	{0x433, 0x413},
	{0x434, 0x414},
	{0x435, 0x415},
	{0x436, 0x416},
	{0x437, 0x417},
	{0x438, 0x418},
	{0x439, 0x419},
	{0x43a, 0x41a},
	{0x43b, 0x41b},
	{0x43c, 0x41c},
	{0x43d, 0x41d},
	{0x43e, 0x41e},
	{0x43f, 0x41f},
	{0x440, 0x420},
	{0x441, 0x421},
	{0x442, 0x422},
	{0x443, 0x423},
	{0x444, 0x424},
	{0x445, 0x425},
	{0x446, 0x426},
	{0x447, 0x427},
	{0x448, 0x428},
	{0x449, 0x429},
	{0x44a, 0x42a},
	{0x44b, 0x42b},
	{0x44c, 0x42c},
	{0x44d, 0x42d},
	{0x44e, 0x42e},
	{0x44f, 0x42f},
	{0x450, 0x400},
	{0x451, 0x401},
	{0x452, 0x402},
	{0x453, 0x403},
	{0x454, 0x404},
	{0x455, 0x405},
	{0x456, 0x406},
	{0x457, 0x407},
	{0x458, 0x408},
	{0x459, 0x409},
	{0x45a, 0x40a},
	{0x45b, 0x40b},
	{0x45c, 0x40c},
	{0x45d, 0x40d},
	{0x45e, 0x40e},
	{0x45f, 0x40f},
	{0x461, 0x460},
	{0x463, 0x462},
	{0x465, 0x464},
	{0x467, 0x466},
	{0x469, 0x468},
	{0x46b, 0x46a},
	{0x46d, 0x46c},
	{0x46f, 0x46e},
	{0x471, 0x470},
	{0x473, 0x472},
	{0x475, 0x474},
	{0x477, 0x476},
	{0x479, 0x478},
	{0x47b, 0x47a},
	{0x47d, 0x47c},
	{0x47f, 0x47e},
	{0x481, 0x480},
	{0x48b, 0x48a},
	{0x48d, 0x48c},
	{0x48f, 0x48e},
	{0x491, 0x490},
	{0x493, 0x492},
	{0x495, 0x494},
	{0x497, 0x496},
	{0x499, 0x498},
	{0x49b, 0x49a},
	{0x49d, 0x49c},
	{0x49f, 0x49e},
	{0x4a1, 0x4a0},
	{0x4a3, 0x4a2},
	{0x4a5, 0x4a4},
	{0x4a7, 0x4a6},
	{0x4a9, 0x4a8},
	{0x4ab, 0x4aa},
	{0x4ad, 0x4ac},
	{0x4af, 0x4ae},
	{0x4b1, 0x4b0},
	{0x4b3, 0x4b2},
	{0x4b5, 0x4b4},
	{0x4b7, 0x4b6},
	{0x4b9, 0x4b8},
	{0x4bb, 0x4ba},
	{0x4bd, 0x4bc},
	{0x4bf, 0x4be},
	{0x4c2, 0x4c1},
	{0x4c4, 0x4c3},
	{0x4c6, 0x4c5},
	{0x4c8, 0x4c7},
	{0x4ca, 0x4c9},
	{0x4cc, 0x4cb},
	{0x4ce, 0x4cd},
	{0x4d1, 0x4d0},
	{0x4d3, 0x4d2},
	{0x4d5, 0x4d4},
	{0x4d7, 0x4d6},
	{0x4d9, 0x4d8},
	{0x4db, 0x4da},
	{0x4dd, 0x4dc},
	{0x4df, 0x4de},
	{0x4e1, 0x4e0},
	{0x4e3, 0x4e2},
	{0x4e5, 0x4e4},
	{0x4e7, 0x4e6},
	{0x4e9, 0x4e8},
	{0x4eb, 0x4ea},
	{0x4ed, 0x4ec},
	{0x4ef, 0x4ee},
	{0x4f1, 0x4f0},
	{0x4f3, 0x4f2},
	{0x4f5, 0x4f4},
	{0x4f9, 0x4f8},
	{0x501, 0x500},
	{0x503, 0x502},
	{0x505, 0x504},
	{0x507, 0x506},
	{0x509, 0x508},
	{0x50b, 0x50a},
	{0x50d, 0x50c},
	{0x50f, 0x50e},
	{0x561, 0x531},
	{0x562, 0x532},
	{0x563, 0x533},
	{0x564, 0x534},
	{0x565, 0x535},
	{0x566, 0x536},
	{0x567, 0x537},
	{0x568, 0x538},
	{0x569, 0x539},
	{0x56a, 0x53a},
	{0x56b, 0x53b},
	{0x56c, 0x53c},
	{0x56d, 0x53d},
	{0x56e, 0x53e},
	{0x56f, 0x53f},
	{0x570, 0x540},
	{0x571, 0x541},
	{0x572, 0x542},
	{0x573, 0x543},
	{0x574, 0x544},
	{0x575, 0x545},
	{0x576, 0x546},
	{0x577, 0x547},
	{0x578, 0x548},
	{0x579, 0x549},
	{0x57a, 0x54a},
	{0x57b, 0x54b},
	{0x57c, 0x54c},
	{0x57d, 0x54d},
	{0x57e, 0x54e},
	{0x57f, 0x54f},
	{0x580, 0x550},
	{0x581, 0x551},
	{0x582, 0x552},
	{0x583, 0x553},
	{0x584, 0x554},
	{0x585, 0x555},
	{0x586, 0x556},
	{0x1e01, 0x1e00},
	{0x1e03, 0x1e02},
	{0x1e05, 0x1e04},
	{0x1e07, 0x1e06},
	{0x1e09, 0x1e08},
	{0x1e0b, 0x1e0a},
	{0x1e0d, 0x1e0c},
	{0x1e0f, 0x1e0e},
	{0x1e11, 0x1e10},
	{0x1e13, 0x1e12},
	{0x1e15, 0x1e14},
	{0x1e17, 0x1e16},
	{0x1e19, 0x1e18},
	{0x1e1b, 0x1e1a},
	{0x1e1d, 0x1e1c},
	{0x1e1f, 0x1e1e},
	{0x1e21, 0x1e20},
	{0x1e23, 0x1e22},
	{0x1e25, 0x1e24},
	{0x1e27, 0x1e26},
	{0x1e29, 0x1e28},
	{0x1e2b, 0x1e2a},
	{0x1e2d, 0x1e2c},
	{0x1e2f, 0x1e2e},
	{0x1e31, 0x1e30},
	{0x1e33, 0x1e32},
	{0x1e35, 0x1e34},
	{0x1e37, 0x1e36},
	{0x1e39, 0x1e38},
	{0x1e3b, 0x1e3a},
	{0x1e3d, 0x1e3c},
	{0x1e3f, 0x1e3e},
	{0x1e41, 0x1e40},
	{0x1e43, 0x1e42},
	{0x1e45, 0x1e44},
	{0x1e47, 0x1e46},
	{0x1e49, 0x1e48},
	{0x1e4b, 0x1e4a},
	{0x1e4d, 0x1e4c},
	{0x1e4f, 0x1e4e},
	{0x1e51, 0x1e50},
	{0x1e53, 0x1e52},
	{0x1e55, 0x1e54},
	{0x1e57, 0x1e56},
	{0x1e59, 0x1e58},
	{0x1e5b, 0x1e5a},
	{0x1e5d, 0x1e5c},
	{0x1e5f, 0x1e5e},
	{0x1e61, 0x1e60},
	{0x1e63, 0x1e62},
	{0x1e65, 0x1e64},
	{0x1e67, 0x1e66},
	{0x1e69, 0x1e68},
	{0x1e6b, 0x1e6a},
	{0x1e6d, 0x1e6c},
	{0x1e6f, 0x1e6e},
	{0x1e71, 0x1e70},
	{0x1e73, 0x1e72},
	{0x1e75, 0x1e74},
	{0x1e77, 0x1e76},
	{0x1e79, 0x1e78},
	{0x1e7b, 0x1e7a},
	{0x1e7d, 0x1e7c},
	{0x1e7f, 0x1e7e},
	{0x1e81, 0x1e80},
	{0x1e83, 0x1e82},
	{0x1e85, 0x1e84},
	{0x1e87, 0x1e86},
	{0x1e89, 0x1e88},
	{0x1e8b, 0x1e8a},
	{0x1e8d, 0x1e8c},
	{0x1e8f, 0x1e8e},
	{0x1e91, 0x1e90},
	{0x1e93, 0x1e92},
	{0x1e95, 0x1e94},
	{0x1e9b, 0x1e60},
	{0x1ea1, 0x1ea0},
	{0x1ea3, 0x1ea2},
	{0x1ea5, 0x1ea4},
	{0x1ea7, 0x1ea6},
	{0x1ea9, 0x1ea8},
	{0x1eab, 0x1eaa},
	{0x1ead, 0x1eac},
	{0x1eaf, 0x1eae},
	{0x1eb1, 0x1eb0},
	{0x1eb3, 0x1eb2},
	{0x1eb5, 0x1eb4},
	{0x1eb7, 0x1eb6},
	{0x1eb9, 0x1eb8},
	{0x1ebb, 0x1eba},
	{0x1ebd, 0x1ebc},
	{0x1ebf, 0x1ebe},
	{0x1ec1, 0x1ec0},
	{0x1ec3, 0x1ec2},
	{0x1ec5, 0x1ec4},
	{0x1ec7, 0x1ec6},
	{0x1ec9, 0x1ec8},
	{0x1ecb, 0x1eca},
	{0x1ecd, 0x1ecc},
	{0x1ecf, 0x1ece},
	{0x1ed1, 0x1ed0},
	{0x1ed3, 0x1ed2},
	{0x1ed5, 0x1ed4},
	{0x1ed7, 0x1ed6},
	{0x1ed9, 0x1ed8},
	{0x1edb, 0x1eda},
	{0x1edd, 0x1edc},
	{0x1edf, 0x1ede},
	{0x1ee1, 0x1ee0},
	{0x1ee3, 0x1ee2},
	{0x1ee5, 0x1ee4},
	{0x1ee7, 0x1ee6},
	{0x1ee9, 0x1ee8},
	{0x1eeb, 0x1eea},
	{0x1eed, 0x1eec},
	{0x1eef, 0x1eee},
	{0x1ef1, 0x1ef0},
	{0x1ef3, 0x1ef2},
	{0x1ef5, 0x1ef4},
	{0x1ef7, 0x1ef6},
	{0x1ef9, 0x1ef8},
	{0x1f00, 0x1f08},
	{0x1f01, 0x1f09},
	{0x1f02, 0x1f0a},
	{0x1f03, 0x1f0b},
	{0x1f04, 0x1f0c},
	{0x1f05, 0x1f0d},
	{0x1f06, 0x1f0e},
	{0x1f07, 0x1f0f},
	{0x1f10, 0x1f18},
	{0x1f11, 0x1f19},
	{0x1f12, 0x1f1a},
	{0x1f13, 0x1f1b},
	{0x1f14, 0x1f1c},
	{0x1f15, 0x1f1d},
	{0x1f20, 0x1f28},
	{0x1f21, 0x1f29},
	{0x1f22, 0x1f2a},
	{0x1f23, 0x1f2b},
	{0x1f24, 0x1f2c},
	{0x1f25, 0x1f2d},
	{0x1f26, 0x1f2e},
	{0x1f27, 0x1f2f},
	{0x1f30, 0x1f38},
	{0x1f31, 0x1f39},
	{0x1f32, 0x1f3a},
	{0x1f33, 0x1f3b},
	{0x1f34, 0x1f3c},
	{0x1f35, 0x1f3d},
	{0x1f36, 0x1f3e},
	{0x1f37, 0x1f3f},
	{0x1f40, 0x1f48},
	{0x1f41, 0x1f49},
	{0x1f42, 0x1f4a},
	{0x1f43, 0x1f4b},
	{0x1f44, 0x1f4c},
	{0x1f45, 0x1f4d},
	{0x1f51, 0x1f59},
	{0x1f53, 0x1f5b},
	{0x1f55, 0x1f5d},
	{0x1f57, 0x1f5f},
	{0x1f60, 0x1f68},
	{0x1f61, 0x1f69},
	{0x1f62, 0x1f6a},
	{0x1f63, 0x1f6b},
	{0x1f64, 0x1f6c},
	{0x1f65, 0x1f6d},
	{0x1f66, 0x1f6e},
	{0x1f67, 0x1f6f},
	{0x1f70, 0x1fba},
	{0x1f71, 0x1fbb},
	{0x1f72, 0x1fc8},
	{0x1f73, 0x1fc9},
	{0x1f74, 0x1fca},
	{0x1f75, 0x1fcb},
	{0x1f76, 0x1fda},
	{0x1f77, 0x1fdb},
	{0x1f78, 0x1ff8},
	{0x1f79, 0x1ff9},
	{0x1f7a, 0x1fea},
	{0x1f7b, 0x1feb},
	{0x1f7c, 0x1ffa},
	{0x1f7d, 0x1ffb},
	{0x1f80, 0x1f88},
	{0x1f81, 0x1f89},
	{0x1f82, 0x1f8a},
	{0x1f83, 0x1f8b},
	{0x1f84, 0x1f8c},
	{0x1f85, 0x1f8d},
	{0x1f86, 0x1f8e},
	{0x1f87, 0x1f8f},
	{0x1f90, 0x1f98},
	{0x1f91, 0x1f99},
	{0x1f92, 0x1f9a},
	{0x1f93, 0x1f9b},
	{0x1f94, 0x1f9c},
	{0x1f95, 0x1f9d},
	{0x1f96, 0x1f9e},
	{0x1f97, 0x1f9f},
	{0x1fa0, 0x1fa8},
	{0x1fa1, 0x1fa9},
	{0x1fa2, 0x1faa},
	{0x1fa3, 0x1fab},
	{0x1fa4, 0x1fac},
	{0x1fa5, 0x1fad},
	{0x1fa6, 0x1fae},
	{0x1fa7, 0x1faf},
	{0x1fb0, 0x1fb8},
	{0x1fb1, 0x1fb9},
	{0x1fb3, 0x1fbc},
	{0x1fbe, 0x399},
	{0x1fc3, 0x1fcc},
	{0x1fd0, 0x1fd8},
	{0x1fd1, 0x1fd9},
	{0x1fe0, 0x1fe8},
	{0x1fe1, 0x1fe9},
	{0x1fe5, 0x1fec},
	{0x1ff3, 0x1ffc},
	{0x2170, 0x2160},
	{0x2171, 0x2161},
	{0x2172, 0x2162},
	{0x2173, 0x2163},
	{0x2174, 0x2164},
	{0x2175, 0x2165},
	{0x2176, 0x2166},
	{0x2177, 0x2167},
	{0x2178, 0x2168},
	{0x2179, 0x2169},
	{0x217a, 0x216a},
	{0x217b, 0x216b},
	{0x217c, 0x216c},
	{0x217d, 0x216d},
	{0x217e, 0x216e},
	{0x217f, 0x216f},
	{0x24d0, 0x24b6},
	{0x24d1, 0x24b7},
	{0x24d2, 0x24b8},
	{0x24d3, 0x24b9},
	{0x24d4, 0x24ba},
	{0x24d5, 0x24bb},
	{0x24d6, 0x24bc},
	{0x24d7, 0x24bd},
	{0x24d8, 0x24be},
	{0x24d9, 0x24bf},
	{0x24da, 0x24c0},
	{0x24db, 0x24c1},
	{0x24dc, 0x24c2},
	{0x24dd, 0x24c3},
	{0x24de, 0x24c4},
	{0x24df, 0x24c5},
	{0x24e0, 0x24c6},
	{0x24e1, 0x24c7},
	{0x24e2, 0x24c8},
	{0x24e3, 0x24c9},
	{0x24e4, 0x24ca},
	{0x24e5, 0x24cb},
	{0x24e6, 0x24cc},
	{0x24e7, 0x24cd},
	{0x24e8, 0x24ce},
	{0x24e9, 0x24cf},
	{0xff41, 0xff21},
	{0xff42, 0xff22},
	{0xff43, 0xff23},
	{0xff44, 0xff24},
	{0xff45, 0xff25},
	{0xff46, 0xff26},
	{0xff47, 0xff27},
	{0xff48, 0xff28},
	{0xff49, 0xff29},
	{0xff4a, 0xff2a},
	{0xff4b, 0xff2b},
	{0xff4c, 0xff2c},
	{0xff4d, 0xff2d},
	{0xff4e, 0xff2e},
	{0xff4f, 0xff2f},
	{0xff50, 0xff30},
	{0xff51, 0xff31},
	{0xff52, 0xff32},
	{0xff53, 0xff33},
	{0xff54, 0xff34},
	{0xff55, 0xff35},
	{0xff56, 0xff36},
	{0xff57, 0xff37},
	{0xff58, 0xff38},
	{0xff59, 0xff39},
	{0xff5a, 0xff3a},
	{0x10428, 0x10400},
	{0x10429, 0x10401},
	{0x1042a, 0x10402},
	{0x1042b, 0x10403},
	{0x1042c, 0x10404},
	{0x1042d, 0x10405},
	{0x1042e, 0x10406},
	{0x1042f, 0x10407},
	{0x10430, 0x10408},
	{0x10431, 0x10409},
	{0x10432, 0x1040a},
	{0x10433, 0x1040b},
	{0x10434, 0x1040c},
	{0x10435, 0x1040d},
	{0x10436, 0x1040e},
	{0x10437, 0x1040f},
	{0x10438, 0x10410},
	{0x10439, 0x10411},
	{0x1043a, 0x10412},
	{0x1043b, 0x10413},
	{0x1043c, 0x10414},
	{0x1043d, 0x10415},
	{0x1043e, 0x10416},
	{0x1043f, 0x10417},
	{0x10440, 0x10418},
	{0x10441, 0x10419},
	{0x10442, 0x1041a},
	{0x10443, 0x1041b},
	{0x10444, 0x1041c},
	{0x10445, 0x1041d},
	{0x10446, 0x1041e},
	{0x10447, 0x1041f},
	{0x10448, 0x10420},
	{0x10449, 0x10421},
	{0x1044a, 0x10422},
	{0x1044b, 0x10423},
	{0x1044c, 0x10424},
	{0x1044d, 0x10425},
	{0x1044e, 0x10426},
	{0x1044f, 0x10427},
};

const CaseMapping* const lowerMappingsEnd = lowerMappings +
	(sizeof(lowerMappings)/sizeof(lowerMappings[0]));

const CaseMapping* const upperMappingsEnd = upperMappings +
	(sizeof(upperMappings)/sizeof(upperMappings[0]));

struct MappingOrdering
{
	bool operator()(const CaseMapping& x, const CaseMapping& y)
	{
		return x.codePoint < y.codePoint;
	}
};

struct Transformer
{
	Transformer(const CaseMapping* const begin, const CaseMapping* const end)
		: m_begin(begin)
		, m_end(end)
	{
	}

	UInt32 operator()(UInt32 in) const
	{
		CaseMapping val = { in, 0 };
		const CaseMapping* i = std::lower_bound(m_begin, m_end, val, MappingOrdering());
		if (i == m_end || i->codePoint != in)
		{
			return in;
		}
		else
		{
			return i->mapping;
		}
	}
	const CaseMapping* const m_begin;
	const CaseMapping* const m_end;
};

} // end unnamed namespace

/////////////////////////////////////////////////////////////////////////////
bool toUpperCaseInPlace(char* input)
{
	return transformInPlace(input, Transformer(upperMappings, upperMappingsEnd));
}

/////////////////////////////////////////////////////////////////////////////
String toUpperCase(const char* input)
{
	return transform(input, Transformer(upperMappings, upperMappingsEnd));
}

/////////////////////////////////////////////////////////////////////////////
bool toLowerCaseInPlace(char* input)
{
	return transformInPlace(input, Transformer(lowerMappings, lowerMappingsEnd));
}

/////////////////////////////////////////////////////////////////////////////
String toLowerCase(const char* input)
{
	return transform(input, Transformer(lowerMappings, lowerMappingsEnd));
}

} // end namespace OW_UTF8Utils

} // end namespace OW_NAMESPACE


