/*******************************************************************************
* Copyright (C) 2001-2004 Vintela, Inc. All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of source code must retain the above copyright notice,
*    this list of conditions and the following disclaimer.
*
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this list of conditions and the following disclaimer in the documentation
*    and/or other materials provided with the distribution.
*
*  - Neither the name of Vintela, Inc. nor the names of its
*    contributors may be used to endorse or promote products derived from this
*    software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL Vintela, Inc. OR THE CONTRIBUTORS
* BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/**
 * @author Dan Nuffer
 */


#include "TestSuite.hpp"
#include "TestCaller.hpp"
#include "OW_UTF8UtilsTestCases.hpp"
#include "OW_UTF8Utils.hpp"
#include "OW_Char16.hpp"
#include "OW_String.hpp"

using namespace OpenWBEM;

void OW_UTF8UtilsTestCases::setUp()
{
}

void OW_UTF8UtilsTestCases::tearDown()
{
}

void OW_UTF8UtilsTestCases::testCharCount()
{
	unitAssert(UTF8Utils::charCount("") == 0);
	unitAssert(UTF8Utils::charCount("a") == 1);
	unitAssert(UTF8Utils::charCount("ab") == 2);
	// some examples taken from icu many.txt
	// I used KWrite and selected UTF8 encoding to edit this stuff.
	unitAssert(UTF8Utils::charCount("外国語の勉強と教え") == 9);
	unitAssert(UTF8Utils::charCount("Изучение и обучение иностранных языков") == 38);
	unitAssert(UTF8Utils::charCount("語文教學・语文教学") == 9);
	unitAssert(UTF8Utils::charCount("Enseñanza y estudio de idiomas") == 30);
	unitAssert(UTF8Utils::charCount("Изучаване и Преподаване на Чужди Езипи") == 38);
	unitAssert(UTF8Utils::charCount("ქართული ენის შესწავლა და სწავლება") == 33);
	unitAssert(UTF8Utils::charCount("'læŋɡwidʒ 'lɘr:niŋ ænd 'ti:ʃiŋ") == 30);
	unitAssert(UTF8Utils::charCount("Lus kawm thaib qhia") == 19);
	unitAssert(UTF8Utils::charCount("Ngôn Ngữ, Sự học,") == 17);
	unitAssert(UTF8Utils::charCount("‭‫ללמוד וללמד את השֵפה") == 22);
	unitAssert(UTF8Utils::charCount("L'enseignement et l'étude des langues") == 37);
	unitAssert(UTF8Utils::charCount("㜊㞕㧍㒟㦮 㐀㛲㭘㒟") == 10);
	unitAssert(UTF8Utils::charCount("Nauka języków obcych") == 20);
	unitAssert(UTF8Utils::charCount("Γλωσσική Εκμὰθηση και Διδασκαλία") == 32);
	unitAssert(UTF8Utils::charCount("‭‫ﺗﺪﺭﯾﺲ ﻭ ﯾﺎﺩﮔﯿﺮﯼ ﺯﺑﺎﻥ") == 22);
	unitAssert(UTF8Utils::charCount("Sprachlernen und -lehren") == 24);
	unitAssert(UTF8Utils::charCount("‭‫ﺗﻌﻠ‫ُّ‪‫ﻢ ﻭﺗﺪﺭﻳﺲ ﺍﻟﻌﺮﺑﻴﺔ") == 26);
	unitAssert(UTF8Utils::charCount("เรียนและสอนภาษา") == 15);

	// some from UTF-8-test.txt
	unitAssert(UTF8Utils::charCount("1ࠀ𐀀") == 4);
	unitAssert(UTF8Utils::charCount("߿￿") == 4);

}

// you can get these numbers by using a hex editor and doing
// conversions with iconv which comes with glibc. Use utf8 and ucs4 or ucs2
// for the encodings.
void OW_UTF8UtilsTestCases::testUTF8toUCS2()
{
	unitAssert(UTF8Utils::UTF8toUCS2("a") == 'a');
	unitAssert(UTF8Utils::UTF8toUCS2("") == 0x80);
	unitAssert(UTF8Utils::UTF8toUCS2("ࠀ") == 0x800);
	unitAssert(UTF8Utils::UTF8toUCS2("￿") == 0xFFFF);
	unitAssert(UTF8Utils::UTF8toUCS2("\xe3\x82\xa6") == 0x30a6);
}

void OW_UTF8UtilsTestCases::testUCS2toUTF8()
{
	unitAssert(UTF8Utils::UCS2toUTF8('a') == "a");
	unitAssert(UTF8Utils::UCS2toUTF8(0x80) == "");
	unitAssert(UTF8Utils::UCS2toUTF8(0x800) == "ࠀ");
	unitAssert(UTF8Utils::UCS2toUTF8(0xFFFF) == "￿");
	unitAssert(UTF8Utils::UCS2toUTF8(0x30a6) == "\xe3\x82\xa6");
}

void OW_UTF8UtilsTestCases::testUTF8toUCS4()
{
	unitAssert(UTF8Utils::UTF8toUCS4("a") == 'a');
	unitAssert(UTF8Utils::UTF8toUCS4("") == 0x80);
	unitAssert(UTF8Utils::UTF8toUCS4("ࠀ") == 0x800);
	unitAssert(UTF8Utils::UTF8toUCS4("￿") == 0xFFFF);
	unitAssert(UTF8Utils::UTF8toUCS4("\xe3\x82\xa6") == 0x30a6);
	// this is the current largest unicode char 
	// (according to UnicodeData.txt from Unicode 4.0)
	unitAssert(UTF8Utils::UTF8toUCS4("\xf0\xaf\xa8\x9d") == 0x02fa1d);
}

void OW_UTF8UtilsTestCases::testUCS4toUTF8()
{
	unitAssert(UTF8Utils::UCS4toUTF8('a') == "a");
	unitAssert(UTF8Utils::UCS4toUTF8(0x80) == "");
	unitAssert(UTF8Utils::UCS4toUTF8(0x800) == "ࠀ");
	unitAssert(UTF8Utils::UCS4toUTF8(0xFFFF) == "￿");
	unitAssert(UTF8Utils::UCS4toUTF8(0x30a6) == "\xe3\x82\xa6");
	// this is the current largest unicode char 
	// (according to UnicodeData.txt from Unicode 4.0)
	unitAssert(UTF8Utils::UCS4toUTF8(0x02fa1d) == "\xf0\xaf\xa8\x9d");
}

void OW_UTF8UtilsTestCases::testcompareToIgnoreCase()
{
	unitAssert(UTF8Utils::compareToIgnoreCase("foo", "foo") == 0);
	unitAssert(UTF8Utils::compareToIgnoreCase("foo", "Foo") == 0);
	unitAssert(UTF8Utils::compareToIgnoreCase("foo", "FOO") == 0);
	unitAssert(UTF8Utils::compareToIgnoreCase("foo", "fooX") < 0);
	unitAssert(UTF8Utils::compareToIgnoreCase("fooX", "foo") > 0);
	unitAssert(UTF8Utils::compareToIgnoreCase("I", "i") == 0);
	// Turkish capital I with a dot on top.
	unitAssert(UTF8Utils::compareToIgnoreCase("I", "\xc4\xb1") == 0);


	unitAssert(UTF8Utils::compareToIgnoreCase("A", "B") < 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("B", "A") > 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("AB", "a") > 0)

// This takes too long to compile.  If compareToIgnoreCase ever changes, come activate this sequence of tests.
#if 0
	// generated by OW_GenCaseFoldingCompareTests.cpp
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1", "\x1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x2", "\x2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3", "\x3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4", "\x4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5", "\x5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6", "\x6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7", "\x7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8", "\x8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x9", "\x9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa", "\xa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb", "\xb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc", "\xc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd", "\xd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe", "\xe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf", "\xf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x10", "\x10") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x11", "\x11") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x12", "\x12") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x13", "\x13") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x14", "\x14") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x15", "\x15") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x16", "\x16") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x17", "\x17") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x18", "\x18") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x19", "\x19") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1a", "\x1a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1b", "\x1b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1c", "\x1c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1d", "\x1d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1e", "\x1e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1f", "\x1f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x20", "\x20") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x21", "\x21") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x22", "\x22") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x23", "\x23") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x24", "\x24") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x25", "\x25") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x26", "\x26") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x27", "\x27") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x28", "\x28") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x29", "\x29") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x2a", "\x2a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x2b", "\x2b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x2c", "\x2c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x2d", "\x2d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x2e", "\x2e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x2f", "\x2f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x30", "\x30") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x31", "\x31") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x32", "\x32") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x33", "\x33") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x34", "\x34") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x35", "\x35") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x36", "\x36") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x37", "\x37") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x38", "\x38") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x39", "\x39") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3a", "\x3a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3b", "\x3b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3c", "\x3c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3d", "\x3d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3e", "\x3e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3f", "\x3f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x40", "\x40") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x41", "\x41") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x42", "\x42") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x43", "\x43") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x44", "\x44") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x45", "\x45") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x46", "\x46") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x47", "\x47") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x48", "\x48") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x49", "\x49") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4a", "\x4a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4b", "\x4b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4c", "\x4c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4d", "\x4d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4e", "\x4e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4f", "\x4f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x50", "\x50") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x51", "\x51") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x52", "\x52") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x53", "\x53") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x54", "\x54") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x55", "\x55") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x56", "\x56") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x57", "\x57") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x58", "\x58") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x59", "\x59") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5a", "\x5a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5b", "\x5b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5c", "\x5c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5d", "\x5d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5e", "\x5e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5f", "\x5f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x60", "\x60") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x61", "\x61") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x62", "\x62") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x63", "\x63") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x64", "\x64") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x65", "\x65") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66", "\x66") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x67", "\x67") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x68", "\x68") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x69", "\x69") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6a", "\x6a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6b", "\x6b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6c", "\x6c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6d", "\x6d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6e", "\x6e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6f", "\x6f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x70", "\x70") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x71", "\x71") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x72", "\x72") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x73", "\x73") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x74", "\x74") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x75", "\x75") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x76", "\x76") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x77", "\x77") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x78", "\x78") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x79", "\x79") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7a", "\x7a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7b", "\x7b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7c", "\x7c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7d", "\x7d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7e", "\x7e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7f", "\x7f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x80", "\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x81", "\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x82", "\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x83", "\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x84", "\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x85", "\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x86", "\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x87", "\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x88", "\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x89", "\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8a", "\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8b", "\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8c", "\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8d", "\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8e", "\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8f", "\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x90", "\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x91", "\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x92", "\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x93", "\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x94", "\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x95", "\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x96", "\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x97", "\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x98", "\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x99", "\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x9a", "\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x9b", "\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x9c", "\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x9d", "\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x9e", "\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x9f", "\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa0", "\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa1", "\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa2", "\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa3", "\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa4", "\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa5", "\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa6", "\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa7", "\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa8", "\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa9", "\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xaa", "\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xab", "\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xac", "\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xad", "\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xae", "\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xaf", "\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb0", "\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb1", "\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb2", "\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb3", "\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb4", "\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb5", "\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb6", "\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb7", "\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb8", "\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb9", "\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xba", "\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xbb", "\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xbc", "\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xbd", "\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xbe", "\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xbf", "\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc0", "\xc0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc1", "\xc1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc2", "\xc2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3", "\xc3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4", "\xc4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5", "\xc5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6", "\xc6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7", "\xc7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8", "\xc8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9", "\xc9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca", "\xca") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcb", "\xcb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcc", "\xcc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcd", "\xcd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce", "\xce") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf", "\xcf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0", "\xd0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1", "\xd1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2", "\xd2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3", "\xd3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4", "\xd4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5", "\xd5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6", "\xd6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd7", "\xd7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd8", "\xd8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd9", "\xd9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xda", "\xda") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xdb", "\xdb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xdc", "\xdc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xdd", "\xdd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xde", "\xde") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xdf", "\xdf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe0", "\xe0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1", "\xe1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2", "\xe2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe3", "\xe3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe4", "\xe4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe5", "\xe5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe6", "\xe6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe7", "\xe7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe8", "\xe8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe9", "\xe9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xea", "\xea") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xeb", "\xeb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xec", "\xec") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xed", "\xed") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xee", "\xee") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef", "\xef") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0", "\xf0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf1", "\xf1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf2", "\xf2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf3", "\xf3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf4", "\xf4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf5", "\xf5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf6", "\xf6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf7", "\xf7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf8", "\xf8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf9", "\xf9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xfa", "\xfa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xfb", "\xfb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xfc", "\xfc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xfd", "\xfd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xfe", "\xfe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xff", "\xff") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x41", "\x61") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x61", "\x41") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x42", "\x62") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x62", "\x42") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x43", "\x63") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x63", "\x43") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x44", "\x64") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x64", "\x44") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x45", "\x65") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x65", "\x45") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x46", "\x66") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66", "\x46") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x47", "\x67") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x67", "\x47") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x48", "\x68") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x68", "\x48") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x49", "\x69") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x69", "\x49") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x49", "\xc4\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb1", "\x49") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4a", "\x6a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6a", "\x4a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4b", "\x6b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6b", "\x4b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4c", "\x6c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6c", "\x4c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4d", "\x6d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6d", "\x4d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4e", "\x6e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6e", "\x4e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x4f", "\x6f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6f", "\x4f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x50", "\x70") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x70", "\x50") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x51", "\x71") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x71", "\x51") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x52", "\x72") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x72", "\x52") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x53", "\x73") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x73", "\x53") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x54", "\x74") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x74", "\x54") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x55", "\x75") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x75", "\x55") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x56", "\x76") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x76", "\x56") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x57", "\x77") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x77", "\x57") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x58", "\x78") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x78", "\x58") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x59", "\x79") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x79", "\x59") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5a", "\x7a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x7a", "\x5a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc2\xb5", "\xce\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbc", "\xc2\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x80", "\xc3\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa0", "\xc3\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x81", "\xc3\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa1", "\xc3\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x82", "\xc3\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa2", "\xc3\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x83", "\xc3\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa3", "\xc3\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x84", "\xc3\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa4", "\xc3\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x85", "\xc3\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa5", "\xc3\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x86", "\xc3\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa6", "\xc3\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x87", "\xc3\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa7", "\xc3\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x88", "\xc3\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa8", "\xc3\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x89", "\xc3\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa9", "\xc3\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x8a", "\xc3\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xaa", "\xc3\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x8b", "\xc3\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xab", "\xc3\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x8c", "\xc3\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xac", "\xc3\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x8d", "\xc3\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xad", "\xc3\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x8e", "\xc3\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xae", "\xc3\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x8f", "\xc3\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xaf", "\xc3\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x90", "\xc3\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb0", "\xc3\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x91", "\xc3\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb1", "\xc3\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x92", "\xc3\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb2", "\xc3\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x93", "\xc3\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb3", "\xc3\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x94", "\xc3\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb4", "\xc3\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x95", "\xc3\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb5", "\xc3\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x96", "\xc3\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb6", "\xc3\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x98", "\xc3\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb8", "\xc3\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x99", "\xc3\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb9", "\xc3\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x9a", "\xc3\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xba", "\xc3\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x9b", "\xc3\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xbb", "\xc3\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x9c", "\xc3\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xbc", "\xc3\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x9d", "\xc3\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xbd", "\xc3\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x9e", "\xc3\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xbe", "\xc3\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\x9f", "\x73\x73") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x73\x73", "\xc3\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x80", "\xc4\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x81", "\xc4\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x82", "\xc4\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x83", "\xc4\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x84", "\xc4\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x85", "\xc4\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x86", "\xc4\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x87", "\xc4\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x88", "\xc4\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x89", "\xc4\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x8a", "\xc4\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x8b", "\xc4\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x8c", "\xc4\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x8d", "\xc4\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x8e", "\xc4\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x8f", "\xc4\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x90", "\xc4\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x91", "\xc4\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x92", "\xc4\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x93", "\xc4\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x94", "\xc4\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x95", "\xc4\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x96", "\xc4\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x97", "\xc4\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x98", "\xc4\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x99", "\xc4\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x9a", "\xc4\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x9b", "\xc4\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x9c", "\xc4\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x9d", "\xc4\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x9e", "\xc4\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x9f", "\xc4\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa0", "\xc4\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa1", "\xc4\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa2", "\xc4\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa3", "\xc4\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa4", "\xc4\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa5", "\xc4\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa6", "\xc4\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa7", "\xc4\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa8", "\xc4\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa9", "\xc4\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xaa", "\xc4\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xab", "\xc4\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xac", "\xc4\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xad", "\xc4\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xae", "\xc4\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xaf", "\xc4\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb0", "\x69\xcc\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x69\xcc\x87", "\xc4\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb0", "\x69") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x69", "\xc4\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb2", "\xc4\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb3", "\xc4\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb4", "\xc4\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb5", "\xc4\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb6", "\xc4\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb7", "\xc4\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb9", "\xc4\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xba", "\xc4\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xbb", "\xc4\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xbc", "\xc4\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xbd", "\xc4\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xbe", "\xc4\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xbf", "\xc5\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x80", "\xc4\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x81", "\xc5\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x82", "\xc5\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x83", "\xc5\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x84", "\xc5\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x85", "\xc5\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x86", "\xc5\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x87", "\xc5\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x88", "\xc5\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x89", "\xca\xbc\x6e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\xbc\x6e", "\xc5\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x8a", "\xc5\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x8b", "\xc5\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x8c", "\xc5\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x8d", "\xc5\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x8e", "\xc5\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x8f", "\xc5\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x90", "\xc5\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x91", "\xc5\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x92", "\xc5\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x93", "\xc5\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x94", "\xc5\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x95", "\xc5\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x96", "\xc5\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x97", "\xc5\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x98", "\xc5\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x99", "\xc5\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x9a", "\xc5\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x9b", "\xc5\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x9c", "\xc5\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x9d", "\xc5\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x9e", "\xc5\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x9f", "\xc5\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa0", "\xc5\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa1", "\xc5\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa2", "\xc5\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa3", "\xc5\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa4", "\xc5\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa5", "\xc5\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa6", "\xc5\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa7", "\xc5\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa8", "\xc5\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa9", "\xc5\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xaa", "\xc5\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xab", "\xc5\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xac", "\xc5\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xad", "\xc5\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xae", "\xc5\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xaf", "\xc5\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb0", "\xc5\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb1", "\xc5\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb2", "\xc5\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb3", "\xc5\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb4", "\xc5\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb5", "\xc5\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb6", "\xc5\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb7", "\xc5\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb8", "\xc3\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xbf", "\xc5\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb9", "\xc5\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xba", "\xc5\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xbb", "\xc5\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xbc", "\xc5\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xbd", "\xc5\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xbe", "\xc5\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xbf", "\x73") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x73", "\xc5\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x81", "\xc9\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\x93", "\xc6\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x82", "\xc6\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x83", "\xc6\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x84", "\xc6\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x85", "\xc6\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x86", "\xc9\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\x94", "\xc6\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x87", "\xc6\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x88", "\xc6\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x89", "\xc9\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\x96", "\xc6\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x8a", "\xc9\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\x97", "\xc6\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x8b", "\xc6\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x8c", "\xc6\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x8e", "\xc7\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x9d", "\xc6\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x8f", "\xc9\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\x99", "\xc6\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x90", "\xc9\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\x9b", "\xc6\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x91", "\xc6\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x92", "\xc6\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x93", "\xc9\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xa0", "\xc6\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x94", "\xc9\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xa3", "\xc6\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x96", "\xc9\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xa9", "\xc6\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x97", "\xc9\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xa8", "\xc6\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x98", "\xc6\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x99", "\xc6\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x9c", "\xc9\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xaf", "\xc6\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x9d", "\xc9\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xb2", "\xc6\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x9f", "\xc9\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xb5", "\xc6\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa0", "\xc6\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa1", "\xc6\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa2", "\xc6\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa3", "\xc6\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa4", "\xc6\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa5", "\xc6\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa6", "\xca\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\x80", "\xc6\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa7", "\xc6\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa8", "\xc6\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa9", "\xca\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\x83", "\xc6\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xac", "\xc6\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xad", "\xc6\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xae", "\xca\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\x88", "\xc6\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xaf", "\xc6\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb0", "\xc6\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb1", "\xca\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\x8a", "\xc6\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb2", "\xca\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\x8b", "\xc6\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb3", "\xc6\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb4", "\xc6\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb5", "\xc6\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb6", "\xc6\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb7", "\xca\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\x92", "\xc6\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb8", "\xc6\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb9", "\xc6\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xbc", "\xc6\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xbd", "\xc6\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x84", "\xc7\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x86", "\xc7\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x85", "\xc7\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x86", "\xc7\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x87", "\xc7\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x89", "\xc7\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x88", "\xc7\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x89", "\xc7\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8a", "\xc7\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8c", "\xc7\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8b", "\xc7\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8c", "\xc7\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8d", "\xc7\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8e", "\xc7\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8f", "\xc7\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x90", "\xc7\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x91", "\xc7\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x92", "\xc7\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x93", "\xc7\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x94", "\xc7\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x95", "\xc7\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x96", "\xc7\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x97", "\xc7\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x98", "\xc7\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x99", "\xc7\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x9a", "\xc7\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x9b", "\xc7\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x9c", "\xc7\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x9e", "\xc7\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x9f", "\xc7\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa0", "\xc7\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa1", "\xc7\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa2", "\xc7\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa3", "\xc7\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa4", "\xc7\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa5", "\xc7\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa6", "\xc7\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa7", "\xc7\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa8", "\xc7\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa9", "\xc7\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xaa", "\xc7\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xab", "\xc7\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xac", "\xc7\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xad", "\xc7\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xae", "\xc7\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xaf", "\xc7\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb0", "\x6a\xcc\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6a\xcc\x8c", "\xc7\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb1", "\xc7\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb3", "\xc7\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb2", "\xc7\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb3", "\xc7\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb4", "\xc7\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb5", "\xc7\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb6", "\xc6\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x95", "\xc7\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb7", "\xc6\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xbf", "\xc7\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb8", "\xc7\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb9", "\xc7\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xba", "\xc7\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xbb", "\xc7\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xbc", "\xc7\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xbd", "\xc7\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xbe", "\xc7\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xbf", "\xc7\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x80", "\xc8\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x81", "\xc8\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x82", "\xc8\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x83", "\xc8\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x84", "\xc8\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x85", "\xc8\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x86", "\xc8\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x87", "\xc8\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x88", "\xc8\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x89", "\xc8\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x8a", "\xc8\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x8b", "\xc8\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x8c", "\xc8\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x8d", "\xc8\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x8e", "\xc8\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x8f", "\xc8\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x90", "\xc8\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x91", "\xc8\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x92", "\xc8\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x93", "\xc8\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x94", "\xc8\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x95", "\xc8\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x96", "\xc8\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x97", "\xc8\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x98", "\xc8\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x99", "\xc8\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x9a", "\xc8\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x9b", "\xc8\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x9c", "\xc8\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x9d", "\xc8\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x9e", "\xc8\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x9f", "\xc8\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa0", "\xc6\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x9e", "\xc8\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa2", "\xc8\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa3", "\xc8\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa4", "\xc8\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa5", "\xc8\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa6", "\xc8\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa7", "\xc8\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa8", "\xc8\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa9", "\xc8\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xaa", "\xc8\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xab", "\xc8\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xac", "\xc8\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xad", "\xc8\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xae", "\xc8\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xaf", "\xc8\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xb0", "\xc8\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xb1", "\xc8\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xb2", "\xc8\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xb3", "\xc8\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcd\x85", "\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9", "\xcd\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x86", "\xce\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xac", "\xce\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x88", "\xce\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xad", "\xce\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x89", "\xce\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xae", "\xce\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x8a", "\xce\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xaf", "\xce\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x8c", "\xcf\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x8c", "\xce\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x8e", "\xcf\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x8d", "\xce\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x8f", "\xcf\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x8e", "\xce\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x90", "\xce\xb9\xcc\x88\xcc\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xcc\x88\xcc\x81", "\xce\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x91", "\xce\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb1", "\xce\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x92", "\xce\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb2", "\xce\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x93", "\xce\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb3", "\xce\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x94", "\xce\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb4", "\xce\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x95", "\xce\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb5", "\xce\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x96", "\xce\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb6", "\xce\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x97", "\xce\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb7", "\xce\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x98", "\xce\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb8", "\xce\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x99", "\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9", "\xce\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x9a", "\xce\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xba", "\xce\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x9b", "\xce\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbb", "\xce\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x9c", "\xce\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbc", "\xce\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x9d", "\xce\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbd", "\xce\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x9e", "\xce\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbe", "\xce\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\x9f", "\xce\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbf", "\xce\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa0", "\xcf\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x80", "\xce\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa1", "\xcf\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x81", "\xce\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa3", "\xcf\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x83", "\xce\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa4", "\xcf\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x84", "\xce\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa5", "\xcf\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85", "\xce\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa6", "\xcf\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x86", "\xce\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa7", "\xcf\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x87", "\xce\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa8", "\xcf\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x88", "\xce\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xa9", "\xcf\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x89", "\xce\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xaa", "\xcf\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x8a", "\xce\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xab", "\xcf\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x8b", "\xce\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb0", "\xcf\x85\xcc\x88\xcc\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x88\xcc\x81", "\xce\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x82", "\xcf\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x83", "\xcf\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x90", "\xce\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb2", "\xcf\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x91", "\xce\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb8", "\xcf\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x95", "\xcf\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x86", "\xcf\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x96", "\xcf\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x80", "\xcf\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x98", "\xcf\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x99", "\xcf\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x9a", "\xcf\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x9b", "\xcf\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x9c", "\xcf\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x9d", "\xcf\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x9e", "\xcf\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x9f", "\xcf\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa0", "\xcf\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa1", "\xcf\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa2", "\xcf\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa3", "\xcf\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa4", "\xcf\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa5", "\xcf\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa6", "\xcf\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa7", "\xcf\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa8", "\xcf\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa9", "\xcf\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xaa", "\xcf\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xab", "\xcf\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xac", "\xcf\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xad", "\xcf\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xae", "\xcf\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xaf", "\xcf\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb0", "\xce\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xba", "\xcf\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb1", "\xcf\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x81", "\xcf\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb4", "\xce\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb8", "\xcf\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb5", "\xce\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb5", "\xcf\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb7", "\xcf\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb8", "\xcf\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb9", "\xcf\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb2", "\xcf\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xba", "\xcf\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xbb", "\xcf\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x80", "\xd1\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x90", "\xd0\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x81", "\xd1\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x91", "\xd0\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x82", "\xd1\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x92", "\xd0\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x83", "\xd1\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x93", "\xd0\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x84", "\xd1\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x94", "\xd0\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x85", "\xd1\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x95", "\xd0\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x86", "\xd1\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x96", "\xd0\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x87", "\xd1\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x97", "\xd0\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x88", "\xd1\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x98", "\xd0\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x89", "\xd1\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x99", "\xd0\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x8a", "\xd1\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x9a", "\xd0\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x8b", "\xd1\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x9b", "\xd0\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x8c", "\xd1\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x9c", "\xd0\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x8d", "\xd1\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x9d", "\xd0\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x8e", "\xd1\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x9e", "\xd0\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x8f", "\xd1\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x9f", "\xd0\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x90", "\xd0\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb0", "\xd0\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x91", "\xd0\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb1", "\xd0\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x92", "\xd0\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb2", "\xd0\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x93", "\xd0\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb3", "\xd0\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x94", "\xd0\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb4", "\xd0\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x95", "\xd0\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb5", "\xd0\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x96", "\xd0\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb6", "\xd0\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x97", "\xd0\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb7", "\xd0\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x98", "\xd0\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb8", "\xd0\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x99", "\xd0\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb9", "\xd0\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x9a", "\xd0\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xba", "\xd0\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x9b", "\xd0\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xbb", "\xd0\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x9c", "\xd0\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xbc", "\xd0\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x9d", "\xd0\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xbd", "\xd0\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x9e", "\xd0\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xbe", "\xd0\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\x9f", "\xd0\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xbf", "\xd0\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa0", "\xd1\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x80", "\xd0\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa1", "\xd1\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x81", "\xd0\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa2", "\xd1\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x82", "\xd0\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa3", "\xd1\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x83", "\xd0\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa4", "\xd1\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x84", "\xd0\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa5", "\xd1\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x85", "\xd0\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa6", "\xd1\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x86", "\xd0\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa7", "\xd1\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x87", "\xd0\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa8", "\xd1\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x88", "\xd0\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xa9", "\xd1\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x89", "\xd0\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xaa", "\xd1\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8a", "\xd0\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xab", "\xd1\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8b", "\xd0\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xac", "\xd1\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8c", "\xd0\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xad", "\xd1\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8d", "\xd0\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xae", "\xd1\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8e", "\xd0\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xaf", "\xd1\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8f", "\xd0\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa0", "\xd1\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa1", "\xd1\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa2", "\xd1\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa3", "\xd1\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa4", "\xd1\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa5", "\xd1\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa6", "\xd1\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa7", "\xd1\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa8", "\xd1\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa9", "\xd1\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xaa", "\xd1\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xab", "\xd1\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xac", "\xd1\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xad", "\xd1\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xae", "\xd1\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xaf", "\xd1\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb0", "\xd1\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb1", "\xd1\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb2", "\xd1\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb3", "\xd1\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb4", "\xd1\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb5", "\xd1\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb6", "\xd1\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb7", "\xd1\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb8", "\xd1\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb9", "\xd1\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xba", "\xd1\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xbb", "\xd1\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xbc", "\xd1\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xbd", "\xd1\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xbe", "\xd1\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xbf", "\xd1\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x80", "\xd2\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x81", "\xd2\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x8a", "\xd2\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x8b", "\xd2\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x8c", "\xd2\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x8d", "\xd2\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x8e", "\xd2\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x8f", "\xd2\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x90", "\xd2\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x91", "\xd2\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x92", "\xd2\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x93", "\xd2\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x94", "\xd2\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x95", "\xd2\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x96", "\xd2\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x97", "\xd2\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x98", "\xd2\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x99", "\xd2\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x9a", "\xd2\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x9b", "\xd2\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x9c", "\xd2\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x9d", "\xd2\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x9e", "\xd2\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x9f", "\xd2\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa0", "\xd2\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa1", "\xd2\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa2", "\xd2\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa3", "\xd2\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa4", "\xd2\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa5", "\xd2\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa6", "\xd2\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa7", "\xd2\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa8", "\xd2\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa9", "\xd2\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xaa", "\xd2\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xab", "\xd2\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xac", "\xd2\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xad", "\xd2\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xae", "\xd2\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xaf", "\xd2\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb0", "\xd2\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb1", "\xd2\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb2", "\xd2\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb3", "\xd2\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb4", "\xd2\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb5", "\xd2\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb6", "\xd2\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb7", "\xd2\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb8", "\xd2\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb9", "\xd2\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xba", "\xd2\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xbb", "\xd2\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xbc", "\xd2\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xbd", "\xd2\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xbe", "\xd2\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xbf", "\xd2\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x81", "\xd3\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x82", "\xd3\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x83", "\xd3\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x84", "\xd3\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x85", "\xd3\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x86", "\xd3\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x87", "\xd3\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x88", "\xd3\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x89", "\xd3\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x8a", "\xd3\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x8b", "\xd3\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x8c", "\xd3\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x8d", "\xd3\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x8e", "\xd3\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x90", "\xd3\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x91", "\xd3\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x92", "\xd3\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x93", "\xd3\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x94", "\xd3\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x95", "\xd3\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x96", "\xd3\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x97", "\xd3\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x98", "\xd3\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x99", "\xd3\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x9a", "\xd3\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x9b", "\xd3\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x9c", "\xd3\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x9d", "\xd3\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x9e", "\xd3\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x9f", "\xd3\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa0", "\xd3\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa1", "\xd3\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa2", "\xd3\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa3", "\xd3\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa4", "\xd3\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa5", "\xd3\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa6", "\xd3\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa7", "\xd3\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa8", "\xd3\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa9", "\xd3\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xaa", "\xd3\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xab", "\xd3\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xac", "\xd3\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xad", "\xd3\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xae", "\xd3\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xaf", "\xd3\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb0", "\xd3\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb1", "\xd3\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb2", "\xd3\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb3", "\xd3\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb4", "\xd3\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb5", "\xd3\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb8", "\xd3\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb9", "\xd3\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x80", "\xd4\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x81", "\xd4\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x82", "\xd4\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x83", "\xd4\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x84", "\xd4\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x85", "\xd4\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x86", "\xd4\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x87", "\xd4\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x88", "\xd4\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x89", "\xd4\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x8a", "\xd4\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x8b", "\xd4\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x8c", "\xd4\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x8d", "\xd4\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x8e", "\xd4\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x8f", "\xd4\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb1", "\xd5\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa1", "\xd4\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb2", "\xd5\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa2", "\xd4\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb3", "\xd5\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa3", "\xd4\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb4", "\xd5\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa4", "\xd4\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb5", "\xd5\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa5", "\xd4\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb6", "\xd5\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa6", "\xd4\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb7", "\xd5\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa7", "\xd4\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb8", "\xd5\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa8", "\xd4\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xb9", "\xd5\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa9", "\xd4\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xba", "\xd5\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xaa", "\xd4\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xbb", "\xd5\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xab", "\xd4\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xbc", "\xd5\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xac", "\xd4\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xbd", "\xd5\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xad", "\xd4\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xbe", "\xd5\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xae", "\xd4\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\xbf", "\xd5\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xaf", "\xd4\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x80", "\xd5\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb0", "\xd5\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x81", "\xd5\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb1", "\xd5\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x82", "\xd5\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb2", "\xd5\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x83", "\xd5\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb3", "\xd5\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x84", "\xd5\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb4", "\xd5\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x85", "\xd5\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb5", "\xd5\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x86", "\xd5\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb6", "\xd5\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x87", "\xd5\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb7", "\xd5\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x88", "\xd5\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb8", "\xd5\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x89", "\xd5\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb9", "\xd5\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x8a", "\xd5\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xba", "\xd5\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x8b", "\xd5\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xbb", "\xd5\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x8c", "\xd5\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xbc", "\xd5\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x8d", "\xd5\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xbd", "\xd5\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x8e", "\xd5\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xbe", "\xd5\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x8f", "\xd5\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xbf", "\xd5\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x90", "\xd6\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x80", "\xd5\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x91", "\xd6\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x81", "\xd5\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x92", "\xd6\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x82", "\xd5\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x93", "\xd6\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x83", "\xd5\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x94", "\xd6\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x84", "\xd5\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x95", "\xd6\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x85", "\xd5\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\x96", "\xd6\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x86", "\xd5\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x87", "\xd5\xa5\xd6\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa5\xd6\x82", "\xd6\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x80", "\xe1\xb8\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x81", "\xe1\xb8\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x82", "\xe1\xb8\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x83", "\xe1\xb8\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x84", "\xe1\xb8\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x85", "\xe1\xb8\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x86", "\xe1\xb8\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x87", "\xe1\xb8\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x88", "\xe1\xb8\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x89", "\xe1\xb8\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x8a", "\xe1\xb8\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x8b", "\xe1\xb8\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x8c", "\xe1\xb8\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x8d", "\xe1\xb8\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x8e", "\xe1\xb8\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x8f", "\xe1\xb8\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x90", "\xe1\xb8\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x91", "\xe1\xb8\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x92", "\xe1\xb8\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x93", "\xe1\xb8\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x94", "\xe1\xb8\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x95", "\xe1\xb8\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x96", "\xe1\xb8\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x97", "\xe1\xb8\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x98", "\xe1\xb8\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x99", "\xe1\xb8\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x9a", "\xe1\xb8\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x9b", "\xe1\xb8\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x9c", "\xe1\xb8\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x9d", "\xe1\xb8\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x9e", "\xe1\xb8\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x9f", "\xe1\xb8\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa0", "\xe1\xb8\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa1", "\xe1\xb8\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa2", "\xe1\xb8\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa3", "\xe1\xb8\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa4", "\xe1\xb8\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa5", "\xe1\xb8\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa6", "\xe1\xb8\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa7", "\xe1\xb8\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa8", "\xe1\xb8\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa9", "\xe1\xb8\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xaa", "\xe1\xb8\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xab", "\xe1\xb8\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xac", "\xe1\xb8\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xad", "\xe1\xb8\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xae", "\xe1\xb8\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xaf", "\xe1\xb8\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb0", "\xe1\xb8\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb1", "\xe1\xb8\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb2", "\xe1\xb8\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb3", "\xe1\xb8\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb4", "\xe1\xb8\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb5", "\xe1\xb8\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb6", "\xe1\xb8\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb7", "\xe1\xb8\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb8", "\xe1\xb8\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb9", "\xe1\xb8\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xba", "\xe1\xb8\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xbb", "\xe1\xb8\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xbc", "\xe1\xb8\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xbd", "\xe1\xb8\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xbe", "\xe1\xb8\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xbf", "\xe1\xb8\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x80", "\xe1\xb9\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x81", "\xe1\xb9\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x82", "\xe1\xb9\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x83", "\xe1\xb9\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x84", "\xe1\xb9\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x85", "\xe1\xb9\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x86", "\xe1\xb9\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x87", "\xe1\xb9\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x88", "\xe1\xb9\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x89", "\xe1\xb9\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x8a", "\xe1\xb9\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x8b", "\xe1\xb9\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x8c", "\xe1\xb9\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x8d", "\xe1\xb9\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x8e", "\xe1\xb9\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x8f", "\xe1\xb9\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x90", "\xe1\xb9\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x91", "\xe1\xb9\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x92", "\xe1\xb9\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x93", "\xe1\xb9\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x94", "\xe1\xb9\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x95", "\xe1\xb9\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x96", "\xe1\xb9\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x97", "\xe1\xb9\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x98", "\xe1\xb9\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x99", "\xe1\xb9\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x9a", "\xe1\xb9\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x9b", "\xe1\xb9\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x9c", "\xe1\xb9\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x9d", "\xe1\xb9\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x9e", "\xe1\xb9\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x9f", "\xe1\xb9\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa0", "\xe1\xb9\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa1", "\xe1\xb9\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa2", "\xe1\xb9\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa3", "\xe1\xb9\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa4", "\xe1\xb9\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa5", "\xe1\xb9\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa6", "\xe1\xb9\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa7", "\xe1\xb9\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa8", "\xe1\xb9\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa9", "\xe1\xb9\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xaa", "\xe1\xb9\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xab", "\xe1\xb9\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xac", "\xe1\xb9\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xad", "\xe1\xb9\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xae", "\xe1\xb9\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xaf", "\xe1\xb9\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb0", "\xe1\xb9\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb1", "\xe1\xb9\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb2", "\xe1\xb9\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb3", "\xe1\xb9\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb4", "\xe1\xb9\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb5", "\xe1\xb9\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb6", "\xe1\xb9\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb7", "\xe1\xb9\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb8", "\xe1\xb9\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb9", "\xe1\xb9\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xba", "\xe1\xb9\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xbb", "\xe1\xb9\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xbc", "\xe1\xb9\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xbd", "\xe1\xb9\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xbe", "\xe1\xb9\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xbf", "\xe1\xb9\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x80", "\xe1\xba\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x81", "\xe1\xba\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x82", "\xe1\xba\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x83", "\xe1\xba\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x84", "\xe1\xba\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x85", "\xe1\xba\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x86", "\xe1\xba\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x87", "\xe1\xba\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x88", "\xe1\xba\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x89", "\xe1\xba\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x8a", "\xe1\xba\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x8b", "\xe1\xba\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x8c", "\xe1\xba\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x8d", "\xe1\xba\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x8e", "\xe1\xba\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x8f", "\xe1\xba\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x90", "\xe1\xba\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x91", "\xe1\xba\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x92", "\xe1\xba\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x93", "\xe1\xba\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x94", "\xe1\xba\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x95", "\xe1\xba\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x96", "\x68\xcc\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x68\xcc\xb1", "\xe1\xba\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x97", "\x74\xcc\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x74\xcc\x88", "\xe1\xba\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x98", "\x77\xcc\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x77\xcc\x8a", "\xe1\xba\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x99", "\x79\xcc\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x79\xcc\x8a", "\xe1\xba\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x9a", "\x61\xca\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x61\xca\xbe", "\xe1\xba\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x9b", "\xe1\xb9\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa1", "\xe1\xba\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa0", "\xe1\xba\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa1", "\xe1\xba\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa2", "\xe1\xba\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa3", "\xe1\xba\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa4", "\xe1\xba\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa5", "\xe1\xba\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa6", "\xe1\xba\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa7", "\xe1\xba\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa8", "\xe1\xba\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa9", "\xe1\xba\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xaa", "\xe1\xba\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xab", "\xe1\xba\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xac", "\xe1\xba\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xad", "\xe1\xba\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xae", "\xe1\xba\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xaf", "\xe1\xba\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb0", "\xe1\xba\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb1", "\xe1\xba\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb2", "\xe1\xba\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb3", "\xe1\xba\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb4", "\xe1\xba\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb5", "\xe1\xba\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb6", "\xe1\xba\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb7", "\xe1\xba\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb8", "\xe1\xba\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb9", "\xe1\xba\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xba", "\xe1\xba\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xbb", "\xe1\xba\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xbc", "\xe1\xba\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xbd", "\xe1\xba\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xbe", "\xe1\xba\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xbf", "\xe1\xba\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x80", "\xe1\xbb\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x81", "\xe1\xbb\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x82", "\xe1\xbb\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x83", "\xe1\xbb\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x84", "\xe1\xbb\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x85", "\xe1\xbb\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x86", "\xe1\xbb\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x87", "\xe1\xbb\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x88", "\xe1\xbb\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x89", "\xe1\xbb\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x8a", "\xe1\xbb\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x8b", "\xe1\xbb\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x8c", "\xe1\xbb\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x8d", "\xe1\xbb\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x8e", "\xe1\xbb\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x8f", "\xe1\xbb\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x90", "\xe1\xbb\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x91", "\xe1\xbb\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x92", "\xe1\xbb\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x93", "\xe1\xbb\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x94", "\xe1\xbb\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x95", "\xe1\xbb\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x96", "\xe1\xbb\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x97", "\xe1\xbb\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x98", "\xe1\xbb\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x99", "\xe1\xbb\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x9a", "\xe1\xbb\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x9b", "\xe1\xbb\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x9c", "\xe1\xbb\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x9d", "\xe1\xbb\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x9e", "\xe1\xbb\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x9f", "\xe1\xbb\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa0", "\xe1\xbb\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa1", "\xe1\xbb\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa2", "\xe1\xbb\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa3", "\xe1\xbb\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa4", "\xe1\xbb\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa5", "\xe1\xbb\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa6", "\xe1\xbb\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa7", "\xe1\xbb\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa8", "\xe1\xbb\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa9", "\xe1\xbb\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xaa", "\xe1\xbb\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xab", "\xe1\xbb\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xac", "\xe1\xbb\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xad", "\xe1\xbb\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xae", "\xe1\xbb\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xaf", "\xe1\xbb\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb0", "\xe1\xbb\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb1", "\xe1\xbb\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb2", "\xe1\xbb\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb3", "\xe1\xbb\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb4", "\xe1\xbb\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb5", "\xe1\xbb\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb6", "\xe1\xbb\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb7", "\xe1\xbb\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb8", "\xe1\xbb\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb9", "\xe1\xbb\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x88", "\xe1\xbc\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x80", "\xe1\xbc\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x89", "\xe1\xbc\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x81", "\xe1\xbc\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x8a", "\xe1\xbc\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x82", "\xe1\xbc\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x8b", "\xe1\xbc\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x83", "\xe1\xbc\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x8c", "\xe1\xbc\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x84", "\xe1\xbc\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x8d", "\xe1\xbc\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x85", "\xe1\xbc\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x8e", "\xe1\xbc\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x86", "\xe1\xbc\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x8f", "\xe1\xbc\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x87", "\xe1\xbc\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x98", "\xe1\xbc\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x90", "\xe1\xbc\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x99", "\xe1\xbc\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x91", "\xe1\xbc\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x9a", "\xe1\xbc\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x92", "\xe1\xbc\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x9b", "\xe1\xbc\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x93", "\xe1\xbc\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x9c", "\xe1\xbc\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x94", "\xe1\xbc\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x9d", "\xe1\xbc\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x95", "\xe1\xbc\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa8", "\xe1\xbc\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa0", "\xe1\xbc\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa9", "\xe1\xbc\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa1", "\xe1\xbc\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xaa", "\xe1\xbc\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa2", "\xe1\xbc\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xab", "\xe1\xbc\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa3", "\xe1\xbc\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xac", "\xe1\xbc\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa4", "\xe1\xbc\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xad", "\xe1\xbc\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa5", "\xe1\xbc\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xae", "\xe1\xbc\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa6", "\xe1\xbc\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xaf", "\xe1\xbc\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa7", "\xe1\xbc\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb8", "\xe1\xbc\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb0", "\xe1\xbc\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb9", "\xe1\xbc\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb1", "\xe1\xbc\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xba", "\xe1\xbc\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb2", "\xe1\xbc\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xbb", "\xe1\xbc\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb3", "\xe1\xbc\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xbc", "\xe1\xbc\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb4", "\xe1\xbc\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xbd", "\xe1\xbc\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb5", "\xe1\xbc\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xbe", "\xe1\xbc\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb6", "\xe1\xbc\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xbf", "\xe1\xbc\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb7", "\xe1\xbc\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x88", "\xe1\xbd\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x80", "\xe1\xbd\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x89", "\xe1\xbd\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x81", "\xe1\xbd\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x8a", "\xe1\xbd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x82", "\xe1\xbd\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x8b", "\xe1\xbd\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x83", "\xe1\xbd\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x8c", "\xe1\xbd\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x84", "\xe1\xbd\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x8d", "\xe1\xbd\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x85", "\xe1\xbd\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x90", "\xcf\x85\xcc\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x93", "\xe1\xbd\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x92", "\xcf\x85\xcc\x93\xcc\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x93\xcc\x80", "\xe1\xbd\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x94", "\xcf\x85\xcc\x93\xcc\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x93\xcc\x81", "\xe1\xbd\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x96", "\xcf\x85\xcc\x93\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x93\xcd\x82", "\xe1\xbd\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x99", "\xe1\xbd\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x91", "\xe1\xbd\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x9b", "\xe1\xbd\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x93", "\xe1\xbd\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x9d", "\xe1\xbd\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x95", "\xe1\xbd\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x9f", "\xe1\xbd\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x97", "\xe1\xbd\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa8", "\xe1\xbd\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa0", "\xe1\xbd\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa9", "\xe1\xbd\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa1", "\xe1\xbd\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xaa", "\xe1\xbd\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa2", "\xe1\xbd\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xab", "\xe1\xbd\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa3", "\xe1\xbd\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xac", "\xe1\xbd\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa4", "\xe1\xbd\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xad", "\xe1\xbd\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa5", "\xe1\xbd\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xae", "\xe1\xbd\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa6", "\xe1\xbd\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xaf", "\xe1\xbd\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa7", "\xe1\xbd\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x80", "\xe1\xbc\x80\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x80\xce\xb9", "\xe1\xbe\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x81", "\xe1\xbc\x81\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x81\xce\xb9", "\xe1\xbe\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x82", "\xe1\xbc\x82\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x82\xce\xb9", "\xe1\xbe\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x83", "\xe1\xbc\x83\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x83\xce\xb9", "\xe1\xbe\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x84", "\xe1\xbc\x84\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x84\xce\xb9", "\xe1\xbe\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x85", "\xe1\xbc\x85\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x85\xce\xb9", "\xe1\xbe\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x86", "\xe1\xbc\x86\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x86\xce\xb9", "\xe1\xbe\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x87", "\xe1\xbc\x87\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x87\xce\xb9", "\xe1\xbe\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x88", "\xe1\xbc\x80\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x80\xce\xb9", "\xe1\xbe\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x88", "\xe1\xbe\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x80", "\xe1\xbe\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x89", "\xe1\xbc\x81\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x81\xce\xb9", "\xe1\xbe\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x89", "\xe1\xbe\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x81", "\xe1\xbe\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8a", "\xe1\xbc\x82\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x82\xce\xb9", "\xe1\xbe\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8a", "\xe1\xbe\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x82", "\xe1\xbe\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8b", "\xe1\xbc\x83\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x83\xce\xb9", "\xe1\xbe\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8b", "\xe1\xbe\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x83", "\xe1\xbe\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8c", "\xe1\xbc\x84\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x84\xce\xb9", "\xe1\xbe\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8c", "\xe1\xbe\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x84", "\xe1\xbe\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8d", "\xe1\xbc\x85\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x85\xce\xb9", "\xe1\xbe\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8d", "\xe1\xbe\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x85", "\xe1\xbe\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8e", "\xe1\xbc\x86\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x86\xce\xb9", "\xe1\xbe\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8e", "\xe1\xbe\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x86", "\xe1\xbe\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8f", "\xe1\xbc\x87\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x87\xce\xb9", "\xe1\xbe\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x8f", "\xe1\xbe\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x87", "\xe1\xbe\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x90", "\xe1\xbc\xa0\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa0\xce\xb9", "\xe1\xbe\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x91", "\xe1\xbc\xa1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa1\xce\xb9", "\xe1\xbe\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x92", "\xe1\xbc\xa2\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa2\xce\xb9", "\xe1\xbe\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x93", "\xe1\xbc\xa3\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa3\xce\xb9", "\xe1\xbe\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x94", "\xe1\xbc\xa4\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa4\xce\xb9", "\xe1\xbe\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x95", "\xe1\xbc\xa5\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa5\xce\xb9", "\xe1\xbe\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x96", "\xe1\xbc\xa6\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa6\xce\xb9", "\xe1\xbe\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x97", "\xe1\xbc\xa7\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa7\xce\xb9", "\xe1\xbe\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x98", "\xe1\xbc\xa0\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa0\xce\xb9", "\xe1\xbe\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x98", "\xe1\xbe\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x90", "\xe1\xbe\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x99", "\xe1\xbc\xa1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa1\xce\xb9", "\xe1\xbe\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x99", "\xe1\xbe\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x91", "\xe1\xbe\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9a", "\xe1\xbc\xa2\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa2\xce\xb9", "\xe1\xbe\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9a", "\xe1\xbe\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x92", "\xe1\xbe\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9b", "\xe1\xbc\xa3\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa3\xce\xb9", "\xe1\xbe\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9b", "\xe1\xbe\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x93", "\xe1\xbe\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9c", "\xe1\xbc\xa4\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa4\xce\xb9", "\xe1\xbe\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9c", "\xe1\xbe\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x94", "\xe1\xbe\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9d", "\xe1\xbc\xa5\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa5\xce\xb9", "\xe1\xbe\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9d", "\xe1\xbe\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x95", "\xe1\xbe\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9e", "\xe1\xbc\xa6\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa6\xce\xb9", "\xe1\xbe\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9e", "\xe1\xbe\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x96", "\xe1\xbe\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9f", "\xe1\xbc\xa7\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa7\xce\xb9", "\xe1\xbe\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x9f", "\xe1\xbe\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x97", "\xe1\xbe\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa0", "\xe1\xbd\xa0\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa0\xce\xb9", "\xe1\xbe\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa1", "\xe1\xbd\xa1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa1\xce\xb9", "\xe1\xbe\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa2", "\xe1\xbd\xa2\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa2\xce\xb9", "\xe1\xbe\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa3", "\xe1\xbd\xa3\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa3\xce\xb9", "\xe1\xbe\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa4", "\xe1\xbd\xa4\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa4\xce\xb9", "\xe1\xbe\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa5", "\xe1\xbd\xa5\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa5\xce\xb9", "\xe1\xbe\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa6", "\xe1\xbd\xa6\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa6\xce\xb9", "\xe1\xbe\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa7", "\xe1\xbd\xa7\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa7\xce\xb9", "\xe1\xbe\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa8", "\xe1\xbd\xa0\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa0\xce\xb9", "\xe1\xbe\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa8", "\xe1\xbe\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa0", "\xe1\xbe\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa9", "\xe1\xbd\xa1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa1\xce\xb9", "\xe1\xbe\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa9", "\xe1\xbe\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa1", "\xe1\xbe\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xaa", "\xe1\xbd\xa2\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa2\xce\xb9", "\xe1\xbe\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xaa", "\xe1\xbe\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa2", "\xe1\xbe\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xab", "\xe1\xbd\xa3\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa3\xce\xb9", "\xe1\xbe\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xab", "\xe1\xbe\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa3", "\xe1\xbe\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xac", "\xe1\xbd\xa4\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa4\xce\xb9", "\xe1\xbe\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xac", "\xe1\xbe\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa4", "\xe1\xbe\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xad", "\xe1\xbd\xa5\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa5\xce\xb9", "\xe1\xbe\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xad", "\xe1\xbe\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa5", "\xe1\xbe\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xae", "\xe1\xbd\xa6\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa6\xce\xb9", "\xe1\xbe\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xae", "\xe1\xbe\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa6", "\xe1\xbe\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xaf", "\xe1\xbd\xa7\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa7\xce\xb9", "\xe1\xbe\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xaf", "\xe1\xbe\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa7", "\xe1\xbe\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb2", "\xe1\xbd\xb0\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb0\xce\xb9", "\xe1\xbe\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb3", "\xce\xb1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb1\xce\xb9", "\xe1\xbe\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb4", "\xce\xac\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xac\xce\xb9", "\xe1\xbe\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb6", "\xce\xb1\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb1\xcd\x82", "\xe1\xbe\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb7", "\xce\xb1\xcd\x82\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb1\xcd\x82\xce\xb9", "\xe1\xbe\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb8", "\xe1\xbe\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb0", "\xe1\xbe\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb9", "\xe1\xbe\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb1", "\xe1\xbe\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xba", "\xe1\xbd\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb0", "\xe1\xbe\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xbb", "\xe1\xbd\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb1", "\xe1\xbe\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xbc", "\xce\xb1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb1\xce\xb9", "\xe1\xbe\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xbc", "\xe1\xbe\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb3", "\xe1\xbe\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xbe", "\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9", "\xe1\xbe\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x82", "\xe1\xbd\xb4\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb4\xce\xb9", "\xe1\xbf\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x83", "\xce\xb7\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb7\xce\xb9", "\xe1\xbf\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x84", "\xce\xae\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xae\xce\xb9", "\xe1\xbf\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x86", "\xce\xb7\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb7\xcd\x82", "\xe1\xbf\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x87", "\xce\xb7\xcd\x82\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb7\xcd\x82\xce\xb9", "\xe1\xbf\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x88", "\xe1\xbd\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb2", "\xe1\xbf\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x89", "\xe1\xbd\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb3", "\xe1\xbf\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x8a", "\xe1\xbd\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb4", "\xe1\xbf\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x8b", "\xe1\xbd\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb5", "\xe1\xbf\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x8c", "\xce\xb7\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb7\xce\xb9", "\xe1\xbf\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x8c", "\xe1\xbf\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x83", "\xe1\xbf\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x92", "\xce\xb9\xcc\x88\xcc\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xcc\x88\xcc\x80", "\xe1\xbf\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x93", "\xce\xb9\xcc\x88\xcc\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xcc\x88\xcc\x81", "\xe1\xbf\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x96", "\xce\xb9\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xcd\x82", "\xe1\xbf\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x97", "\xce\xb9\xcc\x88\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xcc\x88\xcd\x82", "\xe1\xbf\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x98", "\xe1\xbf\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x90", "\xe1\xbf\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x99", "\xe1\xbf\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x91", "\xe1\xbf\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x9a", "\xe1\xbd\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb6", "\xe1\xbf\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\x9b", "\xe1\xbd\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb7", "\xe1\xbf\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa2", "\xcf\x85\xcc\x88\xcc\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x88\xcc\x80", "\xe1\xbf\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa3", "\xcf\x85\xcc\x88\xcc\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x88\xcc\x81", "\xe1\xbf\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa4", "\xcf\x81\xcc\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x81\xcc\x93", "\xe1\xbf\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa6", "\xcf\x85\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcd\x82", "\xe1\xbf\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa7", "\xcf\x85\xcc\x88\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x88\xcd\x82", "\xe1\xbf\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa8", "\xe1\xbf\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa0", "\xe1\xbf\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa9", "\xe1\xbf\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa1", "\xe1\xbf\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xaa", "\xe1\xbd\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xba", "\xe1\xbf\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xab", "\xe1\xbd\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xbb", "\xe1\xbf\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xac", "\xe1\xbf\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa5", "\xe1\xbf\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb2", "\xe1\xbd\xbc\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xbc\xce\xb9", "\xe1\xbf\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb3", "\xcf\x89\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x89\xce\xb9", "\xe1\xbf\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb4", "\xcf\x8e\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x8e\xce\xb9", "\xe1\xbf\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb6", "\xcf\x89\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x89\xcd\x82", "\xe1\xbf\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb7", "\xcf\x89\xcd\x82\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x89\xcd\x82\xce\xb9", "\xe1\xbf\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb8", "\xe1\xbd\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb8", "\xe1\xbf\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb9", "\xe1\xbd\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb9", "\xe1\xbf\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xba", "\xe1\xbd\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xbc", "\xe1\xbf\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xbb", "\xe1\xbd\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xbd", "\xe1\xbf\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xbc", "\xcf\x89\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x89\xce\xb9", "\xe1\xbf\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xbc", "\xe1\xbf\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb3", "\xe1\xbf\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x84\xa6", "\xcf\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x89", "\xe2\x84\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x84\xaa", "\x6b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6b", "\xe2\x84\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x84\xab", "\xc3\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa5", "\xe2\x84\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa0", "\xe2\x85\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb0", "\xe2\x85\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa1", "\xe2\x85\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb1", "\xe2\x85\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa2", "\xe2\x85\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb2", "\xe2\x85\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa3", "\xe2\x85\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb3", "\xe2\x85\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa4", "\xe2\x85\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb4", "\xe2\x85\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa5", "\xe2\x85\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb5", "\xe2\x85\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa6", "\xe2\x85\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb6", "\xe2\x85\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa7", "\xe2\x85\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb7", "\xe2\x85\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa8", "\xe2\x85\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb8", "\xe2\x85\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xa9", "\xe2\x85\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb9", "\xe2\x85\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xaa", "\xe2\x85\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xba", "\xe2\x85\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xab", "\xe2\x85\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xbb", "\xe2\x85\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xac", "\xe2\x85\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xbc", "\xe2\x85\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xad", "\xe2\x85\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xbd", "\xe2\x85\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xae", "\xe2\x85\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xbe", "\xe2\x85\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xaf", "\xe2\x85\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xbf", "\xe2\x85\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xb6", "\xe2\x93\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x90", "\xe2\x92\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xb7", "\xe2\x93\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x91", "\xe2\x92\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xb8", "\xe2\x93\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x92", "\xe2\x92\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xb9", "\xe2\x93\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x93", "\xe2\x92\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xba", "\xe2\x93\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x94", "\xe2\x92\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xbb", "\xe2\x93\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x95", "\xe2\x92\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xbc", "\xe2\x93\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x96", "\xe2\x92\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xbd", "\xe2\x93\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x97", "\xe2\x92\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xbe", "\xe2\x93\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x98", "\xe2\x92\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x92\xbf", "\xe2\x93\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x99", "\xe2\x92\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x80", "\xe2\x93\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9a", "\xe2\x93\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x81", "\xe2\x93\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9b", "\xe2\x93\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x82", "\xe2\x93\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9c", "\xe2\x93\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x83", "\xe2\x93\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9d", "\xe2\x93\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x84", "\xe2\x93\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9e", "\xe2\x93\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x85", "\xe2\x93\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9f", "\xe2\x93\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x86", "\xe2\x93\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa0", "\xe2\x93\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x87", "\xe2\x93\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa1", "\xe2\x93\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x88", "\xe2\x93\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa2", "\xe2\x93\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x89", "\xe2\x93\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa3", "\xe2\x93\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x8a", "\xe2\x93\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa4", "\xe2\x93\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x8b", "\xe2\x93\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa5", "\xe2\x93\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x8c", "\xe2\x93\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa6", "\xe2\x93\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x8d", "\xe2\x93\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa7", "\xe2\x93\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x8e", "\xe2\x93\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa8", "\xe2\x93\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x8f", "\xe2\x93\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa9", "\xe2\x93\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x80", "\x66\x66") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66\x66", "\xef\xac\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x81", "\x66\x69") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66\x69", "\xef\xac\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x82", "\x66\x6c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66\x6c", "\xef\xac\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x83", "\x66\x66\x69") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66\x66\x69", "\xef\xac\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x84", "\x66\x66\x6c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66\x66\x6c", "\xef\xac\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x85", "\x73\x74") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x73\x74", "\xef\xac\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x86", "\x73\x74") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x73\x74", "\xef\xac\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x93", "\xd5\xb4\xd5\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb4\xd5\xb6", "\xef\xac\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x94", "\xd5\xb4\xd5\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb4\xd5\xa5", "\xef\xac\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x95", "\xd5\xb4\xd5\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb4\xd5\xab", "\xef\xac\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x96", "\xd5\xbe\xd5\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xbe\xd5\xb6", "\xef\xac\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xac\x97", "\xd5\xb4\xd5\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb4\xd5\xad", "\xef\xac\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa1", "\xef\xbd\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x81", "\xef\xbc\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa2", "\xef\xbd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x82", "\xef\xbc\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa3", "\xef\xbd\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x83", "\xef\xbc\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa4", "\xef\xbd\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x84", "\xef\xbc\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa5", "\xef\xbd\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x85", "\xef\xbc\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa6", "\xef\xbd\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x86", "\xef\xbc\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa7", "\xef\xbd\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x87", "\xef\xbc\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa8", "\xef\xbd\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x88", "\xef\xbc\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xa9", "\xef\xbd\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x89", "\xef\xbc\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xaa", "\xef\xbd\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x8a", "\xef\xbc\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xab", "\xef\xbd\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x8b", "\xef\xbc\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xac", "\xef\xbd\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x8c", "\xef\xbc\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xad", "\xef\xbd\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x8d", "\xef\xbc\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xae", "\xef\xbd\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x8e", "\xef\xbc\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xaf", "\xef\xbd\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x8f", "\xef\xbc\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb0", "\xef\xbd\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x90", "\xef\xbc\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb1", "\xef\xbd\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x91", "\xef\xbc\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb2", "\xef\xbd\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x92", "\xef\xbc\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb3", "\xef\xbd\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x93", "\xef\xbc\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb4", "\xef\xbd\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x94", "\xef\xbc\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb5", "\xef\xbd\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x95", "\xef\xbc\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb6", "\xef\xbd\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x96", "\xef\xbc\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb7", "\xef\xbd\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x97", "\xef\xbc\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb8", "\xef\xbd\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x98", "\xef\xbc\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xb9", "\xef\xbd\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x99", "\xef\xbc\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbc\xba", "\xef\xbd\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x9a", "\xef\xbc\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x80", "\xf0\x90\x90\xa8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa8", "\xf0\x90\x90\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x81", "\xf0\x90\x90\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa9", "\xf0\x90\x90\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x82", "\xf0\x90\x90\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xaa", "\xf0\x90\x90\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x83", "\xf0\x90\x90\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xab", "\xf0\x90\x90\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x84", "\xf0\x90\x90\xac") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xac", "\xf0\x90\x90\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x85", "\xf0\x90\x90\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xad", "\xf0\x90\x90\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x86", "\xf0\x90\x90\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xae", "\xf0\x90\x90\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x87", "\xf0\x90\x90\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xaf", "\xf0\x90\x90\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x88", "\xf0\x90\x90\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb0", "\xf0\x90\x90\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x89", "\xf0\x90\x90\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb1", "\xf0\x90\x90\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x8a", "\xf0\x90\x90\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb2", "\xf0\x90\x90\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x8b", "\xf0\x90\x90\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb3", "\xf0\x90\x90\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x8c", "\xf0\x90\x90\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb4", "\xf0\x90\x90\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x8d", "\xf0\x90\x90\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb5", "\xf0\x90\x90\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x8e", "\xf0\x90\x90\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb6", "\xf0\x90\x90\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x8f", "\xf0\x90\x90\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb7", "\xf0\x90\x90\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x90", "\xf0\x90\x90\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb8", "\xf0\x90\x90\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x91", "\xf0\x90\x90\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb9", "\xf0\x90\x90\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x92", "\xf0\x90\x90\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xba", "\xf0\x90\x90\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x93", "\xf0\x90\x90\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xbb", "\xf0\x90\x90\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x94", "\xf0\x90\x90\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xbc", "\xf0\x90\x90\x94") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x95", "\xf0\x90\x90\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xbd", "\xf0\x90\x90\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x96", "\xf0\x90\x90\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xbe", "\xf0\x90\x90\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x97", "\xf0\x90\x90\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xbf", "\xf0\x90\x90\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x98", "\xf0\x90\x91\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x80", "\xf0\x90\x90\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x99", "\xf0\x90\x91\x81") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x81", "\xf0\x90\x90\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x9a", "\xf0\x90\x91\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x82", "\xf0\x90\x90\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x9b", "\xf0\x90\x91\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x83", "\xf0\x90\x90\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x9c", "\xf0\x90\x91\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x84", "\xf0\x90\x90\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x9d", "\xf0\x90\x91\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x85", "\xf0\x90\x90\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x9e", "\xf0\x90\x91\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x86", "\xf0\x90\x90\x9e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\x9f", "\xf0\x90\x91\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x87", "\xf0\x90\x90\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa0", "\xf0\x90\x91\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x88", "\xf0\x90\x90\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa1", "\xf0\x90\x91\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x89", "\xf0\x90\x90\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa2", "\xf0\x90\x91\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x8a", "\xf0\x90\x90\xa2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa3", "\xf0\x90\x91\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x8b", "\xf0\x90\x90\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa4", "\xf0\x90\x91\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x8c", "\xf0\x90\x90\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa5", "\xf0\x90\x91\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x8d", "\xf0\x90\x90\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa6", "\xf0\x90\x91\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x8e", "\xf0\x90\x90\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa7", "\xf0\x90\x91\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x8f", "\xf0\x90\x90\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1\x2\x3\x4\x5\x6\x7\x8\x9\xa", "\x1\x2\x3\x4\x5\x6\x7\x8\x9\xa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb\xc\xd\xe\xf\x10\x11\x12\x13\x14", "\xb\xc\xd\xe\xf\x10\x11\x12\x13\x14") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e", "\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x1f\x20\x21\x22\x23\x24\x25\x26\x27\x28", "\x1f\x20\x21\x22\x23\x24\x25\x26\x27\x28") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x29\x2a\x2b\x2c\x2d\x2e\x2f\x30\x31\x32", "\x29\x2a\x2b\x2c\x2d\x2e\x2f\x30\x31\x32") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x33\x34\x35\x36\x37\x38\x39\x3a\x3b\x3c", "\x33\x34\x35\x36\x37\x38\x39\x3a\x3b\x3c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x3d\x3e\x3f\x40\x41\x42\x43\x44\x45\x46", "\x3d\x3e\x3f\x40\x41\x42\x43\x44\x45\x46") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x47\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f\x50", "\x47\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f\x50") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x51\x52\x53\x54\x55\x56\x57\x58\x59\x5a", "\x51\x52\x53\x54\x55\x56\x57\x58\x59\x5a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x5b\x5c\x5d\x5e\x5f\x60\x61\x62\x63\x64", "\x5b\x5c\x5d\x5e\x5f\x60\x61\x62\x63\x64") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x65\x66\x67\x68\x69\x6a\x6b\x6c\x6d\x6e", "\x65\x66\x67\x68\x69\x6a\x6b\x6c\x6d\x6e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6f\x70\x71\x72\x73\x74\x75\x76\x77\x78", "\x6f\x70\x71\x72\x73\x74\x75\x76\x77\x78") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x79\x7a\x7b\x7c\x7d\x7e\x7f\x80\x81\x82", "\x79\x7a\x7b\x7c\x7d\x7e\x7f\x80\x81\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c", "\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96", "\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f\xa0", "\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8\xa9\xaa", "\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8\xa9\xaa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xab\xac\xad\xae\xaf\xb0\xb1\xb2\xb3\xb4", "\xab\xac\xad\xae\xaf\xb0\xb1\xb2\xb3\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xb5\xb6\xb7\xb8\xb9\xba\xbb\xbc\xbd\xbe", "\xb5\xb6\xb7\xb8\xb9\xba\xbb\xbc\xbd\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xbf\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8", "\xbf\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0\xd1\xd2", "\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0\xd1\xd2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xd4\xd5\xd6\xd7\xd8\xd9\xda\xdb\xdc", "\xd3\xd4\xd5\xd6\xd7\xd8\xd9\xda\xdb\xdc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xdd\xde\xdf\xe0\xe1\xe2\xe3\xe4\xe5\xe6", "\xdd\xde\xdf\xe0\xe1\xe2\xe3\xe4\xe5\xe6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe7\xe8\xe9\xea\xeb\xec\xed\xee\xef\xf0", "\xe7\xe8\xe9\xea\xeb\xec\xed\xee\xef\xf0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9\xfa", "\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9\xfa") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xfb\xfc\xfd\xfe\xff\x41\x61\x42\x62\x43", "\xfb\xfc\xfd\xfe\xff\x61\x41\x62\x42\x63") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x63\x44\x64\x45\x65\x46\x66\x47\x67\x48", "\x43\x64\x44\x65\x45\x66\x46\x67\x47\x68") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x68\x49\x69\x49\xc4\xb1\x4a\x6a\x4b\x6b\x4c", "\x48\x69\x49\xc4\xb1\x49\x6a\x4a\x6b\x4b\x6c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x6c\x4d\x6d\x4e\x6e\x4f\x6f\x50\x70\x51", "\x4c\x6d\x4d\x6e\x4e\x6f\x4f\x70\x50\x71") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x71\x52\x72\x53\x73\x54\x74\x55\x75\x56", "\x51\x72\x52\x73\x53\x74\x54\x75\x55\x76") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x76\x57\x77\x58\x78\x59\x79\x5a\x7a\xc2\xb5", "\x56\x77\x57\x78\x58\x79\x59\x7a\x5a\xce\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbc\xc3\x80\xc3\xa0\xc3\x81\xc3\xa1\xc3\x82\xc3\xa2\xc3\x83\xc3\xa3\xc3\x84", "\xc2\xb5\xc3\xa0\xc3\x80\xc3\xa1\xc3\x81\xc3\xa2\xc3\x82\xc3\xa3\xc3\x83\xc3\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa4\xc3\x85\xc3\xa5\xc3\x86\xc3\xa6\xc3\x87\xc3\xa7\xc3\x88\xc3\xa8\xc3\x89", "\xc3\x84\xc3\xa5\xc3\x85\xc3\xa6\xc3\x86\xc3\xa7\xc3\x87\xc3\xa8\xc3\x88\xc3\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xa9\xc3\x8a\xc3\xaa\xc3\x8b\xc3\xab\xc3\x8c\xc3\xac\xc3\x8d\xc3\xad\xc3\x8e", "\xc3\x89\xc3\xaa\xc3\x8a\xc3\xab\xc3\x8b\xc3\xac\xc3\x8c\xc3\xad\xc3\x8d\xc3\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xae\xc3\x8f\xc3\xaf\xc3\x90\xc3\xb0\xc3\x91\xc3\xb1\xc3\x92\xc3\xb2\xc3\x93", "\xc3\x8e\xc3\xaf\xc3\x8f\xc3\xb0\xc3\x90\xc3\xb1\xc3\x91\xc3\xb2\xc3\x92\xc3\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb3\xc3\x94\xc3\xb4\xc3\x95\xc3\xb5\xc3\x96\xc3\xb6\xc3\x98\xc3\xb8\xc3\x99", "\xc3\x93\xc3\xb4\xc3\x94\xc3\xb5\xc3\x95\xc3\xb6\xc3\x96\xc3\xb8\xc3\x98\xc3\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xb9\xc3\x9a\xc3\xba\xc3\x9b\xc3\xbb\xc3\x9c\xc3\xbc\xc3\x9d\xc3\xbd\xc3\x9e", "\xc3\x99\xc3\xba\xc3\x9a\xc3\xbb\xc3\x9b\xc3\xbc\xc3\x9c\xc3\xbd\xc3\x9d\xc3\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc3\xbe\xc3\x9f\x73\x73\xc4\x80\xc4\x81\xc4\x82\xc4\x83\xc4\x84\xc4\x85\xc4\x86", "\xc3\x9e\x73\x73\xc3\x9f\xc4\x81\xc4\x80\xc4\x83\xc4\x82\xc4\x85\xc4\x84\xc4\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x87\xc4\x88\xc4\x89\xc4\x8a\xc4\x8b\xc4\x8c\xc4\x8d\xc4\x8e\xc4\x8f\xc4\x90", "\xc4\x86\xc4\x89\xc4\x88\xc4\x8b\xc4\x8a\xc4\x8d\xc4\x8c\xc4\x8f\xc4\x8e\xc4\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x91\xc4\x92\xc4\x93\xc4\x94\xc4\x95\xc4\x96\xc4\x97\xc4\x98\xc4\x99\xc4\x9a", "\xc4\x90\xc4\x93\xc4\x92\xc4\x95\xc4\x94\xc4\x97\xc4\x96\xc4\x99\xc4\x98\xc4\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\x9b\xc4\x9c\xc4\x9d\xc4\x9e\xc4\x9f\xc4\xa0\xc4\xa1\xc4\xa2\xc4\xa3\xc4\xa4", "\xc4\x9a\xc4\x9d\xc4\x9c\xc4\x9f\xc4\x9e\xc4\xa1\xc4\xa0\xc4\xa3\xc4\xa2\xc4\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xa5\xc4\xa6\xc4\xa7\xc4\xa8\xc4\xa9\xc4\xaa\xc4\xab\xc4\xac\xc4\xad\xc4\xae", "\xc4\xa4\xc4\xa7\xc4\xa6\xc4\xa9\xc4\xa8\xc4\xab\xc4\xaa\xc4\xad\xc4\xac\xc4\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xaf\xc4\xb0\x69\xcc\x87\xc4\xb0\x69\xc4\xb2\xc4\xb3\xc4\xb4\xc4\xb5\xc4\xb6", "\xc4\xae\x69\xcc\x87\xc4\xb0\x69\xc4\xb0\xc4\xb3\xc4\xb2\xc4\xb5\xc4\xb4\xc4\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc4\xb7\xc4\xb9\xc4\xba\xc4\xbb\xc4\xbc\xc4\xbd\xc4\xbe\xc4\xbf\xc5\x80\xc5\x81", "\xc4\xb6\xc4\xba\xc4\xb9\xc4\xbc\xc4\xbb\xc4\xbe\xc4\xbd\xc5\x80\xc4\xbf\xc5\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x82\xc5\x83\xc5\x84\xc5\x85\xc5\x86\xc5\x87\xc5\x88\xc5\x89\xca\xbc\x6e\xc5\x8a", "\xc5\x81\xc5\x84\xc5\x83\xc5\x86\xc5\x85\xc5\x88\xc5\x87\xca\xbc\x6e\xc5\x89\xc5\x8b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x8b\xc5\x8c\xc5\x8d\xc5\x8e\xc5\x8f\xc5\x90\xc5\x91\xc5\x92\xc5\x93\xc5\x94", "\xc5\x8a\xc5\x8d\xc5\x8c\xc5\x8f\xc5\x8e\xc5\x91\xc5\x90\xc5\x93\xc5\x92\xc5\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x95\xc5\x96\xc5\x97\xc5\x98\xc5\x99\xc5\x9a\xc5\x9b\xc5\x9c\xc5\x9d\xc5\x9e", "\xc5\x94\xc5\x97\xc5\x96\xc5\x99\xc5\x98\xc5\x9b\xc5\x9a\xc5\x9d\xc5\x9c\xc5\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\x9f\xc5\xa0\xc5\xa1\xc5\xa2\xc5\xa3\xc5\xa4\xc5\xa5\xc5\xa6\xc5\xa7\xc5\xa8", "\xc5\x9e\xc5\xa1\xc5\xa0\xc5\xa3\xc5\xa2\xc5\xa5\xc5\xa4\xc5\xa7\xc5\xa6\xc5\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xa9\xc5\xaa\xc5\xab\xc5\xac\xc5\xad\xc5\xae\xc5\xaf\xc5\xb0\xc5\xb1\xc5\xb2", "\xc5\xa8\xc5\xab\xc5\xaa\xc5\xad\xc5\xac\xc5\xaf\xc5\xae\xc5\xb1\xc5\xb0\xc5\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xb3\xc5\xb4\xc5\xb5\xc5\xb6\xc5\xb7\xc5\xb8\xc3\xbf\xc5\xb9\xc5\xba\xc5\xbb", "\xc5\xb2\xc5\xb5\xc5\xb4\xc5\xb7\xc5\xb6\xc3\xbf\xc5\xb8\xc5\xba\xc5\xb9\xc5\xbc") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc5\xbc\xc5\xbd\xc5\xbe\xc5\xbf\x73\xc6\x81\xc9\x93\xc6\x82\xc6\x83\xc6\x84", "\xc5\xbb\xc5\xbe\xc5\xbd\x73\xc5\xbf\xc9\x93\xc6\x81\xc6\x83\xc6\x82\xc6\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x85\xc6\x86\xc9\x94\xc6\x87\xc6\x88\xc6\x89\xc9\x96\xc6\x8a\xc9\x97\xc6\x8b", "\xc6\x84\xc9\x94\xc6\x86\xc6\x88\xc6\x87\xc9\x96\xc6\x89\xc9\x97\xc6\x8a\xc6\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\x8c\xc6\x8e\xc7\x9d\xc6\x8f\xc9\x99\xc6\x90\xc9\x9b\xc6\x91\xc6\x92\xc6\x93", "\xc6\x8b\xc7\x9d\xc6\x8e\xc9\x99\xc6\x8f\xc9\x9b\xc6\x90\xc6\x92\xc6\x91\xc9\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xa0\xc6\x94\xc9\xa3\xc6\x96\xc9\xa9\xc6\x97\xc9\xa8\xc6\x98\xc6\x99\xc6\x9c", "\xc6\x93\xc9\xa3\xc6\x94\xc9\xa9\xc6\x96\xc9\xa8\xc6\x97\xc6\x99\xc6\x98\xc9\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc9\xaf\xc6\x9d\xc9\xb2\xc6\x9f\xc9\xb5\xc6\xa0\xc6\xa1\xc6\xa2\xc6\xa3\xc6\xa4", "\xc6\x9c\xc9\xb2\xc6\x9d\xc9\xb5\xc6\x9f\xc6\xa1\xc6\xa0\xc6\xa3\xc6\xa2\xc6\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xa5\xc6\xa6\xca\x80\xc6\xa7\xc6\xa8\xc6\xa9\xca\x83\xc6\xac\xc6\xad\xc6\xae", "\xc6\xa4\xca\x80\xc6\xa6\xc6\xa8\xc6\xa7\xca\x83\xc6\xa9\xc6\xad\xc6\xac\xca\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xca\x88\xc6\xaf\xc6\xb0\xc6\xb1\xca\x8a\xc6\xb2\xca\x8b\xc6\xb3\xc6\xb4\xc6\xb5", "\xc6\xae\xc6\xb0\xc6\xaf\xca\x8a\xc6\xb1\xca\x8b\xc6\xb2\xc6\xb4\xc6\xb3\xc6\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc6\xb6\xc6\xb7\xca\x92\xc6\xb8\xc6\xb9\xc6\xbc\xc6\xbd\xc7\x84\xc7\x86\xc7\x85", "\xc6\xb5\xca\x92\xc6\xb7\xc6\xb9\xc6\xb8\xc6\xbd\xc6\xbc\xc7\x86\xc7\x84\xc7\x86") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x86\xc7\x87\xc7\x89\xc7\x88\xc7\x89\xc7\x8a\xc7\x8c\xc7\x8b\xc7\x8c\xc7\x8d", "\xc7\x85\xc7\x89\xc7\x87\xc7\x89\xc7\x88\xc7\x8c\xc7\x8a\xc7\x8c\xc7\x8b\xc7\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x8e\xc7\x8f\xc7\x90\xc7\x91\xc7\x92\xc7\x93\xc7\x94\xc7\x95\xc7\x96\xc7\x97", "\xc7\x8d\xc7\x90\xc7\x8f\xc7\x92\xc7\x91\xc7\x94\xc7\x93\xc7\x96\xc7\x95\xc7\x98") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\x98\xc7\x99\xc7\x9a\xc7\x9b\xc7\x9c\xc7\x9e\xc7\x9f\xc7\xa0\xc7\xa1\xc7\xa2", "\xc7\x97\xc7\x9a\xc7\x99\xc7\x9c\xc7\x9b\xc7\x9f\xc7\x9e\xc7\xa1\xc7\xa0\xc7\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xa3\xc7\xa4\xc7\xa5\xc7\xa6\xc7\xa7\xc7\xa8\xc7\xa9\xc7\xaa\xc7\xab\xc7\xac", "\xc7\xa2\xc7\xa5\xc7\xa4\xc7\xa7\xc7\xa6\xc7\xa9\xc7\xa8\xc7\xab\xc7\xaa\xc7\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xad\xc7\xae\xc7\xaf\xc7\xb0\x6a\xcc\x8c\xc7\xb1\xc7\xb3\xc7\xb2\xc7\xb3\xc7\xb4", "\xc7\xac\xc7\xaf\xc7\xae\x6a\xcc\x8c\xc7\xb0\xc7\xb3\xc7\xb1\xc7\xb3\xc7\xb2\xc7\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xb5\xc7\xb6\xc6\x95\xc7\xb7\xc6\xbf\xc7\xb8\xc7\xb9\xc7\xba\xc7\xbb\xc7\xbc", "\xc7\xb4\xc6\x95\xc7\xb6\xc6\xbf\xc7\xb7\xc7\xb9\xc7\xb8\xc7\xbb\xc7\xba\xc7\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc7\xbd\xc7\xbe\xc7\xbf\xc8\x80\xc8\x81\xc8\x82\xc8\x83\xc8\x84\xc8\x85\xc8\x86", "\xc7\xbc\xc7\xbf\xc7\xbe\xc8\x81\xc8\x80\xc8\x83\xc8\x82\xc8\x85\xc8\x84\xc8\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x87\xc8\x88\xc8\x89\xc8\x8a\xc8\x8b\xc8\x8c\xc8\x8d\xc8\x8e\xc8\x8f\xc8\x90", "\xc8\x86\xc8\x89\xc8\x88\xc8\x8b\xc8\x8a\xc8\x8d\xc8\x8c\xc8\x8f\xc8\x8e\xc8\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x91\xc8\x92\xc8\x93\xc8\x94\xc8\x95\xc8\x96\xc8\x97\xc8\x98\xc8\x99\xc8\x9a", "\xc8\x90\xc8\x93\xc8\x92\xc8\x95\xc8\x94\xc8\x97\xc8\x96\xc8\x99\xc8\x98\xc8\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\x9b\xc8\x9c\xc8\x9d\xc8\x9e\xc8\x9f\xc8\xa0\xc6\x9e\xc8\xa2\xc8\xa3\xc8\xa4", "\xc8\x9a\xc8\x9d\xc8\x9c\xc8\x9f\xc8\x9e\xc6\x9e\xc8\xa0\xc8\xa3\xc8\xa2\xc8\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xa5\xc8\xa6\xc8\xa7\xc8\xa8\xc8\xa9\xc8\xaa\xc8\xab\xc8\xac\xc8\xad\xc8\xae", "\xc8\xa4\xc8\xa7\xc8\xa6\xc8\xa9\xc8\xa8\xc8\xab\xc8\xaa\xc8\xad\xc8\xac\xc8\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xc8\xaf\xc8\xb0\xc8\xb1\xc8\xb2\xc8\xb3\xcd\x85\xce\xb9\xce\x86\xce\xac\xce\x88", "\xc8\xae\xc8\xb1\xc8\xb0\xc8\xb3\xc8\xb2\xce\xb9\xcd\x85\xce\xac\xce\x86\xce\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xad\xce\x89\xce\xae\xce\x8a\xce\xaf\xce\x8c\xcf\x8c\xce\x8e\xcf\x8d\xce\x8f", "\xce\x88\xce\xae\xce\x89\xce\xaf\xce\x8a\xcf\x8c\xce\x8c\xcf\x8d\xce\x8e\xcf\x8e") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x8e\xce\x90\xce\xb9\xcc\x88\xcc\x81\xce\x91\xce\xb1\xce\x92\xce\xb2\xce\x93\xce\xb3\xce\x94", "\xce\x8f\xce\xb9\xcc\x88\xcc\x81\xce\x90\xce\xb1\xce\x91\xce\xb2\xce\x92\xce\xb3\xce\x93\xce\xb4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb4\xce\x95\xce\xb5\xce\x96\xce\xb6\xce\x97\xce\xb7\xce\x98\xce\xb8\xce\x99", "\xce\x94\xce\xb5\xce\x95\xce\xb6\xce\x96\xce\xb7\xce\x97\xce\xb8\xce\x98\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xce\x9a\xce\xba\xce\x9b\xce\xbb\xce\x9c\xce\xbc\xce\x9d\xce\xbd\xce\x9e", "\xce\x99\xce\xba\xce\x9a\xce\xbb\xce\x9b\xce\xbc\xce\x9c\xce\xbd\xce\x9d\xce\xbe") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xbe\xce\x9f\xce\xbf\xce\xa0\xcf\x80\xce\xa1\xcf\x81\xce\xa3\xcf\x83\xce\xa4", "\xce\x9e\xce\xbf\xce\x9f\xcf\x80\xce\xa0\xcf\x81\xce\xa1\xcf\x83\xce\xa3\xcf\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x84\xce\xa5\xcf\x85\xce\xa6\xcf\x86\xce\xa7\xcf\x87\xce\xa8\xcf\x88\xce\xa9", "\xce\xa4\xcf\x85\xce\xa5\xcf\x86\xce\xa6\xcf\x87\xce\xa7\xcf\x88\xce\xa8\xcf\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x89\xce\xaa\xcf\x8a\xce\xab\xcf\x8b\xce\xb0\xcf\x85\xcc\x88\xcc\x81\xcf\x82\xcf\x83\xcf\x90", "\xce\xa9\xcf\x8a\xce\xaa\xcf\x8b\xce\xab\xcf\x85\xcc\x88\xcc\x81\xce\xb0\xcf\x83\xcf\x82\xce\xb2") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb2\xcf\x91\xce\xb8\xcf\x95\xcf\x86\xcf\x96\xcf\x80\xcf\x98\xcf\x99\xcf\x9a", "\xcf\x90\xce\xb8\xcf\x91\xcf\x86\xcf\x95\xcf\x80\xcf\x96\xcf\x99\xcf\x98\xcf\x9b") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x9b\xcf\x9c\xcf\x9d\xcf\x9e\xcf\x9f\xcf\xa0\xcf\xa1\xcf\xa2\xcf\xa3\xcf\xa4", "\xcf\x9a\xcf\x9d\xcf\x9c\xcf\x9f\xcf\x9e\xcf\xa1\xcf\xa0\xcf\xa3\xcf\xa2\xcf\xa5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xa5\xcf\xa6\xcf\xa7\xcf\xa8\xcf\xa9\xcf\xaa\xcf\xab\xcf\xac\xcf\xad\xcf\xae", "\xcf\xa4\xcf\xa7\xcf\xa6\xcf\xa9\xcf\xa8\xcf\xab\xcf\xaa\xcf\xad\xcf\xac\xcf\xaf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xaf\xcf\xb0\xce\xba\xcf\xb1\xcf\x81\xcf\xb4\xce\xb8\xcf\xb5\xce\xb5\xcf\xb7", "\xcf\xae\xce\xba\xcf\xb0\xcf\x81\xcf\xb1\xce\xb8\xcf\xb4\xce\xb5\xcf\xb5\xcf\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\xb8\xcf\xb9\xcf\xb2\xcf\xba\xcf\xbb\xd0\x80\xd1\x90\xd0\x81\xd1\x91\xd0\x82", "\xcf\xb7\xcf\xb2\xcf\xb9\xcf\xbb\xcf\xba\xd1\x90\xd0\x80\xd1\x91\xd0\x81\xd1\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x92\xd0\x83\xd1\x93\xd0\x84\xd1\x94\xd0\x85\xd1\x95\xd0\x86\xd1\x96\xd0\x87", "\xd0\x82\xd1\x93\xd0\x83\xd1\x94\xd0\x84\xd1\x95\xd0\x85\xd1\x96\xd0\x86\xd1\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x97\xd0\x88\xd1\x98\xd0\x89\xd1\x99\xd0\x8a\xd1\x9a\xd0\x8b\xd1\x9b\xd0\x8c", "\xd0\x87\xd1\x98\xd0\x88\xd1\x99\xd0\x89\xd1\x9a\xd0\x8a\xd1\x9b\xd0\x8b\xd1\x9c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x9c\xd0\x8d\xd1\x9d\xd0\x8e\xd1\x9e\xd0\x8f\xd1\x9f\xd0\x90\xd0\xb0\xd0\x91", "\xd0\x8c\xd1\x9d\xd0\x8d\xd1\x9e\xd0\x8e\xd1\x9f\xd0\x8f\xd0\xb0\xd0\x90\xd0\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb1\xd0\x92\xd0\xb2\xd0\x93\xd0\xb3\xd0\x94\xd0\xb4\xd0\x95\xd0\xb5\xd0\x96", "\xd0\x91\xd0\xb2\xd0\x92\xd0\xb3\xd0\x93\xd0\xb4\xd0\x94\xd0\xb5\xd0\x95\xd0\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xb6\xd0\x97\xd0\xb7\xd0\x98\xd0\xb8\xd0\x99\xd0\xb9\xd0\x9a\xd0\xba\xd0\x9b", "\xd0\x96\xd0\xb7\xd0\x97\xd0\xb8\xd0\x98\xd0\xb9\xd0\x99\xd0\xba\xd0\x9a\xd0\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd0\xbb\xd0\x9c\xd0\xbc\xd0\x9d\xd0\xbd\xd0\x9e\xd0\xbe\xd0\x9f\xd0\xbf\xd0\xa0", "\xd0\x9b\xd0\xbc\xd0\x9c\xd0\xbd\xd0\x9d\xd0\xbe\xd0\x9e\xd0\xbf\xd0\x9f\xd1\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x80\xd0\xa1\xd1\x81\xd0\xa2\xd1\x82\xd0\xa3\xd1\x83\xd0\xa4\xd1\x84\xd0\xa5", "\xd0\xa0\xd1\x81\xd0\xa1\xd1\x82\xd0\xa2\xd1\x83\xd0\xa3\xd1\x84\xd0\xa4\xd1\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x85\xd0\xa6\xd1\x86\xd0\xa7\xd1\x87\xd0\xa8\xd1\x88\xd0\xa9\xd1\x89\xd0\xaa", "\xd0\xa5\xd1\x86\xd0\xa6\xd1\x87\xd0\xa7\xd1\x88\xd0\xa8\xd1\x89\xd0\xa9\xd1\x8a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8a\xd0\xab\xd1\x8b\xd0\xac\xd1\x8c\xd0\xad\xd1\x8d\xd0\xae\xd1\x8e\xd0\xaf", "\xd0\xaa\xd1\x8b\xd0\xab\xd1\x8c\xd0\xac\xd1\x8d\xd0\xad\xd1\x8e\xd0\xae\xd1\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\x8f\xd1\xa0\xd1\xa1\xd1\xa2\xd1\xa3\xd1\xa4\xd1\xa5\xd1\xa6\xd1\xa7\xd1\xa8", "\xd0\xaf\xd1\xa1\xd1\xa0\xd1\xa3\xd1\xa2\xd1\xa5\xd1\xa4\xd1\xa7\xd1\xa6\xd1\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xa9\xd1\xaa\xd1\xab\xd1\xac\xd1\xad\xd1\xae\xd1\xaf\xd1\xb0\xd1\xb1\xd1\xb2", "\xd1\xa8\xd1\xab\xd1\xaa\xd1\xad\xd1\xac\xd1\xaf\xd1\xae\xd1\xb1\xd1\xb0\xd1\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xb3\xd1\xb4\xd1\xb5\xd1\xb6\xd1\xb7\xd1\xb8\xd1\xb9\xd1\xba\xd1\xbb\xd1\xbc", "\xd1\xb2\xd1\xb5\xd1\xb4\xd1\xb7\xd1\xb6\xd1\xb9\xd1\xb8\xd1\xbb\xd1\xba\xd1\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd1\xbd\xd1\xbe\xd1\xbf\xd2\x80\xd2\x81\xd2\x8a\xd2\x8b\xd2\x8c\xd2\x8d\xd2\x8e", "\xd1\xbc\xd1\xbf\xd1\xbe\xd2\x81\xd2\x80\xd2\x8b\xd2\x8a\xd2\x8d\xd2\x8c\xd2\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x8f\xd2\x90\xd2\x91\xd2\x92\xd2\x93\xd2\x94\xd2\x95\xd2\x96\xd2\x97\xd2\x98", "\xd2\x8e\xd2\x91\xd2\x90\xd2\x93\xd2\x92\xd2\x95\xd2\x94\xd2\x97\xd2\x96\xd2\x99") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\x99\xd2\x9a\xd2\x9b\xd2\x9c\xd2\x9d\xd2\x9e\xd2\x9f\xd2\xa0\xd2\xa1\xd2\xa2", "\xd2\x98\xd2\x9b\xd2\x9a\xd2\x9d\xd2\x9c\xd2\x9f\xd2\x9e\xd2\xa1\xd2\xa0\xd2\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xa3\xd2\xa4\xd2\xa5\xd2\xa6\xd2\xa7\xd2\xa8\xd2\xa9\xd2\xaa\xd2\xab\xd2\xac", "\xd2\xa2\xd2\xa5\xd2\xa4\xd2\xa7\xd2\xa6\xd2\xa9\xd2\xa8\xd2\xab\xd2\xaa\xd2\xad") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xad\xd2\xae\xd2\xaf\xd2\xb0\xd2\xb1\xd2\xb2\xd2\xb3\xd2\xb4\xd2\xb5\xd2\xb6", "\xd2\xac\xd2\xaf\xd2\xae\xd2\xb1\xd2\xb0\xd2\xb3\xd2\xb2\xd2\xb5\xd2\xb4\xd2\xb7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd2\xb7\xd2\xb8\xd2\xb9\xd2\xba\xd2\xbb\xd2\xbc\xd2\xbd\xd2\xbe\xd2\xbf\xd3\x81", "\xd2\xb6\xd2\xb9\xd2\xb8\xd2\xbb\xd2\xba\xd2\xbd\xd2\xbc\xd2\xbf\xd2\xbe\xd3\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x82\xd3\x83\xd3\x84\xd3\x85\xd3\x86\xd3\x87\xd3\x88\xd3\x89\xd3\x8a\xd3\x8b", "\xd3\x81\xd3\x84\xd3\x83\xd3\x86\xd3\x85\xd3\x88\xd3\x87\xd3\x8a\xd3\x89\xd3\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x8c\xd3\x8d\xd3\x8e\xd3\x90\xd3\x91\xd3\x92\xd3\x93\xd3\x94\xd3\x95\xd3\x96", "\xd3\x8b\xd3\x8e\xd3\x8d\xd3\x91\xd3\x90\xd3\x93\xd3\x92\xd3\x95\xd3\x94\xd3\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\x97\xd3\x98\xd3\x99\xd3\x9a\xd3\x9b\xd3\x9c\xd3\x9d\xd3\x9e\xd3\x9f\xd3\xa0", "\xd3\x96\xd3\x99\xd3\x98\xd3\x9b\xd3\x9a\xd3\x9d\xd3\x9c\xd3\x9f\xd3\x9e\xd3\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xa1\xd3\xa2\xd3\xa3\xd3\xa4\xd3\xa5\xd3\xa6\xd3\xa7\xd3\xa8\xd3\xa9\xd3\xaa", "\xd3\xa0\xd3\xa3\xd3\xa2\xd3\xa5\xd3\xa4\xd3\xa7\xd3\xa6\xd3\xa9\xd3\xa8\xd3\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xab\xd3\xac\xd3\xad\xd3\xae\xd3\xaf\xd3\xb0\xd3\xb1\xd3\xb2\xd3\xb3\xd3\xb4", "\xd3\xaa\xd3\xad\xd3\xac\xd3\xaf\xd3\xae\xd3\xb1\xd3\xb0\xd3\xb3\xd3\xb2\xd3\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd3\xb5\xd3\xb8\xd3\xb9\xd4\x80\xd4\x81\xd4\x82\xd4\x83\xd4\x84\xd4\x85\xd4\x86", "\xd3\xb4\xd3\xb9\xd3\xb8\xd4\x81\xd4\x80\xd4\x83\xd4\x82\xd4\x85\xd4\x84\xd4\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd4\x87\xd4\x88\xd4\x89\xd4\x8a\xd4\x8b\xd4\x8c\xd4\x8d\xd4\x8e\xd4\x8f\xd4\xb1", "\xd4\x86\xd4\x89\xd4\x88\xd4\x8b\xd4\x8a\xd4\x8d\xd4\x8c\xd4\x8f\xd4\x8e\xd5\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa1\xd4\xb2\xd5\xa2\xd4\xb3\xd5\xa3\xd4\xb4\xd5\xa4\xd4\xb5\xd5\xa5\xd4\xb6", "\xd4\xb1\xd5\xa2\xd4\xb2\xd5\xa3\xd4\xb3\xd5\xa4\xd4\xb4\xd5\xa5\xd4\xb5\xd5\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xa6\xd4\xb7\xd5\xa7\xd4\xb8\xd5\xa8\xd4\xb9\xd5\xa9\xd4\xba\xd5\xaa\xd4\xbb", "\xd4\xb6\xd5\xa7\xd4\xb7\xd5\xa8\xd4\xb8\xd5\xa9\xd4\xb9\xd5\xaa\xd4\xba\xd5\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xab\xd4\xbc\xd5\xac\xd4\xbd\xd5\xad\xd4\xbe\xd5\xae\xd4\xbf\xd5\xaf\xd5\x80", "\xd4\xbb\xd5\xac\xd4\xbc\xd5\xad\xd4\xbd\xd5\xae\xd4\xbe\xd5\xaf\xd4\xbf\xd5\xb0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb0\xd5\x81\xd5\xb1\xd5\x82\xd5\xb2\xd5\x83\xd5\xb3\xd5\x84\xd5\xb4\xd5\x85", "\xd5\x80\xd5\xb1\xd5\x81\xd5\xb2\xd5\x82\xd5\xb3\xd5\x83\xd5\xb4\xd5\x84\xd5\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb5\xd5\x86\xd5\xb6\xd5\x87\xd5\xb7\xd5\x88\xd5\xb8\xd5\x89\xd5\xb9\xd5\x8a", "\xd5\x85\xd5\xb6\xd5\x86\xd5\xb7\xd5\x87\xd5\xb8\xd5\x88\xd5\xb9\xd5\x89\xd5\xba") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xba\xd5\x8b\xd5\xbb\xd5\x8c\xd5\xbc\xd5\x8d\xd5\xbd\xd5\x8e\xd5\xbe\xd5\x8f", "\xd5\x8a\xd5\xbb\xd5\x8b\xd5\xbc\xd5\x8c\xd5\xbd\xd5\x8d\xd5\xbe\xd5\x8e\xd5\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xbf\xd5\x90\xd6\x80\xd5\x91\xd6\x81\xd5\x92\xd6\x82\xd5\x93\xd6\x83\xd5\x94", "\xd5\x8f\xd6\x80\xd5\x90\xd6\x81\xd5\x91\xd6\x82\xd5\x92\xd6\x83\xd5\x93\xd6\x84") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd6\x84\xd5\x95\xd6\x85\xd5\x96\xd6\x86\xd6\x87\xd5\xa5\xd6\x82\xe1\xb8\x80\xe1\xb8\x81\xe1\xb8\x82", "\xd5\x94\xd6\x85\xd5\x95\xd6\x86\xd5\x96\xd5\xa5\xd6\x82\xd6\x87\xe1\xb8\x81\xe1\xb8\x80\xe1\xb8\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x83\xe1\xb8\x84\xe1\xb8\x85\xe1\xb8\x86\xe1\xb8\x87\xe1\xb8\x88\xe1\xb8\x89\xe1\xb8\x8a\xe1\xb8\x8b\xe1\xb8\x8c", "\xe1\xb8\x82\xe1\xb8\x85\xe1\xb8\x84\xe1\xb8\x87\xe1\xb8\x86\xe1\xb8\x89\xe1\xb8\x88\xe1\xb8\x8b\xe1\xb8\x8a\xe1\xb8\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x8d\xe1\xb8\x8e\xe1\xb8\x8f\xe1\xb8\x90\xe1\xb8\x91\xe1\xb8\x92\xe1\xb8\x93\xe1\xb8\x94\xe1\xb8\x95\xe1\xb8\x96", "\xe1\xb8\x8c\xe1\xb8\x8f\xe1\xb8\x8e\xe1\xb8\x91\xe1\xb8\x90\xe1\xb8\x93\xe1\xb8\x92\xe1\xb8\x95\xe1\xb8\x94\xe1\xb8\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\x97\xe1\xb8\x98\xe1\xb8\x99\xe1\xb8\x9a\xe1\xb8\x9b\xe1\xb8\x9c\xe1\xb8\x9d\xe1\xb8\x9e\xe1\xb8\x9f\xe1\xb8\xa0", "\xe1\xb8\x96\xe1\xb8\x99\xe1\xb8\x98\xe1\xb8\x9b\xe1\xb8\x9a\xe1\xb8\x9d\xe1\xb8\x9c\xe1\xb8\x9f\xe1\xb8\x9e\xe1\xb8\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xa1\xe1\xb8\xa2\xe1\xb8\xa3\xe1\xb8\xa4\xe1\xb8\xa5\xe1\xb8\xa6\xe1\xb8\xa7\xe1\xb8\xa8\xe1\xb8\xa9\xe1\xb8\xaa", "\xe1\xb8\xa0\xe1\xb8\xa3\xe1\xb8\xa2\xe1\xb8\xa5\xe1\xb8\xa4\xe1\xb8\xa7\xe1\xb8\xa6\xe1\xb8\xa9\xe1\xb8\xa8\xe1\xb8\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xab\xe1\xb8\xac\xe1\xb8\xad\xe1\xb8\xae\xe1\xb8\xaf\xe1\xb8\xb0\xe1\xb8\xb1\xe1\xb8\xb2\xe1\xb8\xb3\xe1\xb8\xb4", "\xe1\xb8\xaa\xe1\xb8\xad\xe1\xb8\xac\xe1\xb8\xaf\xe1\xb8\xae\xe1\xb8\xb1\xe1\xb8\xb0\xe1\xb8\xb3\xe1\xb8\xb2\xe1\xb8\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xb5\xe1\xb8\xb6\xe1\xb8\xb7\xe1\xb8\xb8\xe1\xb8\xb9\xe1\xb8\xba\xe1\xb8\xbb\xe1\xb8\xbc\xe1\xb8\xbd\xe1\xb8\xbe", "\xe1\xb8\xb4\xe1\xb8\xb7\xe1\xb8\xb6\xe1\xb8\xb9\xe1\xb8\xb8\xe1\xb8\xbb\xe1\xb8\xba\xe1\xb8\xbd\xe1\xb8\xbc\xe1\xb8\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb8\xbf\xe1\xb9\x80\xe1\xb9\x81\xe1\xb9\x82\xe1\xb9\x83\xe1\xb9\x84\xe1\xb9\x85\xe1\xb9\x86\xe1\xb9\x87\xe1\xb9\x88", "\xe1\xb8\xbe\xe1\xb9\x81\xe1\xb9\x80\xe1\xb9\x83\xe1\xb9\x82\xe1\xb9\x85\xe1\xb9\x84\xe1\xb9\x87\xe1\xb9\x86\xe1\xb9\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x89\xe1\xb9\x8a\xe1\xb9\x8b\xe1\xb9\x8c\xe1\xb9\x8d\xe1\xb9\x8e\xe1\xb9\x8f\xe1\xb9\x90\xe1\xb9\x91\xe1\xb9\x92", "\xe1\xb9\x88\xe1\xb9\x8b\xe1\xb9\x8a\xe1\xb9\x8d\xe1\xb9\x8c\xe1\xb9\x8f\xe1\xb9\x8e\xe1\xb9\x91\xe1\xb9\x90\xe1\xb9\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x93\xe1\xb9\x94\xe1\xb9\x95\xe1\xb9\x96\xe1\xb9\x97\xe1\xb9\x98\xe1\xb9\x99\xe1\xb9\x9a\xe1\xb9\x9b\xe1\xb9\x9c", "\xe1\xb9\x92\xe1\xb9\x95\xe1\xb9\x94\xe1\xb9\x97\xe1\xb9\x96\xe1\xb9\x99\xe1\xb9\x98\xe1\xb9\x9b\xe1\xb9\x9a\xe1\xb9\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\x9d\xe1\xb9\x9e\xe1\xb9\x9f\xe1\xb9\xa0\xe1\xb9\xa1\xe1\xb9\xa2\xe1\xb9\xa3\xe1\xb9\xa4\xe1\xb9\xa5\xe1\xb9\xa6", "\xe1\xb9\x9c\xe1\xb9\x9f\xe1\xb9\x9e\xe1\xb9\xa1\xe1\xb9\xa0\xe1\xb9\xa3\xe1\xb9\xa2\xe1\xb9\xa5\xe1\xb9\xa4\xe1\xb9\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xa7\xe1\xb9\xa8\xe1\xb9\xa9\xe1\xb9\xaa\xe1\xb9\xab\xe1\xb9\xac\xe1\xb9\xad\xe1\xb9\xae\xe1\xb9\xaf\xe1\xb9\xb0", "\xe1\xb9\xa6\xe1\xb9\xa9\xe1\xb9\xa8\xe1\xb9\xab\xe1\xb9\xaa\xe1\xb9\xad\xe1\xb9\xac\xe1\xb9\xaf\xe1\xb9\xae\xe1\xb9\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xb1\xe1\xb9\xb2\xe1\xb9\xb3\xe1\xb9\xb4\xe1\xb9\xb5\xe1\xb9\xb6\xe1\xb9\xb7\xe1\xb9\xb8\xe1\xb9\xb9\xe1\xb9\xba", "\xe1\xb9\xb0\xe1\xb9\xb3\xe1\xb9\xb2\xe1\xb9\xb5\xe1\xb9\xb4\xe1\xb9\xb7\xe1\xb9\xb6\xe1\xb9\xb9\xe1\xb9\xb8\xe1\xb9\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xb9\xbb\xe1\xb9\xbc\xe1\xb9\xbd\xe1\xb9\xbe\xe1\xb9\xbf\xe1\xba\x80\xe1\xba\x81\xe1\xba\x82\xe1\xba\x83\xe1\xba\x84", "\xe1\xb9\xba\xe1\xb9\xbd\xe1\xb9\xbc\xe1\xb9\xbf\xe1\xb9\xbe\xe1\xba\x81\xe1\xba\x80\xe1\xba\x83\xe1\xba\x82\xe1\xba\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x85\xe1\xba\x86\xe1\xba\x87\xe1\xba\x88\xe1\xba\x89\xe1\xba\x8a\xe1\xba\x8b\xe1\xba\x8c\xe1\xba\x8d\xe1\xba\x8e", "\xe1\xba\x84\xe1\xba\x87\xe1\xba\x86\xe1\xba\x89\xe1\xba\x88\xe1\xba\x8b\xe1\xba\x8a\xe1\xba\x8d\xe1\xba\x8c\xe1\xba\x8f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\x8f\xe1\xba\x90\xe1\xba\x91\xe1\xba\x92\xe1\xba\x93\xe1\xba\x94\xe1\xba\x95\xe1\xba\x96\x68\xcc\xb1\xe1\xba\x97", "\xe1\xba\x8e\xe1\xba\x91\xe1\xba\x90\xe1\xba\x93\xe1\xba\x92\xe1\xba\x95\xe1\xba\x94\x68\xcc\xb1\xe1\xba\x96\x74\xcc\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x74\xcc\x88\xe1\xba\x98\x77\xcc\x8a\xe1\xba\x99\x79\xcc\x8a\xe1\xba\x9a\x61\xca\xbe\xe1\xba\x9b\xe1\xb9\xa1\xe1\xba\xa0", "\xe1\xba\x97\x77\xcc\x8a\xe1\xba\x98\x79\xcc\x8a\xe1\xba\x99\x61\xca\xbe\xe1\xba\x9a\xe1\xb9\xa1\xe1\xba\x9b\xe1\xba\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xa1\xe1\xba\xa2\xe1\xba\xa3\xe1\xba\xa4\xe1\xba\xa5\xe1\xba\xa6\xe1\xba\xa7\xe1\xba\xa8\xe1\xba\xa9\xe1\xba\xaa", "\xe1\xba\xa0\xe1\xba\xa3\xe1\xba\xa2\xe1\xba\xa5\xe1\xba\xa4\xe1\xba\xa7\xe1\xba\xa6\xe1\xba\xa9\xe1\xba\xa8\xe1\xba\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xab\xe1\xba\xac\xe1\xba\xad\xe1\xba\xae\xe1\xba\xaf\xe1\xba\xb0\xe1\xba\xb1\xe1\xba\xb2\xe1\xba\xb3\xe1\xba\xb4", "\xe1\xba\xaa\xe1\xba\xad\xe1\xba\xac\xe1\xba\xaf\xe1\xba\xae\xe1\xba\xb1\xe1\xba\xb0\xe1\xba\xb3\xe1\xba\xb2\xe1\xba\xb5") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xb5\xe1\xba\xb6\xe1\xba\xb7\xe1\xba\xb8\xe1\xba\xb9\xe1\xba\xba\xe1\xba\xbb\xe1\xba\xbc\xe1\xba\xbd\xe1\xba\xbe", "\xe1\xba\xb4\xe1\xba\xb7\xe1\xba\xb6\xe1\xba\xb9\xe1\xba\xb8\xe1\xba\xbb\xe1\xba\xba\xe1\xba\xbd\xe1\xba\xbc\xe1\xba\xbf") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xba\xbf\xe1\xbb\x80\xe1\xbb\x81\xe1\xbb\x82\xe1\xbb\x83\xe1\xbb\x84\xe1\xbb\x85\xe1\xbb\x86\xe1\xbb\x87\xe1\xbb\x88", "\xe1\xba\xbe\xe1\xbb\x81\xe1\xbb\x80\xe1\xbb\x83\xe1\xbb\x82\xe1\xbb\x85\xe1\xbb\x84\xe1\xbb\x87\xe1\xbb\x86\xe1\xbb\x89") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x89\xe1\xbb\x8a\xe1\xbb\x8b\xe1\xbb\x8c\xe1\xbb\x8d\xe1\xbb\x8e\xe1\xbb\x8f\xe1\xbb\x90\xe1\xbb\x91\xe1\xbb\x92", "\xe1\xbb\x88\xe1\xbb\x8b\xe1\xbb\x8a\xe1\xbb\x8d\xe1\xbb\x8c\xe1\xbb\x8f\xe1\xbb\x8e\xe1\xbb\x91\xe1\xbb\x90\xe1\xbb\x93") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x93\xe1\xbb\x94\xe1\xbb\x95\xe1\xbb\x96\xe1\xbb\x97\xe1\xbb\x98\xe1\xbb\x99\xe1\xbb\x9a\xe1\xbb\x9b\xe1\xbb\x9c", "\xe1\xbb\x92\xe1\xbb\x95\xe1\xbb\x94\xe1\xbb\x97\xe1\xbb\x96\xe1\xbb\x99\xe1\xbb\x98\xe1\xbb\x9b\xe1\xbb\x9a\xe1\xbb\x9d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\x9d\xe1\xbb\x9e\xe1\xbb\x9f\xe1\xbb\xa0\xe1\xbb\xa1\xe1\xbb\xa2\xe1\xbb\xa3\xe1\xbb\xa4\xe1\xbb\xa5\xe1\xbb\xa6", "\xe1\xbb\x9c\xe1\xbb\x9f\xe1\xbb\x9e\xe1\xbb\xa1\xe1\xbb\xa0\xe1\xbb\xa3\xe1\xbb\xa2\xe1\xbb\xa5\xe1\xbb\xa4\xe1\xbb\xa7") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xa7\xe1\xbb\xa8\xe1\xbb\xa9\xe1\xbb\xaa\xe1\xbb\xab\xe1\xbb\xac\xe1\xbb\xad\xe1\xbb\xae\xe1\xbb\xaf\xe1\xbb\xb0", "\xe1\xbb\xa6\xe1\xbb\xa9\xe1\xbb\xa8\xe1\xbb\xab\xe1\xbb\xaa\xe1\xbb\xad\xe1\xbb\xac\xe1\xbb\xaf\xe1\xbb\xae\xe1\xbb\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbb\xb1\xe1\xbb\xb2\xe1\xbb\xb3\xe1\xbb\xb4\xe1\xbb\xb5\xe1\xbb\xb6\xe1\xbb\xb7\xe1\xbb\xb8\xe1\xbb\xb9\xe1\xbc\x88", "\xe1\xbb\xb0\xe1\xbb\xb3\xe1\xbb\xb2\xe1\xbb\xb5\xe1\xbb\xb4\xe1\xbb\xb7\xe1\xbb\xb6\xe1\xbb\xb9\xe1\xbb\xb8\xe1\xbc\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x80\xe1\xbc\x89\xe1\xbc\x81\xe1\xbc\x8a\xe1\xbc\x82\xe1\xbc\x8b\xe1\xbc\x83\xe1\xbc\x8c\xe1\xbc\x84\xe1\xbc\x8d", "\xe1\xbc\x88\xe1\xbc\x81\xe1\xbc\x89\xe1\xbc\x82\xe1\xbc\x8a\xe1\xbc\x83\xe1\xbc\x8b\xe1\xbc\x84\xe1\xbc\x8c\xe1\xbc\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x85\xe1\xbc\x8e\xe1\xbc\x86\xe1\xbc\x8f\xe1\xbc\x87\xe1\xbc\x98\xe1\xbc\x90\xe1\xbc\x99\xe1\xbc\x91\xe1\xbc\x9a", "\xe1\xbc\x8d\xe1\xbc\x86\xe1\xbc\x8e\xe1\xbc\x87\xe1\xbc\x8f\xe1\xbc\x90\xe1\xbc\x98\xe1\xbc\x91\xe1\xbc\x99\xe1\xbc\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x92\xe1\xbc\x9b\xe1\xbc\x93\xe1\xbc\x9c\xe1\xbc\x94\xe1\xbc\x9d\xe1\xbc\x95\xe1\xbc\xa8\xe1\xbc\xa0\xe1\xbc\xa9", "\xe1\xbc\x9a\xe1\xbc\x93\xe1\xbc\x9b\xe1\xbc\x94\xe1\xbc\x9c\xe1\xbc\x95\xe1\xbc\x9d\xe1\xbc\xa0\xe1\xbc\xa8\xe1\xbc\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa1\xe1\xbc\xaa\xe1\xbc\xa2\xe1\xbc\xab\xe1\xbc\xa3\xe1\xbc\xac\xe1\xbc\xa4\xe1\xbc\xad\xe1\xbc\xa5\xe1\xbc\xae", "\xe1\xbc\xa9\xe1\xbc\xa2\xe1\xbc\xaa\xe1\xbc\xa3\xe1\xbc\xab\xe1\xbc\xa4\xe1\xbc\xac\xe1\xbc\xa5\xe1\xbc\xad\xe1\xbc\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa6\xe1\xbc\xaf\xe1\xbc\xa7\xe1\xbc\xb8\xe1\xbc\xb0\xe1\xbc\xb9\xe1\xbc\xb1\xe1\xbc\xba\xe1\xbc\xb2\xe1\xbc\xbb", "\xe1\xbc\xae\xe1\xbc\xa7\xe1\xbc\xaf\xe1\xbc\xb0\xe1\xbc\xb8\xe1\xbc\xb1\xe1\xbc\xb9\xe1\xbc\xb2\xe1\xbc\xba\xe1\xbc\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xb3\xe1\xbc\xbc\xe1\xbc\xb4\xe1\xbc\xbd\xe1\xbc\xb5\xe1\xbc\xbe\xe1\xbc\xb6\xe1\xbc\xbf\xe1\xbc\xb7\xe1\xbd\x88", "\xe1\xbc\xbb\xe1\xbc\xb4\xe1\xbc\xbc\xe1\xbc\xb5\xe1\xbc\xbd\xe1\xbc\xb6\xe1\xbc\xbe\xe1\xbc\xb7\xe1\xbc\xbf\xe1\xbd\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x80\xe1\xbd\x89\xe1\xbd\x81\xe1\xbd\x8a\xe1\xbd\x82\xe1\xbd\x8b\xe1\xbd\x83\xe1\xbd\x8c\xe1\xbd\x84\xe1\xbd\x8d", "\xe1\xbd\x88\xe1\xbd\x81\xe1\xbd\x89\xe1\xbd\x82\xe1\xbd\x8a\xe1\xbd\x83\xe1\xbd\x8b\xe1\xbd\x84\xe1\xbd\x8c\xe1\xbd\x85") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x85\xe1\xbd\x90\xcf\x85\xcc\x93\xe1\xbd\x92\xcf\x85\xcc\x93\xcc\x80\xe1\xbd\x94\xcf\x85\xcc\x93\xcc\x81\xe1\xbd\x96\xcf\x85\xcc\x93\xcd\x82\xe1\xbd\x99", "\xe1\xbd\x8d\xcf\x85\xcc\x93\xe1\xbd\x90\xcf\x85\xcc\x93\xcc\x80\xe1\xbd\x92\xcf\x85\xcc\x93\xcc\x81\xe1\xbd\x94\xcf\x85\xcc\x93\xcd\x82\xe1\xbd\x96\xe1\xbd\x91") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\x91\xe1\xbd\x9b\xe1\xbd\x93\xe1\xbd\x9d\xe1\xbd\x95\xe1\xbd\x9f\xe1\xbd\x97\xe1\xbd\xa8\xe1\xbd\xa0\xe1\xbd\xa9", "\xe1\xbd\x99\xe1\xbd\x93\xe1\xbd\x9b\xe1\xbd\x95\xe1\xbd\x9d\xe1\xbd\x97\xe1\xbd\x9f\xe1\xbd\xa0\xe1\xbd\xa8\xe1\xbd\xa1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa1\xe1\xbd\xaa\xe1\xbd\xa2\xe1\xbd\xab\xe1\xbd\xa3\xe1\xbd\xac\xe1\xbd\xa4\xe1\xbd\xad\xe1\xbd\xa5\xe1\xbd\xae", "\xe1\xbd\xa9\xe1\xbd\xa2\xe1\xbd\xaa\xe1\xbd\xa3\xe1\xbd\xab\xe1\xbd\xa4\xe1\xbd\xac\xe1\xbd\xa5\xe1\xbd\xad\xe1\xbd\xa6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa6\xe1\xbd\xaf\xe1\xbd\xa7\xe1\xbe\x80\xe1\xbc\x80\xce\xb9\xe1\xbe\x81\xe1\xbc\x81\xce\xb9\xe1\xbe\x82\xe1\xbc\x82\xce\xb9\xe1\xbe\x83", "\xe1\xbd\xae\xe1\xbd\xa7\xe1\xbd\xaf\xe1\xbc\x80\xce\xb9\xe1\xbe\x80\xe1\xbc\x81\xce\xb9\xe1\xbe\x81\xe1\xbc\x82\xce\xb9\xe1\xbe\x82\xe1\xbc\x83\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x83\xce\xb9\xe1\xbe\x84\xe1\xbc\x84\xce\xb9\xe1\xbe\x85\xe1\xbc\x85\xce\xb9\xe1\xbe\x86\xe1\xbc\x86\xce\xb9\xe1\xbe\x87\xe1\xbc\x87\xce\xb9\xe1\xbe\x88", "\xe1\xbe\x83\xe1\xbc\x84\xce\xb9\xe1\xbe\x84\xe1\xbc\x85\xce\xb9\xe1\xbe\x85\xe1\xbc\x86\xce\xb9\xe1\xbe\x86\xe1\xbc\x87\xce\xb9\xe1\xbe\x87\xe1\xbc\x80\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x80\xce\xb9\xe1\xbe\x88\xe1\xbe\x80\xe1\xbe\x89\xe1\xbc\x81\xce\xb9\xe1\xbe\x89\xe1\xbe\x81\xe1\xbe\x8a\xe1\xbc\x82\xce\xb9\xe1\xbe\x8a", "\xe1\xbe\x88\xe1\xbe\x80\xe1\xbe\x88\xe1\xbc\x81\xce\xb9\xe1\xbe\x89\xe1\xbe\x81\xe1\xbe\x89\xe1\xbc\x82\xce\xb9\xe1\xbe\x8a\xe1\xbe\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x82\xe1\xbe\x8b\xe1\xbc\x83\xce\xb9\xe1\xbe\x8b\xe1\xbe\x83\xe1\xbe\x8c\xe1\xbc\x84\xce\xb9\xe1\xbe\x8c\xe1\xbe\x84\xe1\xbe\x8d", "\xe1\xbe\x8a\xe1\xbc\x83\xce\xb9\xe1\xbe\x8b\xe1\xbe\x83\xe1\xbe\x8b\xe1\xbc\x84\xce\xb9\xe1\xbe\x8c\xe1\xbe\x84\xe1\xbe\x8c\xe1\xbc\x85\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\x85\xce\xb9\xe1\xbe\x8d\xe1\xbe\x85\xe1\xbe\x8e\xe1\xbc\x86\xce\xb9\xe1\xbe\x8e\xe1\xbe\x86\xe1\xbe\x8f\xe1\xbc\x87\xce\xb9\xe1\xbe\x8f", "\xe1\xbe\x8d\xe1\xbe\x85\xe1\xbe\x8d\xe1\xbc\x86\xce\xb9\xe1\xbe\x8e\xe1\xbe\x86\xe1\xbe\x8e\xe1\xbc\x87\xce\xb9\xe1\xbe\x8f\xe1\xbe\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x87\xe1\xbe\x90\xe1\xbc\xa0\xce\xb9\xe1\xbe\x91\xe1\xbc\xa1\xce\xb9\xe1\xbe\x92\xe1\xbc\xa2\xce\xb9\xe1\xbe\x93\xe1\xbc\xa3\xce\xb9\xe1\xbe\x94", "\xe1\xbe\x8f\xe1\xbc\xa0\xce\xb9\xe1\xbe\x90\xe1\xbc\xa1\xce\xb9\xe1\xbe\x91\xe1\xbc\xa2\xce\xb9\xe1\xbe\x92\xe1\xbc\xa3\xce\xb9\xe1\xbe\x93\xe1\xbc\xa4\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa4\xce\xb9\xe1\xbe\x95\xe1\xbc\xa5\xce\xb9\xe1\xbe\x96\xe1\xbc\xa6\xce\xb9\xe1\xbe\x97\xe1\xbc\xa7\xce\xb9\xe1\xbe\x98\xe1\xbc\xa0\xce\xb9\xe1\xbe\x98", "\xe1\xbe\x94\xe1\xbc\xa5\xce\xb9\xe1\xbe\x95\xe1\xbc\xa6\xce\xb9\xe1\xbe\x96\xe1\xbc\xa7\xce\xb9\xe1\xbe\x97\xe1\xbc\xa0\xce\xb9\xe1\xbe\x98\xe1\xbe\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x90\xe1\xbe\x99\xe1\xbc\xa1\xce\xb9\xe1\xbe\x99\xe1\xbe\x91\xe1\xbe\x9a\xe1\xbc\xa2\xce\xb9\xe1\xbe\x9a\xe1\xbe\x92\xe1\xbe\x9b", "\xe1\xbe\x98\xe1\xbc\xa1\xce\xb9\xe1\xbe\x99\xe1\xbe\x91\xe1\xbe\x99\xe1\xbc\xa2\xce\xb9\xe1\xbe\x9a\xe1\xbe\x92\xe1\xbe\x9a\xe1\xbc\xa3\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbc\xa3\xce\xb9\xe1\xbe\x9b\xe1\xbe\x93\xe1\xbe\x9c\xe1\xbc\xa4\xce\xb9\xe1\xbe\x9c\xe1\xbe\x94\xe1\xbe\x9d\xe1\xbc\xa5\xce\xb9\xe1\xbe\x9d", "\xe1\xbe\x9b\xe1\xbe\x93\xe1\xbe\x9b\xe1\xbc\xa4\xce\xb9\xe1\xbe\x9c\xe1\xbe\x94\xe1\xbe\x9c\xe1\xbc\xa5\xce\xb9\xe1\xbe\x9d\xe1\xbe\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\x95\xe1\xbe\x9e\xe1\xbc\xa6\xce\xb9\xe1\xbe\x9e\xe1\xbe\x96\xe1\xbe\x9f\xe1\xbc\xa7\xce\xb9\xe1\xbe\x9f\xe1\xbe\x97\xe1\xbe\xa0", "\xe1\xbe\x9d\xe1\xbc\xa6\xce\xb9\xe1\xbe\x9e\xe1\xbe\x96\xe1\xbe\x9e\xe1\xbc\xa7\xce\xb9\xe1\xbe\x9f\xe1\xbe\x97\xe1\xbe\x9f\xe1\xbd\xa0\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa0\xce\xb9\xe1\xbe\xa1\xe1\xbd\xa1\xce\xb9\xe1\xbe\xa2\xe1\xbd\xa2\xce\xb9\xe1\xbe\xa3\xe1\xbd\xa3\xce\xb9\xe1\xbe\xa4\xe1\xbd\xa4\xce\xb9\xe1\xbe\xa5", "\xe1\xbe\xa0\xe1\xbd\xa1\xce\xb9\xe1\xbe\xa1\xe1\xbd\xa2\xce\xb9\xe1\xbe\xa2\xe1\xbd\xa3\xce\xb9\xe1\xbe\xa3\xe1\xbd\xa4\xce\xb9\xe1\xbe\xa4\xe1\xbd\xa5\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa5\xce\xb9\xe1\xbe\xa6\xe1\xbd\xa6\xce\xb9\xe1\xbe\xa7\xe1\xbd\xa7\xce\xb9\xe1\xbe\xa8\xe1\xbd\xa0\xce\xb9\xe1\xbe\xa8\xe1\xbe\xa0\xe1\xbe\xa9", "\xe1\xbe\xa5\xe1\xbd\xa6\xce\xb9\xe1\xbe\xa6\xe1\xbd\xa7\xce\xb9\xe1\xbe\xa7\xe1\xbd\xa0\xce\xb9\xe1\xbe\xa8\xe1\xbe\xa0\xe1\xbe\xa8\xe1\xbd\xa1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa1\xce\xb9\xe1\xbe\xa9\xe1\xbe\xa1\xe1\xbe\xaa\xe1\xbd\xa2\xce\xb9\xe1\xbe\xaa\xe1\xbe\xa2\xe1\xbe\xab\xe1\xbd\xa3\xce\xb9\xe1\xbe\xab", "\xe1\xbe\xa9\xe1\xbe\xa1\xe1\xbe\xa9\xe1\xbd\xa2\xce\xb9\xe1\xbe\xaa\xe1\xbe\xa2\xe1\xbe\xaa\xe1\xbd\xa3\xce\xb9\xe1\xbe\xab\xe1\xbe\xa3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xa3\xe1\xbe\xac\xe1\xbd\xa4\xce\xb9\xe1\xbe\xac\xe1\xbe\xa4\xe1\xbe\xad\xe1\xbd\xa5\xce\xb9\xe1\xbe\xad\xe1\xbe\xa5\xe1\xbe\xae", "\xe1\xbe\xab\xe1\xbd\xa4\xce\xb9\xe1\xbe\xac\xe1\xbe\xa4\xe1\xbe\xac\xe1\xbd\xa5\xce\xb9\xe1\xbe\xad\xe1\xbe\xa5\xe1\xbe\xad\xe1\xbd\xa6\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xa6\xce\xb9\xe1\xbe\xae\xe1\xbe\xa6\xe1\xbe\xaf\xe1\xbd\xa7\xce\xb9\xe1\xbe\xaf\xe1\xbe\xa7\xe1\xbe\xb2\xe1\xbd\xb0\xce\xb9\xe1\xbe\xb3", "\xe1\xbe\xae\xe1\xbe\xa6\xe1\xbe\xae\xe1\xbd\xa7\xce\xb9\xe1\xbe\xaf\xe1\xbe\xa7\xe1\xbe\xaf\xe1\xbd\xb0\xce\xb9\xe1\xbe\xb2\xce\xb1\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb1\xce\xb9\xe1\xbe\xb4\xce\xac\xce\xb9\xe1\xbe\xb6\xce\xb1\xcd\x82\xe1\xbe\xb7\xce\xb1\xcd\x82\xce\xb9\xe1\xbe\xb8\xe1\xbe\xb0\xe1\xbe\xb9", "\xe1\xbe\xb3\xce\xac\xce\xb9\xe1\xbe\xb4\xce\xb1\xcd\x82\xe1\xbe\xb6\xce\xb1\xcd\x82\xce\xb9\xe1\xbe\xb7\xe1\xbe\xb0\xe1\xbe\xb8\xe1\xbe\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbe\xb1\xe1\xbe\xba\xe1\xbd\xb0\xe1\xbe\xbb\xe1\xbd\xb1\xe1\xbe\xbc\xce\xb1\xce\xb9\xe1\xbe\xbc\xe1\xbe\xb3\xe1\xbe\xbe", "\xe1\xbe\xb9\xe1\xbd\xb0\xe1\xbe\xba\xe1\xbd\xb1\xe1\xbe\xbb\xce\xb1\xce\xb9\xe1\xbe\xbc\xe1\xbe\xb3\xe1\xbe\xbc\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xe1\xbf\x82\xe1\xbd\xb4\xce\xb9\xe1\xbf\x83\xce\xb7\xce\xb9\xe1\xbf\x84\xce\xae\xce\xb9\xe1\xbf\x86\xce\xb7\xcd\x82\xe1\xbf\x87", "\xe1\xbe\xbe\xe1\xbd\xb4\xce\xb9\xe1\xbf\x82\xce\xb7\xce\xb9\xe1\xbf\x83\xce\xae\xce\xb9\xe1\xbf\x84\xce\xb7\xcd\x82\xe1\xbf\x86\xce\xb7\xcd\x82\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb7\xcd\x82\xce\xb9\xe1\xbf\x88\xe1\xbd\xb2\xe1\xbf\x89\xe1\xbd\xb3\xe1\xbf\x8a\xe1\xbd\xb4\xe1\xbf\x8b\xe1\xbd\xb5\xe1\xbf\x8c", "\xe1\xbf\x87\xe1\xbd\xb2\xe1\xbf\x88\xe1\xbd\xb3\xe1\xbf\x89\xe1\xbd\xb4\xe1\xbf\x8a\xe1\xbd\xb5\xe1\xbf\x8b\xce\xb7\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb7\xce\xb9\xe1\xbf\x8c\xe1\xbf\x83\xe1\xbf\x92\xce\xb9\xcc\x88\xcc\x80\xe1\xbf\x93\xce\xb9\xcc\x88\xcc\x81\xe1\xbf\x96\xce\xb9\xcd\x82\xe1\xbf\x97", "\xe1\xbf\x8c\xe1\xbf\x83\xe1\xbf\x8c\xce\xb9\xcc\x88\xcc\x80\xe1\xbf\x92\xce\xb9\xcc\x88\xcc\x81\xe1\xbf\x93\xce\xb9\xcd\x82\xe1\xbf\x96\xce\xb9\xcc\x88\xcd\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xce\xb9\xcc\x88\xcd\x82\xe1\xbf\x98\xe1\xbf\x90\xe1\xbf\x99\xe1\xbf\x91\xe1\xbf\x9a\xe1\xbd\xb6\xe1\xbf\x9b\xe1\xbd\xb7\xe1\xbf\xa2", "\xe1\xbf\x97\xe1\xbf\x90\xe1\xbf\x98\xe1\xbf\x91\xe1\xbf\x99\xe1\xbd\xb6\xe1\xbf\x9a\xe1\xbd\xb7\xe1\xbf\x9b\xcf\x85\xcc\x88\xcc\x80") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xcf\x85\xcc\x88\xcc\x80\xe1\xbf\xa3\xcf\x85\xcc\x88\xcc\x81\xe1\xbf\xa4\xcf\x81\xcc\x93\xe1\xbf\xa6\xcf\x85\xcd\x82\xe1\xbf\xa7\xcf\x85\xcc\x88\xcd\x82\xe1\xbf\xa8", "\xe1\xbf\xa2\xcf\x85\xcc\x88\xcc\x81\xe1\xbf\xa3\xcf\x81\xcc\x93\xe1\xbf\xa4\xcf\x85\xcd\x82\xe1\xbf\xa6\xcf\x85\xcc\x88\xcd\x82\xe1\xbf\xa7\xe1\xbf\xa0") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xa0\xe1\xbf\xa9\xe1\xbf\xa1\xe1\xbf\xaa\xe1\xbd\xba\xe1\xbf\xab\xe1\xbd\xbb\xe1\xbf\xac\xe1\xbf\xa5\xe1\xbf\xb2", "\xe1\xbf\xa8\xe1\xbf\xa1\xe1\xbf\xa9\xe1\xbd\xba\xe1\xbf\xaa\xe1\xbd\xbb\xe1\xbf\xab\xe1\xbf\xa5\xe1\xbf\xac\xe1\xbd\xbc\xce\xb9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xbc\xce\xb9\xe1\xbf\xb3\xcf\x89\xce\xb9\xe1\xbf\xb4\xcf\x8e\xce\xb9\xe1\xbf\xb6\xcf\x89\xcd\x82\xe1\xbf\xb7\xcf\x89\xcd\x82\xce\xb9\xe1\xbf\xb8", "\xe1\xbf\xb2\xcf\x89\xce\xb9\xe1\xbf\xb3\xcf\x8e\xce\xb9\xe1\xbf\xb4\xcf\x89\xcd\x82\xe1\xbf\xb6\xcf\x89\xcd\x82\xce\xb9\xe1\xbf\xb7\xe1\xbd\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbd\xb8\xe1\xbf\xb9\xe1\xbd\xb9\xe1\xbf\xba\xe1\xbd\xbc\xe1\xbf\xbb\xe1\xbd\xbd\xe1\xbf\xbc\xcf\x89\xce\xb9\xe1\xbf\xbc", "\xe1\xbf\xb8\xe1\xbd\xb9\xe1\xbf\xb9\xe1\xbd\xbc\xe1\xbf\xba\xe1\xbd\xbd\xe1\xbf\xbb\xcf\x89\xce\xb9\xe1\xbf\xbc\xe1\xbf\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe1\xbf\xb3\xe2\x84\xa6\xcf\x89\xe2\x84\xaa\x6b\xe2\x84\xab\xc3\xa5\xe2\x85\xa0\xe2\x85\xb0\xe2\x85\xa1", "\xe1\xbf\xbc\xcf\x89\xe2\x84\xa6\x6b\xe2\x84\xaa\xc3\xa5\xe2\x84\xab\xe2\x85\xb0\xe2\x85\xa0\xe2\x85\xb1") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb1\xe2\x85\xa2\xe2\x85\xb2\xe2\x85\xa3\xe2\x85\xb3\xe2\x85\xa4\xe2\x85\xb4\xe2\x85\xa5\xe2\x85\xb5\xe2\x85\xa6", "\xe2\x85\xa1\xe2\x85\xb2\xe2\x85\xa2\xe2\x85\xb3\xe2\x85\xa3\xe2\x85\xb4\xe2\x85\xa4\xe2\x85\xb5\xe2\x85\xa5\xe2\x85\xb6") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xb6\xe2\x85\xa7\xe2\x85\xb7\xe2\x85\xa8\xe2\x85\xb8\xe2\x85\xa9\xe2\x85\xb9\xe2\x85\xaa\xe2\x85\xba\xe2\x85\xab", "\xe2\x85\xa6\xe2\x85\xb7\xe2\x85\xa7\xe2\x85\xb8\xe2\x85\xa8\xe2\x85\xb9\xe2\x85\xa9\xe2\x85\xba\xe2\x85\xaa\xe2\x85\xbb") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x85\xbb\xe2\x85\xac\xe2\x85\xbc\xe2\x85\xad\xe2\x85\xbd\xe2\x85\xae\xe2\x85\xbe\xe2\x85\xaf\xe2\x85\xbf\xe2\x92\xb6", "\xe2\x85\xab\xe2\x85\xbc\xe2\x85\xac\xe2\x85\xbd\xe2\x85\xad\xe2\x85\xbe\xe2\x85\xae\xe2\x85\xbf\xe2\x85\xaf\xe2\x93\x90") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x90\xe2\x92\xb7\xe2\x93\x91\xe2\x92\xb8\xe2\x93\x92\xe2\x92\xb9\xe2\x93\x93\xe2\x92\xba\xe2\x93\x94\xe2\x92\xbb", "\xe2\x92\xb6\xe2\x93\x91\xe2\x92\xb7\xe2\x93\x92\xe2\x92\xb8\xe2\x93\x93\xe2\x92\xb9\xe2\x93\x94\xe2\x92\xba\xe2\x93\x95") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x95\xe2\x92\xbc\xe2\x93\x96\xe2\x92\xbd\xe2\x93\x97\xe2\x92\xbe\xe2\x93\x98\xe2\x92\xbf\xe2\x93\x99\xe2\x93\x80", "\xe2\x92\xbb\xe2\x93\x96\xe2\x92\xbc\xe2\x93\x97\xe2\x92\xbd\xe2\x93\x98\xe2\x92\xbe\xe2\x93\x99\xe2\x92\xbf\xe2\x93\x9a") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9a\xe2\x93\x81\xe2\x93\x9b\xe2\x93\x82\xe2\x93\x9c\xe2\x93\x83\xe2\x93\x9d\xe2\x93\x84\xe2\x93\x9e\xe2\x93\x85", "\xe2\x93\x80\xe2\x93\x9b\xe2\x93\x81\xe2\x93\x9c\xe2\x93\x82\xe2\x93\x9d\xe2\x93\x83\xe2\x93\x9e\xe2\x93\x84\xe2\x93\x9f") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\x9f\xe2\x93\x86\xe2\x93\xa0\xe2\x93\x87\xe2\x93\xa1\xe2\x93\x88\xe2\x93\xa2\xe2\x93\x89\xe2\x93\xa3\xe2\x93\x8a", "\xe2\x93\x85\xe2\x93\xa0\xe2\x93\x86\xe2\x93\xa1\xe2\x93\x87\xe2\x93\xa2\xe2\x93\x88\xe2\x93\xa3\xe2\x93\x89\xe2\x93\xa4") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa4\xe2\x93\x8b\xe2\x93\xa5\xe2\x93\x8c\xe2\x93\xa6\xe2\x93\x8d\xe2\x93\xa7\xe2\x93\x8e\xe2\x93\xa8\xe2\x93\x8f", "\xe2\x93\x8a\xe2\x93\xa5\xe2\x93\x8b\xe2\x93\xa6\xe2\x93\x8c\xe2\x93\xa7\xe2\x93\x8d\xe2\x93\xa8\xe2\x93\x8e\xe2\x93\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xe2\x93\xa9\xef\xac\x80\x66\x66\xef\xac\x81\x66\x69\xef\xac\x82\x66\x6c\xef\xac\x83\x66\x66\x69\xef\xac\x84", "\xe2\x93\x8f\x66\x66\xef\xac\x80\x66\x69\xef\xac\x81\x66\x6c\xef\xac\x82\x66\x66\x69\xef\xac\x83\x66\x66\x6c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\x66\x66\x6c\xef\xac\x85\x73\x74\xef\xac\x86\x73\x74\xef\xac\x93\xd5\xb4\xd5\xb6\xef\xac\x94\xd5\xb4\xd5\xa5\xef\xac\x95", "\xef\xac\x84\x73\x74\xef\xac\x85\x73\x74\xef\xac\x86\xd5\xb4\xd5\xb6\xef\xac\x93\xd5\xb4\xd5\xa5\xef\xac\x94\xd5\xb4\xd5\xab") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xd5\xb4\xd5\xab\xef\xac\x96\xd5\xbe\xd5\xb6\xef\xac\x97\xd5\xb4\xd5\xad\xef\xbc\xa1\xef\xbd\x81\xef\xbc\xa2\xef\xbd\x82\xef\xbc\xa3", "\xef\xac\x95\xd5\xbe\xd5\xb6\xef\xac\x96\xd5\xb4\xd5\xad\xef\xac\x97\xef\xbd\x81\xef\xbc\xa1\xef\xbd\x82\xef\xbc\xa2\xef\xbd\x83") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x83\xef\xbc\xa4\xef\xbd\x84\xef\xbc\xa5\xef\xbd\x85\xef\xbc\xa6\xef\xbd\x86\xef\xbc\xa7\xef\xbd\x87\xef\xbc\xa8", "\xef\xbc\xa3\xef\xbd\x84\xef\xbc\xa4\xef\xbd\x85\xef\xbc\xa5\xef\xbd\x86\xef\xbc\xa6\xef\xbd\x87\xef\xbc\xa7\xef\xbd\x88") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x88\xef\xbc\xa9\xef\xbd\x89\xef\xbc\xaa\xef\xbd\x8a\xef\xbc\xab\xef\xbd\x8b\xef\xbc\xac\xef\xbd\x8c\xef\xbc\xad", "\xef\xbc\xa8\xef\xbd\x89\xef\xbc\xa9\xef\xbd\x8a\xef\xbc\xaa\xef\xbd\x8b\xef\xbc\xab\xef\xbd\x8c\xef\xbc\xac\xef\xbd\x8d") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x8d\xef\xbc\xae\xef\xbd\x8e\xef\xbc\xaf\xef\xbd\x8f\xef\xbc\xb0\xef\xbd\x90\xef\xbc\xb1\xef\xbd\x91\xef\xbc\xb2", "\xef\xbc\xad\xef\xbd\x8e\xef\xbc\xae\xef\xbd\x8f\xef\xbc\xaf\xef\xbd\x90\xef\xbc\xb0\xef\xbd\x91\xef\xbc\xb1\xef\xbd\x92") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x92\xef\xbc\xb3\xef\xbd\x93\xef\xbc\xb4\xef\xbd\x94\xef\xbc\xb5\xef\xbd\x95\xef\xbc\xb6\xef\xbd\x96\xef\xbc\xb7", "\xef\xbc\xb2\xef\xbd\x93\xef\xbc\xb3\xef\xbd\x94\xef\xbc\xb4\xef\xbd\x95\xef\xbc\xb5\xef\xbd\x96\xef\xbc\xb6\xef\xbd\x97") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xef\xbd\x97\xef\xbc\xb8\xef\xbd\x98\xef\xbc\xb9\xef\xbd\x99\xef\xbc\xba\xef\xbd\x9a\xf0\x90\x90\x80\xf0\x90\x90\xa8\xf0\x90\x90\x81", "\xef\xbc\xb7\xef\xbd\x98\xef\xbc\xb8\xef\xbd\x99\xef\xbc\xb9\xef\xbd\x9a\xef\xbc\xba\xf0\x90\x90\xa8\xf0\x90\x90\x80\xf0\x90\x90\xa9") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xa9\xf0\x90\x90\x82\xf0\x90\x90\xaa\xf0\x90\x90\x83\xf0\x90\x90\xab\xf0\x90\x90\x84\xf0\x90\x90\xac\xf0\x90\x90\x85\xf0\x90\x90\xad\xf0\x90\x90\x86", "\xf0\x90\x90\x81\xf0\x90\x90\xaa\xf0\x90\x90\x82\xf0\x90\x90\xab\xf0\x90\x90\x83\xf0\x90\x90\xac\xf0\x90\x90\x84\xf0\x90\x90\xad\xf0\x90\x90\x85\xf0\x90\x90\xae") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xae\xf0\x90\x90\x87\xf0\x90\x90\xaf\xf0\x90\x90\x88\xf0\x90\x90\xb0\xf0\x90\x90\x89\xf0\x90\x90\xb1\xf0\x90\x90\x8a\xf0\x90\x90\xb2\xf0\x90\x90\x8b", "\xf0\x90\x90\x86\xf0\x90\x90\xaf\xf0\x90\x90\x87\xf0\x90\x90\xb0\xf0\x90\x90\x88\xf0\x90\x90\xb1\xf0\x90\x90\x89\xf0\x90\x90\xb2\xf0\x90\x90\x8a\xf0\x90\x90\xb3") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb3\xf0\x90\x90\x8c\xf0\x90\x90\xb4\xf0\x90\x90\x8d\xf0\x90\x90\xb5\xf0\x90\x90\x8e\xf0\x90\x90\xb6\xf0\x90\x90\x8f\xf0\x90\x90\xb7\xf0\x90\x90\x90", "\xf0\x90\x90\x8b\xf0\x90\x90\xb4\xf0\x90\x90\x8c\xf0\x90\x90\xb5\xf0\x90\x90\x8d\xf0\x90\x90\xb6\xf0\x90\x90\x8e\xf0\x90\x90\xb7\xf0\x90\x90\x8f\xf0\x90\x90\xb8") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xb8\xf0\x90\x90\x91\xf0\x90\x90\xb9\xf0\x90\x90\x92\xf0\x90\x90\xba\xf0\x90\x90\x93\xf0\x90\x90\xbb\xf0\x90\x90\x94\xf0\x90\x90\xbc\xf0\x90\x90\x95", "\xf0\x90\x90\x90\xf0\x90\x90\xb9\xf0\x90\x90\x91\xf0\x90\x90\xba\xf0\x90\x90\x92\xf0\x90\x90\xbb\xf0\x90\x90\x93\xf0\x90\x90\xbc\xf0\x90\x90\x94\xf0\x90\x90\xbd") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x90\xbd\xf0\x90\x90\x96\xf0\x90\x90\xbe\xf0\x90\x90\x97\xf0\x90\x90\xbf\xf0\x90\x90\x98\xf0\x90\x91\x80\xf0\x90\x90\x99\xf0\x90\x91\x81\xf0\x90\x90\x9a", "\xf0\x90\x90\x95\xf0\x90\x90\xbe\xf0\x90\x90\x96\xf0\x90\x90\xbf\xf0\x90\x90\x97\xf0\x90\x91\x80\xf0\x90\x90\x98\xf0\x90\x91\x81\xf0\x90\x90\x99\xf0\x90\x91\x82") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x82\xf0\x90\x90\x9b\xf0\x90\x91\x83\xf0\x90\x90\x9c\xf0\x90\x91\x84\xf0\x90\x90\x9d\xf0\x90\x91\x85\xf0\x90\x90\x9e\xf0\x90\x91\x86\xf0\x90\x90\x9f", "\xf0\x90\x90\x9a\xf0\x90\x91\x83\xf0\x90\x90\x9b\xf0\x90\x91\x84\xf0\x90\x90\x9c\xf0\x90\x91\x85\xf0\x90\x90\x9d\xf0\x90\x91\x86\xf0\x90\x90\x9e\xf0\x90\x91\x87") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x87\xf0\x90\x90\xa0\xf0\x90\x91\x88\xf0\x90\x90\xa1\xf0\x90\x91\x89\xf0\x90\x90\xa2\xf0\x90\x91\x8a\xf0\x90\x90\xa3\xf0\x90\x91\x8b\xf0\x90\x90\xa4", "\xf0\x90\x90\x9f\xf0\x90\x91\x88\xf0\x90\x90\xa0\xf0\x90\x91\x89\xf0\x90\x90\xa1\xf0\x90\x91\x8a\xf0\x90\x90\xa2\xf0\x90\x91\x8b\xf0\x90\x90\xa3\xf0\x90\x91\x8c") == 0)
	unitAssert(UTF8Utils::compareToIgnoreCase("\xf0\x90\x91\x8c\xf0\x90\x90\xa5\xf0\x90\x91\x8d\xf0\x90\x90\xa6\xf0\x90\x91\x8e\xf0\x90\x90\xa7\xf0\x90\x91\x8f", "\xf0\x90\x90\xa4\xf0\x90\x91\x8d\xf0\x90\x90\xa5\xf0\x90\x91\x8e\xf0\x90\x90\xa6\xf0\x90\x91\x8f\xf0\x90\x90\xa7") == 0)
#endif
}

void OW_UTF8UtilsTestCases::testToLowerCaseInPlace()
{
	char buf[1024];
	strcpy(buf, "ab");
	unitAssert(UTF8Utils::toLowerCaseInPlace(buf));
	unitAssert(String(buf) == "ab");
	strcpy(buf, "AbC");
	unitAssert(UTF8Utils::toLowerCaseInPlace(buf));
	unitAssert(String(buf) == "abc");

	// invalid utf8 chars
	strcpy(buf, "\xFF AbC\xFE\xFD D \xFC");
	unitAssert(UTF8Utils::toLowerCaseInPlace(buf));
	unitAssert(String(buf) == "\xFF abc\xFE\xFD d \xFC");

	// differing length	- these should all work.  With Unicode 4.0.1, there
	// are *no* utf8 sequences that get longer with lower-casing.
	strcpy(buf, "\xc4\xb0");
	unitAssert(UTF8Utils::toLowerCaseInPlace(buf));
	unitAssert(String(buf) == "\x69");
	
	strcpy(buf, "\xe2\x84\xa6");
	unitAssert(UTF8Utils::toLowerCaseInPlace(buf));
	unitAssert(String(buf) == "\xcf\x89");

	strcpy(buf, "\xe2\x84\xaa");
	unitAssert(UTF8Utils::toLowerCaseInPlace(buf));
	unitAssert(String(buf) == "\x6b");

	// 2, 3, 4 char lengths
	strcpy(buf, "\xc3\x80 A \xc4\x80 b C\xe1\xb8\x80 \xf0\x90\x90\x94");
	unitAssert(UTF8Utils::toLowerCaseInPlace(buf));
	unitAssert(String(buf) == "\xc3\xa0 a \xc4\x81 b c\xe1\xb8\x81 \xf0\x90\x90\xbc");

}

void OW_UTF8UtilsTestCases::testToUpperCaseInPlace()
{
	char buf[1024];
	strcpy(buf, "AB");
	unitAssert(UTF8Utils::toUpperCaseInPlace(buf));
	unitAssert(String(buf) == "AB");
	strcpy(buf, "AbC");
	unitAssert(UTF8Utils::toUpperCaseInPlace(buf));
	unitAssert(String(buf) == "ABC");

	// invalid utf8 chars
	strcpy(buf, "\xFF AbC\xFE\xFD d \xFC");
	unitAssert(UTF8Utils::toUpperCaseInPlace(buf));
	unitAssert(String(buf) == "\xFF ABC\xFE\xFD D \xFC");

	// differing length	- these should all work.  With Unicode 4.0.1, there
	// are *no* utf8 sequences that get longer with upper-casing.
	strcpy(buf, "\xc4\xb1");
	unitAssert(UTF8Utils::toUpperCaseInPlace(buf));
	unitAssert(String(buf) == "\x49");
	
	strcpy(buf, "\xe1\xbe\xbe");
	unitAssert(UTF8Utils::toUpperCaseInPlace(buf));
	unitAssert(String(buf) == "\xce\x99");

	// 2, 3, 4 char lengths
	strcpy(buf, "\xd4\x81 a \xc4\x81 b c\xe1\xb8\x81 \xf0\x90\x90\xbc");
	unitAssert(UTF8Utils::toUpperCaseInPlace(buf));
	unitAssert(String(buf) == "\xd4\x80 A \xc4\x80 B C\xe1\xb8\x80 \xf0\x90\x90\x94");
}

void OW_UTF8UtilsTestCases::testToLowerCase()
{
	unitAssert(UTF8Utils::toLowerCase("ab") == "ab");
	unitAssert(UTF8Utils::toLowerCase("AbC") == "abc");

	// invalid utf8 chars
	unitAssert(UTF8Utils::toLowerCase("\xFF AbC\xFE\xFD D \xFC") == "\xFF abc\xFE\xFD d \xFC");

	// differing length	- these should all work.  With Unicode 4.0.1, there
	// are *no* utf8 sequences that get longer with lower-casing.
	unitAssert(UTF8Utils::toLowerCase("\xc4\xb0") == "\x69");
	
	unitAssert(UTF8Utils::toLowerCase("\xe2\x84\xa6") == "\xcf\x89");

	unitAssert(UTF8Utils::toLowerCase("\xe2\x84\xaa") == "\x6b");

	// 2, 3, 4 char lengths
	unitAssert(UTF8Utils::toLowerCase("\xc3\x80 A \xc4\x80 b C\xe1\xb8\x80 \xf0\x90\x90\x94") == "\xc3\xa0 a \xc4\x81 b c\xe1\xb8\x81 \xf0\x90\x90\xbc");
}

void OW_UTF8UtilsTestCases::testToUpperCase()
{
	unitAssert(UTF8Utils::toUpperCase("AB") == "AB");
	unitAssert(UTF8Utils::toUpperCase("AbC") == "ABC");

	// invalid utf8 chars
	unitAssert(UTF8Utils::toUpperCase("\xFF AbC\xFE\xFD d \xFC") == "\xFF ABC\xFE\xFD D \xFC");

	// differing length	- these should all work.  With Unicode 4.0.1, there
	// are *no* utf8 sequences that get longer with upper-casing.
	unitAssert(UTF8Utils::toUpperCase("\xc4\xb1") == "\x49");
	
	unitAssert(UTF8Utils::toUpperCase("\xe1\xbe\xbe") == "\xce\x99");

	// 2, 3, 4 char lengths
	unitAssert(UTF8Utils::toUpperCase("\xd4\x81 a \xc4\x81 b c\xe1\xb8\x81 \xf0\x90\x90\xbc") == "\xd4\x80 A \xc4\x80 B C\xe1\xb8\x80 \xf0\x90\x90\x94");
}

Test* OW_UTF8UtilsTestCases::suite()
{
	TestSuite *testSuite = new TestSuite ("OW_UTF8Utils");

	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testCharCount);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testUTF8toUCS2);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testUCS2toUTF8);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testUTF8toUCS4);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testUCS4toUTF8);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testcompareToIgnoreCase);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testToLowerCaseInPlace);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testToUpperCaseInPlace);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testToLowerCase);
	ADD_TEST_TO_SUITE(OW_UTF8UtilsTestCases, testToUpperCase);

	return testSuite;
}

