/*
 * cmap.c - CMap table manipulation.
 *
    Copyright (C) 2001  Yao Zhang

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * yaoz@users.sourceforge.net
 */
#include <iconv.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ot.h"

CMapTable *new_cmap_table(void)
{
    CMapTable *cmapTable;

    cmapTable = (CMapTable *)calloc(1, sizeof(CMapTable));

    return cmapTable;
}

void delete_cmap_table(CMapTable *cmapTable)
{
    int i, n;

    n = cmapTable->cmapIndex.numberSubtables;
    for (i = 0; i < n; i++) {
        if (cmapTable->cmapArray[i].cmapSubtable.u2c != NULL) {
            u2c_close(cmapTable->cmapArray[i].cmapSubtable.u2c);
        }
        switch (cmapTable->cmapArray[i].cmapSubtable.format.f0.format) {
        case 2:
            if (cmapTable->cmapArray[i].cmapSubtable.format.f2.subHeaders != NULL) {
                free(cmapTable->cmapArray[i].cmapSubtable.format.f2.subHeaders);
            }
            break;
        default:
            break;
        }
    }
    free(cmapTable->cmapArray);

    if (cmapTable->buffer != NULL) {
        free(cmapTable->buffer);
    }
    free(cmapTable);
}

void read_cmap_table(char *file, FontDirectory *fontDirectory,
                           CMapTable *cmapTable)
{
    register uint8_t *p;
    int i, n;
    TableDirectoryEntry *tableDirectoryEntry;

    /* Find the 'cmap' directory */
    n = fontDirectory->offsetSubtable.numTables;
    for (i = 0; i < n; i++) {
        tableDirectoryEntry = &fontDirectory->tableDirectory[i];

        if (strncmp(tableDirectoryEntry->tag, CMAP_TAG, 4) == 0) {
            break;
        }
    }

    if (i >= n) {
        return;
    }

    /* Read from file to memory */
    cmapTable->length = tableDirectoryEntry->length;
    cmapTable->buffer = (uint8_t *)calloc(cmapTable->length, sizeof(uint8_t));
    read_table(cmapTable->buffer, cmapTable->length,
               file, tableDirectoryEntry->offset);

    /* CMap Index */
    p = cmapTable->buffer;
    cmapTable->cmapIndex.version = GET2B(p);
    cmapTable->cmapIndex.numberSubtables = GET2B(p);

    n = cmapTable->cmapIndex.numberSubtables;

    cmapTable->cmapArray = (CMapArray *)calloc(n, sizeof(CMapArray));

    /* CMap Encoding Subtables */
    for (i = 0; i < n; i++) {
        cmapTable->cmapArray[i].cmapEncodingSubtable.platformID = GET2B(p);
        cmapTable->cmapArray[i].cmapEncodingSubtable.platformSpecificID = GET2B(p);
        cmapTable->cmapArray[i].cmapEncodingSubtable.offset = GET4B(p);
    }

    /* CMap Subtables */
    for (i = 0; i < n; i++) {
        int k, count;
        uint16_t format;
        CMapSubtable *cmapSubtable;

        p = cmapTable->buffer;
        p += cmapTable->cmapArray[i].cmapEncodingSubtable.offset;

        cmapSubtable = &cmapTable->cmapArray[i].cmapSubtable;
        format = GET2B(p);
        switch (format) {
        case 0:
            cmapSubtable->format.f0.format = format;
            cmapSubtable->format.f0.length = GET2B(p);
            cmapSubtable->format.f0.language = GET2B(p);
            break;
        case 2:
            cmapSubtable->format.f2.format = format;
            cmapSubtable->format.f2.length = GET2B(p);
            cmapSubtable->format.f2.language = GET2B(p);
            count = 0;
            for (k = 0; k < 256; k++) {
                cmapSubtable->format.f2.subHeaderKeys[k] = GET2B(p);
                if (cmapSubtable->format.f2.subHeaderKeys[k]/8 > count) {
                    count = cmapSubtable->format.f2.subHeaderKeys[k]/8;
                }
            }
            count++;
            cmapSubtable->format.f2.subHeaders
                    = (F2SubHeader *)calloc(count, sizeof(F2SubHeader));
            for (k = 0; k < count; k++) {
                cmapSubtable->format.f2.subHeaders[k].firstCode = GET2B(p);
                cmapSubtable->format.f2.subHeaders[k].entryCount = GET2B(p);
                cmapSubtable->format.f2.subHeaders[k].idDelta = GET2B(p);
                cmapSubtable->format.f2.subHeaders[k].idRangeOffset = GET2B(p);
            }
            if ((cmapTable->cmapArray[i].cmapEncodingSubtable.platformID == 3)
             && (cmapTable->cmapArray[i].cmapEncodingSubtable.platformSpecificID == 3)) {
                U2C *u2c;
                char answer[8];

                fprintf(stderr, "The font has a PRC GB encoding mapping table.\n");
                fprintf(stderr, "It can be either GB2312 or GB12345.\n");
                fprintf(stderr, "Only if you know that the glyphs are in\n");
                fprintf(stderr, "TRADITIONAL Chinese form, choose GB12345.\n");
                fprintf(stderr, "Are you sure you want to use GB12345?(y/n): ");
                fgets(answer, 8, stdin);
                if ((answer[0] == 'y') || (answer[0] == 'Y')) {
                    u2c = u2c_open(GB12345);
                } else {
                    u2c = u2c_open(GB2312);
                }
                for (k = 0; k < u2c->length; k++) {
                    uint32_t gb;

                    gb = u2c->array[k].c;
                    u2c->array[k].c = f2_glyph_index((gb>>8)&0xFF, gb&0xFF,
                                                     cmapTable, i);
                }
                cmapTable->cmapArray[i].cmapSubtable.u2c = u2c;
            }
            else if ((cmapTable->cmapArray[i].cmapEncodingSubtable.platformID == 3)
             && (cmapTable->cmapArray[i].cmapEncodingSubtable.platformSpecificID == 4)) {
                U2C *u2c;
                char answer[8];

                u2c = u2c_open(BIG5);
                
                for (k = 0; k < u2c->length; k++) {
                    uint32_t big5;

                    big5 = u2c->array[k].c;
                    u2c->array[k].c = f2_glyph_index((big5>>8)&0xFF, big5&0xFF,
                                                     cmapTable, i);
                }
                cmapTable->cmapArray[i].cmapSubtable.u2c = u2c;
            }
            break;
        case 4:
            cmapSubtable->format.f4.format = format;
            cmapSubtable->format.f4.length = GET2B(p);
            cmapSubtable->format.f4.language = GET2B(p);
            cmapSubtable->format.f4.segCountX2 = GET2B(p);
            cmapSubtable->format.f4.searchRange = GET2B(p);
            cmapSubtable->format.f4.entrySelector = GET2B(p);
            cmapSubtable->format.f4.rangeShift = GET2B(p);
            break;
        case 6:
            cmapSubtable->format.f6.format = format;
            cmapSubtable->format.f6.length = GET2B(p);
            cmapSubtable->format.f6.language = GET2B(p);
            cmapSubtable->format.f6.firstCode = GET2B(p);
            cmapSubtable->format.f6.entryCount = GET2B(p);
            break;
        case 8:
            cmapSubtable->format.f8.format = format;
            cmapSubtable->format.f8.reserved = GET2B(p);
            cmapSubtable->format.f8.length = GET4B(p);
            cmapSubtable->format.f8.language = GET4B(p);
            break;
        case 10:
            cmapSubtable->format.f10.format = format;
            cmapSubtable->format.f10.reserved = GET2B(p);
            cmapSubtable->format.f10.length = GET4B(p);
            cmapSubtable->format.f10.language = GET4B(p);
            cmapSubtable->format.f10.startCharCode = GET4B(p);
            cmapSubtable->format.f10.numChars = GET4B(p);
            break;
        case 12:
            cmapSubtable->format.f12.format = format;
            cmapSubtable->format.f12.reserved = GET2B(p);
            cmapSubtable->format.f12.length = GET4B(p);
            cmapSubtable->format.f12.language = GET4B(p);
            cmapSubtable->format.f12.nGroups = GET4B(p);
            break;
        default:
            break;
        }
    }

    return;
}

void print_cmap_index(CMapIndex *cmapIndex)
{
    printf("  Version: %u, ", cmapIndex->version);
    printf("Number of encoding subtables: %u.\n", cmapIndex->numberSubtables);
}

void print_cmap_encoding_and_subtable(CMapArray *cmapArrayEntry)
{
    CMapEncodingSubtable *cmapEncodingSubtable;
    CMapSubtable *cmapSubtable;
    int length, language;

    cmapEncodingSubtable = &cmapArrayEntry->cmapEncodingSubtable;
    cmapSubtable = &cmapArrayEntry->cmapSubtable;
    length = language = -1;
    switch (cmapSubtable->format.f0.format) {
    case 0:
        length = cmapSubtable->format.f0.length;
        language = cmapSubtable->format.f0.language;
        break;
    case 2:
        length = cmapSubtable->format.f2.length;
        language = cmapSubtable->format.f2.language;
        break;
    case 4:
        length = cmapSubtable->format.f4.length;
        language = cmapSubtable->format.f4.language;
        break;
    case 6:
        length = cmapSubtable->format.f6.length;
        language = cmapSubtable->format.f6.language;
        break;
    case 8:
        length = cmapSubtable->format.f8.length;
        language = cmapSubtable->format.f8.language;
        break;
    case 10:
        length = cmapSubtable->format.f10.length;
        language = cmapSubtable->format.f10.language;
        break;
    case 12:
        length = cmapSubtable->format.f12.length;
        language = cmapSubtable->format.f12.language;
        break;
    default:
        break;
    }
    printf("  (%s,%s,%s) at %d+%d: format %d.\n",
           platform_string(cmapEncodingSubtable->platformID),
           platform_specific_string(cmapEncodingSubtable->platformID, cmapEncodingSubtable->platformSpecificID),
           language == 0 ? "(language-independent)" : language_string(cmapEncodingSubtable->platformID, language),
           cmapEncodingSubtable->offset,
           length,
           cmapSubtable->format.f0.format);
}

uint16_t f2_glyph_index(uint8_t b1, uint8_t b2, CMapTable *cmapTable, int n)
{
    int k;
    CMapFormat2 *f2;
    uint16_t offset;
    unsigned char *p;
    uint16_t r;

    f2 = &cmapTable->cmapArray[n].cmapSubtable.format.f2;

    if (b1 == 0) {
        k = f2->subHeaderKeys[b2]/8;
        if (k != 0) {
            return 0;
        }
    } else {
        k = f2->subHeaderKeys[b1]/8;
        if (k == 0) {
            return 0;
        }
    }

    if ((b2 < f2->subHeaders[k].firstCode)
     || (b2 >= (f2->subHeaders[k].firstCode+f2->subHeaders[k].entryCount))) {
        return 0;
    }

    offset = cmapTable->cmapArray[n].cmapEncodingSubtable.offset;
    offset += 3*2+256*2+k*8+6+f2->subHeaders[k].idRangeOffset;
    offset += (b2 - f2->subHeaders[k].firstCode)*2;

    p = &cmapTable->buffer[offset];

    r = GET2B(p);

    if (r != 0) {
        r = ((int)r+(int)f2->subHeaders[k].idDelta) % 65536;
    }

    return r;
}

/* Create a Format 4 buffer from a Unicode to Glyph Index table. */
void cmap_format_4(U2C *u2c, uint16_t language,
                   uint8_t **bufferp, uint32_t *lengthp)
{
    register uint8_t *p;
    int i, segCount;
    struct Segment {
        uint16_t startCode;
        uint16_t endCode;
        uint16_t delta;
    } *segment;
    uint16_t delta;
    uint8_t *buffer;
    uint32_t length;

    segment = (struct Segment *)calloc(u2c->length, sizeof(struct Segment));

    segCount = 0;
    segment[segCount].startCode = u2c->array[0].u;
    segment[segCount].endCode = u2c->array[0].u;
    segment[segCount].delta = 65536+u2c->array[0].c - u2c->array[0].u;
    for (i = 1; i < u2c->length; i++) {
        if (u2c->array[i].c != 0) {
            delta = 65536+u2c->array[i].c - u2c->array[i].u;
            if (((u2c->array[i].u - u2c->array[i-1].u) == 1)
             && (delta == segment[segCount].delta)) {
                segment[segCount].endCode = u2c->array[i].u;
            } else {
                segCount++;
                segment[segCount].startCode = u2c->array[i].u;
                segment[segCount].endCode = u2c->array[i].u;
                segment[segCount].delta = delta;
            }
        }
    }
    segCount++;

    length = 7*2 + segCount*2 + 2 + segCount*2 + segCount*2 + segCount*2;
    length = ((length+3)/4)*4;

    buffer = (uint8_t *)calloc(length, 1);

    p = buffer;
    SET2B(p, 4);
    SET2B(p, length);
    SET2B(p, language);
    SET2B(p, segCount*2);

    i = floor(log(segCount)/log(2));
    SET2B(p, 2*(1<<i));
    SET2B(p, i);
    SET2B(p, 2*segCount - 2*(1<<i));

    for (i = 0; i < segCount; i++) {
        SET2B(p, segment[i].endCode);
    }
    p += 2;					/* reservedPad: 0 */
    for (i = 0; i < segCount; i++) {
        SET2B(p, segment[i].startCode);
    }
    for (i = 0; i < segCount; i++) {
        SET2B(p, segment[i].delta);
    }
#if 0
    for (i = 0; i < segCount; i++) {		/* idRangeOffset: 0 */
        SET2B(p, 0);
    }
						/* glyphIndexArray: none */
#endif

    if (bufferp != NULL) {
        *bufferp = buffer;
    } else {
        free(buffer);
    }

    if (lengthp != NULL) {
        *lengthp = length;
    }

    free(segment);

    return;
}

/*
 * If the cmap table has a Unicode cmap, the returned cmap buffer
 * is a copy of the original;
 * If the cmap table has no Unicode cmap, the returned cmap buffer
 * is the original plus a new equivalent Unicode cmap.
 * In both cases, the cmapTable itselft is not altered.
 */
void add_unicode_cmap(CMapTable *cmapTable,
                      uint8_t **bufferp, uint32_t *lengthp)
{
    int hasUnicodeCMap;
    U2C *u2c;
    uint16_t language;
    int i, n;
    uint8_t *buffer;
    uint32_t length;

    hasUnicodeCMap = 0;
    u2c = NULL;

    n = cmapTable->cmapIndex.numberSubtables;

    for (i = 0; i < n; i++) {
        CMapEncodingSubtable *cmapEncodingSubtable;

        cmapEncodingSubtable = &cmapTable->cmapArray[i].cmapEncodingSubtable;

        if (cmapEncodingSubtable->platformID == 0) {
            hasUnicodeCMap = 1;
        } else if ((cmapEncodingSubtable->platformID == 3)
                && (cmapEncodingSubtable->platformSpecificID == 1)) {
            hasUnicodeCMap = 1;
        } else if ((cmapEncodingSubtable->platformID == 3)
                && (cmapEncodingSubtable->platformSpecificID == 10)) {
            hasUnicodeCMap = 1;
        }

        if ((cmapEncodingSubtable->platformID == 3)
                && ((cmapEncodingSubtable->platformSpecificID == 3) || 
                    (cmapEncodingSubtable->platformSpecificID == 4))) {
            if (cmapTable->cmapArray[i].cmapSubtable.format.f0.format == 2) {
                u2c = cmapTable->cmapArray[i].cmapSubtable.u2c;
                language = cmapTable->cmapArray[i].cmapSubtable.format.f2.language;
            }
        }
    }

    if (hasUnicodeCMap || (u2c == NULL)) {
        length = cmapTable->length;
        buffer = (uint8_t *)calloc(length, 1);
        memcpy(buffer, cmapTable->buffer, length);
    } else {
        register uint8_t *p;
        uint8_t *ubuffer;
        uint32_t ulength;
        uint32_t offset;
        int inserted;

        length = 2*2 + (n+1)*(2*2+4);
        for (i = 0; i < n; i++) {
            CMapEncodingSubtable *cmapEncodingSubtable;
            uint32_t olength;

            cmapEncodingSubtable = &cmapTable->cmapArray[i].cmapEncodingSubtable;

            olength = 0;
            switch (cmapTable->cmapArray[i].cmapSubtable.format.f0.format) {
            case 0:
            case 2:
            case 4:
            case 6:
                olength = cmapTable->cmapArray[i].cmapSubtable.format.f0.length;
                break;
            case 8:
            case 10:
            case 12:
                olength = cmapTable->cmapArray[i].cmapSubtable.format.f8.length;
                break;
            default:
                fprintf(stderr, "The font has an unknown cmap.  Cannot proceed.\n");
                exit(1);
                break;
            }
            olength = ((olength+3)/4)*4;
            length += olength;
        }
        cmap_format_4(u2c, language, &ubuffer, &ulength);
        length += ulength;

        buffer = (uint8_t *)calloc(length, 1);

        p = buffer;
        p += 2;					/* version: 0 */
        SET2B(p, n+1);				/* numberSubtables */

        offset = 2*2 + (n+1)*(2*2+4);
        inserted = 0;
        for (i = 0; i < n; i++) {
            CMapEncodingSubtable *cmapEncodingSubtable;
            uint8_t *obuffer;
            uint32_t olength;

            cmapEncodingSubtable = &cmapTable->cmapArray[i].cmapEncodingSubtable;

            if (inserted == 0) {
                if ((cmapEncodingSubtable->platformID >= 3)
                 && (cmapEncodingSubtable->platformSpecificID > 1)) {
                    /* insert it here */
                    SET2B(p, 3);	/* Microsoft platform */
                    SET2B(p, 1);	/* Unicode encoding */
                    SET4B(p, offset);
                    memcpy(buffer+offset, ubuffer, ulength);
                    offset += ulength;
                    inserted = 1;
                }
            }

            SET2B(p, cmapEncodingSubtable->platformID);
            SET2B(p, cmapEncodingSubtable->platformSpecificID);
            SET4B(p, offset);
            obuffer = cmapTable->buffer + cmapEncodingSubtable->offset;
            olength = 0;
            switch (cmapTable->cmapArray[i].cmapSubtable.format.f0.format) {
            case 0:
            case 2:
            case 4:
            case 6:
                olength = cmapTable->cmapArray[i].cmapSubtable.format.f0.length;
                break;
            case 8:
            case 10:
            case 12:
                olength = cmapTable->cmapArray[i].cmapSubtable.format.f8.length;
                break;
            default:
                break;
            }
            if (olength > 0) {
                memcpy(buffer+offset, obuffer, olength);
            }
            olength = ((olength+3)/4)*4;
            offset += olength;
        }


    }

    if (bufferp != NULL) {
        *bufferp = buffer;
    } else {
        free(buffer);
    }

    if (lengthp != NULL) {
        *lengthp = length;
    }
}
