/*
 * palm-db-tools: Access to List-format databases.
 * Copyright (C) 1999-2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Provides access to List databases:
 * http://www.magma.ca/~roo/list/list.html
 */

#include <string>
#include <cstring>

#include "libsupport/strop.h"

#include "ListDB.h"

using namespace PalmLib::FlatFile;

bool PalmLib::FlatFile::ListDB::classify(PalmLib::Database& pdb)
{
    return (pdb.creator() == PalmLib::mktag('L','S','d','b'))
        && (pdb.type() == PalmLib::mktag('D','A','T','A'));
}

bool PalmLib::FlatFile::ListDB::match_name(const std::string& name)
{
    return (name == "ListDB") || (name == "listdb") || (name == "list");
}

PalmLib::FlatFile::ListDB::ListDB(PalmLib::Database& pdb) : Database("listdb", pdb)
{
    // Unpack the application information block.
    ListAppInfoType hdr;
    hdr.unpack(pdb.getAppInfoBlock());
    m_display_style = hdr.displayStyle;
    m_write_protect = false;

    // The schema is always the same no matter what List database.
    appendField(hdr.customField1, Field::STRING);
    appendField(hdr.customField2, Field::STRING);
    appendField("memo", Field::STRING);

    // Read and convert all of the records.
    for (unsigned i = 0; i < pdb.getNumRecords(); ++i) {
        PalmLib::Record pdb_record = pdb.getRecord(i);

        // Make sure that the record has enough data for the offset table.
        if (pdb_record.raw_size() < 3)
            throw PalmLib::error("record is corrupt");

        // Pull out the field offsets.
        pi_char_t* p = pdb_record.raw_data();
        pi_char_t offset1 = p[0];
        pi_char_t offset2 = p[1];
        pi_char_t offset3 = p[2];

        // Make sure the offsets stay within the record and are in order.
        if (offset1 >= pdb_record.raw_size()
            || offset2 >= pdb_record.raw_size()
            || offset3 >= pdb_record.raw_size())
            throw PalmLib::error("record is corrupt");

        PalmLib::FlatFile::Field field1, field2, field3;

        field1.type = PalmLib::FlatFile::Field::STRING;
        if (offset1 != 0) 
            field1.v_string = std::string((char *) (p + offset1),
                                          strlen((char *) (p + offset1)));
        else
            field1.v_string = "";

        field2.type = PalmLib::FlatFile::Field::STRING;
        if (offset2 != 0)
            field2.v_string = std::string((char *) (p + offset2),
                                          strlen((char *) (p + offset2)));
        else
            field2.v_string = "";

        field3.type = PalmLib::FlatFile::Field::STRING;
        if (offset3 != 0)
            field3.v_string = std::string((char *) (p + offset3),
                                          strlen((char *) (p + offset3)));
        else
            field3.v_string = "";

        Record record;
        record.appendField(field1);
        record.appendField(field2);
        record.appendField(field3);
        appendRecord(record);
    }
}

void PalmLib::FlatFile::ListDB::outputPDB(PalmLib::Database& pdb) const
{
    unsigned i;

    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile, Database, outputPDB, (pdb));

    // Set the database's type and creator.
    pdb.type(PalmLib::mktag('D','A','T','A'));
    pdb.creator(PalmLib::mktag('L','S','d','b'));

    // Build the application information block.
    ListAppInfoType hdr;
    hdr.renamedCategories = 0;
    hdr.categoryLabels[0] = "Unfiled";
    for (i = 1; i < 16; ++i) {
        hdr.categoryLabels[i] = "";
    }
    for (i = 0; i < 16; i++) {
        hdr.categoryUniqIDs[i] = i;
    }
    hdr.lastUniqID = 15;
    hdr.displayStyle = m_display_style;
    hdr.writeProtect = m_write_protect;
    hdr.lastCategory = 0;
    hdr.customField1 = field_name(0);
    hdr.customField2 = field_name(1);

    // Pack and set the application information block.
    pdb.setAppInfoBlock(hdr.pack());

    // Output the records.
    for (i = 0; i < getNumRecords(); ++i) {
        PalmLib::FlatFile::Record record = getRecord(i);

        // Make sure that we have the correct number and type of fields.
        if (record.fields().size() != 3)
            throw PalmLib::error("more than 3 fields not supported");

        // Calculate the size needed for the record.
        size_t size = 3;
        for (unsigned int j = 0; j < getNumOfFields(); j++) {
#ifdef __LIBSTDCPP_PARTIAL_SUPPORT__
            const Field field = record.fields()[j];
#else
            const Field field = record.fields().at(j);
#endif
            if (field.type != Field::STRING)
                throw PalmLib::error("unsupported field type");
            size += 1 + field.v_string.length();
        }

        // Pack the fields into the record.
        pi_char_t* buf = new pi_char_t[size];
        pi_char_t* p = buf + 3;

        buf[0] = static_cast<pi_char_t> (3);
#ifdef __OLD_CPP_ISO__
        for (j = 0; j < getNumOfFields(); j++) {
#else
        for (unsigned int j = 0; j < getNumOfFields(); j++) {
#endif
#ifdef __LIBSTDCPP_PARTIAL_SUPPORT__
            const Field field = record.fields()[j];
#else
            const Field field = record.fields().at(j);
#endif
            strcpy((char *) p, field.v_string.c_str());
            p += field.v_string.length() + 1;
            if (j < 2)
                buf[j + 1] = static_cast<pi_char_t> (p - buf);
        }

        // Allocate and set a new PalmOS record.
        PalmLib::Record pdb_record(0, 0, buf, size);
        pdb.appendRecord(pdb_record);
        delete [] buf;
    }
}

unsigned PalmLib::FlatFile::ListDB::getMaxNumOfFields() const
{
    return 3;
}

bool PalmLib::FlatFile::ListDB::supportsFieldType(const Field::FieldType& type) const
{
    switch (type) {
    case Field::STRING:
        return true;
    default:
        return false;
    }
}

unsigned PalmLib::FlatFile::ListDB::getMaxNumOfListViews() const
{
    return 1;
}

void PalmLib::FlatFile::ListDB::setOption(const std::string& name,
                                          const std::string& value)
{
    if (name == "list-display-style") {
        if (value == "field1-field2")
            m_display_style = FIELD1_FIELD2;
        else if (value == "field2-field1")
            m_display_style = FIELD2_FIELD1;
    } else if (name == "readonly" || name == "read-only") {
        // override the read-only option
        m_write_protect = StrOps::string2boolean(value);
        Database::setOption("read-only", "false");
    } else {
        SUPERCLASS(PalmLib::FlatFile, Database, setOption, (name, value));
    }
}

PalmLib::FlatFile::Database::options_list_t
PalmLib::FlatFile::ListDB::getOptions(void) const
{
    typedef PalmLib::FlatFile::Database::options_list_t::value_type value;
    PalmLib::FlatFile::Database::options_list_t result;

    result = SUPERCLASS(PalmLib::FlatFile, Database, getOptions, ());

    switch (m_display_style) {
    case FIELD1_FIELD2:
        result.push_back(value("list-display-style", "field1-field2"));
        break;
    case FIELD2_FIELD1:
        result.push_back(value("list-display-style", "field2-field1"));
        break;
    }

    // override the read-only option
    bool overrode = false;
    for (PalmLib::FlatFile::Database::options_list_t::iterator i
             = result.begin(); i != result.end(); ++i) {
        const std::string& name = (*i).first;

        if (name == "read-only") {
            if (m_write_protect)
                (*i).second = "true";
            else
                (*i).second = "false";
            overrode = true;
        }
    }

    // If we didn't override the read-only option, then set it.
    if (!overrode && m_write_protect)
        result.push_back(value("read-only", "true"));

    return result;
}

void PalmLib::FlatFile::ListDB::doneWithSchema()
{
    // Let the superclass have a chance.
    SUPERCLASS(PalmLib::FlatFile, Database, doneWithSchema, ());

    // Ensure that 3 fields have been specified.
    if (getNumOfFields() != 3)
        throw PalmLib::error("all List databases require 3 fields");
}

PalmLib::Block PalmLib::FlatFile::ListDB::ListAppInfoType::pack()
{
    int i;

    // The List application info block is allocated to a fixed 512 byte size.
    PalmLib::Block block(512);
    PalmLib::Block::pointer p = block.data();

    PalmLib::set_short(p, renamedCategories);
    p += 2;

    for (i = 0; i < 16; ++i) {
        strncpy((char *) p, categoryLabels[i].c_str(), 15);
        p += 16;
    }
    for (i = 0; i < 16; ++i) {
        *p++ = categoryUniqIDs[i];
    }
    *p++ = lastUniqID;

    switch (displayStyle) {
    case FIELD1_FIELD2:
        *p++ = 0;
        break;
    case FIELD2_FIELD1:
        *p++ = 1;
        break;
    }
    *p++ = writeProtect ? 1 : 0;
    *p++ = lastCategory;

    strncpy((char *) p, customField1.c_str(), 15);
    p += 16;

    strncpy((char *) p, customField2.c_str(), 15);
    p += 16;

    // Return the constructed data block.
    return block;
}

void PalmLib::FlatFile::ListDB::ListAppInfoType::unpack(const PalmLib::Block& block)
{
    int i;

    // Check the size of the app info block.
    if (block.size() < (2 + (16 * 16) + (16 * 1) + 1 + 3 + 16 + 16))
        throw PalmLib::error("header is corrupt");

    // Point at the start of the buffer.
    const pi_char_t* p = block.data();

    // Unpack the application info block.
    renamedCategories = PalmLib::get_short(p);
    p += 2;

    for (i = 0; i < 16; ++i) {
        categoryLabels[i] = std::string((char *) p);
        p += 16;
    }
    for (i = 0; i < 16; ++i) {
        categoryUniqIDs[i] = *p++;
    }
    lastUniqID = *p++;
    
    switch (*p++) {
    default:
    case 0:
        displayStyle = FIELD1_FIELD2;
        break;
    case 1:
        displayStyle = FIELD2_FIELD1;
        break;
    }

    writeProtect = (*p++ != 0) ? true : false;
    lastCategory = *p++;

    customField1 = std::string((char *) p);
    p += 16;

    customField2 = std::string((char *) p);
    p += 16;
}
