#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <stdexcept>
#include <utility>

#ifndef WIN32
# if TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
# else
#  if HAVE_SYS_TIME_H
#   include <sys/time.h>
#  else
#   include <time.h>
#  endif
# endif
#else
# include <time.h>
#endif

#include "libpalm/File.h"
#include "libflatfile/Database.h"
#include "libflatfile/Factory.h"
#include "clp.h"
#include "strop.h"
#include "infofile.h"
#include "flatfile/pdbtools.h"

extern std::ostream* err;

#define errorout(s) error << linenum << ": " << (s) << std::endl
#define warning(s) error << linenum << ": warning: " << (s) << std::endl

void DataFile::InfoFile::runParser(DataFile::InfoFile::Parser * p_Parser) const
{    
    std::ostringstream error;

    // Open the information file.
    std::ifstream f(m_FileName.c_str());
    if (!f) {
    error << "unable to open '" << m_FileName.c_str() << "'\n";
        *err << error.str();
    throw CLP::parse_error(error.str());
    }

    int linenum = 0;
    while (1) {
        // Read the next line into the buffer.
        std::string line(StrOps::readline(f));
        if (!f)
            break;

        ++linenum;
        // Strip any trailing newline characters.
        line = StrOps::strip_back(line, "\r\n");

        // Strip trailing whitespace.
        line = StrOps::strip_back(line, " \t");

        // Strip leading whitespace.
        line = StrOps::strip_front(line, " \t");

        // Skip this line if it is blank.
        if (line.length() == 0)
            continue;
        
        // Parse the line into an argv-style array.
        std::vector<std::string> array;
        try {
            array = StrOps::str_to_array(line, " \t", true, true);
        } catch (const StrOps::csv_parse_error& e) {
            errorout(e.what());
            throw CLP::parse_error(error.str());
        }

        // Skip this line if no arguments were found.
        if (array.size() == 0)
            continue;
        
        p_Parser->parse(linenum, array);
    }
    f.close();
}

void DataFile::InfoFile::read(DataFile::CSVConfig& state) const
{
    runParser((DataFile::InfoFile::Parser *) new ConfigParser(state));
}

void DataFile::InfoFile::read(PalmLib::FlatFile::Database& db) const
{
    runParser((DataFile::InfoFile::Parser *) new DatabaseParser(db));
}

void DataFile::InfoFile::ConfigParser::parse(int linenum, std::vector<std::string> array)
{
    std::ostringstream error;
    StrOps::lower(array[0]);
    if (array[0] == "extended") {
        if (array.size() != 2) {
            errorout("the extended directive takes 1 argument");
            throw CLP::parse_error(error.str());
        }
        m_Config.extended_csv_mode = StrOps::string2boolean(array[1]);
    } else if (array[0] == "quoted") {
        if (array.size() != 2) {
            errorout("the quoted directive takes 1 argument");
            throw CLP::parse_error(error.str());
        }
        m_Config.quoted_string = StrOps::string2boolean(array[1]);
    } else if (array[0] == "csvfile") {
        if (array.size() != 2) {
            errorout("option directives take 1 arguments");
            throw CLP::parse_error(error.str());
        }
        m_Config.csv_fname = array[1];
    } else if (array[0] == "separator") {
        if (array.size() != 2) {
            errorout("option directives take 1 arguments");
            throw CLP::parse_error(error.str());
        }
        m_Config.field_sep = array[1];
    } else if (array[0] == "format") {
        if (array.size() != 3) {
            errorout("format directives take 1 arguments");
            throw CLP::parse_error(error.str());
        }
        if (array[1] == std::string("date", 4))
            m_Config.format_date = array[2];
        else if (array[1] == std::string("time", 4))
            m_Config.format_time = array[2];
    }
}

void DataFile::InfoFile::DatabaseParser::parse(int linenum, std::vector<std::string> array)
{
    std::ostringstream error;
    StrOps::lower(array[0]);
    if (array[0] == "title") {
        if (array.size() != 2) {
            errorout("title directive only takes 1 argument");
            throw CLP::parse_error(error.str());
        }
        m_DB.title(array[1]);
    } else if (array[0] == "field") {
        if (array.size() < 3 || array.size() > 4) {
            errorout("field directive takes 3 or 4 arguments");
            throw CLP::parse_error(error.str());
        }
        
        try {
            if (array.size() == 3) {
                PalmLib::FlatFile::Field::FieldType type = StrOps::string2type(array[2]);
                m_DB.appendField(array[1], type);
            } else {
                PalmLib::FlatFile::Field::FieldType type = StrOps::string2type(array[2]);
                m_DB.appendField(array[1], type, array[3]);
            }
        } catch (const std::string& errstr) {
            errorout(errstr);
            *err << error.str();
            throw CLP::parse_error(error.str());
        } catch (const std::exception& e) {
            errorout(e.what());
            *err << error.str();
            throw CLP::parse_error(error.str());
        }
    } else if (array[0] == "view") {
        PalmLib::FlatFile::ListView lv;

        // Ignore attempts to go beyond the maximum number of list views.
        if (m_DB.getMaxNumOfListViews() != 0
                && m_DB.getNumOfListViews() + 1 > m_DB.getMaxNumOfListViews()) {
            warning("too many view directives for this database type");
            return;
        }

        // Ensure that we have enough arguments.
        if (array.size() < 3) {
            errorout("view directive takes at least 3 arguments");
            throw CLP::parse_error(error.str());
        }
        if ((array.size() % 2) != 0) {
            errorout("missing field name or width in view directive");
            throw CLP::parse_error(error.str());
        }

        // The first argument is the view name.
        lv.name = array[1];

        unsigned int firstfield = 2;
        if (array[2] == "editor") {
            lv.editoruse = StrOps::string2boolean(array[3]);
            firstfield = 4;
        }
        // Loop through the remaining arguments to find the fields
        // and widths for the rest of the list view.
        for (unsigned i = firstfield; i < array.size() - 1; i += 2) {
            const std::string& field_name = array[i];
            const std::string& width_string = array[i + 1];
            int field_index = -1;
            int width = 80;

            // Search for the field index corresponding to the field name.
            for (unsigned j = 0; j < m_DB.getNumOfFields(); ++j) {
                if (field_name == m_DB.field_name(j)) field_index = j;
            }
            if (field_index < 0) {
                errorout("unknown field name in field directive: " + field_name);
                throw CLP::parse_error(error.str());
            }

            // Extract the field width from the string.
            StrOps::convert_string(width_string, width);

            // Check the width for validity.
            if (width < 10 || width > 160) {
                errorout("field width must be greater than 10 and less than 160");
                throw CLP::parse_error(error.str());
            }

            // Add this column to the list view.
            PalmLib::FlatFile::ListViewColumn col(field_index, width);
            lv.push_back(col);
        }

        // Add this list view to the database.
        try {
            m_DB.appendListView(lv);
        } catch (const std::exception& e) {
            errorout(e.what());
            throw CLP::parse_error(error.str());
        }
    } else if (array[0] == "option") {
        if (array.size() != 3) {
            errorout("option directives take 2 arguments");
            throw CLP::parse_error(error.str());
        }
        m_DB.setOption(array[1], array[2]);
    } else if (array[0] == "about") {
        if (array.size() != 2) {
            errorout("about directives take 1 arguments");
            throw CLP::parse_error(error.str());
        }
        m_DB.setAboutInformation( array[1]);
    }
}

void DataFile::InfoFile::TypeParser::parse(int linenum, std::vector<std::string> array)
{
    std::ostringstream error;
    StrOps::lower(array[0]);
    if (array[0] == "type") {
        if (array.size() != 2) {
            errorout("title directive only takes 1 argument");
            *err << error.str();
            throw CLP::parse_error(error.str());
        }
        m_Type = array[1];
    }
}

void DataFile::InfoFile::PDBPathParser::parse(int linenum, std::vector<std::string> array)
{
    std::ostringstream error;
    StrOps::lower(array[0]);
    if (array[0] == "pdbpath") {
        if (array.size() != 2) {
            errorout("title directive only takes 1 argument");
            *err << error.str();
            throw CLP::parse_error(error.str());
        }
        m_Path = array[1];
    }
}

#undef warning
#undef errorout

std::string DataFile::InfoFile::readType() const
{
    std::string l_Type("db");
    runParser((DataFile::InfoFile::Parser *) new TypeParser(l_Type));    
    return (const std::string)l_Type;
}

std::string DataFile::InfoFile::readPDBPath() const
{
    std::string l_Type = ".";
    runParser((DataFile::InfoFile::Parser *) new PDBPathParser(l_Type));    
    return (const std::string)l_Type;
}

static bool
has_trivial_listview(const PalmLib::FlatFile::Database& flatfile)
{
    if (flatfile.getMaxNumOfListViews() != 1)
        return false;

    if (flatfile.getNumOfListViews() != 1)
        return false;

    PalmLib::FlatFile::ListView lv = flatfile.getListView(0);

    unsigned field = 0;
    PalmLib::FlatFile::ListView::const_iterator iter = lv.begin();
    for (; iter != lv.end(); ++iter, ++field) {
        const PalmLib::FlatFile::ListViewColumn& col = (*iter);
        if (col.field != field) return false;
    }

    if (field != flatfile.getNumOfFields())
        return false;

    return true;
}

void
DataFile::InfoFile::write(const PalmLib::FlatFile::Database& flatfile,
                 DataFile::CSVConfig& state, std::string p_PDBPath)
{
    std::ostringstream error;

    // Open the output file.
    std::ofstream l_Info(m_FileName.c_str());
    if (!l_Info) {
        error << "unable to open metadata file\n";
        throw CLP::parse_error(error.str());
    }

    writeDBInfo(l_Info, flatfile, state.extended_csv_mode);
    writeCSVInfo(l_Info, state);
    writePDBInfo(l_Info, p_PDBPath, state.extended_csv_mode);

    // Close the output file.
    l_Info.close();
}

void
DataFile::InfoFile::write(DataFile::CSVConfig& state, std::string p_PDBPath)
{
    std::ostringstream error;

    // Open the output file.
    std::ofstream l_Info(m_FileName.c_str());
    if (!l_Info) {
        error << "unable to open metadata file\n";
        *err << error.str();
        throw CLP::parse_error(error.str());
    }

    writeCSVInfo(l_Info, state);
    writePDBInfo(l_Info, p_PDBPath);

    // Close the output file.
    l_Info.close();
}

void DataFile::InfoFile::writeDBInfo(std::ofstream& p_Info, const PalmLib::FlatFile::Database& flatfile, bool p_extended)
{    
    p_Info << "# Database informations\n";
    /* Output the database type. */
    p_Info << "type " << StrOps::quote_string(flatfile.type(), p_extended) << "\n";

    /* Output the database title. */
    p_Info << "title " << StrOps::quote_string(flatfile.title(), p_extended) << "\n";

    /* Output the database structure. */
    if (!has_trivial_listview(flatfile)) {
        for (unsigned int i = 0; i < flatfile.getNumOfFields(); ++i) {
            p_Info << "field " << StrOps::quote_string(flatfile.field_name(i), p_extended)
                 << " " << StrOps::type2string(flatfile.field_type(i)) << " ";
        if (!flatfile.field(i).argument().empty())
                p_Info << StrOps::quote_string(flatfile.field(i).argument(), p_extended) << std::endl;
        else
            p_Info << std::endl;
        }
    } else {
        PalmLib::FlatFile::ListView lv = flatfile.getListView(0);
        PalmLib::FlatFile::ListView::const_iterator p = lv.begin();

        for (unsigned int i = 0; i < flatfile.getNumOfFields(); ++i, ++p) {
            p_Info << "field " << StrOps::quote_string(flatfile.field_name(i), p_extended)
                 << " " << StrOps::type2string(flatfile.field_type(i)) << " "
                 << (*p).width << std::endl;
        }
    }

    // If this database supports multiple views, output them all.
    if (! has_trivial_listview(flatfile)) {
        for (unsigned int i = 0; i < flatfile.getNumOfListViews(); ++i) {
            PalmLib::FlatFile::ListView lv = flatfile.getListView(i);

            p_Info << "view " << StrOps::quote_string(lv.name, p_extended) << " ";
            for (PalmLib::FlatFile::ListView::const_iterator p = lv.begin(); p != lv.end(); ++p) {
                const PalmLib::FlatFile::ListViewColumn& col = (*p);
                p_Info << " " << StrOps::quote_string(flatfile.field_name(col.field), p_extended)
                     << " " << col.width;
            }
            p_Info << std::endl;
        }
    }

    // Output any extra options that this database format supports.
    const PalmLib::FlatFile::Database::options_list_t opts = flatfile.getOptions();
    for (PalmLib::FlatFile::Database::options_list_t::const_iterator p = opts.begin();
         p != opts.end(); ++p) {
        p_Info << "option " << (*p).first << ' ' << (*p).second << std::endl;
    }

    if (!flatfile.getAboutInformation().empty())
        p_Info << "about" <<StrOps::quote_string(flatfile.getAboutInformation(), p_extended) << std::endl;

}

void DataFile::InfoFile::writeCSVInfo(std::ofstream& p_Info, DataFile::CSVConfig& state)
{
    p_Info << "# CSV informations\n";
    /* Output extended CSV mode. */
    if (state.extended_csv_mode)
        p_Info << "extended on\n";
    else
        p_Info << "extended off\n";

    /* Output quoted CSV string data. */
    if (!state.quoted_string)
        p_Info << "quoted off\n";

    if (state.field_sep != std::string(",",1))
        p_Info << "separator " << state.field_sep << std::endl;
    
    p_Info << "format time " << StrOps::quote_string(state.format_time, state.extended_csv_mode) << std::endl;
    p_Info << "format date " << StrOps::quote_string(state.format_date, state.extended_csv_mode) << std::endl;

    if (!state.csv_fname.empty())
        p_Info << "csvfile " << StrOps::quote_string(state.csv_fname, state.extended_csv_mode) << std::endl;

}

void DataFile::InfoFile::writePDBInfo(std::ofstream& p_Info, std::string p_PDBPath, bool p_extended)
{
    p_Info << "# PDB informations\n";
    p_Info << "pdbpath " << StrOps::quote_string(p_PDBPath, p_extended) << std::endl;
}
