/*
 * Code to handle dealing with the configuration file.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: conf.c,v 1.6 1999/05/19 09:42:47 kefka Exp $
 */

#include <config.h>
#include <stdarg.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "error.h"
#include "conf.h"
#include "x10.h"
#include "conf-proto.h"
#include "stacklist.h"
#include "snprintf.h"

/* Points flex to our conf file. */
extern FILE *yyin;
extern int yyparse();

/* Uglies needed to track bison errors. */
extern int yynerrs;

/* Local prototypes. */
static void conf_clear(void);
static int conf_check(void);

/* Configuration options. */
char *conf_tty;
uid_t conf_daemon_socket_uid;
gid_t conf_daemon_socket_gid;
int conf_daemon_socket_mode;
uid_t conf_daemon_monitor_socket_uid;
gid_t conf_daemon_monitor_socket_gid;
int conf_daemon_monitor_socket_mode;
int conf_housecode;
char *conf_daemon_dir;
int conf_daemon_command_timeout;

Stacklist conf_devicelist;
Stacklist conf_aliaslist;
Stacklist conf_macrolist;

/* These configuration options are just for convenience. */
char conf_daemon_monitor_socket_path[PATH_MAX];
char conf_daemon_socket_path[PATH_MAX];

/* Translation table for homecodes. */
unsigned char housecode_table[] = {
	HOUSECODE_A,
	HOUSECODE_B,
	HOUSECODE_C,
	HOUSECODE_D,
	HOUSECODE_E,
	HOUSECODE_F,
	HOUSECODE_G,
	HOUSECODE_H,
	HOUSECODE_I,
	HOUSECODE_J,
	HOUSECODE_K,
	HOUSECODE_L,
	HOUSECODE_M,
	HOUSECODE_N,
	HOUSECODE_O,
	HOUSECODE_P
};

/* Translation table for devicecodes. */
unsigned char devicecode_table[] = {
	DEVICECODE_1,
	DEVICECODE_2,
	DEVICECODE_3,
	DEVICECODE_4,
	DEVICECODE_5,
	DEVICECODE_6,
	DEVICECODE_7,
	DEVICECODE_8,
	DEVICECODE_9,
	DEVICECODE_10,
	DEVICECODE_11,
	DEVICECODE_12,
	DEVICECODE_13,
	DEVICECODE_14,
	DEVICECODE_15,
	DEVICECODE_16
};


/* Parse the conf file. */
void conf_parse(char *filename) {
	
	/* Clear the configuration options. */
	conf_clear();
	
	/* Open our conf file to parse. */
	yyin=fopen(filename, "r");
	if(!yyin) {
		fatal("Could not open conf file '%s'.", filename);
	}
	
	/* Parse the file. */
	if(yyparse() || yynerrs) {
		fatal("Could not parse conf file.");
	}

	/* Make sure we were given all the needed configurations. */
	if(!conf_check()) {
		fatal("Could not use conf file, exiting.");
	}
	
	/* Create the values for the daemon socket paths (for convenience). */
	if(snprintf(conf_daemon_monitor_socket_path, PATH_MAX, "%s/%s", conf_daemon_dir, DAEMON_MONITOR_SOCKET_FILE) == -1) {
		fatal("Daemon monitor socket's path too large.");
	}
	if(snprintf(conf_daemon_socket_path, PATH_MAX, "%s/%s", conf_daemon_dir, DAEMON_SOCKET_FILE) == -1) {
		fatal("Daemon socket's path too large.");
	}
	
	/* 
	 * Sort all the aliases in the aliaslist so we can group commands to
	 * them efficiently.
	 */
	conf_aliaslist_sort(&conf_aliaslist);
	
	return;
}


/* Check that all the needed configurations were given. */
static int conf_check(void) {
	int retval=1;
	
	if(!conf_tty) {
		error("TTY not specified.");
		retval=0;
	}
	if(conf_daemon_socket_uid == -1 || conf_daemon_socket_gid == -1) {
		error("DAEMON_SOCKET_NAME not specified.");
		retval=0;
	}
	if(conf_daemon_monitor_socket_uid == -1 || conf_daemon_socket_gid == -1) {
		error("DAEMON_MONITOR_SOCKET_NAME not specified.");
		retval=0;
	}
	if(conf_housecode == -1) {
		error("HOUSECODE not specified.");
		retval=0;
	}
	if(!conf_daemon_dir) {
		error("DAEMON_DIR not specified.");
		retval=0;
	}
	if(conf_daemon_command_timeout == -1) {
		error("DAEMON_COMMAND_TIMEOUT not specified.");
		retval=0;
	}
	
	return(retval);
}


/* 
 * Clear all optional configuration settings so we know if we had them, and
 * clear one of each required setting's options so we know if we didn't have
 * them.
 */
static void conf_clear(void) {
	
	conf_tty=NULL;
	conf_daemon_socket_uid=conf_daemon_socket_gid=-1;
	conf_daemon_monitor_socket_uid=conf_daemon_monitor_socket_gid=-1;
	conf_housecode=-1;
	conf_daemon_dir=NULL;
	conf_daemon_command_timeout=-1;
	stacklist_init(&conf_devicelist);
	stacklist_init(&conf_aliaslist);
	stacklist_init(&conf_macrolist);
	
	return;
}


/* Free all the memory associated with the configuration. */
void conf_free(void) {

	free(conf_tty);
	free(conf_daemon_dir);
	conf_devicelist_free(&conf_devicelist);
	conf_aliaslist_free(&conf_aliaslist);
	conf_macrolist_free(&conf_macrolist);
	
	return;
}


/* Creaate a new device. */
Device *conf_device_create(char *name, unsigned char housecode, unsigned char devicecode, int type) {
	Device *device;
	
	/* Create a new device entry. */
	device=malloc(sizeof(Device));
	if(!device) fatal("Out of memory.");
	
	/* Add the info to the device. */
	device->housecode=housecode;
	device->devicecode=devicecode;
	device->type=type;
	device->name=name;
	device->status_known=0;
	
	return(device);
}


/* Create a new alias. */
Alias *conf_alias_create(char *name, int devices, ...) {
	Alias *alias;
	int i;
	va_list ap;
	va_start(ap, devices);
	
	/* Create a new device entry. */
	alias=malloc(sizeof(Alias));
	if(!alias) fatal("Out of memory.");
	
	/* Add the info to the device. */
	alias->name=name;
	alias->devices=0;
	alias->device=NULL;
	
	/* Each additional argument is a device, add them all. */
	for(i=0; i < devices; i++) {
		conf_alias_insert_device(alias, va_arg(ap, Device *));
	}
	
	va_end(ap);
	return(alias);
}


/* Insert a device pointer into an alias. */
void conf_alias_insert_device(Alias *alias, Device *device) {
	
	/* Make space for the new device pointer. */
	alias->device=realloc(alias->device, (alias->devices + 1) * sizeof(Device *));
	if(!alias->device) fatal("Out of memory.");
	
	/* Add the pointer. */
	alias->device[alias->devices]=device;
	alias->devices++;
	
	return;
}


/* Free all the memory of a devicelist. */
void conf_devicelist_free(Stacklist *devicelist) {
	Stacklist_Entry *stacklist_entry;
	Device *device;
	
	/* Free each device. */
	for(stacklist_entry=devicelist->first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
		device=(Device *) stacklist_entry->info;
		free(device->name);
		free(device);
	}
	
	/* Free the list itself. */
	stacklist_free(devicelist);
	
	return;	
}


/* Free all the memory of an aliaslist. */
void conf_aliaslist_free(Stacklist *aliaslist) {
	Stacklist_Entry *stacklist_entry;
	Alias *alias;
	
	/* Free each alias. */
	for(stacklist_entry=aliaslist->first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
		alias=(Alias *) stacklist_entry->info;
		free(alias->name);
		free(alias->device);
		free(alias);
	}
	
	/* Free the list itself. */
	stacklist_free(aliaslist);
	
	return;	
}


/* Free all the memory of a macrolist. */
void conf_macrolist_free(Stacklist *macrolist) {
	Stacklist_Entry *stacklist_entry;
	Macro *macro;
	
	/* Free each macro. */
	for(stacklist_entry=macrolist->first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
		macro=(Macro *) stacklist_entry->info;
		free(macro->name);
		/* *** Need to free the state table. */
		free(macro);
	}
	
	/* Free the list itself. */
	stacklist_free(macrolist);
	
	return;	
}


/* Sort the devices in aliases in an aliaslist. */
void conf_aliaslist_sort(Stacklist *aliaslist) {
	Stacklist_Entry *stacklist_entry;
	
	/* Sort each alias. */
	for(stacklist_entry=aliaslist->first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
		conf_alias_sort((Alias *) stacklist_entry->info);
	}
	
	return;
}


/* 
 * Sort the devices in an alias to allow easy grouping of the devices into
 * commands.
 */
void conf_alias_sort(Alias *alias) {
	
	/* *** */

	return;
}


/* 
 * Lookup an identifier in the aliaslist.  If not found return NULL,
 * otherwise the alias we found.
 */
Alias *conf_aliaslist_lookup(char *name) {
	Stacklist_Entry *stacklist_entry;
	Alias *alias;
	
	/* Check each alias againt this name and see if we match one. */
	for(stacklist_entry=conf_aliaslist.first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
		alias=(Alias *) stacklist_entry->info;
		
		/* Check if the names match (case insensitive). */
		if(strcasecmp(name, alias->name) == 0) {
			
			/* They matched, this is a good one. */
			return(alias);
		}
	}
	
	/* We didn't match. */
	return(NULL);
}
