/*
 * Code to handle the daemon and client's socket needs.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: socket.c,v 1.4 1999/09/19 19:08:34 kefka Exp $
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <sys/time.h>
#include <string.h>
#include "error.h"
#include "socket.h"

	
/* Create the daemon socket. */	
int socket_create(char *socket_name, int mode, int uid, int gid) {
	struct sockaddr_un sock_addr;
	int daemon_socket;
	
	/* 
	 * *** What mode does it create the socket with?  User only, or
	 * does it use the umask?  If so, that's bad, it will mean a
	 * permissions race.
	 */
	
	/* Create a socket for talking to the user program. */
	daemon_socket=socket (AF_UNIX, SOCK_STREAM, 0);
	if(daemon_socket == -1) {
		fatal("Could not create daemon socket.");
	}
	
	/* Bind the socket to the filesystem. */
	sock_addr.sun_family=AF_UNIX;
	strcpy(sock_addr.sun_path, socket_name);
	if(bind(daemon_socket, (struct sockaddr *) &sock_addr, sizeof(sock_addr)) != 0) {
		fatal("Could not bind socket '%s'.", socket_name);
	}
	
	/* Clear the socket's permissions while we change the ownership. */
	if(chmod(socket_name, 0) != 0) {
		fatal("Could not clear permissions on socket '%s'.", socket_name);
	}
	
	/* 
	 * Change the ownership of the socket to the ones specified.  I'd
	 * rather use fchown, but it fails quietly on bound sockets?  If we
	 * got here, we are guaranteed that the socket exists by socket_name
	 * anyway, so it should be safe.
	 */
	if(chown(socket_name, uid, gid) != 0) {
		fatal("Could not chown socket '%s'.", socket_name);
	}
	
	/* Set the socket's real permissions. */
	if(chmod(socket_name, mode) != 0) {
		fatal("Could not set permissions on socket '%s'.", socket_name);
	}
	
	/* Set the socket to listen for connections. */
	if(listen(daemon_socket, SOMAXCONN) != 0) {
		fatal("Could not set socket to listen.");
	}
	
	/* Return this socket. */
	return(daemon_socket);
}


/* 
 * Connect to the daemon socket.
 *
 * Returns the fd of the socket.
 */
int socket_connect(char *socket_name) {
	struct sockaddr_un sock_addr;
	int daemon_socket;
	
	/* Create a socket for talking to the daemon program. */
	daemon_socket=socket(AF_UNIX, SOCK_STREAM, 0);
	if(daemon_socket == -1) {
		fatal("Could not create socket.");
	}
	
	/* Connect the socket to the daemon's socket. */
	sock_addr.sun_family=AF_UNIX;
	strcpy(sock_addr.sun_path, socket_name);
	if(connect(daemon_socket, (struct sockaddr *) &sock_addr, sizeof(sock_addr)) != 0) {
		fatal("Could not connect to socket '%s'.", socket_name);
	}
	
	/* Return this socket. */
	return(daemon_socket);
}


/*
 * Read from a socket.
 *
 * The socket must be non-blocking, or the timeout is useless.  We return
 * true if we worked, and false if we timed out.
 */
int socket_read(int socket, void *buffer, int size, int timeout) {
	int received;
	int readval;
	
	/* Keep reading until we have the whole message. */
	for(received=0; received < size;) {
		
		/* Wait until data becomes available. */
		if(!socket_wait_read(socket, timeout)) {
			debug(DEBUG_EXPECTED, "Socket read timed out.");
			return(0);
		}
		
		/* Read as much as we can. */
		readval=read(socket, buffer, size - received);
		if(readval <= 0) {
			debug(DEBUG_EXPECTED, "Read in socket read failed.");
			return(0);
		}
		
		/* Update the number of bytes received. */
		received += readval;
	}
	
	/* We got it. */
	return(1);
}


/* 
 * Wait for a socket to become readable.
 *
 * If it didn't become readable in the ammount of time given, return false,
 * otherwise true.
 *
 * The timeout given is in milliseconds.  If it is -1, it is infinite.
 */
int socket_wait_read(int socket, int timeout) {
	fd_set read_fd_set;
	struct timeval tv;
	struct timeval *tvp;
	
	/* If the timeout is -1, we have no timeout. */
	if(timeout == -1) {
		tvp=NULL;
	}
	else {
		tvp=&tv;
	}
	
	/* Wait for data to be readable. */
	FD_ZERO(&read_fd_set);
	FD_SET(socket, &read_fd_set);
	tv.tv_sec = timeout/1000000u;
	tv.tv_usec = timeout%1000000u;
	if(select(socket+1, &read_fd_set, NULL, NULL, tvp)) {
		
		/* We got some data, return ok. */
		return(1);
	}
	
	else {
		
		/* We didn't get any data, this is a fail. */
		return(0);
	}
}
