#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <image_from_tga.h>

/* Traiter le cas de la conservation ou pas du canal alpha */
/* gerer les extensions. */

/* Tous les types d'images que l'on connait */
enum TgaType {
  TGA_TYPE_NO_IMG       = 0,
  TGA_TYPE_CMAP_IMG     = 1,
  TGA_TYPE_RGB_IMG      = 2,
  TGA_TYPE_BW_IMG       = 3,
  TGA_TYPE_RLE_CMAP_IMG = 9,
  TGA_TYPE_RLE_RGB_IMG  = 10,
  TGA_TYPE_RLE_BW_IMG   = 11,
};

/* Les differents traitement possible pour le */
/* canal alpha (representation propre a       */
/* l'interpreteur pas au format TGA)          */
enum AlphaType {
  ALPHA_UNDEF,  /* on ne sait pas quoi en faire   */
  ALPHA_NONE,   /* on ne doit pas en tenir compte */
  ALPHA_USE,    /* on doit s'en servir            */
  ALPHA_PREMUL  /* il est present et les valeurs  */
                /* RGB sont premultiplie par sa   */
                /* valeur normalise (0..1)        */
};

/* Entete des fichiers TGA (Disponible au debut du fichier) */
typedef struct _TgaHeader TgaHeader;
struct _TgaHeader {
  uint8 id_size;       /* taille de l'identifiant de l'image (0..255) */
  uint8 cmap_type;     /* 0 = pas de palette, 1 = palette             */
  uint8 type;          /* type de l'image les choix sont dans TgaType */
  uint8 cmap_start[2]; /* position de la premiere couleur definie     */
                       /* dans la palette                             */
  uint8 cmap_size[2];  /* nombre de couleur definies dans la palette  */
  uint8 cmap_bits;     /* nombre de bits pour definir une couleur     */
                       /* dans la palette les valeurs possibles sont  */
                       /* 15, 16, 24 et 32 bits                       */
  uint8 x_origin[2];   /* position de l'image par rapport a l'ecran   */
  uint8 y_origin[2];   /* on se fiche de ces valeurs                  */
  uint8 width[2];      /* largeur de l'image                          */
  uint8 height[2];     /* hauteur de l'image                          */
  uint8 pix_bits;      /* nombre de bits pour definir un point. les   */
                       /* valeurs possibles sont 8, 16, 24 et 32 bits.*/
  uint8 attr;          /* bits 6-7 : doivent etre a 0                 */
                       /* bit 4 : 0 = pixel de gauche a droite,       */
                       /*         1 = droite a gauche                 */
                       /* bit 5 : 0 = ligne de bas en haut,           */
                       /*         1 = ligne de haut en bas            */
                       /* bits 0-3 : nombre de bits d'attributs par   */
                       /*            point (canal alpha)              */
};

/* Disponible a la fin du fichier si il s'agit d'un */
/* fichier Targa a la norme 1989                    */
typedef struct _TgaFooter TgaFooter;
struct _TgaFooter {
  uint8 ext_ofs[4];    /* The Extension Area Offset                   */
  uint8 dev_ofs[4];    /* The Developper Directory Offset             */
  uint8 signature[18]; /* doit etre : TRUEVISION-XFILE.               */
};

/* Extension Area qui peut etre disponible dans */
/* les fichier Targa a la norme 1989            */
typedef struct _TgaExtension TgaExtension;
struct _TgaExtension {
  uint8 ext_size[2];          /* taille de la zone d'extension sur 16 bits */
                              /* normalement 495 si il n'y a pas d'autres  */
                              /* champs que ceux present dans la structure */
  uint8 author_name[41];      /* chaine ASCII du nom de l'auteur. doit     */
                              /* imperativement etre termine par un 0 soit */
                              /* author_name[40] = 0 (en C)                */
  uint8 author_comments[324]; /* commentaires de l'auteur. compose de 4    */
                              /* lignes de 80 caracteres ASCII chacune     */
                              /* terminee par un 0. author_comments[80]=0, */
                              /* author_comments[161]=0,                   */
                              /* author_comments[242]=0 et                 */
                              /* author_comments[323]=0 (en C)             */
  /* date de creation de l'image. si elle n'est pas utilisee, tous les     */
  /* champs doivent etre mis a la valeur 0.                                */
  uint8 date_month[2];        /* mois (16bits)    : 1-12                   */
  uint8 date_day[2];          /* jour (16bits)    : 1-31                   */
  uint8 date_year[2];         /* annee (16bits)   : 4 chiffres (ex : 2000) */
  uint8 date_hour[2];         /* heure (16bits)   : 0-23                   */
  uint8 date_minute[2];       /* minute (16bits)  : 0-59                   */
  uint8 date_second[2];       /* seconde (16bits) : 0-59                   */
  uint8 job_name[41];         /* nom du travail auquel l'image est associee*/
                              /* 40 caracteres ASCII et 0 pour la 41ieme   */
                              /* valeur soit job_name[40] = 0 (en C)       */
  /* temps passe sur cette image. si ces champs ne sont pas utilises, ils  */
  /* doivent tous etre mis a la valeur 0.                                  */
  uint8 job_time_hours[2];    /* heures (16bits)   : 0-65535               */
  uint8 job_time_minutes[2];  /* minutes (16bits)  : 0-59                  */
  uint8 job_time_seconds[2];  /* secondes (16bits) : 0-59                  */
  uint8 soft_id[41];          /* 40 ASCII caracteres pour identifier le    */
                              /* logiciel qui a cree cette image. la       */
                              /* 41ieme valeur doit etre 0 (soft_id[40]=0) */
  uint8 soft_ver_num[2];      /* version du logiciel * 100 (4,17 => 417)   */
  uint8 soft_ver_letter[1];   /* code ASCII de la lettre correspondant a   */
                              /* la version du logiciel (a,b...)           */
  uint8 key_color[4];         /* couleur d'incruste. si elle n'est pas     */
                              /* utilisee tout est a 0. sinon la couleur : */
                              /* key_color[0] = bleu                       */
                              /* key_color[1] = vert                       */
                              /* key_color[2] = rouge                      */
                              /* key_color[3] = canal alpha (transparence) */
  /* rapport entre largeur et hauteur des pixels si ils ne sont pas carres */
  /* si ces valeurs ne sont pas utilisees TGA_UINT16(pix_ratio_height) = 0 */
  uint8 pix_ratio_width[2];   /* largeur des pixel par rapport a la hauteur*/
  uint8 pix_ratio_height[2];  /* hauteur des pixel par rapport a la largeur*/
  /* TGA_UINT16(gamma_numerator)/TGA_UINT16(gamma_denominator) correspond  */
  /* au gamma de l'image. les valeurs sont comprises entre 0.0 et 10.0     */
  uint8 gamma_numerator[2];
  uint8 gamma_denominator[2];  
  uint8 color_correc_ofs[4];  /* offset sur 32 bits de la table de         */
                              /* correction des couleurs par rapport au    */
                              /* debut du fichier. si il n'y a pas de table*/
                              /* l'offest doit etre 0.                     */
  uint8 postage_stamp_ofs[4]; /* offset sur 32 bits de la representation   */
                              /* reduite de l'image par rapport au debut du*/
                              /* fichier. si il n'y a pas d'image reduite  */
                              /* l'offest doit etre 0.                     */
  uint8 scan_line_ofs[4];     /* offset sur 32 bits de la table d'offest   */
                              /* des lignes de l'image par rapport au      */
                              /* debut du fichier. si il n'y a pas de table*/
                              /* d'offset des lignes, l'offset doit etre 0.*/
  uint8 attr_type;            /* attribut du canal Alpha.                  */
                              /* 0 : pas de donnees alphas disponibles     */
                              /* 1 : donnees de type inconnu dans le canal */
                              /*     alpha. on peut les ignorer.           */
                              /* 2 : donnees de type inconnu dans le canal */
                              /*     alpha. il faut les conserver.         */
                              /* 3 : canal alpha disponible.               */
                              /* 4 : canal alpha disponible. les donnees   */
                              /*     deja premultipler par le canal alpha  */
};

/* renvoi un uint16 a partir de l'adresse */
/* d'un tableau contenant deux uint8      */
#define TGA_UINT16(ptr) ((ptr)[0] | ((ptr)[1] << 8))
/* renvoi un uint32 a partir de l'adresse */
/* d'un tableau contenant quatre uint8    */
#define TGA_UINT32(ptr) ((ptr)[0] | ((ptr)[1] << 8) | ((ptr)[2] << 16)\
                       | ((ptr)[3] << 24))

#define TGA_SET_UINT16(ptr,val) {(ptr)[0] = (val&0xFF); (ptr)[1] = ((val)>>8)&0xFF;}

/* Renvoi l'image ou NULL si impossible */
image *image_new_from_tga(char *file_name)
{
  pix   vpix = WHITE;    /* definition du dernier pixel a repeter */
                         /* pour les paquets encodes RLE          */
  image *vimage;         /* structure qui contiendra l'image dans */
                         /* notre format interne                  */
  pix   *dst;            /* pointeur qui permet de ce deplacer    */
                         /* dans le buffer de pixel de l'image    */
                         /* finale vimage                         */
  pix   *dst_tmp;
  int   handle;          /* descripteur du fichier                */
  int   file_size;       /* taille du fichier                     */
  uint8 *buf;            /* buffer contenant tout le fichier      */
  uint8 *buf_move;       /* pointeur qui permet de ce deplacer sur*/
                         /* le fichier                            */
  pix   *palette = NULL; /* tableau de pixels qui represente la   */
                         /* palette si il en existe une.          */
  TgaHeader *vTgaHeader; /* pointe l'entete TGA                   */
  TgaFooter *vTgaFooter; /* pointe l'entete de fin de fichier TGA */
  TgaExtension *vTgaExtension; /* pointe sur la definition de la  */
                         /* zone des Extensions si elle existe.   */
  uint8 cmap_bpp;        /* nombre d'octet pour coder un element  */
                         /* de la palette si elle existe          */
  uint8 pix_bpp;         /* nombre d'octet pour coder un pixel    */
  uint16 cmap_start = 0; /* valeur de la premiere valeur indexee  */
                         /* dans la palette.                      */
  uint16 cmap_end = 0;   /* derniere valeur indexee de la palette */
  int32 hstep;           /* increment du pointeur dst pour passer */
                         /* au prochain point horizontal          */
  int32 vstep;           /* increment du pointeur dst pour passer */
                         /* a la prochaine ligne                  */
  uint16 nb_rle = 0;     /* nombre de points identiques           */
  uint16 nb_raw = 0;     /* nombre de points differents           */
  int    remain_size;    /* taille du fichier restant a           */
                         /* interpreter. utilise pour le RLE.     */
  int32  remain_width;   /* nombre de points encore attendus pour */
                         /* une ligne compressee avec RLE.        */
  uint8  alpha_type = ALPHA_UNDEF; 
                         /* s'il y a un canal alpha, quoi en faire*/
  uint32 i,x,y;
  uint16 vuint16;

  /* ouverture du fichier en lecture */
  if((handle = open(file_name,O_RDONLY)) == -1) return(NULL);
  /* on retire la taille du fichier en se */
  /* positionnant a la fin du fichier     */
  if((file_size = lseek(handle, 0, SEEK_END)) == -1)
  { close(handle); return(NULL); }
  /* on se repositionne au debut du fichier */
  if(lseek(handle, 0, SEEK_SET) == -1)
  { close(handle); return(NULL); }
  /* si il n'y a pas la place de l'entete dans le     */
  /* fichier, ce ne peut deja pas etre un fichier TGA */
  if((file_size < sizeof(TgaHeader)))
  { close(handle); return(NULL); }
  /* on alloue de la memoire pour tout le fichier */
  if((buf = (uint8 *)malloc(file_size*sizeof(uint8))) == NULL)
  { perror("malloc failed."); exit(1); }
  /* on lit tout le fichier que l'on place */
  /* dans le tampon "buf"                  */
  if(read(handle, buf, file_size) == -1)
  { free(buf); close(handle); return(NULL); }
  /* on referme le fichier car on */
  /* n'a pas plus besoin de lui   */
  close(handle);

  /* interprete le debut du fichier */
  /* comme l'entete TGA             */
  vTgaHeader = (TgaHeader *)buf;

  /* taille du fichier restant a interpreter.              */
  /* on supprime la taille de l'entete puisqu'on l'utilise */
  remain_size = file_size - sizeof(TgaHeader);

  /* interprete la fin du fichier comme      */
  /* la fin d'un fichier TGA a la norme 1989 */
  vTgaFooter = (TgaFooter *)(buf+file_size-sizeof(TgaFooter));  

  /* pour l'instant, on considere qu'il n'y */
  /* a pas de zone pour les extensions.     */
  vTgaExtension = NULL;

  /* on determine si il y a effectivement une          */
  /* entete TGA a la norme de 1989 a la fin du fichier */
  if((remain_size < sizeof(TgaFooter)) ||
     (strncmp(vTgaFooter->signature,"TRUEVISION-XFILE.",18) != 0))
  { /* la taille du fichier ou la non presence de la signature presente */
    /* dans l'entete en fin de fichier indique qu'elle n'existe pas.    */
    /* l'affectation de la valeur NULL materialise ce fait.             */
    vTgaFooter = NULL;
  }  else {
    /* la structure TgaFooter existe bien                  */
    /* on retire sa taille de ce qu'il reste a interpreter */
    remain_size -= sizeof(TgaFooter);
    
    /* on regarde si la zone d'extension existe bien */
    if(TGA_UINT32(vTgaFooter->ext_ofs) != 0)
    {
      /* on verifie, la coherence de la taille du      */
      /* fichier et la position de la zone d'extension */
      if(TGA_UINT32(vTgaFooter->ext_ofs)+sizeof(TgaExtension) > file_size)
      { free(buf); return(NULL); }

      /* on place le pointeur de la zone d'extension */
      /* sur la zone d'extension.                    */
      vTgaExtension = (TgaExtension *)(buf+TGA_UINT32(vTgaFooter->ext_ofs));

      /* on verifie la coherence de la taille reelle de la zone */
      /* d'extension (en effet, il peut y avoir des champs que  */
      /* l'on ne connaissait pas).                              */
      if((remain_size < TGA_UINT16(vTgaExtension->ext_size)) ||
         (TGA_UINT32(vTgaFooter->ext_ofs)
          +TGA_UINT16(vTgaExtension->ext_size) > file_size) ||
         (TGA_UINT16(vTgaExtension->ext_size) < sizeof(TgaExtension)))
      { free(buf); return(NULL); }

      /* on retire a ce qu'il reste a interpreter la taille */
      /* de la zone d'extension.                            */
      remain_size -= TGA_UINT16(vTgaExtension->ext_size);
      
      /* prend en note le type de gestion du canal alpha */
      switch(vTgaExtension->attr_type)
      {
        case 0 :
          /* il n'y a pas de canal aplha */
          alpha_type = ALPHA_NONE;
          /* on verifie qu'il y a en effet pas de bits */
          /* disponible pour le canal alpha.           */
          if((vTgaHeader->attr & 0x0F) != 0)
          { free(buf); return(NULL); }
          break;
        case 1 :
          /* on peut ignorer les infos presentes dans le canal alpha */
          alpha_type = ALPHA_NONE;
          break;
        case 2 :
          /* il y a des informations dans le canal alpha mais  */
          /* on ne sait pas quoi en faire. theoriquement on    */
          /* doit les conserver mais pourquoi en faire puisque */
          /* l'on ne sait pas a quoi elles servent. on n'en    */
          /* tient donc pas compte.                            */
          alpha_type = ALPHA_NONE;
          break;
        case 3 :
          /* il y a un canal alpha, on va donc s'en servir */
          alpha_type = ALPHA_USE;
          break;
        case 4 :
          /* il y a un canal alpha, on va donc s'en servir */
          /* les donnees RGB sont deja premultiplie par    */
          /* le canal alpha. il faudra annuler cela.       */
          alpha_type = ALPHA_PREMUL;
          break;
        default :
          /* on ne connait pas ces cas */
          free(buf); return(NULL);
          break;
      }
    }
  }

  /* si il y a un canal alpha et que l'on n'a pas put      */
  /* avoir d'informations sur quoi en faire dans la zone   */
  /* d'extension. alors on prend la decision de l'utiliser */
  /* (decision purement arbitraire)                        */
  if(((vTgaHeader->attr & 0x0F) != 0) && (alpha_type == ALPHA_UNDEF))
  { alpha_type = ALPHA_USE; }

  /* si on ne connait pas le codage de l'image, */
  /* on ne peut rien faire. on quitte.          */
  if((vTgaHeader->type != TGA_TYPE_NO_IMG) &&
     (vTgaHeader->type != TGA_TYPE_CMAP_IMG) &&
     (vTgaHeader->type != TGA_TYPE_RGB_IMG) &&
     (vTgaHeader->type != TGA_TYPE_BW_IMG) &&
     (vTgaHeader->type != TGA_TYPE_RLE_CMAP_IMG) &&
     (vTgaHeader->type != TGA_TYPE_RLE_RGB_IMG) &&
     (vTgaHeader->type != TGA_TYPE_RLE_BW_IMG))
  { free(buf); return(NULL); }

  /* si le fichier ne contient pas d'image, */
  /* on s'arrete tout de suite.             */
  if(vTgaHeader->type == TGA_TYPE_NO_IMG)
  { free(buf); return(NULL); }

  /* on verifie la coherence de la palette   */
  /* si elle n'existe pas, les autres champs */
  /* concernant la palette doivent etre a 0  */
  /* de meme la presence ou l'absence de la  */
  /* palette doit correspondre avec le type  */
  /* de codage des donnees.                  */
  if(((vTgaHeader->cmap_type != 0) && (vTgaHeader->cmap_type != 1)) ||
     ((vTgaHeader->cmap_type == 0) &&
      (TGA_UINT16(vTgaHeader->cmap_start) != 0) &&
      (TGA_UINT16(vTgaHeader->cmap_size) != 0) &&
      (vTgaHeader->cmap_bits != 0)) ||
     ((vTgaHeader->cmap_type == 0) &&
      ((vTgaHeader->type == TGA_TYPE_CMAP_IMG) ||
       (vTgaHeader->type == TGA_TYPE_RLE_CMAP_IMG))) ||
     ((vTgaHeader->cmap_type == 1) &&
      ((vTgaHeader->type == TGA_TYPE_RGB_IMG) ||
       (vTgaHeader->type == TGA_TYPE_BW_IMG) ||
       (vTgaHeader->type == TGA_TYPE_RLE_RGB_IMG) ||
       (vTgaHeader->type == TGA_TYPE_RLE_BW_IMG))))
  { free(buf); return(NULL); }

  /* verifie que la taille du fichier est en accord */
  /* avec la taille de l'identifiant de l'image     */
  if(remain_size < vTgaHeader->id_size)
  {
    free(buf); return(NULL);
  } else {
    /* une zone de moins a interpreter */
    remain_size -= vTgaHeader->id_size;
  }

  /* place le pointeur buf_move sur les premieres donnees         */
  /* a interpreter (palette ou donnees s'il n'y a pas de palette) */
  buf_move = buf + sizeof(TgaHeader) + vTgaHeader->id_size;

  /* si il y une palette alors on l'utilise. */
  if(vTgaHeader->cmap_type == 1)
  {
    /* on regarde si on supporte le type de codage de */
    /* la palette. si ce n'est pas le cas, on quitte  */
    if((vTgaHeader->cmap_bits != 15) && (vTgaHeader->cmap_bits != 16) &&
       (vTgaHeader->cmap_bits != 24) && (vTgaHeader->cmap_bits != 32))
    { free(buf); return(NULL); }

    /* on regarde si on sait quoi faire du canal alpha si il existe */
    if(((alpha_type == ALPHA_USE) || (alpha_type == ALPHA_PREMUL)) &&
       (((vTgaHeader->cmap_bits == 16) && ((vTgaHeader->attr & 0x0F) != 1)) ||
        ((vTgaHeader->cmap_bits == 32) && ((vTgaHeader->attr & 0x0F) != 8))))
    { free(buf); return(NULL); }

    /* calcul le nombre d'octet pour coder un element de la palette */
    cmap_bpp = (vTgaHeader->cmap_bits+7)>>3;

    /* on regarde si le codage des pixels correspond     */
    /* aux modes que l'on sait traiter avec une palette. */
    /* si ce n'est pas le cas, on quitte.                */
    if((vTgaHeader->pix_bits != 8) && (vTgaHeader->pix_bits != 16))
    { free(buf); return(NULL); }

    /* on verifie que les champs taille et    */
    /* debut de la palette sont coherents     */
    /* par rapport au mode de codage supporte */
    if((TGA_UINT16(vTgaHeader->cmap_size) == 0) ||
       ((vTgaHeader->pix_bits == 8) &&
        (TGA_UINT16(vTgaHeader->cmap_start) +
         TGA_UINT16(vTgaHeader->cmap_size) > 256)) ||
       ((vTgaHeader->pix_bits == 16) &&
        ((uint32)TGA_UINT16(vTgaHeader->cmap_start) +
         (uint32)TGA_UINT16(vTgaHeader->cmap_size) > 65536)))
    { free(buf); return(NULL); }

    cmap_start = TGA_UINT16(vTgaHeader->cmap_start);
    cmap_end   = cmap_start + TGA_UINT16(vTgaHeader->cmap_size) - 1;

    /* on verifie que la taille du fichier est */
    /* toujours plausible avec la presence de  */
    /* cette palette.                          */
    if(remain_size < cmap_bpp*TGA_UINT16(vTgaHeader->cmap_size))
    { free(buf); return(NULL); }
    
    /* on retire la taille de la palette a ce qu'il reste a interpreter */
    remain_size -= cmap_bpp*TGA_UINT16(vTgaHeader->cmap_size);
    
    /* on alloue la place necessaire pour cette palette */
    if((palette = (pix *)malloc(
      TGA_UINT16(vTgaHeader->cmap_size)*sizeof(pix))) == NULL)
    { free(buf); perror("malloc failed "); exit(1); }

    /* on charge cette palette */
    for(i = 0; i < TGA_UINT16(vTgaHeader->cmap_size); i++)
    {
      switch(vTgaHeader->cmap_bits) {
        case 15:
          vuint16 = TGA_UINT16(buf_move);
          buf_move += 2;
          palette[i] = COL(
            (uint8)((vuint16 >> 7) & 0xF8),
            (uint8)((vuint16 >> 2) & 0xF8),
            (uint8)(vuint16 << 3));
          break;
        case 16:
          vuint16 = TGA_UINT16(buf_move);
          buf_move += 2;
          /* si le 16ieme bit est mis c'est que cette couleur */
          /* est transparente. si on traite la transparence,  */
          /* on prend en compte                               */
          if(((alpha_type == ALPHA_USE) || (alpha_type == ALPHA_PREMUL))
            && (vuint16 & 0x8000))
          {
            palette[i] = COL_FULL(
              (uint8)((vuint16 >> 7) & 0xF8),
              (uint8)((vuint16 >> 2) & 0xF8),
              (uint8)(vuint16 << 3), 0x00);
          } else {
            palette[i] = COL(
              (uint8)((vuint16 >> 7) & 0xF8),
              (uint8)((vuint16 >> 2) & 0xF8),
              (uint8)(vuint16 << 3));
          }
          break;
        case 24:
          palette[i] = COL(buf_move[2], buf_move[1], buf_move[0]);
          buf_move += 3;
          break;
        case 32:
          /* pareil que pour 24 bits avec le canal alpha en plus */
          if(alpha_type == ALPHA_USE) {
            palette[i] = COL_FULL(buf_move[2], buf_move[1], buf_move[0],
              buf_move[3]);
            buf_move += 4;
          } else if(alpha_type == ALPHA_PREMUL) {
            /* on supprime la premultiplication des couleurs */
            /* par le canal alpha avant le stockage.         */
            palette[i] = COL_FULL(
              MIN(255,(((uint32)buf_move[2])*255)/buf_move[3]),
              MIN(255,(((uint32)buf_move[1])*255)/buf_move[3]),
              MIN(255,(((uint32)buf_move[0])*255)/buf_move[3]),
              buf_move[3]);
            buf_move += 4;
          } else {
            /* on ne tient pas compte du canal alpha */
            palette[i] = COL(buf_move[2], buf_move[1], buf_move[0]);
            buf_move += 4;
          }
          break;
      }
    }
  } /* fin traitement de la palette */

  /* si palette il y avait, a la charge */
  /* buf_move pointe maintenant sur les */
  /* donnees qui decrivent les points   */

  /* on regarde si il s'agit d'un format que l'on sait      */
  /* traiter on ne verifie ici que les formats sans palette */
  /* on supporte les modes RGB 15, 16, 24 et 32 bits et les */
  /* modes noir et blanc 8 et 16 bits.                      */
  if((((vTgaHeader->type == TGA_TYPE_RGB_IMG) ||
       (vTgaHeader->type == TGA_TYPE_RLE_RGB_IMG)) &&
      (vTgaHeader->pix_bits != 15) && (vTgaHeader->pix_bits != 16) &&
      (vTgaHeader->pix_bits != 24) && (vTgaHeader->pix_bits != 32)) ||
     (((vTgaHeader->type == TGA_TYPE_BW_IMG) ||
       (vTgaHeader->type == TGA_TYPE_RLE_BW_IMG)) &&
      (vTgaHeader->pix_bits != 8) && (vTgaHeader->pix_bits != 16)))
  { free(palette); free(buf); return(NULL); }

  /* nombre d'octet pour coder un pixel */
  pix_bpp = (vTgaHeader->pix_bits+7)>>3;

  /* on verifie que la taille du fichier permet  */
  /* bien de coder le format des donnees. si le  */
  /* codage n'utilise pas la compression RLE, on */
  /* est sur que l'on n'aura pas de debordement. */
  /* Par contre si on utilise la compression RLE */
  /* , ce test ne suffit pas, il faudra encore   */
  /* s'en assurer lors de la decompression.      */
  if((((vTgaHeader->type == TGA_TYPE_RGB_IMG) ||
       (vTgaHeader->type == TGA_TYPE_BW_IMG)  ||
       (vTgaHeader->type == TGA_TYPE_CMAP_IMG)) &&
      (remain_size < pix_bpp*TGA_UINT16(vTgaHeader->width)
       *TGA_UINT16(vTgaHeader->height))) ||
     (((vTgaHeader->type == TGA_TYPE_RLE_RGB_IMG) ||
       (vTgaHeader->type == TGA_TYPE_RLE_BW_IMG)  ||
       (vTgaHeader->type == TGA_TYPE_RLE_CMAP_IMG)) &&
      (remain_size < (1+pix_bpp)*TGA_UINT16(vTgaHeader->height))))
  { free(palette); free(buf); return(NULL); }

  /* alloue une nouvelle image de la            */
  /* taille de l'image contenue dans le fichier */
  vimage = image_new(
    (uint32)TGA_UINT16(vTgaHeader->width),
    (uint32)TGA_UINT16(vTgaHeader->height));

  if(vTgaHeader->attr & 0x20)
  {
    /* on lit les lignes en partant du haut alors */
    /* on place dst au debut de la premiere ligne */
    dst   = vimage->buf;
    vstep = vimage->width;
  } else {
    /* on lit les lignes en partant du bas alors  */
    /* on place dst au debut de la derniere ligne */
    dst   = vimage->buf + (vimage->height-1)*vimage->width;
    vstep = -vimage->width;
  }

  if(vTgaHeader->attr & 0x10)
  {
    /* les lignes sont ecrites de droite a gauche       */
    /* le premier pixel est donc le dernier de la ligne */
    dst += vimage->width-1;
    hstep = -1;
  } else {
    hstep = 1;
  }

  /* parcours toutes les lignes */
  for(y=0;y<vimage->height;y++)
  {
    /* on traite d'abord les cas sans compression RLE */
    if((vTgaHeader->type == TGA_TYPE_CMAP_IMG) ||
       (vTgaHeader->type == TGA_TYPE_RGB_IMG)  ||
       (vTgaHeader->type == TGA_TYPE_BW_IMG))
    {
      dst_tmp = dst;
      /* on fait tous les points d'une ligne */
      for(x=0;x<vimage->width;x++)
      {
        /* on fait tous les cas */
        switch(vTgaHeader->pix_bits) {
          case 8:
            if(vTgaHeader->type == TGA_TYPE_CMAP_IMG)
            {
              /* on verifie que l'indexe existe dans la palette */
              if((*buf_move < cmap_start) || (*buf_move > cmap_end))
              { free(palette); free(buf); image_free(vimage); return(NULL); }
              *dst = palette[(*buf_move)-cmap_start];
            }
            if(vTgaHeader->type == TGA_TYPE_BW_IMG)
            {
              *dst = COL(*buf_move,*buf_move,*buf_move);
            }
            buf_move++;
            break;
          case 15:
            /* il n'y a qu'un mode 15 bits que l'on connait */
            vuint16 = TGA_UINT16(buf_move);
            *dst = COL(
              (uint8)((vuint16 >> 7)&0xF8),
              (uint8)((vuint16 >> 2)&0xF8),
              (uint8)(vuint16 << 3));
            buf_move += 2;
            break;
          case 16:
            vuint16 = TGA_UINT16(buf_move);
            if(vTgaHeader->type == TGA_TYPE_CMAP_IMG)
            {
              /* on verifie que l'indexe existe dans la palette */
              if((vuint16 < cmap_start) || (vuint16 > cmap_end))
              { free(palette); free(buf); image_free(vimage); return(NULL); }
              *dst = palette[vuint16-cmap_start];
            }
            if(vTgaHeader->type == TGA_TYPE_BW_IMG)
            {
              *dst = COL(vuint16>>8, vuint16>>8, vuint16>>8);
            }
            if(vTgaHeader->type == TGA_TYPE_RGB_IMG)
            {
              /* le codage est :            */
              /* bit  15    = Interrupt bit */
              /* bits 10-14 = rouge         */
              /* bits 5-9   = vert          */
              /* bits 0-4   = blue          */
              
              /* si le bit Interrupt est mis  */
              /* alors ce point est invisible */
              if(((alpha_type == ALPHA_USE) || (alpha_type == ALPHA_PREMUL))
                && (vuint16 & 0x8000))
              {
                *dst = COL_FULL(
                  (uint8)((vuint16 >> 7)&0xF8),
                  (uint8)((vuint16 >> 2)&0xF8),
                  (uint8)(vuint16 << 3), 0x00);
              } else {
                *dst = COL(
                  (uint8)((vuint16 >> 7)&0xF8),
                  (uint8)((vuint16 >> 2)&0xF8),
                  (uint8)(vuint16 << 3));
              }
            }
            buf_move += 2;
            break;
          case 24:
            /* un seul mode 24 bits */
            *dst = COL(buf_move[2], buf_move[1], buf_move[0]);
            buf_move += 3;
            break;
          case 32:
            /* un seul mode 32 bits */
            if(alpha_type == ALPHA_USE) {
              *dst = COL_FULL(buf_move[2], buf_move[1], buf_move[0],
                buf_move[3]);
              buf_move += 4;
            } else if(alpha_type == ALPHA_PREMUL) {
              /* on supprime la premultiplication des couleurs */
              /* par le canal alpha avant le stockage.         */
              *dst = COL_FULL(
                MIN(255,(((uint32)buf_move[2])*255)/buf_move[3]),
                MIN(255,(((uint32)buf_move[1])*255)/buf_move[3]),
                MIN(255,(((uint32)buf_move[0])*255)/buf_move[3]),
                buf_move[3]);
              buf_move += 4;
            } else {
              /* on ne tient pas compte du canal alpha */
              *dst = COL(buf_move[2], buf_move[1], buf_move[0]);
              buf_move += 4;
            }
            break;
        }
        /* on passe au point suivant a ecrire */
        dst += hstep;
      }
      dst = dst_tmp;
    } else {
      /* on traite maintenant les modes avec compression RLE */
      dst_tmp = dst;

      /* on parcours une ligne */
      for(remain_width=vimage->width;remain_width>0;)
      {
        /* si il ne reste plus de place pour lire l'octet    */
        /* de control c'est que le fichier n'est pas correct */
        if(--remain_size < 0)
        { free(palette); free(buf); image_free(vimage); return(NULL); }

        if((*buf_move) & 0x80)
        { /* le point qui va suivre sera repete nb_rle fois */
          nb_rle = ((*(buf_move++)) & 0x7F) + 1;

          /* s'il ne reste plus de place pour lire le point */
          /* c'est que le fichier n'est pas correct         */
          remain_size -= pix_bpp;
          if(remain_size < 0)
          { free(palette); free(buf); image_free(vimage); return(NULL); }

          /* on regarde si le nombre de points ne deborde pas la ligne */
          /* si c'est le cas c'est que le fichier n'est pas correct    */
          remain_width -= nb_rle;
          if(remain_width < 0)
          { free(palette); free(buf); image_free(vimage); return(NULL); }

          /* on lit le point a repeter */
          /* on fait tous les cas      */
          switch(vTgaHeader->pix_bits) {
            case 8:
              if(vTgaHeader->type == TGA_TYPE_RLE_CMAP_IMG)
              {
                /* on verifie que l'index existe dans la palette */
                if((*buf_move < cmap_start) || (*buf_move > cmap_end))
                { free(palette); free(buf); image_free(vimage); return(NULL); }
                vpix = palette[(*buf_move)-cmap_start];
              }
              if(vTgaHeader->type == TGA_TYPE_RLE_BW_IMG)
              {
                vpix = COL(*buf_move,*buf_move,*buf_move);
              }
              buf_move++;
              break;
            case 15:
              /* il n'y a qu'un mode 15 bits que l'on connait */
              vuint16 = TGA_UINT16(buf_move);
              vpix = COL(
                (uint8)((vuint16 >> 7)&0xF8),
                (uint8)((vuint16 >> 2)&0xF8),
                (uint8)(vuint16 << 3));
              buf_move += 2;
              break;
            case 16:
              vuint16 = TGA_UINT16(buf_move);
              if(vTgaHeader->type == TGA_TYPE_RLE_CMAP_IMG)
              {
                /* on verifie que l'indexe existe dans la palette */
                if((vuint16 < cmap_start) || (vuint16 > cmap_end))
                { free(palette); free(buf); image_free(vimage); return(NULL); }
                vpix = palette[vuint16-cmap_start];
              }
              if(vTgaHeader->type == TGA_TYPE_RLE_BW_IMG)
              {
                vpix = COL(vuint16>>8, vuint16>>8, vuint16>>8);
              }
              if(vTgaHeader->type == TGA_TYPE_RLE_RGB_IMG)
              {
                /* le codage est :            */
                /* bit  15    = Interrupt bit */
                /* bits 10-14 = rouge         */
                /* bits 5-9   = vert          */
                /* bits 0-4   = blue          */

                /* si le bit Interrupt est mis  */
                /* alors ce point est invisible */
                if(((alpha_type == ALPHA_USE) || (alpha_type == ALPHA_PREMUL))
                  && (vuint16 & 0x8000))
                {
                  vpix = COL_FULL(
                    (uint8)((vuint16 >> 7)&0xF8),
                    (uint8)((vuint16 >> 2)&0xF8),
                    (uint8)(vuint16 << 3), 0x00);
                } else {
                  vpix = COL(
                    (uint8)((vuint16 >> 7)&0xF8),
                    (uint8)((vuint16 >> 2)&0xF8),
                    (uint8)(vuint16 << 3));
                }
              }
              buf_move += 2;
              break;
            case 24:
              /* un seul mode 24 bits */
              vpix = COL(buf_move[2], buf_move[1], buf_move[0]);
              buf_move += 3;
              break;
            case 32:
              /* un seul mode 32 bits */
              if(alpha_type == ALPHA_USE) {
                vpix = COL_FULL(buf_move[2], buf_move[1], buf_move[0],
                  buf_move[3]);
                buf_move += 4;
              } else if(alpha_type == ALPHA_PREMUL) {
                /* on supprime la premultiplication des couleurs */
                /* par le canal alpha avant le stockage.         */
                vpix = COL_FULL(
                  MIN(255,(((uint32)buf_move[2])*255)/buf_move[3]),
                  MIN(255,(((uint32)buf_move[1])*255)/buf_move[3]),
                  MIN(255,(((uint32)buf_move[0])*255)/buf_move[3]),
                  buf_move[3]);
                buf_move += 4;
              } else {
                /* on ne tient pas compte du canal alpha */
                vpix = COL(buf_move[2], buf_move[1], buf_move[0]);
                buf_move += 4;
              }
              break;
          }
           
          /* on recopie nb_rle fois le point que l'on vient de lire */
          for(;nb_rle!=0;nb_rle--)
          {
            *dst = vpix;
            dst += hstep;
          }
        } else {
          /* il y aura nb_raw point decrit apres */
          nb_raw = ((*(buf_move++)) & 0x7F) + 1;

          /* s'il ne reste plus de place pour lire les points  */
          /* c'est que le fichier n'est pas correct            */
          remain_size -= pix_bpp*nb_raw;
          if(remain_size < 0)
          { free(palette); free(buf); image_free(vimage); return(NULL); }

          /* on regarde si le nombre de points ne deborde pas la ligne */
          /* si c'est le cas c'est que le fichier n'est pas correct    */
          remain_width -= nb_raw;
          if(remain_width < 0)
          { free(palette); free(buf); image_free(vimage); return(NULL); }
          
          /* on lit les nb_raw point */
          for(;nb_raw!=0;nb_raw--)
          {
            /* on fait tous les cas      */
            switch(vTgaHeader->pix_bits) {
              case 8:
                if(vTgaHeader->type == TGA_TYPE_RLE_CMAP_IMG)
                {
                  /* on verifie que l'index existe dans la palette */
                  if((*buf_move < cmap_start) || (*buf_move > cmap_end))
                  { free(palette); free(buf); image_free(vimage); return(NULL); }
                  *dst = palette[(*buf_move)-cmap_start];
                }
                if(vTgaHeader->type == TGA_TYPE_RLE_BW_IMG)
                {
                  *dst = COL(*buf_move,*buf_move,*buf_move);
                }
                buf_move++;
                break;
              case 15:
                /* il n'y a qu'un mode 15 bits que l'on connait */
                vuint16 = TGA_UINT16(buf_move);
                *dst = COL(
                  (uint8)((vuint16 >> 7)&0xF8),
                  (uint8)((vuint16 >> 2)&0xF8),
                  (uint8)(vuint16 << 3));
                buf_move += 2;
                break;
              case 16:
                vuint16 = TGA_UINT16(buf_move);
                if(vTgaHeader->type == TGA_TYPE_RLE_CMAP_IMG)
                {
                  /* on verifie que l'indexe existe dans la palette */
                  if((vuint16 < cmap_start) || (vuint16 > cmap_end))
                  { free(palette); free(buf); image_free(vimage); return(NULL); }
                  *dst = palette[vuint16-cmap_start];
                }
                if(vTgaHeader->type == TGA_TYPE_RLE_BW_IMG)
                {
                  *dst = COL(vuint16>>8, vuint16>>8, vuint16>>8);
                }
                if(vTgaHeader->type == TGA_TYPE_RLE_RGB_IMG)
                {
                  /* le codage est :            */
                  /* bit  15    = Interrupt bit */
                  /* bits 10-14 = rouge         */
                  /* bits 5-9   = vert          */
                  /* bits 0-4   = blue          */

                  /* si le bit Interrupt est mis  */
                  /* alors ce point est invisible */
                  if(((alpha_type == ALPHA_USE)
                      || (alpha_type == ALPHA_PREMUL))
                    && (vuint16 & 0x8000))
                  {
                    *dst = COL_FULL(
                      (uint8)((vuint16 >> 7)&0xF8),
                      (uint8)((vuint16 >> 2)&0xF8),
                      (uint8)(vuint16 << 3), 0x00);
                  } else {
                    *dst = COL(
                      (uint8)((vuint16 >> 7)&0xF8),
                      (uint8)((vuint16 >> 2)&0xF8),
                      (uint8)(vuint16 << 3));
                  }
                }
                buf_move += 2;
                break;
              case 24:
                /* un seul mode 24 bits */
                *dst = COL(buf_move[2], buf_move[1], buf_move[0]);
                buf_move += 3;
                break;
              case 32:
                /* un seul mode 32 bits */
                if(alpha_type == ALPHA_USE) {
                  *dst = COL_FULL(buf_move[2], buf_move[1], buf_move[0],
                    buf_move[3]);
                  buf_move += 4;
                } else if(alpha_type == ALPHA_PREMUL) {
                  /* on supprime la premultiplication des couleurs */
                  /* par le canal alpha avant le stockage.         */
                  *dst = COL_FULL(
                    MIN(255,(((uint32)buf_move[2])*255)/buf_move[3]),
                    MIN(255,(((uint32)buf_move[1])*255)/buf_move[3]),
                    MIN(255,(((uint32)buf_move[0])*255)/buf_move[3]),
                    buf_move[3]);
                  buf_move += 4;
                } else {
                  /* on ne tient pas compte du canal alpha */
                  *dst = COL(buf_move[2], buf_move[1], buf_move[0]);
                  buf_move += 4;
                }
                break;
            }
            dst += hstep;
          }
        }
      } /* fin de parcours d'une ligne compressee */
      dst = dst_tmp;
    }
    /* on passe a la ligne suivante */
    dst += vstep;
  } /* fin du parcours des lignes */

  /* libere les buffers temporaires */
  free(palette); free(buf);
  /* retourne l'image lue */
  return(vimage);
}

/* Sauvegarde l'image pimage dans le fichier file     */
/* sous le format TGA. Return -1 = si ERREUR, 0 sinon */
int image_save_to_tga(image *pimage, char *file)
{
  /* A FAIRE */
  return(-1);
}
