/****************************************************/
/* Bibliothque qui charge des objets 3d  partir   */
/* de fichiers DXF                                  */
/* obj_3d_from_dxf.c                                */
/*                                                  */
/* Ecrit par : Daniel Lacroix (all rights reserved) */
/*                                                  */
/****************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <obj_3d_from_dxf.h>

#define DIFFUSE  0.4
#define SPECULAR 0.6
#define SHINE    20.0

typedef struct {
  uint32 code;
  char   value[2048];
} lexeme;

/************************************************/
/* Recupere une unitee de base des fichiers DXF */
static int get_lexeme(FILE *file, lexeme *lex)
{
  int length;
  char buf[2048];

  if(fgets(buf, 2048, file) == NULL)
    return(-1);
  lex->code = atoi(buf);
  
  if(fgets(lex->value, 2048, file) == NULL)
    return(-1);

  length = strlen(lex->value);

  /* supprime le retour  la ligne si il est prsent */
  if((length >= 1) && (lex->value[length-1] == '\n'))
  {
    lex->value[length-1] = 0;
    if((length >= 2) && (lex->value[length-2] == 13))
      lex->value[length-2] = 0;
  }

  /*fprintf(stderr,"code : %d, value : \"%s\"\n",lex->code,lex->value);*/
    
  return(0);
}
/************************************************/

/*************************************************/
/* Renvoi l'objet 3d contenu dans le fichier DXF */
/* ou NULL si impossible                         */
obj_3d *obj_3d_new_from_dxf(char *filename)
{
  FILE   *file = NULL;
  lexeme lex;
  obj_3d *vobj_3d = NULL;
  
  int    polyline_type = 0;
  int    nb_vertex     = 0;
  int    nb_face       = 0;
  int    current_face  = 0;
  int    current_vertex= 0;
  
  float  x = 0.0,y = 0.0,z = 0.0;
  boolean got_x = FALSE, got_y = FALSE, got_z = FALSE;
  int    f1 = 0,f2 = 0,f3 = 0;
  boolean got_f1 = FALSE, got_f2 = FALSE, got_f3 = FALSE;
  int    i;

  /* alloue la memoire pour la structure de base de l'objet 3D */
  if((vobj_3d = (obj_3d *)malloc(sizeof(obj_3d))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* mode de rendu */
  vobj_3d->fill_mode = FLAT;
  /* numero d'identifiant de l'objet (utilise dans obj_buf) */
  vobj_3d->id      = 0;
  /* nombre de polygone de cet objet */
  vobj_3d->nb_face = 0;
  /* definition de tout les polygones */
  vobj_3d->face    = NULL;
  /* nombre de point de cet objet */
  vobj_3d->nb_dot  = 0;
  /* definition de tout les points (utilise dans les polygones) */
  vobj_3d->dot     = NULL;
  /* tampon pour les calculs des points de l'objet apres transformation */
  vobj_3d->dot_tmp = NULL;
  /* vecteur normaux (utilise dans les polygones) */
  vobj_3d->normal  = NULL;
  /* vecteur normaux en chaque sommet */
  vobj_3d->normal_dot = NULL;
  /* transformation a appliquer sur l'objet (null si aucune) */
  vobj_3d->obj_mat = NULL;

  /* ouverture du fichier ASC */
  if((file = fopen(filename, "r")) == NULL)
    goto error;

  /* recupere le premier lexeme */
  if(get_lexeme(file,&lex)) goto error;
  
  while((lex.code != 0) || strcmp(lex.value,"EOF")) {

    if((lex.code != 0) || strcmp(lex.value,"SECTION")) goto error;
  
    /* on recupere le type de la section */
    if(get_lexeme(file,&lex)) goto error;
    
    if(lex.code != 2) goto error;

    /* en fonction du type de section on fait ce que l'on doit faire */
    if(!strcmp(lex.value,"ENTITIES"))
    {      
      if(get_lexeme(file,&lex)) goto error;
      
      if((lex.code != 0) || strcmp(lex.value,"POLYLINE"))
      {
        /* on saute cette section */
        do {
          if(get_lexeme(file,&lex)) goto error;
        } while((lex.code != 0) || strcmp(lex.value,"ENDSEC"));
      } else {
        
        do {
          if(get_lexeme(file,&lex)) goto error;
          
          switch(lex.code) {
            case 70 :
              polyline_type = atoi(lex.value);
              break;
            case 71 :
              nb_vertex = atoi(lex.value);
              break;
            case 72 :
              nb_face = atoi(lex.value);
              break;
          }
        } while(lex.code != 0);
        
        /* on verifie si on a recuperer un polyline que l'on connait */
        if(polyline_type != 64) goto error;

        /* il faudrait calculer dynamiquement ces valeurs */
        vobj_3d->nb_dot = nb_vertex;
        vobj_3d->nb_face = nb_face;
        
        /* alloue la mmoire pour les points */
        if((vobj_3d->dot = (dot_3d *)malloc(vobj_3d->nb_dot*sizeof(dot_3d))) == NULL)
        { perror("malloc failed "); exit(1); }

         /* alloue la mmoire pour les faces */
        if((vobj_3d->face = (face *)malloc(vobj_3d->nb_face*sizeof(face))) == NULL)
        { perror("malloc failed "); exit(1); }        
        
        while(!((lex.code == 0) && !strcmp(lex.value,"SEQEND"))) {

          if((lex.code != 0) || strcmp(lex.value,"VERTEX")) goto error;
          
          got_x = got_y = got_z = FALSE;
          got_f1 = got_f2 = got_f3 = FALSE;
          
          do {
            if(get_lexeme(file,&lex)) goto error;
            
            switch(lex.code) {
              case 10 :
                for(i=0;lex.value[i];i++) if(lex.value[i] == ',') lex.value[i] = '.';
                x = atof(lex.value);
                got_x = TRUE;
                break;
              case 20 :
                for(i=0;lex.value[i];i++) if(lex.value[i] == ',') lex.value[i] = '.';
                y = atof(lex.value);
                got_y = TRUE;
                break;
              case 30 :
                for(i=0;lex.value[i];i++) if(lex.value[i] == ',') lex.value[i] = '.';
                z = atof(lex.value);
                got_z = TRUE;
                break;
              case 71 :
                f1 = atoi(lex.value);
                got_f1 = TRUE;
                break;
              case 72 :
                f2 = atoi(lex.value);
                got_f2 = TRUE;
                break;
              case 73 :
                f3 = atoi(lex.value);
                got_f3 = TRUE;
                break;
            }
            
          } while(lex.code != 0);
          
          if(got_f1 && got_f2 && got_f3)
          {
            f1 = ABS(f1)-1; f2 = ABS(f2)-1; f3 = ABS(f3)-1;
            
            vobj_3d->face[current_face].pt1 = f1;
            vobj_3d->face[current_face].pt2 = f2;
            vobj_3d->face[current_face].pt3 = f3;
            /* fixe des proprites arbitraires */
            /* de rflexion de la lumire      */
            vobj_3d->face[current_face].diffuse    = DIFFUSE;
            vobj_3d->face[current_face].speculaire = SPECULAR;
            vobj_3d->face[current_face].shine      = SHINE;
            current_face++;
            
            /*fprintf(stderr,"face : %d %d %d\n",f1,f2,f3);*/
          
          } else if(got_x && got_y && got_z) {
            vobj_3d->dot[current_vertex].x = x;
            vobj_3d->dot[current_vertex].y = y;
            vobj_3d->dot[current_vertex].z = z;
            current_vertex++;
            /*fprintf(stderr,"dot : %f %f %f\n",x,y,z);*/
            
          } else goto error; 
        } /* fin while !SEQEND */

        /*fprintf(stderr,"nb_face : %d, nb_dot : %d\n",current_face,current_vertex);*/

        /* lit le reste du POLYLINE */        
        do {
          if(get_lexeme(file,&lex)) goto error;
        } while(lex.code != 0);
        
      }
      /* on verifie que l'on est bien a la fin de la section */
      if((lex.code != 0) || (strcmp(lex.value,"ENDSEC"))) goto error;
      
    } else {
      /* on saute cette section */
      do {
        if(get_lexeme(file,&lex)) goto error;
      } while((lex.code != 0) || strcmp(lex.value,"ENDSEC"));
    }
  
    if(get_lexeme(file,&lex)) goto error;
  }

  /* on ferme le fichier */
  fclose(file);
  
  return(vobj_3d);

  /* si il y a une erreur, on fait le menage avant de quitter */  
error:
  if(vobj_3d)
  {
    if(vobj_3d->face)
      free(vobj_3d->face);
    if(vobj_3d->dot)
      free(vobj_3d->dot);
    if(vobj_3d->dot_tmp)
      free(vobj_3d->dot_tmp);
    if(vobj_3d->normal)
      free(vobj_3d->normal);
    if(vobj_3d->normal_dot)
      free(vobj_3d->normal_dot);
    if(vobj_3d->obj_mat)
      free(vobj_3d->obj_mat);
    free(vobj_3d);
  }
  if(file)
    fclose(file);
  return(NULL);
}
/*************************************************/

