#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <general.h>
#include <image.h>
#include <image_trans.h>
#include <image_from_bmp.h>
#include <image_from_gif.h>
#include <image_from_tga.h>
#include <image_from_pbm.h>
#include <image_from_pgm.h>
#include <image_from_ppm.h>
#include <image_from_jpeg.h>

/* resize algorithme */
enum { NEAREST, BILINEAR, DCT, CATMULL, TV, TV2, MOST, DOWN, DOWN_ROUGHT };

/* input/output file format */
enum { TYPE_PPM, TYPE_PGM, TYPE_PBM, TYPE_BMP, TYPE_GIF, TYPE_TGA, TYPE_JPG, TYPE_UNK };

void usage(void)
{
  printf(
  "Usage: resize_image [OPTION] SRC_FILE -w NEW_WIDTH -h NEW_HEIGHT -o DST_FILE\n"\
  "resize the SRC_FILE jpeg image into DST_FILE with the\n"\
  "dimension (NEW_WIDTHxNEW_HEIGHT)\n\n"\
  "  -w, --width       result width (at least one of width or height is needed)\n"\
  "  -h, --height      result height (at least one of width or height is needed)\n"\
  "  -b, --bilinear    resize using a bilinear interpolation (DEFAULT)\n"\
  "  -n, --nearest     resize using the nearest dot\n"\
  "  -d, --dct         resize using a 2D DCT interpolation (very long)\n"\
  "  -s, --bspline     resize using a Catmull-Rom BSpline\n"\
  "  -t, --tv          resize using a TV like display\n"\
  "  -t2,--tv2         resize using a TV like display\n"\
  "  -m, --most        resize using a most present pixel\n"\
  "  -m2,--most2       resize using a most present pixel\n"\
  "  -do,--down        downsize\n"\
  "  -dr,--down_rought downsize but with less blur\n"\
  "  -it FILETYPE      force the image file format of the input image. Possible\n"\
  "                    choice are : PPM, PGM, PBM, BMP, GIF, TGA, JPG. If you\n"\
  "                    choose JPG, input file '-' means stdin.\n"\
  "  -ot FILETYPE      force the image file format of the output image. Possible\n"\
  "                    choice are : PPM, JPG. If you choose JPG, output file '-'\n"\
  "                    means stdout."\
  "      --help        display this help and exit\n"
  );
}

uint8 trans_type(char *type)
{
  uint8 res = TYPE_UNK;
  
  if(!strcmp(type,"PPM")) {
    res = TYPE_PPM;
  } else if(!strcmp(type,"PGM")) {
    res = TYPE_PGM;  
  } else if(!strcmp(type,"PBM")) {
    res = TYPE_PBM;
  } else if(!strcmp(type,"BMP")) {
    res = TYPE_BMP;
  } else if(!strcmp(type,"GIF")) {
    res = TYPE_GIF;
  } else if(!strcmp(type,"TGA")) {
    res = TYPE_TGA;
  } else if(!strcmp(type,"JPG")) {
    res = TYPE_JPG;
  }
  return(res);
}

uint8 guess_type(char *file_src)
{
  uint8 res = TYPE_UNK;
  
  if((strlen(file_src) >= 3) && (!strncmp(&(file_src[strlen(file_src)-3]),"ppm",3)))
  {
    res = TYPE_PPM;
  } else if((strlen(file_src) >= 3) && (!strncmp(&(file_src[strlen(file_src)-3]),"pgm",3))) {
    res = TYPE_PGM;
  } else if((strlen(file_src) >= 3) && (!strncmp(&(file_src[strlen(file_src)-3]),"pbm",3))) {
    res = TYPE_PBM;
  } else if((strlen(file_src) >= 3) && (!strncmp(&(file_src[strlen(file_src)-3]),"bmp",3))) {
    res = TYPE_BMP;
  } else if((strlen(file_src) >= 3) && (!strncmp(&(file_src[strlen(file_src)-3]),"gif",3))) {
    res = TYPE_GIF;
  } else if((strlen(file_src) >= 3) && (!strncmp(&(file_src[strlen(file_src)-3]),"tga",3))) {
    res = TYPE_TGA;
  } else if(((strlen(file_src) >= 3) && (!strncmp(&(file_src[strlen(file_src)-3]),"jpg",3))) ||
            ((strlen(file_src) >= 4) && (!strncmp(&(file_src[strlen(file_src)-4]),"jpeg",4)))) {
    res = TYPE_JPG;
  }
  return(res);
}

int main(int argc, char **argv)
{
  image *vimage = NULL, *vimage_resize = NULL;
  int i;
  char *file_src = NULL;
  char *file_dst = NULL;
  uint8 mode = BILINEAR;
  uint8 in_format  = TYPE_UNK;
  uint8 out_format = TYPE_UNK;
  int32 width  = 0;
  int32 height = 0;

  /* read command line arguments */  
  for(i = 1; i < argc; i++)
  {
    if(!strcmp(argv[i],"--help")) { usage(); exit(2); }
    else if((!strcmp(argv[i],"-b")) || (!strcmp(argv[i],"--bilinear")))    mode = BILINEAR;
    else if((!strcmp(argv[i],"-d")) || (!strcmp(argv[i],"--dct")))         mode = DCT;
    else if((!strcmp(argv[i],"-s")) || (!strcmp(argv[i],"--bspline")))     mode = CATMULL;
    else if((!strcmp(argv[i],"-t")) || (!strcmp(argv[i],"--tv")))          mode = TV;
    else if((!strcmp(argv[i],"-t2"))|| (!strcmp(argv[i],"--tv2")))         mode = TV2;
    else if((!strcmp(argv[i],"-do"))|| (!strcmp(argv[i],"--down")))        mode = DOWN;
    else if((!strcmp(argv[i],"-dr"))|| (!strcmp(argv[i],"--down_rought"))) mode = DOWN_ROUGHT;
    else if((!strcmp(argv[i],"-m")) || (!strcmp(argv[i],"--most")))        mode = MOST;
    else if((!strcmp(argv[i],"-n")) || (!strcmp(argv[i],"--nearest")))     mode = NEAREST;
    else if(!strcmp(argv[i],"-it")) {
      i++;
      if((i < argc) && (in_format == TYPE_UNK)) {
        in_format = trans_type(argv[i]);
        if(in_format == TYPE_UNK)  { usage(); exit(2); }
      } else { usage(); exit(2); }
    } else if(!strcmp(argv[i],"-ot")) {
      i++;
      if((i < argc) && (out_format == TYPE_UNK)) {
        out_format = trans_type(argv[i]);
        if(out_format == TYPE_UNK)  { usage(); exit(2); }
      } else { usage(); exit(2); }
    } else if(!strcmp(argv[i],"-o")) {
      i++;
      if((i < argc) && (file_dst == NULL)) {
        file_dst = argv[i];
      } else { usage(); exit(2); }
    } else if((!strcmp(argv[i],"-w")) || (!strcmp(argv[i],"--width"))) {
      i++;
      if((i < argc) && (width == 0)) {
        width = atoi(argv[i]);
        if(width <= 0) { usage(); exit(2); }
      } else { usage(); exit(2); }
    } else if((!strcmp(argv[i],"-h")) || (!strcmp(argv[i],"--height"))) {
      i++;
      if((i < argc) && (height == 0)) {
        height = atoi(argv[i]);
        if(height <= 0) { usage(); exit(2); }
      } else { usage(); exit(2); }
    } else {
      if(file_src == NULL) {
        file_src = argv[i];
        /* file src MUST be followed by its new dimension */
/*        i++; if(i >= argc) { usage(); exit(2); }
        width  = atoi(argv[i]);
        i++; if(i >= argc) { usage(); exit(2); }
        height = atoi(argv[i]);
*/        /* negative or null is not allowed */
/*        if((width <= 0) || (height <= 0)) { usage(); exit(2); }*/
      } else { usage(); exit(2); }
    }
  }

  if(((width <= 0) && (height <= 0)) || (file_src == NULL) || (file_dst == NULL))
  { usage(); exit(2); }

  if(in_format == TYPE_UNK) {
    in_format = guess_type(file_src);
    if(in_format == TYPE_UNK) {
      fprintf(stderr,"input file format unknown.\n");
      exit(2);
    }
  }

  if(out_format == TYPE_UNK) {
    out_format = guess_type(file_dst);
    if(out_format == TYPE_UNK) {
      fprintf(stderr,"output file format unknown.\n");
      exit(2);
    }
  }

  switch(in_format) {
    case TYPE_PPM :
      vimage = image_new_from_ppm(file_src);
      break;
    case TYPE_PGM :
      vimage = image_new_from_pgm(file_src);
      break;
    case TYPE_PBM :
      vimage = image_new_from_pbm(file_src);
      break;
    case TYPE_BMP :
      vimage = image_new_from_bmp(file_src);
      break;
    case TYPE_GIF :
      vimage = image_new_from_gif(file_src);
      break;
    case TYPE_TGA :
      vimage = image_new_from_tga(file_src);
      break;
    case TYPE_JPG :    
      if(!strcmp(file_src,"-"))
        vimage = image_new_from_jpeg_stream(stdin);
      else
        vimage = image_new_from_jpeg(file_src);
      break;
  }

  if(vimage == NULL)
  { printf("ERROR while loading source file \"%s\"\n",file_src); exit(2); }

  /*
   * If we only know the widht or the height build 
   * the other using the image ratio
   */
  if(height <= 0) {
    height = (width * vimage->height) / vimage->width;
  }
  if(width <= 0) {
    width = (height * vimage->width) / vimage->height;
  }

  /* create the destination image */  
  vimage_resize = image_new(width,height);

  switch(mode)
  {
    case NEAREST     :
      image_resize_nearest(vimage_resize,vimage,width,height);
      break;
    case BILINEAR    :
      image_resize_bilinear(vimage_resize,vimage,width,height);
      break;
    case DCT         :
      printf("Be patient it can be long (1024x768 destination image takes"\
             " 15 minutes on a PII 333Mhz)\n\n");
      image_resize_dct(vimage_resize,vimage,width,height);
      break;
    case CATMULL     :
      printf("Be patient it can be long\n\n");
      image_resize_catmull_rom(vimage_resize,vimage,width,height);
      break;
    case TV          :
      image_resize_tv(vimage_resize,vimage);
      break;
    case TV2         :
      image_resize_tv2(vimage_resize,vimage);
      break;
    case MOST        :
      image_resize_most(vimage_resize,vimage);
      break;
    case DOWN        :
      image_downsize(vimage_resize,vimage,width,height);
      break;
    case DOWN_ROUGHT :
      image_downsize_rought(vimage_resize,vimage,width,height);
      break;
  }

  /* save the resulting image */
  switch(out_format) {
    case TYPE_PPM :
      image_save_to_ppm(vimage_resize, file_dst);
      break;
    case TYPE_JPG :
      if(!strcmp(file_dst,"-"))
	image_save_to_jpeg_stream(vimage_resize, stdout);
      else
	image_save_to_jpeg(vimage_resize, file_dst);
      break;
    default :
      fprintf(stderr,"unsupported output file format.\n");
      exit(2);
  }

  return(0);
}
