# Code for the Gnome route editor
#
# Copyright (C) 1996-2001 Chris Lawrence
# This file may be freely distributed under the terms of the RoutePlanner
# license.  A copy should appear as 'LICENSE' in the archive that this
# file was included in.

import gnome, rpdbase, rpcountry, GtkExtra

gnome.app_id = 'RouteEdit'
gnome.app_version = rpdbase.VERID

import string, sys, os, glob, commands, time, libglade, GDK
import rpcity, rproute, bisect
from rpcitylist import CityList
from rpprogress import ProgressWin

from gtk import *
from gnome.ui import *
from GDK import *
from rpunits import *
import gnome.config

GUIFILE = '/usr/share/routeplanner/routeedit.glade'

def clip(minval, x, maxval):
    return min(max(minval, x), maxval)

def GtkOptionMenu_selected(optionmenu):
    menu = optionmenu.get_menu()
    children = menu.children()
    selected = menu.get_active()
    i = 0

    for child in children:
        if child == selected: return i
        i = i+1

class Application:
    def __init__(self, filename=''):
        if os.path.exists('routeedit.glade'):
            self.wtree = libglade.GladeXML('routeedit.glade')
        else:
            self.wtree = libglade.GladeXML(GUIFILE)
        
        self.appwin = self.wtree.get_widget('app1')
        dict = {}
        for key in dir(self.__class__):
            dict[key] = getattr(self, key)

        self.wtree.signal_autoconnect(dict)
        self.db = rpdbase.RPDatabase()

        self.citylist = self.wtree.get_widget('citylist')
        self.roadlist = self.wtree.get_widget('roadlist')

        self.dbname = ''
        self.modified = FALSE
        self.new_database()
        if filename:
            self.open_database(filename)

        menu = GtkMenu()
        menu2 = GtkMenu()
        menu.show()
        menu2.show()
        for type in rpdbase.classify:
            item = GtkMenuItem(type)
            item.show()
            menu.append(item)
            item2 = GtkMenuItem(type)
            item2.show()
            menu2.append(item2)

        selector = self.wtree.get_widget('routesel')
        selector.set_menu(menu)
        selector = self.wtree.get_widget('routesel2')
        selector.set_menu(menu2)

        menu = GtkMenu()
        menu.show()

        countries = rpcountry.countries.items()
        countries.sort(lambda x, y: cmp(x[1], y[1]))
        self.countrylookup = countries

        countries = []
        for code, country in self.countrylookup:
            countries.append(country)

        combo = self.wtree.get_widget('countrynamecombo')
        combo.set_popdown_strings(countries)
        combo.set_value_in_list(TRUE, FALSE)

        self.citywin = None
        self.findtext = ''
        self.appwin.connect("destroy", self.quit) 
        self.appwin.show()

    def set_titlebar(self):
        modw = ''
        if self.modified: modw = '* '

        if self.dbname:
            self.appwin.set_title('RouteEdit: '+modw+self.dbname)
        else:
            self.appwin.set_title('RouteEdit: '+modw+'(untitled)')

    def citybuttons(self, state=TRUE):
        self.wtree.get_widget('removecity').set_sensitive(state)
        self.wtree.get_widget('cityprops').set_sensitive(state)
        self.wtree.get_widget('addroute').set_sensitive(state)

    def routebuttons(self, state=TRUE):
        widgets = ['extendroute', 'removeroute', 'jump', 'routeproperties',
                   'splitroute']
        for widget in widgets:
            self.wtree.get_widget(widget).set_sensitive(state)

    def new_database(self):
        del self.db
        self.db = rpdbase.RPDatabase()
        self.citylist.clear()
        self.roadlist.clear()
        self.modified = FALSE
        self.dbname = ''
        self.update_propswin()
        self.citybuttons(FALSE)
        self.routebuttons(FALSE)
        self.wtree.get_widget('routeframe').set_label("Routes")
        self.set_titlebar()

    def open_database(self, filename):
        progwin = ProgressWin(os.path.basename(filename)+'...', self.appwin)
        progwin.show()
        try:
            self.db = rpdbase.RPDatabase(filename, quiet=1,
                                         progressbar=progwin)
        except (rpdbase.invalidformat, IOError), x:
            progwin.destroy()
            error = GnomeErrorDialog('Unable to open '+filename+':\n\n'+
                                     str(x), self.appwin)
            error.run_and_close()
            return

        progwin.destroy()
        #self.make_widgets_sensitive()
        self.roadlist.clear()
        self.citylist.freeze()
        self.citylist.clear()

        i = 0
        for city in self.db.cities:
            self.citylist.append( [str(city)] )
            self.citylist.set_row_data(i, i)
            i = i+1
            
        self.citylist.thaw()
        self.citybuttons(FALSE)
        self.wtree.get_widget('routeframe').set_label("Routes")
        self.routebuttons(FALSE)
        self.modified = FALSE
        self.dbname = filename
        self.update_propswin()
        self.set_titlebar()

    def save_database(self, filename):
        progwin = ProgressWin(os.path.basename(filename)+'...', self.appwin)
        progwin.show()
        try:
            self.db.Save(filename, progressbar=progwin)
        except (rpdbase.invalidformat, IOError, OSError), x:
            progwin.destroy()
            error = GnomeErrorDialog('Unable to save '+filename+':\n\n'+
                                     str(x), self.appwin)
            error.run_and_close()
            return

        #self.make_widgets_sensitive()
        progwin.destroy()
        self.modified = FALSE
        self.dbname = filename
        self.set_titlebar()

    def quit(self, *args):
        if self.modified:
            x = GnomeQuestionDialog('Database modified.  Do you really want '
                                    'to quit?', self.really_quit,
                                    self.appwin)
            x.run_and_close()
            return TRUE
        else:
            self.really_quit()

    def really_quit(self, clicked=0):
        if not clicked:
            self.appwin.hide()
            raise SystemExit
        else:
            self.appwin.show()

    def on_new_file1_activate(self, *args):
        if self.modified:
            x = GnomeQuestionDialog('Database modified.  Do you really want '
                                    'to start a new one?', self.really_new,
                                    self.appwin)
            x.run_and_close()
        else:
            self.really_new()

    def really_new(self, clicked=0):
        if not clicked: self.new_database()

    def on_open1_activate(self, *args):
        if self.modified:
            x = GnomeQuestionDialog('Database modified.  Do you really want '
                                    'to open another?', self.really_open,
                                    self.appwin)
            x.run_and_close()
        else:
            self.really_open()

    def really_open(self, clicked=0):
        if not clicked:
            self.fw = GtkFileSelection('Select database')
            self.fw.set_transient_for(self.appwin)
            self.fw.selection_entry.connect("activate",
                                             self.on_open_ok_clicked)
            self.fw.ok_button.connect("pressed", self.on_open_ok_clicked)
            self.fw.cancel_button.connect("pressed", self.fw.destroy)
            self.fw.set_modal(TRUE)
            self.fw.hide_on_delete(TRUE)
            self.fw.set_filename(self.dbname)
            self.fw.show()

    def update_propswin(self):
        self.wtree.get_widget('author').set_text(self.db.author)
        self.wtree.get_widget('authoremail').set_text(self.db.author_email)
        self.wtree.get_widget('comment').set_text(self.db.comment)
        xmap = { 'metric': UNITS_METRIC, 'units_us': UNITS_US,
                 'imperial': UNITS_IMPERIAL }
        for widget, units in xmap.items():
            self.wtree.get_widget(widget).set_active(FALSE)
            if self.db.units == units:
                self.wtree.get_widget(widget).set_active(TRUE)

    def on_propswin_changed(self, *args):
        changed = FALSE
        
        if self.db.author != self.wtree.get_widget('author').get_text():
            changed = TRUE
        elif self.db.author_email != \
             self.wtree.get_widget('authoremail').get_text():
            changed = TRUE
        elif self.db.comment != self.wtree.get_widget('comment').get_text():
            changed = TRUE
        else:
            xmap = { 'metric': UNITS_METRIC, 'units_us': UNITS_US,
                     'imperial': UNITS_IMPERIAL }
            for widget, units in xmap.items():
                if self.wtree.get_widget(widget).get_active():
                    if self.db.units != units:
                        changed = TRUE

        self.wtree.get_widget('dbaseproperties').set_modified(changed)

    def on_propswin_apply(self, *args):
        self.db.author = self.wtree.get_widget('author').get_text()
        self.db.author_email = self.wtree.get_widget('authoremail').get_text()
        self.db.comment = self.wtree.get_widget('comment').get_text()
        xmap = { 'metric': UNITS_METRIC, 'units_us': UNITS_US,
                 'imperial': UNITS_IMPERIAL }
        for widget, units in xmap.items():
            if self.wtree.get_widget(widget).get_active():
                self.db.units = units

        self.modified = TRUE
        self.set_titlebar()

    def on_open_ok_clicked(self, *args):
        self.fw.hide()
        self.open_database(self.fw.get_filename())
        self.fw.destroy()

    def on_save1_activate(self, *args):
        if not self.dbname:
            self.on_save_as1_activate()
        else:
            self.save_database(self.dbname)

    def on_save_as1_activate(self, *args):
        self.fw = GtkFileSelection('Save database as')
        self.fw.set_transient_for(self.appwin)
        self.fw.selection_entry.connect("activate",
                                         self.on_save_ok_clicked)
        self.fw.ok_button.connect("pressed", self.on_save_ok_clicked)
        self.fw.cancel_button.connect("pressed", self.fw.destroy)
        self.fw.set_modal(TRUE)
        self.fw.hide_on_delete(TRUE)
        self.fw.set_filename(self.dbname)
        self.fw.show()
        
    def on_save_ok_clicked(self, *args):
        self.fw.hide()
        self.save_database(self.fw.get_filename())
        self.fw.destroy()

    def on_about1_activate(self, *args):
        about = self.wtree.get_widget('about')
        about.close_hides(TRUE)
        about.show()

    def on_about1_close(self, *args):
        self.wtree.get_widget('about').hide()

    def on_properties1_activate(self, *args):
        props = self.wtree.get_widget('dbaseproperties')
        props.close_hides(TRUE)
        props.show()

    def on_check_database_activate(self, *args):
        bits = {}
        
        for city in self.db.cities:
            routes = city.roads
            bits[len(routes)] = bits.get(len(routes), []) + [city.printable()]

        output = ""
        for i in range(0, 3):
            output = output + "Cities with %d routes:\n  " % i
            output = output + (string.join(bits.get(i, []), "\n  ") or
                               "None") + "\n\n"
        output = output[:-2]
        self.wtree.get_widget('check_less').show_string(output)
        self.wtree.get_widget('checkdialog').show()

    def on_citylist_select_row(self, *args):
        self.roadlist.freeze()
        if self.roadlist.selection:
            row = self.roadlist.selection[0]
            thisroute = self.roadlist.get_row_data(row)[0]
        else:
            thisroute = None
        
        self.roadlist.clear()
        city = self.db.cities[args[1]]
        cityname = city.printable()
        self.wtree.get_widget('routeframe').set_label('Routes from '+cityname)
        i = 0
        selected = FALSE
        for route in city.roads:
            picked = route[2]
            self.roadlist.append([ str(picked), route[0].name,
                                   str(route[0].distance),
                                   rpdbase.declassify[route[0].speedcode] ])
            self.roadlist.set_row_data(i, route)
            if thisroute is route[0]:
                self.roadlist.select_row(i, 0)
                selected = TRUE
            i = i+1

        self.citybuttons(TRUE)
        self.routebuttons(selected)
        self.roadlist.thaw()

    def on_roadlist_select_row(self, *args):
        self.routebuttons(TRUE)

    def on_find1_activate(self, *args):
        self.wtree.get_widget('findwin').show()
        self.wtree.get_widget('citytofind').grab_focus()

    def on_startfind_clicked(self, *args):
        self.wtree.get_widget('findwin').hide()
        text = self.wtree.get_widget('citytofind').get_text()
        self.findtext = text
        self.citylist.unselect_all()
        self.on_find_again1_activate()

    def on_find_again1_activate(self, *args):
        if not self.findtext:
            GnomeWarningDialog('Nothing to find.', self.appwin).run_and_close()
            return

        self.findtext = string.lower(self.findtext)
        found = FALSE
        if self.citylist.selection:
            selection = self.citylist.selection[0]
            i = selection+1
            tosearch = self.db.cities[i:]
            word = 'more '
        else:
            i = 0
            tosearch = self.db.cities
            word = ''

        self.citylist.unselect_all()
        for city in tosearch:
            if string.find(string.lower(str(city)), self.findtext) >= 0:
                self.citylist.select_row(i, 0)
                self.citylist.moveto(i)
                found = TRUE
                break
            
            i = i+1

        if not found:
            GnomeWarningDialog('No '+word+'cities matching '+self.findtext+'.',
                               self.appwin).run_and_close()

    def on_citylist_button_press_event(self, button, event, *args):
        if event.type == GDK._2BUTTON_PRESS and event.button == 1 \
           and self.citylist.selection:
            self.on_cityprops_clicked()

    def on_countrybutton_clicked(self, *args):
        self.wtree.get_widget('countrysel').show()

    def on_cityok_clicked(self, *args):
        citywin = self.wtree.get_widget('cityprop')
        if self.editing >= 0:
            city = self.db.cities[self.editing]
        else:
            city = rpcity.City()

        cname, sname = (self.wtree.get_widget('cityname').get_text(),
                        self.wtree.get_widget('statename').get_text())
        sname = rpdbase.expand_state(sname)
        country = self.wtree.get_widget('countryname').get_text()
        code = rpcountry.code_for[country]

        cities = self.db.CitiesMatching(cname+', '+sname)
        cities = filter(lambda x, city=city: x is not city, cities)
	for c in cities:
            if c.city == cname and c.state == sname and \
               c.country == code:
                GnomeErrorDialog('There is already a city with this name.',
                                 citywin)
                return

        city.update_info(cname, sname, code,
                         self.wtree.get_widget('longitude').get_value(),
                         self.wtree.get_widget('latitude').get_value(),
                         self.wtree.get_widget('interchange').get_active())

        self.modified = TRUE
        self.set_titlebar()
        if self.editing >= 0:
            del self.db.cities[self.editing]

        # Insert city in list
        thiscity = city
        self.citylist.freeze()
        self.db.cities.append(city)
        self.db.Sort()

        i = 0
        self.citylist.clear()
        for city in self.db.cities:
            self.citylist.append( [str(city)] )
            self.citylist.set_row_data(i, i)
            if city is thiscity:
                self.citylist.select_row(i, 0)
                self.citylist.moveto(i)
            i = i+1

        self.citylist.thaw()

    def on_addcity_clicked(self, *args):
        citywin = self.wtree.get_widget('cityprop')
        self.wtree.get_widget('cityname').set_text('')
        #self.wtree.get_widget('statename').set_text('')
        #self.wtree.get_widget('countryname').set_text('United States')
        self.wtree.get_widget('longitude').set_value(0.0)
        self.wtree.get_widget('latitude').set_value(0.0)
        self.wtree.get_widget('interchange').set_active(FALSE)
        self.wtree.get_widget('cityname').grab_focus()
        self.editing = -1
        citywin.show()

    def on_statename_activate(self, *args):
        sname = self.wtree.get_widget('statename').get_text()
        if sname and len(sname) == 2:
            sname = rpdbase.expand_state(sname)

        if sname:
            self.wtree.get_widget('statename').set_text(sname)
            cname = rpdbase.autodetect_country(sname)
            if cname:
                cname = rpcountry.expand_country(cname)
                self.wtree.get_widget('countryname').set_text(cname)

    def on_cityprops_clicked(self, *args):
        if not self.citylist.selection:
            return

        self.editing = self.citylist.selection[0]
        city = self.db.cities[self.editing]

        citywin = self.wtree.get_widget('cityprop')
        self.wtree.get_widget('cityname').set_text(city.city)
        self.wtree.get_widget('statename').set_text(city.state)
        cname = rpcountry.expand_country(city.country)
        self.wtree.get_widget('countryname').set_text(cname)
        self.wtree.get_widget('longitude').set_value(city.longitude)
        self.wtree.get_widget('latitude').set_value(city.latitude)
        self.wtree.get_widget('interchange').set_active(city.intersection)
        self.wtree.get_widget('cityname').grab_focus()
        citywin.show()

    def on_removecity_clicked(self, *args):
        if not self.citylist.selection:
            return

        GnomeQuestionDialog('Remove this city and all connected routes?',
                            self.do_remove_city, self.appwin).run_and_close()

    def do_remove_city(self, chosen):
        if chosen:
            return
        
        item = self.citylist.selection[0]
        thiscity = self.db.cities[item]
        roads = map(lambda x: x[0], thiscity.roads)
        for road in roads:
            for city in road.city:
                x = city.roads
                city.roads = []
                for route in x:
                    if route[0] != road:
                        city.roads.append(route)
            
            try:
                self.db.routes.remove(road)
            except ValueError:
                pass

        del roads
        del self.db.cities[item]
        self.db.Rehash()

        self.citylist.freeze()

        i = 0
        self.citylist.clear()
        for city in self.db.cities:
            self.citylist.append( [str(city)] )
            self.citylist.set_row_data(i, i)
            i = i+1

        self.citylist.thaw()
        self.citylist.select_row(item, 0)
        self.citylist.moveto(item)

        self.modified = TRUE
        self.set_titlebar()

    def on_roadlist_button_press_event(self, button, event, *args):
        if event.type == GDK._2BUTTON_PRESS and event.button == 1 \
           and self.roadlist.selection:
            self.on_routeproperties_clicked()

    def on_jump_clicked(self, *args):
        sel = self.roadlist.selection
        if not sel: return

        road = self.roadlist.get_row_data(sel[0])
        item = self.db.cities.index(road[2])
        self.citylist.select_row(item, 0)
        self.citylist.moveto(item)

    def on_removeroute_clicked(self, *args):
        sel = self.roadlist.selection
        if not sel: return

        row = sel[0]
        road = self.roadlist.get_row_data(row)[0]
        for city in road.city:
            x = city.roads
            city.roads = []
            for route in x:
                if route[0] != road:
                    city.roads.append(route)
            
        self.db.routes.remove(road)
        self.roadlist.remove(row)

        self.modified = TRUE
        self.set_titlebar()

    def show_via_entries(self, show=TRUE):
        for widget in ['vialabel', 'city3', 'choosecity3', 'exitlabel3',
                       'exit3', 'rightbox']:
            if show:
                self.wtree.get_widget(widget).show()
            else:
                self.wtree.get_widget(widget).hide()
        self.update_box_labels()

    def update_box_labels(self):
        fullbox = self.wtree.get_widget('rightbox').flags(VISIBLE)
        if fullbox:
            via = self.wtree.get_widget('city3').get_text()
        else:
            via = self.wtree.get_widget('city2').get_text()
        
        self.wtree.get_widget('leftbox').set_label(
            'From '+self.wtree.get_widget('city1').get_text()+' to '+via)
        if fullbox:
            self.wtree.get_widget('rightbox').set_label(
                'From '+via+' to '+self.wtree.get_widget('city2').get_text())
        self.wtree.get_widget('leftbox').draw_default()
        self.wtree.get_widget('rightbox').draw_default()

    def on_addroute_clicked(self, *args):
        thiscity = self.db.cities[self.citylist.selection[0]]
        self.wtree.get_widget('city1').set_text(str(thiscity))
        for widget in ['routename', 'city2', 'exit1', 'exit2']:
            self.wtree.get_widget(widget).set_text('')
        for widget in ['toll', 'scenic', 'ferry']:
            self.wtree.get_widget(widget).set_active(FALSE)
        if self.db.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(TRUE)
        else:
            self.wtree.get_widget('miles').set_active(TRUE)
        self.wtree.get_widget('distance').set_value(0)
        self.wtree.get_widget('speedtime').set_value(0)
        self.editing = -1
        selector = self.wtree.get_widget('routesel')
        selector.set_history(0)
        self.show_via_entries(FALSE)
        self.update_box_labels()
        self.wtree.get_widget('routeok').set_sensitive(FALSE)
        self.wtree.get_widget('routeprop').show()
        self.wtree.get_widget('routename').grab_focus()
        self.mode = 'add'
        
    def on_routeproperties_clicked(self, *args):
        thiscity = self.db.cities[self.citylist.selection[0]]
        sel = self.roadlist.selection
        if not sel: return

        row = sel[0]
        road = self.roadlist.get_row_data(row)
        rinfo, thatcity = road[1], road[2]
        thisexit, thatexit = road[0].exits[rinfo], road[0].exits[1-rinfo]
        self.editing = row

        self.wtree.get_widget('routename').set_text(road[0].name)
        self.wtree.get_widget('city1').set_text(str(thiscity))
        self.wtree.get_widget('city2').set_text(str(thatcity))
        self.wtree.get_widget('exit1').set_text(thisexit)
        self.wtree.get_widget('exit2').set_text(thatexit)

        for widget in ['toll', 'scenic', 'ferry']:
            self.wtree.get_widget(widget).set_active(
                road[0].flagmap.get(widget, FALSE))

        if road[0].distance.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(TRUE)
        else:
            self.wtree.get_widget('miles').set_active(TRUE)
        
        self.wtree.get_widget('distance').set_value(road[0].distance.value)

        pos = rpdbase.classify.index(rpdbase.declassify[road[0].speedcode])

        selector = self.wtree.get_widget('routesel')
        selector.set_history(pos)
        
        self.wtree.get_widget('speedtime').set_value(road[0].speed)
        self.show_via_entries(FALSE)
        self.wtree.get_widget('routeok').set_sensitive(TRUE)
        self.wtree.get_widget('routename').grab_focus()
        self.wtree.get_widget('routeprop').show()
        self.mode = 'edit'

    def on_routeok_clicked(self, *args):
        name = self.wtree.get_widget('routename').get_text()
        city1 = self.wtree.get_widget('city1').get_text()
        city2 = self.wtree.get_widget('city2').get_text()
        city3 = self.wtree.get_widget('city3').get_text()
        exit1 = self.wtree.get_widget('exit1').get_text()
        exit2 = self.wtree.get_widget('exit2').get_text()
        exit3 = self.wtree.get_widget('exit3').get_text()

        if not name or not city1 or not city2 or \
           (self.mode in ('break', 'extend') and not city3):
            win = self.wtree.get_widget('routeprop')
            GnomeErrorDialog('You must include the route name and cities.',
                             win).run_and_close()
            return

        if self.wtree.get_widget('km').get_active():
            units = UNITS_METRIC
        else:
            units = UNITS_US

        if self.wtree.get_widget('km2').get_active():
            units2 = UNITS_METRIC
        else:
            units2 = UNITS_US

        distance = Distance(self.wtree.get_widget('distance').get_value(),
                            units)
        distance2 = Distance(self.wtree.get_widget('distance2').get_value(),
                             units2)
        if distance == 0 or (self.mode != 'edit' and distance2 == 0):
            win = self.wtree.get_widget('routeprop')
            GnomeErrorDialog('Distances must be greater than zero.',
                             win).run_and_close()
            return

        if self.editing >= 0:
            road = self.roadlist.get_row_data(self.editing)[0]
            for city in road.city:
                x = city.roads
                city.roads = []
                for route in x:
                    if route[0] != road:
                        city.roads.append(route)
            try:
                self.db.routes.remove(road)
            except:
                pass

        flagmap = {}
        for widget in ['toll', 'scenic', 'ferry']:
            flagmap[widget] = self.wtree.get_widget(widget).get_active()

        speed = self.wtree.get_widget('speedtime').get_value_as_int()
        code = GtkOptionMenu_selected(self.wtree.get_widget('routesel'))
        speedcode = rpdbase.classifications[rpdbase.classify[code]]

        if self.mode in ('add', 'edit'):
            route = rproute.Route([ city1, city2, str(distance), speed,
                                    speedcode, exit1, exit2, name, flagmap ],
                                  self.db.cityhash)
            self.db.routes.append(route)
        else:
            route = rproute.Route([ city1, city3, str(distance), speed,
                                    speedcode, exit1, exit3, name, flagmap ],
                                  self.db.cityhash)
            self.db.routes.append(route)

            flagmap = {}
            for widget in ['toll', 'scenic', 'ferry']:
                flagmap[widget] = self.wtree.get_widget(widget+'2').\
                                  get_active()

            speed = self.wtree.get_widget('speedtime2').get_value_as_int()
            code = GtkOptionMenu_selected(self.wtree.get_widget('routesel2'))
            speedcode = rpdbase.classifications[rpdbase.classify[code]]

            route2 = rproute.Route([ city2, city3, str(distance2), speed,
                                     speedcode, exit2, exit3, name, flagmap ],
                                   self.db.cityhash)
            self.db.routes.append(route2)
            if self.mode == 'extend':
                # Highlight the new segment
                route, route2 = route2, route
            
        thisroute = route
        self.modified = TRUE
        self.set_titlebar()

        pos = self.citylist.selection[0]
        thiscity = self.db.cities[pos]
        selected = FALSE

        self.roadlist.clear()
        i = 0
        for route in thiscity.roads:
            picked = route[2]
            self.roadlist.append([ str(picked), route[0].name,
                                   str(route[0].distance),
                                   rpdbase.declassify[route[0].speedcode] ])
            self.roadlist.set_row_data(i, route)
            if route[0] is thisroute:
                self.roadlist.select_row(i, 0)
                self.roadlist.moveto(i)
                selected = TRUE
            i = i+1

        self.citybuttons(TRUE)
        self.routebuttons(selected)
        self.roadlist.thaw()

    def on_extendroute_clicked(self, *args):
        thiscity = self.db.cities[self.citylist.selection[0]]
        sel = self.roadlist.selection
        if not sel: return

        row = sel[0]
        road = self.roadlist.get_row_data(row)
        rinfo, thatcity = road[1], road[2]
        thisexit, thatexit = road[0].exits[rinfo], road[0].exits[1-rinfo]
        self.editing = row

        self.show_via_entries(TRUE)
        self.wtree.get_widget('routename').set_text(road[0].name)
        self.wtree.get_widget('city3').set_text(str(thiscity))
        self.wtree.get_widget('city2').set_text('')
        self.wtree.get_widget('city1').set_text(str(thatcity))
        
        self.wtree.get_widget('exit3').set_text(thisexit)
        self.wtree.get_widget('exit2').set_text('')
        self.wtree.get_widget('exit1').set_text(thatexit)

        for widget in ['toll', 'scenic', 'ferry']:
            status = road[0].flagmap.get(widget, FALSE)
            self.wtree.get_widget(widget).set_active(status)
            self.wtree.get_widget(widget+'2').set_active(status)

        if road[0].distance.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(TRUE)
            self.wtree.get_widget('km2').set_active(TRUE)
        else:
            self.wtree.get_widget('miles').set_active(TRUE)
            self.wtree.get_widget('miles2').set_active(TRUE)
        
        self.wtree.get_widget('distance').set_value(float(road[0].distance))
        self.wtree.get_widget('distance2').set_value(0)

        pos = rpdbase.classify.index(rpdbase.declassify[road[0].speedcode])

        selector = self.wtree.get_widget('routesel')
        selector.set_history(pos)
        selector = self.wtree.get_widget('routesel2')
        selector.set_history(pos)
        
        self.wtree.get_widget('speedtime').set_value(road[0].speed)
        self.wtree.get_widget('speedtime2').set_value(road[0].speed)
        #self.wtree.get_widget('lock').hide()
        self.wtree.get_widget('lock').set_sensitive(FALSE)
        self.wtree.get_widget('lock').set_active(FALSE)
        self.wtree.get_widget('routeok').set_sensitive(FALSE)
        self.update_box_labels()
        self.wtree.get_widget('routename').grab_focus()
        self.wtree.get_widget('routeprop').show()
        self.mode = 'extend'

    def on_splitroute_clicked(self, *args):
        thiscity = self.db.cities[self.citylist.selection[0]]
        sel = self.roadlist.selection
        if not sel: return

        row = sel[0]
        road = self.roadlist.get_row_data(row)
        rinfo, thatcity = road[1], road[2]
        thisexit, thatexit = road[0].exits[rinfo], road[0].exits[1-rinfo]
        self.editing = row

        self.show_via_entries(TRUE)
        self.wtree.get_widget('routename').set_text(road[0].name)
        self.wtree.get_widget('city1').set_text(str(thiscity))
        self.wtree.get_widget('city2').set_text(str(thatcity))
        self.wtree.get_widget('city3').set_text('')
        
        self.wtree.get_widget('exit1').set_text(thisexit)
        self.wtree.get_widget('exit2').set_text(thatexit)
        self.wtree.get_widget('exit3').set_text('')

        for widget in ['toll', 'scenic', 'ferry']:
            status = road[0].flagmap.get(widget, FALSE)
            self.wtree.get_widget(widget).set_active(status)
            self.wtree.get_widget(widget+'2').set_active(status)

        if road[0].distance.units == UNITS_METRIC:
            self.wtree.get_widget('km').set_active(TRUE)
            self.wtree.get_widget('km2').set_active(TRUE)
        else:
            self.wtree.get_widget('miles').set_active(TRUE)
            self.wtree.get_widget('miles2').set_active(TRUE)

        self.totaldist = road[0].distance
        self.wtree.get_widget('distance').set_value(float(self.totaldist))
        self.wtree.get_widget('distance2').set_value(0)

        pos = rpdbase.classify.index(rpdbase.declassify[road[0].speedcode])

        selector = self.wtree.get_widget('routesel')
        selector.set_history(pos)
        selector = self.wtree.get_widget('routesel2')
        selector.set_history(pos)
        
        self.wtree.get_widget('speedtime').set_value(road[0].speed)
        self.wtree.get_widget('speedtime2').set_value(road[0].speed)
        self.wtree.get_widget('lock').set_sensitive(TRUE)
        self.wtree.get_widget('lock').set_active(TRUE)
        #self.wtree.get_widget('lock').show()
        self.wtree.get_widget('routeok').set_sensitive(FALSE)
        self.update_box_labels()
        self.wtree.get_widget('routename').grab_focus()
        self.wtree.get_widget('routeprop').show()
        self.mode = 'break'

    def on_miles_toggled(self, *args):
        if not self.wtree.get_widget('routeprop').flags(VISIBLE):
            return
        
        from_u, to_u = UNITS_US, UNITS_METRIC
        if self.wtree.get_widget('miles').get_active():
            from_u, to_u = UNITS_METRIC, UNITS_US
        
        dwidget = self.wtree.get_widget('distance')
        distance = Distance(dwidget.get_value(), from_u)
        dwidget.set_value(float(distance.AsUnit(to_u)))

        # Don't convert time
        code = GtkOptionMenu_selected(self.wtree.get_widget('routesel'))
        speedcode = rpdbase.classifications[rpdbase.classify[code]]
        if speedcode == 1: return

        dwidget = self.wtree.get_widget('speedtime')
        distance = Distance(dwidget.get_value(), from_u)
        dwidget.set_value(float(distance.AsUnit(to_u)))

    def on_choosecity1_clicked(self, *args):
        if not self.citywin: self.citywin = CityList(self.wtree)
        self.citywin.startup('', self.db, self.set_city_widget,
                             self.wtree.get_widget('city1'))

    def on_choosecity2_clicked(self, *args):
        if not self.citywin: self.citywin = CityList(self.wtree)
        self.citywin.startup('', self.db, self.set_city_widget,
                             self.wtree.get_widget('city2'))

    def on_choosecity3_clicked(self, *args):
        if not self.citywin: self.citywin = CityList(self.wtree)
        self.citywin.startup('', self.db, self.set_city_widget,
                             self.wtree.get_widget('city3'))

    def set_city_widget(self, cityid, citynum, widget):
        widget.set_text(str(cityid))
        self.update_box_labels()
        self.update_sensitivity()

    def update_sensitivity(self, *args):
        city1 = self.wtree.get_widget('city1').get_text()
        city2 = self.wtree.get_widget('city2').get_text()
        city3 = self.wtree.get_widget('city3').get_text()
        name = self.wtree.get_widget('routename').get_text()
        val = FALSE
        if city1 and city2 and name and \
           (self.mode not in ('break', 'extend') or city3):
            val = TRUE
        self.wtree.get_widget('routeok').set_sensitive(val)

    def on_lock_toggled(self, *args):
        locked = self.wtree.get_widget('lock').get_active()
        self.wtree.get_widget('miles2').set_sensitive(not locked)
        self.wtree.get_widget('km2').set_sensitive(not locked)
        if locked:
            self.wtree.get_widget('miles2').set_active(
                self.wtree.get_widget('miles').get_active())
            self.wtree.get_widget('km2').set_active(
                self.wtree.get_widget('km').get_active())

    def on_distance_changed(self, *args):
        if not self.wtree.get_widget('rightbox').flags(VISIBLE):
            return
        if not self.wtree.get_widget('lock').get_active():
            return

        units = UNITS_US
        if self.wtree.get_widget('km').get_active():
            units = UNITS_METRIC
        
        widget = args[0]
        widget.update()
        total = float(self.totaldist.AsUnit(units))
        if widget == self.wtree.get_widget('distance'):
            dist1 = widget.get_value()
            dist2 = total - dist1
        else:
            dist2 = widget.get_value()
            dist1 = total - dist2

        dist1 = clip(0, dist1, total)
        dist2 = clip(0, dist2, total)

        self.wtree.get_widget('distance').set_value(dist1)
        self.wtree.get_widget('distance2').set_value(dist2)

def main(filename=''):
    app = Application(filename)
    mainloop()
