/******************************************************************************
 *
 * sci.c  SCreen Input program.
 * Copyright (C) 1999 Robert Canup, Sotiris Vassilopoulos, Tor Sigurdsson
 * Jason Felice and Mike Ayers
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/
/******************************************************************************
 * 
 * This program is designed to create a general purpose screen input and edit
 * facility for Unix like operating systems. It is designed to read a template
 * and allow the user to fill in and edit template data structures.
 *
 *****************************************************************************/

#include "config.h"
#define _POSIX_SOURCE 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/* Such an ugly thing is needed for some BSDs, which have both a -lcurses and
 * a -lncurses, and a <ncurses.h> and <curses.h>. */
#ifdef HAVE_NCURSES_H
#include <ncurses.h>
#else
#ifdef HAVE_CURSES_H
#include <curses.h>
#endif
#endif

#include <signal.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <locale.h>

#include "getopt.h"
#include "scipp.h"

#define MAX_WIDTH 256 /* Maximum screen width */
#define MAX_LINES 256 /* Maximum scren height */
#define MAX_PATH 256
#define MAX_FIELDS 2048 /* Maximum number of fields on a screen */
#define MAX_FLD_LEN MAX_WIDTH
#define MWIDTH 60 /* Message box width */
#define MHEIGHT 6 /* Message box height */
#define HOSTNAME 64 /* Longest hostname allowed */
#define LOOPFLAG 0x01 /* Indicates loop mode option */
#define SEQFLAG 0x08 /* Indicates -s (sequential) flag option */
#define HILITFLAG 0x10 /* Indicates -h (high lite) flag option */
#define DATEFLAG 0x20 /* Indicates -d (date timestamping) flag option */
#define REVRSE 0x40 /* Indicates CR instead of tab for field change */
#define NOHELP 0x80 /* Indicates no help is wanted */
#define VOID 0x00

/* The following flags are for the fA[n].flags field */
#define NUMFLAG 0x01 /* Indicate a numeric data entry field */
#define CAPFLAG 0x02 /* Indicate a field to capitalize */
#define ALPHAFLAG 0x04 /* Indicate an alpha only field */
#define ALPHPFLAG 0x08 /* Indicate an an alpha or punctuation */
#define NUMOFLAG 0x10 /* Indicate numbers only field */
#define FORMFLAG 0x20 /* Indicate a formatted input field */
#define COLORON 0x40 /* Indicate color is on in this field */
#define MANDFLAG 0x80 /* Indicate at least one char in field flag */
#define MANDATORYFLAG 0x100 /* Indicate full field fill flag */
#define YESNO 0x200 /* Indicate a Yes/No field */
#define EXTERNAL 0x400 /* Indicate an external program input field */
#define SENDONLY 0x800 /* Send data to an external program only */
#define SENDAND 0x1000 /* Send data to external prog and data base */
#define PICKBOX 0x2000 /* Create a pick box for this field */
#define MFLAG 0xa0 /* Both MANDFLAG and FOMRFLAG set */
#define MMFLAG 0x120  /* BOTH MANDATORYFLAG and FORMFLAG set */
#define OTHERFLAGS 0x3d /* Turn off flags not needed by Y/N flag */

int number_of_fields;
int bgColor;
int colorOK;
int padchar;
int fieldKey;
int screenKey;
int fieldChar = '.';
int prevKey = KEY_UP;
int nextKey = KEY_DOWN;
unsigned char SCI_DELIM;
char *localle;
SciReaderPtr file = NULL;
FILE *file1;
char *templt = "template.sci";
char permstring[MAX_WIDTH];
char permstring1[MAX_WIDTH];
char *outfile= "scidata.out";
char *pickDelim=",";
char picky[2];
char enmsg[] = "Press the Enter key to continue";
char manmsg[] = "In this field at least one character is mandatory";
char nummsg[] = "This field accepts only numbers, '-','+', or '.'";
char numomsg[] = "This field accepts only numbers";
char mandmsg[] = "It is mandatory that this field be complete";
char ynmsg[] = "This is a Yes/No field; enter 'y' or 'n'";
char alphamsg[] = "This field accepts only alphabetic characters";
char lockmsg[] = "Lock file detected. Cntrl - C to abort or";
char exterr[] = "External program not found";

/* This is the data for getopt_long () */
static char shortopts[] = "lf:S:F:D:p:t:o:B:P:N:shdrH";
static struct option longopts[] = 
{
	{ "loop", 0, 0, 'l' },
	{ "field-char", 1, 0, 'f' },
	{ "screen-key", 1, 0, 's' },
	{ "field-key", 1, 0, 'F' },
	{ "delimiter-char", 1, 0, 'D' },
	{ "pad-char", 1, 0, 'p' },
	{ "template", 1, 0, 't' },
	{ "output", 1, 0, 'o' },
	{ "pickBox-delimiter",1,0,'B' },
	{ "prevkey",1,0,'P' },
	{ "nextkey",1,0,'N' },
	{ "sequential", 0, 0, 's' },
	{ "highlight", 0, 0, 'h' },
	{ "hilite", 0, 0, 'h' },
	{ "datestamp", 0, 0, 'd' },
	{ "reverse", 0, 0, 'r' },
	{ "no-help", 0, 0, 'H' },
	{ NULL }
};

/* This is for argToKey, so that you can specify key names. */
static struct sci_key_t
{
	char *name;
	int code;
}
sci_keys[] =
{
#include "keys.x"
	{ "F1", KEY_F(1) },
	{ "F2", KEY_F(2) },
	{ "F3", KEY_F(3) },
	{ "F4", KEY_F(4) },
	{ "F5", KEY_F(5) },
	{ "F6", KEY_F(6) },
	{ "F7", KEY_F(7) },
	{ "F8", KEY_F(8) },
	{ "F9", KEY_F(9) },
	{ "F10", KEY_F(10) },
	{ "F11", KEY_F(11) },
	{ "F12", KEY_F(12) },
	{ "F13", KEY_F(13) },
	{ "F14", KEY_F(14) },
	{ "F15", KEY_F(15) },
	{ "F16", KEY_F(16) },
	{ "F17", KEY_F(17) },
	{ "F18", KEY_F(18) },
	{ "F19", KEY_F(19) },
	{ "F20", KEY_F(10) },
	{ "F21", KEY_F(11) },
	{ "F22", KEY_F(12) },
	{ "F23", KEY_F(13) },
	{ "F24", KEY_F(14) },
	{ NULL }
};

struct fild 
{
	int	xpos;	/* Start of field x posistion */
	int ypos;	/* Start of field y posistion */
	int cxpos;	/* Current x position of cursor in field */
	int cypos;	/* Current y position of cursor in field */
	int length; /* Maximum possible length of field */
	int size;   /* Current size of field */
	int flags;  /* Attributes of field */
	int decpt;	/* Position of decimal point in formatted input */
	int charColor; /* Color of character */
	int deffsize; /* To fix padchar problems */
	char deff[MAX_WIDTH]; /* default string for field */
	char fstring[MAX_WIDTH]; /* Format editing string */
	char fileName[MAX_WIDTH]; /* #:nx input string */
	char sendName[MAX_WIDTH]; /* #:ns or #:nS output file string */
	char pickBox[MAX_WIDTH]; /* Storage for pickBox items */
} ;

struct fild fA[MAX_FIELDS];

int setopt(int argc, char *argv[],int passNumber);
int getTemPlate(SciReaderPtr temPlateName);
int editScreen(int numOfFields,int options);
int writeData(FILE *ofile,int number_of_fields,int options);
void initinput(void);
void die(int dnum);
void clearFields(int number_of_fields, int options);
void popup(char *string);
int SCV_backspace(int size, int ypos, int xpos, int cxpos, int length);
int SCV_delete(int size, int ypos, int xpos, int cxpos);
int SCV_insert(int size, int ypos, int xpos, int cxpos, int ch);
int externField(int fieldNumber);
int picBox(int n);

#ifdef HAVE_USLEEP
int usleep(unsigned long usec);
#endif /* HAVE_USLEEP */
#ifndef __USE_BSD
int gethostname(char *name, size_t len);
#endif /* __USE_BSD */
       
void clearFields(int number_of_fields, int options)
{
int n;
int x;
 	for(n = 0; n < number_of_fields; n++)
	{
		fA[n].size = fA[n].deffsize;
		fA[n].cxpos = fA[n].xpos + fA[n].size;
		fA[n].cypos = fA[n].ypos;
 		wmove(stdscr,fA[n].ypos,fA[n].xpos);
		if((options & HILITFLAG) == HILITFLAG)
			wattron(stdscr,A_BOLD);
		if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
			wattron(stdscr,COLOR_PAIR(fA[n].charColor));
	 	for(x = 0; x < fA[n].length; x++)
	 	{
			if((fA[n].flags & FORMFLAG) != FORMFLAG)
			{
				waddch(stdscr,fA[n].deff[x]);
			}
			else
			{
				if(x == (fA[n].decpt-1))
					fA[n].fstring[x] = '.';
				else
					fA[n].fstring[x] = ' ';
				waddch(stdscr,fA[n].fstring[x]);
				fA[n].size = 0;
			}
		}
		if((fA[n].flags & FORMFLAG) == FORMFLAG)
			fA[n].fstring[(fA[n].decpt-1)] = ' ';
			wattroff(stdscr,COLOR_PAIR(fA[n].charColor));
		if((options & HILITFLAG) == HILITFLAG)
			wattroff(stdscr,A_BOLD);
	}
	refresh();
}		

void initinput(void)
{
int x;
	initscr();
	start_color();
	colorOK = has_colors();
	for(x = 1; x < COLORS; x++)
	{
		init_pair(x,x,bgColor);
	}
	raw();
	noecho();
	nonl();
	meta(stdscr, TRUE); /* Turn on 8 bits */
	keypad(stdscr, TRUE);
	clear();
	refresh();
}

void popup(char *string)
{
char ch;
int y,x;
WINDOW *ppup;
/* The algorithm is to create a window in the middle of the screen, pop it up,
 * display the message passed to it, wait for a user close window keystroke,
 * and restore the screen to previous condition.
 */
 	y = ((MHEIGHT / 2) -1);
 	x = MWIDTH / 2;
	ppup = newwin(MHEIGHT,MWIDTH,((LINES /2) - (MHEIGHT / 2)), 
		((COLS /2) -(MWIDTH /2)));
	if( ppup == NULL)
		return ; /* This won't do - no memory available */
	box(ppup,'|','-');
	wmove(ppup,y,(x - (strlen(string)/2)));
	wprintw(ppup,"%s",string);
	wmove(ppup,(y+1),(x - (strlen(enmsg)/2)));
	wprintw(ppup,"%s",enmsg);
	touchwin(ppup);
	refresh();
	ch = ' ';
	while(ch != 0x0d)
	{
		ch = wgetch(ppup) ;
		if(ch == 0x03)
			raise(SIGINT) ; /* Abort out on Cntrl - C */
	}
	delwin(ppup);
	touchwin(stdscr);
	refresh();
}

/* Translate an argument to a key.  Either a numeric (ASCII) value
 * can be specified, or a single character.  Also, you may specify
 * the name of a key according to curses.h (e.g. KEY_xxx macro,
 * with the leading KEY_ removed).  So now you can specify "NPAGE"
 * for example. */
static int argToKey (char *optarg)
{
	if (strlen (optarg) > 1)
	{
		struct sci_key_t *keyptr = sci_keys;
		while (keyptr->name) 
		{
			if (!strcmp (keyptr->name, optarg))
				return keyptr->code;
			keyptr++;
		}
		return strtol (optarg, NULL, 0);
	}
	return optarg[0];
}

int setopt(int argc, char *argv[],int passNumber)
{
int ch;
int options ; /* start with no options set */
int opt;
int loptind;
int aRgc;
char **Argv;
int x;
int adder;
static int argCount = 0;
char *aRgv;
char **arGv;
char *argV;
char parseString[MAX_WIDTH];
char pStr[MAX_WIDTH];
	options = 0;
	loptind = 0;
	aRgv = getenv("SCIPARMS");
 	if(aRgv  == NULL)
	{
		if(passNumber == 0)
		argCount = argc;
		while ((opt = getopt_long (argc, argv, shortopts, longopts, &loptind)) != -1) 
		{
			switch (opt)
			{
			case 'l':
				options |= LOOPFLAG;
				break;

			case 'f':
				fieldChar = argToKey (optarg);
				break;
		
			case 'S':
				screenKey = argToKey (optarg);
				break;

			case 'F':
				fieldKey = argToKey (optarg);
				break;

			case 'D':
				SCI_DELIM = argToKey (optarg);
				break;

			case 'p': /* pad char to use */
				padchar = argToKey (optarg);
				break;
	
			case 't': /* template name */
				templt = optarg;
				break;

			case 'o':
				outfile = optarg;
				break;
		
			case 'B': /* Pick field delimiter key */
				ch = argToKey (optarg);
				sprintf(picky,"%c",ch);
				pickDelim = picky;			
				break;

			case 'P': /* Previous key */
				prevKey = argToKey (optarg);
				break;
			
			case 'N': /* Next key */
				nextKey = argToKey (optarg);
				break;

			case 's':
				options |= SEQFLAG;
				break;

			case 'h':
				options |= HILITFLAG;
				break;

			case 'd':
				options |= DATEFLAG;
				break;

			case 'r':
				options |= REVRSE;
				break;

			case 'H':
				options |= NOHELP;
				break;

			case '?':
			case ':':
				raise(SIGINT) ; /* Argument error. */
			}
		}
	}
	else
	{
		strcpy(parseString,"sci ");
		strncat(parseString,aRgv,(size_t) MAX_WIDTH-4);
		aRgc = 0;
		strcpy(pStr,parseString);
		aRgv = strtok(pStr," ");
		if(aRgv != NULL)
		{
			aRgc = 1;
			do
			{
				aRgv = strtok(NULL," ");
					if(aRgv != NULL)
						++aRgc;
			} while(aRgv != NULL);
		} /* We now have argument count in aRgc */
		if(passNumber == 1)
			aRgc += argCount;
		/* So we have to allocate the array of pointers to the strings */
		Argv = calloc((size_t) aRgc,(size_t) sizeof(char *));
		if(Argv == NULL)
			return (options) ; /* Exit, not able to do it */
		/* The next thing we have to do is build the array arGv */
		
		if((argV = calloc((size_t) aRgc,(size_t) MAX_WIDTH)) != NULL)
		{
			strcpy(pStr,parseString);
			aRgv = strtok(pStr," ");
			adder = 0;
			for(x = argCount; x < aRgc; ++x)
			{
				if(aRgv != NULL)
				{
					strcpy(argV + adder,aRgv);
					Argv[x]  = argV + adder;
					adder += (strlen(aRgv) +1);
				}
				aRgv = strtok(NULL," ");
			}					
			arGv = Argv;
			while ((opt = getopt_long (aRgc, arGv, shortopts, longopts, &loptind)) != -1) 
			{
				switch (opt)
				{
				case 'l':
					options |= LOOPFLAG;
					break;

				case 'f':
					if(passNumber == 0)
					{
						fieldChar = argToKey (optarg);
						break;
					}
						break;
				case 'S':
					screenKey = argToKey (optarg);
					break;

				case 'F':
					fieldKey = argToKey (optarg);
					break;

				case 'D':
					SCI_DELIM = argToKey (optarg);
					break;
				case 'p': /* pad char to use */
					if(passNumber == 0)
					{
						padchar = argToKey (optarg);
						break;
					}
					break;
				case 't': /* template name */
					if(passNumber == 0)
					{
					/* Have to use permanent storage instead of heap */
						strcpy(permstring,optarg);
						templt = permstring;
						break;
					}
					break;
				case 'o':
					/* Have to use permanent storage instead of heap */
					strcpy(permstring1,optarg);
					outfile = permstring1;
					break;
		
				case 'B': /* Pick field delimiter key */
					ch = argToKey (optarg);
					sprintf(picky,"%c",ch);
					pickDelim = picky;			
					break;

				case 'P': /* Previous key */
					prevKey = argToKey (optarg);
					break;
				
				case 'N': /* Next key */
					nextKey = argToKey (optarg);
					break;

				case 's':
					options |= SEQFLAG;
					break;

				case 'h':
					options |= HILITFLAG;
					break;

				case 'd':
					options |= DATEFLAG;
					break;

				case 'r':
					options |= REVRSE;
					break;

				case 'H':
					options |= NOHELP;
					break;

				case '?':
				case ':':
					free(argV);
					free(Argv);
					raise(SIGINT) ; /* Argument error. */
				}
			}

			free(argV);
		}
		if(passNumber == 0)
			argCount = aRgc;
		free(Argv);
	}
	return (options);
}

int getTemPlate(SciReaderPtr temPlateName)
{
/* In this routine we read in the template and print it to the screen up to
 * either the end of file or a comment field. We then display the screen. As
 * we are filling in the screen we also fill in the fld array which keeps 
 * track of the posistions of each field on the screen.
 */
unsigned char ch;
char strng[MAX_WIDTH];
char string1[MAX_WIDTH];
char inpstrng[MAX_WIDTH];
char *strptr;
int brk;
int y;
int yy;
int zz;
int x;
int Ch;
int inField;
int inArrayNum;
int iColCnt = 0;

	brk = 0;
	inField = -1;
	inArrayNum = 0;
	Ch = sciReaderGetChar(temPlateName);
	while((Ch != EOF) && (brk == 0) && (inArrayNum < MAX_FIELDS))
	{
		ch = Ch;
                scrollok(stdscr, TRUE);
                if ( ch != 10 )
                  iColCnt++;
                else
                  iColCnt = 0;
		if(ch == fieldChar)
		{
			if(inField != 0)
			{
				getyx(stdscr,fA[inArrayNum].ypos,fA[inArrayNum].xpos);
				fA[inArrayNum].cypos = fA[inArrayNum].ypos;
				fA[inArrayNum].cxpos = fA[inArrayNum].xpos;
				fA[inArrayNum].length = 0;
				inField = 0;
			}
			if(inField == 0)
				fA[inArrayNum].length++;
		}
		if(ch != fieldChar)
		{
			if(inField == 0)
			{
				inArrayNum++;
				inField = -1;
			}
		}
		if((ch == '#') && (iColCnt == 1))
		{
			brk = -1; 
		}
		if(brk == 0)
		{
			if(ch == 0x0d)
			{
				waddch(stdscr,ch);
				waddch(stdscr,0x0a);
			}
			else
				waddch(stdscr,ch);
		}
		Ch = sciReaderGetChar(temPlateName);
	}			
	refresh();
	if(brk != -1)
	{
		return (inArrayNum);
	}
	ch = Ch;
	while(Ch != EOF)
	/* If we get here we know we exited at the start of the comment fields.
	 * The proceedure is to continue scanning; check for
	 * a #: combo to alert us to a command. Each time we get here we are
	 * at the start of a new line and have found a '#' character there. We
	 * have fetched the next char and are ready to process it.
	 */
	{
	 	ch = Ch;
	 	while(ch == ':')
	 	{
			/* Now we check for a field number */
			Ch = sciReaderGetChar(temPlateName);
			if(Ch == EOF)
				return (inArrayNum);
			ch = Ch;
			if(!isdigit(ch))
				return (inArrayNum);
			/* Ok we have at least one number after a ':' so build 
			 * a numeric string up
			 */
			y = 0;
			while((isdigit(ch)) && (y < MAX_WIDTH -1))
			{
				strng[y] = ch;
				Ch = sciReaderGetChar(temPlateName);
				if(Ch == EOF)
					return (inArrayNum);
				ch = Ch;
				++y;
			}							
			strng[y] = 0x00; /* Put closing string at end */
			y = atoi(strng);
			/* We now know which field has the command. 
			 * if the field number is too large do nada.
			 * y is off by one so we correct it.
			 */
			 --y;
			if(y > inArrayNum)
					return (inArrayNum);
			/* Everything is OK so get the command for field */
			if((ch != 'n') && (ch != 'd') && (ch != 'c') && (ch != 'a')
				&& (ch != 'A') && (ch != 'N') && (ch != 'f') && (ch != 'C')
				&& (ch != 'm') && (ch != 'M') && (ch != 'y') && (ch != 'b') 
				&& (ch != 'x') && (ch != 's') && (ch != 'S') && (ch != 'p'))
				return (inArrayNum);
			if(y >= -1)
			{
				if((ch == 'a') && (y != -1))
					fA[y].flags |= ALPHPFLAG; /* Set the alpha-punctuation flag */
				if((ch == 'A') && (y != -1))
					fA[y].flags |= ALPHAFLAG; /* Set the alpha only flag */
				if((ch == 'n') && (y != -1))
					fA[y].flags |= NUMFLAG; /* Set the numeric flag */
				if((ch == 'N') && (y != -1))
					fA[y].flags |= NUMOFLAG; /* Set the numbers only flag */
				if(ch == 'c')
				{
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
							fA[yy].flags |= CAPFLAG;
					}
					else
						fA[y].flags |= CAPFLAG; /* Set the capitilization flag */
				}
				if(ch == 'm')
				{
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
							fA[yy].flags |= MANDFLAG;
					}
					else
						fA[y].flags |= MANDFLAG;
				}
				if(ch == 'M')
				{
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
							fA[yy].flags |= MANDATORYFLAG;
					}
					else
						fA[y].flags |= MANDATORYFLAG;
				}
				if(ch == 'y')
				{
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
						{
							fA[yy].flags |= YESNO;
							fA[yy].flags &= (-OTHERFLAGS - 1);
							if(fA[yy].length != 1)
							{
								wmove(stdscr,fA[yy].ypos,fA[yy].xpos);
								waddch(stdscr,padchar);
								for(zz = 1; zz < fA[yy].length; ++zz)
									waddch(stdscr,' ');
								fA[yy].length = 1;
							}
						} 
					}
					else
					{
						fA[y].flags |= YESNO;
						fA[y].flags &= (-OTHERFLAGS - 1);
						if(fA[y].length != 1)
						{
							wmove(stdscr,fA[y].ypos,fA[y].xpos);
							waddch(stdscr,padchar);
							for(zz = 1; zz < fA[y].length; ++zz)
								waddch(stdscr,' ');
							fA[y].length = 1;
						}
					}
				}
				if((ch == 'f') && (y != -1))
				{
				/* We have a valid command so insert chars into place */
				/* We retrieve the format characters */
					fA[y].flags |= FORMFLAG; /* Set the format flag */
					strcpy(string1,""); /* Blank the string */
					x = 0;
					do
					{
						Ch = sciReaderGetChar(temPlateName);
						if(Ch == EOF)
							return (inArrayNum);
						ch = Ch;
						if(ch != 0x0a)
							string1[x] = ch;
						if(ch == 0x0a)
						{
							string1[x] = 0; /* Terminate string */
							x = fA[y].length;
							strptr = strchr(string1, '.');
							if(strptr != NULL)
							{
								fA[y].decpt = strptr - string1;
								++fA[y].decpt;
							}
							else
								fA[y].decpt = 0;
							break;
						}
						x++;
					} while(1 != 2);
				}
				if((ch == 'd') && (y != -1))
				{
				/* We have a valid command so insert default char into place */
					for(x = 0; x < fA[y].length; x++)
					{
						Ch = sciReaderGetChar(temPlateName);
						if(Ch == EOF)
							return (inArrayNum);
						ch = Ch;
						if(ch != 0x0a)
						{
							fA[y].deff[x] = ch;
							fA[y].deffsize++;
						}			
						if(ch == 0x0a)
							x = fA[y].length;
					}
					if(ch != 0x0a)
						ch = VOID; /* Don't pass on garbage */
				}
				if(ch == 'x')
				{
					/* We have a link to an external program so set it up */
					/* First get external program string into program */
					x = 0; 
					strcpy(inpstrng,"");
					do
					{
						Ch = sciReaderGetChar(temPlateName);
						if(Ch == EOF)
							return (inArrayNum);
						ch = Ch;
						if(ch != 0x0a)
							inpstrng[x] = ch;
						if(ch == 0x0a)
							inpstrng[x] = 0x00;
						++x;
					} while((ch != 0x0a) && (x < MAX_WIDTH -1));
					if(ch != 0x0a)
						ch = VOID;
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
						{
							fA[yy].flags |= EXTERNAL;
							strcpy(fA[y].fileName,inpstrng);
						}
					}
					else
					{
						fA[y].flags |= EXTERNAL;
						strcpy(fA[y].fileName,inpstrng);
					}
				}				
				if(ch == 'p')
				{
					/* First get popup items into program */
					x = 0; 
					strcpy(inpstrng,"");
					do
					{
						Ch = sciReaderGetChar(temPlateName);
						if(Ch == EOF)
							return (inArrayNum);
						ch = Ch;
						if(ch != 0x0a)
							inpstrng[x] = ch;
						if(ch == 0x0a)
							inpstrng[x] = 0x00;
						++x;
					} while((ch != 0x0a) && (x < MAX_WIDTH -1));
					if(ch != 0x0a)
						ch = VOID;
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
						{
							fA[yy].flags |= PICKBOX;
							strcpy(fA[y].pickBox,inpstrng);
						}
					}
					else
					{
						fA[y].flags |= PICKBOX;
						strcpy(fA[y].pickBox,inpstrng);
					}
				}				
				if(ch == 's')
				{
					/* We have a link to an external program so set it up */
					/* First get external program string into program */
					x = 0; 
					strcpy(inpstrng,"");
					do
					{
						Ch = sciReaderGetChar(temPlateName);
						if(Ch == EOF)
							return (inArrayNum);
						ch = Ch;
						if(ch != 0x0a)
							inpstrng[x] = ch;
						if(ch == 0x0a)
							inpstrng[x] = 0x00;
						++x;
					} while((ch != 0x0a) && (x < MAX_WIDTH -1));
					if(ch != 0x0a)
						ch = VOID;
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
						{
							fA[yy].flags |= SENDONLY;
							strcpy(fA[y].sendName,inpstrng);
						}
					}
					else
					{
						fA[y].flags |= SENDONLY;
						strcpy(fA[y].sendName,inpstrng);
					}
				}				
				if(ch == 'S')
				{
					/* We have a link to an external program so set it up */
					/* First get external program string into program */
					x = 0; 
					strcpy(inpstrng,"");
					do
					{
						Ch = sciReaderGetChar(temPlateName);
						if(Ch == EOF)
							return (inArrayNum);
						ch = Ch;
						if(ch != 0x0a)
							inpstrng[x] = ch;
						if(ch == 0x0a)
							inpstrng[x] = 0x00;
						++x;
					} while((ch != 0x0a) && (x < MAX_WIDTH -1));
					if(ch != 0x0a)
						ch = VOID;
					if(y == -1)
					{
						for(yy = 0; yy < inArrayNum; yy++)
						{
							fA[yy].flags |= SENDAND;
							strcpy(fA[y].sendName,inpstrng);
						}
					}
					else
					{
						fA[y].flags |= SENDAND;
						strcpy(fA[y].sendName,inpstrng);
					}
				}				
			}
			if(ch == 'C')
			{
				/* We have color! So take care of it */
				/* First set the color flags in array */
				if(y == -1)
				{
					for(yy = 0; yy < inArrayNum; yy++)
						fA[yy].flags |= COLORON;
				}
				else
					fA[y].flags |= COLORON;
			/* Next get the next character from template */
				Ch = sciReaderGetChar(temPlateName);
				if(Ch == EOF)
					return (inArrayNum);
				ch = Ch;
				switch (ch)
				{
					case 'g':
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_GREEN;
						else
							fA[y].charColor = COLOR_GREEN;
						break;
					case 'w':
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_WHITE;
						else
							fA[y].charColor = COLOR_WHITE;
						break;
					case 'r':
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_RED;
						else
							fA[y].charColor = COLOR_RED;
						break;
					case 'b':
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_BLUE;
						else
							fA[y].charColor = COLOR_BLUE;
						break;
					case 'c':
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_CYAN;
						else
							fA[y].charColor = COLOR_CYAN;
						break;
					case 'y':
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_YELLOW;
						else
							fA[y].charColor = COLOR_YELLOW;
						break;
					case 'm':
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_MAGENTA;
						else
							fA[y].charColor = COLOR_MAGENTA;
						break;
					default:
						if(y == -1)
							for(yy = 0; yy < inArrayNum; yy++)
								fA[yy].charColor = COLOR_WHITE;
						else
							fA[y].charColor = COLOR_WHITE;
						break;
				}
			}
			if(ch == 'b')
			{
				/* We have a background color! So take care of it */
				/* First set the background color flags in array */
				/* Next get the next character from template */
				Ch = sciReaderGetChar(temPlateName);
				if(Ch == EOF)
					return (inArrayNum);
				ch = Ch;
				switch (ch)
				{
					case 'g':
						if(y == -1)
							bgColor = COLOR_GREEN;
						break;
					case 'w':
						if(y == -1)
							bgColor = COLOR_WHITE;
						break;
					case 'r':
						if(y == -1)
							bgColor = COLOR_RED;
						break;
					case 'b':
							bgColor = COLOR_BLUE;
						break;
					case 'c':
						if(y == -1)
							bgColor = COLOR_CYAN;
						break;
					case 'y':
						if(y == -1)
							bgColor = COLOR_YELLOW;
						break;
					case 'm':
						if(y == -1)
							bgColor = COLOR_MAGENTA;
						break;
					default:
						if(y == -1)
							bgColor = COLOR_BLACK;
						break;
				}
			}
		}
		while(ch != 0x0a)
		{
			/* If the last char is not a new line search till we get one */		
				Ch = sciReaderGetChar(temPlateName);
				if(Ch == EOF)
					return (inArrayNum);
				ch = Ch;
		}		
		/* At end of previous line. Get next line start */
		Ch = sciReaderGetChar(temPlateName);
		if(Ch == EOF)
			return (inArrayNum);
		ch = Ch;
		if(ch != '#')
			return (inArrayNum);
		Ch = sciReaderGetChar(temPlateName);
		if(Ch == EOF)
			return (inArrayNum);
	}
	return (inArrayNum);
}


static void formAdjust(int *psignFlag, char tstring[MAX_WIDTH], int n)
{
  fA[n].cxpos = (fA[n].xpos + fA[n].length);
  fA[n].cypos = fA[n].ypos;
  wmove(stdscr,fA[n].cypos,fA[n].xpos);
  fA[n].fstring[(fA[n].length -1)] = 0x00; 
  memmove(tstring,fA[n].fstring,(size_t)fA[n].decpt);
  tstring[(fA[n].decpt -1)] = '.'; /* Put in decimal pt */
  tstring[(fA[n].decpt )] = 0x00; 
  strcat(tstring,&fA[n].fstring[(fA[n].decpt-1)]);
  wmove(stdscr,fA[n].ypos,fA[n].xpos);
  wprintw(stdscr,tstring);
  if(fA[n].size == 0)
    *psignFlag = 0;
  else
    *psignFlag = 1;
}

int editScreen(int numOfFields,int options)
{
int n;
int x;
int temp;
int tflag;
int minusFlag;
int signFlag;
int ch;
char tstring[MAX_WIDTH];
	minusFlag = 0;
	signFlag = 0;
	--numOfFields; /* Indexing off by one */
	n = 0;
	wmove(stdscr,fA[n].cypos,fA[n].cxpos);
	
	if((fA[n].flags & FORMFLAG) == FORMFLAG)
	{
	  formAdjust(&signFlag, tstring, n);
	}
	if((options & HILITFLAG) == HILITFLAG)
		wattron(stdscr,A_BOLD);
	if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
		wattron(stdscr,COLOR_PAIR(fA[n].charColor));
	if((options & REVRSE) == REVRSE)
	{
		temp = screenKey ;
		screenKey = fieldKey ;
		fieldKey = temp ;
	}
	if(((fA[n].flags & EXTERNAL) == EXTERNAL) && (fA[n].size == 0))
		ch = externField(n);
	else
		ch = wgetch(stdscr);
	tflag = 1;
	if(((fA[n].flags & MANDFLAG) == MANDFLAG)  && (fA[n].size > 0))
		tflag = 0; 
	if(((fA[n].flags & MANDATORYFLAG) == MANDATORYFLAG) &&
		(fA[n].size == fA[n].length)) 
			tflag = 0;
	if(((fA[n].flags & MFLAG) == MFLAG) && (fA[n].size == (fA[n].length - 1)))
			tflag = 0;
	if(((fA[n].flags & MMFLAG) == MMFLAG) && (fA[n].size == (fA[n].length -1)))
			tflag = 0;
	if(((fA[n].flags & MANDFLAG) != MANDFLAG)  
		&& ((fA[n].flags & MANDATORYFLAG) != MANDATORYFLAG))
			tflag = 0;
	if((tflag == 1) && (ch == screenKey))
	{ 
		if(((fA[n].flags & MANDFLAG) == MANDFLAG) 
		&& ((options & NOHELP) != NOHELP)) 
			popup(manmsg);
		if(((fA[n].flags & MANDATORYFLAG) == MANDATORYFLAG)
		&& ((options & NOHELP) != NOHELP)) 
			popup(mandmsg);
		ch = VOID;
	}
	while(ch != screenKey) /* CR char unless -r option */ 
	{
		if(ch == nextKey)
			ch = fieldKey;
		if((ch == fieldKey) || (ch == prevKey)) /* Tab char unless -r option */
		{
			tflag = 1;
			if(((fA[n].flags & MANDFLAG) == MANDFLAG)  && (fA[n].size > 0))
				tflag = 0; 
			if(((fA[n].flags & MANDATORYFLAG) == MANDATORYFLAG) && 
				(fA[n].size == fA[n].length)) 
					tflag = 0;
			if(((fA[n].flags & MFLAG) == MFLAG) && (fA[n].size == (fA[n].length - 1)))
				tflag = 0;
			if(((fA[n].flags & MMFLAG) == MMFLAG) && (fA[n].size == (fA[n].length -1)))
				tflag = 0;
			if(((fA[n].flags & MANDFLAG) != MANDFLAG)  
				&& ((fA[n].flags & MANDATORYFLAG) != MANDATORYFLAG))
					tflag = 0;
			if(tflag == 0)
			{ 
				if((ch == fieldKey) && ( n < numOfFields))
				{
					wattroff(stdscr,COLOR_PAIR(fA[n].charColor));
					++n;
					wmove(stdscr,fA[n].cypos,fA[n].cxpos);
					if((options & HILITFLAG) == HILITFLAG)
						wattron(stdscr,A_BOLD);
					if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
						wattron(stdscr,COLOR_PAIR(fA[n].charColor));
				}
				else if(ch == fieldKey)
				{
					wattroff(stdscr,COLOR_PAIR(fA[n].charColor));
					n = 0;
					wmove(stdscr,fA[n].cypos,fA[n].cxpos);
					if((options & HILITFLAG) == HILITFLAG)
						wattron(stdscr,A_BOLD);
					if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
						wattron(stdscr,COLOR_PAIR(fA[n].charColor));
				}
				else if((ch == prevKey) && (n > 0))
				{
					wattroff(stdscr,COLOR_PAIR(fA[n].charColor));
					--n;
					wmove(stdscr,fA[n].cypos,fA[n].cxpos);
					if((options & HILITFLAG) == HILITFLAG)
						wattron(stdscr,A_BOLD);
					if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
						wattron(stdscr,COLOR_PAIR(fA[n].charColor));
				}
				else if((ch == prevKey) && (n == 0))
				{
					n = numOfFields;
					wattroff(stdscr,COLOR_PAIR(fA[n].charColor));
					wmove(stdscr,fA[n].cypos,fA[n].cxpos);
					if((options & HILITFLAG) == HILITFLAG)
						wattron(stdscr,A_BOLD);
					if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
						wattron(stdscr,COLOR_PAIR(fA[n].charColor));
				}
				if((fA[n].flags & PICKBOX) == PICKBOX)
					picBox(n);
				ch = VOID;
			}
			else
			{
				if(((fA[n].flags & MANDFLAG) == MANDFLAG)
				&& ((options & NOHELP) != NOHELP)) 
					popup(manmsg);
				if(((fA[n].flags & MANDATORYFLAG) == MANDATORYFLAG)
				&& ((options & NOHELP) != NOHELP)) 
					popup(mandmsg);
				ch = VOID;
			}
			if((fA[n].flags & FORMFLAG) == FORMFLAG)
			{
			  formAdjust(&signFlag, tstring, n);
			}
			refresh();
		}
		if(ch == KEY_LEFT)
		{
			if(fA[n].cxpos > fA[n].xpos)
			{
				wmove(stdscr, fA[n].cypos, --fA[n].cxpos);
				refresh();
			}
		}
		if(ch == KEY_RIGHT)
		{
			if((fA[n].cxpos < (fA[n].xpos) + fA[n].size))
			{
				wmove(stdscr, fA[n].cypos, ++fA[n].cxpos);
				refresh();
			}
		}
		if(ch == KEY_HOME)
		{
			fA[n].cxpos = fA[n].xpos;
			wmove(stdscr, fA[n].cypos, fA[n].cxpos);
			refresh();
		}
		if(ch == KEY_END)
		{
			fA[n].cxpos = fA[n].xpos + fA[n].size;
			wmove(stdscr, fA[n].cypos, fA[n].cxpos);
			refresh();
		}
		if((fA[n].flags & FORMFLAG) == FORMFLAG)
		{
			if((ch == 0x08) || (ch == 0x7f) || (ch == 0x7) || (ch == KEY_BACKSPACE)) /* backspace */
			{
				if(fA[n].size > 0 )
				{
					/* Make room for period at start */
					memmove(&fA[n].fstring[1],&fA[n].fstring[0],(size_t)fA[n].length -1);
					fA[n].fstring[0] = ' ';
					fA[n].fstring[fA[n].length -1] = 0x00;
					memmove(tstring,fA[n].fstring,(size_t)fA[n].decpt);
					tstring[(fA[n].decpt -1)] = '.'; /* Put in decimal pt */
					tstring[(fA[n].decpt )] = 0x00; 
					strcat(tstring,&fA[n].fstring[(fA[n].decpt-1)]);
					wmove(stdscr,fA[n].ypos,fA[n].xpos);
					wprintw(stdscr,tstring);
					refresh();
					fA[n].size--;
					if(fA[n].size == 0)
						signFlag = 0;
				}
					ch = VOID;
			}
			if(ch == 0x15) /* Cntrl u */
			{
				wmove(stdscr,fA[n].ypos,fA[n].xpos);
				if((options & HILITFLAG) == HILITFLAG)
					wattron(stdscr,A_BOLD);
				if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
					wattron(stdscr,COLOR_PAIR(fA[n].charColor));
				strcpy(fA[n].fstring,"");
				for(x = 0; x < fA[n].length; x++)
					if(x == (fA[n].decpt-1))
						strcat(fA[n].fstring,".");
					else
						strcat(fA[n].fstring," ");
				wprintw(stdscr,fA[n].fstring);
				fA[n].fstring[(fA[n].decpt-1)] = ' ';
				memmove(fA[n].fstring,&fA[n].fstring[1],fA[n].length);
				fA[n].size = 0;
				signFlag = 0;
				refresh();
				ch = VOID;
			}
			if(ch == 0x03)	/* Cntrl - c */
				raise(SIGINT);
			if(((isdigit(ch) || (((ch == '-') || (ch == '+'))
				&& (signFlag == 0)))))
			{
				if(fA[n].size < (fA[n].length -1))
				{
				/* Upon entry fA[n].fstring is one short of its maximum length */
				/* Shorten string by one to make room for character */
					if(fA[n].fstring[(fA[n].decpt-1)] == '.')
						fA[n].fstring[(fA[n].decpt-1)] = ' ';
					memmove(&fA[n].fstring[0],&fA[n].fstring[1], (size_t)(fA[n].length -1));
					fA[n].fstring[(fA[n].length - 2)] = ch;
					memmove(tstring,fA[n].fstring,(size_t)fA[n].decpt);
					tstring[(fA[n].decpt -1)] = '.'; /* Put in decimal pt */
					tstring[(fA[n].decpt )] = 0x00; 
					strcat(tstring,&fA[n].fstring[(fA[n].decpt-1)]);
					wmove(stdscr,fA[n].ypos,fA[n].xpos);
					wprintw(stdscr,tstring);
					refresh();
					fA[n].size++;
					signFlag = 1;
				}
					ch = VOID;
			}	
			ch = wgetch(stdscr);
		}
		else
		{
			if((ch == 0x08) || (ch == 0x7f) || (ch == 0x7) || (ch == KEY_BACKSPACE)) /* backspace */
			{
				if(fA[n].cxpos > fA[n].xpos)
				{
					SCV_backspace(fA[n].size, fA[n].ypos, fA[n].xpos, fA[n].cxpos, fA[n].length);
					--fA[n].cxpos;
					wmove(stdscr,fA[n].cypos,fA[n].cxpos);
					refresh();
					fA[n].size--;
				}
				ch = VOID;
			}
			if (ch == KEY_DC)  /* Delete key */
			{
				if (fA[n].cxpos != fA[n].xpos + fA[n].size)
				{
					SCV_delete(fA[n].size, fA[n].ypos, fA[n].xpos, fA[n].cxpos);
					refresh();
					fA[n].size--;
				}
				ch = VOID;
			}
			if(ch == 0x15) /* Cntrl u */
			{
				fA[n].cxpos = fA[n].xpos; /* Back to start of field */
				wmove(stdscr,fA[n].cypos,fA[n].cxpos);
				if((options & HILITFLAG) == HILITFLAG)
					wattron(stdscr,A_BOLD);
				if((colorOK) && ((fA[n].flags & COLORON) == COLORON))
					wattron(stdscr,COLOR_PAIR(fA[n].charColor));
				for(x = 0; x < fA[n].length; x++)
				{
					waddch(stdscr,padchar);
					++fA[n].cxpos;
				}
				fA[n].cxpos = fA[n].xpos;
				wmove(stdscr,fA[n].cypos,fA[n].cxpos);
				fA[n].size = 0;
				refresh();
				if((fA[n].flags & PICKBOX) == PICKBOX)
					picBox(n);
				ch = VOID;
			}
			if(((fA[n].flags & EXTERNAL) == EXTERNAL) && (fA[n].size == 0))
				ch = externField(n);
			if(ch == 0x03)	/* Cntrl - c */
				raise(SIGINT);

			if((fA[n].flags & NUMOFLAG) == NUMOFLAG)
			{
				if((!isdigit(ch))  && (ch != VOID)) 
				{
					if((options & NOHELP) != NOHELP) 
						popup(numomsg);
					ch = VOID;
				}
			}
			
			if((fA[n].flags & ALPHAFLAG) == ALPHAFLAG)
			{
				/* Check for alphabetic character only - no spaces allowed */
				if((!isalpha(ch)) && (ch != VOID))
				{
					if((options & NOHELP) != NOHELP) 
						popup(alphamsg);
					ch = VOID;
				}
			}

			if((fA[n].flags & ALPHPFLAG) == ALPHPFLAG)
			{
				/* Check for alpha or punctuation char - allow spaces */
				if((!isalpha(ch)) && !strchr("!\"#$%&'()*+,-./;:?@ ",ch) && (ch != VOID) &&(ch < 0x80))
				{
					if((options & NOHELP) != NOHELP) 
						popup(alphamsg);
					ch = VOID;
				}
			}
			if((fA[n].flags & YESNO) == YESNO)
			{
				if(ch != VOID)
				{
					if((ch != 'y') && (ch != 'n') && (ch != 'Y') && (ch != 'N'))
					{
						if((options & NOHELP) != NOHELP) 
							popup(ynmsg);
						ch = VOID;
					}
				}				
			}
		

			if((fA[n].flags & NUMFLAG) == 0x00)
			{
				if((unsigned char) ch < 0x80)
				{
					if((ch < ' ') || (ch > 'z')) 
					{
						ch = VOID;
					}	
				}
			}
			if((fA[n].flags & NUMFLAG) == NUMFLAG)
			{
				if(ch != VOID) 
				{
					if(!isdigit(ch)) 
					{
						if(ch != '-') 
						{
							if(ch != '+')
							{
								if(ch != '.')
								{
									if((options & NOHELP) != NOHELP) 
										popup(nummsg);
									ch = VOID;
								}
							}
						}
					}
				}
			}

			if((fA[n].flags & CAPFLAG) == CAPFLAG)
			{
				temp = toupper(ch);
				ch = (char) temp;
			}

			if((fA[n].size < fA[n].length) && (ch != VOID))
			{
				SCV_insert(fA[n].size, fA[n].ypos, fA[n].xpos,fA[n].cxpos,ch);
				refresh();
				fA[n].cxpos++;
				fA[n].size++;
			}
			ch = wgetch(stdscr);
		}
		tflag = 1;
		if(((fA[n].flags & MANDFLAG) == MANDFLAG)  && (fA[n].size > 0))
			tflag = 0; 
		if(((fA[n].flags & MANDATORYFLAG) == MANDATORYFLAG) && 
			(fA[n].size == fA[n].length)) 
				tflag = 0;
		if(((fA[n].flags & MFLAG) == MFLAG) && (fA[n].size == (fA[n].length - 1)))
			tflag = 0;
		if(((fA[n].flags & MMFLAG) == MMFLAG) && (fA[n].size == (fA[n].length - 1)))
			tflag = 0;
		if(((fA[n].flags & MANDFLAG) != MANDFLAG)  
			&& ((fA[n].flags & MANDATORYFLAG) != MANDATORYFLAG))
				tflag = 0;
		if((tflag == 1) && (ch == screenKey))
		{ 
			if(((fA[n].flags & MANDFLAG) == MANDFLAG)
			&& ((options & NOHELP) != NOHELP)) 
				popup(manmsg);
			if(((fA[n].flags & MANDATORYFLAG) == MANDATORYFLAG)
			&& ((options & NOHELP) != NOHELP)) 
				popup(mandmsg);
			ch = VOID;
		}
	}
	return (0);
}

int SCV_backspace(int size, int ypos, int xpos, int cxpos, int length)
{
int x;
char b[2];
	b[1] = '\0';
	if(cxpos == size + xpos)
	{
		for(x = cxpos - 1; x < xpos + size -2; x++)
		{
			wmove(stdscr, ypos, (x + 1));
			b[0] = winch(stdscr);
			mvwaddstr(stdscr,ypos, x, b);
		}
	}
	else
	{
		for(x = cxpos - 1 ; x < xpos + size; x++)
		{
			wmove(stdscr, ypos, (x + 1));
			b[0] = winch(stdscr);
			mvwaddstr(stdscr,ypos, x, b);
		}
	}
	b[0] = padchar;
	if(cxpos == size + xpos )
	{
		mvwaddstr(stdscr,ypos, xpos + size -1, b);
	}
	if((cxpos < size + xpos) && (size != length))
	{
		mvwaddstr(stdscr , ypos, xpos + size, b);
	}
	if((cxpos < size + xpos) && (size == length))
	{
		mvwaddstr(stdscr, ypos, xpos + size -1, b);
	}
	return(1);
}

int SCV_delete(int size, int ypos, int xpos, int cxpos)
{
	int x;
	char b[2];
	b[1] = '\0';
	for (x = cxpos; x < xpos + size; x++) 
	{
		wmove(stdscr, ypos, x + 1);
		b[0] = winch(stdscr);
		mvwaddstr(stdscr, ypos, x, b);
	}
	b[0] = padchar;
	mvwaddstr(stdscr, ypos, xpos + size - 1, b);
	wmove(stdscr, ypos, cxpos);
	return (1);
}
 

int SCV_insert(int size, int ypos, int xpos, int cxpos, int ch)
{
int x;
char arry[MAX_WIDTH + 3];
	/* copy string */
	for(x = 0; x < size; x++)
	{
		wmove(stdscr, ypos, xpos + x);
		arry[x] = winch(stdscr);
	}
	arry[size] = '\0';
	arry[size + 1] = '\0';
	/* make room for the character */
	for(x = size; x > cxpos - xpos; x--)
	{
		arry[x] = arry[x -1];
	}
	arry[cxpos - xpos] = ch;
	mvwaddstr(stdscr, ypos, xpos, arry);
	wmove(stdscr, ypos, cxpos + 1);
	return(1);
}
 
int externField(int fieldNumber)
{
/* When called this routine will enter data into the specified field from 
 * an external program. This allows Cntrl - u field clearing to 
 * enter a new value into the field.
 */
FILE *file;
int ch;
char temp2[MAX_WIDTH];
char *temp;
	if(strlen(fA[fieldNumber].fileName) != 0) 
	{
		strcpy(temp2,fA[fieldNumber].fileName);
		temp = strtok(temp2," ");
		if((ch = access(temp,R_OK)) == 0)
		{
			file = popen(fA[fieldNumber].fileName,"r");
			if(file != NULL)
			{
				do
				{
					ch = sciReaderGetChar((SciReaderPtr)file);
					if(ch >= ' ')
					{
						waddch(stdscr,ch);
						fA[fieldNumber].cxpos++;
						fA[fieldNumber].size++;
					}
				} while((fA[fieldNumber].size < fA[fieldNumber].length)
					&& (ch != EOF) && (ch != 0x0a));
				pclose(file);
			}
		}
		else
		{
			popup(exterr);
			ch = wgetch(stdscr);
			if(ch == 0x03) /* Cntrl - c exit */
				raise(SIGINT);
			return(ch);
		}
	}
	return(1);
}

int writeData(FILE *ofile,int number_of_fields,int options)
{
int j;
int k;
int n;
int w;
int x;
int y;
int xx;
int ch;
int perflag;
time_t tme;
pid_t z;
char cch;
char arry[MAX_WIDTH];
char ary[MAX_PATH+4];
char num[MAX_WIDTH];
char lock[MAX_PATH+HOSTNAME+10];
char hostn[HOSTNAME+2];
char temp2[MAX_WIDTH];
char *temp;
FILE *fille;
struct stat buff;
/* Multi-user file locking section */
	y = gethostname(hostn,HOSTNAME);
	if(y != 0) 
		raise(SIGINT); /* If this fails we're dead anyway. */
	z = getpid();
	sprintf (lock,"%s-%s-%d", templt, hostn, z); /* Build our unique lock file name */
	w = 2;
	srand(z); /* Set up the random number generator */
	k = 0; /* This is a counter to see how long we are locked */
	do
	{
		y = link(templt,lock);
		stat(lock,&buff);
		w = buff.st_nlink;
		if(w > 2)
		{
			unlink(lock);
			#ifdef HAVE_USLEEP
				j = 1000 + (int) (1000.0 * rand()/(RAND_MAX));
				usleep(j); /* usleep may not be portable */
			#endif /* HAVE_USLEEP */
			k++;
			if(k > 20)
			{
				popup(lockmsg);
				k = 0;
			}
		}
	}
	while(w != 2);
/* End of Multi-user file locking section. */
	for(n = 0;n < number_of_fields; n++)
	{
		wmove(stdscr,fA[n].ypos,fA[n].xpos);
		fA[n].cxpos = fA[n].xpos;
		if((fA[n].flags & FORMFLAG) != FORMFLAG)
		{
			for(x = 0;x < fA[n].size;x++)
			{
				arry[x] = winch(stdscr);
				wmove(stdscr,fA[n].ypos,++fA[n].cxpos);
			}
		}
		else
		{
			xx = 0;
			perflag = 0;
			for(x = 0 ; x < fA[n].length; x++)
			{
				cch = winch(stdscr);
				wmove(stdscr,fA[n].ypos,++fA[n].cxpos);
				if((cch != ' ') || (perflag != 0))
				{
					if(cch == ' ')
					{
						cch = '0';
						fA[n].size++; /* Account for added char */
					}
					arry[xx] = cch;
					xx++;
					perflag = 1;
					
				}
			}
			fA[n].size++; /* Account for decimal point */
		}
		if(fA[n].size != 0)
		{
			if(((fA[n].flags & SENDONLY) == SENDONLY)
				|| ((fA[n].flags & SENDAND) == SENDAND))
			{
				if(strlen(fA[n].sendName) != 0) 
				{
					strcpy(temp2,fA[n].sendName);
					temp = strtok(temp2," ");
					fflush(ofile);
					if((ch = access(temp,X_OK)) == 0)
					{
						fille = popen(fA[n].sendName,"w");
						if(fille != NULL)
						{
							x = fwrite(&arry[0],1,(size_t) fA[n].size,fille);
							if(x != fA[n].size)
							{
								y = unlink(lock);
								pclose(fille);
								raise(SIGINT);
							}								
							pclose(fille);
						}
					}
				}	
			}
			if((fA[n].flags & SENDONLY) != SENDONLY)
			{
				x = fwrite(&arry[0],1,(size_t) fA[n].size,ofile);
				if(x != fA[n].size) 
				{
					y = unlink(lock);
					raise(SIGINT);
				}
			}
		}
		if(n == (number_of_fields - 1))
		{
			if((options & SEQFLAG) != 0)
			{
				strcpy(num,"0");
				strcpy(ary,templt);
				strcat(ary,".num");
				file1 = fopen(ary,"a+");
				if(file1 != NULL)
				{
					rewind(file1);
					fgets(num,MAX_WIDTH,file1);
					fclose(file1);
					x = atoi(num);
					x++;
					strcpy(num,"");
					sprintf(num,"%i",x);
					file1 = fopen(ary,"w");
					if(file1 != NULL)
					{
						fputs(num,file1);
						fclose(file1);
					}
				}
				arry[0] = SCI_DELIM;
				x = fwrite(&arry[0],1,1,ofile);
				if(x != 1)
				{
					y = unlink(lock);
					raise(SIGINT);
				}
				x = fwrite(num,1,strlen(num),ofile);
				if(x != strlen(num)) 
				{
					y = unlink(lock);
					raise(SIGINT);
				}
			}
			if((options & DATEFLAG) != 0)
			{
				tme = time(NULL);
				if(tme != -1)
				{
					strcpy(num,"");
					sprintf(num,"%i",(int) tme);	
					arry[0] = SCI_DELIM;
					x = fwrite(&arry[0],1,1,ofile);
					if(x != 1)
					{
						y = unlink(lock);
						raise(SIGINT);
					}
					x = fwrite(num,1,strlen(num),ofile);
					if(x != strlen(num)) 
					{
						y = unlink(lock);
						raise(SIGINT);
					}
				}
			}				
		}
		arry[0] = SCI_DELIM;
		if(n == (number_of_fields -1))
			arry[0] = 0x0a;
		x = fwrite(&arry[0],1,1,ofile);
		if(x != 1)
		{
			y = unlink(lock);
			raise(SIGINT);
		}
	}
	y = unlink(lock);
	if(y != 0) 
		raise(SIGINT);
	return (0);
}

int picBox(int n)
{
/* This is the routine to create a pick box and paste the results of the pick
 * into a data field. The pick box is 4 characters wider than the longest item 
 * in the list (up to the width of the screen ). So the first thing to do is
 * to see what is the biggest item. The item width is limited by the length of 
 * the field.
 */
int p;
int q;
int r;
int w;
int x;
int y;
int screenPos;
int z[MAX_WIDTH];
int ch;
char tstrng[MAX_WIDTH];
char *strt;
WINDOW *ppup;
WINDOW *popup;
	x = 0;
	y = 0;
	strcpy(tstrng,fA[n].pickBox);
	if((strlen(tstrng) > 0))
	{
		strt = strtok(tstrng,pickDelim);
		if(strt != NULL)
		{
			x = strlen(strt);
			z[y]=x; /* Save the length of the string for later use */
			++y;
			while(strt != NULL)
			{
				strt = strtok(NULL,pickDelim);
				if(strt != NULL)
				{
					z[y] = strlen(strt);
					if(z[y] > x)
					{
						x = z[y];
					}
					++y;
				}
			}
			/* When we get here y is the number of items, and x is the 
			 * length of the longest item. The array z[] has the lengths 
			 * of each string. We can now build the pickBoxwindow. 
			 */
			if(x < COLS - 4)
			 	p = x + 4;
			else 
			 	p = COLS;
			if(y < LINES - 3)
			 	q = y + 3;
			else
				q = LINES; 
			ppup = newwin(q,p,0,(COLS - p));
			if( ppup == NULL)
				return 0 ; /* This won't do - no memory available */
			box(ppup,'|','-');
			if(y < LINES -3) /* No scroll If we don't have too many items */
			{
				strcpy(tstrng,fA[n].pickBox);
				strt = strtok(tstrng,pickDelim);			
				wmove(ppup, 2,2);
				for(w = 0; w < y; ++w)
				{
					wprintw(ppup,"%s",strt);
					wmove(ppup, (w + 3),2 );
					strt = strtok(NULL,pickDelim);
				} 
				wmove(ppup,1,2);
				wrefresh(ppup);
				r = 0;
				keypad(ppup,TRUE);
				do
				{
					ch = wgetch(ppup);
					if(ch == KEY_DOWN)
					{
						if(r < y)
							++r;
						else
							r = 0;
					}
					if(ch == KEY_UP)
					{
						if(r > 0)
							--r;
						else
							r = y;
					}
					if(ch == KEY_PPAGE)
					{
						r = 0;
					}
					if(ch == KEY_NPAGE)
					{
						r = y ;
					}
					wmove(ppup, (r + 1),2);			
					wrefresh(ppup);
				} while((ch != screenKey) && (ch != fieldKey));
				if(ch == screenKey)
				{         
					--r; /* Account for blank line at start */
					if(r >= 0)
					{
						/* First clear the destination field */
						for(x = 0 ; x < fA[n].length; ++x)
						{
							wmove(stdscr,fA[n].ypos,(fA[n].xpos + x));
							waddch(stdscr,padchar);
						}
						fA[n].size = 0;
						fA[n].cxpos = fA[n].xpos;
						for(x=0;x < z[r]; ++x)
						{
							ch = winch(ppup);
							wmove(stdscr,fA[n].ypos,fA[n].cxpos++);
							waddch(stdscr,ch);
							fA[n].size++;
							wmove(ppup,r+2,x+3);
						}
					}
				}
			}
			else 
			/* There are more items than would fit on a single screen. It is
			 * necessary to create a scrolling window. As before y is the 
			 * number of items in the menu, x is the largest item, and z[] 
			 * contains the size of each menu item. We have created a new
			 * window box. We have to create another window inside of it
			 * which will scroll. p is the the height of the box, q is the
			 * width. To make the scrolling box we reduce each by 2 and
			 * we position it at one down from the box, one to the right.
			 */
			{
				wrefresh(ppup);
				p -= 4;
				q -= 2;
				popup = newwin(q,p,1,(COLS - p -2));
				if( popup == NULL)
				{
					delwin(ppup); /* close the box window. */
					touchwin(stdscr);
					refresh();
					return 0 ; /* This won't do - no memory available */
				}
				/* Fill the on screen array */
				strcpy(tstrng,fA[n].pickBox);
				strt = strtok(tstrng,pickDelim);			
				wmove(popup, 1,0);
				keypad(popup,TRUE);
				for(w = 0; w < LINES -3; ++w)
				{
					wprintw(popup,"%s",strt);
					wmove(popup, (w + 2),0 );
					strt = strtok(NULL,pickDelim);
				} 
				wrefresh(popup);										
				r = -1; 
				screenPos = 0; 
				/* r points to the element in the array we start it at one
				 * before the array's first element to account for the
				 * blank line at the start of the pick box. ScreenPos points
				 * to the line on the screen where the cursor is.
				 */
				wmove(popup,0,0);
				do 
				{
					ch = wgetch(popup);
					if(ch == KEY_DOWN)
					{
						if(screenPos < LINES -3)
						{
							++screenPos;
							++r;
							wmove(popup,screenPos,0);
						}
						else
						{
						/* We have a scroll situation to handle */
							if(r < y - 1)
							{
								scrollok(popup,TRUE);
								++r; /* Index to next position in list */
								wscrl(popup,1);
								strcpy(tstrng,fA[n].pickBox);
								strt = strtok(tstrng,pickDelim);
								for(p = 0; p < r; ++p)
								{
									strt = strtok(NULL,pickDelim);
								}
								scrollok(popup,FALSE);
								wprintw(popup,"%s",strt);
								wmove(popup,LINES -3,0);
								wrefresh(popup);
							}
						}
					}
					if(ch == KEY_UP)
					{
						if(screenPos > 0)
						{
							--screenPos;
							--r;
							wmove(popup,screenPos,0);
						}
						else
						if(r > -1)
						{
							scrollok(popup,TRUE);
							--r; /* Index to prev position in list */
							wscrl(popup,-1);
							if(r != -1)
							{
								strcpy(tstrng,fA[n].pickBox);
								strt = strtok(tstrng,pickDelim);
								for(p = 0; p < r; ++p)
								{
									strt = strtok(NULL,pickDelim);
								}
								scrollok(popup,FALSE);
								wprintw(popup,"%s",strt);
								wmove(popup,0,0);
								wrefresh(popup);
							}
						}
					}
					if(ch == KEY_PPAGE)
					{
						wmove(popup,0,0);
						scrollok(popup,FALSE);
						werase(popup);			
						strcpy(tstrng,fA[n].pickBox);
						strt = strtok(tstrng,pickDelim);			
						wmove(popup, 1,0);
						keypad(popup,TRUE);
						for(w = 0; w < LINES -3; ++w)
						{
							wprintw(popup,"%s",strt);
							wmove(popup, (w + 2),0 );
							strt = strtok(NULL,pickDelim);
						} 
						wmove(popup, 0, 0);
						wrefresh(popup);
						screenPos = 0;
						r = -1;										
					}
					if(ch == KEY_NPAGE)
					{
						wmove(popup,0,0);
						scrollok(popup,FALSE);
						werase(popup);			
						strcpy(tstrng,fA[n].pickBox);
						strt = strtok(tstrng,pickDelim);			
						for(w = 0; w < ((y -1) - (LINES -3)); w++)
						{
							strt = strtok(NULL,pickDelim);
						}
						wmove(popup, 0,0);
						keypad(popup,TRUE);
						for(w = 0; w < LINES -2; ++w)
						{
							wprintw(popup,"%s",strt);
							wmove(popup, (w + 1),0 );
							strt = strtok(NULL,pickDelim);
						} 
						wmove(popup,(LINES -3), 0);
						wrefresh(popup);										
						screenPos = LINES - 3;
						r = y - 1;
					}
				} while((ch != screenKey) && (ch != fieldKey));
				if(ch == screenKey)
				{
					if(r >= 0)
					{
						/* First clear the destination field */
						for(x = 0 ; x < fA[n].length; ++x)
						{
							wmove(stdscr,fA[n].ypos,(fA[n].xpos + x));
							waddch(stdscr,padchar);
						}
						fA[n].size = 0;
						fA[n].cxpos = fA[n].xpos;
						for(x=0;x < z[r]; ++x)
						{
							ch = winch(popup);
							wmove(stdscr,fA[n].ypos,fA[n].cxpos++);
							waddch(stdscr,ch);
							fA[n].size++;
							wmove(popup,screenPos,x+1);
						}
					}
				}
				delwin(popup); 
			} 
			delwin(ppup);
			touchwin(stdscr);
			refresh();							
		}
	}
	return(x);
}

int main(int argc, char *argv[])
{
int x;
int y;
char *aRgv;
int options;
	options = 0;
	x = 0;
	bgColor = COLOR_BLACK;
	padchar = '.'; /* Set up default pad character */
	screenKey = 0x0d;
	fieldKey = 0x09;
	SCI_DELIM = 0x09;
/* First clear out the data structure to hold the field data */
	number_of_fields = 0;
	localle=setlocale(LC_ALL, "" );
	if(localle == NULL)
	{
		fprintf(stderr,"Could not set locale!\n");
		exit(1);
	}
	initinput();
	signal(SIGINT,  die);
	for(x=0;x<MAX_FIELDS;x++)
	{
		fA[x].xpos = 0;
		fA[x].ypos = 0;
		fA[x].cxpos = 0;
		fA[x].cypos = 0;
		fA[x].length = 0;
		fA[x].size = 0;
		fA[x].flags = 0;
		fA[x].decpt = 0;
		fA[x].deffsize = 0;
		fA[x].charColor = COLOR_WHITE;
		for(y=0;y < MAX_WIDTH; y++)
		{
			fA[x].deff[y] = padchar;
			fA[x].fstring[y] = padchar;
			fA[x].fileName[y] = 0 ;
			fA[x].pickBox[y] = 0 ;
		}	
	}
	options = setopt(argc, argv,0); 
	file = sciReaderNew (templt);
	if(file == NULL)
		raise(SIGINT);
	number_of_fields = getTemPlate(file);
	sciReaderDestroy (file);
	file = NULL;

	for(x = 1; x < COLORS ; x++)
	{
		init_pair(x,x,bgColor);
	}

	aRgv = getenv("SCIPARMS");
 	if(aRgv  != NULL)
		options = setopt(argc, argv,1); 
	
	{
		FILE *file;
		file = fopen(outfile,"a");
		if(file == NULL)
			raise(SIGINT);

/* After the template and output files are open handle the input in the
 * main loop.
 */
		do
		{
			clearFields(number_of_fields,options);
			editScreen(number_of_fields,options);
			writeData(file,number_of_fields,options);
		} while((options & LOOPFLAG) != 0);

		fclose(file);
	}
	wmove(stdscr, COLS -1, LINES -1);
	endwin();
	return (0);
}

void die(int dnum)
{
	signal(SIGINT, SIG_IGN);
	if (file != NULL)
		sciReaderDestroy (file);
	wmove(stdscr, COLS -1, LINES -1);
	endwin();
	exit(1);
}

