/******************************************************************************
 *
 * scipp.c  Preprocessor implementation for the SCreen Input program.
 * Copyright (C) 1999 Jason M. Felice
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************/

#include "config.h"
#define _POSIX_SOURCE 1
#include <stdio.h>
#include <ctype.h>
#define __USE_MISC
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <signal.h>
#include <stdarg.h>
#include <errno.h>
#include <curses.h>

#include "scipp.h"

static void sciReaderReadLine (SciReaderPtr This);
static int sciReaderProcessCommand (SciReaderPtr This);
static char *sciReaderGetToken (SciReaderPtr This);
static void sciReaderPreProcessLine (SciReaderPtr This);
static int sciReaderDef (SciReaderPtr This);
static int sciReaderIfDef (SciReaderPtr This);
static int sciReaderIfNDef (SciReaderPtr This);
static int sciReaderElse (SciReaderPtr This);
static int sciReaderEndIf (SciReaderPtr This);
static int sciReaderUndef (SciReaderPtr This);
static int sciReaderInclude (SciReaderPtr This);
static int sciReaderCenter (SciReaderPtr This);

/******************************************************************************
 * SciReaderFile implementation.
 *****************************************************************************/

SciReaderFilePtr sciReaderFileNew (const char *name)
{
	SciReaderFilePtr This;
	FILE *f;

	if ((f = fopen (name, "r")) == NULL)
		return NULL;
	if ((This = (SciReaderFilePtr)malloc (sizeof (SciReaderFile))) == NULL)
	{
		fclose (f);
		return NULL;
	}
	This->f = f;
	This->line = 0;
	This->next = NULL;
	if ((This->name = (char *)malloc (strlen (name) + 1)) == NULL)
	{
		fclose (f);
		free (This);
		return NULL;
	}
	strcpy (This->name, name);
	return This;
}

void sciReaderFileDestroy (SciReaderFilePtr This)
{
	if (This->name != NULL)
		free (This->name);
	if (This->f != NULL)
		fclose (This->f);
	free (This);
}

char *sciReaderFileReadLine (SciReaderFilePtr This)
{
	char *ret;
	int allocated;
	int len;
	int ch;

	if (feof (This->f))
		return NULL; /* End of file */

	allocated = 256;
	len = 0;
	if ((ret = (char *)malloc (allocated)) == NULL)
		return NULL; /* Out of memory */
	
	do
	{
		ch = fgetc (This->f);
		if (ch != 0x1a && ch != EOF && ch != '\r')
		{
			ret[len++] = (char)ch;
			if (len >= allocated)
			{
				allocated += 256;
				ret = (char *)realloc (ret, allocated);
				if (ret == NULL)
					return NULL; /* Out of memory */
			}
		}
	} while (ch != EOF && ch != '\n');

	if (ch == EOF && len == 0)
	{
		free (ret);
		return NULL; /* End of file */
	}

	ret[len] = '\0';
	This->line++;
	return ret;
}

/******************************************************************************
 * SciReader implementation.
 *****************************************************************************/

SciReaderPtr sciReaderNew (const char *file)
{
	SciReaderPtr This;
	SciReaderFilePtr filePtr;

	if ((This = (SciReaderPtr)malloc (sizeof (SciReader))) == NULL)
		return NULL; /* Out of memory */

	if ((filePtr = sciReaderFileNew (file)) == NULL)
	{
		free (This);
		return NULL; /* File error */
	}

	This->conditionStack = NULL;
	This->fileStack = filePtr;
	This->lineBuffer = NULL;
	This->lineIndex = 0;
	This->macroPrefix = '@';
	return This;
}

void sciReaderDestroy (SciReaderPtr This)
{
	SciReaderFilePtr topFile, nextFile;
	SciReaderCondPtr topCond, nextCond;

	topFile = This->fileStack;
	while (topFile != NULL)
	{
		nextFile = topFile->next;
		sciReaderFileDestroy (topFile);
		topFile = nextFile;
	}

	topCond = This->conditionStack;
	while (topCond != NULL)
	{
		nextCond = topCond->next;
		free (topCond);
		topCond = nextCond;
	}

	free (This);
}

int sciReaderGetChar (SciReaderPtr This)
{
	if (This->lineBuffer == NULL || This->lineBuffer[This->lineIndex] == '\0')
	{
		if (This->fileStack == NULL)
			return EOF; /* End of file(s) */
		sciReaderReadLine (This);
	}

	if (This->lineBuffer == NULL)
		return EOF; /* End of file(s) */

	return This->lineBuffer[This->lineIndex++];
}

static void sciReaderReadLine (SciReaderPtr This)
{
	This->lineIndex = 0;
	while (1)
	{
		if (This->lineBuffer != NULL)
		{
			free (This->lineBuffer);
			This->lineBuffer = NULL;
		}

		if (This->fileStack == NULL)
			break; /* End of file(s) */

		This->lineBuffer = sciReaderFileReadLine (This->fileStack);
		if (This->lineBuffer == NULL)
		{
			/* End of an input file, pop one off the stack. */
			SciReaderFilePtr topFile = This->fileStack;
			This->fileStack = This->fileStack->next;
			sciReaderFileDestroy (topFile);	
			continue;
		}

		if (This->lineBuffer[0] == '#' && This->lineBuffer[1] != ':')
			continue; /* A comment or a executable magic */

		if (This->lineBuffer[0] == '@')
		{
			int ignoreLine;
			ignoreLine = sciReaderProcessCommand (This);
			if (ignoreLine)
				continue;
		}

		/* This way, we drop lines between @ifdef ... @endif depending 
		 * on how the expressions were evaluated. */
		if (This->conditionStack == NULL || This->conditionStack->value)
		{
			/* Interpolate $VAR and ${VAR} strings. */
			sciReaderPreProcessLine (This);
			break;
		}
	}
}

static int sciReaderProcessCommand (SciReaderPtr This)
{
	char cmdName[20];
	int cmdLen = 0;
	char *scanp = This->lineBuffer + 1;

	while (*scanp != '\0' && (*scanp == '_' || isalpha(*scanp) || isdigit(*scanp)))
	{
		if (cmdLen >= 19)
			break;
		cmdName[cmdLen++] = *scanp++;
	}
	cmdName[cmdLen] = '\0';

	while (*scanp && isspace (*scanp))
		scanp++;

	This->ppScan = scanp;

	if (!strcmp (cmdName, "def"))
		return sciReaderDef (This);
	else if (!strcmp (cmdName, "ifdef"))
		return sciReaderIfDef (This);
	else if (!strcmp (cmdName, "ifndef"))
		return sciReaderIfNDef (This);
	else if (!strcmp (cmdName, "else"))
		return sciReaderElse (This);
	else if (!strcmp (cmdName, "endif"))
		return sciReaderEndIf (This);
	else if (!strcmp (cmdName, "undef"))
		return sciReaderUndef (This);
	else if (!strcmp (cmdName, "include"))
		return sciReaderInclude (This);
	else if (!strcmp (cmdName, "center"))
		return sciReaderCenter (This);
	else
		sciReaderError (This, "@%s: unrecognized preprocessor command.", cmdName);

	return 0; /* UNREACH */
}

void sciReaderError (SciReaderPtr This, const char *fmt, ...)
{
	va_list vl;
	char errorBuffer[256];

	va_start (vl,fmt);
	if (This->fileStack != NULL)
		sprintf (errorBuffer, "%s:%d: ", This->fileStack->name, This->fileStack->line);
	else
		strcpy (errorBuffer, "file?:line?: ");
	vsprintf (errorBuffer + strlen (errorBuffer), fmt, vl);
	va_end (vl);

	/* FIXME: The error message gets written to stderr before curses
	 * is shut down. */
	fprintf (stderr, "%s\n", errorBuffer);
	raise (SIGINT);
}

/* Read the next token (parameter) from the directive line.
 *
 * FIXME: Currently we don't support quoting or variable expansion. */
static char *sciReaderGetToken (SciReaderPtr This)
{
	char *scanp = This->ppScan;
	char *val;
	int valAllocated = 256;
	int valLen = 0;

	while (*scanp && isspace (*scanp))
		scanp++;

	if (*scanp == '\0')
		return NULL;

	val = (char *)malloc (valAllocated);
	if (val == NULL)
		sciReaderError (This, "Out of memory");

	while (*scanp && !isspace (*scanp))
	{
		val[valLen++] = *scanp;
		if (valLen >= valAllocated)
		{
			val = realloc (val, valAllocated += 256);
			if (val == NULL)
				sciReaderError (This, "Out of memory");
		}
		scanp++;
	}
	val[valLen] = '\0';

	This->ppScan = scanp;
	return val;
}

/* @def var [value ...]
 *
 * This command is similar to the C preprocessor #define directive, except
 * that it operates on environment variables instead of a macro space.
 * Additional values are concatenated (with spaces in between)
 * as in the @center command below.  *REC II 7/12/99* */
static int sciReaderDef (SciReaderPtr This)
{
	char *name, *value;
	char *text = NULL;
	char *segment;
	
	/* Don't do this if we are in an @if that's false. */
	if (This->conditionStack != NULL && !This->conditionStack->value)
		return 1;

	if ((name = sciReaderGetToken (This)) == NULL)
		sciReaderError (This, "@def: missing argument.");
	while ((segment = sciReaderGetToken (This)) != NULL)
	{
		if (text == NULL)
			text = segment;
		else
		{
			char *newText = (char *)malloc (strlen (text) +
					strlen (segment) + 2);
			strcpy (newText, text);
			strcat (newText, " ");
			strcat (newText, segment);
			free (text);
			text = newText;
			free (segment);
		}
	}
	value = text;
	sciReaderSetenv (This, name, value);
	free (name);
	if (value != NULL)
		free (value);
	return 1;
}

/* @ifdef var
 *
 * Pushes a new condition onto the condition stack.  If `var' exists in
 * the current environment and the topmost condition is true, the new
 * condition is set to true. */
static int sciReaderIfDef (SciReaderPtr This)
{
	SciReaderCondPtr newCond;
	char *name;

	if ((name = sciReaderGetToken (This)) == NULL)
		sciReaderError (This, "@ifdef: missing argument.");

	if ((newCond = (SciReaderCondPtr)malloc (sizeof (SciReaderCond))) == NULL)
		sciReaderError (This, "Out of memory");

	if (This->conditionStack == NULL || This->conditionStack->value == 1)
	{
		newCond->value = (sciReaderGetenv (This, name) != NULL);
		newCond->canFlip = 1;
	}
	else
	{
		newCond->value = 0;
		newCond->canFlip = 0;
	}

	newCond->next = This->conditionStack;
	This->conditionStack = newCond;
	return 1;
}

/* @ifndef var
 *
 * Oposite of @ifdef. */
static int sciReaderIfNDef (SciReaderPtr This)
{
	SciReaderCondPtr newCond;
	char *name;

	if ((name = sciReaderGetToken (This)) == NULL)
		sciReaderError (This, "@ifndef: missing argument.");

	if ((newCond = (SciReaderCondPtr)malloc (sizeof (SciReaderCond))) == NULL)
		sciReaderError (This, "Out of memory");

	if (This->conditionStack == NULL || This->conditionStack->value == 1)
	{
		newCond->value = (sciReaderGetenv (This, name) == NULL);
		newCond->canFlip = 1;
	}
	else
	{
		newCond->value = 0;
		newCond->canFlip = 0;
	}

	newCond->next = This->conditionStack;
	This->conditionStack = newCond;
	return 1;
}

/* @else */
static int sciReaderElse (SciReaderPtr This)
{
	if (This->conditionStack == NULL)
		sciReaderError (This, "@else: Nothing to @else!");

	if (This->conditionStack->canFlip)
	{
		This->conditionStack->value = !This->conditionStack->value;
		This->conditionStack->canFlip = 0;
	}
	return 1;
}

/* @endif */
static int sciReaderEndIf (SciReaderPtr This)
{
	SciReaderCondPtr topCond;

	if (This->conditionStack == NULL)
		sciReaderError (This, "@endif: Nothing to @endif!");

	topCond = This->conditionStack;
	This->conditionStack = This->conditionStack->next;
	free (topCond);
	return 1;
}

/* @undef var [var ...]
 *
 * This command unsets environment variable(s). */
static int sciReaderUndef (SciReaderPtr This)
{
	char *name;

	/* Don't do this if we are in an @if that's false. */
	if (This->conditionStack != NULL && !This->conditionStack->value)
		return 1;


	while ((name = sciReaderGetToken (This)) != NULL)
	{
		sciReaderUnsetenv (This, name);
		free (name);
	}
	return 1;
}

/* @include file
 *
 * This command performs the same function as the C preprocessor's #include
 * directive. */
static int sciReaderInclude (SciReaderPtr This)
{
	char *name = sciReaderGetToken (This);
	SciReaderFilePtr newFile;
	if (name == NULL)
		sciReaderError (This, "@include: parameter required.");

	/* Don't do this if we are in an @if that's false. */
	if (This->conditionStack != NULL && !This->conditionStack->value)
	{
		free (name);
		return 1;
	}

	newFile = sciReaderFileNew (name);
	if (newFile == NULL)
	{
		free (name);
		sciReaderError (This, "@include: %s: %s.", name, strerror (errno));
	}

	newFile->next = This->fileStack;
	This->fileStack = newFile;
	free (name);
	return 1;
}

/* @center text [text ...]
 *
 * This command concatenates all the `text' arguments (inserting spaces between
 * the arguments), retrieves the current screen width, and replaces the current
 * input line with a copy of the concatentated text, except that it has been
 * padded such that it is centered. */
static int sciReaderCenter (SciReaderPtr This)
{
	char *text = NULL;
	char *segment;
	int y, x;

	/* Don't do this if we are in an @if that's false. */
	if (This->conditionStack != NULL && !This->conditionStack->value)
		return 1;

	while ((segment = sciReaderGetToken (This)) != NULL)
	{
		if (text == NULL)
			text = segment;
		else
		{
			char *newText = (char *)malloc (strlen (text) +
					strlen (segment) + 2);
			strcpy (newText, text);
			strcat (newText, " ");
			strcat (newText, segment);
			free (text);
			text = newText;
			free (segment);
		}
	}

	/* Get the width of the screen. */
	/* FIXME: I don't like #include <curses.h> in this file, and I don't
	 * like using curses functions here - JMF */
	getmaxyx (stdscr, y, x);

	if (text != NULL)
	{
		if (x > strlen (text))
		{
			/* y is the amount to left-pad, segment is the output. */
			y = (x - strlen (text)) / 2;
			segment = (char *)malloc (strlen (text) + y + 2);
			memset (segment, ' ', y);
			strcpy (segment + y, text);
			strcat (segment, "\n");
			free (text);
			text = segment;
		}
	}
	else
	{
		/* No text to center - insert a blank line. */
		if ((text = (char *)malloc (1)) == NULL)
			sciReaderError (This, "Out of memory");
		text[0] = '\0';
	}

	/* Replace the current line with the formatted output. */
	if (This->lineBuffer != NULL)
		free (This->lineBuffer);

	This->lineBuffer = text;
	return 0;
}

/* Replace @VAR and @{VAR} with the value of the environment variable
 * VAR.
 *
 * I used to scan through twice, once to obtain the length, the 
 * second time to build the string.  This was to reduce the number of
 * malloc/realloc calls.  After coding this, however, I realized what a
 * maintenance nightmare it would be.  This is a sort of compromise,
 * since we allocate with a blocking factor of 256. */
static void sciReaderPreProcessLine (SciReaderPtr This)
{
	char *scanPtr = This->lineBuffer;
	int lineLength = strlen (This->lineBuffer);
	char varName[64];
	int varNameLen;
	const char *varValue;
	char *dstLine;
	int dstIndex;
	int dstAlloc;
	int newAlloc;

	/* Allocate in increments of 256 bytes. */
	dstAlloc = (((lineLength + 1) / 256) + 1) * 256;
	dstIndex = 0;
	dstLine = (char *)malloc (dstAlloc);
	if (dstLine == NULL)
		sciReaderError (This, "Out of memory");

	while (*scanPtr)
	{
		while (*scanPtr && *scanPtr != This->macroPrefix)
		{
			dstLine[dstIndex] = *scanPtr;
			scanPtr++;
			dstIndex++;
		}

		if (*scanPtr == This->macroPrefix) 
		{
			/* Encountered a variable, scan through, subtracting
			 * the length of the name, including `$', `{', and
			 * `}', and get the name of the variable. */
			scanPtr++;
			lineLength--;
			varNameLen = 0;

			if (*scanPtr == '{')
			{
				scanPtr++;
				lineLength--;
				while (*scanPtr && *scanPtr != '}')
				{
					varName[varNameLen++] = *scanPtr;
					scanPtr++;
					lineLength--;
					if (varNameLen >= sizeof (varName))
						sciReaderError (This, "variable name too long.");
				}
				if (*scanPtr == '}')
				{
					scanPtr++;
					lineLength--;
				}
			}
			else if (isalpha (*scanPtr) || *scanPtr == '_')
			{
				while (*scanPtr && (isalpha(*scanPtr) ||
							isdigit(*scanPtr) ||
							*scanPtr == '_'))
				{
					varName[varNameLen++] = *scanPtr;
					if (varNameLen >= sizeof (varName))
						sciReaderError (This, "variable name too long.");
					scanPtr++;
					lineLength--;
				}
			}
			else
			{
				/* As in Perl, we have weird `special' variable names which
				 * are one-character-long punctuation. */
				varName[varNameLen++] = *scanPtr++;
				lineLength --;
			}

			/* Add the length of the variable's value. */
			varName[varNameLen] = '\0';
			if ((varValue = sciReaderGetenv (This, varName)) != NULL)
				lineLength += strlen (varValue);

			/* Reallocate if we need to. */
			newAlloc = (((lineLength + 1) / 256) + 1) * 256;
			if (dstAlloc != newAlloc)
			{
				dstLine = (char *)realloc (dstLine, newAlloc);
				dstAlloc = newAlloc;
			}

			/* Substitute */
			if (varValue != NULL)
			{
				strcpy (dstLine + dstIndex, varValue);
				dstIndex += strlen (varValue);
			}
		}
	}

	/* NUL-terminate and replace the current line with the pre-processed
	 * line. */
	dstLine[dstIndex] = '\0';
	free (This->lineBuffer);
	This->lineBuffer = dstLine;
}

void sciReaderSetenv (SciReaderPtr This, const char *name, const char *value)
{
#ifdef HAVE_SETENV
	setenv (name, value, 1);
#else
	char *buf;
	if ((buf = (char *)malloc (strlen (name) + (value ? strlen (value) : 0) + 2))
			== NULL)
		sciReaderError (This, "Out of memory");

	sprintf (buf, "%s=%s", name, (value ? value : ""));
	putenv (buf);
	free (buf);
#endif
}

const char *sciReaderGetenv (SciReaderPtr This, const char *name)
{
	/* FIXME: This should evaluate to a stringized version of This->macroPrefix */
	if (!strcmp (name, "@"))
		return "@";

	return getenv (name);
}

void sciReaderUnsetenv (SciReaderPtr This, const char *name)
{
#ifdef HAVE_UNSETENV
	unsetenv (name);
#endif
	/* FIXME: What to do? */
}
