#ifndef SCIPPH
#define SCIPPH

/******************************************************************************
 *
 * scipp.h  Preprocessor implementation for the SCreen Input program.
 * Copyright (C) 1999 Jason M. Felice
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************/

/******************************************************************************
 * SciReaderFile is a single input file, not taking into account @include
 * directives.  It is mostly here to replace FILE * with the exception that
 * it keeps track of the next-down @include'd file.
 *****************************************************************************/
struct SciReaderFileTag {
	struct SciReaderFileTag *next; 	/* For the @include stack */
	FILE *f;			/* The stdio stream */
	char *name;			/* A human-readable name for errors */
	int line;			/* A line number for errors */
};

typedef struct SciReaderFileTag SciReaderFile;
typedef struct SciReaderFileTag *SciReaderFilePtr;

extern SciReaderFilePtr sciReaderFileNew (const char *fname);
extern void sciReaderFileDestroy (SciReaderFilePtr This);

/* Result is malloc'd and must be free'd. */
extern char *sciReaderFileReadLine (SciReaderFilePtr This);

/******************************************************************************
 * SciReaderCond is a stack node for SciReader also, since nesting of
 * @if statements requires that we keep track of more than a simple
 * yes/no as to whether we emit text.
 *****************************************************************************/
struct SciReaderCondTag {
	struct SciReaderCondTag *next;	/* For the @if/@endif condition stack */
	int value;			/* Is this condition currently true? */
	int canFlip;			/* Specifies the effect of an @else or
					   @elsif */
};

typedef struct SciReaderCondTag SciReaderCond;
typedef struct SciReaderCondTag *SciReaderCondPtr;

/******************************************************************************
 * SciReader is the master structure that controls the whole shebang. 
 *****************************************************************************/
struct SciReaderTag {
	SciReaderFilePtr fileStack;	/* Stack of @include'd files */
	SciReaderCondPtr conditionStack;/* Stack of @if/@endif conditions */
	char *lineBuffer;		/* Current line we are reading */
	int lineIndex;			/* Index into line of current char */
	char *ppScan;			/* Used for parsing macro arguments */
	char macroPrefix;		/* Macro prefix character (def: @) */
};

typedef struct SciReaderTag SciReader;
typedef struct SciReaderTag *SciReaderPtr;

extern SciReaderPtr sciReaderNew (const char *file);
extern void sciReaderDestroy (SciReaderPtr This);
extern void sciReaderError (SciReaderPtr This, const char *fmt, ...);
extern void sciReaderSetenv (SciReaderPtr This, const char *name, const char *value);
extern const char *sciReaderGetenv (SciReaderPtr This, const char *name);
extern void sciReaderUnsetenv (SciReaderPtr This, const char *name);
extern int sciReaderGetChar (SciReaderPtr This);

#endif
