/*
 * tnmMibTree.c --
 *
 *	Some utilities to build a tree that contains the data found 
 *	in MIB definitions.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/*
 * The following table is used to hash nodes by name before building
 * the MIB tree. This has the nice effect that nodes with the same
 * parent can be found in the list that saves the collisions.
 */

#define NODEHASHSIZE	127
static Tnm_MibNode	*nodehashtab[NODEHASHSIZE];

/*
 * Hashtable used to store textual conventions by name. This
 * allows fast lookups. The nodeHashTable is used to lookup
 * MIB names.
 */

static Tcl_HashTable *tcHashTable = NULL;
static Tcl_HashTable *nodeHashTable = NULL;

/*
 * Forward declarations for procedures defined later in this file:
 */

static Tnm_MibNode*
LookupOID		_ANSI_ARGS_((Tnm_MibNode *root, char *label,
				     int *offset, int exact));
static Tnm_MibNode*
LookupLabelOID		_ANSI_ARGS_((Tnm_MibNode *root, char *label,
				     int *offset, int exact));
static Tnm_MibNode*
LookupLabel		_ANSI_ARGS_((Tnm_MibNode *root, char *start, 
				     char *label, char *moduleName,
				     int *offset, int exact, int fuzzy));
static void
HashNode		_ANSI_ARGS_((Tnm_MibNode *node));

static Tnm_MibNode*
BuildTree		_ANSI_ARGS_((Tnm_MibNode *nlist));

static void
BuildSubTree		_ANSI_ARGS_((Tnm_MibNode *root));

static void
HashNodeList		_ANSI_ARGS_((Tnm_MibNode *nlist));

static int
HashNodeLabel		_ANSI_ARGS_((char *label));


/*
 *----------------------------------------------------------------------
 *
 * LookupOID --
 *
 *	This procedure searches for the node given by soid. It converts 
 *	the oid from string to binary representation and uses to subids
 *	to go from the root to the requested node. 
 *
 * Results:
 *	The pointer to the node or NULL if the node was not found.
 *	The offset to any trailing subidentifiers is written to offset, 
 *	if offset is not a NULL pointer.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static Tnm_MibNode*
LookupOID(root, label, offset, exact)
    Tnm_MibNode *root;
    char *label;
    int *offset;
    int exact;
{
    Tnm_Oid *id;
    int i, len;
    Tnm_MibNode *p, *q = NULL;
    char *s = label;

    if (offset) *offset = -1;

    id = Tnm_StrToOid(label, &len);

    for (p = root; p ; p = p->nextPtr) {
	if (id[0] == p->subid) break;
    }
    if (!p) return NULL;

    while (offset && s && ispunct(*s)) s++;
    while (offset && s && isdigit(*s)) s++;

    for (q = p, i = 1; i < len; p = q, i++) {
	for (q = p->childPtr; q; q = q->nextPtr) {
	    if (q->subid == id[i]) break;
	}
	if (!q) {
	    if (! exact) {
		if (offset) {
		    *offset = s - label;
		}
		return p;
	    } else {
		return NULL; 
	    }
	}
	while (offset && s && ispunct(*s)) s++;
	while (offset && s && isdigit(*s)) s++;
    }

    return q;
}

/*
 *----------------------------------------------------------------------
 *
 * LookupLabelOID --
 *
 *	This procedure searches for the a MIB node given the very 
 *	common format label.oid where label is a simple name of a MIB 
 *	node and oid an object identifier in dotted notation. If this 
 *	is true and the lookup is not exact, search for the label in 
 *	the hash table and append the object identifier part to the 
 *	oid of the label. (This a special case for frequently used 
 *	names like snmpInTooBigs.0 which tend otherwise to be slow.)
 *
 * Results:
 *	The pointer to the node or NULL if the node was not found.
 *	The offset to any trailing subidentifiers is written to offset, 
 *	if offset is not a NULL pointer.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static Tnm_MibNode*
LookupLabelOID(root, label, offset, exact)
    Tnm_MibNode *root;
    char *label;
    int *offset;
    int exact;
{
    Tcl_HashEntry *entryPtr = NULL;
    Tnm_MibNode *nodePtr = NULL;

    if (exact) {
	return NULL;
    }

    if (nodeHashTable) {
	char *name = ckstrdup(label);
	char *oid = name;
	while (*oid && *oid != '.') {
	    oid++;
	}
	if (*oid && Tnm_IsOid(oid)) {
	    *oid++ = '\0';
	    entryPtr = Tcl_FindHashEntry(nodeHashTable, name);
	    if (entryPtr) {
		nodePtr = (Tnm_MibNode *) Tcl_GetHashValue(entryPtr);
	    }
	    if (nodePtr) {
		if (offset) {
		    *offset = oid - name - 1;
		}
		ckfree(name);
		return nodePtr;
	    }
	}
	ckfree(name);
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * LookupLabel --
 *
 *	This procedure extracts the first element (head) of the label
 *	argument and recursively searches for a tree node named head.
 *	If found, it checks whether the remaining label elements (tail)
 *	can be resolved from this node.
 *
 * Results:
 *	The pointer to the node or NULL if the node was not found.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static Tnm_MibNode*
LookupLabel(root, start, label, moduleName, offset, exact, fuzzy)
    Tnm_MibNode *root;
    char *start;
    char *label;
    char *moduleName;
    int *offset;
    int exact;
    int fuzzy;
{
    char head[TNM_OIDMAXLEN * 8];
    char *tail = label, *p = head;
    Tnm_MibNode *tp = NULL, *brother;
    int num = 1;

    if (!root) return NULL;

    if (offset) *offset = -1;

    while (*tail && *tail != '.') {
	num = isdigit(*tail) ? num : 0;
	*p++ = *tail++;
    }
    *p = '\0';
    if (*tail == '.') tail++;
    num = num ? atoi(head) : -1;

    for (brother = root; brother; brother = brother->nextPtr) {
        if (((strcmp (head, brother->label) == 0)
	     && (*moduleName == '\0'
		 || strcmp(moduleName, brother->moduleName) == 0))
	    || (num == brother->subid)) {
	    if (! *tail) {
		tp = brother;
	    } else if (brother->childPtr) {
		tp = LookupLabel(brother->childPtr, start, tail, 
				 moduleName, offset, exact, 0);
	    } else if (! exact) {
		tp = brother;
	    }
	    if (tp) {
		if (offset && (*offset < tail-start-1) && *offset != -2) {
		    *offset = *tail ? tail-start-1 : -2;
		}
		return tp;
	    }
	}
	if (fuzzy && brother->childPtr) {
	    tp = LookupLabel(brother->childPtr, start, label, 
			     moduleName, offset, exact, 1);
	    if (tp) {
		if (offset && (*offset < tail-start-1) && *offset != -2) {
		    *offset = *tail ? tail-start-1 : -2;
		}
		return tp;
	    }
	}
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibFindNode --
 *
 *	This procedure takes the name of a MIB node and searches for
 *	the corresponding node in the MIB tree. The object identifier
 *	of the node is stored at soid if soid is not NULL. soid must
 *	be large enough to hold the complete path. Tnm_MibFindNode()
 *	calls LookupOID() if the name is an object identier for fast
 *	MIB searches. Otherwise we call slow LookupLabel() to compare
 *	the tree node labels.
 *
 * Results:
 *	The pointer to the node or NULL if the node was not found.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tnm_MibNode*
Tnm_MibFindNode(name, offset, exact)
    char *name;
    int *offset;
    int exact;
{
    Tnm_MibNode *nodePtr = NULL;
    char *expanded;
    char moduleName[255];
    char *p = NULL;
    int moduleNameLen = 0;

    *moduleName = '\0';
    p = strchr(name, '!');
    if (p) {
	moduleNameLen = p - name;
	if (moduleNameLen < 255) {
	    strncpy(moduleName, name, moduleNameLen);
	    moduleName[moduleNameLen] = '\0';
	} else {
	    strcpy(moduleName, "********");
	}
	name += moduleNameLen + 1;
    }

    expanded = Tnm_HexToOid(name);
    if (expanded) name = expanded;

    if (Tnm_IsOid(name)) {
	nodePtr = LookupOID(tnm_MibTree, name, offset, exact);
    } else {
	Tcl_HashEntry *entryPtr = NULL;
	if (nodeHashTable) {
	    entryPtr = Tcl_FindHashEntry(nodeHashTable, name);
	}
	if (entryPtr) {
	    nodePtr = (Tnm_MibNode *) Tcl_GetHashValue(entryPtr);
	}
	if (! nodePtr) {
	    nodePtr = LookupLabelOID(tnm_MibTree, name, offset, exact);
	}
	if (! nodePtr) {
	    nodePtr = LookupLabel(tnm_MibTree, name, name, moduleName, 
				  offset, exact, 1);
	}
    }

    /*
     * If we have a file name prefix, we must ensure that the node
     * is really the node we were looking for.
     */

    if (nodePtr && *moduleName) {
	if (strcmp(moduleName, nodePtr->moduleName) == 0) {
	    if (offset && *offset > 0) {
		*offset += moduleNameLen + 1;
	    }
	} else {
	    nodePtr = NULL;
	}
    }

    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibAddTC --
 *
 *	This procedure adds a Tnm_MibTC structure to the set of known
 *	textual conventions which are saved in a hash table. The
 *	Tnm_MibTC structure is also linked into the TC list.
 *
 * Results:
 *	The pointer to the Tnm_MibTC structure.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tnm_MibTC*
Tnm_MibAddTC(tcPtr)
    Tnm_MibTC *tcPtr;
{
    char *name = tcPtr->name;
    Tcl_HashEntry *entryPtr;
    int isnew;

    if (! tcHashTable) {
	tcHashTable = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(tcHashTable, TCL_STRING_KEYS);
    }
    
    entryPtr = Tcl_CreateHashEntry(tcHashTable, name, &isnew);
    
    if (! isnew) {
	return (Tnm_MibTC *) Tcl_GetHashValue(entryPtr);
    }

    tcPtr->nextPtr = tnm_MibTCList;
    tnm_MibTCList = tcPtr;
    
    Tcl_SetHashValue(entryPtr, (ClientData) tcPtr);

    return tcPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibGetTC --
 *
 *	This procedure searches for a textual convention given by name.
 *
 * Results:
 *	The pointer to the Tnm_MibTC structure or NULL if the name
 *	can not be resolved.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tnm_MibTC*
Tnm_MibFindTC(name)
    char *name;
{
    Tcl_HashEntry *entryPtr;
    
    if (! tcHashTable) {
	return NULL;
    }

    entryPtr = Tcl_FindHashEntry(tcHashTable, name);
    if (entryPtr == NULL) {
	return NULL;
    }

    return (Tnm_MibTC *) Tcl_GetHashValue(entryPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibNewNode --
 *
 *	This procedure allocates a new tree element and does some
 *	basic initializations.
 *
 * Results:
 *	The pointer the new Tnm_MibNode structure is returned.
 *
 * Side effects:
 *	Memory is allocated.
 *
 *----------------------------------------------------------------------
 */

Tnm_MibNode*
Tnm_MibNewNode(label)
    char *label;
{
    Tnm_MibNode *nodePtr = (Tnm_MibNode *) ckalloc(sizeof(Tnm_MibNode));
    memset((char *) nodePtr, 0, sizeof(Tnm_MibNode));
    if (label) {
	nodePtr->label = ckstrdup(label);
    }
    nodePtr->syntax = ASN1_OBJECT_IDENTIFIER;
    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * HashNode --
 *
 *	This procedure maintans a hash table which is used to lookup
 *	MIB nodes by names. This works well as long as the label of
 *	the node is unique. Otherwise, we have to recurse on the tree.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
HashNode(nodePtr)
    Tnm_MibNode *nodePtr;
{
    char *name = nodePtr->label;
    Tcl_HashEntry *entryPtr;
    int isnew;
    
    if (! nodeHashTable) {
	nodeHashTable = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(nodeHashTable, TCL_STRING_KEYS);
    }
    
    entryPtr = Tcl_CreateHashEntry(nodeHashTable, name, &isnew);
    
    if (! isnew) {
	if (nodePtr != (Tnm_MibNode *) Tcl_GetHashValue(entryPtr)) {

	    /*
	     * leave in hashtable, but mark with NULL ptr:
	     */

	    Tcl_SetHashValue(entryPtr, (ClientData) 0);
	}
	return;
    }

    Tcl_SetHashValue(entryPtr, (ClientData) nodePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * BuildTree --
 *
 *	This procedure builds a MIB tree for the nodes given in the
 *	nodeList parameter.
 *
 * Results:
 *	Returns a pointer to root if successful and NULL otherwise.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static Tnm_MibNode*
BuildTree(nodeList)
    Tnm_MibNode *nodeList;
{
    Tnm_MibNode *ccitt, *iso, *joint;
    
    HashNodeList(nodeList);

    /*
     * Initialize the MIB tree with three default nodes: 
     * ccitt(0), iso(1) and joint-iso-ccitt (2).
     */

    ccitt = Tnm_MibNewNode("ccitt");
    ccitt->parentName = ckstrdup("(unknown)");
    ccitt->syntax = ASN1_OBJECT_IDENTIFIER;
    
    iso = Tnm_MibNewNode("iso");
    iso->parentName = ckstrdup("(unknown)");
    iso->subid = 1;
    iso->syntax = ASN1_OBJECT_IDENTIFIER;
    ccitt->nextPtr = iso;

    joint = Tnm_MibNewNode("joint-iso-ccitt");
    joint->parentName = ckstrdup("(unknown)");
    joint->subid = 2;
    joint->syntax = ASN1_OBJECT_IDENTIFIER;
    iso->nextPtr = joint;
    
    /* 
     * Build the tree out of these three nodes.
     */

    BuildSubTree(ccitt);
    BuildSubTree(iso);
    BuildSubTree(joint);

    /*
     * Return the first node in the tree (perhaps "iso" would be
     * better, but this way is consistent)
     */
    
    return ccitt;
}

/*
 *----------------------------------------------------------------------
 *
 * BuildSubTree --
 *
 *	This procedure finds all the children of root in the list of
 *	nodes and links them into the tree and out of the node list.
 *	nodeList parameter.
 *
 * Results:
 *	Returns a pointer to root if successful and NULL otherwise.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
BuildSubTree(root)
    Tnm_MibNode *root;
{
    Tnm_MibNode **np, **ptr;
    int	hash = HashNodeLabel(root->label);

    /*
     * Loop through all nodes whose parent is root. They are all
     * members of the list which starts at the same hash key.
     */

    for (np = &nodehashtab[hash]; *np != NULL; ) {
	
	if (root->label[0] == ((*np)->parentName)[0]
	    && (strcmp(root->label, (*np)->parentName) == 0)) {

	    Tnm_MibNode *thisNode = *np;
	    *np = (*np)->nextPtr;

	    thisNode->fileName = tnm_MibFileName;

/*** XXX: This should be freed if the data came from the parser but
          it may not be freed if the data came from a frozen file.
          Same below.
	    if (thisNode->parentName) {
		ckfree(thisNode->parentName);
		thisNode->parentName = NULL;
	    }
***/
	    thisNode->parentPtr = root;
	    thisNode->childPtr = NULL;
	    thisNode->nextPtr  = NULL;
	    
	    /* 
	     * Link node in the tree. First skip over all nodes with a
	     * subid less than the new subid. Insert the node if the
	     * node does not already exist. Otherwise free this node.
	     */

	    ptr = &root->childPtr;
	    while (*ptr && (*ptr)->subid < thisNode->subid) {
		ptr = &(*ptr)->nextPtr;
	    }
	    if (*ptr && (*ptr)->subid == thisNode->subid) {
/*** XXX	if (thisNode->label) ckfree((char *) thisNode->label);
		ckfree((char *) thisNode);		***/
	    } else {
		thisNode->nextPtr = *ptr;
                *ptr = thisNode;
		HashNode(thisNode);
	    }

	    BuildSubTree(*ptr);			/* recurse on child */

	} else {
	    np = &(*np)->nextPtr;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibAddNode --
 *
 *	This procedure adds all nodes in nodeList to the MIB tree
 *	given by rootPtr. It first initializes the hash table and
 *	calls BuildSubTree() to move nodes from the hash table into 
 *	the MIB tree.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The nodes are moved from the nodeList into the correct 
 *	position in the MIB tree.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_MibAddNode(rootPtr, nodeList)
    Tnm_MibNode **rootPtr;
    Tnm_MibNode *nodeList;
{
    Tnm_MibNode *nodePtr;
    Tnm_MibNode *tree;
    Tnm_MibNode *root = *rootPtr;
    int i;

    if (! nodeList) return;

    if (! root) {
	*rootPtr = BuildTree(nodeList);
    }

   /*
    * Test if the parent of the first node exists. We expect that we
    * load individual subtrees where the parent of the first node
    * defines the anchor. This must already exist. Note, the first 
    * node is the last node in our nodeList.
    */

    for (nodePtr = nodeList; nodePtr->nextPtr; nodePtr = nodePtr->nextPtr) ;
    tree = Tnm_MibFindNode(nodePtr->parentName, NULL, 1);
    HashNodeList(nodeList);
    if (tree) {
	BuildSubTree(tree);
    }
    
   /*
    * Now the slow part: If there are any nodes left in the hash table,
    * we either have two isolated subtrees or the last node in nodeList
    * was not the root of our tree. We scan for nodes which have known
    * parents in the existing tree and call BuildSubTree for those
    * parents.
    *
    * Perhaps we should check for candidates in the nodelist, as this
    * would allow some local scope. XXX
    */

  repeat:
    for (i = 0; i < NODEHASHSIZE; i++) {
	for (nodePtr = nodehashtab[i]; nodePtr; nodePtr = nodePtr->nextPtr) {
	    tree = Tnm_MibFindNode(nodePtr->parentName, NULL, 1);
	    if (tree) {
		BuildSubTree(tree);
		goto repeat;
	    }
	}
    }

    /* 
     * Finally print out all the node labels that are still in the
     * hashtable. This should allow people to fix the ordering of
     * MIB load commands.
     */
    
    for (i = 0; i < NODEHASHSIZE; i++) {
	for (nodePtr = nodehashtab[i]; nodePtr; nodePtr = nodePtr->nextPtr) {
	    fprintf(stderr, "%s: no parent %s for node %s\n", 
		    tnm_MibFileName, nodePtr->parentName, nodePtr->label);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * HashNodeList --
 *
 *	This procedure moves the nodes from the nodeList parameter
 *	into the node hash table. The hash function is computed
 *	upon the parent names so that all nodes with the same parent 
 *	are in one collision list.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The nodes are moved from the nodeList into lists sorted by
 *	the name of the parent node.
 *
 *----------------------------------------------------------------------
 */

static void
HashNodeList(nodeList)
    Tnm_MibNode *nodeList;
{
    int	hash;
    Tnm_MibNode *nodePtr, *nextp;

    memset((char *) nodehashtab, 0, sizeof(nodehashtab));

    for (nodePtr = nodeList; nodePtr != NULL; ) {
	if (! nodePtr->parentName) {
	    fprintf(stderr, "%s: %s has no parent in the MIB tree!\n",
		    tnm_MibFileName, nodePtr->label);
	    return;
	}

	hash = HashNodeLabel(nodePtr->parentName);

	nextp = nodePtr->nextPtr;
	nodePtr->nextPtr = nodehashtab[hash];
	nodehashtab[hash] = nodePtr;
	nodePtr = nextp;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * HashNodeLabel --
 *
 *	This procedure computes the hash function used to hash
 *	MIB nodes by parent name.
 *
 * Results:
 *	The hash value of the given label.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
HashNodeLabel(label)
    char *label;
{
    int hash = 0;
    char *cp;
    
    for (cp = label; *cp; cp++) {
	hash += *cp;
    }

    return (hash % NODEHASHSIZE);
}

