/*
 * tnmSnmpReq.c --
 *
 *	This file contains all functions that take a request to send a
 *	SNMP packet over the network.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/* 
 * Flag that controls hexdump. See the watch command for its use.
 */

extern int	hexdump;

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
EncodeMessage		_ANSI_ARGS_((Tcl_Interp *interp,
				     SNMP_Session *sess, SNMP_PDU *pdu,
				     u_char *packet, int *packetlen));
#ifdef TNM_SNMPv2U
static int
EncodeUsecParameter	_ANSI_ARGS_((SNMP_Session *session, SNMP_PDU *pdu, 
				     u_char *parameter));
#endif

static u_char*
EncodePDU		_ANSI_ARGS_((Tcl_Interp *interp, 
				     SNMP_Session *sess, SNMP_PDU *pdu,
				     u_char *packet, int *packetlen));


/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpEncode --
 *
 *	This procedure converts the pdu into BER transfer syntax and 
 *	sends it from this entity (either a manager or an agent) on
 *	this system to a remote entity.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpEncode(interp, session, pdu, proc, clientData)
    Tcl_Interp *interp;
    SNMP_Session *session;
    SNMP_PDU *pdu;
    Tnm_SnmpRequestProc *proc;
    ClientData clientData;
{
    int	retry = 0, packetlen = 0, rc = 0;
    u_char packet[TNM_SNMP_MAXSIZE];

    memset((char *) packet, 0, sizeof(packet));
    packetlen = 0;

    /*
     * Some special care must be taken to conform to SNMPv1 sessions:
     * SNMPv2 getbulk requests must be turned into getnext request 
     * and SNMPv1 error codes must be mapped on SNMPv1 error codes
     * (e.g. genErr as nothing more appropriate is available).
     *
     * This is based on the mapping presented in Marhall Rose and
     * Keith McCloghrie: "How to Manage your Network using SNMP"
     * page 95.
     */

    if (session->version == TNM_SNMPv1) {
        if (pdu->type == TNM_SNMP_GETBULK) {
	    pdu->type = TNM_SNMP_GETNEXT;
	    pdu->error_status = TNM_SNMP_NOERROR;
	    pdu->error_index  = 0;
	}
	if (pdu->error_status > TNM_SNMP_GENERR) {
	    switch (pdu->error_status) {
	      case TNM_SNMP_NOACCESS:
	      case TNM_SNMP_NOCREATION:
	      case TNM_SNMP_AUTHORIZATIONERROR:
	      case TNM_SNMP_NOTWRITABLE:
	      case TNM_SNMP_INCONSISTENTNAME:
		pdu->error_status = TNM_SNMP_NOSUCHNAME; break;
	      case TNM_SNMP_WRONGTYPE:
	      case TNM_SNMP_WRONGLENGTH:
	      case TNM_SNMP_WRONGENCODING:
	      case TNM_SNMP_WRONGVALUE:
	      case TNM_SNMP_INCONSISTENTVALUE:
		pdu->error_status = TNM_SNMP_BADVALUE; break;	
	      case TNM_SNMP_RESOURCEUNAVAILABLE:
	      case TNM_SNMP_COMMITFAILED:
	      case TNM_SNMP_UNDOFAILED:
		pdu->error_status = TNM_SNMP_GENERR; break;
	      default:
		pdu->error_status = TNM_SNMP_GENERR; break;
	    }
	}
    }

    /*
     * Encode message into ASN1 BER transfer syntax. Authentication or
     * encryption is done within the following procedures if it is an
     * authentic or private message.
     */

    rc = EncodeMessage(interp, session, pdu, packet, &packetlen);
    if (rc != TCL_OK) {
	return TCL_ERROR;
    }

    switch (pdu->type) {
      case TNM_SNMP_GET:	snmpStats.snmpOutGetRequests++; break;
      case TNM_SNMP_GETNEXT:	snmpStats.snmpOutGetNexts++; break;
      case TNM_SNMP_SET:	snmpStats.snmpOutSetRequests++; break;
      case TNM_SNMP_RESPONSE:	snmpStats.snmpOutGetResponses++; break;
      case TNM_SNMPv1_TRAP:	snmpStats.snmpOutTraps++; break;
    }
    
    /*
     * Show the contents of the PDU - mostly for debugging.
     */
    
    Tnm_SnmpEvalBinding(interp, session, pdu, TNM_SNMP_SEND_EVENT);

    Tnm_SnmpDumpPDU(interp, pdu);

    /*
     * A trap message or a response? - send it and we are done!
     */
    
    if (pdu->type == TNM_SNMPv1_TRAP || pdu->type == TNM_SNMPv2_TRAP 
	|| pdu->type == TNM_SNMP_RESPONSE || pdu->type == TNM_SNMP_REPORT) {
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U) {
	    Tnm_SnmpUsecAuth(session, packet, packetlen);
	}
#endif
	if (Tnm_SnmpSend(interp, packet, packetlen, &pdu->addr) != TCL_OK) {
	    return TCL_ERROR;
	}
	Tcl_ResetResult(interp);
	return TCL_OK;
    }
  
    /*
     * Asychronous request: queue request and we are done.
     */

    if (proc) {
	Tnm_SnmpRequest *rPtr;
	rPtr = Tnm_SnmpCreateRequest(pdu->request_id, packet, packetlen,
				     proc, clientData, interp);
	Tnm_SnmpQueueRequest(session, rPtr);
	sprintf(interp->result, "%d", (int) pdu->request_id);
	return TCL_OK;
    }
    
    /*
     * Synchronous request: send packet and wait for response.
     */
    
    for (retry = 0; retry <= session->retries; retry++) {
	int id;
#ifdef SNMP_BENCH
	Tnm_SnmpMark stats;

	memset((char *) &stats, 0, sizeof(stats));
#endif

      repeat:
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U) {
	    Tnm_SnmpUsecAuth(session, packet, packetlen);
	}
#endif
	Tnm_SnmpDelay(session);
	if (Tnm_SnmpSend(interp, packet, packetlen, &pdu->addr) == TCL_ERROR) {
	    return TCL_ERROR;
	}

#ifdef SNMP_BENCH
	if (stats.sendSize == 0) {
	    stats.sendSize = tnmSnmpBenchMark.sendSize;
	    stats.sendTime = tnmSnmpBenchMark.sendTime;
	}
#endif

	while (Tnm_SnmpWait(session->timeout*1000/(session->retries+1)) > 0) {
	    u_char packet[TNM_SNMP_MAXSIZE];
	    int rc, packetlen = TNM_SNMP_MAXSIZE;
	    struct sockaddr_in from;
	    
	    if (Tnm_SnmpRecv(interp, packet, &packetlen, &from) != TCL_OK) {
		return TCL_ERROR;
	    }
	    
	    rc = Tnm_SnmpDecode(interp, packet, packetlen, &from,
				session, &id);
	    if (rc == TCL_BREAK) {
		if (retry++ <= session->retries + 1) {
		    goto repeat;
		}
	    }
	    if (rc == TCL_OK) {
		if (id == pdu->request_id) {
#ifdef SNMP_BENCH
		    stats.recvSize = tnmSnmpBenchMark.recvSize;
		    stats.recvTime = tnmSnmpBenchMark.recvTime;
		    session->stats = stats;
#endif
		    return TCL_OK;
		}
		rc = TCL_CONTINUE;
	    }
	    
	    if (rc == TCL_CONTINUE) {
		if (hexdump) {
		    fprintf(stderr, "%s\n", interp->result);
		}
		continue;
	    }
	    if (rc == TCL_ERROR) {
		return TCL_ERROR;
	    }
	}
    }
    
    Tcl_SetResult(interp, "noResponse", TCL_STATIC);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * EncodeMessage --
 *
 *	This procedure takes a session and a PDU and serializes the
 *	ASN1 pdu as an octet string into the buffer pointed to by 
 *	packet using the "Basic Encoding Rules". See RFC 1157 and 
 *	RFC 1910 for the message header description. The main parts
 *	are the version number, the community string and the SNMP PDU.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
EncodeMessage(interp, session, pdu, packet, packetlen)
    Tcl_Interp *interp;
    SNMP_Session *session;
    SNMP_PDU *pdu;
    u_char *packet;
    int *packetlen;
{
    u_char *messageLen;
    u_char *p = packet;
#ifdef TNM_SNMPv2U
#define PARAM_MAX_LENGTH 340
    u_char buffer[PARAM_MAX_LENGTH], *parameter = NULL;
    int version = 0, parameterLen = 0;

    if (session->qos & USEC_QOS_PRIV) {
	Tcl_SetResult(interp, "encryption not supported", TCL_STATIC);
	return TCL_ERROR;
    }
#endif

    *p++ = (ASN1_UNIVERSAL | ASN1_CONSTRUCTED | ASN1_SEQUENCE);
    messageLen = p++;
    *packetlen += 2;

    switch (session->version) {
      case TNM_SNMPv1:
	version = 0;
	parameter = (pdu->type == TNM_SNMP_SET && session->writeCommunity)
	            ? (u_char *) session->writeCommunity
		    : (u_char *) session->readCommunity;
	parameterLen = strlen((char *) parameter);
	break;
#ifdef TNM_SNMPv2C
      case TNM_SNMPv2C:
	version = 1;
	parameter = (pdu->type == TNM_SNMP_SET && session->writeCommunity)
	            ? (u_char *) session->writeCommunity
		    : (u_char *) session->readCommunity;
	parameterLen = strlen((char *) parameter);
        break;
#endif
#ifdef TNM_SNMPv2U
      case TNM_SNMPv2U:
	version = 2;
	parameter = buffer;
	parameterLen = EncodeUsecParameter(session, pdu, parameter);
	break;
    }
#endif
    p = Tnm_BerEncInt(p, packetlen, ASN1_INTEGER, version);
    p = Tnm_BerEncOctetString(p, packetlen, ASN1_OCTET_STRING,
			      (char *) parameter, parameterLen);

    p = EncodePDU(interp, session, pdu, p, packetlen);
    if (p == NULL) {
	if (*interp->result == '\0') {
	    Tcl_SetResult(interp, Tnm_BerError(), TCL_STATIC);
	}
	return TCL_ERROR;
    }

    p = Tnm_BerEncLength(p, packetlen, messageLen, p - (messageLen + 1));
    return TCL_OK;
}

#ifdef TNM_SNMPv2U
/*
 *----------------------------------------------------------------------
 *
 * EncodeUsecParameter --
 *
 *	This procedure builds the parameters string. Note, some fields
 *	are left blank as they are filled in later in Tnm_SnmpUsecAuth().
 *	This way we can patch in new agentTime or agentBoot values in 
 *	case our clock drifts away.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
EncodeUsecParameter(session, pdu, parameter)
    SNMP_Session *session;
    SNMP_PDU *pdu;
    u_char *parameter;
{
    u_char *p = parameter;
    u_int boots = 0, clock = 0;

    /* 
     * The first byte is the model indicator, which is 0 for the USEC 
     * model. The second byte is the quality of service (qos) field.
     */

    *p++ = USEC_MODEL;

    if ((pdu->type == TNM_SNMP_GET) 
	|| (pdu->type == TNM_SNMP_GETNEXT)
	|| (pdu->type == TNM_SNMP_GETBULK) 
	|| (pdu->type == TNM_SNMP_SET)) {
	*p++ = session->qos | USEC_QOS_REPORT;
    } else {
	*p++ = session->qos;
    }

    /*
     * The following bytes contain the agent identifier. This value
     * is only filled if we authenticate the message or if we send
     * a notification.
     */

    if ((session->qos & USEC_QOS_REPORT)
	|| (session->qos & USEC_QOS_PRIV)
	|| (pdu->type == TNM_SNMPv2_TRAP)
	|| (pdu->type == TNM_SNMP_INFORM)) {
	memcpy(p, session->agentID, USEC_MAX_AGENTID);
    } else {
	memset(p, 0, USEC_MAX_AGENTID);
    }
    p += USEC_MAX_AGENTID;
    
    /* 
     * The next 4 bytes contain the number of agent boots followed by
     * the current agent time which is calculated by using the time
     * offset saved in the session structure. Note, these fields are
     * patched later and set to zero if the message is not authenticated.
     */

    if (session->qos & USEC_QOS_AUTH || session->qos & USEC_QOS_PRIV) {
	boots = session->agentBoots;
	clock = time((time_t *) NULL) - session->agentTime;
    }

    *p++ = (boots >> 24) & 0xff;
    *p++ = (boots >> 16) & 0xff;
    *p++ = (boots >> 8) & 0xff;
    *p++ = boots & 0xff;
    *p++ = (clock >> 24) & 0xff;
    *p++ = (clock >> 16) & 0xff;
    *p++ = (clock >> 8) & 0xff;
    *p++ = clock & 0xff;

#ifdef DEBUG_USEC
    fprintf(stderr, "encode: agentBoots = %u agentTime = %u\n",
	    session->agentBoots, session->agentTime);
#endif

    /*
     * The following two bytes contain the max message size we accept.
     */
    
    *p++ = (session->maxSize >> 8) & 0xff;
    *p++ = session->maxSize & 0xff;
    
    /*
     * The next variable length field contains the user name. The first
     * byte is the length of the user name.
     */

    *p++ = session->userNameLen;
    memcpy(p, session->userName, session->userNameLen);
    p += session->userNameLen;

    /*
     * The next variable length field is the authentication digest. Its
     * length is 0 for unauthenticated messages or 16 for the MD5 digest
     * algorithm. Note, this field is patched later.
     */
    
    if (session->qos & USEC_QOS_AUTH || session->qos & USEC_QOS_PRIV) {
	*p++ = TNM_MD5_SIZE;
	p += TNM_MD5_SIZE;
    } else {
	*p++ = 0;
    }

    /* 
     * Note, the context identifier is variable length but the length
     * is not given as it is implicit contained in the length of the
     * octet string.  
     */
        
    memcpy(p, session->cntxt, session->cntxtLen);
    p += session->cntxtLen;
    return (p - parameter);
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpUsecAuth --
 *
 *	This procedure patches the USEC authentication information
 *	into a BER encoded SNMP USEC packet. We therefore decode the 
 *	packet until we have found the parameter string. This is still 
 *	a lot faster than doing BER encodings every time the packet 
 *	is sent. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tnm_SnmpUsecAuth(session, packet, packetlen)
    SNMP_Session *session;
    u_char *packet;
    int packetlen;
{
    u_char *parm, *p = packet;
    int dummy = packetlen;
    u_int boots = 0, clock = 0;
    u_char digest[TNM_MD5_SIZE];

    /*
     * Get the parameter section out of the message header.
     */
    
    if (*p ++ != (ASN1_UNIVERSAL | ASN1_CONSTRUCTED | ASN1_SEQUENCE)) {
	return;
    }
    p = Tnm_BerDecLength(p, &dummy, (u_int *) &dummy);
    p = Tnm_BerDecInt(p, &dummy, ASN1_INTEGER, &dummy);
    p = Tnm_BerDecOctetString(p, &dummy, ASN1_OCTET_STRING, 
			       (char **) &parm, &dummy);
    if (! p) return;

    /*
     * Set the agentID, the agentBoots and the agentTime values.
     */

    p = parm + 2;
    memcpy(p, session->agentID, USEC_MAX_AGENTID);
    p += USEC_MAX_AGENTID;

    if (session->qos & USEC_QOS_AUTH || session->qos & USEC_QOS_PRIV) {
	boots = session->agentBoots;
	clock = time((time_t *) NULL) - session->agentTime;
	*p++ = (boots >> 24) & 0xff;
	*p++ = (boots >> 16) & 0xff;
	*p++ = (boots >> 8) & 0xff;
	*p++ = boots & 0xff;
	*p++ = (clock >> 24) & 0xff;
	*p++ = (clock >> 16) & 0xff;
	*p++ = (clock >> 8) & 0xff;
	*p++ = clock & 0xff;
	
#ifdef DEBUG_USEC
	fprintf(stderr, "auth: agentBoots = %u agentTime = %u\n",
		session->agentBoots, session->agentTime);
#endif

	/*
	 * Skip the max message size field.
	 */
	
	p += 2;
	
	/*
	 * Skip the user name field, check the digest len (should be 16),
	 * copy the key into the packet, compute the digest and copy the
	 * result back into the packet.
	 */
	
	p += *p;
	p++;
	if (*p++ != TNM_MD5_SIZE) return;
	memcpy(p, session->authKey, TNM_MD5_SIZE);
	Tnm_SnmpMD5Digest(packet, packetlen, session->authKey, digest);
	memcpy(p, digest, TNM_MD5_SIZE);
    }
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * DecodePDU --
 *
 *	This procedure takes a session and a PDU and serializes it 
 *	into an ASN.1 PDU hold in the buffer pointed to by packet
 *	using the "Basic Encoding Rules".
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static u_char*
EncodePDU(interp, session, pdu, packet, packetlen)
    Tcl_Interp *interp;
    SNMP_Session *session;
    SNMP_PDU *pdu;
    u_char *packet;
    int	*packetlen;
{    
    u_char *PDU_len = NULL, *VarBind_len = NULL, *VarBindList_len = NULL;
    
    int i, vblc, vbc;
    char **vblv, **vbv;

    Tnm_Oid *oid;
    int oidlen;

    /*
     * encode pdu type ( tag: [pdu_type] IMPLICIT PDU )
     */
    
    *packet++  = (ASN1_CONTEXT | ASN1_CONSTRUCTED | pdu->type);
    PDU_len    = packet++;
    *packetlen += 2;

    if (pdu->type == TNM_SNMPv1_TRAP) {

	int generic = 0, specific = 0, len;
	struct sockaddr_in addr;

	/*
	 * RFC 1907 defines the standard trap using snmpTraps which
	 * is registered at 1.3.6.1.6.3.1.1.5. A TRAP-TYPE macro (RFC 
	 * 1215) is converted into a SNMPv2 oid using the rules in
	 * RFC 1908 and these rules use the snmp node registered
	 * at 1.3.6.1.2.1.11. We have to accept both definitions
	 * for the standard traps to be backward compatible.
	 */

	oid = Tnm_StrToOid(pdu->trapOID, &oidlen);
	if (strncmp(pdu->trapOID, "1.3.6.1.6.3.1.1.5", 17) == 0) {
	    Tnm_Oid *tmp;
	    generic = oid[oidlen-1] - 1;
	    specific = 0;
	    tmp = Tnm_StrToOid("1.3.6.1.4.1.1575", &len); /* tubs */
	    packet = Tnm_BerEncOID(packet, packetlen, tmp, len);
	} else if (strncmp(pdu->trapOID, "1.3.6.1.2.1.11.0", 16) == 0) {
	    Tnm_Oid *tmp;
            generic = oid[oidlen-1];
            specific = 0;
            tmp = Tnm_StrToOid("1.3.6.1.4.1.1575", &len); /* tubs */
            packet = Tnm_BerEncOID(packet, packetlen, tmp, len);
	} else {
	    generic = 6;
	    specific = oid[oidlen-1];
	    packet = Tnm_BerEncOID(packet, packetlen, oid, oidlen-2);
	}
	if (packet == NULL) {
	    Tcl_SetResult(interp, 
			  "failed to encode enterprise object identifier", 
			  TCL_STATIC);
	    return NULL;
	}
 
	if (TnmSetIPAddress(interp, Tcl_GetHostName(), &addr) != TCL_OK) {
	    return NULL;
	}
	packet = Tnm_BerEncOctetString(packet, packetlen, ASN1_IPADDRESS,
				       (char *) &addr.sin_addr, 4);

	packet = Tnm_BerEncInt(packet, packetlen, ASN1_INTEGER, generic);
	packet = Tnm_BerEncInt(packet, packetlen, ASN1_INTEGER, specific);
	packet = Tnm_BerEncInt(packet, packetlen, ASN1_TIMETICKS, 
			       Tnm_SnmpSysUpTime());

    } else {
    
	packet = Tnm_BerEncInt(packet, packetlen,
				 ASN1_INTEGER, pdu->request_id);
	packet = Tnm_BerEncInt(packet, packetlen,
				 ASN1_INTEGER, pdu->error_status);
	switch (pdu->error_status) {
	  case TNM_SNMP_TOOBIG:		snmpStats.snmpOutTooBigs++; break;
	  case TNM_SNMP_NOSUCHNAME:	snmpStats.snmpOutNoSuchNames++; break;
	  case TNM_SNMP_BADVALUE:	snmpStats.snmpOutBadValues++; break;
	  case TNM_SNMP_READONLY:	break; /* not used */
	  case TNM_SNMP_GENERR:		snmpStats.snmpOutGenErrs++; break;
	}
	packet = Tnm_BerEncInt(packet, packetlen,
			       ASN1_INTEGER, pdu->error_index);
    }

    /*
     * encode VarBindList ( SEQUENCE of VarBind )
     */
    
    *packet++       = (ASN1_UNIVERSAL | ASN1_CONSTRUCTED | ASN1_SEQUENCE);
    VarBindList_len = packet++;
    *packetlen      += 2;
    
    /*
     * split the varbind list and loop over all elements
     */
    
    if (Tcl_SplitList(interp, Tcl_DStringValue(&pdu->varbind), &vblc, &vblv)
	!= TCL_OK) {
	return NULL;
    }

    if (pdu->type == TNM_SNMPv2_TRAP || pdu->type == TNM_SNMP_INFORM) {

	/* 
	 * Encode two default varbinds: sysUpTime.0 and snmpTrapOID.0
	 * as defined in section 4.2.7 of RFC 1905.
	 */
	   
	*packet++   = (ASN1_UNIVERSAL | ASN1_CONSTRUCTED | ASN1_SEQUENCE);
	VarBind_len = packet++;
	*packetlen  += 2;

	oid = Tnm_StrToOid("1.3.6.1.2.1.1.3.0", &oidlen);
	packet = Tnm_BerEncOID(packet, packetlen, oid, oidlen);

	packet = Tnm_BerEncInt(packet, packetlen, ASN1_TIMETICKS, 
				Tnm_SnmpSysUpTime());
	packet = Tnm_BerEncLength(packet, packetlen, VarBind_len,
				   packet - (VarBind_len + 1));

	*packet++   = (ASN1_UNIVERSAL | ASN1_CONSTRUCTED | ASN1_SEQUENCE);
	VarBind_len = packet++;
	*packetlen  += 2;

	oid = Tnm_StrToOid("1.3.6.1.6.3.1.1.4.1.0", &oidlen);
	packet = Tnm_BerEncOID(packet, packetlen, oid, oidlen);

	oid = Tnm_StrToOid(pdu->trapOID, &oidlen);
	packet = Tnm_BerEncOID(packet, packetlen, oid, oidlen);
	
	packet = Tnm_BerEncLength(packet, packetlen, VarBind_len,
				   packet - (VarBind_len + 1));
    }
    
    for (i = 0; i < vblc; i++) {
	
	char *value;
	int asn1_type = ASN1_OTHER;
	
	/*
	 * split a single varbind into its components
	 */
	
	if (Tcl_SplitList(interp, vblv[i], &vbc, &vbv) != TCL_OK) {
	    ckfree((char *) vblv);
	    return NULL;
	}

	if (vbc == 0) {
	    Tcl_SetResult(interp, "missing OBJECT IDENTIFIER", TCL_STATIC);
	    ckfree((char *) vblv);
            return NULL;
	}
	
	/*
	 * encode each VarBind ( SEQUENCE name, value )
	 */
	
	*packet++   = (ASN1_UNIVERSAL | ASN1_CONSTRUCTED | ASN1_SEQUENCE);
	VarBind_len = packet++;
	*packetlen  += 2;

	/*
	 * encode the object identifier, perhaps consulting the MIB
	 */
	
	if (Tnm_IsOid(vbv[0])) {
	    oid = Tnm_StrToOid(vbv[0], &oidlen);
	} else {
	    char *tmp = Tnm_MibGetOid(vbv[0], 0);
	    if (! tmp) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "invalid object identifier \"",
				 vbv[0], "\"", (char *) NULL);
		ckfree((char *) vbv);
		ckfree((char *) vblv);
		return NULL;
	    }
	    oid = Tnm_StrToOid(tmp, &oidlen);
	}

	packet = Tnm_BerEncOID(packet, packetlen, oid, oidlen);
	if (packet == NULL) {
	    break;
	}

	/*
	 * guess the asn1 type field and the value
	 */

	switch (vbc) {
	  case 1:
	    value = "";
	    asn1_type = ASN1_NULL;
	    break;
	  case 2:
	    value = vbv[1];
	    asn1_type = Tnm_MibGetBaseSyntax(vbv[0], 0);
	    break;
	  default:
	    value = vbv[2];

	    /*
	     * Check if there is an exception in the asn1 type field.
	     * Convert this into an appropriate NULL type if we create
	     * a response PDU. Otherwise, ignore this stuff and use
	     * the type found in the MIB.
	     */

	    if (pdu->type == TNM_SNMP_RESPONSE) {
		if (strcmp(vbv[1], "noSuchObject") == 0) {
		    asn1_type = ASN1_NO_SUCH_OBJECT;
		} else if (strcmp(vbv[1], "noSuchInstance") == 0) {
		    asn1_type = ASN1_NO_SUCH_INSTANCE;
		} else if (strcmp(vbv[1], "endOfMibView") == 0) {
		    asn1_type = ASN1_END_OF_MIB_VIEW;
		} else {
		    asn1_type = TnmGetTableKey(tnmSnmpTypeTable, vbv[1]);
		    if (asn1_type < 0) {
			asn1_type = ASN1_OTHER;
		    }
		}
	    } else {
		asn1_type = TnmGetTableKey(tnmSnmpTypeTable, vbv[1]);
		if (asn1_type < 0) {
		    asn1_type = ASN1_OTHER;
		}
	    }

	    if (asn1_type == ASN1_OTHER) {
		Tnm_MibTC *tcPtr;
		tcPtr = Tnm_MibFindTC(vbv[1]);
		if (tcPtr) {
		    asn1_type = tcPtr->syntax;
		}
	    }
	    break;
	}

	if (asn1_type == ASN1_OTHER) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "unknown type \"", vbv[1], "\"",
			     (char *) NULL);
	    return NULL;
	}
	
	/*
         * If it's a SET request (or something equivalent), we'll 
	 * have to encode the VALUE.
	 */

	if (pdu->type == TNM_SNMP_SET || pdu->type == TNM_SNMP_RESPONSE
	    || pdu->type == TNM_SNMPv1_TRAP || pdu->type == TNM_SNMPv2_TRAP
	    || pdu->type == TNM_SNMP_INFORM || pdu->type == TNM_SNMP_REPORT) {

	    switch (asn1_type) {
	      case ASN1_INTEGER:
	      case ASN1_COUNTER32:
	      case ASN1_GAUGE32:
		{   int int_val, rc;
		    rc = Tcl_GetInt(interp, value, &int_val);
		    if (rc != TCL_OK) {
			char *tmp = Tnm_MibScan(vbv[0], 0, value);
			if (tmp && *tmp) {
			    Tcl_ResetResult(interp);
			    rc = Tcl_GetInt(interp, tmp, &int_val);
			}
			if (rc != TCL_OK) return NULL;
		    }
		    packet = Tnm_BerEncInt(packet, packetlen,
					    asn1_type, int_val);
		}
		break;
	      case ASN1_COUNTER64:
		{   int int_val, rc;
		    if (sizeof(int) >= 8) {
			rc = Tcl_GetInt(interp, value, &int_val);
			if (rc != TCL_OK) {
			    char *tmp = Tnm_MibScan(vbv[0], 0, value);
			    if (tmp && *tmp) {
				Tcl_ResetResult(interp);
				rc = Tcl_GetInt(interp, tmp, &int_val);
			    }
			    if (rc != TCL_OK) return NULL;
			}
			packet = Tnm_BerEncInt(packet, packetlen,
						ASN1_COUNTER64, int_val);
		    } else {
			double d;
			rc = Tcl_GetDouble(interp, value, &d);
			if (rc != TCL_OK) {
			    return NULL;
			}
			if (d < 0) {
			    Tcl_SetResult(interp, "negativ counter value",
					  TCL_STATIC);
			    return NULL;
			}
			packet = Tnm_BerEncCounter64(packet, packetlen, d);
		    }
		}
		break;
	      case ASN1_BIT_STRING:
		packet = Tnm_BerEncNull(packet, packetlen, ASN1_NULL);
		break;
	      case ASN1_TIMETICKS:
		{   u_int d, h, m, s, f, n, val;
		    n = sscanf(value, "%dd %d:%d:%d.%d", 
			       &d, &h, &m, &s, &f);
		    if (n == 5) {
			val = d * 8640000 + h * 360000 
					+ m * 6000 + s * 100 + f;
		    } else {
			val = 0;
			while (isdigit(*value)) {
			    val = 10 * val + *value - '0';
			    value++;
			}
		    }
		    packet = Tnm_BerEncInt(packet, packetlen,
					    ASN1_TIMETICKS, val);
		}
                break;
	      case ASN1_IPADDRESS:
		{   int a, b, c, d, addr = inet_addr(value);
		    int cnt = sscanf(value, "%d.%d.%d.%d", &a, &b, &c, &d);
		    if ((addr == -1 && strcmp(value, "255.255.255.255") != 0)
			|| (cnt != 4)) {
			Tcl_SetResult(interp, "invalid IP address", 
				      TCL_STATIC);
			return NULL;
		    }
		    packet = Tnm_BerEncOctetString(packet, packetlen, 
				     ASN1_IPADDRESS, (char *) &addr, 4);
		}
		break;
	      case ASN1_OCTET_STRING:
		{   char *hex = value;
		    int len;
		    static char *bin = NULL;
		    static int binLen = 0;
		    char *scan = Tnm_MibScan(vbv[0], 0, value);
		    if (scan) hex = scan;
		    if (*hex) {
		        len = strlen(hex);
		        if (binLen < len + 1) {
			    if (bin) ckfree(bin);
			    binLen = len + 1;
			    bin = ckalloc(binLen);
			}
			if (Tnm_SnmpHexToBin(hex, bin, &len) < 0) {
			    Tcl_SetResult(interp, 
					  "illegal octet string value",
					  TCL_STATIC);
			    return NULL;
			}
		    } else {
			len = 0;
		    }
		    packet = Tnm_BerEncOctetString(packet, packetlen,
				     ASN1_OCTET_STRING, bin, len);
		}
		break;
	      case ASN1_OPAQUE:
	        {   char *hex = value;
		    int len;
		    static char *bin = NULL;
		    static int binLen = 0;
		    if (*hex) {
			len = strlen(hex);
			if (binLen < len + 1) {
			    if (bin) ckfree(bin);
			    binLen = len + 1;
			    bin = ckalloc(binLen);
			}
			if (Tnm_SnmpHexToBin(hex, bin, &len) < 0) {
			    Tcl_SetResult(interp, "illegal Opaque value",
					  TCL_STATIC);
			    return NULL;
			}
		    } else {
			len = 0;
		    }
		    packet = Tnm_BerEncOctetString(packet, packetlen,
						   ASN1_OPAQUE, bin, len);    
		}
		break;
	      case ASN1_OBJECT_IDENTIFIER:
		if (! Tnm_IsOid(value)) {
		    char *tmp = Tnm_MibGetOid(value, 0);
		    if (!tmp) {
			Tcl_AppendResult(interp, 
					 "illegal object identifier \"",
					 value, "\"", (char *) NULL);
			return NULL;
		    }
		    oid = Tnm_StrToOid(tmp, &oidlen);
		} else {
		    oid = Tnm_StrToOid(value, &oidlen);
		}
		packet = Tnm_BerEncOID(packet, packetlen, oid, oidlen);
		break;
	      case ASN1_NO_SUCH_OBJECT:
	      case ASN1_NO_SUCH_INSTANCE:
	      case ASN1_END_OF_MIB_VIEW:
	      case ASN1_NULL:
		packet = Tnm_BerEncNull(packet, packetlen, asn1_type);
		break;
	      default:
		sprintf(interp->result, "unknown asn1 type 0x%.2x",
			asn1_type);
		return NULL;
	    }
	} else {
	    packet = Tnm_BerEncNull(packet, packetlen, ASN1_NULL);
	}

	packet = Tnm_BerEncLength(packet, packetlen, VarBind_len,
				   packet - (VarBind_len + 1));

	ckfree((char *) vbv);
    }

    ckfree((char *) vblv);

    packet = Tnm_BerEncLength(packet, packetlen, VarBindList_len,
			       packet - (VarBindList_len + 1));
    packet = Tnm_BerEncLength(packet, packetlen, PDU_len,
			       packet - (PDU_len + 1));
    return packet;
}
