%module skf

%{
/* -------------------------------------------- */
/* SWIG interface C definitions			*/
/* -------------------------------------------- */
/* ********************************************************************
**
** skf_convert.i: skf extension interfaces
**
** Copyright (c) 2005-2012 Seiji Kaneko. All rights reserved.
** Everyone is permitted to use this program in source and binary
** form, with or without modification if and only if the following
** conditions are met:
** 1. Redistributions of source code must retain the above copyright
**   notice, copyright notice written in source code, additional list
**   of conditions and the following disclaimer.
** 2. Redistributions in machine readable form must reproduce the 
**   above copyright notice and the following disclaimer in the
**   documentation or other material provided with the distribution.
** 3. Neither the name of the copyright holders nor the names of its 
**   contributors may be used to endorse or promote products derived
**   from this software without specific prior written permission.
**********************************************************************
** Disclaimer: This software is provided and distributed AS IS, 
**	without any implicit or explicit warranties, and not
**	guaranteed to be error-free. In no event shall the author be
**	liable for any direct, indirect or incidental damages,
**	including, but not limited to, loss of data, use or profits
**	responsibility for any direct or indirect damages or results
**	arising by using whole or a part of this software.
**********************************************************************
** RUBY EXTENSION
** Usage: Skf.guess(option_string, string_to_convert) -- returns states
**	  Skf.convert(option_string, string_to_convert) -- conversion
**	  Skf.init(option_string, string_to_convert) -- initialize
**	  Skf.destruct(option_string, string_to_convert) -- free buffers
**
** skf options except -b,-u works as shown in documents. Environment
** variables skfenv and SKFENV is ignored regardless uid.
** conversion refers previous state hold inside skf. To suppress this
** behavior, use Skf.init explicitly.
******************************************************************* */
/* $Id: skf_convert.i,v 1.4 2013/04/07 03:34:21 seiji Exp seiji $ */

/* invocation for some ghosts */
#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <unistd.h>
#include <errno.h>

#ifdef HAVE_STRING_H
#include <string.h>
#else
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#endif

/* --- perl swig --- */
#ifdef SWIGPERL
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
#endif

/* --- ruby swig --- */
#ifdef SWIGRUBY

#if defined(SKF_RUBY20)
#define SKF_RUBY2
#endif

/* 1.9-handling: HAS_RUBY_RUBY_H may be broken on Mac OS X */
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
#include "ruby/ruby.h"
#include "ruby/encoding.h"
#else
#include "ruby.h"
#endif
#endif

/* --- php swig --- */
#ifdef SWIGPHP
#include "php.h"
#endif

/* --- python swig --- */
#ifdef SWIGPYTHON
#include "Python.h"
/* ifdef SKF_PYTHON3 */
#define LWLINTSTRING 
#endif

/* --- others --- */

/* --- skf -------- */
#include "skf.h"
#include "skf_fileio.h"
#include "convert.h"
#include "oconv.h"

#include "skf_convert.h"

/* Note:
   maximum output size is ((input_string_length * 5) + 1) for skf (non-unic*de).
   maximum output size is ((input_string_length * 4) + 6) for skf (unic*de).
   maximum output size is ((input_string_length * 10) + 6) for skf (unic*de w. decomp).
   See skf_fileio.h and .c
*/
#define INCSIZE		256
#define GUESSSIZE	128

#define NO_INIT		0
#define FORCE_INIT	1
#define GUESS_INIT	2

#ifdef __cplusplus
}
#endif

int	swig_state = 0;
int	skf_swig_result = 0;

int	errorcode = 0;		/* return error result condition   */
int	in_saved_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
int	ruby_out_locale_index = 0;
int	ruby_out_ascii_index = 0;
#endif
/* --------------------------------------------------------------- */
/* misc utilities						   */
/* --------------------------------------------------------------- */
void skf_exit(int eval)
{
    errorcode = skf_swig_result;
#if defined(SWIGRUBY)
    rb_raise(rb_eSignal,"skf detected fatal error");
#endif
#if defined(SWIGPERL)
    croak("skf detected fatal error");
#endif
}

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
int get_rstr_enc(cname)
char *cname;
{
    return(skf_search_chname(cname));
}
#endif

/* --------------------------------------------------------------- */
/* we need function to calculate string length			   */
/* --------------------------------------------------------------- */
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
size_t  skf_swig_strlen(str,maxlen)
@SKFSTRINGS@ *str;
int maxlen;
{
    int len;
    
    len = str->length;
    if (len > maxlen) len = maxlen;

    return(len);
}
#else
size_t  skf_swig_strlen(str,maxlen)
@SKFSTRINGS@ *str;
int maxlen;
{
    int i;
    size_t len;

    for (i=0,len=0;((i<maxlen) && (*str!='\0')); i++,str++,len++);

    return(len);
}
#endif

/* --------------------------------------------------------------- */
/* input/output string conversion				   */
/* --------------------------------------------------------------- */
static struct Skf_localestring	*istrdef;
static struct Skf_localestring	*ostrdef;
static int optstr_len = 0;
static int iencode = -1;
/* --------------------------------------------------------------- */
/* output control for LWL interface				   */
/* --------------------------------------------------------------- */
int	skf_olimit = 0;
unsigned char	*skfobuf = NULL;

/* initialize */
void	skf_ioinit(skfoFILE *fout,int mode)
{

    skf_swig_result = 0;
    errorcode = 0;

/* --- output-side buffer prepare -------------------------------- */
    if (ostrdef == NULL) {
    	ostrdef = (struct Skf_localestring *)
			malloc(sizeof(struct Skf_localestring));
	if (ostrdef == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	    	/* should be break at this point */
	};
    } else;

    if (skfobuf == NULL) {
#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"buffer allocation\n");
#endif
	skf_olimit = SKF_STRBUFLEN;
    	skfobuf = (unsigned char *)malloc(skf_olimit * sizeof(unsigned char));
	if (skfobuf == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	};
    };
    ostrdef->sstr = skfobuf;
    ostrdef->length = 0;
    ostrdef->codeset = out_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    if ((mode == GUESS_INIT) || is_o_encode) {
    		/* return ascii for guess and ACE/MIMEs		   */
	ostrdef->lwl_codeset =
		rb_enc_find_index("US_ASCII");
    } else if (mode == FORCE_INIT) {
	ostrdef->lwl_codeset =
		rb_enc_find_index(i_codeset[out_codeset].cname);
    } else;	/* continuing.. do not initialize		   */
#else
    ostrdef->lwl_codeset = -1;
#endif

    return;
}

/* dummy initialize */
void	skf_dmyinit()
{

    skf_swig_result = 0;
    errorcode = 0;
/* --- output-side buffer prepare -------------------------------- */
    if (ostrdef == NULL) {
    	ostrdef = (struct Skf_localestring *)
			malloc(sizeof(struct Skf_localestring));
	if (ostrdef == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	    	/* should be break at this point */
	};
    } else;

    if (skfobuf == NULL) {
#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"buffer allocation\n");
#endif
	skf_olimit = SKF_STRBUFLEN;
    	skfobuf = (unsigned char *)malloc(4 * sizeof(unsigned char));
	if (skfobuf == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	};
    };
    skfobuf[0] = ' ';
    skfobuf[1] = '\0';
    ostrdef->sstr = skfobuf;
    ostrdef->length = 1;
    ostrdef->codeset = out_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    ostrdef->lwl_codeset =
		rb_enc_find_index("US_ASCII");
#else
    ostrdef->lwl_codeset = -1;
#endif

    return;
}

/* --------------------------------------------------------------- */
/* putchar for LWL interface					   */
/* --------------------------------------------------------------- */
int	lwl_putchar(c)
int	c;
 {
    unsigned char	*newbuf;

    if (ostrdef->length >= skf_olimit) {
#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"buffer re-allocation\n");
#endif
	skf_olimit += BUFINCSIZE;
	newbuf = realloc(skfobuf, (skf_olimit * sizeof(unsigned char)));
	if (newbuf == NULL) {
	    skferr(SKF_OBUFREERR,0,skf_olimit);
	    return(-1);
	};
	skfobuf = newbuf;
	ostrdef->sstr = newbuf;
    };
    skfobuf[ostrdef->length] = c;
    ostrdef->length = ostrdef->length + 1;

    return(0);
}

#ifdef LWLINTSTRING
/* --------------------------------------------------------------- */
char	*skfstrstrconv(@SKFSTRINGS@ *sstr,int len)
{
    char	*dstr;
    int		i;

    dstr = calloc(len+1,sizeof(char));
    for (i=0;i<len;i++) dstr[i] = (char) sstr[i];
    sstr[len] = '\0';

    return(dstr);
}
#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
/* --------------------------------------------------------------- */
struct Skf_localestring *skf_rbstring2skfstring(VALUE rstr)
{
    struct Skf_localestring	*sstrdef;


    if ((sstrdef = calloc(1,sizeof(struct Skf_localestring))) == NULL) {
    	skferr(SKF_MALLOCERR,0,0);
    } else {
	sstrdef->sstr = (unsigned char *)RSTRING_PTR(rstr);
	sstrdef->length = RSTRING_LEN(rstr);
	sstrdef->codeset = 
	    skf_search_cname((char *)rb_enc_name(rb_enc_get(rstr)));
	sstrdef->lwl_codeset = -1;	/* do not assume input locale */
    };

    istrdef = sstrdef;
    return sstrdef;
}

#else
struct Skf_localestring *skf_lwlstring2skfstring(char *rstr)
{
    struct Skf_localestring	*sstrdef;
    int i;
    char *pstr = rstr;
    unsigned char	 *istr;
    int	 buflen = LWL_MAXLEN;

    if (istrdef == NULL) {
	if ((sstrdef = calloc(1,sizeof(struct Skf_localestring))) == NULL) {
	    skferr(SKF_MALLOCERR,0,0);
	    skf_exit(EXIT_FAILURE);
	} else;
    } else {
    	sstrdef = istrdef;
    };
    if ((sstrdef->sstr) == NULL) {
	if ((istr = calloc(LWL_MAXLEN,sizeof(unsigned char))) == NULL) {
	/* Note: Maybe I should free istr at this point, but do nothing. */
	    skferr(SKF_MALLOCERR,0,0);
	    skf_exit(EXIT_FAILURE);
	} else;
    } else {
    	istr = sstrdef->sstr;
    };

    for (i=0;((i<LWL_MAXLEN) && (*pstr!='\0')); i++,pstr++) {
	istr[i] = (*pstr) & 0xff;
	if (i >= (buflen - 2)) {
	    istr = realloc(istr,(size_t)(sizeof(int) *(buflen * 2)));
	    if (istr == NULL) {
		skferr(SKF_MALLOCERR,0,0);
		skf_exit(EXIT_FAILURE);
	    } else;
	    break;
	} else;
    };

    istr[i] = sEOF;
    sstrdef->sstr = istr;
    sstrdef->length = i;
    sstrdef->codeset = -1;
    sstrdef->lwl_codeset = -1;	/* do not assume input locale */

    istrdef = sstrdef;
    return sstrdef;
}

#endif
/* ---------------------------------------------------------------
    common output routine.
    output buffer should be prepared before calling.
*/
static void r_skf_convert(struct Skf_localestring *lstr, long ibuflen,
				int mode,int ienc)
{
    int sy;

/* --- codeset specific initialization is done in convert -------- */
/* --- preparing output buffers ---------------------------------- */
    skf_ioinit((skfoFILE *) 0,mode);

/* --- preconversion output prepare ------------------------------ */
    if (o_add_bom) show_endian_out();
    if (add_annon) print_announce(out_codeset);

    reset_kanji_shift;
    init_all_stats();

/* pass parameter to ioinit input side	-------------------------- */
    stdibuf = lstr->sstr;
    buf_p = lstr->length;

#ifdef SKFDEBUG
    if (is_vv_debug) {
	fprintf(stderr,"#buf_p:%ld#",buf_p);
    } else;
#endif

    if (mode == FORCE_INIT) {
	if (o_add_bom) show_endian_out();
	show_lang_tag();

/* --- preconversion output prepare ------------------------------ */
	if (add_annon) print_announce(out_codeset);

/* --- fold value fix -------------------------------------------- */
#ifdef FOLD_SUPPORT
	fold_value_setup();
#endif
    };
/* --- debug analyze ---- */
#ifdef SKFDEBUG
    debug_analyze();
#endif
/* --- open provided strings ------------------------------------- */
/* Note: this point messed up int and long. Be warned.		   */

    sy = ibuflen;
    if (sy > ibuflen) {
	skferr(SKF_DEBUGERR_2,(long)0,(long)0);
	skf_exit(EXIT_SUCCESS);
    };

    skf_fpntr = 0;
    Qflush();

    in_codeset = ienc;	/* this may be blunder. Go anyway	   */

/* --- conversion loop ------------------------------------------- */
    sy = skf_in_converter((FILE *)0);

    in_saved_codeset = in_codeset;

    if (is_jis(conv_cap)) JIS_finish_procedure();
    if (is_euc(conv_cap)) EUC_finish_procedure();
    if (is_msfam(conv_cap)) SJIS_finish_procedure();
    if (is_ucs_utf7(conv_cap)) utf7_finish_procedure();
    if (is_ucs_utf8(conv_cap)) utf8_finish_procedure();
    if (is_ucs_utf16(conv_cap)) ucod_finish_procedure();
    if (out_bg(conv_cap)) BG_finish_procedure();
    if (is_ucs_brgt(conv_cap)) BRGT_finish_procedure();
#ifdef SKFDEBUG
    if (is_v_debug) fprintf(stderr,"\n[EOF]\n");
#endif
    errorcode = 0;
    return;
}

/*
    routines visible from (i.e. provided to) scripting languages
    convert, quickconvert, guess
*/
@SKFSTRINGS@ *convert(char *optstr, @SKFSTRINGS@ *cstr)
{
    long	ibuflen;
    int		result;
    struct	Skf_localestring *lwlstr;

    in_saved_codeset = -1;

    if (swig_state == 0) {
#ifdef SKFDEBUG
	if (is_vv_debug) fprintf(stderr,"\nextension initialize\n");
#endif
	skf_script_init(); swig_state = 1;
    };

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    lwlstr = cstr;
    ibuflen = get_rstr_len(cstr);
#else
    lwlstr = skf_lwlstring2skfstring(cstr);
    ibuflen = lwlstr->length;
#endif

#ifdef DEBUG
    if (is_vv_debug) {
    	fprintf(stderr,"inputside-len%d,ptr:%lx\n",cstr->length,cstr->sstr)
    } else;
#endif

    result = skf_script_param_parse(optstr,optstr_len);

    if (result < 0) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
	skf_dmyinit();	/* generate dmy ostrdef			   */
	return (ostrdef);
#else
	return ((char *)skfobuf);
#endif
    };

    iencode = -1;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    ruby_out_locale_index = rb_enc_find_index(i_codeset[out_codeset].cname);
    iencode = cstr->codeset;
#endif
/* --- conversion call ------------------------------------------- */
    r_skf_convert(lwlstr,ibuflen,FORCE_INIT,iencode);
    lwl_putchar(0x00);	/* add terminater			   */
    errorcode = skf_swig_result;

#ifdef DEBUG
    if (is_vv_debug) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,ostrdef->sstr);
#else
	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,cstr);
#endif
    };

#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    if (cstr != NULL) free(cstr);
    return (ostrdef);
#else
    return ((char *)skfobuf);
#endif
}

@SKFSTRINGS@ *quickconvert(char *optstr, @SKFSTRINGS@ *cstr)
{
    long	ibuflen;
    int		result = 0;
    struct	Skf_localestring *lwlstr;

    if (swig_state == 0) {
#ifdef SKFDEBUG
	if (is_vv_debug) fprintf(stderr,"\nextension initialize\n");
#endif
	skf_script_init(); swig_state = 1;
    };
    debug_opt = 0;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    lwlstr = cstr;
    ibuflen = get_rstr_len(cstr);
#else
    lwlstr = skf_lwlstring2skfstring(cstr);
    ibuflen = lwlstr->length;
#endif

    lwlstr->codeset = in_saved_codeset;

    if (optstr != NULL) result = skf_script_param_parse(optstr,optstr_len);
    if (result < 0) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
	skf_dmyinit();	/* generate dmy ostrdef			   */
	return (ostrdef);
#else
	return ((char *)skfobuf);
#endif
    };

/* do not initialize iencode */
/* --- conversion call ------------------------------------------- */
    r_skf_convert(lwlstr,ibuflen,FORCE_INIT,iencode);
    lwl_putchar(0x00);	/* add terminater			   */
    errorcode = skf_swig_result;

#ifdef DEBUG
    if (is_vv_debug) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,ostrdef->sstr);
#else
	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,cstr);
#endif
    };

#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    if (cstr != NULL) free(cstr);
    return (ostrdef);
#else
    return ((char *)skfobuf);
#endif
}

@SKFSTRINGS@ *guess(char *optstr, @SKFSTRINGS@ *cstr)
{
    long	ibuflen;
    int		result;
    struct	Skf_localestring *lwlstr;

    skf_script_init();		/* guess does not use old value	   */
    in_saved_codeset = -1;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    lwlstr = cstr;
    ibuflen = get_rstr_len(cstr);
#else
    lwlstr = skf_lwlstring2skfstring(cstr);
    ibuflen = lwlstr->length;
#endif

    result = skf_script_param_parse(optstr,optstr_len);
    if (result < 0) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
	skf_dmyinit();	/* generate dmy ostrdef			   */
	return (ostrdef);
#else
	return ((char *)skfobuf);
#endif
    };
    set_input_inquiry;
    iencode = -1;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    ruby_out_ascii_index = rb_enc_find_index("US_ASCII");
    iencode = cstr->codeset;
#endif
/* --- conversion call ------------------------------------------- */
    r_skf_convert(lwlstr,ibuflen,FORCE_INIT,iencode);
    lwl_putchar(0x00);	/* add terminater			   */
    errorcode = skf_swig_result;

#ifdef DEBUG
    if (is_vv_debug) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,ostrdef->sstr);
#else
	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,cstr);
#endif
    };

#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
    if (cstr != NULL) free(cstr);
    return (ostrdef);
#else
    return ((char *)skfobuf);
#endif
}

void destruct()
{
    skf_script_init();
    if (skfobuf != NULL) free(skfobuf);
    skfobuf = NULL;
}

char *inputcode()
{
    return(i_codeset[in_codeset].cname);
}

%}

%import "skf_convert.h"
%newobject convert;
%newobject quickconvert;
%newobject guess;
%newobject inputcode;
%immutable cstr;
%immutable optstr;

#if defined(SWIGRUBY)
%typemap (in) (SKFSTRINGS *cstr) {
	$1 = skf_rbstring2skfstring($input);
}

%typemap (in) (char *optstr) {
	$1 = RSTRING_PTR($input);
}

%typemap (out) (SKFSTRINGS *) {
    {
	VALUE res;
	int	i;
	char	*resstr;
	unsigned char	*oostr;
	long	olen,oolen;

	oolen = (olen = $1->length);
	if (olen < 0) olen = 8;

	/* olen includes terminate \0. ruby does not expect this value, so */
	/* we have to fix up the return value.				   */
	olen--;

	res = rb_str_new(0, olen + 5);
	rb_str_set_len(res,olen);
	resstr = RSTRING_PTR(res);
	oostr = $1->sstr;
	if (o_encode) {
	    rb_enc_associate(res, rb_usascii_encoding());
	} else {
	    rb_enc_associate(res,
	      rb_enc_from_index(rb_enc_find_index(i_codeset[out_codeset].cname)));
	};
	for (i=0;i<$1->length;i++) {
	    if (oolen < 0) {
	    	*(resstr++) = ' ';
	    } else {
		*(resstr++) = *(oostr++);
	    };
	};
	$result = res;
    };
}
#endif

/* module definitions */
/* #ifdef	HAVE_FAST_LWLSTRLEN */
/* %apply @SKFSTRINGS@ *INPUT { @SKFSTRINGS@ *cstr }; */
@SKFSTRINGS@ *convert(char *optstr, @SKFSTRINGS@ *cstr);
@SKFSTRINGS@ *quickconvert(char *optstr, @SKFSTRINGS@ *cstr);
@SKFSTRINGS@ *guess(char *optstr, @SKFSTRINGS@ *cstr);
@SKFSTRINGS@ *inputcode();

/* %rename(init) skf_script_init; */
void skf_script_init();
void destruct();

int  in_codeset;
int  out_codeset;
int  errorcode;
